
/**
 * 文本提取模块 - 用于从网页中提取主要内容和格式化文本
 */
(function (window) {
    'use strict';

    let currentScore = 0;
    // --- 内部辅助函数 ---

    /**
     * 计算元素中的纯文本长度，忽略script和style标签
     * @param {Element} element - 要计算的DOM元素
     * @return {number} 纯文本长度
     */
    function getPlainTextLength(element) {
        if (!element) return 0;

        let textLength = 0;
        for (const node of element.childNodes) {
            if (node.nodeType === Node.TEXT_NODE) {
                textLength += node.textContent.trim().length;
            } else if (node.nodeType === Node.ELEMENT_NODE) {
                // 忽略script和style标签
                if (node.tagName !== 'SCRIPT' && node.tagName !== 'STYLE') {
                    // 递归计算子元素的文本长度
                    textLength += getPlainTextLength(node);
                }
            }
        }
        return textLength;
    }

    /**
     * 计算元素的总HTML文本长度（outerHTML长度）
     * @param {Element} element - 要计算的DOM元素
     * @return {number} HTML文本长度
     */
    function getAllHtmlTextLength(element) {
        if (!element || !element.outerHTML) return 0;
        return element.outerHTML.length;
    }

    /**
     * 计算元素的文本密度
     * @param {Element} element - 要计算的DOM元素
     * @return {number} 文本密度值
     */
    function calculateDensity(element) {
        const plainTextLen = getPlainTextLength(element);
        const allHtmlTextLen = getAllHtmlTextLength(element);

        if (allHtmlTextLen === 0) {
            return 0;
        }

        // 使用基本密度公式：密度 = 纯文本长度 / HTML总长度
        return plainTextLen / allHtmlTextLen;
    }

    /**
     * 计算元素的内容得分
     * @param {Element} element - 要计算的DOM元素
     * @return {number} 内容得分
     */
    function calculateScore(element) {
        // 首先通过文本密度计算基础得分
        let density = calculateDensity(element);
        let score = density;

        // 段落数量加分
        let numberOfPtag = element.querySelectorAll('p').length;
        if (numberOfPtag > 1) {
            score += numberOfPtag * 0.03;
        }
        //换行同样加分
        let numberOfBrtag = element.querySelectorAll('br').length;
        if (numberOfBrtag > 1) {
            score += numberOfBrtag * 0.02;
        }
        
        // 其他可选加分规则（已注释）
        if (element.tagName.toLowerCase() === 'article') {
            score += 0.5;  // 文章标签加分
        }
        for (let cls of element.classList) {
            cls = cls.toLowerCase();
            if(cls == 'content' || cls == 'maincontent') {
                score += 0.7; // 精确匹配content或maincontent加分
                break;
            }
            else if (cls.includes('content') ) {
                score += 0.5; // 包含content的类名加分
                break;
            } 
            if(cls == 'article') {
                score += 0.7; // 精确匹配article的类名加分
                break;
            }
            else if(cls.includes('article')) {
                score += 0.5; // 包含article的类名加分
                break;
            }
    
           if(cls.includes('comment')) {
                score -= 0.5; // 评论类名减分
            }
        }
        console.log("score:" + score + " element:" + element.tagName + "  class:" + element.classList);
        return score;
    }

    /**
     * 获取页面中最可能包含正文内容的元素
     * @return {Element|null} 主要内容元素或null
     */
    function getMainContentElement() {
        const containerTags = ['DIV', 'section', 'article', 'main', 'TH', 'UL', 'LI', 'DL', 'DT', 'DD'];
        let maxScore = 0;
        let mainContentElement = null;
        const allElements = document.querySelectorAll(containerTags.join(','));
        
        allElements.forEach(element => {
            // 跳过过小或隐藏元素
            if (element.textContent.trim().length < 50 || element.offsetWidth === 0 || element.offsetHeight === 0) {
                return;
            }
            let score = calculateScore(element);
            if (score > maxScore) {
                maxScore = score;
                mainContentElement = element;
            }
        });
        //调试
//        alert(mainContentElement.tagName + " score:" + maxScore + "  classList:" + mainContentElement.textContent);
        currentScore = maxScore;
        return mainContentElement;
    }
    /**
     * 获取当前内容的得分
     * @return {number} 当前内容得分
     */
    function getContentTextScore() {
        return currentScore;
    }

    /**
     * 提取元素中的纯文本内容，保留段落结构
     * @param {Element} element - 要提取文本的DOM元素
     * @return {string} 格式化后的文本内容
     */
    function extractText(element) {
        if (!element) return '';
        
        // 常见块级元素，遇到这些自动加换行
        const blockTags = new Set([
            'DIV', 'P', 'SECTION', 'ARTICLE', 'BR', 'LI', 'UL', 'OL', 'TABLE', 'TR', 'TD', 'TH', 
            'HEADER', 'FOOTER', 'NAV', 'MAIN', 'ASIDE', 'DL', 'DT', 'DD', 'HR', 'FIGURE', 
            'ADDRESS', 'BLOCKQUOTE', 'PRE', 'FORM', 'H1', 'H2', 'H3', 'H4', 'H5', 'H6'
        ]);
        
        let text = '';
        for (const node of element.childNodes) {
            if (node.nodeType === Node.TEXT_NODE) {
                text += node.textContent;
            } else if (node.nodeType === Node.ELEMENT_NODE) {
                if (node.tagName === 'SCRIPT' || node.tagName === 'STYLE') continue;
                let childText = extractText(node);
                if (blockTags.has(node.tagName)) {
                    // 块级元素前后加换行，避免多余换行
                    if (text && !text.endsWith('\n')) text += '\n';
                    text += childText.trim();
                    if (!text.endsWith('\n')) text += '\n';
                } else {
                    text += childText;
                }
            }
        }
        // 合理去除多余空行
        return text.replace(/\n{2,}/g, '\n').trim();
    }
    
    // --- 导出公共API ---
    
    // 创建命名空间
    window.TextExtractor = window.TextExtractor || {};
    
    // 导出公共方法
    window.TextExtractor.getMainContentElement = getMainContentElement;
    window.TextExtractor.extractText = extractText;
    window.TextExtractor.getContentTextScore = getContentTextScore;
    
    // 便捷方法：直接提取页面主要内容的文本
    window.TextExtractor.extractMainText = function() {
        const mainElement = getMainContentElement();
        return mainElement ? extractText(mainElement) : '';
    };
    
})(window);