/* Inter-process communication for PPE.
   Created by Daniel Vershinin
   2015-2016 (c) Polarr, Inc */

/* Base-64 */

var Base64 = {}

;(function() {
  "use strict"

  var chars = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/"

  // Use a lookup table to find the index.
  var lookup = new Uint8Array(256)
  for (var i = 0; i < chars.length; i++) {
    lookup[chars.charCodeAt(i)] = i
  }

  Base64.encode = function(arraybuffer) {
    var bytes = new Uint8Array(arraybuffer),
      i,
      len = bytes.length,
      base64 = ""

    for (i = 0; i < len; i += 3) {
      base64 += chars[bytes[i] >> 2]
      base64 += chars[((bytes[i] & 3) << 4) | (bytes[i + 1] >> 4)]
      base64 += chars[((bytes[i + 1] & 15) << 2) | (bytes[i + 2] >> 6)]
      base64 += chars[bytes[i + 2] & 63]
    }

    if (len % 3 === 2) {
      base64 = base64.substring(0, base64.length - 1) + "="
    } else if (len % 3 === 1) {
      base64 = base64.substring(0, base64.length - 2) + "=="
    }

    return base64
  }

  Base64.decode = function(base64) {
    var bufferLength = base64.length * 0.75,
      len = base64.length,
      i,
      p = 0,
      encoded1,
      encoded2,
      encoded3,
      encoded4

    if (base64[base64.length - 1] === "=") {
      bufferLength--
      if (base64[base64.length - 2] === "=") {
        bufferLength--
      }
    }

    var arraybuffer = new ArrayBuffer(bufferLength),
      bytes = new Uint8Array(arraybuffer)

    for (i = 0; i < len; i += 4) {
      encoded1 = lookup[base64.charCodeAt(i)]
      encoded2 = lookup[base64.charCodeAt(i + 1)]
      encoded3 = lookup[base64.charCodeAt(i + 2)]
      encoded4 = lookup[base64.charCodeAt(i + 3)]

      bytes[p++] = (encoded1 << 2) | (encoded2 >> 4)
      bytes[p++] = ((encoded2 & 15) << 4) | (encoded3 >> 2)
      bytes[p++] = ((encoded3 & 3) << 6) | (encoded4 & 63)
    }

    return arraybuffer
  }
})()

/*function base64ArrayBuffer(arrayBuffer)
{
  var base64    = ''
  var encodings = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/'

  var bytes         = new Uint8Array(arrayBuffer)
  var byteLength    = bytes.byteLength
  var byteRemainder = byteLength % 3
  var mainLength    = byteLength - byteRemainder

  var a, b, c, d
  var chunk

  // Main loop deals with bytes in chunks of 3
  for (var i = 0; i < mainLength; i = i + 3) {
    // Combine the three bytes into a single integer
    chunk = (bytes[i] << 16) | (bytes[i + 1] << 8) | bytes[i + 2]

    // Use bitmasks to extract 6-bit segments from the triplet
    a = (chunk & 16515072) >> 18 // 16515072 = (2^6 - 1) << 18
    b = (chunk & 258048)   >> 12 // 258048   = (2^6 - 1) << 12
    c = (chunk & 4032)     >>  6 // 4032     = (2^6 - 1) << 6
    d = chunk & 63               // 63       = 2^6 - 1

    // Convert the raw binary segments to the appropriate ASCII encoding
    base64 += encodings[a] + encodings[b] + encodings[c] + encodings[d]
  }

  // Deal with the remaining bytes and padding
  if (byteRemainder == 1) {
    chunk = bytes[mainLength]

    a = (chunk & 252) >> 2 // 252 = (2^6 - 1) << 2

    // Set the 4 least significant bits to zero
    b = (chunk & 3)   << 4 // 3   = 2^2 - 1

    base64 += encodings[a] + encodings[b] + '=='
  } else if (byteRemainder == 2) {
    chunk = (bytes[mainLength] << 8) | bytes[mainLength + 1]

    a = (chunk & 64512) >> 10 // 64512 = (2^6 - 1) << 10
    b = (chunk & 1008)  >>  4 // 1008  = (2^6 - 1) << 4

    // Set the 2 least significant bits to zero
    c = (chunk & 15)    <<  2 // 15    = 2^4 - 1

    base64 += encodings[a] + encodings[b] + encodings[c] + '='
  }
  
  return base64
}*/

/* Nested objects */

var NestedObjects = {
  get: function(obj, path) {
    var parsed = NestedObjects.parse(path)
    return NestedObjects.getPathValue(parsed, obj)
  },

  set: function(obj, path, val) {
    var parsed = NestedObjects.parse(path)
    NestedObjects.setPathValue(parsed, val, obj)
  },

  parse: function(path) {
    var str = (path || "").replace(/\[/g, ".[")
    var parts = str.match(/(\\\.|[^.]+?)+/g)
    var re = /\[(\d+)\]$/
    var ret = []
    var mArr = null

    for (var i = 0, len = parts.length; i < len; i++) {
      mArr = re.exec(parts[i])
      ret.push(mArr ? { i: parseFloat(mArr[1]) } : { p: parts[i] })
    }

    return ret
  },

  defined: function(val) {
    return !(!val && "undefined" === typeof val)
  },

  getPathValue: function(parsed, obj) {
    var tmp = obj
    var res

    for (var i = 0, l = parsed.length; i < l; i++) {
      var part = parsed[i]
      if (tmp) {
        if (NestedObjects.defined(part.p)) tmp = tmp[part.p]
        else if (NestedObjects.defined(part.i)) tmp = tmp[part.i]
        if (i == l - 1) res = tmp
      } else {
        res = undefined
      }
    }

    return res
  },

  setPathValue: function(parsed, val, obj) {
    var tmp = obj
    var i = 0
    var l = parsed.length
    var part

    for (; i < l; i++) {
      part = parsed[i]

      if (NestedObjects.defined(tmp) && i == l - 1) {
        var x = NestedObjects.defined(part.p) ? part.p : part.i
        tmp[x] = val
      } else if (NestedObjects.defined(tmp)) {
        if (NestedObjects.defined(part.p) && tmp[part.p]) {
          tmp = tmp[part.p]
        } else if (NestedObjects.defined(part.i) && tmp[part.i]) {
          tmp = tmp[part.i]
        } else {
          var next = parsed[i + 1]
          var x = NestedObjects.defined(part.p) ? part.p : part.i
          var y = NestedObjects.defined(next.p) ? {} : []
          tmp[x] = y
          tmp = tmp[x]
        }
      } else {
        if (i == l - 1) tmp = val
        else if (NestedObjects.defined(part.p)) tmp = {}
        else if (NestedObjects.defined(part.i)) tmp = []
      }
    }
  },
}

function LRUcache(size) {
  this.size = typeof size === "number" ? size : 0
  if (this.size) {
    this.elements = new Array()
  }
}

LRUcache.prototype.add = function(key, value) {
  var i,
    flag = 0,
    len = this.elements.length,
    key = typeof key === "string" ? key : -1

  if (key != -1) {
    for (i = 0; i < len; i++) {
      if (this.elements[i].k == key) {
        for (j = i; j < len - 1; j++) this.elements[j] = this.elements[j + 1]
        this.elements[j] = { k: key, v: value }
        flag = 1
      }
    }
    if (!flag) {
      if (len == this.size) {
        this.elements.shift()
      }
      this.elements.push({ k: key, v: value })
    }
    return true
  }
  return false
}

LRUcache.prototype.get = function(key) {
  var i,
    len = this.elements.length,
    key = typeof key == "string" ? key : -1

  if (key != -1) {
    for (i = 0; i < len; i++) {
      if (this.elements[i].k == key) {
        var tmp = this.elements[i].v

        for (j = i; j < len - 1; j++) this.elements[j] = this.elements[j + 1]

        this.elements.pop()
        this.elements.push({ k: key, v: tmp })
        return tmp
      }
    }
  }
  return -1
}

LRUcache.prototype.keys = function() {
  var i,
    len = this.elements.length
  var result = []

  for (i = 0; i < len; i++) {
    result.push(this.elements[i].k)
  }
  return result
}

/* Message that contains identifier and automatically manages extraction of partials */

function POMessage(name, contents, identifier) {
  this.name = name
  this.contents = contents
  this.replyIdentifier = -1

  this.partials = []
  this.partialsLength = 0

  if (typeof identifier === "undefined") {
    this.identifier = POMessage.getNextMessageIdentifier()
  } else {
    this.identifier = identifier
  }
}

/* POMessage instance methods */

//Getting message identifier
POMessage.prototype.getIdentifier = function() {
  return this.identifier.toString()
}

//Serialising a message to be transmitted over some data pipe
POMessage.prototype.serialize = function() {
  var message = {}
  message[POMessage.MESSAGE_KEY_NAME] = this.name
  message[POMessage.MESSAGE_KEY_DATA] = this.contents
  message[POMessage.MESSAGE_KEY_ID] = this.identifier

  var newMessage = JSON.parse(JSON.stringify(message))
  //Encoding partials in the message
  POMessage.encodePartials(newMessage)

  return newMessage
}

/* POMessage static methods */

POMessage.getNextMessageIdentifier = function() {
  return POMessage.MESSAGE_LAST_IDENTIFIER++
}

//Extracting partials
POMessage.encodePartials = function(message, path) {
  if (typeof path === "undefined") path = ""

  function encodePartialsData(data, path) {
    for (var key in data) {
      if (data[key] && data[key].buffer instanceof ArrayBuffer && data[key].byteLength !== undefined) {
        data[key] = { data: Base64.encode(data[key]) }
        data[key]["type"] = POMessage.MESSAGE_KEY_PARTIAL
        continue
      }

      if (typeof data[key] === "object") {
        encodePartialsData(data[key], path.length != 0 ? path + "." + key : key)
      }
    }
  }

  encodePartialsData(message, path)
}

POMessage.decodePartials = function(message, path) {
  if (typeof path === "undefined") path = ""

  function decodePartialsData(data, path) {
    for (var key in data) {
      if (data[key] != null && typeof data[key] === "object") {
        if (data[key]["type"] === POMessage.MESSAGE_KEY_PARTIAL) {
          data[key] = Base64.decode(data[key]["data"])
          continue
        } else {
          decodePartialsData(data[key], path.length != 0 ? path + "." + key : key)
        }
      }
    }
  }

  decodePartialsData(message, path)
}

POMessage.deserialize = function(contents) {
  var message = new POMessage(
    contents[POMessage.MESSAGE_KEY_NAME],
    contents[POMessage.MESSAGE_KEY_DATA],
    contents[POMessage.MESSAGE_KEY_ID]
  )

  POMessage.decodePartials(message.contents)

  if (contents[POMessage.MESSAGE_KEY_REPLY_ID] != null) {
    message.replyIdentifier = contents[POMessage.MESSAGE_KEY_REPLY_ID]
  }

  return message
}

/* POMessage constants */

POMessage.MESSAGE_KEY_DATA = "_i_data"
POMessage.MESSAGE_KEY_ID = "_i_id"
POMessage.MESSAGE_KEY_REPLY_ID = "_i_reply_id"
POMessage.MESSAGE_KEY_NAME = "_i_name"
POMessage.MESSAGE_KEY_PARTIAL = "_i_partial"
POMessage.MESSAGE_CALLBACK = "_i_callback"
POMessage.MESSAGE_LAST_IDENTIFIER = 0

/* Client */

/* Generic client responsible for sending messages to the server and receiving callbacks */
function POIPCClient(channel, debugChannel) {
  //Checking for android
  if (typeof window["ipcAndroid"] !== "undefined") {
    var postMessage = function(object) {
      window["ipcAndroid"].postMessage(JSON.stringify(object))
    }
    var recevieMessage = function(name, object) {
      var paramObject
      var jsonResult
      if (name === "ppe.text.getStyle") {
        // Note: cache should not be re-used by repeated calls to JSON.stringify.
        paramObject = JSON.stringify(object, function(key, value) {
          if (
            key === "children" ||
            key === "data" ||
            key === "onBlur" ||
            key === "onChange" ||
            key === "onFocus" ||
            key === "color"
          ) {
            return
          }
          return value
        })

        jsonResult = window["ipcAndroid"].recevieMessage(name, paramObject)
      } else if (name === "ppe.text.getFont.data") {
        if (object["res"]) {
          jsonResult = window["ipcAndroid"].recevieArrayMessage(name, object["name"], Uint8Array.from(object["res"]))
        }
      } else if (name === "ppe.state.get") {
        paramObject = JSON.stringify(cachedBrushs.keys())
        jsonResult = window["ipcAndroid"].recevieMessage(name, paramObject)
      } else if (name === "ppe.state.set") {
        var local_adjustments = object.state.local_adjustments
        if (local_adjustments) {
          local_adjustments.forEach(function(adjustment) {
            if (adjustment.type === "brush") {
              var brushes = adjustment.brush
              if (brushes) {
                brushes.forEach(function(brush) {
                  if (brush.pointId && window["ipcAndroid"].checkBrushCache(brush.pointId)) {
                    if (cachedBrushs.get(brush.pointId) === -1) {
                      cachedBrushs.add(brush.pointId, brush.points)
                    }
                    brush.points = []
                  } else {
                    var guid = getGUID()
                    brush.pointId = guid

                    if (cachedBrushs.get(brush.pointId) === -1) {
                      cachedBrushs.add(brush.pointId, brush.points)
                    }
                  }
                })
              }
            }
          })
        }

        paramObject = JSON.stringify(object)

        jsonResult = window["ipcAndroid"].recevieMessage(name, paramObject)
      } else {
        paramObject = JSON.stringify(object)

        jsonResult = window["ipcAndroid"].recevieMessage(name, paramObject)
      }

      if (jsonResult) {
        // check its object or not
        if (jsonResult.indexOf("_value:") == 0) {
          return jsonResult.substring(7)
        } else {
          var jsonObject = JSON.parse(jsonResult)
          if (jsonObject.hasOwnProperty("_value")) {
            return jsonObject._value
          }

          if (name === "ppe.state.get") {
            var local_adjustments = jsonObject.local_adjustments
            if (local_adjustments) {
              local_adjustments.forEach(function(adjustment) {
                if (adjustment.type === "brush") {
                  var brushes = adjustment.brush
                  if (brushes) {
                    brushes.forEach(function(brush) {
                      if (!brush.points || brush.points.length === 0) {
                        var points = cachedBrushs.get(brush.pointId)
                        if (points !== -1) {
                          brush.points = points
                        }
                      }
                    })
                  }
                }
              })
            }
          }

          return jsonObject
        }
      }

      return null
    }

    window["webkit"] = { messageHandlers: {} }
    window.webkit.messageHandlers[channel] = { postMessage: postMessage, recevieMessage: recevieMessage }
    window.webkit.messageHandlers[debugChannel] = { postMessage: function() {} }
  }

  //Server properties
  this.server = window.webkit.messageHandlers[channel]
  this.debugServer = window.webkit.messageHandlers[debugChannel]

  //Message handler
  this.messageHandler = null
}

var getGUID = function() {
  function s4() {
    return Math.floor((1 + Math.random()) * 0x10000)
      .toString(16)
      .substring(1)
  }
  return s4() + s4() + "-" + s4() + "-" + s4() + "-" + s4() + "-" + s4() + s4() + s4()
}

var cachedBrushs = new LRUcache(100)

//Setting the messsages handler
POIPCClient.prototype.setMessagesHandler = function(handler) {
  this.messageHandler = handler
}

POIPCClient.prototype.sendMessage = function(message) {
  //Notifying about the message been sent
  this.debugServer.postMessage(message.identifier)
  var result = this.server.recevieMessage(message.name, message.contents)
  if (result === "NULL") {
    return null
  }
  if (result != null) {
    return result
  }

  //Posting the main message
  this.server.postMessage(message.serialize())

  return []
}

/* Called by server */
POIPCClient.prototype.acceptMessage = function(contents) {
  var message = POMessage.deserialize(contents)
  //console.log('client_call:' +message.name);
  //Delivering instantly
  if (this.messageHandler != null) {
    this.messageHandler.call(this, message)
  }
}

/* Called by server */
POIPCClient.prototype.acceptObjectMessage = function(name, contents) {
  //console.log('client_call:' +name);

  if (POIPCStorage.sharedStorage.ipc_messagesNameWithCallbacks[name] != null) {
    POIPCStorage.sharedStorage.ipc_messagesNameWithCallbacks[name].call(this, contents)
    delete POIPCStorage.sharedStorage.ipc_messagesNameWithCallbacks[name]
  } else if (eventListeners[name] != null) {
    var jsonObject = JSON.parse(contents)
    eventListeners[name].call(this, jsonObject)
  }
}

//Shared client instance
POIPCClient.sharedClient = new POIPCClient("ipc", "ipc-debug")

/* IPC manager */
//Event listeners
var eventListeners = {}

function POIPCManager() {
  var _this = this

  //Action on incoming messages
  POIPCClient.sharedClient.setMessagesHandler(function(message) {
    if (
      message.replyIdentifier !== -1 &&
      POIPCStorage.sharedStorage.ipc_messagesWithCallbacks[message.replyIdentifier] != null
    ) {
      POIPCStorage.sharedStorage.ipc_messagesWithCallbacks[message.replyIdentifier].call(_this, message.contents)
      delete POIPCStorage.sharedStorage.ipc_messagesWithCallbacks[message.replyIdentifier]
      return
    }

    if (POIPCStorage.sharedStorage.ipc_messagesNameWithCallbacks[message.name] != null) {
      POIPCStorage.sharedStorage.ipc_messagesNameWithCallbacks[message.name].call(_this, message.contents)
      delete POIPCStorage.sharedStorage.ipc_messagesNameWithCallbacks[message.name]
      return
    }

    if (eventListeners[message.name] != null) {
      // if request need a callback
      if (message.contents[POMessage.MESSAGE_CALLBACK] != null) {
        // Callback by web of a request by server before
        var requestCallback = function(res) {
          var resMessage = new POMessage(message.contents[POMessage.MESSAGE_CALLBACK], res)
          POIPCClient.sharedClient.sendMessage(resMessage)
        }
        //console.log('ipc_callback:' +message.name);

        eventListeners[message.name].call(_this, message.contents["value"], requestCallback)
      } else {
        //console.log('ipc_callback:' +message.name);

        eventListeners[message.name].call(_this, message.contents)
      }
    }
  })
}

POIPCManager.prototype.waitForMessage = function(name, callback, option) {
  if (option && option.remove) {
    eventListeners[name] = null
  } else if (callback != null) {
    eventListeners[name] = callback
  }
}

POIPCManager.prototype.sendMessage = function(name, contents, callback) {
  var message = new POMessage(name, contents)
  if (callback) {
    POIPCStorage.sharedStorage.ipc_messagesWithCallbacks[message.identifier] = callback
    POIPCStorage.sharedStorage.ipc_messagesNameWithCallbacks[name] = callback
  }

  var result = POIPCClient.sharedClient.sendMessage(message)
  return result
}

if (typeof window["ipcAndroid"] !== "undefined") {
  POIPCManager.serviceImportURL = window["ipcAndroid"].serviceImportURL()
  POIPCManager.sharedManager = new POIPCManager()
} else {
  POIPCManager = undefined
}

/* Storage definition */

function POIPCStorage() {
  this.manager = POIPCManager.sharedManager
  //Messages with callbacks
  this.ipc_messagesWithCallbacks = {}
  this.ipc_messagesNameWithCallbacks = {}
}

POIPCStorage.sharedStorage = new POIPCStorage()

POIPCStorage.prototype.getObject = function(key, callback) {
  this.manager.sendMessage("POIPCKVS_GET", { key: key }, function(result) {
    if (callback != null) callback(result)
  })
}

POIPCStorage.prototype.setObject = function(key, value, callback) {
  this.manager.sendMessage("POIPCKVS_SET", { key: key, value: value }, function(result) {
    if (callback != null) {
      callback(result)
    }
  })
}

/* Binary helpers */

function POIPCBinary(data) {
  return data
}
