#extension GL_OES_EGL_image_external : require
precision mediump float;
varying vec2 textureCoordinate;
uniform samplerExternalOES inputImageTexture;

uniform int uEnhanceQuality;
uniform float uContrastAdjust;
uniform float uBrightnessAdjust;
uniform float uSaturationAdjust;

uniform vec2  blurCenter;
uniform float blurRadius;
uniform float aspectRatio;
uniform float textureWidthOffset;
uniform float textureHeightOffset;

const int GAUSSIAN_SAMPLES = 9;
vec2 blurCoordinates[GAUSSIAN_SAMPLES];

const float epsilon = 1e-10;
const vec3 halfVal = vec3(0.5, 0.5, 0.5);

// rgb转换到hsl色彩空间
vec3 rgbToHsl(vec3 rgb) {
    vec4 p = (rgb.g < rgb.b) ? vec4(rgb.bg, -1.0, 2.0 / 3.0) : vec4(rgb.gb, 0.0, -1.0 / 3.0);
    vec4 q = (rgb.r < p.x) ? vec4(p.xyw, rgb.r) : vec4(rgb.r, p.yzx);
    float c = q.x - min(q.w, q.y);
    float h = abs((q.w - q.y) / (6.0 * c + epsilon) + q.z);
    float l = q.x - c * 0.5;
    float s = c / (1.0 - abs(l * 2.0 - 1.0) + epsilon);
    return vec3(h, s, l);
}

// hsl转换到rgb色彩空间
vec3 hslToRgb(vec3 hsl) {
    float r = abs(hsl.x * 6.0 - 3.0) - 1.0;
    float g = 2.0 - abs(hsl.x * 6.0 - 2.0);
    float b = 2.0 - abs(hsl.x * 6.0 - 4.0);
    vec3 rgb = clamp(vec3(r, g, b), 0.0, 1.0);
    float c = (1.0 - abs(2.0 * hsl.z - 1.0)) * hsl.y;
    return (rgb - 0.5) * c + hsl.z;
}

// 高斯滤波
vec4 gaussianBlur() {

    vec2 blurStep;
    int multiplier = 0;
    vec2 singleStepOffset = vec2(textureWidthOffset, textureHeightOffset);

    for (int i = 0; i < GAUSSIAN_SAMPLES; i++) {
        multiplier = (i - ((GAUSSIAN_SAMPLES - 1) / 2));
        blurStep = float(multiplier) * singleStepOffset;
        blurCoordinates[i] = textureCoordinate.xy + blurStep;
    }

    vec2 textureCoordinateToUse = vec2(textureCoordinate.x, (textureCoordinate.y * aspectRatio + 0.5 - 0.5 * aspectRatio));
    float dist = distance(blurCenter, textureCoordinateToUse);

    if (dist < blurRadius) {
        vec4 sum = vec4(0.0);

        sum += texture2D(inputImageTexture, blurCoordinates[0]) * 0.05;
        sum += texture2D(inputImageTexture, blurCoordinates[1]) * 0.09;
        sum += texture2D(inputImageTexture, blurCoordinates[2]) * 0.12;
        sum += texture2D(inputImageTexture, blurCoordinates[3]) * 0.15;
        sum += texture2D(inputImageTexture, blurCoordinates[4]) * 0.18;
        sum += texture2D(inputImageTexture, blurCoordinates[5]) * 0.15;
        sum += texture2D(inputImageTexture, blurCoordinates[6]) * 0.12;
        sum += texture2D(inputImageTexture, blurCoordinates[7]) * 0.09;
        sum += texture2D(inputImageTexture, blurCoordinates[8]) * 0.05;

        return sum;
    } else {
        return texture2D(inputImageTexture, textureCoordinate);
    }
}

void main() {
    vec4 rgbColor = texture2D(inputImageTexture, textureCoordinate);

    if (uEnhanceQuality == 1) {
        vec3 hslColor = rgbToHsl(rgbColor.rgb);
        // auto adjust saturation
        float saturation = 1.0 + smoothstep(0.05, 0.8, hslColor.y) / 2.0;
        saturation = clamp(saturation, 1.0, uSaturationAdjust);
        float average = (rgbColor.r + rgbColor.g + rgbColor.b) / 3.0;
        // clip black color
        saturation = step(0.05, average) == 1.0 ? saturation : 1.0;
        hslColor.y = clamp(hslColor.y * saturation, 0.0, 1.0);
        // auto adjust brightness
        float diff = 1.0 - smoothstep(0.0, 0.7, hslColor.z);
        diff = uBrightnessAdjust + diff * 0.05;
        float brightnessAdjust = hslColor.z <= 0.7 ? diff : 0.0;
        hslColor.z = clamp(hslColor.z + brightnessAdjust, 0.0, 1.0);
        rgbColor = vec4(hslToRgb(hslColor), rgbColor.a);
        // adjust contrast
        vec3 contrastColor = uContrastAdjust * (rgbColor.rgb - halfVal) + halfVal;
        rgbColor.r = clamp(contrastColor.r, 0.0, 1.0);
        rgbColor.g = clamp(contrastColor.g, 0.0, 1.0);
        rgbColor.b = clamp(contrastColor.b, 0.0, 1.0);
    }

    gl_FragColor = rgbColor;
}