#version 100
precision mediump float;

// Input varyings from vertex shader
varying float v_Life;
varying float v_ModPhase;

// Uniforms
uniform vec2 u_Resolution;
uniform float u_Time;
uniform vec4 u_Color;        // Primary particle color (often white)
uniform vec4 u_AccentColor;  // Accent color for effects (often cyan/blue)
uniform float u_TransitionTime;
uniform int u_State;
uniform int u_PrevState;
uniform float u_AudioReactivity;
uniform float u_AudioReactivityMultiplier;

// Define states as constants
const int STATE_ENTRY = 0;
const int STATE_THINKING = 1;
const int STATE_SPEAKING = 2;
const int STATE_LISTENING = 3;

// Constants
const float PI = 3.14159265359;

// Helper functions
float adjustBrightness(float value, float brightnessAdjustment) {
  return clamp(value + brightnessAdjustment, 0.0, 1.0);
}

vec3 adjustBrightness(vec3 color, float brightnessAdjustment) {
  return clamp(color + vec3(brightnessAdjustment), 0.0, 1.0);
}

void main() {
  // Calculate circular shape with a soft glow
  vec2 centered = gl_PointCoord - 0.5;
  float dist = length(centered);
  
  // Create a slightly softer inner glow effect
  float innerGlow = 1.0 - smoothstep(0.0, 0.15, dist);
  float outerGlow = 1.0 - smoothstep(0.15, 0.3, dist);
  
  // Combine for a soft core with subtle edges
  float alpha = (innerGlow * 0.9 + outerGlow * 0.1) * v_Life;
  
  // Apply particle visibility based on state
  float randomValue = fract(v_ModPhase / (2.0 * PI));
  
  // Set visibility threshold based on state
  float transitionDuration = 0.5;
  float transitionProgress = min(u_TransitionTime / transitionDuration, 1.0);
  float visibilityThreshold = 1.0;
  
  if (u_State == STATE_ENTRY) {
    float startThreshold = u_PrevState == STATE_THINKING ? 0.75 : 1.0;
    visibilityThreshold = mix(startThreshold, 0.25, transitionProgress);
  } else if (u_State == STATE_THINKING) {
    float startThreshold = u_PrevState == STATE_ENTRY ? 0.25 : 1.0;
    visibilityThreshold = mix(startThreshold, 0.75, transitionProgress);
  } else {
    float startThreshold = u_PrevState == STATE_THINKING
      ? 0.75
      : u_PrevState == STATE_ENTRY ? 0.25 : 1.0;
    visibilityThreshold = mix(startThreshold, 1.0, transitionProgress);
  }
  
  // Make particle fully transparent if random value exceeds threshold
  alpha *= (randomValue > visibilityThreshold) ? 0.0 : 1.0;
  
  // Add glint effect based on modulation phase
  float normPhase = fract(v_ModPhase / (2.0 * PI));
  float glint = abs(sin(2.0 * PI * (normPhase + u_Time * 0.3) + u_Time * 3.0));
  float blendFactor = smoothstep(0.8, 1.0, glint);
  
  vec3 baseColor = u_Color.rgb;
  vec3 accentColor = u_AccentColor.rgb;
  
  // Override with uniform colors if provided and valid
  if (u_Color.a > 0.0) {
    baseColor = u_Color.rgb;
  }
  if (u_AccentColor.a > 0.0) {
    accentColor = u_AccentColor.rgb;
  }
  
  // Mix colors with blend factor
  vec3 finalColor = mix(baseColor, accentColor, blendFactor);
  
  // Apply audio reactivity brightness boost
  float brightnessBoost = (u_State == STATE_SPEAKING || u_State == STATE_LISTENING) ?
    (u_AudioReactivity * u_AudioReactivityMultiplier * 0.8) : 0.0;
  
  finalColor = adjustBrightness(finalColor, brightnessBoost);
  
  // Set final color with alpha
  gl_FragColor = vec4(clamp(finalColor, 0.0, 1.0), alpha);
} 