(()=>{var __webpack_modules__={3089:(__unused_webpack_module,exports)=>{"use strict";eval('var __webpack_unused_export__;\n\n/**\n * Implementation of Myers\' online approximate string matching algorithm [1],\n * with additional optimizations suggested by [2].\n *\n * This has O((k/w) * n) complexity where `n` is the length of the text, `k` is\n * the maximum number of errors allowed (always <= the pattern length) and `w`\n * is the word size. Because JS only supports bitwise operations on 32 bit\n * integers, `w` is 32.\n *\n * As far as I am aware, there aren\'t any online algorithms which are\n * significantly better for a wide range of input parameters. The problem can be\n * solved faster using "filter then verify" approaches which first filter out\n * regions of the text that cannot match using a "cheap" check and then verify\n * the remaining potential matches. The verify step requires an algorithm such\n * as this one however.\n *\n * The algorithm\'s approach is essentially to optimize the classic dynamic\n * programming solution to the problem by computing columns of the matrix in\n * word-sized chunks (ie. dealing with 32 chars of the pattern at a time) and\n * avoiding calculating regions of the matrix where the minimum error count is\n * guaranteed to exceed the input threshold.\n *\n * The paper consists of two parts, the first describes the core algorithm for\n * matching patterns <= the size of a word (implemented by `advanceBlock` here).\n * The second uses the core algorithm as part of a larger block-based algorithm\n * to handle longer patterns.\n *\n * [1] G. Myers, “A Fast Bit-Vector Algorithm for Approximate String Matching\n * Based on Dynamic Programming,” vol. 46, no. 3, pp. 395–415, 1999.\n *\n * [2] Šošić, M. (2014). An simd dynamic programming c/c++ library (Doctoral\n * dissertation, Fakultet Elektrotehnike i računarstva, Sveučilište u Zagrebu).\n */\n__webpack_unused_export__ = ({ value: true });\nfunction reverse(s) {\n    return s\n        .split("")\n        .reverse()\n        .join("");\n}\n/**\n * Given the ends of approximate matches for `pattern` in `text`, find\n * the start of the matches.\n *\n * @param findEndFn - Function for finding the end of matches in\n * text.\n * @return Matches with the `start` property set.\n */\nfunction findMatchStarts(text, pattern, matches) {\n    var patRev = reverse(pattern);\n    return matches.map(function (m) {\n        // Find start of each match by reversing the pattern and matching segment\n        // of text and searching for an approx match with the same number of\n        // errors.\n        var minStart = Math.max(0, m.end - pattern.length - m.errors);\n        var textRev = reverse(text.slice(minStart, m.end));\n        // If there are multiple possible start points, choose the one that\n        // maximizes the length of the match.\n        var start = findMatchEnds(textRev, patRev, m.errors).reduce(function (min, rm) {\n            if (m.end - rm.end < min) {\n                return m.end - rm.end;\n            }\n            return min;\n        }, m.end);\n        return {\n            start: start,\n            end: m.end,\n            errors: m.errors\n        };\n    });\n}\n/**\n * Return 1 if a number is non-zero or zero otherwise, without using\n * conditional operators.\n *\n * This should get inlined into `advanceBlock` below by the JIT.\n *\n * Adapted from https://stackoverflow.com/a/3912218/434243\n */\nfunction oneIfNotZero(n) {\n    return ((n | -n) >> 31) & 1;\n}\n/**\n * Block calculation step of the algorithm.\n *\n * From Fig 8. on p. 408 of [1], additionally optimized to replace conditional\n * checks with bitwise operations as per Section 4.2.3 of [2].\n *\n * @param ctx - The pattern context object\n * @param peq - The `peq` array for the current character (`ctx.peq.get(ch)`)\n * @param b - The block level\n * @param hIn - Horizontal input delta ∈ {1,0,-1}\n * @return Horizontal output delta ∈ {1,0,-1}\n */\nfunction advanceBlock(ctx, peq, b, hIn) {\n    var pV = ctx.P[b];\n    var mV = ctx.M[b];\n    var hInIsNegative = hIn >>> 31; // 1 if hIn < 0 or 0 otherwise.\n    var eq = peq[b] | hInIsNegative;\n    // Step 1: Compute horizontal deltas.\n    var xV = eq | mV;\n    var xH = (((eq & pV) + pV) ^ pV) | eq;\n    var pH = mV | ~(xH | pV);\n    var mH = pV & xH;\n    // Step 2: Update score (value of last row of this block).\n    var hOut = oneIfNotZero(pH & ctx.lastRowMask[b]) -\n        oneIfNotZero(mH & ctx.lastRowMask[b]);\n    // Step 3: Update vertical deltas for use when processing next char.\n    pH <<= 1;\n    mH <<= 1;\n    mH |= hInIsNegative;\n    pH |= oneIfNotZero(hIn) - hInIsNegative; // set pH[0] if hIn > 0\n    pV = mH | ~(xV | pH);\n    mV = pH & xV;\n    ctx.P[b] = pV;\n    ctx.M[b] = mV;\n    return hOut;\n}\n/**\n * Find the ends and error counts for matches of `pattern` in `text`.\n *\n * Only the matches with the lowest error count are reported. Other matches\n * with error counts <= maxErrors are discarded.\n *\n * This is the block-based search algorithm from Fig. 9 on p.410 of [1].\n */\nfunction findMatchEnds(text, pattern, maxErrors) {\n    if (pattern.length === 0) {\n        return [];\n    }\n    // Clamp error count so we can rely on the `maxErrors` and `pattern.length`\n    // rows being in the same block below.\n    maxErrors = Math.min(maxErrors, pattern.length);\n    var matches = [];\n    // Word size.\n    var w = 32;\n    // Index of maximum block level.\n    var bMax = Math.ceil(pattern.length / w) - 1;\n    // Context used across block calculations.\n    var ctx = {\n        P: new Uint32Array(bMax + 1),\n        M: new Uint32Array(bMax + 1),\n        lastRowMask: new Uint32Array(bMax + 1)\n    };\n    ctx.lastRowMask.fill(1 << 31);\n    ctx.lastRowMask[bMax] = 1 << (pattern.length - 1) % w;\n    // Dummy "peq" array for chars in the text which do not occur in the pattern.\n    var emptyPeq = new Uint32Array(bMax + 1);\n    // Map of UTF-16 character code to bit vector indicating positions in the\n    // pattern that equal that character.\n    var peq = new Map();\n    // Version of `peq` that only stores mappings for small characters. This\n    // allows faster lookups when iterating through the text because a simple\n    // array lookup can be done instead of a hash table lookup.\n    var asciiPeq = [];\n    for (var i = 0; i < 256; i++) {\n        asciiPeq.push(emptyPeq);\n    }\n    // Calculate `ctx.peq` - a map of character values to bitmasks indicating\n    // positions of that character within the pattern, where each bit represents\n    // a position in the pattern.\n    for (var c = 0; c < pattern.length; c += 1) {\n        var val = pattern.charCodeAt(c);\n        if (peq.has(val)) {\n            // Duplicate char in pattern.\n            continue;\n        }\n        var charPeq = new Uint32Array(bMax + 1);\n        peq.set(val, charPeq);\n        if (val < asciiPeq.length) {\n            asciiPeq[val] = charPeq;\n        }\n        for (var b = 0; b <= bMax; b += 1) {\n            charPeq[b] = 0;\n            // Set all the bits where the pattern matches the current char (ch).\n            // For indexes beyond the end of the pattern, always set the bit as if the\n            // pattern contained a wildcard char in that position.\n            for (var r = 0; r < w; r += 1) {\n                var idx = b * w + r;\n                if (idx >= pattern.length) {\n                    continue;\n                }\n                var match = pattern.charCodeAt(idx) === val;\n                if (match) {\n                    charPeq[b] |= 1 << r;\n                }\n            }\n        }\n    }\n    // Index of last-active block level in the column.\n    var y = Math.max(0, Math.ceil(maxErrors / w) - 1);\n    // Initialize maximum error count at bottom of each block.\n    var score = new Uint32Array(bMax + 1);\n    for (var b = 0; b <= y; b += 1) {\n        score[b] = (b + 1) * w;\n    }\n    score[bMax] = pattern.length;\n    // Initialize vertical deltas for each block.\n    for (var b = 0; b <= y; b += 1) {\n        ctx.P[b] = ~0;\n        ctx.M[b] = 0;\n    }\n    // Process each char of the text, computing the error count for `w` chars of\n    // the pattern at a time.\n    for (var j = 0; j < text.length; j += 1) {\n        // Lookup the bitmask representing the positions of the current char from\n        // the text within the pattern.\n        var charCode = text.charCodeAt(j);\n        var charPeq = void 0;\n        if (charCode < asciiPeq.length) {\n            // Fast array lookup.\n            charPeq = asciiPeq[charCode];\n        }\n        else {\n            // Slower hash table lookup.\n            charPeq = peq.get(charCode);\n            if (typeof charPeq === "undefined") {\n                charPeq = emptyPeq;\n            }\n        }\n        // Calculate error count for blocks that we definitely have to process for\n        // this column.\n        var carry = 0;\n        for (var b = 0; b <= y; b += 1) {\n            carry = advanceBlock(ctx, charPeq, b, carry);\n            score[b] += carry;\n        }\n        // Check if we also need to compute an additional block, or if we can reduce\n        // the number of blocks processed for the next column.\n        if (score[y] - carry <= maxErrors &&\n            y < bMax &&\n            (charPeq[y + 1] & 1 || carry < 0)) {\n            // Error count for bottom block is under threshold, increase the number of\n            // blocks processed for this column & next by 1.\n            y += 1;\n            ctx.P[y] = ~0;\n            ctx.M[y] = 0;\n            var maxBlockScore = y === bMax ? pattern.length % w : w;\n            score[y] =\n                score[y - 1] +\n                    maxBlockScore -\n                    carry +\n                    advanceBlock(ctx, charPeq, y, carry);\n        }\n        else {\n            // Error count for bottom block exceeds threshold, reduce the number of\n            // blocks processed for the next column.\n            while (y > 0 && score[y] >= maxErrors + w) {\n                y -= 1;\n            }\n        }\n        // If error count is under threshold, report a match.\n        if (y === bMax && score[y] <= maxErrors) {\n            if (score[y] < maxErrors) {\n                // Discard any earlier, worse matches.\n                matches.splice(0, matches.length);\n            }\n            matches.push({\n                start: -1,\n                end: j + 1,\n                errors: score[y]\n            });\n            // Because `search` only reports the matches with the lowest error count,\n            // we can "ratchet down" the max error threshold whenever a match is\n            // encountered and thereby save a small amount of work for the remainder\n            // of the text.\n            maxErrors = score[y];\n        }\n    }\n    return matches;\n}\n/**\n * Search for matches for `pattern` in `text` allowing up to `maxErrors` errors.\n *\n * Returns the start, and end positions and error counts for each lowest-cost\n * match. Only the "best" matches are returned.\n */\nfunction search(text, pattern, maxErrors) {\n    var matches = findMatchEnds(text, pattern, maxErrors);\n    return findMatchStarts(text, pattern, matches);\n}\nexports.Z = search;\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///3089\n')},1924:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\n\nvar callBind = __webpack_require__(5559);\n\nvar $indexOf = callBind(GetIntrinsic('String.prototype.indexOf'));\n\nmodule.exports = function callBoundIntrinsic(name, allowMissing) {\n\tvar intrinsic = GetIntrinsic(name, !!allowMissing);\n\tif (typeof intrinsic === 'function' && $indexOf(name, '.prototype.') > -1) {\n\t\treturn callBind(intrinsic);\n\t}\n\treturn intrinsic;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2NhbGwtYmluZC9jYWxsQm91bmQuanM/NTQ1ZSJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBYTs7QUFFYixtQkFBbUIsbUJBQU8sQ0FBQyxHQUFlOztBQUUxQyxlQUFlLG1CQUFPLENBQUMsSUFBSTs7QUFFM0I7O0FBRUE7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0EiLCJmaWxlIjoiMTkyNC5qcyIsInNvdXJjZXNDb250ZW50IjpbIid1c2Ugc3RyaWN0JztcblxudmFyIEdldEludHJpbnNpYyA9IHJlcXVpcmUoJ2dldC1pbnRyaW5zaWMnKTtcblxudmFyIGNhbGxCaW5kID0gcmVxdWlyZSgnLi8nKTtcblxudmFyICRpbmRleE9mID0gY2FsbEJpbmQoR2V0SW50cmluc2ljKCdTdHJpbmcucHJvdG90eXBlLmluZGV4T2YnKSk7XG5cbm1vZHVsZS5leHBvcnRzID0gZnVuY3Rpb24gY2FsbEJvdW5kSW50cmluc2ljKG5hbWUsIGFsbG93TWlzc2luZykge1xuXHR2YXIgaW50cmluc2ljID0gR2V0SW50cmluc2ljKG5hbWUsICEhYWxsb3dNaXNzaW5nKTtcblx0aWYgKHR5cGVvZiBpbnRyaW5zaWMgPT09ICdmdW5jdGlvbicgJiYgJGluZGV4T2YobmFtZSwgJy5wcm90b3R5cGUuJykgPiAtMSkge1xuXHRcdHJldHVybiBjYWxsQmluZChpbnRyaW5zaWMpO1xuXHR9XG5cdHJldHVybiBpbnRyaW5zaWM7XG59O1xuIl0sInNvdXJjZVJvb3QiOiIifQ==\n//# sourceURL=webpack-internal:///1924\n")},5559:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar bind = __webpack_require__(8612);\nvar GetIntrinsic = __webpack_require__(210);\n\nvar $apply = GetIntrinsic('%Function.prototype.apply%');\nvar $call = GetIntrinsic('%Function.prototype.call%');\nvar $reflectApply = GetIntrinsic('%Reflect.apply%', true) || bind.call($call, $apply);\n\nvar $gOPD = GetIntrinsic('%Object.getOwnPropertyDescriptor%', true);\nvar $defineProperty = GetIntrinsic('%Object.defineProperty%', true);\nvar $max = GetIntrinsic('%Math.max%');\n\nif ($defineProperty) {\n\ttry {\n\t\t$defineProperty({}, 'a', { value: 1 });\n\t} catch (e) {\n\t\t// IE 8 has a broken defineProperty\n\t\t$defineProperty = null;\n\t}\n}\n\nmodule.exports = function callBind(originalFunction) {\n\tvar func = $reflectApply(bind, $call, arguments);\n\tif ($gOPD && $defineProperty) {\n\t\tvar desc = $gOPD(func, 'length');\n\t\tif (desc.configurable) {\n\t\t\t// original length, plus the receiver, minus any additional arguments (after the receiver)\n\t\t\t$defineProperty(\n\t\t\t\tfunc,\n\t\t\t\t'length',\n\t\t\t\t{ value: 1 + $max(0, originalFunction.length - (arguments.length - 1)) }\n\t\t\t);\n\t\t}\n\t}\n\treturn func;\n};\n\nvar applyBind = function applyBind() {\n\treturn $reflectApply(bind, $apply, arguments);\n};\n\nif ($defineProperty) {\n\t$defineProperty(module.exports, 'apply', { value: applyBind });\n} else {\n\tmodule.exports.apply = applyBind;\n}\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///5559\n")},4289:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar keys = __webpack_require__(2215);\nvar hasSymbols = typeof Symbol === 'function' && typeof Symbol('foo') === 'symbol';\n\nvar toStr = Object.prototype.toString;\nvar concat = Array.prototype.concat;\nvar origDefineProperty = Object.defineProperty;\n\nvar isFunction = function (fn) {\n\treturn typeof fn === 'function' && toStr.call(fn) === '[object Function]';\n};\n\nvar arePropertyDescriptorsSupported = function () {\n\tvar obj = {};\n\ttry {\n\t\torigDefineProperty(obj, 'x', { enumerable: false, value: obj });\n\t\t// eslint-disable-next-line no-unused-vars, no-restricted-syntax\n\t\tfor (var _ in obj) { // jscs:ignore disallowUnusedVariables\n\t\t\treturn false;\n\t\t}\n\t\treturn obj.x === obj;\n\t} catch (e) { /* this is IE 8. */\n\t\treturn false;\n\t}\n};\nvar supportsDescriptors = origDefineProperty && arePropertyDescriptorsSupported();\n\nvar defineProperty = function (object, name, value, predicate) {\n\tif (name in object && (!isFunction(predicate) || !predicate())) {\n\t\treturn;\n\t}\n\tif (supportsDescriptors) {\n\t\torigDefineProperty(object, name, {\n\t\t\tconfigurable: true,\n\t\t\tenumerable: false,\n\t\t\tvalue: value,\n\t\t\twritable: true\n\t\t});\n\t} else {\n\t\tobject[name] = value;\n\t}\n};\n\nvar defineProperties = function (object, map) {\n\tvar predicates = arguments.length > 2 ? arguments[2] : {};\n\tvar props = keys(map);\n\tif (hasSymbols) {\n\t\tprops = concat.call(props, Object.getOwnPropertySymbols(map));\n\t}\n\tfor (var i = 0; i < props.length; i += 1) {\n\t\tdefineProperty(object, props[i], map[props[i]], predicates[props[i]]);\n\t}\n};\n\ndefineProperties.supportsDescriptors = !!supportsDescriptors;\n\nmodule.exports = defineProperties;\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///4289\n")},2672:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\n\nvar CodePointAt = __webpack_require__(9044);\nvar IsInteger = __webpack_require__(7868);\nvar Type = __webpack_require__(1915);\n\nvar MAX_SAFE_INTEGER = __webpack_require__(5376);\n\nvar $TypeError = GetIntrinsic('%TypeError%');\n\n// https://262.ecma-international.org/6.0/#sec-advancestringindex\n\nmodule.exports = function AdvanceStringIndex(S, index, unicode) {\n\tif (Type(S) !== 'String') {\n\t\tthrow new $TypeError('Assertion failed: `S` must be a String');\n\t}\n\tif (!IsInteger(index) || index < 0 || index > MAX_SAFE_INTEGER) {\n\t\tthrow new $TypeError('Assertion failed: `length` must be an integer >= 0 and <= 2**53');\n\t}\n\tif (Type(unicode) !== 'Boolean') {\n\t\tthrow new $TypeError('Assertion failed: `unicode` must be a Boolean');\n\t}\n\tif (!unicode) {\n\t\treturn index + 1;\n\t}\n\tvar length = S.length;\n\tif ((index + 1) >= length) {\n\t\treturn index + 1;\n\t}\n\tvar cp = CodePointAt(S, index);\n\treturn index + cp['[[CodeUnitCount]]'];\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvQWR2YW5jZVN0cmluZ0luZGV4LmpzP2Q1ZmMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQWE7O0FBRWIsbUJBQW1CLG1CQUFPLENBQUMsR0FBZTs7QUFFMUMsa0JBQWtCLG1CQUFPLENBQUMsSUFBZTtBQUN6QyxnQkFBZ0IsbUJBQU8sQ0FBQyxJQUFhO0FBQ3JDLFdBQVcsbUJBQU8sQ0FBQyxJQUFROztBQUUzQix1QkFBdUIsbUJBQU8sQ0FBQyxJQUEyQjs7QUFFMUQ7O0FBRUE7O0FBRUE7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQSIsImZpbGUiOiIyNjcyLmpzIiwic291cmNlc0NvbnRlbnQiOlsiJ3VzZSBzdHJpY3QnO1xuXG52YXIgR2V0SW50cmluc2ljID0gcmVxdWlyZSgnZ2V0LWludHJpbnNpYycpO1xuXG52YXIgQ29kZVBvaW50QXQgPSByZXF1aXJlKCcuL0NvZGVQb2ludEF0Jyk7XG52YXIgSXNJbnRlZ2VyID0gcmVxdWlyZSgnLi9Jc0ludGVnZXInKTtcbnZhciBUeXBlID0gcmVxdWlyZSgnLi9UeXBlJyk7XG5cbnZhciBNQVhfU0FGRV9JTlRFR0VSID0gcmVxdWlyZSgnLi4vaGVscGVycy9tYXhTYWZlSW50ZWdlcicpO1xuXG52YXIgJFR5cGVFcnJvciA9IEdldEludHJpbnNpYygnJVR5cGVFcnJvciUnKTtcblxuLy8gaHR0cHM6Ly8yNjIuZWNtYS1pbnRlcm5hdGlvbmFsLm9yZy82LjAvI3NlYy1hZHZhbmNlc3RyaW5naW5kZXhcblxubW9kdWxlLmV4cG9ydHMgPSBmdW5jdGlvbiBBZHZhbmNlU3RyaW5nSW5kZXgoUywgaW5kZXgsIHVuaWNvZGUpIHtcblx0aWYgKFR5cGUoUykgIT09ICdTdHJpbmcnKSB7XG5cdFx0dGhyb3cgbmV3ICRUeXBlRXJyb3IoJ0Fzc2VydGlvbiBmYWlsZWQ6IGBTYCBtdXN0IGJlIGEgU3RyaW5nJyk7XG5cdH1cblx0aWYgKCFJc0ludGVnZXIoaW5kZXgpIHx8IGluZGV4IDwgMCB8fCBpbmRleCA+IE1BWF9TQUZFX0lOVEVHRVIpIHtcblx0XHR0aHJvdyBuZXcgJFR5cGVFcnJvcignQXNzZXJ0aW9uIGZhaWxlZDogYGxlbmd0aGAgbXVzdCBiZSBhbiBpbnRlZ2VyID49IDAgYW5kIDw9IDIqKjUzJyk7XG5cdH1cblx0aWYgKFR5cGUodW5pY29kZSkgIT09ICdCb29sZWFuJykge1xuXHRcdHRocm93IG5ldyAkVHlwZUVycm9yKCdBc3NlcnRpb24gZmFpbGVkOiBgdW5pY29kZWAgbXVzdCBiZSBhIEJvb2xlYW4nKTtcblx0fVxuXHRpZiAoIXVuaWNvZGUpIHtcblx0XHRyZXR1cm4gaW5kZXggKyAxO1xuXHR9XG5cdHZhciBsZW5ndGggPSBTLmxlbmd0aDtcblx0aWYgKChpbmRleCArIDEpID49IGxlbmd0aCkge1xuXHRcdHJldHVybiBpbmRleCArIDE7XG5cdH1cblx0dmFyIGNwID0gQ29kZVBvaW50QXQoUywgaW5kZXgpO1xuXHRyZXR1cm4gaW5kZXggKyBjcFsnW1tDb2RlVW5pdENvdW50XV0nXTtcbn07XG4iXSwic291cmNlUm9vdCI6IiJ9\n//# sourceURL=webpack-internal:///2672\n")},7800:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\nvar callBound = __webpack_require__(1924);\n\nvar $TypeError = GetIntrinsic('%TypeError%');\n\nvar IsArray = __webpack_require__(4400);\n\nvar $apply = GetIntrinsic('%Reflect.apply%', true) || callBound('%Function.prototype.apply%');\n\n// https://ecma-international.org/ecma-262/6.0/#sec-call\n\nmodule.exports = function Call(F, V) {\n\tvar argumentsList = arguments.length > 2 ? arguments[2] : [];\n\tif (!IsArray(argumentsList)) {\n\t\tthrow new $TypeError('Assertion failed: optional `argumentsList`, if provided, must be a List');\n\t}\n\treturn $apply(F, V, argumentsList);\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvQ2FsbC5qcz80OTA4Il0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFhOztBQUViLG1CQUFtQixtQkFBTyxDQUFDLEdBQWU7QUFDMUMsZ0JBQWdCLG1CQUFPLENBQUMsSUFBcUI7O0FBRTdDOztBQUVBLGNBQWMsbUJBQU8sQ0FBQyxJQUFXOztBQUVqQzs7QUFFQTs7QUFFQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQSIsImZpbGUiOiI3ODAwLmpzIiwic291cmNlc0NvbnRlbnQiOlsiJ3VzZSBzdHJpY3QnO1xuXG52YXIgR2V0SW50cmluc2ljID0gcmVxdWlyZSgnZ2V0LWludHJpbnNpYycpO1xudmFyIGNhbGxCb3VuZCA9IHJlcXVpcmUoJ2NhbGwtYmluZC9jYWxsQm91bmQnKTtcblxudmFyICRUeXBlRXJyb3IgPSBHZXRJbnRyaW5zaWMoJyVUeXBlRXJyb3IlJyk7XG5cbnZhciBJc0FycmF5ID0gcmVxdWlyZSgnLi9Jc0FycmF5Jyk7XG5cbnZhciAkYXBwbHkgPSBHZXRJbnRyaW5zaWMoJyVSZWZsZWN0LmFwcGx5JScsIHRydWUpIHx8IGNhbGxCb3VuZCgnJUZ1bmN0aW9uLnByb3RvdHlwZS5hcHBseSUnKTtcblxuLy8gaHR0cHM6Ly9lY21hLWludGVybmF0aW9uYWwub3JnL2VjbWEtMjYyLzYuMC8jc2VjLWNhbGxcblxubW9kdWxlLmV4cG9ydHMgPSBmdW5jdGlvbiBDYWxsKEYsIFYpIHtcblx0dmFyIGFyZ3VtZW50c0xpc3QgPSBhcmd1bWVudHMubGVuZ3RoID4gMiA/IGFyZ3VtZW50c1syXSA6IFtdO1xuXHRpZiAoIUlzQXJyYXkoYXJndW1lbnRzTGlzdCkpIHtcblx0XHR0aHJvdyBuZXcgJFR5cGVFcnJvcignQXNzZXJ0aW9uIGZhaWxlZDogb3B0aW9uYWwgYGFyZ3VtZW50c0xpc3RgLCBpZiBwcm92aWRlZCwgbXVzdCBiZSBhIExpc3QnKTtcblx0fVxuXHRyZXR1cm4gJGFwcGx5KEYsIFYsIGFyZ3VtZW50c0xpc3QpO1xufTtcbiJdLCJzb3VyY2VSb290IjoiIn0=\n//# sourceURL=webpack-internal:///7800\n")},9044:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\n\nvar $TypeError = GetIntrinsic('%TypeError%');\nvar callBound = __webpack_require__(1924);\nvar isLeadingSurrogate = __webpack_require__(1996);\nvar isTrailingSurrogate = __webpack_require__(6935);\n\nvar Type = __webpack_require__(1915);\nvar UTF16DecodeSurrogatePair = __webpack_require__(4969);\n\nvar $charAt = callBound('String.prototype.charAt');\nvar $charCodeAt = callBound('String.prototype.charCodeAt');\n\n// https://262.ecma-international.org/11.0/#sec-codepointat\n\nmodule.exports = function CodePointAt(string, position) {\n\tif (Type(string) !== 'String') {\n\t\tthrow new $TypeError('Assertion failed: `string` must be a String');\n\t}\n\tvar size = string.length;\n\tif (position < 0 || position >= size) {\n\t\tthrow new $TypeError('Assertion failed: `position` must be >= 0, and < the length of `string`');\n\t}\n\tvar first = $charCodeAt(string, position);\n\tvar cp = $charAt(string, position);\n\tvar firstIsLeading = isLeadingSurrogate(first);\n\tvar firstIsTrailing = isTrailingSurrogate(first);\n\tif (!firstIsLeading && !firstIsTrailing) {\n\t\treturn {\n\t\t\t'[[CodePoint]]': cp,\n\t\t\t'[[CodeUnitCount]]': 1,\n\t\t\t'[[IsUnpairedSurrogate]]': false\n\t\t};\n\t}\n\tif (firstIsTrailing || (position + 1 === size)) {\n\t\treturn {\n\t\t\t'[[CodePoint]]': cp,\n\t\t\t'[[CodeUnitCount]]': 1,\n\t\t\t'[[IsUnpairedSurrogate]]': true\n\t\t};\n\t}\n\tvar second = $charCodeAt(string, position + 1);\n\tif (!isTrailingSurrogate(second)) {\n\t\treturn {\n\t\t\t'[[CodePoint]]': cp,\n\t\t\t'[[CodeUnitCount]]': 1,\n\t\t\t'[[IsUnpairedSurrogate]]': true\n\t\t};\n\t}\n\n\treturn {\n\t\t'[[CodePoint]]': UTF16DecodeSurrogatePair(first, second),\n\t\t'[[CodeUnitCount]]': 2,\n\t\t'[[IsUnpairedSurrogate]]': false\n\t};\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///9044\n")},3674:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\n\nvar $TypeError = GetIntrinsic('%TypeError%');\n\nvar Type = __webpack_require__(1915);\n\n// https://ecma-international.org/ecma-262/6.0/#sec-createiterresultobject\n\nmodule.exports = function CreateIterResultObject(value, done) {\n\tif (Type(done) !== 'Boolean') {\n\t\tthrow new $TypeError('Assertion failed: Type(done) is not Boolean');\n\t}\n\treturn {\n\t\tvalue: value,\n\t\tdone: done\n\t};\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvQ3JlYXRlSXRlclJlc3VsdE9iamVjdC5qcz9lYTY3Il0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFhOztBQUViLG1CQUFtQixtQkFBTyxDQUFDLEdBQWU7O0FBRTFDOztBQUVBLFdBQVcsbUJBQU8sQ0FBQyxJQUFROztBQUUzQjs7QUFFQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0EiLCJmaWxlIjoiMzY3NC5qcyIsInNvdXJjZXNDb250ZW50IjpbIid1c2Ugc3RyaWN0JztcblxudmFyIEdldEludHJpbnNpYyA9IHJlcXVpcmUoJ2dldC1pbnRyaW5zaWMnKTtcblxudmFyICRUeXBlRXJyb3IgPSBHZXRJbnRyaW5zaWMoJyVUeXBlRXJyb3IlJyk7XG5cbnZhciBUeXBlID0gcmVxdWlyZSgnLi9UeXBlJyk7XG5cbi8vIGh0dHBzOi8vZWNtYS1pbnRlcm5hdGlvbmFsLm9yZy9lY21hLTI2Mi82LjAvI3NlYy1jcmVhdGVpdGVycmVzdWx0b2JqZWN0XG5cbm1vZHVsZS5leHBvcnRzID0gZnVuY3Rpb24gQ3JlYXRlSXRlclJlc3VsdE9iamVjdCh2YWx1ZSwgZG9uZSkge1xuXHRpZiAoVHlwZShkb25lKSAhPT0gJ0Jvb2xlYW4nKSB7XG5cdFx0dGhyb3cgbmV3ICRUeXBlRXJyb3IoJ0Fzc2VydGlvbiBmYWlsZWQ6IFR5cGUoZG9uZSkgaXMgbm90IEJvb2xlYW4nKTtcblx0fVxuXHRyZXR1cm4ge1xuXHRcdHZhbHVlOiB2YWx1ZSxcblx0XHRkb25lOiBkb25lXG5cdH07XG59O1xuIl0sInNvdXJjZVJvb3QiOiIifQ==\n//# sourceURL=webpack-internal:///3674\n")},7942:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\n\nvar $TypeError = GetIntrinsic('%TypeError%');\n\nvar isPropertyDescriptor = __webpack_require__(9146);\nvar DefineOwnProperty = __webpack_require__(509);\n\nvar FromPropertyDescriptor = __webpack_require__(1922);\nvar IsAccessorDescriptor = __webpack_require__(1876);\nvar IsDataDescriptor = __webpack_require__(4307);\nvar IsPropertyKey = __webpack_require__(1388);\nvar SameValue = __webpack_require__(1221);\nvar ToPropertyDescriptor = __webpack_require__(2449);\nvar Type = __webpack_require__(1915);\n\n// https://ecma-international.org/ecma-262/6.0/#sec-definepropertyorthrow\n\nmodule.exports = function DefinePropertyOrThrow(O, P, desc) {\n\tif (Type(O) !== 'Object') {\n\t\tthrow new $TypeError('Assertion failed: Type(O) is not Object');\n\t}\n\n\tif (!IsPropertyKey(P)) {\n\t\tthrow new $TypeError('Assertion failed: IsPropertyKey(P) is not true');\n\t}\n\n\tvar Desc = isPropertyDescriptor({\n\t\tType: Type,\n\t\tIsDataDescriptor: IsDataDescriptor,\n\t\tIsAccessorDescriptor: IsAccessorDescriptor\n\t}, desc) ? desc : ToPropertyDescriptor(desc);\n\tif (!isPropertyDescriptor({\n\t\tType: Type,\n\t\tIsDataDescriptor: IsDataDescriptor,\n\t\tIsAccessorDescriptor: IsAccessorDescriptor\n\t}, Desc)) {\n\t\tthrow new $TypeError('Assertion failed: Desc is not a valid Property Descriptor');\n\t}\n\n\treturn DefineOwnProperty(\n\t\tIsDataDescriptor,\n\t\tSameValue,\n\t\tFromPropertyDescriptor,\n\t\tO,\n\t\tP,\n\t\tDesc\n\t);\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///7942\n")},1922:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar assertRecord = __webpack_require__(4179);\n\nvar Type = __webpack_require__(1915);\n\n// https://ecma-international.org/ecma-262/6.0/#sec-frompropertydescriptor\n\nmodule.exports = function FromPropertyDescriptor(Desc) {\n\tif (typeof Desc === 'undefined') {\n\t\treturn Desc;\n\t}\n\n\tassertRecord(Type, 'Property Descriptor', 'Desc', Desc);\n\n\tvar obj = {};\n\tif ('[[Value]]' in Desc) {\n\t\tobj.value = Desc['[[Value]]'];\n\t}\n\tif ('[[Writable]]' in Desc) {\n\t\tobj.writable = Desc['[[Writable]]'];\n\t}\n\tif ('[[Get]]' in Desc) {\n\t\tobj.get = Desc['[[Get]]'];\n\t}\n\tif ('[[Set]]' in Desc) {\n\t\tobj.set = Desc['[[Set]]'];\n\t}\n\tif ('[[Enumerable]]' in Desc) {\n\t\tobj.enumerable = Desc['[[Enumerable]]'];\n\t}\n\tif ('[[Configurable]]' in Desc) {\n\t\tobj.configurable = Desc['[[Configurable]]'];\n\t}\n\treturn obj;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvRnJvbVByb3BlcnR5RGVzY3JpcHRvci5qcz80OWYwIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFhOztBQUViLG1CQUFtQixtQkFBTyxDQUFDLElBQXlCOztBQUVwRCxXQUFXLG1CQUFPLENBQUMsSUFBUTs7QUFFM0I7O0FBRUE7QUFDQTtBQUNBO0FBQ0E7O0FBRUE7O0FBRUE7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBIiwiZmlsZSI6IjE5MjIuanMiLCJzb3VyY2VzQ29udGVudCI6WyIndXNlIHN0cmljdCc7XG5cbnZhciBhc3NlcnRSZWNvcmQgPSByZXF1aXJlKCcuLi9oZWxwZXJzL2Fzc2VydFJlY29yZCcpO1xuXG52YXIgVHlwZSA9IHJlcXVpcmUoJy4vVHlwZScpO1xuXG4vLyBodHRwczovL2VjbWEtaW50ZXJuYXRpb25hbC5vcmcvZWNtYS0yNjIvNi4wLyNzZWMtZnJvbXByb3BlcnR5ZGVzY3JpcHRvclxuXG5tb2R1bGUuZXhwb3J0cyA9IGZ1bmN0aW9uIEZyb21Qcm9wZXJ0eURlc2NyaXB0b3IoRGVzYykge1xuXHRpZiAodHlwZW9mIERlc2MgPT09ICd1bmRlZmluZWQnKSB7XG5cdFx0cmV0dXJuIERlc2M7XG5cdH1cblxuXHRhc3NlcnRSZWNvcmQoVHlwZSwgJ1Byb3BlcnR5IERlc2NyaXB0b3InLCAnRGVzYycsIERlc2MpO1xuXG5cdHZhciBvYmogPSB7fTtcblx0aWYgKCdbW1ZhbHVlXV0nIGluIERlc2MpIHtcblx0XHRvYmoudmFsdWUgPSBEZXNjWydbW1ZhbHVlXV0nXTtcblx0fVxuXHRpZiAoJ1tbV3JpdGFibGVdXScgaW4gRGVzYykge1xuXHRcdG9iai53cml0YWJsZSA9IERlc2NbJ1tbV3JpdGFibGVdXSddO1xuXHR9XG5cdGlmICgnW1tHZXRdXScgaW4gRGVzYykge1xuXHRcdG9iai5nZXQgPSBEZXNjWydbW0dldF1dJ107XG5cdH1cblx0aWYgKCdbW1NldF1dJyBpbiBEZXNjKSB7XG5cdFx0b2JqLnNldCA9IERlc2NbJ1tbU2V0XV0nXTtcblx0fVxuXHRpZiAoJ1tbRW51bWVyYWJsZV1dJyBpbiBEZXNjKSB7XG5cdFx0b2JqLmVudW1lcmFibGUgPSBEZXNjWydbW0VudW1lcmFibGVdXSddO1xuXHR9XG5cdGlmICgnW1tDb25maWd1cmFibGVdXScgaW4gRGVzYykge1xuXHRcdG9iai5jb25maWd1cmFibGUgPSBEZXNjWydbW0NvbmZpZ3VyYWJsZV1dJ107XG5cdH1cblx0cmV0dXJuIG9iajtcbn07XG4iXSwic291cmNlUm9vdCI6IiJ9\n//# sourceURL=webpack-internal:///1922\n")},5588:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\n\nvar $TypeError = GetIntrinsic('%TypeError%');\n\nvar inspect = __webpack_require__(631);\n\nvar IsPropertyKey = __webpack_require__(1388);\nvar Type = __webpack_require__(1915);\n\n/**\n * 7.3.1 Get (O, P) - https://ecma-international.org/ecma-262/6.0/#sec-get-o-p\n * 1. Assert: Type(O) is Object.\n * 2. Assert: IsPropertyKey(P) is true.\n * 3. Return O.[[Get]](P, O).\n */\n\nmodule.exports = function Get(O, P) {\n\t// 7.3.1.1\n\tif (Type(O) !== 'Object') {\n\t\tthrow new $TypeError('Assertion failed: Type(O) is not Object');\n\t}\n\t// 7.3.1.2\n\tif (!IsPropertyKey(P)) {\n\t\tthrow new $TypeError('Assertion failed: IsPropertyKey(P) is not true, got ' + inspect(P));\n\t}\n\t// 7.3.1.3\n\treturn O[P];\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvR2V0LmpzPzdjZTMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQWE7O0FBRWIsbUJBQW1CLG1CQUFPLENBQUMsR0FBZTs7QUFFMUM7O0FBRUEsY0FBYyxtQkFBTyxDQUFDLEdBQWdCOztBQUV0QyxvQkFBb0IsbUJBQU8sQ0FBQyxJQUFpQjtBQUM3QyxXQUFXLG1CQUFPLENBQUMsSUFBUTs7QUFFM0I7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBOztBQUVBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQSIsImZpbGUiOiI1NTg4LmpzIiwic291cmNlc0NvbnRlbnQiOlsiJ3VzZSBzdHJpY3QnO1xuXG52YXIgR2V0SW50cmluc2ljID0gcmVxdWlyZSgnZ2V0LWludHJpbnNpYycpO1xuXG52YXIgJFR5cGVFcnJvciA9IEdldEludHJpbnNpYygnJVR5cGVFcnJvciUnKTtcblxudmFyIGluc3BlY3QgPSByZXF1aXJlKCdvYmplY3QtaW5zcGVjdCcpO1xuXG52YXIgSXNQcm9wZXJ0eUtleSA9IHJlcXVpcmUoJy4vSXNQcm9wZXJ0eUtleScpO1xudmFyIFR5cGUgPSByZXF1aXJlKCcuL1R5cGUnKTtcblxuLyoqXG4gKiA3LjMuMSBHZXQgKE8sIFApIC0gaHR0cHM6Ly9lY21hLWludGVybmF0aW9uYWwub3JnL2VjbWEtMjYyLzYuMC8jc2VjLWdldC1vLXBcbiAqIDEuIEFzc2VydDogVHlwZShPKSBpcyBPYmplY3QuXG4gKiAyLiBBc3NlcnQ6IElzUHJvcGVydHlLZXkoUCkgaXMgdHJ1ZS5cbiAqIDMuIFJldHVybiBPLltbR2V0XV0oUCwgTykuXG4gKi9cblxubW9kdWxlLmV4cG9ydHMgPSBmdW5jdGlvbiBHZXQoTywgUCkge1xuXHQvLyA3LjMuMS4xXG5cdGlmIChUeXBlKE8pICE9PSAnT2JqZWN0Jykge1xuXHRcdHRocm93IG5ldyAkVHlwZUVycm9yKCdBc3NlcnRpb24gZmFpbGVkOiBUeXBlKE8pIGlzIG5vdCBPYmplY3QnKTtcblx0fVxuXHQvLyA3LjMuMS4yXG5cdGlmICghSXNQcm9wZXJ0eUtleShQKSkge1xuXHRcdHRocm93IG5ldyAkVHlwZUVycm9yKCdBc3NlcnRpb24gZmFpbGVkOiBJc1Byb3BlcnR5S2V5KFApIGlzIG5vdCB0cnVlLCBnb3QgJyArIGluc3BlY3QoUCkpO1xuXHR9XG5cdC8vIDcuMy4xLjNcblx0cmV0dXJuIE9bUF07XG59O1xuIl0sInNvdXJjZVJvb3QiOiIifQ==\n//# sourceURL=webpack-internal:///5588\n")},3081:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\n\nvar $TypeError = GetIntrinsic('%TypeError%');\n\nvar GetV = __webpack_require__(5423);\nvar IsCallable = __webpack_require__(590);\nvar IsPropertyKey = __webpack_require__(1388);\n\n/**\n * 7.3.9 - https://ecma-international.org/ecma-262/6.0/#sec-getmethod\n * 1. Assert: IsPropertyKey(P) is true.\n * 2. Let func be GetV(O, P).\n * 3. ReturnIfAbrupt(func).\n * 4. If func is either undefined or null, return undefined.\n * 5. If IsCallable(func) is false, throw a TypeError exception.\n * 6. Return func.\n */\n\nmodule.exports = function GetMethod(O, P) {\n\t// 7.3.9.1\n\tif (!IsPropertyKey(P)) {\n\t\tthrow new $TypeError('Assertion failed: IsPropertyKey(P) is not true');\n\t}\n\n\t// 7.3.9.2\n\tvar func = GetV(O, P);\n\n\t// 7.3.9.4\n\tif (func == null) {\n\t\treturn void 0;\n\t}\n\n\t// 7.3.9.5\n\tif (!IsCallable(func)) {\n\t\tthrow new $TypeError(P + 'is not a function');\n\t}\n\n\t// 7.3.9.6\n\treturn func;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvR2V0TWV0aG9kLmpzPzRmYTUiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQWE7O0FBRWIsbUJBQW1CLG1CQUFPLENBQUMsR0FBZTs7QUFFMUM7O0FBRUEsV0FBVyxtQkFBTyxDQUFDLElBQVE7QUFDM0IsaUJBQWlCLG1CQUFPLENBQUMsR0FBYztBQUN2QyxvQkFBb0IsbUJBQU8sQ0FBQyxJQUFpQjs7QUFFN0M7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBOztBQUVBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7O0FBRUE7QUFDQTs7QUFFQTtBQUNBO0FBQ0E7QUFDQTs7QUFFQTtBQUNBO0FBQ0E7QUFDQTs7QUFFQTtBQUNBO0FBQ0EiLCJmaWxlIjoiMzA4MS5qcyIsInNvdXJjZXNDb250ZW50IjpbIid1c2Ugc3RyaWN0JztcblxudmFyIEdldEludHJpbnNpYyA9IHJlcXVpcmUoJ2dldC1pbnRyaW5zaWMnKTtcblxudmFyICRUeXBlRXJyb3IgPSBHZXRJbnRyaW5zaWMoJyVUeXBlRXJyb3IlJyk7XG5cbnZhciBHZXRWID0gcmVxdWlyZSgnLi9HZXRWJyk7XG52YXIgSXNDYWxsYWJsZSA9IHJlcXVpcmUoJy4vSXNDYWxsYWJsZScpO1xudmFyIElzUHJvcGVydHlLZXkgPSByZXF1aXJlKCcuL0lzUHJvcGVydHlLZXknKTtcblxuLyoqXG4gKiA3LjMuOSAtIGh0dHBzOi8vZWNtYS1pbnRlcm5hdGlvbmFsLm9yZy9lY21hLTI2Mi82LjAvI3NlYy1nZXRtZXRob2RcbiAqIDEuIEFzc2VydDogSXNQcm9wZXJ0eUtleShQKSBpcyB0cnVlLlxuICogMi4gTGV0IGZ1bmMgYmUgR2V0VihPLCBQKS5cbiAqIDMuIFJldHVybklmQWJydXB0KGZ1bmMpLlxuICogNC4gSWYgZnVuYyBpcyBlaXRoZXIgdW5kZWZpbmVkIG9yIG51bGwsIHJldHVybiB1bmRlZmluZWQuXG4gKiA1LiBJZiBJc0NhbGxhYmxlKGZ1bmMpIGlzIGZhbHNlLCB0aHJvdyBhIFR5cGVFcnJvciBleGNlcHRpb24uXG4gKiA2LiBSZXR1cm4gZnVuYy5cbiAqL1xuXG5tb2R1bGUuZXhwb3J0cyA9IGZ1bmN0aW9uIEdldE1ldGhvZChPLCBQKSB7XG5cdC8vIDcuMy45LjFcblx0aWYgKCFJc1Byb3BlcnR5S2V5KFApKSB7XG5cdFx0dGhyb3cgbmV3ICRUeXBlRXJyb3IoJ0Fzc2VydGlvbiBmYWlsZWQ6IElzUHJvcGVydHlLZXkoUCkgaXMgbm90IHRydWUnKTtcblx0fVxuXG5cdC8vIDcuMy45LjJcblx0dmFyIGZ1bmMgPSBHZXRWKE8sIFApO1xuXG5cdC8vIDcuMy45LjRcblx0aWYgKGZ1bmMgPT0gbnVsbCkge1xuXHRcdHJldHVybiB2b2lkIDA7XG5cdH1cblxuXHQvLyA3LjMuOS41XG5cdGlmICghSXNDYWxsYWJsZShmdW5jKSkge1xuXHRcdHRocm93IG5ldyAkVHlwZUVycm9yKFAgKyAnaXMgbm90IGEgZnVuY3Rpb24nKTtcblx0fVxuXG5cdC8vIDcuMy45LjZcblx0cmV0dXJuIGZ1bmM7XG59O1xuIl0sInNvdXJjZVJvb3QiOiIifQ==\n//# sourceURL=webpack-internal:///3081\n")},5423:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\n\nvar $TypeError = GetIntrinsic('%TypeError%');\n\nvar IsPropertyKey = __webpack_require__(1388);\nvar ToObject = __webpack_require__(2093);\n\n/**\n * 7.3.2 GetV (V, P)\n * 1. Assert: IsPropertyKey(P) is true.\n * 2. Let O be ToObject(V).\n * 3. ReturnIfAbrupt(O).\n * 4. Return O.[[Get]](P, V).\n */\n\nmodule.exports = function GetV(V, P) {\n\t// 7.3.2.1\n\tif (!IsPropertyKey(P)) {\n\t\tthrow new $TypeError('Assertion failed: IsPropertyKey(P) is not true');\n\t}\n\n\t// 7.3.2.2-3\n\tvar O = ToObject(V);\n\n\t// 7.3.2.4\n\treturn O[P];\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvR2V0Vi5qcz82Y2E0Il0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFhOztBQUViLG1CQUFtQixtQkFBTyxDQUFDLEdBQWU7O0FBRTFDOztBQUVBLG9CQUFvQixtQkFBTyxDQUFDLElBQWlCO0FBQzdDLGVBQWUsbUJBQU8sQ0FBQyxJQUFZOztBQUVuQztBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTs7QUFFQTtBQUNBO0FBQ0E7QUFDQTtBQUNBOztBQUVBO0FBQ0E7O0FBRUE7QUFDQTtBQUNBIiwiZmlsZSI6IjU0MjMuanMiLCJzb3VyY2VzQ29udGVudCI6WyIndXNlIHN0cmljdCc7XG5cbnZhciBHZXRJbnRyaW5zaWMgPSByZXF1aXJlKCdnZXQtaW50cmluc2ljJyk7XG5cbnZhciAkVHlwZUVycm9yID0gR2V0SW50cmluc2ljKCclVHlwZUVycm9yJScpO1xuXG52YXIgSXNQcm9wZXJ0eUtleSA9IHJlcXVpcmUoJy4vSXNQcm9wZXJ0eUtleScpO1xudmFyIFRvT2JqZWN0ID0gcmVxdWlyZSgnLi9Ub09iamVjdCcpO1xuXG4vKipcbiAqIDcuMy4yIEdldFYgKFYsIFApXG4gKiAxLiBBc3NlcnQ6IElzUHJvcGVydHlLZXkoUCkgaXMgdHJ1ZS5cbiAqIDIuIExldCBPIGJlIFRvT2JqZWN0KFYpLlxuICogMy4gUmV0dXJuSWZBYnJ1cHQoTykuXG4gKiA0LiBSZXR1cm4gTy5bW0dldF1dKFAsIFYpLlxuICovXG5cbm1vZHVsZS5leHBvcnRzID0gZnVuY3Rpb24gR2V0VihWLCBQKSB7XG5cdC8vIDcuMy4yLjFcblx0aWYgKCFJc1Byb3BlcnR5S2V5KFApKSB7XG5cdFx0dGhyb3cgbmV3ICRUeXBlRXJyb3IoJ0Fzc2VydGlvbiBmYWlsZWQ6IElzUHJvcGVydHlLZXkoUCkgaXMgbm90IHRydWUnKTtcblx0fVxuXG5cdC8vIDcuMy4yLjItM1xuXHR2YXIgTyA9IFRvT2JqZWN0KFYpO1xuXG5cdC8vIDcuMy4yLjRcblx0cmV0dXJuIE9bUF07XG59O1xuIl0sInNvdXJjZVJvb3QiOiIifQ==\n//# sourceURL=webpack-internal:///5423\n")},1876:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar has = __webpack_require__(7642);\n\nvar assertRecord = __webpack_require__(4179);\n\nvar Type = __webpack_require__(1915);\n\n// https://ecma-international.org/ecma-262/6.0/#sec-isaccessordescriptor\n\nmodule.exports = function IsAccessorDescriptor(Desc) {\n\tif (typeof Desc === 'undefined') {\n\t\treturn false;\n\t}\n\n\tassertRecord(Type, 'Property Descriptor', 'Desc', Desc);\n\n\tif (!has(Desc, '[[Get]]') && !has(Desc, '[[Set]]')) {\n\t\treturn false;\n\t}\n\n\treturn true;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvSXNBY2Nlc3NvckRlc2NyaXB0b3IuanM/NjEzZCJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBYTs7QUFFYixVQUFVLG1CQUFPLENBQUMsSUFBSzs7QUFFdkIsbUJBQW1CLG1CQUFPLENBQUMsSUFBeUI7O0FBRXBELFdBQVcsbUJBQU8sQ0FBQyxJQUFROztBQUUzQjs7QUFFQTtBQUNBO0FBQ0E7QUFDQTs7QUFFQTs7QUFFQTtBQUNBO0FBQ0E7O0FBRUE7QUFDQSIsImZpbGUiOiIxODc2LmpzIiwic291cmNlc0NvbnRlbnQiOlsiJ3VzZSBzdHJpY3QnO1xuXG52YXIgaGFzID0gcmVxdWlyZSgnaGFzJyk7XG5cbnZhciBhc3NlcnRSZWNvcmQgPSByZXF1aXJlKCcuLi9oZWxwZXJzL2Fzc2VydFJlY29yZCcpO1xuXG52YXIgVHlwZSA9IHJlcXVpcmUoJy4vVHlwZScpO1xuXG4vLyBodHRwczovL2VjbWEtaW50ZXJuYXRpb25hbC5vcmcvZWNtYS0yNjIvNi4wLyNzZWMtaXNhY2Nlc3NvcmRlc2NyaXB0b3JcblxubW9kdWxlLmV4cG9ydHMgPSBmdW5jdGlvbiBJc0FjY2Vzc29yRGVzY3JpcHRvcihEZXNjKSB7XG5cdGlmICh0eXBlb2YgRGVzYyA9PT0gJ3VuZGVmaW5lZCcpIHtcblx0XHRyZXR1cm4gZmFsc2U7XG5cdH1cblxuXHRhc3NlcnRSZWNvcmQoVHlwZSwgJ1Byb3BlcnR5IERlc2NyaXB0b3InLCAnRGVzYycsIERlc2MpO1xuXG5cdGlmICghaGFzKERlc2MsICdbW0dldF1dJykgJiYgIWhhcyhEZXNjLCAnW1tTZXRdXScpKSB7XG5cdFx0cmV0dXJuIGZhbHNlO1xuXHR9XG5cblx0cmV0dXJuIHRydWU7XG59O1xuIl0sInNvdXJjZVJvb3QiOiIifQ==\n//# sourceURL=webpack-internal:///1876\n")},4400:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\n\nvar $Array = GetIntrinsic('%Array%');\n\n// eslint-disable-next-line global-require\nvar toStr = !$Array.isArray && __webpack_require__(1924)('Object.prototype.toString');\n\n// https://ecma-international.org/ecma-262/6.0/#sec-isarray\n\nmodule.exports = $Array.isArray || function IsArray(argument) {\n\treturn toStr(argument) === '[object Array]';\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvSXNBcnJheS5qcz8yN2IxIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFhOztBQUViLG1CQUFtQixtQkFBTyxDQUFDLEdBQWU7O0FBRTFDOztBQUVBO0FBQ0EsK0JBQStCLG1CQUFPLENBQUMsSUFBcUI7O0FBRTVEOztBQUVBO0FBQ0E7QUFDQSIsImZpbGUiOiI0NDAwLmpzIiwic291cmNlc0NvbnRlbnQiOlsiJ3VzZSBzdHJpY3QnO1xuXG52YXIgR2V0SW50cmluc2ljID0gcmVxdWlyZSgnZ2V0LWludHJpbnNpYycpO1xuXG52YXIgJEFycmF5ID0gR2V0SW50cmluc2ljKCclQXJyYXklJyk7XG5cbi8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBnbG9iYWwtcmVxdWlyZVxudmFyIHRvU3RyID0gISRBcnJheS5pc0FycmF5ICYmIHJlcXVpcmUoJ2NhbGwtYmluZC9jYWxsQm91bmQnKSgnT2JqZWN0LnByb3RvdHlwZS50b1N0cmluZycpO1xuXG4vLyBodHRwczovL2VjbWEtaW50ZXJuYXRpb25hbC5vcmcvZWNtYS0yNjIvNi4wLyNzZWMtaXNhcnJheVxuXG5tb2R1bGUuZXhwb3J0cyA9ICRBcnJheS5pc0FycmF5IHx8IGZ1bmN0aW9uIElzQXJyYXkoYXJndW1lbnQpIHtcblx0cmV0dXJuIHRvU3RyKGFyZ3VtZW50KSA9PT0gJ1tvYmplY3QgQXJyYXldJztcbn07XG4iXSwic291cmNlUm9vdCI6IiJ9\n//# sourceURL=webpack-internal:///4400\n")},590:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\n// http://262.ecma-international.org/5.1/#sec-9.11\n\nmodule.exports = __webpack_require__(5320);\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvSXNDYWxsYWJsZS5qcz8xNjI5Il0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFhOztBQUViOztBQUVBLDBDQUF1QyIsImZpbGUiOiI1OTAuanMiLCJzb3VyY2VzQ29udGVudCI6WyIndXNlIHN0cmljdCc7XG5cbi8vIGh0dHA6Ly8yNjIuZWNtYS1pbnRlcm5hdGlvbmFsLm9yZy81LjEvI3NlYy05LjExXG5cbm1vZHVsZS5leHBvcnRzID0gcmVxdWlyZSgnaXMtY2FsbGFibGUnKTtcbiJdLCJzb3VyY2VSb290IjoiIn0=\n//# sourceURL=webpack-internal:///590\n")},970:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(8769);\n\nvar $construct = GetIntrinsic('%Reflect.construct%', true);\n\nvar DefinePropertyOrThrow = __webpack_require__(7942);\ntry {\n\tDefinePropertyOrThrow({}, '', { '[[Get]]': function () {} });\n} catch (e) {\n\t// Accessor properties aren't supported\n\tDefinePropertyOrThrow = null;\n}\n\n// https://ecma-international.org/ecma-262/6.0/#sec-isconstructor\n\nif (DefinePropertyOrThrow && $construct) {\n\tvar isConstructorMarker = {};\n\tvar badArrayLike = {};\n\tDefinePropertyOrThrow(badArrayLike, 'length', {\n\t\t'[[Get]]': function () {\n\t\t\tthrow isConstructorMarker;\n\t\t},\n\t\t'[[Enumerable]]': true\n\t});\n\n\tmodule.exports = function IsConstructor(argument) {\n\t\ttry {\n\t\t\t// `Reflect.construct` invokes `IsConstructor(target)` before `Get(args, 'length')`:\n\t\t\t$construct(argument, badArrayLike);\n\t\t} catch (err) {\n\t\t\treturn err === isConstructorMarker;\n\t\t}\n\t};\n} else {\n\tmodule.exports = function IsConstructor(argument) {\n\t\t// unfortunately there's no way to truly check this without try/catch `new argument` in old environments\n\t\treturn typeof argument === 'function' && !!argument.prototype;\n\t};\n}\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///970\n")},4307:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar has = __webpack_require__(7642);\n\nvar assertRecord = __webpack_require__(4179);\n\nvar Type = __webpack_require__(1915);\n\n// https://ecma-international.org/ecma-262/6.0/#sec-isdatadescriptor\n\nmodule.exports = function IsDataDescriptor(Desc) {\n\tif (typeof Desc === 'undefined') {\n\t\treturn false;\n\t}\n\n\tassertRecord(Type, 'Property Descriptor', 'Desc', Desc);\n\n\tif (!has(Desc, '[[Value]]') && !has(Desc, '[[Writable]]')) {\n\t\treturn false;\n\t}\n\n\treturn true;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvSXNEYXRhRGVzY3JpcHRvci5qcz8wNDUyIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFhOztBQUViLFVBQVUsbUJBQU8sQ0FBQyxJQUFLOztBQUV2QixtQkFBbUIsbUJBQU8sQ0FBQyxJQUF5Qjs7QUFFcEQsV0FBVyxtQkFBTyxDQUFDLElBQVE7O0FBRTNCOztBQUVBO0FBQ0E7QUFDQTtBQUNBOztBQUVBOztBQUVBO0FBQ0E7QUFDQTs7QUFFQTtBQUNBIiwiZmlsZSI6IjQzMDcuanMiLCJzb3VyY2VzQ29udGVudCI6WyIndXNlIHN0cmljdCc7XG5cbnZhciBoYXMgPSByZXF1aXJlKCdoYXMnKTtcblxudmFyIGFzc2VydFJlY29yZCA9IHJlcXVpcmUoJy4uL2hlbHBlcnMvYXNzZXJ0UmVjb3JkJyk7XG5cbnZhciBUeXBlID0gcmVxdWlyZSgnLi9UeXBlJyk7XG5cbi8vIGh0dHBzOi8vZWNtYS1pbnRlcm5hdGlvbmFsLm9yZy9lY21hLTI2Mi82LjAvI3NlYy1pc2RhdGFkZXNjcmlwdG9yXG5cbm1vZHVsZS5leHBvcnRzID0gZnVuY3Rpb24gSXNEYXRhRGVzY3JpcHRvcihEZXNjKSB7XG5cdGlmICh0eXBlb2YgRGVzYyA9PT0gJ3VuZGVmaW5lZCcpIHtcblx0XHRyZXR1cm4gZmFsc2U7XG5cdH1cblxuXHRhc3NlcnRSZWNvcmQoVHlwZSwgJ1Byb3BlcnR5IERlc2NyaXB0b3InLCAnRGVzYycsIERlc2MpO1xuXG5cdGlmICghaGFzKERlc2MsICdbW1ZhbHVlXV0nKSAmJiAhaGFzKERlc2MsICdbW1dyaXRhYmxlXV0nKSkge1xuXHRcdHJldHVybiBmYWxzZTtcblx0fVxuXG5cdHJldHVybiB0cnVlO1xufTtcbiJdLCJzb3VyY2VSb290IjoiIn0=\n//# sourceURL=webpack-internal:///4307\n")},7868:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar abs = __webpack_require__(1717);\nvar floor = __webpack_require__(9202);\n\nvar $isNaN = __webpack_require__(1214);\nvar $isFinite = __webpack_require__(3060);\n\n// https://ecma-international.org/ecma-262/6.0/#sec-isinteger\n\nmodule.exports = function IsInteger(argument) {\n\tif (typeof argument !== 'number' || $isNaN(argument) || !$isFinite(argument)) {\n\t\treturn false;\n\t}\n\tvar absValue = abs(argument);\n\treturn floor(absValue) === absValue;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvSXNJbnRlZ2VyLmpzPzkyNzciXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQWE7O0FBRWIsVUFBVSxtQkFBTyxDQUFDLElBQU87QUFDekIsWUFBWSxtQkFBTyxDQUFDLElBQVM7O0FBRTdCLGFBQWEsbUJBQU8sQ0FBQyxJQUFrQjtBQUN2QyxnQkFBZ0IsbUJBQU8sQ0FBQyxJQUFxQjs7QUFFN0M7O0FBRUE7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0EiLCJmaWxlIjoiNzg2OC5qcyIsInNvdXJjZXNDb250ZW50IjpbIid1c2Ugc3RyaWN0JztcblxudmFyIGFicyA9IHJlcXVpcmUoJy4vYWJzJyk7XG52YXIgZmxvb3IgPSByZXF1aXJlKCcuL2Zsb29yJyk7XG5cbnZhciAkaXNOYU4gPSByZXF1aXJlKCcuLi9oZWxwZXJzL2lzTmFOJyk7XG52YXIgJGlzRmluaXRlID0gcmVxdWlyZSgnLi4vaGVscGVycy9pc0Zpbml0ZScpO1xuXG4vLyBodHRwczovL2VjbWEtaW50ZXJuYXRpb25hbC5vcmcvZWNtYS0yNjIvNi4wLyNzZWMtaXNpbnRlZ2VyXG5cbm1vZHVsZS5leHBvcnRzID0gZnVuY3Rpb24gSXNJbnRlZ2VyKGFyZ3VtZW50KSB7XG5cdGlmICh0eXBlb2YgYXJndW1lbnQgIT09ICdudW1iZXInIHx8ICRpc05hTihhcmd1bWVudCkgfHwgISRpc0Zpbml0ZShhcmd1bWVudCkpIHtcblx0XHRyZXR1cm4gZmFsc2U7XG5cdH1cblx0dmFyIGFic1ZhbHVlID0gYWJzKGFyZ3VtZW50KTtcblx0cmV0dXJuIGZsb29yKGFic1ZhbHVlKSA9PT0gYWJzVmFsdWU7XG59O1xuIl0sInNvdXJjZVJvb3QiOiIifQ==\n//# sourceURL=webpack-internal:///7868\n")},1388:module=>{"use strict";eval("\n\n// https://ecma-international.org/ecma-262/6.0/#sec-ispropertykey\n\nmodule.exports = function IsPropertyKey(argument) {\n\treturn typeof argument === 'string' || typeof argument === 'symbol';\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvSXNQcm9wZXJ0eUtleS5qcz83YTIwIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFhOztBQUViOztBQUVBO0FBQ0E7QUFDQSIsImZpbGUiOiIxMzg4LmpzIiwic291cmNlc0NvbnRlbnQiOlsiJ3VzZSBzdHJpY3QnO1xuXG4vLyBodHRwczovL2VjbWEtaW50ZXJuYXRpb25hbC5vcmcvZWNtYS0yNjIvNi4wLyNzZWMtaXNwcm9wZXJ0eWtleVxuXG5tb2R1bGUuZXhwb3J0cyA9IGZ1bmN0aW9uIElzUHJvcGVydHlLZXkoYXJndW1lbnQpIHtcblx0cmV0dXJuIHR5cGVvZiBhcmd1bWVudCA9PT0gJ3N0cmluZycgfHwgdHlwZW9mIGFyZ3VtZW50ID09PSAnc3ltYm9sJztcbn07XG4iXSwic291cmNlUm9vdCI6IiJ9\n//# sourceURL=webpack-internal:///1388\n")},1137:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\n\nvar $match = GetIntrinsic('%Symbol.match%', true);\n\nvar hasRegExpMatcher = __webpack_require__(8420);\n\nvar ToBoolean = __webpack_require__(3683);\n\n// https://ecma-international.org/ecma-262/6.0/#sec-isregexp\n\nmodule.exports = function IsRegExp(argument) {\n\tif (!argument || typeof argument !== 'object') {\n\t\treturn false;\n\t}\n\tif ($match) {\n\t\tvar isRegExp = argument[$match];\n\t\tif (typeof isRegExp !== 'undefined') {\n\t\t\treturn ToBoolean(isRegExp);\n\t\t}\n\t}\n\treturn hasRegExpMatcher(argument);\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvSXNSZWdFeHAuanM/YjJjMCJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBYTs7QUFFYixtQkFBbUIsbUJBQU8sQ0FBQyxHQUFlOztBQUUxQzs7QUFFQSx1QkFBdUIsbUJBQU8sQ0FBQyxJQUFVOztBQUV6QyxnQkFBZ0IsbUJBQU8sQ0FBQyxJQUFhOztBQUVyQzs7QUFFQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0EiLCJmaWxlIjoiMTEzNy5qcyIsInNvdXJjZXNDb250ZW50IjpbIid1c2Ugc3RyaWN0JztcblxudmFyIEdldEludHJpbnNpYyA9IHJlcXVpcmUoJ2dldC1pbnRyaW5zaWMnKTtcblxudmFyICRtYXRjaCA9IEdldEludHJpbnNpYygnJVN5bWJvbC5tYXRjaCUnLCB0cnVlKTtcblxudmFyIGhhc1JlZ0V4cE1hdGNoZXIgPSByZXF1aXJlKCdpcy1yZWdleCcpO1xuXG52YXIgVG9Cb29sZWFuID0gcmVxdWlyZSgnLi9Ub0Jvb2xlYW4nKTtcblxuLy8gaHR0cHM6Ly9lY21hLWludGVybmF0aW9uYWwub3JnL2VjbWEtMjYyLzYuMC8jc2VjLWlzcmVnZXhwXG5cbm1vZHVsZS5leHBvcnRzID0gZnVuY3Rpb24gSXNSZWdFeHAoYXJndW1lbnQpIHtcblx0aWYgKCFhcmd1bWVudCB8fCB0eXBlb2YgYXJndW1lbnQgIT09ICdvYmplY3QnKSB7XG5cdFx0cmV0dXJuIGZhbHNlO1xuXHR9XG5cdGlmICgkbWF0Y2gpIHtcblx0XHR2YXIgaXNSZWdFeHAgPSBhcmd1bWVudFskbWF0Y2hdO1xuXHRcdGlmICh0eXBlb2YgaXNSZWdFeHAgIT09ICd1bmRlZmluZWQnKSB7XG5cdFx0XHRyZXR1cm4gVG9Cb29sZWFuKGlzUmVnRXhwKTtcblx0XHR9XG5cdH1cblx0cmV0dXJuIGhhc1JlZ0V4cE1hdGNoZXIoYXJndW1lbnQpO1xufTtcbiJdLCJzb3VyY2VSb290IjoiIn0=\n//# sourceURL=webpack-internal:///1137\n")},4091:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\n\nvar $ObjectCreate = GetIntrinsic('%Object.create%', true);\nvar $TypeError = GetIntrinsic('%TypeError%');\nvar $SyntaxError = GetIntrinsic('%SyntaxError%');\n\nvar IsArray = __webpack_require__(4400);\nvar Type = __webpack_require__(1915);\n\nvar hasProto = !({ __proto__: null } instanceof Object);\n\n// https://262.ecma-international.org/6.0/#sec-objectcreate\n\nmodule.exports = function OrdinaryObjectCreate(proto) {\n\tif (proto !== null && Type(proto) !== 'Object') {\n\t\tthrow new $TypeError('Assertion failed: `proto` must be null or an object');\n\t}\n\tvar additionalInternalSlotsList = arguments.length < 2 ? [] : arguments[1];\n\tif (!IsArray(additionalInternalSlotsList)) {\n\t\tthrow new $TypeError('Assertion failed: `additionalInternalSlotsList` must be an Array');\n\t}\n\t// var internalSlotsList = ['[[Prototype]]', '[[Extensible]]'];\n\tif (additionalInternalSlotsList.length > 0) {\n\t\tthrow new $SyntaxError('es-abstract does not yet support internal slots');\n\t\t// internalSlotsList.push(...additionalInternalSlotsList);\n\t}\n\t// var O = MakeBasicObject(internalSlotsList);\n\t// setProto(O, proto);\n\t// return O;\n\n\tif ($ObjectCreate) {\n\t\treturn $ObjectCreate(proto);\n\t}\n\tif (hasProto) {\n\t\treturn { __proto__: proto };\n\t}\n\n\tif (proto === null) {\n\t\tthrow new $SyntaxError('native Object.create support is required to create null objects');\n\t}\n\tvar T = function T() {};\n\tT.prototype = proto;\n\treturn new T();\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///4091\n")},356:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\n\nvar $TypeError = GetIntrinsic('%TypeError%');\n\nvar regexExec = __webpack_require__(1924)('RegExp.prototype.exec');\n\nvar Call = __webpack_require__(7800);\nvar Get = __webpack_require__(5588);\nvar IsCallable = __webpack_require__(590);\nvar Type = __webpack_require__(1915);\n\n// https://ecma-international.org/ecma-262/6.0/#sec-regexpexec\n\nmodule.exports = function RegExpExec(R, S) {\n\tif (Type(R) !== 'Object') {\n\t\tthrow new $TypeError('Assertion failed: `R` must be an Object');\n\t}\n\tif (Type(S) !== 'String') {\n\t\tthrow new $TypeError('Assertion failed: `S` must be a String');\n\t}\n\tvar exec = Get(R, 'exec');\n\tif (IsCallable(exec)) {\n\t\tvar result = Call(exec, R, [S]);\n\t\tif (result === null || Type(result) === 'Object') {\n\t\t\treturn result;\n\t\t}\n\t\tthrow new $TypeError('\"exec\" method must return `null` or an Object');\n\t}\n\treturn regexExec(R, S);\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvUmVnRXhwRXhlYy5qcz8zMzc2Il0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFhOztBQUViLG1CQUFtQixtQkFBTyxDQUFDLEdBQWU7O0FBRTFDOztBQUVBLGdCQUFnQixtQkFBTyxDQUFDLElBQXFCOztBQUU3QyxXQUFXLG1CQUFPLENBQUMsSUFBUTtBQUMzQixVQUFVLG1CQUFPLENBQUMsSUFBTztBQUN6QixpQkFBaUIsbUJBQU8sQ0FBQyxHQUFjO0FBQ3ZDLFdBQVcsbUJBQU8sQ0FBQyxJQUFROztBQUUzQjs7QUFFQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBIiwiZmlsZSI6IjM1Ni5qcyIsInNvdXJjZXNDb250ZW50IjpbIid1c2Ugc3RyaWN0JztcblxudmFyIEdldEludHJpbnNpYyA9IHJlcXVpcmUoJ2dldC1pbnRyaW5zaWMnKTtcblxudmFyICRUeXBlRXJyb3IgPSBHZXRJbnRyaW5zaWMoJyVUeXBlRXJyb3IlJyk7XG5cbnZhciByZWdleEV4ZWMgPSByZXF1aXJlKCdjYWxsLWJpbmQvY2FsbEJvdW5kJykoJ1JlZ0V4cC5wcm90b3R5cGUuZXhlYycpO1xuXG52YXIgQ2FsbCA9IHJlcXVpcmUoJy4vQ2FsbCcpO1xudmFyIEdldCA9IHJlcXVpcmUoJy4vR2V0Jyk7XG52YXIgSXNDYWxsYWJsZSA9IHJlcXVpcmUoJy4vSXNDYWxsYWJsZScpO1xudmFyIFR5cGUgPSByZXF1aXJlKCcuL1R5cGUnKTtcblxuLy8gaHR0cHM6Ly9lY21hLWludGVybmF0aW9uYWwub3JnL2VjbWEtMjYyLzYuMC8jc2VjLXJlZ2V4cGV4ZWNcblxubW9kdWxlLmV4cG9ydHMgPSBmdW5jdGlvbiBSZWdFeHBFeGVjKFIsIFMpIHtcblx0aWYgKFR5cGUoUikgIT09ICdPYmplY3QnKSB7XG5cdFx0dGhyb3cgbmV3ICRUeXBlRXJyb3IoJ0Fzc2VydGlvbiBmYWlsZWQ6IGBSYCBtdXN0IGJlIGFuIE9iamVjdCcpO1xuXHR9XG5cdGlmIChUeXBlKFMpICE9PSAnU3RyaW5nJykge1xuXHRcdHRocm93IG5ldyAkVHlwZUVycm9yKCdBc3NlcnRpb24gZmFpbGVkOiBgU2AgbXVzdCBiZSBhIFN0cmluZycpO1xuXHR9XG5cdHZhciBleGVjID0gR2V0KFIsICdleGVjJyk7XG5cdGlmIChJc0NhbGxhYmxlKGV4ZWMpKSB7XG5cdFx0dmFyIHJlc3VsdCA9IENhbGwoZXhlYywgUiwgW1NdKTtcblx0XHRpZiAocmVzdWx0ID09PSBudWxsIHx8IFR5cGUocmVzdWx0KSA9PT0gJ09iamVjdCcpIHtcblx0XHRcdHJldHVybiByZXN1bHQ7XG5cdFx0fVxuXHRcdHRocm93IG5ldyAkVHlwZUVycm9yKCdcImV4ZWNcIiBtZXRob2QgbXVzdCByZXR1cm4gYG51bGxgIG9yIGFuIE9iamVjdCcpO1xuXHR9XG5cdHJldHVybiByZWdleEV4ZWMoUiwgUyk7XG59O1xuIl0sInNvdXJjZVJvb3QiOiIifQ==\n//# sourceURL=webpack-internal:///356\n")},3733:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nmodule.exports = __webpack_require__(8631);\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvUmVxdWlyZU9iamVjdENvZXJjaWJsZS5qcz8wNDI3Il0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFhOztBQUViLDBDQUFxRCIsImZpbGUiOiIzNzMzLmpzIiwic291cmNlc0NvbnRlbnQiOlsiJ3VzZSBzdHJpY3QnO1xuXG5tb2R1bGUuZXhwb3J0cyA9IHJlcXVpcmUoJy4uLzUvQ2hlY2tPYmplY3RDb2VyY2libGUnKTtcbiJdLCJzb3VyY2VSb290IjoiIn0=\n//# sourceURL=webpack-internal:///3733\n")},1221:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar $isNaN = __webpack_require__(1214);\n\n// http://262.ecma-international.org/5.1/#sec-9.12\n\nmodule.exports = function SameValue(x, y) {\n\tif (x === y) { // 0 === -0, but they are not identical.\n\t\tif (x === 0) { return 1 / x === 1 / y; }\n\t\treturn true;\n\t}\n\treturn $isNaN(x) && $isNaN(y);\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvU2FtZVZhbHVlLmpzPzk3OTkiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQWE7O0FBRWIsYUFBYSxtQkFBTyxDQUFDLElBQWtCOztBQUV2Qzs7QUFFQTtBQUNBLGVBQWU7QUFDZixnQkFBZ0Isd0JBQXdCO0FBQ3hDO0FBQ0E7QUFDQTtBQUNBIiwiZmlsZSI6IjEyMjEuanMiLCJzb3VyY2VzQ29udGVudCI6WyIndXNlIHN0cmljdCc7XG5cbnZhciAkaXNOYU4gPSByZXF1aXJlKCcuLi9oZWxwZXJzL2lzTmFOJyk7XG5cbi8vIGh0dHA6Ly8yNjIuZWNtYS1pbnRlcm5hdGlvbmFsLm9yZy81LjEvI3NlYy05LjEyXG5cbm1vZHVsZS5leHBvcnRzID0gZnVuY3Rpb24gU2FtZVZhbHVlKHgsIHkpIHtcblx0aWYgKHggPT09IHkpIHsgLy8gMCA9PT0gLTAsIGJ1dCB0aGV5IGFyZSBub3QgaWRlbnRpY2FsLlxuXHRcdGlmICh4ID09PSAwKSB7IHJldHVybiAxIC8geCA9PT0gMSAvIHk7IH1cblx0XHRyZXR1cm4gdHJ1ZTtcblx0fVxuXHRyZXR1cm4gJGlzTmFOKHgpICYmICRpc05hTih5KTtcbn07XG4iXSwic291cmNlUm9vdCI6IiJ9\n//# sourceURL=webpack-internal:///1221\n")},2946:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\n\nvar $TypeError = GetIntrinsic('%TypeError%');\n\nvar IsPropertyKey = __webpack_require__(1388);\nvar SameValue = __webpack_require__(1221);\nvar Type = __webpack_require__(1915);\n\n// IE 9 does not throw in strict mode when writability/configurability/extensibility is violated\nvar noThrowOnStrictViolation = (function () {\n\ttry {\n\t\tdelete [].length;\n\t\treturn true;\n\t} catch (e) {\n\t\treturn false;\n\t}\n}());\n\n// https://ecma-international.org/ecma-262/6.0/#sec-set-o-p-v-throw\n\nmodule.exports = function Set(O, P, V, Throw) {\n\tif (Type(O) !== 'Object') {\n\t\tthrow new $TypeError('Assertion failed: `O` must be an Object');\n\t}\n\tif (!IsPropertyKey(P)) {\n\t\tthrow new $TypeError('Assertion failed: `P` must be a Property Key');\n\t}\n\tif (Type(Throw) !== 'Boolean') {\n\t\tthrow new $TypeError('Assertion failed: `Throw` must be a Boolean');\n\t}\n\tif (Throw) {\n\t\tO[P] = V; // eslint-disable-line no-param-reassign\n\t\tif (noThrowOnStrictViolation && !SameValue(O[P], V)) {\n\t\t\tthrow new $TypeError('Attempted to assign to readonly property.');\n\t\t}\n\t\treturn true;\n\t}\n\ttry {\n\t\tO[P] = V; // eslint-disable-line no-param-reassign\n\t\treturn noThrowOnStrictViolation ? SameValue(O[P], V) : true;\n\t} catch (e) {\n\t\treturn false;\n\t}\n\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///2946\n")},303:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\n\nvar $species = GetIntrinsic('%Symbol.species%', true);\nvar $TypeError = GetIntrinsic('%TypeError%');\n\nvar IsConstructor = __webpack_require__(970);\nvar Type = __webpack_require__(1915);\n\n// https://ecma-international.org/ecma-262/6.0/#sec-speciesconstructor\n\nmodule.exports = function SpeciesConstructor(O, defaultConstructor) {\n\tif (Type(O) !== 'Object') {\n\t\tthrow new $TypeError('Assertion failed: Type(O) is not Object');\n\t}\n\tvar C = O.constructor;\n\tif (typeof C === 'undefined') {\n\t\treturn defaultConstructor;\n\t}\n\tif (Type(C) !== 'Object') {\n\t\tthrow new $TypeError('O.constructor is not an Object');\n\t}\n\tvar S = $species ? C[$species] : void 0;\n\tif (S == null) {\n\t\treturn defaultConstructor;\n\t}\n\tif (IsConstructor(S)) {\n\t\treturn S;\n\t}\n\tthrow new $TypeError('no constructor found');\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvU3BlY2llc0NvbnN0cnVjdG9yLmpzP2U2NjYiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQWE7O0FBRWIsbUJBQW1CLG1CQUFPLENBQUMsR0FBZTs7QUFFMUM7QUFDQTs7QUFFQSxvQkFBb0IsbUJBQU8sQ0FBQyxHQUFpQjtBQUM3QyxXQUFXLG1CQUFPLENBQUMsSUFBUTs7QUFFM0I7O0FBRUE7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQSIsImZpbGUiOiIzMDMuanMiLCJzb3VyY2VzQ29udGVudCI6WyIndXNlIHN0cmljdCc7XG5cbnZhciBHZXRJbnRyaW5zaWMgPSByZXF1aXJlKCdnZXQtaW50cmluc2ljJyk7XG5cbnZhciAkc3BlY2llcyA9IEdldEludHJpbnNpYygnJVN5bWJvbC5zcGVjaWVzJScsIHRydWUpO1xudmFyICRUeXBlRXJyb3IgPSBHZXRJbnRyaW5zaWMoJyVUeXBlRXJyb3IlJyk7XG5cbnZhciBJc0NvbnN0cnVjdG9yID0gcmVxdWlyZSgnLi9Jc0NvbnN0cnVjdG9yJyk7XG52YXIgVHlwZSA9IHJlcXVpcmUoJy4vVHlwZScpO1xuXG4vLyBodHRwczovL2VjbWEtaW50ZXJuYXRpb25hbC5vcmcvZWNtYS0yNjIvNi4wLyNzZWMtc3BlY2llc2NvbnN0cnVjdG9yXG5cbm1vZHVsZS5leHBvcnRzID0gZnVuY3Rpb24gU3BlY2llc0NvbnN0cnVjdG9yKE8sIGRlZmF1bHRDb25zdHJ1Y3Rvcikge1xuXHRpZiAoVHlwZShPKSAhPT0gJ09iamVjdCcpIHtcblx0XHR0aHJvdyBuZXcgJFR5cGVFcnJvcignQXNzZXJ0aW9uIGZhaWxlZDogVHlwZShPKSBpcyBub3QgT2JqZWN0Jyk7XG5cdH1cblx0dmFyIEMgPSBPLmNvbnN0cnVjdG9yO1xuXHRpZiAodHlwZW9mIEMgPT09ICd1bmRlZmluZWQnKSB7XG5cdFx0cmV0dXJuIGRlZmF1bHRDb25zdHJ1Y3Rvcjtcblx0fVxuXHRpZiAoVHlwZShDKSAhPT0gJ09iamVjdCcpIHtcblx0XHR0aHJvdyBuZXcgJFR5cGVFcnJvcignTy5jb25zdHJ1Y3RvciBpcyBub3QgYW4gT2JqZWN0Jyk7XG5cdH1cblx0dmFyIFMgPSAkc3BlY2llcyA/IENbJHNwZWNpZXNdIDogdm9pZCAwO1xuXHRpZiAoUyA9PSBudWxsKSB7XG5cdFx0cmV0dXJuIGRlZmF1bHRDb25zdHJ1Y3Rvcjtcblx0fVxuXHRpZiAoSXNDb25zdHJ1Y3RvcihTKSkge1xuXHRcdHJldHVybiBTO1xuXHR9XG5cdHRocm93IG5ldyAkVHlwZUVycm9yKCdubyBjb25zdHJ1Y3RvciBmb3VuZCcpO1xufTtcbiJdLCJzb3VyY2VSb290IjoiIn0=\n//# sourceURL=webpack-internal:///303\n")},3683:module=>{"use strict";eval("\n\n// http://262.ecma-international.org/5.1/#sec-9.2\n\nmodule.exports = function ToBoolean(value) { return !!value; };\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvVG9Cb29sZWFuLmpzPzhiMTEiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQWE7O0FBRWI7O0FBRUEsNENBQTRDLGdCQUFnQiIsImZpbGUiOiIzNjgzLmpzIiwic291cmNlc0NvbnRlbnQiOlsiJ3VzZSBzdHJpY3QnO1xuXG4vLyBodHRwOi8vMjYyLmVjbWEtaW50ZXJuYXRpb25hbC5vcmcvNS4xLyNzZWMtOS4yXG5cbm1vZHVsZS5leHBvcnRzID0gZnVuY3Rpb24gVG9Cb29sZWFuKHZhbHVlKSB7IHJldHVybiAhIXZhbHVlOyB9O1xuIl0sInNvdXJjZVJvb3QiOiIifQ==\n//# sourceURL=webpack-internal:///3683\n")},5912:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar ES5ToInteger = __webpack_require__(7195);\n\nvar ToNumber = __webpack_require__(7622);\n\n// https://262.ecma-international.org/11.0/#sec-tointeger\n\nmodule.exports = function ToInteger(value) {\n\tvar number = ToNumber(value);\n\tif (number !== 0) {\n\t\tnumber = ES5ToInteger(number);\n\t}\n\treturn number === 0 ? 0 : number;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvVG9JbnRlZ2VyLmpzPzQyYmYiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQWE7O0FBRWIsbUJBQW1CLG1CQUFPLENBQUMsSUFBZ0I7O0FBRTNDLGVBQWUsbUJBQU8sQ0FBQyxJQUFZOztBQUVuQzs7QUFFQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQSIsImZpbGUiOiI1OTEyLmpzIiwic291cmNlc0NvbnRlbnQiOlsiJ3VzZSBzdHJpY3QnO1xuXG52YXIgRVM1VG9JbnRlZ2VyID0gcmVxdWlyZSgnLi4vNS9Ub0ludGVnZXInKTtcblxudmFyIFRvTnVtYmVyID0gcmVxdWlyZSgnLi9Ub051bWJlcicpO1xuXG4vLyBodHRwczovLzI2Mi5lY21hLWludGVybmF0aW9uYWwub3JnLzExLjAvI3NlYy10b2ludGVnZXJcblxubW9kdWxlLmV4cG9ydHMgPSBmdW5jdGlvbiBUb0ludGVnZXIodmFsdWUpIHtcblx0dmFyIG51bWJlciA9IFRvTnVtYmVyKHZhbHVlKTtcblx0aWYgKG51bWJlciAhPT0gMCkge1xuXHRcdG51bWJlciA9IEVTNVRvSW50ZWdlcihudW1iZXIpO1xuXHR9XG5cdHJldHVybiBudW1iZXIgPT09IDAgPyAwIDogbnVtYmVyO1xufTtcbiJdLCJzb3VyY2VSb290IjoiIn0=\n//# sourceURL=webpack-internal:///5912\n")},8502:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar MAX_SAFE_INTEGER = __webpack_require__(5376);\n\nvar ToInteger = __webpack_require__(5912);\n\nmodule.exports = function ToLength(argument) {\n\tvar len = ToInteger(argument);\n\tif (len <= 0) { return 0; } // includes converting -0 to +0\n\tif (len > MAX_SAFE_INTEGER) { return MAX_SAFE_INTEGER; }\n\treturn len;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvVG9MZW5ndGguanM/OTk5OSJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBYTs7QUFFYix1QkFBdUIsbUJBQU8sQ0FBQyxJQUEyQjs7QUFFMUQsZ0JBQWdCLG1CQUFPLENBQUMsSUFBYTs7QUFFckM7QUFDQTtBQUNBLGdCQUFnQixVQUFVLEVBQUU7QUFDNUIsOEJBQThCLHlCQUF5QjtBQUN2RDtBQUNBIiwiZmlsZSI6Ijg1MDIuanMiLCJzb3VyY2VzQ29udGVudCI6WyIndXNlIHN0cmljdCc7XG5cbnZhciBNQVhfU0FGRV9JTlRFR0VSID0gcmVxdWlyZSgnLi4vaGVscGVycy9tYXhTYWZlSW50ZWdlcicpO1xuXG52YXIgVG9JbnRlZ2VyID0gcmVxdWlyZSgnLi9Ub0ludGVnZXInKTtcblxubW9kdWxlLmV4cG9ydHMgPSBmdW5jdGlvbiBUb0xlbmd0aChhcmd1bWVudCkge1xuXHR2YXIgbGVuID0gVG9JbnRlZ2VyKGFyZ3VtZW50KTtcblx0aWYgKGxlbiA8PSAwKSB7IHJldHVybiAwOyB9IC8vIGluY2x1ZGVzIGNvbnZlcnRpbmcgLTAgdG8gKzBcblx0aWYgKGxlbiA+IE1BWF9TQUZFX0lOVEVHRVIpIHsgcmV0dXJuIE1BWF9TQUZFX0lOVEVHRVI7IH1cblx0cmV0dXJuIGxlbjtcbn07XG4iXSwic291cmNlUm9vdCI6IiJ9\n//# sourceURL=webpack-internal:///8502\n")},7622:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\n\nvar $TypeError = GetIntrinsic('%TypeError%');\nvar $Number = GetIntrinsic('%Number%');\nvar $RegExp = GetIntrinsic('%RegExp%');\nvar $parseInteger = GetIntrinsic('%parseInt%');\n\nvar callBound = __webpack_require__(1924);\nvar regexTester = __webpack_require__(1652);\nvar isPrimitive = __webpack_require__(3126);\n\nvar $strSlice = callBound('String.prototype.slice');\nvar isBinary = regexTester(/^0b[01]+$/i);\nvar isOctal = regexTester(/^0o[0-7]+$/i);\nvar isInvalidHexLiteral = regexTester(/^[-+]0x[0-9a-f]+$/i);\nvar nonWS = ['\\u0085', '\\u200b', '\\ufffe'].join('');\nvar nonWSregex = new $RegExp('[' + nonWS + ']', 'g');\nvar hasNonWS = regexTester(nonWSregex);\n\n// whitespace from: https://es5.github.io/#x15.5.4.20\n// implementation from https://github.com/es-shims/es5-shim/blob/v3.4.0/es5-shim.js#L1304-L1324\nvar ws = [\n\t'\\x09\\x0A\\x0B\\x0C\\x0D\\x20\\xA0\\u1680\\u180E\\u2000\\u2001\\u2002\\u2003',\n\t'\\u2004\\u2005\\u2006\\u2007\\u2008\\u2009\\u200A\\u202F\\u205F\\u3000\\u2028',\n\t'\\u2029\\uFEFF'\n].join('');\nvar trimRegex = new RegExp('(^[' + ws + ']+)|([' + ws + ']+$)', 'g');\nvar $replace = callBound('String.prototype.replace');\nvar $trim = function (value) {\n\treturn $replace(value, trimRegex, '');\n};\n\nvar ToPrimitive = __webpack_require__(8842);\n\n// https://ecma-international.org/ecma-262/6.0/#sec-tonumber\n\nmodule.exports = function ToNumber(argument) {\n\tvar value = isPrimitive(argument) ? argument : ToPrimitive(argument, $Number);\n\tif (typeof value === 'symbol') {\n\t\tthrow new $TypeError('Cannot convert a Symbol value to a number');\n\t}\n\tif (typeof value === 'bigint') {\n\t\tthrow new $TypeError('Conversion from \\'BigInt\\' to \\'number\\' is not allowed.');\n\t}\n\tif (typeof value === 'string') {\n\t\tif (isBinary(value)) {\n\t\t\treturn ToNumber($parseInteger($strSlice(value, 2), 2));\n\t\t} else if (isOctal(value)) {\n\t\t\treturn ToNumber($parseInteger($strSlice(value, 2), 8));\n\t\t} else if (hasNonWS(value) || isInvalidHexLiteral(value)) {\n\t\t\treturn NaN;\n\t\t}\n\t\tvar trimmed = $trim(value);\n\t\tif (trimmed !== value) {\n\t\t\treturn ToNumber(trimmed);\n\t\t}\n\n\t}\n\treturn $Number(value);\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///7622\n")},2093:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\n\nvar $Object = GetIntrinsic('%Object%');\n\nvar RequireObjectCoercible = __webpack_require__(3733);\n\n// https://ecma-international.org/ecma-262/6.0/#sec-toobject\n\nmodule.exports = function ToObject(value) {\n\tRequireObjectCoercible(value);\n\treturn $Object(value);\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvVG9PYmplY3QuanM/ZWY3MCJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBYTs7QUFFYixtQkFBbUIsbUJBQU8sQ0FBQyxHQUFlOztBQUUxQzs7QUFFQSw2QkFBNkIsbUJBQU8sQ0FBQyxJQUEwQjs7QUFFL0Q7O0FBRUE7QUFDQTtBQUNBO0FBQ0EiLCJmaWxlIjoiMjA5My5qcyIsInNvdXJjZXNDb250ZW50IjpbIid1c2Ugc3RyaWN0JztcblxudmFyIEdldEludHJpbnNpYyA9IHJlcXVpcmUoJ2dldC1pbnRyaW5zaWMnKTtcblxudmFyICRPYmplY3QgPSBHZXRJbnRyaW5zaWMoJyVPYmplY3QlJyk7XG5cbnZhciBSZXF1aXJlT2JqZWN0Q29lcmNpYmxlID0gcmVxdWlyZSgnLi9SZXF1aXJlT2JqZWN0Q29lcmNpYmxlJyk7XG5cbi8vIGh0dHBzOi8vZWNtYS1pbnRlcm5hdGlvbmFsLm9yZy9lY21hLTI2Mi82LjAvI3NlYy10b29iamVjdFxuXG5tb2R1bGUuZXhwb3J0cyA9IGZ1bmN0aW9uIFRvT2JqZWN0KHZhbHVlKSB7XG5cdFJlcXVpcmVPYmplY3RDb2VyY2libGUodmFsdWUpO1xuXHRyZXR1cm4gJE9iamVjdCh2YWx1ZSk7XG59O1xuIl0sInNvdXJjZVJvb3QiOiIifQ==\n//# sourceURL=webpack-internal:///2093\n")},8842:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar toPrimitive = __webpack_require__(1503);\n\n// https://ecma-international.org/ecma-262/6.0/#sec-toprimitive\n\nmodule.exports = function ToPrimitive(input) {\n\tif (arguments.length > 1) {\n\t\treturn toPrimitive(input, arguments[1]);\n\t}\n\treturn toPrimitive(input);\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvVG9QcmltaXRpdmUuanM/NGZmOCJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBYTs7QUFFYixrQkFBa0IsbUJBQU8sQ0FBQyxJQUF3Qjs7QUFFbEQ7O0FBRUE7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBIiwiZmlsZSI6Ijg4NDIuanMiLCJzb3VyY2VzQ29udGVudCI6WyIndXNlIHN0cmljdCc7XG5cbnZhciB0b1ByaW1pdGl2ZSA9IHJlcXVpcmUoJ2VzLXRvLXByaW1pdGl2ZS9lczIwMTUnKTtcblxuLy8gaHR0cHM6Ly9lY21hLWludGVybmF0aW9uYWwub3JnL2VjbWEtMjYyLzYuMC8jc2VjLXRvcHJpbWl0aXZlXG5cbm1vZHVsZS5leHBvcnRzID0gZnVuY3Rpb24gVG9QcmltaXRpdmUoaW5wdXQpIHtcblx0aWYgKGFyZ3VtZW50cy5sZW5ndGggPiAxKSB7XG5cdFx0cmV0dXJuIHRvUHJpbWl0aXZlKGlucHV0LCBhcmd1bWVudHNbMV0pO1xuXHR9XG5cdHJldHVybiB0b1ByaW1pdGl2ZShpbnB1dCk7XG59O1xuIl0sInNvdXJjZVJvb3QiOiIifQ==\n//# sourceURL=webpack-internal:///8842\n")},2449:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar has = __webpack_require__(7642);\n\nvar GetIntrinsic = __webpack_require__(210);\n\nvar $TypeError = GetIntrinsic('%TypeError%');\n\nvar Type = __webpack_require__(1915);\nvar ToBoolean = __webpack_require__(3683);\nvar IsCallable = __webpack_require__(590);\n\n// https://262.ecma-international.org/5.1/#sec-8.10.5\n\nmodule.exports = function ToPropertyDescriptor(Obj) {\n\tif (Type(Obj) !== 'Object') {\n\t\tthrow new $TypeError('ToPropertyDescriptor requires an object');\n\t}\n\n\tvar desc = {};\n\tif (has(Obj, 'enumerable')) {\n\t\tdesc['[[Enumerable]]'] = ToBoolean(Obj.enumerable);\n\t}\n\tif (has(Obj, 'configurable')) {\n\t\tdesc['[[Configurable]]'] = ToBoolean(Obj.configurable);\n\t}\n\tif (has(Obj, 'value')) {\n\t\tdesc['[[Value]]'] = Obj.value;\n\t}\n\tif (has(Obj, 'writable')) {\n\t\tdesc['[[Writable]]'] = ToBoolean(Obj.writable);\n\t}\n\tif (has(Obj, 'get')) {\n\t\tvar getter = Obj.get;\n\t\tif (typeof getter !== 'undefined' && !IsCallable(getter)) {\n\t\t\tthrow new $TypeError('getter must be a function');\n\t\t}\n\t\tdesc['[[Get]]'] = getter;\n\t}\n\tif (has(Obj, 'set')) {\n\t\tvar setter = Obj.set;\n\t\tif (typeof setter !== 'undefined' && !IsCallable(setter)) {\n\t\t\tthrow new $TypeError('setter must be a function');\n\t\t}\n\t\tdesc['[[Set]]'] = setter;\n\t}\n\n\tif ((has(desc, '[[Get]]') || has(desc, '[[Set]]')) && (has(desc, '[[Value]]') || has(desc, '[[Writable]]'))) {\n\t\tthrow new $TypeError('Invalid property descriptor. Cannot both specify accessors and a value or writable attribute');\n\t}\n\treturn desc;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvVG9Qcm9wZXJ0eURlc2NyaXB0b3IuanM/MjJjZSJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBYTs7QUFFYixVQUFVLG1CQUFPLENBQUMsSUFBSzs7QUFFdkIsbUJBQW1CLG1CQUFPLENBQUMsR0FBZTs7QUFFMUM7O0FBRUEsV0FBVyxtQkFBTyxDQUFDLElBQVE7QUFDM0IsZ0JBQWdCLG1CQUFPLENBQUMsSUFBYTtBQUNyQyxpQkFBaUIsbUJBQU8sQ0FBQyxHQUFjOztBQUV2Qzs7QUFFQTtBQUNBO0FBQ0E7QUFDQTs7QUFFQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7O0FBRUE7QUFDQTtBQUNBO0FBQ0E7QUFDQSIsImZpbGUiOiIyNDQ5LmpzIiwic291cmNlc0NvbnRlbnQiOlsiJ3VzZSBzdHJpY3QnO1xuXG52YXIgaGFzID0gcmVxdWlyZSgnaGFzJyk7XG5cbnZhciBHZXRJbnRyaW5zaWMgPSByZXF1aXJlKCdnZXQtaW50cmluc2ljJyk7XG5cbnZhciAkVHlwZUVycm9yID0gR2V0SW50cmluc2ljKCclVHlwZUVycm9yJScpO1xuXG52YXIgVHlwZSA9IHJlcXVpcmUoJy4vVHlwZScpO1xudmFyIFRvQm9vbGVhbiA9IHJlcXVpcmUoJy4vVG9Cb29sZWFuJyk7XG52YXIgSXNDYWxsYWJsZSA9IHJlcXVpcmUoJy4vSXNDYWxsYWJsZScpO1xuXG4vLyBodHRwczovLzI2Mi5lY21hLWludGVybmF0aW9uYWwub3JnLzUuMS8jc2VjLTguMTAuNVxuXG5tb2R1bGUuZXhwb3J0cyA9IGZ1bmN0aW9uIFRvUHJvcGVydHlEZXNjcmlwdG9yKE9iaikge1xuXHRpZiAoVHlwZShPYmopICE9PSAnT2JqZWN0Jykge1xuXHRcdHRocm93IG5ldyAkVHlwZUVycm9yKCdUb1Byb3BlcnR5RGVzY3JpcHRvciByZXF1aXJlcyBhbiBvYmplY3QnKTtcblx0fVxuXG5cdHZhciBkZXNjID0ge307XG5cdGlmIChoYXMoT2JqLCAnZW51bWVyYWJsZScpKSB7XG5cdFx0ZGVzY1snW1tFbnVtZXJhYmxlXV0nXSA9IFRvQm9vbGVhbihPYmouZW51bWVyYWJsZSk7XG5cdH1cblx0aWYgKGhhcyhPYmosICdjb25maWd1cmFibGUnKSkge1xuXHRcdGRlc2NbJ1tbQ29uZmlndXJhYmxlXV0nXSA9IFRvQm9vbGVhbihPYmouY29uZmlndXJhYmxlKTtcblx0fVxuXHRpZiAoaGFzKE9iaiwgJ3ZhbHVlJykpIHtcblx0XHRkZXNjWydbW1ZhbHVlXV0nXSA9IE9iai52YWx1ZTtcblx0fVxuXHRpZiAoaGFzKE9iaiwgJ3dyaXRhYmxlJykpIHtcblx0XHRkZXNjWydbW1dyaXRhYmxlXV0nXSA9IFRvQm9vbGVhbihPYmoud3JpdGFibGUpO1xuXHR9XG5cdGlmIChoYXMoT2JqLCAnZ2V0JykpIHtcblx0XHR2YXIgZ2V0dGVyID0gT2JqLmdldDtcblx0XHRpZiAodHlwZW9mIGdldHRlciAhPT0gJ3VuZGVmaW5lZCcgJiYgIUlzQ2FsbGFibGUoZ2V0dGVyKSkge1xuXHRcdFx0dGhyb3cgbmV3ICRUeXBlRXJyb3IoJ2dldHRlciBtdXN0IGJlIGEgZnVuY3Rpb24nKTtcblx0XHR9XG5cdFx0ZGVzY1snW1tHZXRdXSddID0gZ2V0dGVyO1xuXHR9XG5cdGlmIChoYXMoT2JqLCAnc2V0JykpIHtcblx0XHR2YXIgc2V0dGVyID0gT2JqLnNldDtcblx0XHRpZiAodHlwZW9mIHNldHRlciAhPT0gJ3VuZGVmaW5lZCcgJiYgIUlzQ2FsbGFibGUoc2V0dGVyKSkge1xuXHRcdFx0dGhyb3cgbmV3ICRUeXBlRXJyb3IoJ3NldHRlciBtdXN0IGJlIGEgZnVuY3Rpb24nKTtcblx0XHR9XG5cdFx0ZGVzY1snW1tTZXRdXSddID0gc2V0dGVyO1xuXHR9XG5cblx0aWYgKChoYXMoZGVzYywgJ1tbR2V0XV0nKSB8fCBoYXMoZGVzYywgJ1tbU2V0XV0nKSkgJiYgKGhhcyhkZXNjLCAnW1tWYWx1ZV1dJykgfHwgaGFzKGRlc2MsICdbW1dyaXRhYmxlXV0nKSkpIHtcblx0XHR0aHJvdyBuZXcgJFR5cGVFcnJvcignSW52YWxpZCBwcm9wZXJ0eSBkZXNjcmlwdG9yLiBDYW5ub3QgYm90aCBzcGVjaWZ5IGFjY2Vzc29ycyBhbmQgYSB2YWx1ZSBvciB3cml0YWJsZSBhdHRyaWJ1dGUnKTtcblx0fVxuXHRyZXR1cm4gZGVzYztcbn07XG4iXSwic291cmNlUm9vdCI6IiJ9\n//# sourceURL=webpack-internal:///2449\n")},7308:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\n\nvar $String = GetIntrinsic('%String%');\nvar $TypeError = GetIntrinsic('%TypeError%');\n\n// https://ecma-international.org/ecma-262/6.0/#sec-tostring\n\nmodule.exports = function ToString(argument) {\n\tif (typeof argument === 'symbol') {\n\t\tthrow new $TypeError('Cannot convert a Symbol value to a string');\n\t}\n\treturn $String(argument);\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvVG9TdHJpbmcuanM/ZmUxZiJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBYTs7QUFFYixtQkFBbUIsbUJBQU8sQ0FBQyxHQUFlOztBQUUxQztBQUNBOztBQUVBOztBQUVBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQSIsImZpbGUiOiI3MzA4LmpzIiwic291cmNlc0NvbnRlbnQiOlsiJ3VzZSBzdHJpY3QnO1xuXG52YXIgR2V0SW50cmluc2ljID0gcmVxdWlyZSgnZ2V0LWludHJpbnNpYycpO1xuXG52YXIgJFN0cmluZyA9IEdldEludHJpbnNpYygnJVN0cmluZyUnKTtcbnZhciAkVHlwZUVycm9yID0gR2V0SW50cmluc2ljKCclVHlwZUVycm9yJScpO1xuXG4vLyBodHRwczovL2VjbWEtaW50ZXJuYXRpb25hbC5vcmcvZWNtYS0yNjIvNi4wLyNzZWMtdG9zdHJpbmdcblxubW9kdWxlLmV4cG9ydHMgPSBmdW5jdGlvbiBUb1N0cmluZyhhcmd1bWVudCkge1xuXHRpZiAodHlwZW9mIGFyZ3VtZW50ID09PSAnc3ltYm9sJykge1xuXHRcdHRocm93IG5ldyAkVHlwZUVycm9yKCdDYW5ub3QgY29udmVydCBhIFN5bWJvbCB2YWx1ZSB0byBhIHN0cmluZycpO1xuXHR9XG5cdHJldHVybiAkU3RyaW5nKGFyZ3VtZW50KTtcbn07XG4iXSwic291cmNlUm9vdCI6IiJ9\n//# sourceURL=webpack-internal:///7308\n")},1915:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar ES5Type = __webpack_require__(1276);\n\n// https://262.ecma-international.org/11.0/#sec-ecmascript-data-types-and-values\n\nmodule.exports = function Type(x) {\n\tif (typeof x === 'symbol') {\n\t\treturn 'Symbol';\n\t}\n\tif (typeof x === 'bigint') {\n\t\treturn 'BigInt';\n\t}\n\treturn ES5Type(x);\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvVHlwZS5qcz82NzYwIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFhOztBQUViLGNBQWMsbUJBQU8sQ0FBQyxJQUFXOztBQUVqQzs7QUFFQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0EiLCJmaWxlIjoiMTkxNS5qcyIsInNvdXJjZXNDb250ZW50IjpbIid1c2Ugc3RyaWN0JztcblxudmFyIEVTNVR5cGUgPSByZXF1aXJlKCcuLi81L1R5cGUnKTtcblxuLy8gaHR0cHM6Ly8yNjIuZWNtYS1pbnRlcm5hdGlvbmFsLm9yZy8xMS4wLyNzZWMtZWNtYXNjcmlwdC1kYXRhLXR5cGVzLWFuZC12YWx1ZXNcblxubW9kdWxlLmV4cG9ydHMgPSBmdW5jdGlvbiBUeXBlKHgpIHtcblx0aWYgKHR5cGVvZiB4ID09PSAnc3ltYm9sJykge1xuXHRcdHJldHVybiAnU3ltYm9sJztcblx0fVxuXHRpZiAodHlwZW9mIHggPT09ICdiaWdpbnQnKSB7XG5cdFx0cmV0dXJuICdCaWdJbnQnO1xuXHR9XG5cdHJldHVybiBFUzVUeXBlKHgpO1xufTtcbiJdLCJzb3VyY2VSb290IjoiIn0=\n//# sourceURL=webpack-internal:///1915\n")},4969:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\n\nvar $TypeError = GetIntrinsic('%TypeError%');\nvar $fromCharCode = GetIntrinsic('%String.fromCharCode%');\n\nvar isLeadingSurrogate = __webpack_require__(1996);\nvar isTrailingSurrogate = __webpack_require__(6935);\n\n// https://262.ecma-international.org/11.0/#sec-utf16decodesurrogatepair\n\nmodule.exports = function UTF16DecodeSurrogatePair(lead, trail) {\n\tif (!isLeadingSurrogate(lead) || !isTrailingSurrogate(trail)) {\n\t\tthrow new $TypeError('Assertion failed: `lead` must be a leading surrogate char code, and `trail` must be a trailing surrogate char code');\n\t}\n\t// var cp = (lead - 0xD800) * 0x400 + (trail - 0xDC00) + 0x10000;\n\treturn $fromCharCode(lead) + $fromCharCode(trail);\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvVVRGMTZEZWNvZGVTdXJyb2dhdGVQYWlyLmpzP2E0MzIiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQWE7O0FBRWIsbUJBQW1CLG1CQUFPLENBQUMsR0FBZTs7QUFFMUM7QUFDQTs7QUFFQSx5QkFBeUIsbUJBQU8sQ0FBQyxJQUErQjtBQUNoRSwwQkFBMEIsbUJBQU8sQ0FBQyxJQUFnQzs7QUFFbEU7O0FBRUE7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0EiLCJmaWxlIjoiNDk2OS5qcyIsInNvdXJjZXNDb250ZW50IjpbIid1c2Ugc3RyaWN0JztcblxudmFyIEdldEludHJpbnNpYyA9IHJlcXVpcmUoJ2dldC1pbnRyaW5zaWMnKTtcblxudmFyICRUeXBlRXJyb3IgPSBHZXRJbnRyaW5zaWMoJyVUeXBlRXJyb3IlJyk7XG52YXIgJGZyb21DaGFyQ29kZSA9IEdldEludHJpbnNpYygnJVN0cmluZy5mcm9tQ2hhckNvZGUlJyk7XG5cbnZhciBpc0xlYWRpbmdTdXJyb2dhdGUgPSByZXF1aXJlKCcuLi9oZWxwZXJzL2lzTGVhZGluZ1N1cnJvZ2F0ZScpO1xudmFyIGlzVHJhaWxpbmdTdXJyb2dhdGUgPSByZXF1aXJlKCcuLi9oZWxwZXJzL2lzVHJhaWxpbmdTdXJyb2dhdGUnKTtcblxuLy8gaHR0cHM6Ly8yNjIuZWNtYS1pbnRlcm5hdGlvbmFsLm9yZy8xMS4wLyNzZWMtdXRmMTZkZWNvZGVzdXJyb2dhdGVwYWlyXG5cbm1vZHVsZS5leHBvcnRzID0gZnVuY3Rpb24gVVRGMTZEZWNvZGVTdXJyb2dhdGVQYWlyKGxlYWQsIHRyYWlsKSB7XG5cdGlmICghaXNMZWFkaW5nU3Vycm9nYXRlKGxlYWQpIHx8ICFpc1RyYWlsaW5nU3Vycm9nYXRlKHRyYWlsKSkge1xuXHRcdHRocm93IG5ldyAkVHlwZUVycm9yKCdBc3NlcnRpb24gZmFpbGVkOiBgbGVhZGAgbXVzdCBiZSBhIGxlYWRpbmcgc3Vycm9nYXRlIGNoYXIgY29kZSwgYW5kIGB0cmFpbGAgbXVzdCBiZSBhIHRyYWlsaW5nIHN1cnJvZ2F0ZSBjaGFyIGNvZGUnKTtcblx0fVxuXHQvLyB2YXIgY3AgPSAobGVhZCAtIDB4RDgwMCkgKiAweDQwMCArICh0cmFpbCAtIDB4REMwMCkgKyAweDEwMDAwO1xuXHRyZXR1cm4gJGZyb21DaGFyQ29kZShsZWFkKSArICRmcm9tQ2hhckNvZGUodHJhaWwpO1xufTtcbiJdLCJzb3VyY2VSb290IjoiIn0=\n//# sourceURL=webpack-internal:///4969\n")},1717:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\n\nvar $abs = GetIntrinsic('%Math.abs%');\n\n// http://262.ecma-international.org/5.1/#sec-5.2\n\nmodule.exports = function abs(x) {\n\treturn $abs(x);\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvYWJzLmpzPzQ1NjAiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQWE7O0FBRWIsbUJBQW1CLG1CQUFPLENBQUMsR0FBZTs7QUFFMUM7O0FBRUE7O0FBRUE7QUFDQTtBQUNBIiwiZmlsZSI6IjE3MTcuanMiLCJzb3VyY2VzQ29udGVudCI6WyIndXNlIHN0cmljdCc7XG5cbnZhciBHZXRJbnRyaW5zaWMgPSByZXF1aXJlKCdnZXQtaW50cmluc2ljJyk7XG5cbnZhciAkYWJzID0gR2V0SW50cmluc2ljKCclTWF0aC5hYnMlJyk7XG5cbi8vIGh0dHA6Ly8yNjIuZWNtYS1pbnRlcm5hdGlvbmFsLm9yZy81LjEvI3NlYy01LjJcblxubW9kdWxlLmV4cG9ydHMgPSBmdW5jdGlvbiBhYnMoeCkge1xuXHRyZXR1cm4gJGFicyh4KTtcbn07XG4iXSwic291cmNlUm9vdCI6IiJ9\n//# sourceURL=webpack-internal:///1717\n")},9202:module=>{"use strict";eval("\n\n// var modulo = require('./modulo');\nvar $floor = Math.floor;\n\n// http://262.ecma-international.org/5.1/#sec-5.2\n\nmodule.exports = function floor(x) {\n\t// return x - modulo(x, 1);\n\treturn $floor(x);\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvZmxvb3IuanM/MTVkNSJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBYTs7QUFFYjtBQUNBOztBQUVBOztBQUVBO0FBQ0E7QUFDQTtBQUNBIiwiZmlsZSI6IjkyMDIuanMiLCJzb3VyY2VzQ29udGVudCI6WyIndXNlIHN0cmljdCc7XG5cbi8vIHZhciBtb2R1bG8gPSByZXF1aXJlKCcuL21vZHVsbycpO1xudmFyICRmbG9vciA9IE1hdGguZmxvb3I7XG5cbi8vIGh0dHA6Ly8yNjIuZWNtYS1pbnRlcm5hdGlvbmFsLm9yZy81LjEvI3NlYy01LjJcblxubW9kdWxlLmV4cG9ydHMgPSBmdW5jdGlvbiBmbG9vcih4KSB7XG5cdC8vIHJldHVybiB4IC0gbW9kdWxvKHgsIDEpO1xuXHRyZXR1cm4gJGZsb29yKHgpO1xufTtcbiJdLCJzb3VyY2VSb290IjoiIn0=\n//# sourceURL=webpack-internal:///9202\n")},8631:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\n\nvar $TypeError = GetIntrinsic('%TypeError%');\n\n// http://262.ecma-international.org/5.1/#sec-9.10\n\nmodule.exports = function CheckObjectCoercible(value, optMessage) {\n\tif (value == null) {\n\t\tthrow new $TypeError(optMessage || ('Cannot call method on ' + value));\n\t}\n\treturn value;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzUvQ2hlY2tPYmplY3RDb2VyY2libGUuanM/NGQ1MiJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBYTs7QUFFYixtQkFBbUIsbUJBQU8sQ0FBQyxHQUFlOztBQUUxQzs7QUFFQTs7QUFFQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0EiLCJmaWxlIjoiODYzMS5qcyIsInNvdXJjZXNDb250ZW50IjpbIid1c2Ugc3RyaWN0JztcblxudmFyIEdldEludHJpbnNpYyA9IHJlcXVpcmUoJ2dldC1pbnRyaW5zaWMnKTtcblxudmFyICRUeXBlRXJyb3IgPSBHZXRJbnRyaW5zaWMoJyVUeXBlRXJyb3IlJyk7XG5cbi8vIGh0dHA6Ly8yNjIuZWNtYS1pbnRlcm5hdGlvbmFsLm9yZy81LjEvI3NlYy05LjEwXG5cbm1vZHVsZS5leHBvcnRzID0gZnVuY3Rpb24gQ2hlY2tPYmplY3RDb2VyY2libGUodmFsdWUsIG9wdE1lc3NhZ2UpIHtcblx0aWYgKHZhbHVlID09IG51bGwpIHtcblx0XHR0aHJvdyBuZXcgJFR5cGVFcnJvcihvcHRNZXNzYWdlIHx8ICgnQ2Fubm90IGNhbGwgbWV0aG9kIG9uICcgKyB2YWx1ZSkpO1xuXHR9XG5cdHJldHVybiB2YWx1ZTtcbn07XG4iXSwic291cmNlUm9vdCI6IiJ9\n//# sourceURL=webpack-internal:///8631\n")},7195:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar abs = __webpack_require__(2683);\nvar floor = __webpack_require__(9711);\nvar ToNumber = __webpack_require__(7196);\n\nvar $isNaN = __webpack_require__(1214);\nvar $isFinite = __webpack_require__(3060);\nvar $sign = __webpack_require__(4099);\n\n// http://262.ecma-international.org/5.1/#sec-9.4\n\nmodule.exports = function ToInteger(value) {\n\tvar number = ToNumber(value);\n\tif ($isNaN(number)) { return 0; }\n\tif (number === 0 || !$isFinite(number)) { return number; }\n\treturn $sign(number) * floor(abs(number));\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzUvVG9JbnRlZ2VyLmpzP2QzMjciXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQWE7O0FBRWIsVUFBVSxtQkFBTyxDQUFDLElBQU87QUFDekIsWUFBWSxtQkFBTyxDQUFDLElBQVM7QUFDN0IsZUFBZSxtQkFBTyxDQUFDLElBQVk7O0FBRW5DLGFBQWEsbUJBQU8sQ0FBQyxJQUFrQjtBQUN2QyxnQkFBZ0IsbUJBQU8sQ0FBQyxJQUFxQjtBQUM3QyxZQUFZLG1CQUFPLENBQUMsSUFBaUI7O0FBRXJDOztBQUVBO0FBQ0E7QUFDQSxzQkFBc0IsVUFBVTtBQUNoQywwQ0FBMEMsZUFBZTtBQUN6RDtBQUNBIiwiZmlsZSI6IjcxOTUuanMiLCJzb3VyY2VzQ29udGVudCI6WyIndXNlIHN0cmljdCc7XG5cbnZhciBhYnMgPSByZXF1aXJlKCcuL2FicycpO1xudmFyIGZsb29yID0gcmVxdWlyZSgnLi9mbG9vcicpO1xudmFyIFRvTnVtYmVyID0gcmVxdWlyZSgnLi9Ub051bWJlcicpO1xuXG52YXIgJGlzTmFOID0gcmVxdWlyZSgnLi4vaGVscGVycy9pc05hTicpO1xudmFyICRpc0Zpbml0ZSA9IHJlcXVpcmUoJy4uL2hlbHBlcnMvaXNGaW5pdGUnKTtcbnZhciAkc2lnbiA9IHJlcXVpcmUoJy4uL2hlbHBlcnMvc2lnbicpO1xuXG4vLyBodHRwOi8vMjYyLmVjbWEtaW50ZXJuYXRpb25hbC5vcmcvNS4xLyNzZWMtOS40XG5cbm1vZHVsZS5leHBvcnRzID0gZnVuY3Rpb24gVG9JbnRlZ2VyKHZhbHVlKSB7XG5cdHZhciBudW1iZXIgPSBUb051bWJlcih2YWx1ZSk7XG5cdGlmICgkaXNOYU4obnVtYmVyKSkgeyByZXR1cm4gMDsgfVxuXHRpZiAobnVtYmVyID09PSAwIHx8ICEkaXNGaW5pdGUobnVtYmVyKSkgeyByZXR1cm4gbnVtYmVyOyB9XG5cdHJldHVybiAkc2lnbihudW1iZXIpICogZmxvb3IoYWJzKG51bWJlcikpO1xufTtcbiJdLCJzb3VyY2VSb290IjoiIn0=\n//# sourceURL=webpack-internal:///7195\n")},7196:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar ToPrimitive = __webpack_require__(1318);\n\n// http://262.ecma-international.org/5.1/#sec-9.3\n\nmodule.exports = function ToNumber(value) {\n\tvar prim = ToPrimitive(value, Number);\n\tif (typeof prim !== 'string') {\n\t\treturn +prim; // eslint-disable-line no-implicit-coercion\n\t}\n\n\t// eslint-disable-next-line no-control-regex\n\tvar trimmed = prim.replace(/^[ \\t\\x0b\\f\\xa0\\ufeff\\n\\r\\u2028\\u2029\\u1680\\u180e\\u2000\\u2001\\u2002\\u2003\\u2004\\u2005\\u2006\\u2007\\u2008\\u2009\\u200a\\u202f\\u205f\\u3000\\u0085]+|[ \\t\\x0b\\f\\xa0\\ufeff\\n\\r\\u2028\\u2029\\u1680\\u180e\\u2000\\u2001\\u2002\\u2003\\u2004\\u2005\\u2006\\u2007\\u2008\\u2009\\u200a\\u202f\\u205f\\u3000\\u0085]+$/g, '');\n\tif ((/^0[ob]|^[+-]0x/).test(trimmed)) {\n\t\treturn NaN;\n\t}\n\n\treturn +trimmed; // eslint-disable-line no-implicit-coercion\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzUvVG9OdW1iZXIuanM/YmI1YyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBYTs7QUFFYixrQkFBa0IsbUJBQU8sQ0FBQyxJQUFlOztBQUV6Qzs7QUFFQTtBQUNBO0FBQ0E7QUFDQSxlQUFlO0FBQ2Y7O0FBRUE7QUFDQTtBQUNBO0FBQ0E7QUFDQTs7QUFFQSxpQkFBaUI7QUFDakIiLCJmaWxlIjoiNzE5Ni5qcyIsInNvdXJjZXNDb250ZW50IjpbIid1c2Ugc3RyaWN0JztcblxudmFyIFRvUHJpbWl0aXZlID0gcmVxdWlyZSgnLi9Ub1ByaW1pdGl2ZScpO1xuXG4vLyBodHRwOi8vMjYyLmVjbWEtaW50ZXJuYXRpb25hbC5vcmcvNS4xLyNzZWMtOS4zXG5cbm1vZHVsZS5leHBvcnRzID0gZnVuY3Rpb24gVG9OdW1iZXIodmFsdWUpIHtcblx0dmFyIHByaW0gPSBUb1ByaW1pdGl2ZSh2YWx1ZSwgTnVtYmVyKTtcblx0aWYgKHR5cGVvZiBwcmltICE9PSAnc3RyaW5nJykge1xuXHRcdHJldHVybiArcHJpbTsgLy8gZXNsaW50LWRpc2FibGUtbGluZSBuby1pbXBsaWNpdC1jb2VyY2lvblxuXHR9XG5cblx0Ly8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIG5vLWNvbnRyb2wtcmVnZXhcblx0dmFyIHRyaW1tZWQgPSBwcmltLnJlcGxhY2UoL15bIFxcdFxceDBiXFxmXFx4YTBcXHVmZWZmXFxuXFxyXFx1MjAyOFxcdTIwMjlcXHUxNjgwXFx1MTgwZVxcdTIwMDBcXHUyMDAxXFx1MjAwMlxcdTIwMDNcXHUyMDA0XFx1MjAwNVxcdTIwMDZcXHUyMDA3XFx1MjAwOFxcdTIwMDlcXHUyMDBhXFx1MjAyZlxcdTIwNWZcXHUzMDAwXFx1MDA4NV0rfFsgXFx0XFx4MGJcXGZcXHhhMFxcdWZlZmZcXG5cXHJcXHUyMDI4XFx1MjAyOVxcdTE2ODBcXHUxODBlXFx1MjAwMFxcdTIwMDFcXHUyMDAyXFx1MjAwM1xcdTIwMDRcXHUyMDA1XFx1MjAwNlxcdTIwMDdcXHUyMDA4XFx1MjAwOVxcdTIwMGFcXHUyMDJmXFx1MjA1ZlxcdTMwMDBcXHUwMDg1XSskL2csICcnKTtcblx0aWYgKCgvXjBbb2JdfF5bKy1dMHgvKS50ZXN0KHRyaW1tZWQpKSB7XG5cdFx0cmV0dXJuIE5hTjtcblx0fVxuXG5cdHJldHVybiArdHJpbW1lZDsgLy8gZXNsaW50LWRpc2FibGUtbGluZSBuby1pbXBsaWNpdC1jb2VyY2lvblxufTtcbiJdLCJzb3VyY2VSb290IjoiIn0=\n//# sourceURL=webpack-internal:///7196\n")},1318:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\n// http://262.ecma-international.org/5.1/#sec-9.1\n\nmodule.exports = __webpack_require__(2116);\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzUvVG9QcmltaXRpdmUuanM/MDExYyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBYTs7QUFFYjs7QUFFQSwwQ0FBK0MiLCJmaWxlIjoiMTMxOC5qcyIsInNvdXJjZXNDb250ZW50IjpbIid1c2Ugc3RyaWN0JztcblxuLy8gaHR0cDovLzI2Mi5lY21hLWludGVybmF0aW9uYWwub3JnLzUuMS8jc2VjLTkuMVxuXG5tb2R1bGUuZXhwb3J0cyA9IHJlcXVpcmUoJ2VzLXRvLXByaW1pdGl2ZS9lczUnKTtcbiJdLCJzb3VyY2VSb290IjoiIn0=\n//# sourceURL=webpack-internal:///1318\n")},1276:module=>{"use strict";eval("\n\n// https://262.ecma-international.org/5.1/#sec-8\n\nmodule.exports = function Type(x) {\n\tif (x === null) {\n\t\treturn 'Null';\n\t}\n\tif (typeof x === 'undefined') {\n\t\treturn 'Undefined';\n\t}\n\tif (typeof x === 'function' || typeof x === 'object') {\n\t\treturn 'Object';\n\t}\n\tif (typeof x === 'number') {\n\t\treturn 'Number';\n\t}\n\tif (typeof x === 'boolean') {\n\t\treturn 'Boolean';\n\t}\n\tif (typeof x === 'string') {\n\t\treturn 'String';\n\t}\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzUvVHlwZS5qcz81MTgzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFhOztBQUViOztBQUVBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0EiLCJmaWxlIjoiMTI3Ni5qcyIsInNvdXJjZXNDb250ZW50IjpbIid1c2Ugc3RyaWN0JztcblxuLy8gaHR0cHM6Ly8yNjIuZWNtYS1pbnRlcm5hdGlvbmFsLm9yZy81LjEvI3NlYy04XG5cbm1vZHVsZS5leHBvcnRzID0gZnVuY3Rpb24gVHlwZSh4KSB7XG5cdGlmICh4ID09PSBudWxsKSB7XG5cdFx0cmV0dXJuICdOdWxsJztcblx0fVxuXHRpZiAodHlwZW9mIHggPT09ICd1bmRlZmluZWQnKSB7XG5cdFx0cmV0dXJuICdVbmRlZmluZWQnO1xuXHR9XG5cdGlmICh0eXBlb2YgeCA9PT0gJ2Z1bmN0aW9uJyB8fCB0eXBlb2YgeCA9PT0gJ29iamVjdCcpIHtcblx0XHRyZXR1cm4gJ09iamVjdCc7XG5cdH1cblx0aWYgKHR5cGVvZiB4ID09PSAnbnVtYmVyJykge1xuXHRcdHJldHVybiAnTnVtYmVyJztcblx0fVxuXHRpZiAodHlwZW9mIHggPT09ICdib29sZWFuJykge1xuXHRcdHJldHVybiAnQm9vbGVhbic7XG5cdH1cblx0aWYgKHR5cGVvZiB4ID09PSAnc3RyaW5nJykge1xuXHRcdHJldHVybiAnU3RyaW5nJztcblx0fVxufTtcbiJdLCJzb3VyY2VSb290IjoiIn0=\n//# sourceURL=webpack-internal:///1276\n")},2683:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\n\nvar $abs = GetIntrinsic('%Math.abs%');\n\n// http://262.ecma-international.org/5.1/#sec-5.2\n\nmodule.exports = function abs(x) {\n\treturn $abs(x);\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzUvYWJzLmpzP2JkMTAiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQWE7O0FBRWIsbUJBQW1CLG1CQUFPLENBQUMsR0FBZTs7QUFFMUM7O0FBRUE7O0FBRUE7QUFDQTtBQUNBIiwiZmlsZSI6IjI2ODMuanMiLCJzb3VyY2VzQ29udGVudCI6WyIndXNlIHN0cmljdCc7XG5cbnZhciBHZXRJbnRyaW5zaWMgPSByZXF1aXJlKCdnZXQtaW50cmluc2ljJyk7XG5cbnZhciAkYWJzID0gR2V0SW50cmluc2ljKCclTWF0aC5hYnMlJyk7XG5cbi8vIGh0dHA6Ly8yNjIuZWNtYS1pbnRlcm5hdGlvbmFsLm9yZy81LjEvI3NlYy01LjJcblxubW9kdWxlLmV4cG9ydHMgPSBmdW5jdGlvbiBhYnMoeCkge1xuXHRyZXR1cm4gJGFicyh4KTtcbn07XG4iXSwic291cmNlUm9vdCI6IiJ9\n//# sourceURL=webpack-internal:///2683\n")},9711:module=>{"use strict";eval("\n\n// var modulo = require('./modulo');\nvar $floor = Math.floor;\n\n// http://262.ecma-international.org/5.1/#sec-5.2\n\nmodule.exports = function floor(x) {\n\t// return x - modulo(x, 1);\n\treturn $floor(x);\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzUvZmxvb3IuanM/YjVhNiJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBYTs7QUFFYjtBQUNBOztBQUVBOztBQUVBO0FBQ0E7QUFDQTtBQUNBIiwiZmlsZSI6Ijk3MTEuanMiLCJzb3VyY2VzQ29udGVudCI6WyIndXNlIHN0cmljdCc7XG5cbi8vIHZhciBtb2R1bG8gPSByZXF1aXJlKCcuL21vZHVsbycpO1xudmFyICRmbG9vciA9IE1hdGguZmxvb3I7XG5cbi8vIGh0dHA6Ly8yNjIuZWNtYS1pbnRlcm5hdGlvbmFsLm9yZy81LjEvI3NlYy01LjJcblxubW9kdWxlLmV4cG9ydHMgPSBmdW5jdGlvbiBmbG9vcih4KSB7XG5cdC8vIHJldHVybiB4IC0gbW9kdWxvKHgsIDEpO1xuXHRyZXR1cm4gJGZsb29yKHgpO1xufTtcbiJdLCJzb3VyY2VSb290IjoiIn0=\n//# sourceURL=webpack-internal:///9711\n")},8769:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\n// TODO: remove, semver-major\n\nmodule.exports = __webpack_require__(210);\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0L0dldEludHJpbnNpYy5qcz9lOWFjIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFhOztBQUViOztBQUVBLHlDQUF5QyIsImZpbGUiOiI4NzY5LmpzIiwic291cmNlc0NvbnRlbnQiOlsiJ3VzZSBzdHJpY3QnO1xuXG4vLyBUT0RPOiByZW1vdmUsIHNlbXZlci1tYWpvclxuXG5tb2R1bGUuZXhwb3J0cyA9IHJlcXVpcmUoJ2dldC1pbnRyaW5zaWMnKTtcbiJdLCJzb3VyY2VSb290IjoiIn0=\n//# sourceURL=webpack-internal:///8769\n")},509:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\n\nvar $defineProperty = GetIntrinsic('%Object.defineProperty%', true);\n\nif ($defineProperty) {\n\ttry {\n\t\t$defineProperty({}, 'a', { value: 1 });\n\t} catch (e) {\n\t\t// IE 8 has a broken defineProperty\n\t\t$defineProperty = null;\n\t}\n}\n\nvar callBound = __webpack_require__(1924);\n\nvar $isEnumerable = callBound('Object.prototype.propertyIsEnumerable');\n\n// eslint-disable-next-line max-params\nmodule.exports = function DefineOwnProperty(IsDataDescriptor, SameValue, FromPropertyDescriptor, O, P, desc) {\n\tif (!$defineProperty) {\n\t\tif (!IsDataDescriptor(desc)) {\n\t\t\t// ES3 does not support getters/setters\n\t\t\treturn false;\n\t\t}\n\t\tif (!desc['[[Configurable]]'] || !desc['[[Writable]]']) {\n\t\t\treturn false;\n\t\t}\n\n\t\t// fallback for ES3\n\t\tif (P in O && $isEnumerable(O, P) !== !!desc['[[Enumerable]]']) {\n\t\t\t// a non-enumerable existing property\n\t\t\treturn false;\n\t\t}\n\n\t\t// property does not exist at all, or exists but is enumerable\n\t\tvar V = desc['[[Value]]'];\n\t\t// eslint-disable-next-line no-param-reassign\n\t\tO[P] = V; // will use [[Define]]\n\t\treturn SameValue(O[P], V);\n\t}\n\t$defineProperty(O, P, FromPropertyDescriptor(desc));\n\treturn true;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///509\n")},4179:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\n\nvar $TypeError = GetIntrinsic('%TypeError%');\nvar $SyntaxError = GetIntrinsic('%SyntaxError%');\n\nvar has = __webpack_require__(7642);\n\nvar predicates = {\n\t// https://262.ecma-international.org/6.0/#sec-property-descriptor-specification-type\n\t'Property Descriptor': function isPropertyDescriptor(Type, Desc) {\n\t\tif (Type(Desc) !== 'Object') {\n\t\t\treturn false;\n\t\t}\n\t\tvar allowed = {\n\t\t\t'[[Configurable]]': true,\n\t\t\t'[[Enumerable]]': true,\n\t\t\t'[[Get]]': true,\n\t\t\t'[[Set]]': true,\n\t\t\t'[[Value]]': true,\n\t\t\t'[[Writable]]': true\n\t\t};\n\n\t\tfor (var key in Desc) { // eslint-disable-line\n\t\t\tif (has(Desc, key) && !allowed[key]) {\n\t\t\t\treturn false;\n\t\t\t}\n\t\t}\n\n\t\tvar isData = has(Desc, '[[Value]]');\n\t\tvar IsAccessor = has(Desc, '[[Get]]') || has(Desc, '[[Set]]');\n\t\tif (isData && IsAccessor) {\n\t\t\tthrow new $TypeError('Property Descriptors may not be both accessor and data descriptors');\n\t\t}\n\t\treturn true;\n\t}\n};\n\nmodule.exports = function assertRecord(Type, recordType, argumentName, value) {\n\tvar predicate = predicates[recordType];\n\tif (typeof predicate !== 'function') {\n\t\tthrow new $SyntaxError('unknown record type: ' + recordType);\n\t}\n\tif (!predicate(Type, value)) {\n\t\tthrow new $TypeError(argumentName + ' must be a ' + recordType);\n\t}\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///4179\n")},3060:module=>{"use strict";eval("\n\nvar $isNaN = Number.isNaN || function (a) { return a !== a; };\n\nmodule.exports = Number.isFinite || function (x) { return typeof x === 'number' && !$isNaN(x) && x !== Infinity && x !== -Infinity; };\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0L2hlbHBlcnMvaXNGaW5pdGUuanM/YzYxMiJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBYTs7QUFFYiwyQ0FBMkMsZ0JBQWdCOztBQUUzRCxrREFBa0QsaUZBQWlGIiwiZmlsZSI6IjMwNjAuanMiLCJzb3VyY2VzQ29udGVudCI6WyIndXNlIHN0cmljdCc7XG5cbnZhciAkaXNOYU4gPSBOdW1iZXIuaXNOYU4gfHwgZnVuY3Rpb24gKGEpIHsgcmV0dXJuIGEgIT09IGE7IH07XG5cbm1vZHVsZS5leHBvcnRzID0gTnVtYmVyLmlzRmluaXRlIHx8IGZ1bmN0aW9uICh4KSB7IHJldHVybiB0eXBlb2YgeCA9PT0gJ251bWJlcicgJiYgISRpc05hTih4KSAmJiB4ICE9PSBJbmZpbml0eSAmJiB4ICE9PSAtSW5maW5pdHk7IH07XG4iXSwic291cmNlUm9vdCI6IiJ9\n//# sourceURL=webpack-internal:///3060\n")},1996:module=>{"use strict";eval("\n\nmodule.exports = function isLeadingSurrogate(charCode) {\n\treturn typeof charCode === 'number' && charCode >= 0xD800 && charCode <= 0xDBFF;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0L2hlbHBlcnMvaXNMZWFkaW5nU3Vycm9nYXRlLmpzPzYzYmUiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQWE7O0FBRWI7QUFDQTtBQUNBIiwiZmlsZSI6IjE5OTYuanMiLCJzb3VyY2VzQ29udGVudCI6WyIndXNlIHN0cmljdCc7XG5cbm1vZHVsZS5leHBvcnRzID0gZnVuY3Rpb24gaXNMZWFkaW5nU3Vycm9nYXRlKGNoYXJDb2RlKSB7XG5cdHJldHVybiB0eXBlb2YgY2hhckNvZGUgPT09ICdudW1iZXInICYmIGNoYXJDb2RlID49IDB4RDgwMCAmJiBjaGFyQ29kZSA8PSAweERCRkY7XG59O1xuIl0sInNvdXJjZVJvb3QiOiIifQ==\n//# sourceURL=webpack-internal:///1996\n")},1214:module=>{"use strict";eval("\n\nmodule.exports = Number.isNaN || function isNaN(a) {\n\treturn a !== a;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0L2hlbHBlcnMvaXNOYU4uanM/MjA1NyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBYTs7QUFFYjtBQUNBO0FBQ0EiLCJmaWxlIjoiMTIxNC5qcyIsInNvdXJjZXNDb250ZW50IjpbIid1c2Ugc3RyaWN0JztcblxubW9kdWxlLmV4cG9ydHMgPSBOdW1iZXIuaXNOYU4gfHwgZnVuY3Rpb24gaXNOYU4oYSkge1xuXHRyZXR1cm4gYSAhPT0gYTtcbn07XG4iXSwic291cmNlUm9vdCI6IiJ9\n//# sourceURL=webpack-internal:///1214\n")},3126:module=>{"use strict";eval("\n\nmodule.exports = function isPrimitive(value) {\n\treturn value === null || (typeof value !== 'function' && typeof value !== 'object');\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0L2hlbHBlcnMvaXNQcmltaXRpdmUuanM/MTM1OCJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBYTs7QUFFYjtBQUNBO0FBQ0EiLCJmaWxlIjoiMzEyNi5qcyIsInNvdXJjZXNDb250ZW50IjpbIid1c2Ugc3RyaWN0JztcblxubW9kdWxlLmV4cG9ydHMgPSBmdW5jdGlvbiBpc1ByaW1pdGl2ZSh2YWx1ZSkge1xuXHRyZXR1cm4gdmFsdWUgPT09IG51bGwgfHwgKHR5cGVvZiB2YWx1ZSAhPT0gJ2Z1bmN0aW9uJyAmJiB0eXBlb2YgdmFsdWUgIT09ICdvYmplY3QnKTtcbn07XG4iXSwic291cmNlUm9vdCI6IiJ9\n//# sourceURL=webpack-internal:///3126\n")},9146:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\n\nvar has = __webpack_require__(7642);\nvar $TypeError = GetIntrinsic('%TypeError%');\n\nmodule.exports = function IsPropertyDescriptor(ES, Desc) {\n\tif (ES.Type(Desc) !== 'Object') {\n\t\treturn false;\n\t}\n\tvar allowed = {\n\t\t'[[Configurable]]': true,\n\t\t'[[Enumerable]]': true,\n\t\t'[[Get]]': true,\n\t\t'[[Set]]': true,\n\t\t'[[Value]]': true,\n\t\t'[[Writable]]': true\n\t};\n\n\tfor (var key in Desc) { // eslint-disable-line no-restricted-syntax\n\t\tif (has(Desc, key) && !allowed[key]) {\n\t\t\treturn false;\n\t\t}\n\t}\n\n\tif (ES.IsDataDescriptor(Desc) && ES.IsAccessorDescriptor(Desc)) {\n\t\tthrow new $TypeError('Property Descriptors may not be both accessor and data descriptors');\n\t}\n\treturn true;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0L2hlbHBlcnMvaXNQcm9wZXJ0eURlc2NyaXB0b3IuanM/ZmZmZCJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBYTs7QUFFYixtQkFBbUIsbUJBQU8sQ0FBQyxHQUFlOztBQUUxQyxVQUFVLG1CQUFPLENBQUMsSUFBSztBQUN2Qjs7QUFFQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7O0FBRUEsd0JBQXdCO0FBQ3hCO0FBQ0E7QUFDQTtBQUNBOztBQUVBO0FBQ0E7QUFDQTtBQUNBO0FBQ0EiLCJmaWxlIjoiOTE0Ni5qcyIsInNvdXJjZXNDb250ZW50IjpbIid1c2Ugc3RyaWN0JztcblxudmFyIEdldEludHJpbnNpYyA9IHJlcXVpcmUoJ2dldC1pbnRyaW5zaWMnKTtcblxudmFyIGhhcyA9IHJlcXVpcmUoJ2hhcycpO1xudmFyICRUeXBlRXJyb3IgPSBHZXRJbnRyaW5zaWMoJyVUeXBlRXJyb3IlJyk7XG5cbm1vZHVsZS5leHBvcnRzID0gZnVuY3Rpb24gSXNQcm9wZXJ0eURlc2NyaXB0b3IoRVMsIERlc2MpIHtcblx0aWYgKEVTLlR5cGUoRGVzYykgIT09ICdPYmplY3QnKSB7XG5cdFx0cmV0dXJuIGZhbHNlO1xuXHR9XG5cdHZhciBhbGxvd2VkID0ge1xuXHRcdCdbW0NvbmZpZ3VyYWJsZV1dJzogdHJ1ZSxcblx0XHQnW1tFbnVtZXJhYmxlXV0nOiB0cnVlLFxuXHRcdCdbW0dldF1dJzogdHJ1ZSxcblx0XHQnW1tTZXRdXSc6IHRydWUsXG5cdFx0J1tbVmFsdWVdXSc6IHRydWUsXG5cdFx0J1tbV3JpdGFibGVdXSc6IHRydWVcblx0fTtcblxuXHRmb3IgKHZhciBrZXkgaW4gRGVzYykgeyAvLyBlc2xpbnQtZGlzYWJsZS1saW5lIG5vLXJlc3RyaWN0ZWQtc3ludGF4XG5cdFx0aWYgKGhhcyhEZXNjLCBrZXkpICYmICFhbGxvd2VkW2tleV0pIHtcblx0XHRcdHJldHVybiBmYWxzZTtcblx0XHR9XG5cdH1cblxuXHRpZiAoRVMuSXNEYXRhRGVzY3JpcHRvcihEZXNjKSAmJiBFUy5Jc0FjY2Vzc29yRGVzY3JpcHRvcihEZXNjKSkge1xuXHRcdHRocm93IG5ldyAkVHlwZUVycm9yKCdQcm9wZXJ0eSBEZXNjcmlwdG9ycyBtYXkgbm90IGJlIGJvdGggYWNjZXNzb3IgYW5kIGRhdGEgZGVzY3JpcHRvcnMnKTtcblx0fVxuXHRyZXR1cm4gdHJ1ZTtcbn07XG4iXSwic291cmNlUm9vdCI6IiJ9\n//# sourceURL=webpack-internal:///9146\n")},6935:module=>{"use strict";eval("\n\nmodule.exports = function isTrailingSurrogate(charCode) {\n\treturn typeof charCode === 'number' && charCode >= 0xDC00 && charCode <= 0xDFFF;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0L2hlbHBlcnMvaXNUcmFpbGluZ1N1cnJvZ2F0ZS5qcz8wMTM1Il0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFhOztBQUViO0FBQ0E7QUFDQSIsImZpbGUiOiI2OTM1LmpzIiwic291cmNlc0NvbnRlbnQiOlsiJ3VzZSBzdHJpY3QnO1xuXG5tb2R1bGUuZXhwb3J0cyA9IGZ1bmN0aW9uIGlzVHJhaWxpbmdTdXJyb2dhdGUoY2hhckNvZGUpIHtcblx0cmV0dXJuIHR5cGVvZiBjaGFyQ29kZSA9PT0gJ251bWJlcicgJiYgY2hhckNvZGUgPj0gMHhEQzAwICYmIGNoYXJDb2RlIDw9IDB4REZGRjtcbn07XG4iXSwic291cmNlUm9vdCI6IiJ9\n//# sourceURL=webpack-internal:///6935\n")},5376:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\n\nvar $Math = GetIntrinsic('%Math%');\nvar $Number = GetIntrinsic('%Number%');\n\nmodule.exports = $Number.MAX_SAFE_INTEGER || $Math.pow(2, 53) - 1;\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0L2hlbHBlcnMvbWF4U2FmZUludGVnZXIuanM/YjJhNSJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBYTs7QUFFYixtQkFBbUIsbUJBQU8sQ0FBQyxHQUFlOztBQUUxQztBQUNBOztBQUVBIiwiZmlsZSI6IjUzNzYuanMiLCJzb3VyY2VzQ29udGVudCI6WyIndXNlIHN0cmljdCc7XG5cbnZhciBHZXRJbnRyaW5zaWMgPSByZXF1aXJlKCdnZXQtaW50cmluc2ljJyk7XG5cbnZhciAkTWF0aCA9IEdldEludHJpbnNpYygnJU1hdGglJyk7XG52YXIgJE51bWJlciA9IEdldEludHJpbnNpYygnJU51bWJlciUnKTtcblxubW9kdWxlLmV4cG9ydHMgPSAkTnVtYmVyLk1BWF9TQUZFX0lOVEVHRVIgfHwgJE1hdGgucG93KDIsIDUzKSAtIDE7XG4iXSwic291cmNlUm9vdCI6IiJ9\n//# sourceURL=webpack-internal:///5376\n")},1652:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\n\nvar $test = GetIntrinsic('RegExp.prototype.test');\n\nvar callBind = __webpack_require__(5559);\n\nmodule.exports = function regexTester(regex) {\n\treturn callBind($test, regex);\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0L2hlbHBlcnMvcmVnZXhUZXN0ZXIuanM/MGZhYSJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBYTs7QUFFYixtQkFBbUIsbUJBQU8sQ0FBQyxHQUFlOztBQUUxQzs7QUFFQSxlQUFlLG1CQUFPLENBQUMsSUFBVzs7QUFFbEM7QUFDQTtBQUNBIiwiZmlsZSI6IjE2NTIuanMiLCJzb3VyY2VzQ29udGVudCI6WyIndXNlIHN0cmljdCc7XG5cbnZhciBHZXRJbnRyaW5zaWMgPSByZXF1aXJlKCdnZXQtaW50cmluc2ljJyk7XG5cbnZhciAkdGVzdCA9IEdldEludHJpbnNpYygnUmVnRXhwLnByb3RvdHlwZS50ZXN0Jyk7XG5cbnZhciBjYWxsQmluZCA9IHJlcXVpcmUoJ2NhbGwtYmluZCcpO1xuXG5tb2R1bGUuZXhwb3J0cyA9IGZ1bmN0aW9uIHJlZ2V4VGVzdGVyKHJlZ2V4KSB7XG5cdHJldHVybiBjYWxsQmluZCgkdGVzdCwgcmVnZXgpO1xufTtcbiJdLCJzb3VyY2VSb290IjoiIn0=\n//# sourceURL=webpack-internal:///1652\n")},4099:module=>{"use strict";eval("\n\nmodule.exports = function sign(number) {\n\treturn number >= 0 ? 1 : -1;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0L2hlbHBlcnMvc2lnbi5qcz81OTc1Il0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFhOztBQUViO0FBQ0E7QUFDQSIsImZpbGUiOiI0MDk5LmpzIiwic291cmNlc0NvbnRlbnQiOlsiJ3VzZSBzdHJpY3QnO1xuXG5tb2R1bGUuZXhwb3J0cyA9IGZ1bmN0aW9uIHNpZ24obnVtYmVyKSB7XG5cdHJldHVybiBudW1iZXIgPj0gMCA/IDEgOiAtMTtcbn07XG4iXSwic291cmNlUm9vdCI6IiJ9\n//# sourceURL=webpack-internal:///4099\n")},1503:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar hasSymbols = typeof Symbol === 'function' && typeof Symbol.iterator === 'symbol';\n\nvar isPrimitive = __webpack_require__(4149);\nvar isCallable = __webpack_require__(5320);\nvar isDate = __webpack_require__(8923);\nvar isSymbol = __webpack_require__(2636);\n\nvar ordinaryToPrimitive = function OrdinaryToPrimitive(O, hint) {\n\tif (typeof O === 'undefined' || O === null) {\n\t\tthrow new TypeError('Cannot call method on ' + O);\n\t}\n\tif (typeof hint !== 'string' || (hint !== 'number' && hint !== 'string')) {\n\t\tthrow new TypeError('hint must be \"string\" or \"number\"');\n\t}\n\tvar methodNames = hint === 'string' ? ['toString', 'valueOf'] : ['valueOf', 'toString'];\n\tvar method, result, i;\n\tfor (i = 0; i < methodNames.length; ++i) {\n\t\tmethod = O[methodNames[i]];\n\t\tif (isCallable(method)) {\n\t\t\tresult = method.call(O);\n\t\t\tif (isPrimitive(result)) {\n\t\t\t\treturn result;\n\t\t\t}\n\t\t}\n\t}\n\tthrow new TypeError('No default value');\n};\n\nvar GetMethod = function GetMethod(O, P) {\n\tvar func = O[P];\n\tif (func !== null && typeof func !== 'undefined') {\n\t\tif (!isCallable(func)) {\n\t\t\tthrow new TypeError(func + ' returned for property ' + P + ' of object ' + O + ' is not a function');\n\t\t}\n\t\treturn func;\n\t}\n\treturn void 0;\n};\n\n// http://www.ecma-international.org/ecma-262/6.0/#sec-toprimitive\nmodule.exports = function ToPrimitive(input) {\n\tif (isPrimitive(input)) {\n\t\treturn input;\n\t}\n\tvar hint = 'default';\n\tif (arguments.length > 1) {\n\t\tif (arguments[1] === String) {\n\t\t\thint = 'string';\n\t\t} else if (arguments[1] === Number) {\n\t\t\thint = 'number';\n\t\t}\n\t}\n\n\tvar exoticToPrim;\n\tif (hasSymbols) {\n\t\tif (Symbol.toPrimitive) {\n\t\t\texoticToPrim = GetMethod(input, Symbol.toPrimitive);\n\t\t} else if (isSymbol(input)) {\n\t\t\texoticToPrim = Symbol.prototype.valueOf;\n\t\t}\n\t}\n\tif (typeof exoticToPrim !== 'undefined') {\n\t\tvar result = exoticToPrim.call(input, hint);\n\t\tif (isPrimitive(result)) {\n\t\t\treturn result;\n\t\t}\n\t\tthrow new TypeError('unable to convert exotic object to primitive');\n\t}\n\tif (hint === 'default' && (isDate(input) || isSymbol(input))) {\n\t\thint = 'string';\n\t}\n\treturn ordinaryToPrimitive(input, hint === 'default' ? 'number' : hint);\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///1503\n")},2116:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar toStr = Object.prototype.toString;\n\nvar isPrimitive = __webpack_require__(4149);\n\nvar isCallable = __webpack_require__(5320);\n\n// http://ecma-international.org/ecma-262/5.1/#sec-8.12.8\nvar ES5internalSlots = {\n\t'[[DefaultValue]]': function (O) {\n\t\tvar actualHint;\n\t\tif (arguments.length > 1) {\n\t\t\tactualHint = arguments[1];\n\t\t} else {\n\t\t\tactualHint = toStr.call(O) === '[object Date]' ? String : Number;\n\t\t}\n\n\t\tif (actualHint === String || actualHint === Number) {\n\t\t\tvar methods = actualHint === String ? ['toString', 'valueOf'] : ['valueOf', 'toString'];\n\t\t\tvar value, i;\n\t\t\tfor (i = 0; i < methods.length; ++i) {\n\t\t\t\tif (isCallable(O[methods[i]])) {\n\t\t\t\t\tvalue = O[methods[i]]();\n\t\t\t\t\tif (isPrimitive(value)) {\n\t\t\t\t\t\treturn value;\n\t\t\t\t\t}\n\t\t\t\t}\n\t\t\t}\n\t\t\tthrow new TypeError('No default value');\n\t\t}\n\t\tthrow new TypeError('invalid [[DefaultValue]] hint supplied');\n\t}\n};\n\n// http://ecma-international.org/ecma-262/5.1/#sec-9.1\nmodule.exports = function ToPrimitive(input) {\n\tif (isPrimitive(input)) {\n\t\treturn input;\n\t}\n\tif (arguments.length > 1) {\n\t\treturn ES5internalSlots['[[DefaultValue]]'](input, arguments[1]);\n\t}\n\treturn ES5internalSlots['[[DefaultValue]]'](input);\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLXRvLXByaW1pdGl2ZS9lczUuanM/MmYxNyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBYTs7QUFFYjs7QUFFQSxrQkFBa0IsbUJBQU8sQ0FBQyxJQUF1Qjs7QUFFakQsaUJBQWlCLG1CQUFPLENBQUMsSUFBYTs7QUFFdEM7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0EsR0FBRztBQUNIO0FBQ0E7O0FBRUE7QUFDQTtBQUNBO0FBQ0EsY0FBYyxvQkFBb0I7QUFDbEM7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBOztBQUVBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBIiwiZmlsZSI6IjIxMTYuanMiLCJzb3VyY2VzQ29udGVudCI6WyIndXNlIHN0cmljdCc7XG5cbnZhciB0b1N0ciA9IE9iamVjdC5wcm90b3R5cGUudG9TdHJpbmc7XG5cbnZhciBpc1ByaW1pdGl2ZSA9IHJlcXVpcmUoJy4vaGVscGVycy9pc1ByaW1pdGl2ZScpO1xuXG52YXIgaXNDYWxsYWJsZSA9IHJlcXVpcmUoJ2lzLWNhbGxhYmxlJyk7XG5cbi8vIGh0dHA6Ly9lY21hLWludGVybmF0aW9uYWwub3JnL2VjbWEtMjYyLzUuMS8jc2VjLTguMTIuOFxudmFyIEVTNWludGVybmFsU2xvdHMgPSB7XG5cdCdbW0RlZmF1bHRWYWx1ZV1dJzogZnVuY3Rpb24gKE8pIHtcblx0XHR2YXIgYWN0dWFsSGludDtcblx0XHRpZiAoYXJndW1lbnRzLmxlbmd0aCA+IDEpIHtcblx0XHRcdGFjdHVhbEhpbnQgPSBhcmd1bWVudHNbMV07XG5cdFx0fSBlbHNlIHtcblx0XHRcdGFjdHVhbEhpbnQgPSB0b1N0ci5jYWxsKE8pID09PSAnW29iamVjdCBEYXRlXScgPyBTdHJpbmcgOiBOdW1iZXI7XG5cdFx0fVxuXG5cdFx0aWYgKGFjdHVhbEhpbnQgPT09IFN0cmluZyB8fCBhY3R1YWxIaW50ID09PSBOdW1iZXIpIHtcblx0XHRcdHZhciBtZXRob2RzID0gYWN0dWFsSGludCA9PT0gU3RyaW5nID8gWyd0b1N0cmluZycsICd2YWx1ZU9mJ10gOiBbJ3ZhbHVlT2YnLCAndG9TdHJpbmcnXTtcblx0XHRcdHZhciB2YWx1ZSwgaTtcblx0XHRcdGZvciAoaSA9IDA7IGkgPCBtZXRob2RzLmxlbmd0aDsgKytpKSB7XG5cdFx0XHRcdGlmIChpc0NhbGxhYmxlKE9bbWV0aG9kc1tpXV0pKSB7XG5cdFx0XHRcdFx0dmFsdWUgPSBPW21ldGhvZHNbaV1dKCk7XG5cdFx0XHRcdFx0aWYgKGlzUHJpbWl0aXZlKHZhbHVlKSkge1xuXHRcdFx0XHRcdFx0cmV0dXJuIHZhbHVlO1xuXHRcdFx0XHRcdH1cblx0XHRcdFx0fVxuXHRcdFx0fVxuXHRcdFx0dGhyb3cgbmV3IFR5cGVFcnJvcignTm8gZGVmYXVsdCB2YWx1ZScpO1xuXHRcdH1cblx0XHR0aHJvdyBuZXcgVHlwZUVycm9yKCdpbnZhbGlkIFtbRGVmYXVsdFZhbHVlXV0gaGludCBzdXBwbGllZCcpO1xuXHR9XG59O1xuXG4vLyBodHRwOi8vZWNtYS1pbnRlcm5hdGlvbmFsLm9yZy9lY21hLTI2Mi81LjEvI3NlYy05LjFcbm1vZHVsZS5leHBvcnRzID0gZnVuY3Rpb24gVG9QcmltaXRpdmUoaW5wdXQpIHtcblx0aWYgKGlzUHJpbWl0aXZlKGlucHV0KSkge1xuXHRcdHJldHVybiBpbnB1dDtcblx0fVxuXHRpZiAoYXJndW1lbnRzLmxlbmd0aCA+IDEpIHtcblx0XHRyZXR1cm4gRVM1aW50ZXJuYWxTbG90c1snW1tEZWZhdWx0VmFsdWVdXSddKGlucHV0LCBhcmd1bWVudHNbMV0pO1xuXHR9XG5cdHJldHVybiBFUzVpbnRlcm5hbFNsb3RzWydbW0RlZmF1bHRWYWx1ZV1dJ10oaW5wdXQpO1xufTtcbiJdLCJzb3VyY2VSb290IjoiIn0=\n//# sourceURL=webpack-internal:///2116\n")},4149:module=>{"use strict";eval("\n\nmodule.exports = function isPrimitive(value) {\n\treturn value === null || (typeof value !== 'function' && typeof value !== 'object');\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLXRvLXByaW1pdGl2ZS9oZWxwZXJzL2lzUHJpbWl0aXZlLmpzPzRkZTgiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQWE7O0FBRWI7QUFDQTtBQUNBIiwiZmlsZSI6IjQxNDkuanMiLCJzb3VyY2VzQ29udGVudCI6WyIndXNlIHN0cmljdCc7XG5cbm1vZHVsZS5leHBvcnRzID0gZnVuY3Rpb24gaXNQcmltaXRpdmUodmFsdWUpIHtcblx0cmV0dXJuIHZhbHVlID09PSBudWxsIHx8ICh0eXBlb2YgdmFsdWUgIT09ICdmdW5jdGlvbicgJiYgdHlwZW9mIHZhbHVlICE9PSAnb2JqZWN0Jyk7XG59O1xuIl0sInNvdXJjZVJvb3QiOiIifQ==\n//# sourceURL=webpack-internal:///4149\n")},7648:module=>{"use strict";eval("\n\n/* eslint no-invalid-this: 1 */\n\nvar ERROR_MESSAGE = 'Function.prototype.bind called on incompatible ';\nvar slice = Array.prototype.slice;\nvar toStr = Object.prototype.toString;\nvar funcType = '[object Function]';\n\nmodule.exports = function bind(that) {\n    var target = this;\n    if (typeof target !== 'function' || toStr.call(target) !== funcType) {\n        throw new TypeError(ERROR_MESSAGE + target);\n    }\n    var args = slice.call(arguments, 1);\n\n    var bound;\n    var binder = function () {\n        if (this instanceof bound) {\n            var result = target.apply(\n                this,\n                args.concat(slice.call(arguments))\n            );\n            if (Object(result) === result) {\n                return result;\n            }\n            return this;\n        } else {\n            return target.apply(\n                that,\n                args.concat(slice.call(arguments))\n            );\n        }\n    };\n\n    var boundLength = Math.max(0, target.length - args.length);\n    var boundArgs = [];\n    for (var i = 0; i < boundLength; i++) {\n        boundArgs.push('$' + i);\n    }\n\n    bound = Function('binder', 'return function (' + boundArgs.join(',') + '){ return binder.apply(this,arguments); }')(binder);\n\n    if (target.prototype) {\n        var Empty = function Empty() {};\n        Empty.prototype = target.prototype;\n        bound.prototype = new Empty();\n        Empty.prototype = null;\n    }\n\n    return bound;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///7648\n")},8612:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar implementation = __webpack_require__(7648);\n\nmodule.exports = Function.prototype.bind || implementation;\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2Z1bmN0aW9uLWJpbmQvaW5kZXguanM/MGY3YyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBYTs7QUFFYixxQkFBcUIsbUJBQU8sQ0FBQyxJQUFrQjs7QUFFL0MiLCJmaWxlIjoiODYxMi5qcyIsInNvdXJjZXNDb250ZW50IjpbIid1c2Ugc3RyaWN0JztcblxudmFyIGltcGxlbWVudGF0aW9uID0gcmVxdWlyZSgnLi9pbXBsZW1lbnRhdGlvbicpO1xuXG5tb2R1bGUuZXhwb3J0cyA9IEZ1bmN0aW9uLnByb3RvdHlwZS5iaW5kIHx8IGltcGxlbWVudGF0aW9uO1xuIl0sInNvdXJjZVJvb3QiOiIifQ==\n//# sourceURL=webpack-internal:///8612\n")},210:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar undefined;\n\nvar $SyntaxError = SyntaxError;\nvar $Function = Function;\nvar $TypeError = TypeError;\n\n// eslint-disable-next-line consistent-return\nvar getEvalledConstructor = function (expressionSyntax) {\n\ttry {\n\t\treturn $Function('\"use strict\"; return (' + expressionSyntax + ').constructor;')();\n\t} catch (e) {}\n};\n\nvar $gOPD = Object.getOwnPropertyDescriptor;\nif ($gOPD) {\n\ttry {\n\t\t$gOPD({}, '');\n\t} catch (e) {\n\t\t$gOPD = null; // this is IE 8, which has a broken gOPD\n\t}\n}\n\nvar throwTypeError = function () {\n\tthrow new $TypeError();\n};\nvar ThrowTypeError = $gOPD\n\t? (function () {\n\t\ttry {\n\t\t\t// eslint-disable-next-line no-unused-expressions, no-caller, no-restricted-properties\n\t\t\targuments.callee; // IE 8 does not throw here\n\t\t\treturn throwTypeError;\n\t\t} catch (calleeThrows) {\n\t\t\ttry {\n\t\t\t\t// IE 8 throws on Object.getOwnPropertyDescriptor(arguments, '')\n\t\t\t\treturn $gOPD(arguments, 'callee').get;\n\t\t\t} catch (gOPDthrows) {\n\t\t\t\treturn throwTypeError;\n\t\t\t}\n\t\t}\n\t}())\n\t: throwTypeError;\n\nvar hasSymbols = __webpack_require__(1405)();\n\nvar getProto = Object.getPrototypeOf || function (x) { return x.__proto__; }; // eslint-disable-line no-proto\n\nvar needsEval = {};\n\nvar TypedArray = typeof Uint8Array === 'undefined' ? undefined : getProto(Uint8Array);\n\nvar INTRINSICS = {\n\t'%AggregateError%': typeof AggregateError === 'undefined' ? undefined : AggregateError,\n\t'%Array%': Array,\n\t'%ArrayBuffer%': typeof ArrayBuffer === 'undefined' ? undefined : ArrayBuffer,\n\t'%ArrayIteratorPrototype%': hasSymbols ? getProto([][Symbol.iterator]()) : undefined,\n\t'%AsyncFromSyncIteratorPrototype%': undefined,\n\t'%AsyncFunction%': needsEval,\n\t'%AsyncGenerator%': needsEval,\n\t'%AsyncGeneratorFunction%': needsEval,\n\t'%AsyncIteratorPrototype%': needsEval,\n\t'%Atomics%': typeof Atomics === 'undefined' ? undefined : Atomics,\n\t'%BigInt%': typeof BigInt === 'undefined' ? undefined : BigInt,\n\t'%Boolean%': Boolean,\n\t'%DataView%': typeof DataView === 'undefined' ? undefined : DataView,\n\t'%Date%': Date,\n\t'%decodeURI%': decodeURI,\n\t'%decodeURIComponent%': decodeURIComponent,\n\t'%encodeURI%': encodeURI,\n\t'%encodeURIComponent%': encodeURIComponent,\n\t'%Error%': Error,\n\t'%eval%': eval, // eslint-disable-line no-eval\n\t'%EvalError%': EvalError,\n\t'%Float32Array%': typeof Float32Array === 'undefined' ? undefined : Float32Array,\n\t'%Float64Array%': typeof Float64Array === 'undefined' ? undefined : Float64Array,\n\t'%FinalizationRegistry%': typeof FinalizationRegistry === 'undefined' ? undefined : FinalizationRegistry,\n\t'%Function%': $Function,\n\t'%GeneratorFunction%': needsEval,\n\t'%Int8Array%': typeof Int8Array === 'undefined' ? undefined : Int8Array,\n\t'%Int16Array%': typeof Int16Array === 'undefined' ? undefined : Int16Array,\n\t'%Int32Array%': typeof Int32Array === 'undefined' ? undefined : Int32Array,\n\t'%isFinite%': isFinite,\n\t'%isNaN%': isNaN,\n\t'%IteratorPrototype%': hasSymbols ? getProto(getProto([][Symbol.iterator]())) : undefined,\n\t'%JSON%': typeof JSON === 'object' ? JSON : undefined,\n\t'%Map%': typeof Map === 'undefined' ? undefined : Map,\n\t'%MapIteratorPrototype%': typeof Map === 'undefined' || !hasSymbols ? undefined : getProto(new Map()[Symbol.iterator]()),\n\t'%Math%': Math,\n\t'%Number%': Number,\n\t'%Object%': Object,\n\t'%parseFloat%': parseFloat,\n\t'%parseInt%': parseInt,\n\t'%Promise%': typeof Promise === 'undefined' ? undefined : Promise,\n\t'%Proxy%': typeof Proxy === 'undefined' ? undefined : Proxy,\n\t'%RangeError%': RangeError,\n\t'%ReferenceError%': ReferenceError,\n\t'%Reflect%': typeof Reflect === 'undefined' ? undefined : Reflect,\n\t'%RegExp%': RegExp,\n\t'%Set%': typeof Set === 'undefined' ? undefined : Set,\n\t'%SetIteratorPrototype%': typeof Set === 'undefined' || !hasSymbols ? undefined : getProto(new Set()[Symbol.iterator]()),\n\t'%SharedArrayBuffer%': typeof SharedArrayBuffer === 'undefined' ? undefined : SharedArrayBuffer,\n\t'%String%': String,\n\t'%StringIteratorPrototype%': hasSymbols ? getProto(''[Symbol.iterator]()) : undefined,\n\t'%Symbol%': hasSymbols ? Symbol : undefined,\n\t'%SyntaxError%': $SyntaxError,\n\t'%ThrowTypeError%': ThrowTypeError,\n\t'%TypedArray%': TypedArray,\n\t'%TypeError%': $TypeError,\n\t'%Uint8Array%': typeof Uint8Array === 'undefined' ? undefined : Uint8Array,\n\t'%Uint8ClampedArray%': typeof Uint8ClampedArray === 'undefined' ? undefined : Uint8ClampedArray,\n\t'%Uint16Array%': typeof Uint16Array === 'undefined' ? undefined : Uint16Array,\n\t'%Uint32Array%': typeof Uint32Array === 'undefined' ? undefined : Uint32Array,\n\t'%URIError%': URIError,\n\t'%WeakMap%': typeof WeakMap === 'undefined' ? undefined : WeakMap,\n\t'%WeakRef%': typeof WeakRef === 'undefined' ? undefined : WeakRef,\n\t'%WeakSet%': typeof WeakSet === 'undefined' ? undefined : WeakSet\n};\n\nvar doEval = function doEval(name) {\n\tvar value;\n\tif (name === '%AsyncFunction%') {\n\t\tvalue = getEvalledConstructor('async function () {}');\n\t} else if (name === '%GeneratorFunction%') {\n\t\tvalue = getEvalledConstructor('function* () {}');\n\t} else if (name === '%AsyncGeneratorFunction%') {\n\t\tvalue = getEvalledConstructor('async function* () {}');\n\t} else if (name === '%AsyncGenerator%') {\n\t\tvar fn = doEval('%AsyncGeneratorFunction%');\n\t\tif (fn) {\n\t\t\tvalue = fn.prototype;\n\t\t}\n\t} else if (name === '%AsyncIteratorPrototype%') {\n\t\tvar gen = doEval('%AsyncGenerator%');\n\t\tif (gen) {\n\t\t\tvalue = getProto(gen.prototype);\n\t\t}\n\t}\n\n\tINTRINSICS[name] = value;\n\n\treturn value;\n};\n\nvar LEGACY_ALIASES = {\n\t'%ArrayBufferPrototype%': ['ArrayBuffer', 'prototype'],\n\t'%ArrayPrototype%': ['Array', 'prototype'],\n\t'%ArrayProto_entries%': ['Array', 'prototype', 'entries'],\n\t'%ArrayProto_forEach%': ['Array', 'prototype', 'forEach'],\n\t'%ArrayProto_keys%': ['Array', 'prototype', 'keys'],\n\t'%ArrayProto_values%': ['Array', 'prototype', 'values'],\n\t'%AsyncFunctionPrototype%': ['AsyncFunction', 'prototype'],\n\t'%AsyncGenerator%': ['AsyncGeneratorFunction', 'prototype'],\n\t'%AsyncGeneratorPrototype%': ['AsyncGeneratorFunction', 'prototype', 'prototype'],\n\t'%BooleanPrototype%': ['Boolean', 'prototype'],\n\t'%DataViewPrototype%': ['DataView', 'prototype'],\n\t'%DatePrototype%': ['Date', 'prototype'],\n\t'%ErrorPrototype%': ['Error', 'prototype'],\n\t'%EvalErrorPrototype%': ['EvalError', 'prototype'],\n\t'%Float32ArrayPrototype%': ['Float32Array', 'prototype'],\n\t'%Float64ArrayPrototype%': ['Float64Array', 'prototype'],\n\t'%FunctionPrototype%': ['Function', 'prototype'],\n\t'%Generator%': ['GeneratorFunction', 'prototype'],\n\t'%GeneratorPrototype%': ['GeneratorFunction', 'prototype', 'prototype'],\n\t'%Int8ArrayPrototype%': ['Int8Array', 'prototype'],\n\t'%Int16ArrayPrototype%': ['Int16Array', 'prototype'],\n\t'%Int32ArrayPrototype%': ['Int32Array', 'prototype'],\n\t'%JSONParse%': ['JSON', 'parse'],\n\t'%JSONStringify%': ['JSON', 'stringify'],\n\t'%MapPrototype%': ['Map', 'prototype'],\n\t'%NumberPrototype%': ['Number', 'prototype'],\n\t'%ObjectPrototype%': ['Object', 'prototype'],\n\t'%ObjProto_toString%': ['Object', 'prototype', 'toString'],\n\t'%ObjProto_valueOf%': ['Object', 'prototype', 'valueOf'],\n\t'%PromisePrototype%': ['Promise', 'prototype'],\n\t'%PromiseProto_then%': ['Promise', 'prototype', 'then'],\n\t'%Promise_all%': ['Promise', 'all'],\n\t'%Promise_reject%': ['Promise', 'reject'],\n\t'%Promise_resolve%': ['Promise', 'resolve'],\n\t'%RangeErrorPrototype%': ['RangeError', 'prototype'],\n\t'%ReferenceErrorPrototype%': ['ReferenceError', 'prototype'],\n\t'%RegExpPrototype%': ['RegExp', 'prototype'],\n\t'%SetPrototype%': ['Set', 'prototype'],\n\t'%SharedArrayBufferPrototype%': ['SharedArrayBuffer', 'prototype'],\n\t'%StringPrototype%': ['String', 'prototype'],\n\t'%SymbolPrototype%': ['Symbol', 'prototype'],\n\t'%SyntaxErrorPrototype%': ['SyntaxError', 'prototype'],\n\t'%TypedArrayPrototype%': ['TypedArray', 'prototype'],\n\t'%TypeErrorPrototype%': ['TypeError', 'prototype'],\n\t'%Uint8ArrayPrototype%': ['Uint8Array', 'prototype'],\n\t'%Uint8ClampedArrayPrototype%': ['Uint8ClampedArray', 'prototype'],\n\t'%Uint16ArrayPrototype%': ['Uint16Array', 'prototype'],\n\t'%Uint32ArrayPrototype%': ['Uint32Array', 'prototype'],\n\t'%URIErrorPrototype%': ['URIError', 'prototype'],\n\t'%WeakMapPrototype%': ['WeakMap', 'prototype'],\n\t'%WeakSetPrototype%': ['WeakSet', 'prototype']\n};\n\nvar bind = __webpack_require__(8612);\nvar hasOwn = __webpack_require__(7642);\nvar $concat = bind.call(Function.call, Array.prototype.concat);\nvar $spliceApply = bind.call(Function.apply, Array.prototype.splice);\nvar $replace = bind.call(Function.call, String.prototype.replace);\nvar $strSlice = bind.call(Function.call, String.prototype.slice);\n\n/* adapted from https://github.com/lodash/lodash/blob/4.17.15/dist/lodash.js#L6735-L6744 */\nvar rePropName = /[^%.[\\]]+|\\[(?:(-?\\d+(?:\\.\\d+)?)|([\"'])((?:(?!\\2)[^\\\\]|\\\\.)*?)\\2)\\]|(?=(?:\\.|\\[\\])(?:\\.|\\[\\]|%$))/g;\nvar reEscapeChar = /\\\\(\\\\)?/g; /** Used to match backslashes in property paths. */\nvar stringToPath = function stringToPath(string) {\n\tvar first = $strSlice(string, 0, 1);\n\tvar last = $strSlice(string, -1);\n\tif (first === '%' && last !== '%') {\n\t\tthrow new $SyntaxError('invalid intrinsic syntax, expected closing `%`');\n\t} else if (last === '%' && first !== '%') {\n\t\tthrow new $SyntaxError('invalid intrinsic syntax, expected opening `%`');\n\t}\n\tvar result = [];\n\t$replace(string, rePropName, function (match, number, quote, subString) {\n\t\tresult[result.length] = quote ? $replace(subString, reEscapeChar, '$1') : number || match;\n\t});\n\treturn result;\n};\n/* end adaptation */\n\nvar getBaseIntrinsic = function getBaseIntrinsic(name, allowMissing) {\n\tvar intrinsicName = name;\n\tvar alias;\n\tif (hasOwn(LEGACY_ALIASES, intrinsicName)) {\n\t\talias = LEGACY_ALIASES[intrinsicName];\n\t\tintrinsicName = '%' + alias[0] + '%';\n\t}\n\n\tif (hasOwn(INTRINSICS, intrinsicName)) {\n\t\tvar value = INTRINSICS[intrinsicName];\n\t\tif (value === needsEval) {\n\t\t\tvalue = doEval(intrinsicName);\n\t\t}\n\t\tif (typeof value === 'undefined' && !allowMissing) {\n\t\t\tthrow new $TypeError('intrinsic ' + name + ' exists, but is not available. Please file an issue!');\n\t\t}\n\n\t\treturn {\n\t\t\talias: alias,\n\t\t\tname: intrinsicName,\n\t\t\tvalue: value\n\t\t};\n\t}\n\n\tthrow new $SyntaxError('intrinsic ' + name + ' does not exist!');\n};\n\nmodule.exports = function GetIntrinsic(name, allowMissing) {\n\tif (typeof name !== 'string' || name.length === 0) {\n\t\tthrow new $TypeError('intrinsic name must be a non-empty string');\n\t}\n\tif (arguments.length > 1 && typeof allowMissing !== 'boolean') {\n\t\tthrow new $TypeError('\"allowMissing\" argument must be a boolean');\n\t}\n\n\tvar parts = stringToPath(name);\n\tvar intrinsicBaseName = parts.length > 0 ? parts[0] : '';\n\n\tvar intrinsic = getBaseIntrinsic('%' + intrinsicBaseName + '%', allowMissing);\n\tvar intrinsicRealName = intrinsic.name;\n\tvar value = intrinsic.value;\n\tvar skipFurtherCaching = false;\n\n\tvar alias = intrinsic.alias;\n\tif (alias) {\n\t\tintrinsicBaseName = alias[0];\n\t\t$spliceApply(parts, $concat([0, 1], alias));\n\t}\n\n\tfor (var i = 1, isOwn = true; i < parts.length; i += 1) {\n\t\tvar part = parts[i];\n\t\tvar first = $strSlice(part, 0, 1);\n\t\tvar last = $strSlice(part, -1);\n\t\tif (\n\t\t\t(\n\t\t\t\t(first === '\"' || first === \"'\" || first === '`')\n\t\t\t\t|| (last === '\"' || last === \"'\" || last === '`')\n\t\t\t)\n\t\t\t&& first !== last\n\t\t) {\n\t\t\tthrow new $SyntaxError('property names with quotes must have matching quotes');\n\t\t}\n\t\tif (part === 'constructor' || !isOwn) {\n\t\t\tskipFurtherCaching = true;\n\t\t}\n\n\t\tintrinsicBaseName += '.' + part;\n\t\tintrinsicRealName = '%' + intrinsicBaseName + '%';\n\n\t\tif (hasOwn(INTRINSICS, intrinsicRealName)) {\n\t\t\tvalue = INTRINSICS[intrinsicRealName];\n\t\t} else if (value != null) {\n\t\t\tif (!(part in value)) {\n\t\t\t\tif (!allowMissing) {\n\t\t\t\t\tthrow new $TypeError('base intrinsic for ' + name + ' exists, but the property is not available.');\n\t\t\t\t}\n\t\t\t\treturn void undefined;\n\t\t\t}\n\t\t\tif ($gOPD && (i + 1) >= parts.length) {\n\t\t\t\tvar desc = $gOPD(value, part);\n\t\t\t\tisOwn = !!desc;\n\n\t\t\t\t// By convention, when a data property is converted to an accessor\n\t\t\t\t// property to emulate a data property that does not suffer from\n\t\t\t\t// the override mistake, that accessor's getter is marked with\n\t\t\t\t// an `originalValue` property. Here, when we detect this, we\n\t\t\t\t// uphold the illusion by pretending to see that original data\n\t\t\t\t// property, i.e., returning the value rather than the getter\n\t\t\t\t// itself.\n\t\t\t\tif (isOwn && 'get' in desc && !('originalValue' in desc.get)) {\n\t\t\t\t\tvalue = desc.get;\n\t\t\t\t} else {\n\t\t\t\t\tvalue = value[part];\n\t\t\t\t}\n\t\t\t} else {\n\t\t\t\tisOwn = hasOwn(value, part);\n\t\t\t\tvalue = value[part];\n\t\t\t}\n\n\t\t\tif (isOwn && !skipFurtherCaching) {\n\t\t\t\tINTRINSICS[intrinsicRealName] = value;\n\t\t\t}\n\t\t}\n\t}\n\treturn value;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2dldC1pbnRyaW5zaWMvaW5kZXguanM/MDBjZSJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBYTs7QUFFYjs7QUFFQTtBQUNBO0FBQ0E7O0FBRUE7QUFDQTtBQUNBO0FBQ0EsaUNBQWlDLCtDQUErQztBQUNoRixFQUFFO0FBQ0Y7O0FBRUE7QUFDQTtBQUNBO0FBQ0EsVUFBVTtBQUNWLEVBQUU7QUFDRixlQUFlO0FBQ2Y7QUFDQTs7QUFFQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBLG9CQUFvQjtBQUNwQjtBQUNBLEdBQUc7QUFDSDtBQUNBO0FBQ0E7QUFDQSxJQUFJO0FBQ0o7QUFDQTtBQUNBO0FBQ0EsRUFBRTtBQUNGOztBQUVBLGlCQUFpQixtQkFBTyxDQUFDLElBQWE7O0FBRXRDLHNEQUFzRCxvQkFBb0IsR0FBRzs7QUFFN0U7O0FBRUE7O0FBRUE7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBOztBQUVBO0FBQ0E7QUFDQTtBQUNBLHFEQUFxRDtBQUNyRCxFQUFFO0FBQ0YsZ0RBQWdEO0FBQ2hELEVBQUU7QUFDRixzREFBc0Q7QUFDdEQsRUFBRTtBQUNGO0FBQ0E7QUFDQTtBQUNBO0FBQ0EsRUFBRTtBQUNGO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7O0FBRUE7O0FBRUE7QUFDQTs7QUFFQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBOztBQUVBLFdBQVcsbUJBQU8sQ0FBQyxJQUFlO0FBQ2xDLGFBQWEsbUJBQU8sQ0FBQyxJQUFLO0FBQzFCO0FBQ0E7QUFDQTtBQUNBOztBQUVBO0FBQ0E7QUFDQSw4QkFBOEI7QUFDOUI7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBLEVBQUU7QUFDRjtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0EsRUFBRTtBQUNGO0FBQ0E7QUFDQTs7QUFFQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTs7QUFFQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBOztBQUVBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTs7QUFFQTtBQUNBOztBQUVBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBOztBQUVBO0FBQ0E7O0FBRUE7QUFDQTtBQUNBO0FBQ0E7O0FBRUE7QUFDQTtBQUNBO0FBQ0E7QUFDQTs7QUFFQSw4QkFBOEIsa0JBQWtCO0FBQ2hEO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTs7QUFFQTtBQUNBOztBQUVBO0FBQ0E7QUFDQSxHQUFHO0FBQ0g7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBOztBQUVBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBLEtBQUs7QUFDTDtBQUNBO0FBQ0EsSUFBSTtBQUNKO0FBQ0E7QUFDQTs7QUFFQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQSIsImZpbGUiOiIyMTAuanMiLCJzb3VyY2VzQ29udGVudCI6WyIndXNlIHN0cmljdCc7XG5cbnZhciB1bmRlZmluZWQ7XG5cbnZhciAkU3ludGF4RXJyb3IgPSBTeW50YXhFcnJvcjtcbnZhciAkRnVuY3Rpb24gPSBGdW5jdGlvbjtcbnZhciAkVHlwZUVycm9yID0gVHlwZUVycm9yO1xuXG4vLyBlc2xpbnQtZGlzYWJsZS1uZXh0LWxpbmUgY29uc2lzdGVudC1yZXR1cm5cbnZhciBnZXRFdmFsbGVkQ29uc3RydWN0b3IgPSBmdW5jdGlvbiAoZXhwcmVzc2lvblN5bnRheCkge1xuXHR0cnkge1xuXHRcdHJldHVybiAkRnVuY3Rpb24oJ1widXNlIHN0cmljdFwiOyByZXR1cm4gKCcgKyBleHByZXNzaW9uU3ludGF4ICsgJykuY29uc3RydWN0b3I7JykoKTtcblx0fSBjYXRjaCAoZSkge31cbn07XG5cbnZhciAkZ09QRCA9IE9iamVjdC5nZXRPd25Qcm9wZXJ0eURlc2NyaXB0b3I7XG5pZiAoJGdPUEQpIHtcblx0dHJ5IHtcblx0XHQkZ09QRCh7fSwgJycpO1xuXHR9IGNhdGNoIChlKSB7XG5cdFx0JGdPUEQgPSBudWxsOyAvLyB0aGlzIGlzIElFIDgsIHdoaWNoIGhhcyBhIGJyb2tlbiBnT1BEXG5cdH1cbn1cblxudmFyIHRocm93VHlwZUVycm9yID0gZnVuY3Rpb24gKCkge1xuXHR0aHJvdyBuZXcgJFR5cGVFcnJvcigpO1xufTtcbnZhciBUaHJvd1R5cGVFcnJvciA9ICRnT1BEXG5cdD8gKGZ1bmN0aW9uICgpIHtcblx0XHR0cnkge1xuXHRcdFx0Ly8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIG5vLXVudXNlZC1leHByZXNzaW9ucywgbm8tY2FsbGVyLCBuby1yZXN0cmljdGVkLXByb3BlcnRpZXNcblx0XHRcdGFyZ3VtZW50cy5jYWxsZWU7IC8vIElFIDggZG9lcyBub3QgdGhyb3cgaGVyZVxuXHRcdFx0cmV0dXJuIHRocm93VHlwZUVycm9yO1xuXHRcdH0gY2F0Y2ggKGNhbGxlZVRocm93cykge1xuXHRcdFx0dHJ5IHtcblx0XHRcdFx0Ly8gSUUgOCB0aHJvd3Mgb24gT2JqZWN0LmdldE93blByb3BlcnR5RGVzY3JpcHRvcihhcmd1bWVudHMsICcnKVxuXHRcdFx0XHRyZXR1cm4gJGdPUEQoYXJndW1lbnRzLCAnY2FsbGVlJykuZ2V0O1xuXHRcdFx0fSBjYXRjaCAoZ09QRHRocm93cykge1xuXHRcdFx0XHRyZXR1cm4gdGhyb3dUeXBlRXJyb3I7XG5cdFx0XHR9XG5cdFx0fVxuXHR9KCkpXG5cdDogdGhyb3dUeXBlRXJyb3I7XG5cbnZhciBoYXNTeW1ib2xzID0gcmVxdWlyZSgnaGFzLXN5bWJvbHMnKSgpO1xuXG52YXIgZ2V0UHJvdG8gPSBPYmplY3QuZ2V0UHJvdG90eXBlT2YgfHwgZnVuY3Rpb24gKHgpIHsgcmV0dXJuIHguX19wcm90b19fOyB9OyAvLyBlc2xpbnQtZGlzYWJsZS1saW5lIG5vLXByb3RvXG5cbnZhciBuZWVkc0V2YWwgPSB7fTtcblxudmFyIFR5cGVkQXJyYXkgPSB0eXBlb2YgVWludDhBcnJheSA9PT0gJ3VuZGVmaW5lZCcgPyB1bmRlZmluZWQgOiBnZXRQcm90byhVaW50OEFycmF5KTtcblxudmFyIElOVFJJTlNJQ1MgPSB7XG5cdCclQWdncmVnYXRlRXJyb3IlJzogdHlwZW9mIEFnZ3JlZ2F0ZUVycm9yID09PSAndW5kZWZpbmVkJyA/IHVuZGVmaW5lZCA6IEFnZ3JlZ2F0ZUVycm9yLFxuXHQnJUFycmF5JSc6IEFycmF5LFxuXHQnJUFycmF5QnVmZmVyJSc6IHR5cGVvZiBBcnJheUJ1ZmZlciA9PT0gJ3VuZGVmaW5lZCcgPyB1bmRlZmluZWQgOiBBcnJheUJ1ZmZlcixcblx0JyVBcnJheUl0ZXJhdG9yUHJvdG90eXBlJSc6IGhhc1N5bWJvbHMgPyBnZXRQcm90byhbXVtTeW1ib2wuaXRlcmF0b3JdKCkpIDogdW5kZWZpbmVkLFxuXHQnJUFzeW5jRnJvbVN5bmNJdGVyYXRvclByb3RvdHlwZSUnOiB1bmRlZmluZWQsXG5cdCclQXN5bmNGdW5jdGlvbiUnOiBuZWVkc0V2YWwsXG5cdCclQXN5bmNHZW5lcmF0b3IlJzogbmVlZHNFdmFsLFxuXHQnJUFzeW5jR2VuZXJhdG9yRnVuY3Rpb24lJzogbmVlZHNFdmFsLFxuXHQnJUFzeW5jSXRlcmF0b3JQcm90b3R5cGUlJzogbmVlZHNFdmFsLFxuXHQnJUF0b21pY3MlJzogdHlwZW9mIEF0b21pY3MgPT09ICd1bmRlZmluZWQnID8gdW5kZWZpbmVkIDogQXRvbWljcyxcblx0JyVCaWdJbnQlJzogdHlwZW9mIEJpZ0ludCA9PT0gJ3VuZGVmaW5lZCcgPyB1bmRlZmluZWQgOiBCaWdJbnQsXG5cdCclQm9vbGVhbiUnOiBCb29sZWFuLFxuXHQnJURhdGFWaWV3JSc6IHR5cGVvZiBEYXRhVmlldyA9PT0gJ3VuZGVmaW5lZCcgPyB1bmRlZmluZWQgOiBEYXRhVmlldyxcblx0JyVEYXRlJSc6IERhdGUsXG5cdCclZGVjb2RlVVJJJSc6IGRlY29kZVVSSSxcblx0JyVkZWNvZGVVUklDb21wb25lbnQlJzogZGVjb2RlVVJJQ29tcG9uZW50LFxuXHQnJWVuY29kZVVSSSUnOiBlbmNvZGVVUkksXG5cdCclZW5jb2RlVVJJQ29tcG9uZW50JSc6IGVuY29kZVVSSUNvbXBvbmVudCxcblx0JyVFcnJvciUnOiBFcnJvcixcblx0JyVldmFsJSc6IGV2YWwsIC8vIGVzbGludC1kaXNhYmxlLWxpbmUgbm8tZXZhbFxuXHQnJUV2YWxFcnJvciUnOiBFdmFsRXJyb3IsXG5cdCclRmxvYXQzMkFycmF5JSc6IHR5cGVvZiBGbG9hdDMyQXJyYXkgPT09ICd1bmRlZmluZWQnID8gdW5kZWZpbmVkIDogRmxvYXQzMkFycmF5LFxuXHQnJUZsb2F0NjRBcnJheSUnOiB0eXBlb2YgRmxvYXQ2NEFycmF5ID09PSAndW5kZWZpbmVkJyA/IHVuZGVmaW5lZCA6IEZsb2F0NjRBcnJheSxcblx0JyVGaW5hbGl6YXRpb25SZWdpc3RyeSUnOiB0eXBlb2YgRmluYWxpemF0aW9uUmVnaXN0cnkgPT09ICd1bmRlZmluZWQnID8gdW5kZWZpbmVkIDogRmluYWxpemF0aW9uUmVnaXN0cnksXG5cdCclRnVuY3Rpb24lJzogJEZ1bmN0aW9uLFxuXHQnJUdlbmVyYXRvckZ1bmN0aW9uJSc6IG5lZWRzRXZhbCxcblx0JyVJbnQ4QXJyYXklJzogdHlwZW9mIEludDhBcnJheSA9PT0gJ3VuZGVmaW5lZCcgPyB1bmRlZmluZWQgOiBJbnQ4QXJyYXksXG5cdCclSW50MTZBcnJheSUnOiB0eXBlb2YgSW50MTZBcnJheSA9PT0gJ3VuZGVmaW5lZCcgPyB1bmRlZmluZWQgOiBJbnQxNkFycmF5LFxuXHQnJUludDMyQXJyYXklJzogdHlwZW9mIEludDMyQXJyYXkgPT09ICd1bmRlZmluZWQnID8gdW5kZWZpbmVkIDogSW50MzJBcnJheSxcblx0JyVpc0Zpbml0ZSUnOiBpc0Zpbml0ZSxcblx0JyVpc05hTiUnOiBpc05hTixcblx0JyVJdGVyYXRvclByb3RvdHlwZSUnOiBoYXNTeW1ib2xzID8gZ2V0UHJvdG8oZ2V0UHJvdG8oW11bU3ltYm9sLml0ZXJhdG9yXSgpKSkgOiB1bmRlZmluZWQsXG5cdCclSlNPTiUnOiB0eXBlb2YgSlNPTiA9PT0gJ29iamVjdCcgPyBKU09OIDogdW5kZWZpbmVkLFxuXHQnJU1hcCUnOiB0eXBlb2YgTWFwID09PSAndW5kZWZpbmVkJyA/IHVuZGVmaW5lZCA6IE1hcCxcblx0JyVNYXBJdGVyYXRvclByb3RvdHlwZSUnOiB0eXBlb2YgTWFwID09PSAndW5kZWZpbmVkJyB8fCAhaGFzU3ltYm9scyA/IHVuZGVmaW5lZCA6IGdldFByb3RvKG5ldyBNYXAoKVtTeW1ib2wuaXRlcmF0b3JdKCkpLFxuXHQnJU1hdGglJzogTWF0aCxcblx0JyVOdW1iZXIlJzogTnVtYmVyLFxuXHQnJU9iamVjdCUnOiBPYmplY3QsXG5cdCclcGFyc2VGbG9hdCUnOiBwYXJzZUZsb2F0LFxuXHQnJXBhcnNlSW50JSc6IHBhcnNlSW50LFxuXHQnJVByb21pc2UlJzogdHlwZW9mIFByb21pc2UgPT09ICd1bmRlZmluZWQnID8gdW5kZWZpbmVkIDogUHJvbWlzZSxcblx0JyVQcm94eSUnOiB0eXBlb2YgUHJveHkgPT09ICd1bmRlZmluZWQnID8gdW5kZWZpbmVkIDogUHJveHksXG5cdCclUmFuZ2VFcnJvciUnOiBSYW5nZUVycm9yLFxuXHQnJVJlZmVyZW5jZUVycm9yJSc6IFJlZmVyZW5jZUVycm9yLFxuXHQnJVJlZmxlY3QlJzogdHlwZW9mIFJlZmxlY3QgPT09ICd1bmRlZmluZWQnID8gdW5kZWZpbmVkIDogUmVmbGVjdCxcblx0JyVSZWdFeHAlJzogUmVnRXhwLFxuXHQnJVNldCUnOiB0eXBlb2YgU2V0ID09PSAndW5kZWZpbmVkJyA/IHVuZGVmaW5lZCA6IFNldCxcblx0JyVTZXRJdGVyYXRvclByb3RvdHlwZSUnOiB0eXBlb2YgU2V0ID09PSAndW5kZWZpbmVkJyB8fCAhaGFzU3ltYm9scyA/IHVuZGVmaW5lZCA6IGdldFByb3RvKG5ldyBTZXQoKVtTeW1ib2wuaXRlcmF0b3JdKCkpLFxuXHQnJVNoYXJlZEFycmF5QnVmZmVyJSc6IHR5cGVvZiBTaGFyZWRBcnJheUJ1ZmZlciA9PT0gJ3VuZGVmaW5lZCcgPyB1bmRlZmluZWQgOiBTaGFyZWRBcnJheUJ1ZmZlcixcblx0JyVTdHJpbmclJzogU3RyaW5nLFxuXHQnJVN0cmluZ0l0ZXJhdG9yUHJvdG90eXBlJSc6IGhhc1N5bWJvbHMgPyBnZXRQcm90bygnJ1tTeW1ib2wuaXRlcmF0b3JdKCkpIDogdW5kZWZpbmVkLFxuXHQnJVN5bWJvbCUnOiBoYXNTeW1ib2xzID8gU3ltYm9sIDogdW5kZWZpbmVkLFxuXHQnJVN5bnRheEVycm9yJSc6ICRTeW50YXhFcnJvcixcblx0JyVUaHJvd1R5cGVFcnJvciUnOiBUaHJvd1R5cGVFcnJvcixcblx0JyVUeXBlZEFycmF5JSc6IFR5cGVkQXJyYXksXG5cdCclVHlwZUVycm9yJSc6ICRUeXBlRXJyb3IsXG5cdCclVWludDhBcnJheSUnOiB0eXBlb2YgVWludDhBcnJheSA9PT0gJ3VuZGVmaW5lZCcgPyB1bmRlZmluZWQgOiBVaW50OEFycmF5LFxuXHQnJVVpbnQ4Q2xhbXBlZEFycmF5JSc6IHR5cGVvZiBVaW50OENsYW1wZWRBcnJheSA9PT0gJ3VuZGVmaW5lZCcgPyB1bmRlZmluZWQgOiBVaW50OENsYW1wZWRBcnJheSxcblx0JyVVaW50MTZBcnJheSUnOiB0eXBlb2YgVWludDE2QXJyYXkgPT09ICd1bmRlZmluZWQnID8gdW5kZWZpbmVkIDogVWludDE2QXJyYXksXG5cdCclVWludDMyQXJyYXklJzogdHlwZW9mIFVpbnQzMkFycmF5ID09PSAndW5kZWZpbmVkJyA/IHVuZGVmaW5lZCA6IFVpbnQzMkFycmF5LFxuXHQnJVVSSUVycm9yJSc6IFVSSUVycm9yLFxuXHQnJVdlYWtNYXAlJzogdHlwZW9mIFdlYWtNYXAgPT09ICd1bmRlZmluZWQnID8gdW5kZWZpbmVkIDogV2Vha01hcCxcblx0JyVXZWFrUmVmJSc6IHR5cGVvZiBXZWFrUmVmID09PSAndW5kZWZpbmVkJyA/IHVuZGVmaW5lZCA6IFdlYWtSZWYsXG5cdCclV2Vha1NldCUnOiB0eXBlb2YgV2Vha1NldCA9PT0gJ3VuZGVmaW5lZCcgPyB1bmRlZmluZWQgOiBXZWFrU2V0XG59O1xuXG52YXIgZG9FdmFsID0gZnVuY3Rpb24gZG9FdmFsKG5hbWUpIHtcblx0dmFyIHZhbHVlO1xuXHRpZiAobmFtZSA9PT0gJyVBc3luY0Z1bmN0aW9uJScpIHtcblx0XHR2YWx1ZSA9IGdldEV2YWxsZWRDb25zdHJ1Y3RvcignYXN5bmMgZnVuY3Rpb24gKCkge30nKTtcblx0fSBlbHNlIGlmIChuYW1lID09PSAnJUdlbmVyYXRvckZ1bmN0aW9uJScpIHtcblx0XHR2YWx1ZSA9IGdldEV2YWxsZWRDb25zdHJ1Y3RvcignZnVuY3Rpb24qICgpIHt9Jyk7XG5cdH0gZWxzZSBpZiAobmFtZSA9PT0gJyVBc3luY0dlbmVyYXRvckZ1bmN0aW9uJScpIHtcblx0XHR2YWx1ZSA9IGdldEV2YWxsZWRDb25zdHJ1Y3RvcignYXN5bmMgZnVuY3Rpb24qICgpIHt9Jyk7XG5cdH0gZWxzZSBpZiAobmFtZSA9PT0gJyVBc3luY0dlbmVyYXRvciUnKSB7XG5cdFx0dmFyIGZuID0gZG9FdmFsKCclQXN5bmNHZW5lcmF0b3JGdW5jdGlvbiUnKTtcblx0XHRpZiAoZm4pIHtcblx0XHRcdHZhbHVlID0gZm4ucHJvdG90eXBlO1xuXHRcdH1cblx0fSBlbHNlIGlmIChuYW1lID09PSAnJUFzeW5jSXRlcmF0b3JQcm90b3R5cGUlJykge1xuXHRcdHZhciBnZW4gPSBkb0V2YWwoJyVBc3luY0dlbmVyYXRvciUnKTtcblx0XHRpZiAoZ2VuKSB7XG5cdFx0XHR2YWx1ZSA9IGdldFByb3RvKGdlbi5wcm90b3R5cGUpO1xuXHRcdH1cblx0fVxuXG5cdElOVFJJTlNJQ1NbbmFtZV0gPSB2YWx1ZTtcblxuXHRyZXR1cm4gdmFsdWU7XG59O1xuXG52YXIgTEVHQUNZX0FMSUFTRVMgPSB7XG5cdCclQXJyYXlCdWZmZXJQcm90b3R5cGUlJzogWydBcnJheUJ1ZmZlcicsICdwcm90b3R5cGUnXSxcblx0JyVBcnJheVByb3RvdHlwZSUnOiBbJ0FycmF5JywgJ3Byb3RvdHlwZSddLFxuXHQnJUFycmF5UHJvdG9fZW50cmllcyUnOiBbJ0FycmF5JywgJ3Byb3RvdHlwZScsICdlbnRyaWVzJ10sXG5cdCclQXJyYXlQcm90b19mb3JFYWNoJSc6IFsnQXJyYXknLCAncHJvdG90eXBlJywgJ2ZvckVhY2gnXSxcblx0JyVBcnJheVByb3RvX2tleXMlJzogWydBcnJheScsICdwcm90b3R5cGUnLCAna2V5cyddLFxuXHQnJUFycmF5UHJvdG9fdmFsdWVzJSc6IFsnQXJyYXknLCAncHJvdG90eXBlJywgJ3ZhbHVlcyddLFxuXHQnJUFzeW5jRnVuY3Rpb25Qcm90b3R5cGUlJzogWydBc3luY0Z1bmN0aW9uJywgJ3Byb3RvdHlwZSddLFxuXHQnJUFzeW5jR2VuZXJhdG9yJSc6IFsnQXN5bmNHZW5lcmF0b3JGdW5jdGlvbicsICdwcm90b3R5cGUnXSxcblx0JyVBc3luY0dlbmVyYXRvclByb3RvdHlwZSUnOiBbJ0FzeW5jR2VuZXJhdG9yRnVuY3Rpb24nLCAncHJvdG90eXBlJywgJ3Byb3RvdHlwZSddLFxuXHQnJUJvb2xlYW5Qcm90b3R5cGUlJzogWydCb29sZWFuJywgJ3Byb3RvdHlwZSddLFxuXHQnJURhdGFWaWV3UHJvdG90eXBlJSc6IFsnRGF0YVZpZXcnLCAncHJvdG90eXBlJ10sXG5cdCclRGF0ZVByb3RvdHlwZSUnOiBbJ0RhdGUnLCAncHJvdG90eXBlJ10sXG5cdCclRXJyb3JQcm90b3R5cGUlJzogWydFcnJvcicsICdwcm90b3R5cGUnXSxcblx0JyVFdmFsRXJyb3JQcm90b3R5cGUlJzogWydFdmFsRXJyb3InLCAncHJvdG90eXBlJ10sXG5cdCclRmxvYXQzMkFycmF5UHJvdG90eXBlJSc6IFsnRmxvYXQzMkFycmF5JywgJ3Byb3RvdHlwZSddLFxuXHQnJUZsb2F0NjRBcnJheVByb3RvdHlwZSUnOiBbJ0Zsb2F0NjRBcnJheScsICdwcm90b3R5cGUnXSxcblx0JyVGdW5jdGlvblByb3RvdHlwZSUnOiBbJ0Z1bmN0aW9uJywgJ3Byb3RvdHlwZSddLFxuXHQnJUdlbmVyYXRvciUnOiBbJ0dlbmVyYXRvckZ1bmN0aW9uJywgJ3Byb3RvdHlwZSddLFxuXHQnJUdlbmVyYXRvclByb3RvdHlwZSUnOiBbJ0dlbmVyYXRvckZ1bmN0aW9uJywgJ3Byb3RvdHlwZScsICdwcm90b3R5cGUnXSxcblx0JyVJbnQ4QXJyYXlQcm90b3R5cGUlJzogWydJbnQ4QXJyYXknLCAncHJvdG90eXBlJ10sXG5cdCclSW50MTZBcnJheVByb3RvdHlwZSUnOiBbJ0ludDE2QXJyYXknLCAncHJvdG90eXBlJ10sXG5cdCclSW50MzJBcnJheVByb3RvdHlwZSUnOiBbJ0ludDMyQXJyYXknLCAncHJvdG90eXBlJ10sXG5cdCclSlNPTlBhcnNlJSc6IFsnSlNPTicsICdwYXJzZSddLFxuXHQnJUpTT05TdHJpbmdpZnklJzogWydKU09OJywgJ3N0cmluZ2lmeSddLFxuXHQnJU1hcFByb3RvdHlwZSUnOiBbJ01hcCcsICdwcm90b3R5cGUnXSxcblx0JyVOdW1iZXJQcm90b3R5cGUlJzogWydOdW1iZXInLCAncHJvdG90eXBlJ10sXG5cdCclT2JqZWN0UHJvdG90eXBlJSc6IFsnT2JqZWN0JywgJ3Byb3RvdHlwZSddLFxuXHQnJU9ialByb3RvX3RvU3RyaW5nJSc6IFsnT2JqZWN0JywgJ3Byb3RvdHlwZScsICd0b1N0cmluZyddLFxuXHQnJU9ialByb3RvX3ZhbHVlT2YlJzogWydPYmplY3QnLCAncHJvdG90eXBlJywgJ3ZhbHVlT2YnXSxcblx0JyVQcm9taXNlUHJvdG90eXBlJSc6IFsnUHJvbWlzZScsICdwcm90b3R5cGUnXSxcblx0JyVQcm9taXNlUHJvdG9fdGhlbiUnOiBbJ1Byb21pc2UnLCAncHJvdG90eXBlJywgJ3RoZW4nXSxcblx0JyVQcm9taXNlX2FsbCUnOiBbJ1Byb21pc2UnLCAnYWxsJ10sXG5cdCclUHJvbWlzZV9yZWplY3QlJzogWydQcm9taXNlJywgJ3JlamVjdCddLFxuXHQnJVByb21pc2VfcmVzb2x2ZSUnOiBbJ1Byb21pc2UnLCAncmVzb2x2ZSddLFxuXHQnJVJhbmdlRXJyb3JQcm90b3R5cGUlJzogWydSYW5nZUVycm9yJywgJ3Byb3RvdHlwZSddLFxuXHQnJVJlZmVyZW5jZUVycm9yUHJvdG90eXBlJSc6IFsnUmVmZXJlbmNlRXJyb3InLCAncHJvdG90eXBlJ10sXG5cdCclUmVnRXhwUHJvdG90eXBlJSc6IFsnUmVnRXhwJywgJ3Byb3RvdHlwZSddLFxuXHQnJVNldFByb3RvdHlwZSUnOiBbJ1NldCcsICdwcm90b3R5cGUnXSxcblx0JyVTaGFyZWRBcnJheUJ1ZmZlclByb3RvdHlwZSUnOiBbJ1NoYXJlZEFycmF5QnVmZmVyJywgJ3Byb3RvdHlwZSddLFxuXHQnJVN0cmluZ1Byb3RvdHlwZSUnOiBbJ1N0cmluZycsICdwcm90b3R5cGUnXSxcblx0JyVTeW1ib2xQcm90b3R5cGUlJzogWydTeW1ib2wnLCAncHJvdG90eXBlJ10sXG5cdCclU3ludGF4RXJyb3JQcm90b3R5cGUlJzogWydTeW50YXhFcnJvcicsICdwcm90b3R5cGUnXSxcblx0JyVUeXBlZEFycmF5UHJvdG90eXBlJSc6IFsnVHlwZWRBcnJheScsICdwcm90b3R5cGUnXSxcblx0JyVUeXBlRXJyb3JQcm90b3R5cGUlJzogWydUeXBlRXJyb3InLCAncHJvdG90eXBlJ10sXG5cdCclVWludDhBcnJheVByb3RvdHlwZSUnOiBbJ1VpbnQ4QXJyYXknLCAncHJvdG90eXBlJ10sXG5cdCclVWludDhDbGFtcGVkQXJyYXlQcm90b3R5cGUlJzogWydVaW50OENsYW1wZWRBcnJheScsICdwcm90b3R5cGUnXSxcblx0JyVVaW50MTZBcnJheVByb3RvdHlwZSUnOiBbJ1VpbnQxNkFycmF5JywgJ3Byb3RvdHlwZSddLFxuXHQnJVVpbnQzMkFycmF5UHJvdG90eXBlJSc6IFsnVWludDMyQXJyYXknLCAncHJvdG90eXBlJ10sXG5cdCclVVJJRXJyb3JQcm90b3R5cGUlJzogWydVUklFcnJvcicsICdwcm90b3R5cGUnXSxcblx0JyVXZWFrTWFwUHJvdG90eXBlJSc6IFsnV2Vha01hcCcsICdwcm90b3R5cGUnXSxcblx0JyVXZWFrU2V0UHJvdG90eXBlJSc6IFsnV2Vha1NldCcsICdwcm90b3R5cGUnXVxufTtcblxudmFyIGJpbmQgPSByZXF1aXJlKCdmdW5jdGlvbi1iaW5kJyk7XG52YXIgaGFzT3duID0gcmVxdWlyZSgnaGFzJyk7XG52YXIgJGNvbmNhdCA9IGJpbmQuY2FsbChGdW5jdGlvbi5jYWxsLCBBcnJheS5wcm90b3R5cGUuY29uY2F0KTtcbnZhciAkc3BsaWNlQXBwbHkgPSBiaW5kLmNhbGwoRnVuY3Rpb24uYXBwbHksIEFycmF5LnByb3RvdHlwZS5zcGxpY2UpO1xudmFyICRyZXBsYWNlID0gYmluZC5jYWxsKEZ1bmN0aW9uLmNhbGwsIFN0cmluZy5wcm90b3R5cGUucmVwbGFjZSk7XG52YXIgJHN0clNsaWNlID0gYmluZC5jYWxsKEZ1bmN0aW9uLmNhbGwsIFN0cmluZy5wcm90b3R5cGUuc2xpY2UpO1xuXG4vKiBhZGFwdGVkIGZyb20gaHR0cHM6Ly9naXRodWIuY29tL2xvZGFzaC9sb2Rhc2gvYmxvYi80LjE3LjE1L2Rpc3QvbG9kYXNoLmpzI0w2NzM1LUw2NzQ0ICovXG52YXIgcmVQcm9wTmFtZSA9IC9bXiUuW1xcXV0rfFxcWyg/OigtP1xcZCsoPzpcXC5cXGQrKT8pfChbXCInXSkoKD86KD8hXFwyKVteXFxcXF18XFxcXC4pKj8pXFwyKVxcXXwoPz0oPzpcXC58XFxbXFxdKSg/OlxcLnxcXFtcXF18JSQpKS9nO1xudmFyIHJlRXNjYXBlQ2hhciA9IC9cXFxcKFxcXFwpPy9nOyAvKiogVXNlZCB0byBtYXRjaCBiYWNrc2xhc2hlcyBpbiBwcm9wZXJ0eSBwYXRocy4gKi9cbnZhciBzdHJpbmdUb1BhdGggPSBmdW5jdGlvbiBzdHJpbmdUb1BhdGgoc3RyaW5nKSB7XG5cdHZhciBmaXJzdCA9ICRzdHJTbGljZShzdHJpbmcsIDAsIDEpO1xuXHR2YXIgbGFzdCA9ICRzdHJTbGljZShzdHJpbmcsIC0xKTtcblx0aWYgKGZpcnN0ID09PSAnJScgJiYgbGFzdCAhPT0gJyUnKSB7XG5cdFx0dGhyb3cgbmV3ICRTeW50YXhFcnJvcignaW52YWxpZCBpbnRyaW5zaWMgc3ludGF4LCBleHBlY3RlZCBjbG9zaW5nIGAlYCcpO1xuXHR9IGVsc2UgaWYgKGxhc3QgPT09ICclJyAmJiBmaXJzdCAhPT0gJyUnKSB7XG5cdFx0dGhyb3cgbmV3ICRTeW50YXhFcnJvcignaW52YWxpZCBpbnRyaW5zaWMgc3ludGF4LCBleHBlY3RlZCBvcGVuaW5nIGAlYCcpO1xuXHR9XG5cdHZhciByZXN1bHQgPSBbXTtcblx0JHJlcGxhY2Uoc3RyaW5nLCByZVByb3BOYW1lLCBmdW5jdGlvbiAobWF0Y2gsIG51bWJlciwgcXVvdGUsIHN1YlN0cmluZykge1xuXHRcdHJlc3VsdFtyZXN1bHQubGVuZ3RoXSA9IHF1b3RlID8gJHJlcGxhY2Uoc3ViU3RyaW5nLCByZUVzY2FwZUNoYXIsICckMScpIDogbnVtYmVyIHx8IG1hdGNoO1xuXHR9KTtcblx0cmV0dXJuIHJlc3VsdDtcbn07XG4vKiBlbmQgYWRhcHRhdGlvbiAqL1xuXG52YXIgZ2V0QmFzZUludHJpbnNpYyA9IGZ1bmN0aW9uIGdldEJhc2VJbnRyaW5zaWMobmFtZSwgYWxsb3dNaXNzaW5nKSB7XG5cdHZhciBpbnRyaW5zaWNOYW1lID0gbmFtZTtcblx0dmFyIGFsaWFzO1xuXHRpZiAoaGFzT3duKExFR0FDWV9BTElBU0VTLCBpbnRyaW5zaWNOYW1lKSkge1xuXHRcdGFsaWFzID0gTEVHQUNZX0FMSUFTRVNbaW50cmluc2ljTmFtZV07XG5cdFx0aW50cmluc2ljTmFtZSA9ICclJyArIGFsaWFzWzBdICsgJyUnO1xuXHR9XG5cblx0aWYgKGhhc093bihJTlRSSU5TSUNTLCBpbnRyaW5zaWNOYW1lKSkge1xuXHRcdHZhciB2YWx1ZSA9IElOVFJJTlNJQ1NbaW50cmluc2ljTmFtZV07XG5cdFx0aWYgKHZhbHVlID09PSBuZWVkc0V2YWwpIHtcblx0XHRcdHZhbHVlID0gZG9FdmFsKGludHJpbnNpY05hbWUpO1xuXHRcdH1cblx0XHRpZiAodHlwZW9mIHZhbHVlID09PSAndW5kZWZpbmVkJyAmJiAhYWxsb3dNaXNzaW5nKSB7XG5cdFx0XHR0aHJvdyBuZXcgJFR5cGVFcnJvcignaW50cmluc2ljICcgKyBuYW1lICsgJyBleGlzdHMsIGJ1dCBpcyBub3QgYXZhaWxhYmxlLiBQbGVhc2UgZmlsZSBhbiBpc3N1ZSEnKTtcblx0XHR9XG5cblx0XHRyZXR1cm4ge1xuXHRcdFx0YWxpYXM6IGFsaWFzLFxuXHRcdFx0bmFtZTogaW50cmluc2ljTmFtZSxcblx0XHRcdHZhbHVlOiB2YWx1ZVxuXHRcdH07XG5cdH1cblxuXHR0aHJvdyBuZXcgJFN5bnRheEVycm9yKCdpbnRyaW5zaWMgJyArIG5hbWUgKyAnIGRvZXMgbm90IGV4aXN0IScpO1xufTtcblxubW9kdWxlLmV4cG9ydHMgPSBmdW5jdGlvbiBHZXRJbnRyaW5zaWMobmFtZSwgYWxsb3dNaXNzaW5nKSB7XG5cdGlmICh0eXBlb2YgbmFtZSAhPT0gJ3N0cmluZycgfHwgbmFtZS5sZW5ndGggPT09IDApIHtcblx0XHR0aHJvdyBuZXcgJFR5cGVFcnJvcignaW50cmluc2ljIG5hbWUgbXVzdCBiZSBhIG5vbi1lbXB0eSBzdHJpbmcnKTtcblx0fVxuXHRpZiAoYXJndW1lbnRzLmxlbmd0aCA+IDEgJiYgdHlwZW9mIGFsbG93TWlzc2luZyAhPT0gJ2Jvb2xlYW4nKSB7XG5cdFx0dGhyb3cgbmV3ICRUeXBlRXJyb3IoJ1wiYWxsb3dNaXNzaW5nXCIgYXJndW1lbnQgbXVzdCBiZSBhIGJvb2xlYW4nKTtcblx0fVxuXG5cdHZhciBwYXJ0cyA9IHN0cmluZ1RvUGF0aChuYW1lKTtcblx0dmFyIGludHJpbnNpY0Jhc2VOYW1lID0gcGFydHMubGVuZ3RoID4gMCA/IHBhcnRzWzBdIDogJyc7XG5cblx0dmFyIGludHJpbnNpYyA9IGdldEJhc2VJbnRyaW5zaWMoJyUnICsgaW50cmluc2ljQmFzZU5hbWUgKyAnJScsIGFsbG93TWlzc2luZyk7XG5cdHZhciBpbnRyaW5zaWNSZWFsTmFtZSA9IGludHJpbnNpYy5uYW1lO1xuXHR2YXIgdmFsdWUgPSBpbnRyaW5zaWMudmFsdWU7XG5cdHZhciBza2lwRnVydGhlckNhY2hpbmcgPSBmYWxzZTtcblxuXHR2YXIgYWxpYXMgPSBpbnRyaW5zaWMuYWxpYXM7XG5cdGlmIChhbGlhcykge1xuXHRcdGludHJpbnNpY0Jhc2VOYW1lID0gYWxpYXNbMF07XG5cdFx0JHNwbGljZUFwcGx5KHBhcnRzLCAkY29uY2F0KFswLCAxXSwgYWxpYXMpKTtcblx0fVxuXG5cdGZvciAodmFyIGkgPSAxLCBpc093biA9IHRydWU7IGkgPCBwYXJ0cy5sZW5ndGg7IGkgKz0gMSkge1xuXHRcdHZhciBwYXJ0ID0gcGFydHNbaV07XG5cdFx0dmFyIGZpcnN0ID0gJHN0clNsaWNlKHBhcnQsIDAsIDEpO1xuXHRcdHZhciBsYXN0ID0gJHN0clNsaWNlKHBhcnQsIC0xKTtcblx0XHRpZiAoXG5cdFx0XHQoXG5cdFx0XHRcdChmaXJzdCA9PT0gJ1wiJyB8fCBmaXJzdCA9PT0gXCInXCIgfHwgZmlyc3QgPT09ICdgJylcblx0XHRcdFx0fHwgKGxhc3QgPT09ICdcIicgfHwgbGFzdCA9PT0gXCInXCIgfHwgbGFzdCA9PT0gJ2AnKVxuXHRcdFx0KVxuXHRcdFx0JiYgZmlyc3QgIT09IGxhc3Rcblx0XHQpIHtcblx0XHRcdHRocm93IG5ldyAkU3ludGF4RXJyb3IoJ3Byb3BlcnR5IG5hbWVzIHdpdGggcXVvdGVzIG11c3QgaGF2ZSBtYXRjaGluZyBxdW90ZXMnKTtcblx0XHR9XG5cdFx0aWYgKHBhcnQgPT09ICdjb25zdHJ1Y3RvcicgfHwgIWlzT3duKSB7XG5cdFx0XHRza2lwRnVydGhlckNhY2hpbmcgPSB0cnVlO1xuXHRcdH1cblxuXHRcdGludHJpbnNpY0Jhc2VOYW1lICs9ICcuJyArIHBhcnQ7XG5cdFx0aW50cmluc2ljUmVhbE5hbWUgPSAnJScgKyBpbnRyaW5zaWNCYXNlTmFtZSArICclJztcblxuXHRcdGlmIChoYXNPd24oSU5UUklOU0lDUywgaW50cmluc2ljUmVhbE5hbWUpKSB7XG5cdFx0XHR2YWx1ZSA9IElOVFJJTlNJQ1NbaW50cmluc2ljUmVhbE5hbWVdO1xuXHRcdH0gZWxzZSBpZiAodmFsdWUgIT0gbnVsbCkge1xuXHRcdFx0aWYgKCEocGFydCBpbiB2YWx1ZSkpIHtcblx0XHRcdFx0aWYgKCFhbGxvd01pc3NpbmcpIHtcblx0XHRcdFx0XHR0aHJvdyBuZXcgJFR5cGVFcnJvcignYmFzZSBpbnRyaW5zaWMgZm9yICcgKyBuYW1lICsgJyBleGlzdHMsIGJ1dCB0aGUgcHJvcGVydHkgaXMgbm90IGF2YWlsYWJsZS4nKTtcblx0XHRcdFx0fVxuXHRcdFx0XHRyZXR1cm4gdm9pZCB1bmRlZmluZWQ7XG5cdFx0XHR9XG5cdFx0XHRpZiAoJGdPUEQgJiYgKGkgKyAxKSA+PSBwYXJ0cy5sZW5ndGgpIHtcblx0XHRcdFx0dmFyIGRlc2MgPSAkZ09QRCh2YWx1ZSwgcGFydCk7XG5cdFx0XHRcdGlzT3duID0gISFkZXNjO1xuXG5cdFx0XHRcdC8vIEJ5IGNvbnZlbnRpb24sIHdoZW4gYSBkYXRhIHByb3BlcnR5IGlzIGNvbnZlcnRlZCB0byBhbiBhY2Nlc3NvclxuXHRcdFx0XHQvLyBwcm9wZXJ0eSB0byBlbXVsYXRlIGEgZGF0YSBwcm9wZXJ0eSB0aGF0IGRvZXMgbm90IHN1ZmZlciBmcm9tXG5cdFx0XHRcdC8vIHRoZSBvdmVycmlkZSBtaXN0YWtlLCB0aGF0IGFjY2Vzc29yJ3MgZ2V0dGVyIGlzIG1hcmtlZCB3aXRoXG5cdFx0XHRcdC8vIGFuIGBvcmlnaW5hbFZhbHVlYCBwcm9wZXJ0eS4gSGVyZSwgd2hlbiB3ZSBkZXRlY3QgdGhpcywgd2Vcblx0XHRcdFx0Ly8gdXBob2xkIHRoZSBpbGx1c2lvbiBieSBwcmV0ZW5kaW5nIHRvIHNlZSB0aGF0IG9yaWdpbmFsIGRhdGFcblx0XHRcdFx0Ly8gcHJvcGVydHksIGkuZS4sIHJldHVybmluZyB0aGUgdmFsdWUgcmF0aGVyIHRoYW4gdGhlIGdldHRlclxuXHRcdFx0XHQvLyBpdHNlbGYuXG5cdFx0XHRcdGlmIChpc093biAmJiAnZ2V0JyBpbiBkZXNjICYmICEoJ29yaWdpbmFsVmFsdWUnIGluIGRlc2MuZ2V0KSkge1xuXHRcdFx0XHRcdHZhbHVlID0gZGVzYy5nZXQ7XG5cdFx0XHRcdH0gZWxzZSB7XG5cdFx0XHRcdFx0dmFsdWUgPSB2YWx1ZVtwYXJ0XTtcblx0XHRcdFx0fVxuXHRcdFx0fSBlbHNlIHtcblx0XHRcdFx0aXNPd24gPSBoYXNPd24odmFsdWUsIHBhcnQpO1xuXHRcdFx0XHR2YWx1ZSA9IHZhbHVlW3BhcnRdO1xuXHRcdFx0fVxuXG5cdFx0XHRpZiAoaXNPd24gJiYgIXNraXBGdXJ0aGVyQ2FjaGluZykge1xuXHRcdFx0XHRJTlRSSU5TSUNTW2ludHJpbnNpY1JlYWxOYW1lXSA9IHZhbHVlO1xuXHRcdFx0fVxuXHRcdH1cblx0fVxuXHRyZXR1cm4gdmFsdWU7XG59O1xuIl0sInNvdXJjZVJvb3QiOiIifQ==\n//# sourceURL=webpack-internal:///210\n")},1405:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar origSymbol = typeof Symbol !== 'undefined' && Symbol;\nvar hasSymbolSham = __webpack_require__(5419);\n\nmodule.exports = function hasNativeSymbols() {\n\tif (typeof origSymbol !== 'function') { return false; }\n\tif (typeof Symbol !== 'function') { return false; }\n\tif (typeof origSymbol('foo') !== 'symbol') { return false; }\n\tif (typeof Symbol('bar') !== 'symbol') { return false; }\n\n\treturn hasSymbolSham();\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2hhcy1zeW1ib2xzL2luZGV4LmpzPzUxNTYiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQWE7O0FBRWI7QUFDQSxvQkFBb0IsbUJBQU8sQ0FBQyxJQUFTOztBQUVyQztBQUNBLHdDQUF3QyxjQUFjO0FBQ3RELG9DQUFvQyxjQUFjO0FBQ2xELDZDQUE2QyxjQUFjO0FBQzNELHlDQUF5QyxjQUFjOztBQUV2RDtBQUNBIiwiZmlsZSI6IjE0MDUuanMiLCJzb3VyY2VzQ29udGVudCI6WyIndXNlIHN0cmljdCc7XG5cbnZhciBvcmlnU3ltYm9sID0gdHlwZW9mIFN5bWJvbCAhPT0gJ3VuZGVmaW5lZCcgJiYgU3ltYm9sO1xudmFyIGhhc1N5bWJvbFNoYW0gPSByZXF1aXJlKCcuL3NoYW1zJyk7XG5cbm1vZHVsZS5leHBvcnRzID0gZnVuY3Rpb24gaGFzTmF0aXZlU3ltYm9scygpIHtcblx0aWYgKHR5cGVvZiBvcmlnU3ltYm9sICE9PSAnZnVuY3Rpb24nKSB7IHJldHVybiBmYWxzZTsgfVxuXHRpZiAodHlwZW9mIFN5bWJvbCAhPT0gJ2Z1bmN0aW9uJykgeyByZXR1cm4gZmFsc2U7IH1cblx0aWYgKHR5cGVvZiBvcmlnU3ltYm9sKCdmb28nKSAhPT0gJ3N5bWJvbCcpIHsgcmV0dXJuIGZhbHNlOyB9XG5cdGlmICh0eXBlb2YgU3ltYm9sKCdiYXInKSAhPT0gJ3N5bWJvbCcpIHsgcmV0dXJuIGZhbHNlOyB9XG5cblx0cmV0dXJuIGhhc1N5bWJvbFNoYW0oKTtcbn07XG4iXSwic291cmNlUm9vdCI6IiJ9\n//# sourceURL=webpack-internal:///1405\n")},5419:module=>{"use strict";eval("\n\n/* eslint complexity: [2, 18], max-statements: [2, 33] */\nmodule.exports = function hasSymbols() {\n\tif (typeof Symbol !== 'function' || typeof Object.getOwnPropertySymbols !== 'function') { return false; }\n\tif (typeof Symbol.iterator === 'symbol') { return true; }\n\n\tvar obj = {};\n\tvar sym = Symbol('test');\n\tvar symObj = Object(sym);\n\tif (typeof sym === 'string') { return false; }\n\n\tif (Object.prototype.toString.call(sym) !== '[object Symbol]') { return false; }\n\tif (Object.prototype.toString.call(symObj) !== '[object Symbol]') { return false; }\n\n\t// temp disabled per https://github.com/ljharb/object.assign/issues/17\n\t// if (sym instanceof Symbol) { return false; }\n\t// temp disabled per https://github.com/WebReflection/get-own-property-symbols/issues/4\n\t// if (!(symObj instanceof Symbol)) { return false; }\n\n\t// if (typeof Symbol.prototype.toString !== 'function') { return false; }\n\t// if (String(sym) !== Symbol.prototype.toString.call(sym)) { return false; }\n\n\tvar symVal = 42;\n\tobj[sym] = symVal;\n\tfor (sym in obj) { return false; } // eslint-disable-line no-restricted-syntax, no-unreachable-loop\n\tif (typeof Object.keys === 'function' && Object.keys(obj).length !== 0) { return false; }\n\n\tif (typeof Object.getOwnPropertyNames === 'function' && Object.getOwnPropertyNames(obj).length !== 0) { return false; }\n\n\tvar syms = Object.getOwnPropertySymbols(obj);\n\tif (syms.length !== 1 || syms[0] !== sym) { return false; }\n\n\tif (!Object.prototype.propertyIsEnumerable.call(obj, sym)) { return false; }\n\n\tif (typeof Object.getOwnPropertyDescriptor === 'function') {\n\t\tvar descriptor = Object.getOwnPropertyDescriptor(obj, sym);\n\t\tif (descriptor.value !== symVal || descriptor.enumerable !== true) { return false; }\n\t}\n\n\treturn true;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///5419\n")},6410:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar hasSymbols = __webpack_require__(5419);\n\nmodule.exports = function hasToStringTagShams() {\n\treturn hasSymbols() && !!Symbol.toStringTag;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2hhcy10b3N0cmluZ3RhZy9zaGFtcy5qcz8wN2E0Il0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFhOztBQUViLGlCQUFpQixtQkFBTyxDQUFDLElBQW1COztBQUU1QztBQUNBO0FBQ0EiLCJmaWxlIjoiNjQxMC5qcyIsInNvdXJjZXNDb250ZW50IjpbIid1c2Ugc3RyaWN0JztcblxudmFyIGhhc1N5bWJvbHMgPSByZXF1aXJlKCdoYXMtc3ltYm9scy9zaGFtcycpO1xuXG5tb2R1bGUuZXhwb3J0cyA9IGZ1bmN0aW9uIGhhc1RvU3RyaW5nVGFnU2hhbXMoKSB7XG5cdHJldHVybiBoYXNTeW1ib2xzKCkgJiYgISFTeW1ib2wudG9TdHJpbmdUYWc7XG59O1xuIl0sInNvdXJjZVJvb3QiOiIifQ==\n//# sourceURL=webpack-internal:///6410\n")},7642:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar bind = __webpack_require__(8612);\n\nmodule.exports = bind.call(Function.call, Object.prototype.hasOwnProperty);\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2hhcy9zcmMvaW5kZXguanM/YTBkMyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBYTs7QUFFYixXQUFXLG1CQUFPLENBQUMsSUFBZTs7QUFFbEMiLCJmaWxlIjoiNzY0Mi5qcyIsInNvdXJjZXNDb250ZW50IjpbIid1c2Ugc3RyaWN0JztcblxudmFyIGJpbmQgPSByZXF1aXJlKCdmdW5jdGlvbi1iaW5kJyk7XG5cbm1vZHVsZS5leHBvcnRzID0gYmluZC5jYWxsKEZ1bmN0aW9uLmNhbGwsIE9iamVjdC5wcm90b3R5cGUuaGFzT3duUHJvcGVydHkpO1xuIl0sInNvdXJjZVJvb3QiOiIifQ==\n//# sourceURL=webpack-internal:///7642\n")},3715:(__unused_webpack_module,exports,__webpack_require__)=>{eval("var hash = exports;\n\nhash.utils = __webpack_require__(6436);\nhash.common = __webpack_require__(5772);\nhash.sha = __webpack_require__(9041);\nhash.ripemd = __webpack_require__(2949);\nhash.hmac = __webpack_require__(2344);\n\n// Proxy hash functions to the main object\nhash.sha1 = hash.sha.sha1;\nhash.sha256 = hash.sha.sha256;\nhash.sha224 = hash.sha.sha224;\nhash.sha384 = hash.sha.sha384;\nhash.sha512 = hash.sha.sha512;\nhash.ripemd160 = hash.ripemd.ripemd160;\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2hhc2guanMvbGliL2hhc2guanM/N2Q5MiJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBQTs7QUFFQSxhQUFhLG1CQUFPLENBQUMsSUFBYztBQUNuQyxjQUFjLG1CQUFPLENBQUMsSUFBZTtBQUNyQyxXQUFXLG1CQUFPLENBQUMsSUFBWTtBQUMvQixjQUFjLG1CQUFPLENBQUMsSUFBZTtBQUNyQyxZQUFZLG1CQUFPLENBQUMsSUFBYTs7QUFFakM7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0EiLCJmaWxlIjoiMzcxNS5qcyIsInNvdXJjZXNDb250ZW50IjpbInZhciBoYXNoID0gZXhwb3J0cztcblxuaGFzaC51dGlscyA9IHJlcXVpcmUoJy4vaGFzaC91dGlscycpO1xuaGFzaC5jb21tb24gPSByZXF1aXJlKCcuL2hhc2gvY29tbW9uJyk7XG5oYXNoLnNoYSA9IHJlcXVpcmUoJy4vaGFzaC9zaGEnKTtcbmhhc2gucmlwZW1kID0gcmVxdWlyZSgnLi9oYXNoL3JpcGVtZCcpO1xuaGFzaC5obWFjID0gcmVxdWlyZSgnLi9oYXNoL2htYWMnKTtcblxuLy8gUHJveHkgaGFzaCBmdW5jdGlvbnMgdG8gdGhlIG1haW4gb2JqZWN0XG5oYXNoLnNoYTEgPSBoYXNoLnNoYS5zaGExO1xuaGFzaC5zaGEyNTYgPSBoYXNoLnNoYS5zaGEyNTY7XG5oYXNoLnNoYTIyNCA9IGhhc2guc2hhLnNoYTIyNDtcbmhhc2guc2hhMzg0ID0gaGFzaC5zaGEuc2hhMzg0O1xuaGFzaC5zaGE1MTIgPSBoYXNoLnNoYS5zaGE1MTI7XG5oYXNoLnJpcGVtZDE2MCA9IGhhc2gucmlwZW1kLnJpcGVtZDE2MDtcbiJdLCJzb3VyY2VSb290IjoiIn0=\n//# sourceURL=webpack-internal:///3715\n")},5772:(__unused_webpack_module,exports,__webpack_require__)=>{"use strict";eval("\n\nvar utils = __webpack_require__(6436);\nvar assert = __webpack_require__(9746);\n\nfunction BlockHash() {\n  this.pending = null;\n  this.pendingTotal = 0;\n  this.blockSize = this.constructor.blockSize;\n  this.outSize = this.constructor.outSize;\n  this.hmacStrength = this.constructor.hmacStrength;\n  this.padLength = this.constructor.padLength / 8;\n  this.endian = 'big';\n\n  this._delta8 = this.blockSize / 8;\n  this._delta32 = this.blockSize / 32;\n}\nexports.BlockHash = BlockHash;\n\nBlockHash.prototype.update = function update(msg, enc) {\n  // Convert message to array, pad it, and join into 32bit blocks\n  msg = utils.toArray(msg, enc);\n  if (!this.pending)\n    this.pending = msg;\n  else\n    this.pending = this.pending.concat(msg);\n  this.pendingTotal += msg.length;\n\n  // Enough data, try updating\n  if (this.pending.length >= this._delta8) {\n    msg = this.pending;\n\n    // Process pending data in blocks\n    var r = msg.length % this._delta8;\n    this.pending = msg.slice(msg.length - r, msg.length);\n    if (this.pending.length === 0)\n      this.pending = null;\n\n    msg = utils.join32(msg, 0, msg.length - r, this.endian);\n    for (var i = 0; i < msg.length; i += this._delta32)\n      this._update(msg, i, i + this._delta32);\n  }\n\n  return this;\n};\n\nBlockHash.prototype.digest = function digest(enc) {\n  this.update(this._pad());\n  assert(this.pending === null);\n\n  return this._digest(enc);\n};\n\nBlockHash.prototype._pad = function pad() {\n  var len = this.pendingTotal;\n  var bytes = this._delta8;\n  var k = bytes - ((len + this.padLength) % bytes);\n  var res = new Array(k + this.padLength);\n  res[0] = 0x80;\n  for (var i = 1; i < k; i++)\n    res[i] = 0;\n\n  // Append length\n  len <<= 3;\n  if (this.endian === 'big') {\n    for (var t = 8; t < this.padLength; t++)\n      res[i++] = 0;\n\n    res[i++] = 0;\n    res[i++] = 0;\n    res[i++] = 0;\n    res[i++] = 0;\n    res[i++] = (len >>> 24) & 0xff;\n    res[i++] = (len >>> 16) & 0xff;\n    res[i++] = (len >>> 8) & 0xff;\n    res[i++] = len & 0xff;\n  } else {\n    res[i++] = len & 0xff;\n    res[i++] = (len >>> 8) & 0xff;\n    res[i++] = (len >>> 16) & 0xff;\n    res[i++] = (len >>> 24) & 0xff;\n    res[i++] = 0;\n    res[i++] = 0;\n    res[i++] = 0;\n    res[i++] = 0;\n\n    for (t = 8; t < this.padLength; t++)\n      res[i++] = 0;\n  }\n\n  return res;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///5772\n")},2344:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar utils = __webpack_require__(6436);\nvar assert = __webpack_require__(9746);\n\nfunction Hmac(hash, key, enc) {\n  if (!(this instanceof Hmac))\n    return new Hmac(hash, key, enc);\n  this.Hash = hash;\n  this.blockSize = hash.blockSize / 8;\n  this.outSize = hash.outSize / 8;\n  this.inner = null;\n  this.outer = null;\n\n  this._init(utils.toArray(key, enc));\n}\nmodule.exports = Hmac;\n\nHmac.prototype._init = function init(key) {\n  // Shorten key, if needed\n  if (key.length > this.blockSize)\n    key = new this.Hash().update(key).digest();\n  assert(key.length <= this.blockSize);\n\n  // Add padding to key\n  for (var i = key.length; i < this.blockSize; i++)\n    key.push(0);\n\n  for (i = 0; i < key.length; i++)\n    key[i] ^= 0x36;\n  this.inner = new this.Hash().update(key);\n\n  // 0x36 ^ 0x5c = 0x6a\n  for (i = 0; i < key.length; i++)\n    key[i] ^= 0x6a;\n  this.outer = new this.Hash().update(key);\n};\n\nHmac.prototype.update = function update(msg, enc) {\n  this.inner.update(msg, enc);\n  return this;\n};\n\nHmac.prototype.digest = function digest(enc) {\n  this.outer.update(this.inner.digest());\n  return this.outer.digest(enc);\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///2344\n")},2949:(__unused_webpack_module,exports,__webpack_require__)=>{"use strict";eval("\n\nvar utils = __webpack_require__(6436);\nvar common = __webpack_require__(5772);\n\nvar rotl32 = utils.rotl32;\nvar sum32 = utils.sum32;\nvar sum32_3 = utils.sum32_3;\nvar sum32_4 = utils.sum32_4;\nvar BlockHash = common.BlockHash;\n\nfunction RIPEMD160() {\n  if (!(this instanceof RIPEMD160))\n    return new RIPEMD160();\n\n  BlockHash.call(this);\n\n  this.h = [ 0x67452301, 0xefcdab89, 0x98badcfe, 0x10325476, 0xc3d2e1f0 ];\n  this.endian = 'little';\n}\nutils.inherits(RIPEMD160, BlockHash);\nexports.ripemd160 = RIPEMD160;\n\nRIPEMD160.blockSize = 512;\nRIPEMD160.outSize = 160;\nRIPEMD160.hmacStrength = 192;\nRIPEMD160.padLength = 64;\n\nRIPEMD160.prototype._update = function update(msg, start) {\n  var A = this.h[0];\n  var B = this.h[1];\n  var C = this.h[2];\n  var D = this.h[3];\n  var E = this.h[4];\n  var Ah = A;\n  var Bh = B;\n  var Ch = C;\n  var Dh = D;\n  var Eh = E;\n  for (var j = 0; j < 80; j++) {\n    var T = sum32(\n      rotl32(\n        sum32_4(A, f(j, B, C, D), msg[r[j] + start], K(j)),\n        s[j]),\n      E);\n    A = E;\n    E = D;\n    D = rotl32(C, 10);\n    C = B;\n    B = T;\n    T = sum32(\n      rotl32(\n        sum32_4(Ah, f(79 - j, Bh, Ch, Dh), msg[rh[j] + start], Kh(j)),\n        sh[j]),\n      Eh);\n    Ah = Eh;\n    Eh = Dh;\n    Dh = rotl32(Ch, 10);\n    Ch = Bh;\n    Bh = T;\n  }\n  T = sum32_3(this.h[1], C, Dh);\n  this.h[1] = sum32_3(this.h[2], D, Eh);\n  this.h[2] = sum32_3(this.h[3], E, Ah);\n  this.h[3] = sum32_3(this.h[4], A, Bh);\n  this.h[4] = sum32_3(this.h[0], B, Ch);\n  this.h[0] = T;\n};\n\nRIPEMD160.prototype._digest = function digest(enc) {\n  if (enc === 'hex')\n    return utils.toHex32(this.h, 'little');\n  else\n    return utils.split32(this.h, 'little');\n};\n\nfunction f(j, x, y, z) {\n  if (j <= 15)\n    return x ^ y ^ z;\n  else if (j <= 31)\n    return (x & y) | ((~x) & z);\n  else if (j <= 47)\n    return (x | (~y)) ^ z;\n  else if (j <= 63)\n    return (x & z) | (y & (~z));\n  else\n    return x ^ (y | (~z));\n}\n\nfunction K(j) {\n  if (j <= 15)\n    return 0x00000000;\n  else if (j <= 31)\n    return 0x5a827999;\n  else if (j <= 47)\n    return 0x6ed9eba1;\n  else if (j <= 63)\n    return 0x8f1bbcdc;\n  else\n    return 0xa953fd4e;\n}\n\nfunction Kh(j) {\n  if (j <= 15)\n    return 0x50a28be6;\n  else if (j <= 31)\n    return 0x5c4dd124;\n  else if (j <= 47)\n    return 0x6d703ef3;\n  else if (j <= 63)\n    return 0x7a6d76e9;\n  else\n    return 0x00000000;\n}\n\nvar r = [\n  0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15,\n  7, 4, 13, 1, 10, 6, 15, 3, 12, 0, 9, 5, 2, 14, 11, 8,\n  3, 10, 14, 4, 9, 15, 8, 1, 2, 7, 0, 6, 13, 11, 5, 12,\n  1, 9, 11, 10, 0, 8, 12, 4, 13, 3, 7, 15, 14, 5, 6, 2,\n  4, 0, 5, 9, 7, 12, 2, 10, 14, 1, 3, 8, 11, 6, 15, 13\n];\n\nvar rh = [\n  5, 14, 7, 0, 9, 2, 11, 4, 13, 6, 15, 8, 1, 10, 3, 12,\n  6, 11, 3, 7, 0, 13, 5, 10, 14, 15, 8, 12, 4, 9, 1, 2,\n  15, 5, 1, 3, 7, 14, 6, 9, 11, 8, 12, 2, 10, 0, 4, 13,\n  8, 6, 4, 1, 3, 11, 15, 0, 5, 12, 2, 13, 9, 7, 10, 14,\n  12, 15, 10, 4, 1, 5, 8, 7, 6, 2, 13, 14, 0, 3, 9, 11\n];\n\nvar s = [\n  11, 14, 15, 12, 5, 8, 7, 9, 11, 13, 14, 15, 6, 7, 9, 8,\n  7, 6, 8, 13, 11, 9, 7, 15, 7, 12, 15, 9, 11, 7, 13, 12,\n  11, 13, 6, 7, 14, 9, 13, 15, 14, 8, 13, 6, 5, 12, 7, 5,\n  11, 12, 14, 15, 14, 15, 9, 8, 9, 14, 5, 6, 8, 6, 5, 12,\n  9, 15, 5, 11, 6, 8, 13, 12, 5, 12, 13, 14, 11, 8, 5, 6\n];\n\nvar sh = [\n  8, 9, 9, 11, 13, 15, 15, 5, 7, 7, 8, 11, 14, 14, 12, 6,\n  9, 13, 15, 7, 12, 8, 9, 11, 7, 7, 12, 7, 6, 15, 13, 11,\n  9, 7, 15, 11, 8, 6, 6, 14, 12, 13, 5, 14, 13, 13, 7, 5,\n  15, 5, 8, 11, 14, 14, 6, 14, 6, 9, 12, 9, 12, 5, 15, 8,\n  8, 5, 12, 9, 12, 5, 14, 6, 8, 13, 6, 5, 15, 13, 11, 11\n];\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///2949\n")},9041:(__unused_webpack_module,exports,__webpack_require__)=>{"use strict";eval("\n\nexports.sha1 = __webpack_require__(4761);\nexports.sha224 = __webpack_require__(799);\nexports.sha256 = __webpack_require__(9344);\nexports.sha384 = __webpack_require__(772);\nexports.sha512 = __webpack_require__(5900);\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2hhc2guanMvbGliL2hhc2gvc2hhLmpzPzU5MTkiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQWE7O0FBRWIsd0NBQWlDO0FBQ2pDLHlDQUFxQztBQUNyQywwQ0FBcUM7QUFDckMseUNBQXFDO0FBQ3JDLDBDQUFxQyIsImZpbGUiOiI5MDQxLmpzIiwic291cmNlc0NvbnRlbnQiOlsiJ3VzZSBzdHJpY3QnO1xuXG5leHBvcnRzLnNoYTEgPSByZXF1aXJlKCcuL3NoYS8xJyk7XG5leHBvcnRzLnNoYTIyNCA9IHJlcXVpcmUoJy4vc2hhLzIyNCcpO1xuZXhwb3J0cy5zaGEyNTYgPSByZXF1aXJlKCcuL3NoYS8yNTYnKTtcbmV4cG9ydHMuc2hhMzg0ID0gcmVxdWlyZSgnLi9zaGEvMzg0Jyk7XG5leHBvcnRzLnNoYTUxMiA9IHJlcXVpcmUoJy4vc2hhLzUxMicpO1xuIl0sInNvdXJjZVJvb3QiOiIifQ==\n//# sourceURL=webpack-internal:///9041\n")},4761:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar utils = __webpack_require__(6436);\nvar common = __webpack_require__(5772);\nvar shaCommon = __webpack_require__(7038);\n\nvar rotl32 = utils.rotl32;\nvar sum32 = utils.sum32;\nvar sum32_5 = utils.sum32_5;\nvar ft_1 = shaCommon.ft_1;\nvar BlockHash = common.BlockHash;\n\nvar sha1_K = [\n  0x5A827999, 0x6ED9EBA1,\n  0x8F1BBCDC, 0xCA62C1D6\n];\n\nfunction SHA1() {\n  if (!(this instanceof SHA1))\n    return new SHA1();\n\n  BlockHash.call(this);\n  this.h = [\n    0x67452301, 0xefcdab89, 0x98badcfe,\n    0x10325476, 0xc3d2e1f0 ];\n  this.W = new Array(80);\n}\n\nutils.inherits(SHA1, BlockHash);\nmodule.exports = SHA1;\n\nSHA1.blockSize = 512;\nSHA1.outSize = 160;\nSHA1.hmacStrength = 80;\nSHA1.padLength = 64;\n\nSHA1.prototype._update = function _update(msg, start) {\n  var W = this.W;\n\n  for (var i = 0; i < 16; i++)\n    W[i] = msg[start + i];\n\n  for(; i < W.length; i++)\n    W[i] = rotl32(W[i - 3] ^ W[i - 8] ^ W[i - 14] ^ W[i - 16], 1);\n\n  var a = this.h[0];\n  var b = this.h[1];\n  var c = this.h[2];\n  var d = this.h[3];\n  var e = this.h[4];\n\n  for (i = 0; i < W.length; i++) {\n    var s = ~~(i / 20);\n    var t = sum32_5(rotl32(a, 5), ft_1(s, b, c, d), e, W[i], sha1_K[s]);\n    e = d;\n    d = c;\n    c = rotl32(b, 30);\n    b = a;\n    a = t;\n  }\n\n  this.h[0] = sum32(this.h[0], a);\n  this.h[1] = sum32(this.h[1], b);\n  this.h[2] = sum32(this.h[2], c);\n  this.h[3] = sum32(this.h[3], d);\n  this.h[4] = sum32(this.h[4], e);\n};\n\nSHA1.prototype._digest = function digest(enc) {\n  if (enc === 'hex')\n    return utils.toHex32(this.h, 'big');\n  else\n    return utils.split32(this.h, 'big');\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///4761\n")},799:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar utils = __webpack_require__(6436);\nvar SHA256 = __webpack_require__(9344);\n\nfunction SHA224() {\n  if (!(this instanceof SHA224))\n    return new SHA224();\n\n  SHA256.call(this);\n  this.h = [\n    0xc1059ed8, 0x367cd507, 0x3070dd17, 0xf70e5939,\n    0xffc00b31, 0x68581511, 0x64f98fa7, 0xbefa4fa4 ];\n}\nutils.inherits(SHA224, SHA256);\nmodule.exports = SHA224;\n\nSHA224.blockSize = 512;\nSHA224.outSize = 224;\nSHA224.hmacStrength = 192;\nSHA224.padLength = 64;\n\nSHA224.prototype._digest = function digest(enc) {\n  // Just truncate output\n  if (enc === 'hex')\n    return utils.toHex32(this.h.slice(0, 7), 'big');\n  else\n    return utils.split32(this.h.slice(0, 7), 'big');\n};\n\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2hhc2guanMvbGliL2hhc2gvc2hhLzIyNC5qcz8wN2YyIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFhOztBQUViLFlBQVksbUJBQU8sQ0FBQyxJQUFVO0FBQzlCLGFBQWEsbUJBQU8sQ0FBQyxJQUFPOztBQUU1QjtBQUNBO0FBQ0E7O0FBRUE7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7O0FBRUE7QUFDQTtBQUNBO0FBQ0E7O0FBRUE7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0EiLCJmaWxlIjoiNzk5LmpzIiwic291cmNlc0NvbnRlbnQiOlsiJ3VzZSBzdHJpY3QnO1xuXG52YXIgdXRpbHMgPSByZXF1aXJlKCcuLi91dGlscycpO1xudmFyIFNIQTI1NiA9IHJlcXVpcmUoJy4vMjU2Jyk7XG5cbmZ1bmN0aW9uIFNIQTIyNCgpIHtcbiAgaWYgKCEodGhpcyBpbnN0YW5jZW9mIFNIQTIyNCkpXG4gICAgcmV0dXJuIG5ldyBTSEEyMjQoKTtcblxuICBTSEEyNTYuY2FsbCh0aGlzKTtcbiAgdGhpcy5oID0gW1xuICAgIDB4YzEwNTllZDgsIDB4MzY3Y2Q1MDcsIDB4MzA3MGRkMTcsIDB4ZjcwZTU5MzksXG4gICAgMHhmZmMwMGIzMSwgMHg2ODU4MTUxMSwgMHg2NGY5OGZhNywgMHhiZWZhNGZhNCBdO1xufVxudXRpbHMuaW5oZXJpdHMoU0hBMjI0LCBTSEEyNTYpO1xubW9kdWxlLmV4cG9ydHMgPSBTSEEyMjQ7XG5cblNIQTIyNC5ibG9ja1NpemUgPSA1MTI7XG5TSEEyMjQub3V0U2l6ZSA9IDIyNDtcblNIQTIyNC5obWFjU3RyZW5ndGggPSAxOTI7XG5TSEEyMjQucGFkTGVuZ3RoID0gNjQ7XG5cblNIQTIyNC5wcm90b3R5cGUuX2RpZ2VzdCA9IGZ1bmN0aW9uIGRpZ2VzdChlbmMpIHtcbiAgLy8gSnVzdCB0cnVuY2F0ZSBvdXRwdXRcbiAgaWYgKGVuYyA9PT0gJ2hleCcpXG4gICAgcmV0dXJuIHV0aWxzLnRvSGV4MzIodGhpcy5oLnNsaWNlKDAsIDcpLCAnYmlnJyk7XG4gIGVsc2VcbiAgICByZXR1cm4gdXRpbHMuc3BsaXQzMih0aGlzLmguc2xpY2UoMCwgNyksICdiaWcnKTtcbn07XG5cbiJdLCJzb3VyY2VSb290IjoiIn0=\n//# sourceURL=webpack-internal:///799\n")},9344:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar utils = __webpack_require__(6436);\nvar common = __webpack_require__(5772);\nvar shaCommon = __webpack_require__(7038);\nvar assert = __webpack_require__(9746);\n\nvar sum32 = utils.sum32;\nvar sum32_4 = utils.sum32_4;\nvar sum32_5 = utils.sum32_5;\nvar ch32 = shaCommon.ch32;\nvar maj32 = shaCommon.maj32;\nvar s0_256 = shaCommon.s0_256;\nvar s1_256 = shaCommon.s1_256;\nvar g0_256 = shaCommon.g0_256;\nvar g1_256 = shaCommon.g1_256;\n\nvar BlockHash = common.BlockHash;\n\nvar sha256_K = [\n  0x428a2f98, 0x71374491, 0xb5c0fbcf, 0xe9b5dba5,\n  0x3956c25b, 0x59f111f1, 0x923f82a4, 0xab1c5ed5,\n  0xd807aa98, 0x12835b01, 0x243185be, 0x550c7dc3,\n  0x72be5d74, 0x80deb1fe, 0x9bdc06a7, 0xc19bf174,\n  0xe49b69c1, 0xefbe4786, 0x0fc19dc6, 0x240ca1cc,\n  0x2de92c6f, 0x4a7484aa, 0x5cb0a9dc, 0x76f988da,\n  0x983e5152, 0xa831c66d, 0xb00327c8, 0xbf597fc7,\n  0xc6e00bf3, 0xd5a79147, 0x06ca6351, 0x14292967,\n  0x27b70a85, 0x2e1b2138, 0x4d2c6dfc, 0x53380d13,\n  0x650a7354, 0x766a0abb, 0x81c2c92e, 0x92722c85,\n  0xa2bfe8a1, 0xa81a664b, 0xc24b8b70, 0xc76c51a3,\n  0xd192e819, 0xd6990624, 0xf40e3585, 0x106aa070,\n  0x19a4c116, 0x1e376c08, 0x2748774c, 0x34b0bcb5,\n  0x391c0cb3, 0x4ed8aa4a, 0x5b9cca4f, 0x682e6ff3,\n  0x748f82ee, 0x78a5636f, 0x84c87814, 0x8cc70208,\n  0x90befffa, 0xa4506ceb, 0xbef9a3f7, 0xc67178f2\n];\n\nfunction SHA256() {\n  if (!(this instanceof SHA256))\n    return new SHA256();\n\n  BlockHash.call(this);\n  this.h = [\n    0x6a09e667, 0xbb67ae85, 0x3c6ef372, 0xa54ff53a,\n    0x510e527f, 0x9b05688c, 0x1f83d9ab, 0x5be0cd19\n  ];\n  this.k = sha256_K;\n  this.W = new Array(64);\n}\nutils.inherits(SHA256, BlockHash);\nmodule.exports = SHA256;\n\nSHA256.blockSize = 512;\nSHA256.outSize = 256;\nSHA256.hmacStrength = 192;\nSHA256.padLength = 64;\n\nSHA256.prototype._update = function _update(msg, start) {\n  var W = this.W;\n\n  for (var i = 0; i < 16; i++)\n    W[i] = msg[start + i];\n  for (; i < W.length; i++)\n    W[i] = sum32_4(g1_256(W[i - 2]), W[i - 7], g0_256(W[i - 15]), W[i - 16]);\n\n  var a = this.h[0];\n  var b = this.h[1];\n  var c = this.h[2];\n  var d = this.h[3];\n  var e = this.h[4];\n  var f = this.h[5];\n  var g = this.h[6];\n  var h = this.h[7];\n\n  assert(this.k.length === W.length);\n  for (i = 0; i < W.length; i++) {\n    var T1 = sum32_5(h, s1_256(e), ch32(e, f, g), this.k[i], W[i]);\n    var T2 = sum32(s0_256(a), maj32(a, b, c));\n    h = g;\n    g = f;\n    f = e;\n    e = sum32(d, T1);\n    d = c;\n    c = b;\n    b = a;\n    a = sum32(T1, T2);\n  }\n\n  this.h[0] = sum32(this.h[0], a);\n  this.h[1] = sum32(this.h[1], b);\n  this.h[2] = sum32(this.h[2], c);\n  this.h[3] = sum32(this.h[3], d);\n  this.h[4] = sum32(this.h[4], e);\n  this.h[5] = sum32(this.h[5], f);\n  this.h[6] = sum32(this.h[6], g);\n  this.h[7] = sum32(this.h[7], h);\n};\n\nSHA256.prototype._digest = function digest(enc) {\n  if (enc === 'hex')\n    return utils.toHex32(this.h, 'big');\n  else\n    return utils.split32(this.h, 'big');\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///9344\n")},772:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar utils = __webpack_require__(6436);\n\nvar SHA512 = __webpack_require__(5900);\n\nfunction SHA384() {\n  if (!(this instanceof SHA384))\n    return new SHA384();\n\n  SHA512.call(this);\n  this.h = [\n    0xcbbb9d5d, 0xc1059ed8,\n    0x629a292a, 0x367cd507,\n    0x9159015a, 0x3070dd17,\n    0x152fecd8, 0xf70e5939,\n    0x67332667, 0xffc00b31,\n    0x8eb44a87, 0x68581511,\n    0xdb0c2e0d, 0x64f98fa7,\n    0x47b5481d, 0xbefa4fa4 ];\n}\nutils.inherits(SHA384, SHA512);\nmodule.exports = SHA384;\n\nSHA384.blockSize = 1024;\nSHA384.outSize = 384;\nSHA384.hmacStrength = 192;\nSHA384.padLength = 128;\n\nSHA384.prototype._digest = function digest(enc) {\n  if (enc === 'hex')\n    return utils.toHex32(this.h.slice(0, 12), 'big');\n  else\n    return utils.split32(this.h.slice(0, 12), 'big');\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2hhc2guanMvbGliL2hhc2gvc2hhLzM4NC5qcz84Yjk1Il0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFhOztBQUViLFlBQVksbUJBQU8sQ0FBQyxJQUFVOztBQUU5QixhQUFhLG1CQUFPLENBQUMsSUFBTzs7QUFFNUI7QUFDQTtBQUNBOztBQUVBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBOztBQUVBO0FBQ0E7QUFDQTtBQUNBOztBQUVBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQSIsImZpbGUiOiI3NzIuanMiLCJzb3VyY2VzQ29udGVudCI6WyIndXNlIHN0cmljdCc7XG5cbnZhciB1dGlscyA9IHJlcXVpcmUoJy4uL3V0aWxzJyk7XG5cbnZhciBTSEE1MTIgPSByZXF1aXJlKCcuLzUxMicpO1xuXG5mdW5jdGlvbiBTSEEzODQoKSB7XG4gIGlmICghKHRoaXMgaW5zdGFuY2VvZiBTSEEzODQpKVxuICAgIHJldHVybiBuZXcgU0hBMzg0KCk7XG5cbiAgU0hBNTEyLmNhbGwodGhpcyk7XG4gIHRoaXMuaCA9IFtcbiAgICAweGNiYmI5ZDVkLCAweGMxMDU5ZWQ4LFxuICAgIDB4NjI5YTI5MmEsIDB4MzY3Y2Q1MDcsXG4gICAgMHg5MTU5MDE1YSwgMHgzMDcwZGQxNyxcbiAgICAweDE1MmZlY2Q4LCAweGY3MGU1OTM5LFxuICAgIDB4NjczMzI2NjcsIDB4ZmZjMDBiMzEsXG4gICAgMHg4ZWI0NGE4NywgMHg2ODU4MTUxMSxcbiAgICAweGRiMGMyZTBkLCAweDY0Zjk4ZmE3LFxuICAgIDB4NDdiNTQ4MWQsIDB4YmVmYTRmYTQgXTtcbn1cbnV0aWxzLmluaGVyaXRzKFNIQTM4NCwgU0hBNTEyKTtcbm1vZHVsZS5leHBvcnRzID0gU0hBMzg0O1xuXG5TSEEzODQuYmxvY2tTaXplID0gMTAyNDtcblNIQTM4NC5vdXRTaXplID0gMzg0O1xuU0hBMzg0LmhtYWNTdHJlbmd0aCA9IDE5MjtcblNIQTM4NC5wYWRMZW5ndGggPSAxMjg7XG5cblNIQTM4NC5wcm90b3R5cGUuX2RpZ2VzdCA9IGZ1bmN0aW9uIGRpZ2VzdChlbmMpIHtcbiAgaWYgKGVuYyA9PT0gJ2hleCcpXG4gICAgcmV0dXJuIHV0aWxzLnRvSGV4MzIodGhpcy5oLnNsaWNlKDAsIDEyKSwgJ2JpZycpO1xuICBlbHNlXG4gICAgcmV0dXJuIHV0aWxzLnNwbGl0MzIodGhpcy5oLnNsaWNlKDAsIDEyKSwgJ2JpZycpO1xufTtcbiJdLCJzb3VyY2VSb290IjoiIn0=\n//# sourceURL=webpack-internal:///772\n")},5900:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar utils = __webpack_require__(6436);\nvar common = __webpack_require__(5772);\nvar assert = __webpack_require__(9746);\n\nvar rotr64_hi = utils.rotr64_hi;\nvar rotr64_lo = utils.rotr64_lo;\nvar shr64_hi = utils.shr64_hi;\nvar shr64_lo = utils.shr64_lo;\nvar sum64 = utils.sum64;\nvar sum64_hi = utils.sum64_hi;\nvar sum64_lo = utils.sum64_lo;\nvar sum64_4_hi = utils.sum64_4_hi;\nvar sum64_4_lo = utils.sum64_4_lo;\nvar sum64_5_hi = utils.sum64_5_hi;\nvar sum64_5_lo = utils.sum64_5_lo;\n\nvar BlockHash = common.BlockHash;\n\nvar sha512_K = [\n  0x428a2f98, 0xd728ae22, 0x71374491, 0x23ef65cd,\n  0xb5c0fbcf, 0xec4d3b2f, 0xe9b5dba5, 0x8189dbbc,\n  0x3956c25b, 0xf348b538, 0x59f111f1, 0xb605d019,\n  0x923f82a4, 0xaf194f9b, 0xab1c5ed5, 0xda6d8118,\n  0xd807aa98, 0xa3030242, 0x12835b01, 0x45706fbe,\n  0x243185be, 0x4ee4b28c, 0x550c7dc3, 0xd5ffb4e2,\n  0x72be5d74, 0xf27b896f, 0x80deb1fe, 0x3b1696b1,\n  0x9bdc06a7, 0x25c71235, 0xc19bf174, 0xcf692694,\n  0xe49b69c1, 0x9ef14ad2, 0xefbe4786, 0x384f25e3,\n  0x0fc19dc6, 0x8b8cd5b5, 0x240ca1cc, 0x77ac9c65,\n  0x2de92c6f, 0x592b0275, 0x4a7484aa, 0x6ea6e483,\n  0x5cb0a9dc, 0xbd41fbd4, 0x76f988da, 0x831153b5,\n  0x983e5152, 0xee66dfab, 0xa831c66d, 0x2db43210,\n  0xb00327c8, 0x98fb213f, 0xbf597fc7, 0xbeef0ee4,\n  0xc6e00bf3, 0x3da88fc2, 0xd5a79147, 0x930aa725,\n  0x06ca6351, 0xe003826f, 0x14292967, 0x0a0e6e70,\n  0x27b70a85, 0x46d22ffc, 0x2e1b2138, 0x5c26c926,\n  0x4d2c6dfc, 0x5ac42aed, 0x53380d13, 0x9d95b3df,\n  0x650a7354, 0x8baf63de, 0x766a0abb, 0x3c77b2a8,\n  0x81c2c92e, 0x47edaee6, 0x92722c85, 0x1482353b,\n  0xa2bfe8a1, 0x4cf10364, 0xa81a664b, 0xbc423001,\n  0xc24b8b70, 0xd0f89791, 0xc76c51a3, 0x0654be30,\n  0xd192e819, 0xd6ef5218, 0xd6990624, 0x5565a910,\n  0xf40e3585, 0x5771202a, 0x106aa070, 0x32bbd1b8,\n  0x19a4c116, 0xb8d2d0c8, 0x1e376c08, 0x5141ab53,\n  0x2748774c, 0xdf8eeb99, 0x34b0bcb5, 0xe19b48a8,\n  0x391c0cb3, 0xc5c95a63, 0x4ed8aa4a, 0xe3418acb,\n  0x5b9cca4f, 0x7763e373, 0x682e6ff3, 0xd6b2b8a3,\n  0x748f82ee, 0x5defb2fc, 0x78a5636f, 0x43172f60,\n  0x84c87814, 0xa1f0ab72, 0x8cc70208, 0x1a6439ec,\n  0x90befffa, 0x23631e28, 0xa4506ceb, 0xde82bde9,\n  0xbef9a3f7, 0xb2c67915, 0xc67178f2, 0xe372532b,\n  0xca273ece, 0xea26619c, 0xd186b8c7, 0x21c0c207,\n  0xeada7dd6, 0xcde0eb1e, 0xf57d4f7f, 0xee6ed178,\n  0x06f067aa, 0x72176fba, 0x0a637dc5, 0xa2c898a6,\n  0x113f9804, 0xbef90dae, 0x1b710b35, 0x131c471b,\n  0x28db77f5, 0x23047d84, 0x32caab7b, 0x40c72493,\n  0x3c9ebe0a, 0x15c9bebc, 0x431d67c4, 0x9c100d4c,\n  0x4cc5d4be, 0xcb3e42b6, 0x597f299c, 0xfc657e2a,\n  0x5fcb6fab, 0x3ad6faec, 0x6c44198c, 0x4a475817\n];\n\nfunction SHA512() {\n  if (!(this instanceof SHA512))\n    return new SHA512();\n\n  BlockHash.call(this);\n  this.h = [\n    0x6a09e667, 0xf3bcc908,\n    0xbb67ae85, 0x84caa73b,\n    0x3c6ef372, 0xfe94f82b,\n    0xa54ff53a, 0x5f1d36f1,\n    0x510e527f, 0xade682d1,\n    0x9b05688c, 0x2b3e6c1f,\n    0x1f83d9ab, 0xfb41bd6b,\n    0x5be0cd19, 0x137e2179 ];\n  this.k = sha512_K;\n  this.W = new Array(160);\n}\nutils.inherits(SHA512, BlockHash);\nmodule.exports = SHA512;\n\nSHA512.blockSize = 1024;\nSHA512.outSize = 512;\nSHA512.hmacStrength = 192;\nSHA512.padLength = 128;\n\nSHA512.prototype._prepareBlock = function _prepareBlock(msg, start) {\n  var W = this.W;\n\n  // 32 x 32bit words\n  for (var i = 0; i < 32; i++)\n    W[i] = msg[start + i];\n  for (; i < W.length; i += 2) {\n    var c0_hi = g1_512_hi(W[i - 4], W[i - 3]);  // i - 2\n    var c0_lo = g1_512_lo(W[i - 4], W[i - 3]);\n    var c1_hi = W[i - 14];  // i - 7\n    var c1_lo = W[i - 13];\n    var c2_hi = g0_512_hi(W[i - 30], W[i - 29]);  // i - 15\n    var c2_lo = g0_512_lo(W[i - 30], W[i - 29]);\n    var c3_hi = W[i - 32];  // i - 16\n    var c3_lo = W[i - 31];\n\n    W[i] = sum64_4_hi(\n      c0_hi, c0_lo,\n      c1_hi, c1_lo,\n      c2_hi, c2_lo,\n      c3_hi, c3_lo);\n    W[i + 1] = sum64_4_lo(\n      c0_hi, c0_lo,\n      c1_hi, c1_lo,\n      c2_hi, c2_lo,\n      c3_hi, c3_lo);\n  }\n};\n\nSHA512.prototype._update = function _update(msg, start) {\n  this._prepareBlock(msg, start);\n\n  var W = this.W;\n\n  var ah = this.h[0];\n  var al = this.h[1];\n  var bh = this.h[2];\n  var bl = this.h[3];\n  var ch = this.h[4];\n  var cl = this.h[5];\n  var dh = this.h[6];\n  var dl = this.h[7];\n  var eh = this.h[8];\n  var el = this.h[9];\n  var fh = this.h[10];\n  var fl = this.h[11];\n  var gh = this.h[12];\n  var gl = this.h[13];\n  var hh = this.h[14];\n  var hl = this.h[15];\n\n  assert(this.k.length === W.length);\n  for (var i = 0; i < W.length; i += 2) {\n    var c0_hi = hh;\n    var c0_lo = hl;\n    var c1_hi = s1_512_hi(eh, el);\n    var c1_lo = s1_512_lo(eh, el);\n    var c2_hi = ch64_hi(eh, el, fh, fl, gh, gl);\n    var c2_lo = ch64_lo(eh, el, fh, fl, gh, gl);\n    var c3_hi = this.k[i];\n    var c3_lo = this.k[i + 1];\n    var c4_hi = W[i];\n    var c4_lo = W[i + 1];\n\n    var T1_hi = sum64_5_hi(\n      c0_hi, c0_lo,\n      c1_hi, c1_lo,\n      c2_hi, c2_lo,\n      c3_hi, c3_lo,\n      c4_hi, c4_lo);\n    var T1_lo = sum64_5_lo(\n      c0_hi, c0_lo,\n      c1_hi, c1_lo,\n      c2_hi, c2_lo,\n      c3_hi, c3_lo,\n      c4_hi, c4_lo);\n\n    c0_hi = s0_512_hi(ah, al);\n    c0_lo = s0_512_lo(ah, al);\n    c1_hi = maj64_hi(ah, al, bh, bl, ch, cl);\n    c1_lo = maj64_lo(ah, al, bh, bl, ch, cl);\n\n    var T2_hi = sum64_hi(c0_hi, c0_lo, c1_hi, c1_lo);\n    var T2_lo = sum64_lo(c0_hi, c0_lo, c1_hi, c1_lo);\n\n    hh = gh;\n    hl = gl;\n\n    gh = fh;\n    gl = fl;\n\n    fh = eh;\n    fl = el;\n\n    eh = sum64_hi(dh, dl, T1_hi, T1_lo);\n    el = sum64_lo(dl, dl, T1_hi, T1_lo);\n\n    dh = ch;\n    dl = cl;\n\n    ch = bh;\n    cl = bl;\n\n    bh = ah;\n    bl = al;\n\n    ah = sum64_hi(T1_hi, T1_lo, T2_hi, T2_lo);\n    al = sum64_lo(T1_hi, T1_lo, T2_hi, T2_lo);\n  }\n\n  sum64(this.h, 0, ah, al);\n  sum64(this.h, 2, bh, bl);\n  sum64(this.h, 4, ch, cl);\n  sum64(this.h, 6, dh, dl);\n  sum64(this.h, 8, eh, el);\n  sum64(this.h, 10, fh, fl);\n  sum64(this.h, 12, gh, gl);\n  sum64(this.h, 14, hh, hl);\n};\n\nSHA512.prototype._digest = function digest(enc) {\n  if (enc === 'hex')\n    return utils.toHex32(this.h, 'big');\n  else\n    return utils.split32(this.h, 'big');\n};\n\nfunction ch64_hi(xh, xl, yh, yl, zh) {\n  var r = (xh & yh) ^ ((~xh) & zh);\n  if (r < 0)\n    r += 0x100000000;\n  return r;\n}\n\nfunction ch64_lo(xh, xl, yh, yl, zh, zl) {\n  var r = (xl & yl) ^ ((~xl) & zl);\n  if (r < 0)\n    r += 0x100000000;\n  return r;\n}\n\nfunction maj64_hi(xh, xl, yh, yl, zh) {\n  var r = (xh & yh) ^ (xh & zh) ^ (yh & zh);\n  if (r < 0)\n    r += 0x100000000;\n  return r;\n}\n\nfunction maj64_lo(xh, xl, yh, yl, zh, zl) {\n  var r = (xl & yl) ^ (xl & zl) ^ (yl & zl);\n  if (r < 0)\n    r += 0x100000000;\n  return r;\n}\n\nfunction s0_512_hi(xh, xl) {\n  var c0_hi = rotr64_hi(xh, xl, 28);\n  var c1_hi = rotr64_hi(xl, xh, 2);  // 34\n  var c2_hi = rotr64_hi(xl, xh, 7);  // 39\n\n  var r = c0_hi ^ c1_hi ^ c2_hi;\n  if (r < 0)\n    r += 0x100000000;\n  return r;\n}\n\nfunction s0_512_lo(xh, xl) {\n  var c0_lo = rotr64_lo(xh, xl, 28);\n  var c1_lo = rotr64_lo(xl, xh, 2);  // 34\n  var c2_lo = rotr64_lo(xl, xh, 7);  // 39\n\n  var r = c0_lo ^ c1_lo ^ c2_lo;\n  if (r < 0)\n    r += 0x100000000;\n  return r;\n}\n\nfunction s1_512_hi(xh, xl) {\n  var c0_hi = rotr64_hi(xh, xl, 14);\n  var c1_hi = rotr64_hi(xh, xl, 18);\n  var c2_hi = rotr64_hi(xl, xh, 9);  // 41\n\n  var r = c0_hi ^ c1_hi ^ c2_hi;\n  if (r < 0)\n    r += 0x100000000;\n  return r;\n}\n\nfunction s1_512_lo(xh, xl) {\n  var c0_lo = rotr64_lo(xh, xl, 14);\n  var c1_lo = rotr64_lo(xh, xl, 18);\n  var c2_lo = rotr64_lo(xl, xh, 9);  // 41\n\n  var r = c0_lo ^ c1_lo ^ c2_lo;\n  if (r < 0)\n    r += 0x100000000;\n  return r;\n}\n\nfunction g0_512_hi(xh, xl) {\n  var c0_hi = rotr64_hi(xh, xl, 1);\n  var c1_hi = rotr64_hi(xh, xl, 8);\n  var c2_hi = shr64_hi(xh, xl, 7);\n\n  var r = c0_hi ^ c1_hi ^ c2_hi;\n  if (r < 0)\n    r += 0x100000000;\n  return r;\n}\n\nfunction g0_512_lo(xh, xl) {\n  var c0_lo = rotr64_lo(xh, xl, 1);\n  var c1_lo = rotr64_lo(xh, xl, 8);\n  var c2_lo = shr64_lo(xh, xl, 7);\n\n  var r = c0_lo ^ c1_lo ^ c2_lo;\n  if (r < 0)\n    r += 0x100000000;\n  return r;\n}\n\nfunction g1_512_hi(xh, xl) {\n  var c0_hi = rotr64_hi(xh, xl, 19);\n  var c1_hi = rotr64_hi(xl, xh, 29);  // 61\n  var c2_hi = shr64_hi(xh, xl, 6);\n\n  var r = c0_hi ^ c1_hi ^ c2_hi;\n  if (r < 0)\n    r += 0x100000000;\n  return r;\n}\n\nfunction g1_512_lo(xh, xl) {\n  var c0_lo = rotr64_lo(xh, xl, 19);\n  var c1_lo = rotr64_lo(xl, xh, 29);  // 61\n  var c2_lo = shr64_lo(xh, xl, 6);\n\n  var r = c0_lo ^ c1_lo ^ c2_lo;\n  if (r < 0)\n    r += 0x100000000;\n  return r;\n}\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///5900\n")},7038:(__unused_webpack_module,exports,__webpack_require__)=>{"use strict";eval("\n\nvar utils = __webpack_require__(6436);\nvar rotr32 = utils.rotr32;\n\nfunction ft_1(s, x, y, z) {\n  if (s === 0)\n    return ch32(x, y, z);\n  if (s === 1 || s === 3)\n    return p32(x, y, z);\n  if (s === 2)\n    return maj32(x, y, z);\n}\nexports.ft_1 = ft_1;\n\nfunction ch32(x, y, z) {\n  return (x & y) ^ ((~x) & z);\n}\nexports.ch32 = ch32;\n\nfunction maj32(x, y, z) {\n  return (x & y) ^ (x & z) ^ (y & z);\n}\nexports.maj32 = maj32;\n\nfunction p32(x, y, z) {\n  return x ^ y ^ z;\n}\nexports.p32 = p32;\n\nfunction s0_256(x) {\n  return rotr32(x, 2) ^ rotr32(x, 13) ^ rotr32(x, 22);\n}\nexports.s0_256 = s0_256;\n\nfunction s1_256(x) {\n  return rotr32(x, 6) ^ rotr32(x, 11) ^ rotr32(x, 25);\n}\nexports.s1_256 = s1_256;\n\nfunction g0_256(x) {\n  return rotr32(x, 7) ^ rotr32(x, 18) ^ (x >>> 3);\n}\nexports.g0_256 = g0_256;\n\nfunction g1_256(x) {\n  return rotr32(x, 17) ^ rotr32(x, 19) ^ (x >>> 10);\n}\nexports.g1_256 = g1_256;\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2hhc2guanMvbGliL2hhc2gvc2hhL2NvbW1vbi5qcz9hYTU2Il0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFhOztBQUViLFlBQVksbUJBQU8sQ0FBQyxJQUFVO0FBQzlCOztBQUVBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQSxZQUFZOztBQUVaO0FBQ0E7QUFDQTtBQUNBLFlBQVk7O0FBRVo7QUFDQTtBQUNBO0FBQ0EsYUFBYTs7QUFFYjtBQUNBO0FBQ0E7QUFDQSxXQUFXOztBQUVYO0FBQ0E7QUFDQTtBQUNBLGNBQWM7O0FBRWQ7QUFDQTtBQUNBO0FBQ0EsY0FBYzs7QUFFZDtBQUNBO0FBQ0E7QUFDQSxjQUFjOztBQUVkO0FBQ0E7QUFDQTtBQUNBLGNBQWMiLCJmaWxlIjoiNzAzOC5qcyIsInNvdXJjZXNDb250ZW50IjpbIid1c2Ugc3RyaWN0JztcblxudmFyIHV0aWxzID0gcmVxdWlyZSgnLi4vdXRpbHMnKTtcbnZhciByb3RyMzIgPSB1dGlscy5yb3RyMzI7XG5cbmZ1bmN0aW9uIGZ0XzEocywgeCwgeSwgeikge1xuICBpZiAocyA9PT0gMClcbiAgICByZXR1cm4gY2gzMih4LCB5LCB6KTtcbiAgaWYgKHMgPT09IDEgfHwgcyA9PT0gMylcbiAgICByZXR1cm4gcDMyKHgsIHksIHopO1xuICBpZiAocyA9PT0gMilcbiAgICByZXR1cm4gbWFqMzIoeCwgeSwgeik7XG59XG5leHBvcnRzLmZ0XzEgPSBmdF8xO1xuXG5mdW5jdGlvbiBjaDMyKHgsIHksIHopIHtcbiAgcmV0dXJuICh4ICYgeSkgXiAoKH54KSAmIHopO1xufVxuZXhwb3J0cy5jaDMyID0gY2gzMjtcblxuZnVuY3Rpb24gbWFqMzIoeCwgeSwgeikge1xuICByZXR1cm4gKHggJiB5KSBeICh4ICYgeikgXiAoeSAmIHopO1xufVxuZXhwb3J0cy5tYWozMiA9IG1hajMyO1xuXG5mdW5jdGlvbiBwMzIoeCwgeSwgeikge1xuICByZXR1cm4geCBeIHkgXiB6O1xufVxuZXhwb3J0cy5wMzIgPSBwMzI7XG5cbmZ1bmN0aW9uIHMwXzI1Nih4KSB7XG4gIHJldHVybiByb3RyMzIoeCwgMikgXiByb3RyMzIoeCwgMTMpIF4gcm90cjMyKHgsIDIyKTtcbn1cbmV4cG9ydHMuczBfMjU2ID0gczBfMjU2O1xuXG5mdW5jdGlvbiBzMV8yNTYoeCkge1xuICByZXR1cm4gcm90cjMyKHgsIDYpIF4gcm90cjMyKHgsIDExKSBeIHJvdHIzMih4LCAyNSk7XG59XG5leHBvcnRzLnMxXzI1NiA9IHMxXzI1NjtcblxuZnVuY3Rpb24gZzBfMjU2KHgpIHtcbiAgcmV0dXJuIHJvdHIzMih4LCA3KSBeIHJvdHIzMih4LCAxOCkgXiAoeCA+Pj4gMyk7XG59XG5leHBvcnRzLmcwXzI1NiA9IGcwXzI1NjtcblxuZnVuY3Rpb24gZzFfMjU2KHgpIHtcbiAgcmV0dXJuIHJvdHIzMih4LCAxNykgXiByb3RyMzIoeCwgMTkpIF4gKHggPj4+IDEwKTtcbn1cbmV4cG9ydHMuZzFfMjU2ID0gZzFfMjU2O1xuIl0sInNvdXJjZVJvb3QiOiIifQ==\n//# sourceURL=webpack-internal:///7038\n")},6436:(__unused_webpack_module,exports,__webpack_require__)=>{"use strict";eval("\n\nvar assert = __webpack_require__(9746);\nvar inherits = __webpack_require__(5717);\n\nexports.inherits = inherits;\n\nfunction isSurrogatePair(msg, i) {\n  if ((msg.charCodeAt(i) & 0xFC00) !== 0xD800) {\n    return false;\n  }\n  if (i < 0 || i + 1 >= msg.length) {\n    return false;\n  }\n  return (msg.charCodeAt(i + 1) & 0xFC00) === 0xDC00;\n}\n\nfunction toArray(msg, enc) {\n  if (Array.isArray(msg))\n    return msg.slice();\n  if (!msg)\n    return [];\n  var res = [];\n  if (typeof msg === 'string') {\n    if (!enc) {\n      // Inspired by stringToUtf8ByteArray() in closure-library by Google\n      // https://github.com/google/closure-library/blob/8598d87242af59aac233270742c8984e2b2bdbe0/closure/goog/crypt/crypt.js#L117-L143\n      // Apache License 2.0\n      // https://github.com/google/closure-library/blob/master/LICENSE\n      var p = 0;\n      for (var i = 0; i < msg.length; i++) {\n        var c = msg.charCodeAt(i);\n        if (c < 128) {\n          res[p++] = c;\n        } else if (c < 2048) {\n          res[p++] = (c >> 6) | 192;\n          res[p++] = (c & 63) | 128;\n        } else if (isSurrogatePair(msg, i)) {\n          c = 0x10000 + ((c & 0x03FF) << 10) + (msg.charCodeAt(++i) & 0x03FF);\n          res[p++] = (c >> 18) | 240;\n          res[p++] = ((c >> 12) & 63) | 128;\n          res[p++] = ((c >> 6) & 63) | 128;\n          res[p++] = (c & 63) | 128;\n        } else {\n          res[p++] = (c >> 12) | 224;\n          res[p++] = ((c >> 6) & 63) | 128;\n          res[p++] = (c & 63) | 128;\n        }\n      }\n    } else if (enc === 'hex') {\n      msg = msg.replace(/[^a-z0-9]+/ig, '');\n      if (msg.length % 2 !== 0)\n        msg = '0' + msg;\n      for (i = 0; i < msg.length; i += 2)\n        res.push(parseInt(msg[i] + msg[i + 1], 16));\n    }\n  } else {\n    for (i = 0; i < msg.length; i++)\n      res[i] = msg[i] | 0;\n  }\n  return res;\n}\nexports.toArray = toArray;\n\nfunction toHex(msg) {\n  var res = '';\n  for (var i = 0; i < msg.length; i++)\n    res += zero2(msg[i].toString(16));\n  return res;\n}\nexports.toHex = toHex;\n\nfunction htonl(w) {\n  var res = (w >>> 24) |\n            ((w >>> 8) & 0xff00) |\n            ((w << 8) & 0xff0000) |\n            ((w & 0xff) << 24);\n  return res >>> 0;\n}\nexports.htonl = htonl;\n\nfunction toHex32(msg, endian) {\n  var res = '';\n  for (var i = 0; i < msg.length; i++) {\n    var w = msg[i];\n    if (endian === 'little')\n      w = htonl(w);\n    res += zero8(w.toString(16));\n  }\n  return res;\n}\nexports.toHex32 = toHex32;\n\nfunction zero2(word) {\n  if (word.length === 1)\n    return '0' + word;\n  else\n    return word;\n}\nexports.zero2 = zero2;\n\nfunction zero8(word) {\n  if (word.length === 7)\n    return '0' + word;\n  else if (word.length === 6)\n    return '00' + word;\n  else if (word.length === 5)\n    return '000' + word;\n  else if (word.length === 4)\n    return '0000' + word;\n  else if (word.length === 3)\n    return '00000' + word;\n  else if (word.length === 2)\n    return '000000' + word;\n  else if (word.length === 1)\n    return '0000000' + word;\n  else\n    return word;\n}\nexports.zero8 = zero8;\n\nfunction join32(msg, start, end, endian) {\n  var len = end - start;\n  assert(len % 4 === 0);\n  var res = new Array(len / 4);\n  for (var i = 0, k = start; i < res.length; i++, k += 4) {\n    var w;\n    if (endian === 'big')\n      w = (msg[k] << 24) | (msg[k + 1] << 16) | (msg[k + 2] << 8) | msg[k + 3];\n    else\n      w = (msg[k + 3] << 24) | (msg[k + 2] << 16) | (msg[k + 1] << 8) | msg[k];\n    res[i] = w >>> 0;\n  }\n  return res;\n}\nexports.join32 = join32;\n\nfunction split32(msg, endian) {\n  var res = new Array(msg.length * 4);\n  for (var i = 0, k = 0; i < msg.length; i++, k += 4) {\n    var m = msg[i];\n    if (endian === 'big') {\n      res[k] = m >>> 24;\n      res[k + 1] = (m >>> 16) & 0xff;\n      res[k + 2] = (m >>> 8) & 0xff;\n      res[k + 3] = m & 0xff;\n    } else {\n      res[k + 3] = m >>> 24;\n      res[k + 2] = (m >>> 16) & 0xff;\n      res[k + 1] = (m >>> 8) & 0xff;\n      res[k] = m & 0xff;\n    }\n  }\n  return res;\n}\nexports.split32 = split32;\n\nfunction rotr32(w, b) {\n  return (w >>> b) | (w << (32 - b));\n}\nexports.rotr32 = rotr32;\n\nfunction rotl32(w, b) {\n  return (w << b) | (w >>> (32 - b));\n}\nexports.rotl32 = rotl32;\n\nfunction sum32(a, b) {\n  return (a + b) >>> 0;\n}\nexports.sum32 = sum32;\n\nfunction sum32_3(a, b, c) {\n  return (a + b + c) >>> 0;\n}\nexports.sum32_3 = sum32_3;\n\nfunction sum32_4(a, b, c, d) {\n  return (a + b + c + d) >>> 0;\n}\nexports.sum32_4 = sum32_4;\n\nfunction sum32_5(a, b, c, d, e) {\n  return (a + b + c + d + e) >>> 0;\n}\nexports.sum32_5 = sum32_5;\n\nfunction sum64(buf, pos, ah, al) {\n  var bh = buf[pos];\n  var bl = buf[pos + 1];\n\n  var lo = (al + bl) >>> 0;\n  var hi = (lo < al ? 1 : 0) + ah + bh;\n  buf[pos] = hi >>> 0;\n  buf[pos + 1] = lo;\n}\nexports.sum64 = sum64;\n\nfunction sum64_hi(ah, al, bh, bl) {\n  var lo = (al + bl) >>> 0;\n  var hi = (lo < al ? 1 : 0) + ah + bh;\n  return hi >>> 0;\n}\nexports.sum64_hi = sum64_hi;\n\nfunction sum64_lo(ah, al, bh, bl) {\n  var lo = al + bl;\n  return lo >>> 0;\n}\nexports.sum64_lo = sum64_lo;\n\nfunction sum64_4_hi(ah, al, bh, bl, ch, cl, dh, dl) {\n  var carry = 0;\n  var lo = al;\n  lo = (lo + bl) >>> 0;\n  carry += lo < al ? 1 : 0;\n  lo = (lo + cl) >>> 0;\n  carry += lo < cl ? 1 : 0;\n  lo = (lo + dl) >>> 0;\n  carry += lo < dl ? 1 : 0;\n\n  var hi = ah + bh + ch + dh + carry;\n  return hi >>> 0;\n}\nexports.sum64_4_hi = sum64_4_hi;\n\nfunction sum64_4_lo(ah, al, bh, bl, ch, cl, dh, dl) {\n  var lo = al + bl + cl + dl;\n  return lo >>> 0;\n}\nexports.sum64_4_lo = sum64_4_lo;\n\nfunction sum64_5_hi(ah, al, bh, bl, ch, cl, dh, dl, eh, el) {\n  var carry = 0;\n  var lo = al;\n  lo = (lo + bl) >>> 0;\n  carry += lo < al ? 1 : 0;\n  lo = (lo + cl) >>> 0;\n  carry += lo < cl ? 1 : 0;\n  lo = (lo + dl) >>> 0;\n  carry += lo < dl ? 1 : 0;\n  lo = (lo + el) >>> 0;\n  carry += lo < el ? 1 : 0;\n\n  var hi = ah + bh + ch + dh + eh + carry;\n  return hi >>> 0;\n}\nexports.sum64_5_hi = sum64_5_hi;\n\nfunction sum64_5_lo(ah, al, bh, bl, ch, cl, dh, dl, eh, el) {\n  var lo = al + bl + cl + dl + el;\n\n  return lo >>> 0;\n}\nexports.sum64_5_lo = sum64_5_lo;\n\nfunction rotr64_hi(ah, al, num) {\n  var r = (al << (32 - num)) | (ah >>> num);\n  return r >>> 0;\n}\nexports.rotr64_hi = rotr64_hi;\n\nfunction rotr64_lo(ah, al, num) {\n  var r = (ah << (32 - num)) | (al >>> num);\n  return r >>> 0;\n}\nexports.rotr64_lo = rotr64_lo;\n\nfunction shr64_hi(ah, al, num) {\n  return ah >>> num;\n}\nexports.shr64_hi = shr64_hi;\n\nfunction shr64_lo(ah, al, num) {\n  var r = (ah << (32 - num)) | (al >>> num);\n  return r >>> 0;\n}\nexports.shr64_lo = shr64_lo;\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2hhc2guanMvbGliL2hhc2gvdXRpbHMuanM/YzNjMCJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBYTs7QUFFYixhQUFhLG1CQUFPLENBQUMsSUFBcUI7QUFDMUMsZUFBZSxtQkFBTyxDQUFDLElBQVU7O0FBRWpDLGdCQUFnQjs7QUFFaEI7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBOztBQUVBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0EscUJBQXFCLGdCQUFnQjtBQUNyQztBQUNBO0FBQ0E7QUFDQSxTQUFTO0FBQ1Q7QUFDQTtBQUNBLFNBQVM7QUFDVDtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0EsU0FBUztBQUNUO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQSxLQUFLO0FBQ0w7QUFDQTtBQUNBO0FBQ0EsaUJBQWlCLGdCQUFnQjtBQUNqQztBQUNBO0FBQ0EsR0FBRztBQUNILGVBQWUsZ0JBQWdCO0FBQy9CO0FBQ0E7QUFDQTtBQUNBO0FBQ0EsZUFBZTs7QUFFZjtBQUNBO0FBQ0EsaUJBQWlCLGdCQUFnQjtBQUNqQztBQUNBO0FBQ0E7QUFDQSxhQUFhOztBQUViO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0EsYUFBYTs7QUFFYjtBQUNBO0FBQ0EsaUJBQWlCLGdCQUFnQjtBQUNqQztBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBLGVBQWU7O0FBRWY7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0EsYUFBYTs7QUFFYjtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQSxhQUFhOztBQUViO0FBQ0E7QUFDQTtBQUNBO0FBQ0EsNEJBQTRCLGdCQUFnQjtBQUM1QztBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQSxjQUFjOztBQUVkO0FBQ0E7QUFDQSx3QkFBd0IsZ0JBQWdCO0FBQ3hDO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBLEtBQUs7QUFDTDtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0EsZUFBZTs7QUFFZjtBQUNBO0FBQ0E7QUFDQSxjQUFjOztBQUVkO0FBQ0E7QUFDQTtBQUNBLGNBQWM7O0FBRWQ7QUFDQTtBQUNBO0FBQ0EsYUFBYTs7QUFFYjtBQUNBO0FBQ0E7QUFDQSxlQUFlOztBQUVmO0FBQ0E7QUFDQTtBQUNBLGVBQWU7O0FBRWY7QUFDQTtBQUNBO0FBQ0EsZUFBZTs7QUFFZjtBQUNBO0FBQ0E7O0FBRUE7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBLGFBQWE7O0FBRWI7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBLGdCQUFnQjs7QUFFaEI7QUFDQTtBQUNBO0FBQ0E7QUFDQSxnQkFBZ0I7O0FBRWhCO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTs7QUFFQTtBQUNBO0FBQ0E7QUFDQSxrQkFBa0I7O0FBRWxCO0FBQ0E7QUFDQTtBQUNBO0FBQ0Esa0JBQWtCOztBQUVsQjtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBOztBQUVBO0FBQ0E7QUFDQTtBQUNBLGtCQUFrQjs7QUFFbEI7QUFDQTs7QUFFQTtBQUNBO0FBQ0Esa0JBQWtCOztBQUVsQjtBQUNBO0FBQ0E7QUFDQTtBQUNBLGlCQUFpQjs7QUFFakI7QUFDQTtBQUNBO0FBQ0E7QUFDQSxpQkFBaUI7O0FBRWpCO0FBQ0E7QUFDQTtBQUNBLGdCQUFnQjs7QUFFaEI7QUFDQTtBQUNBO0FBQ0E7QUFDQSxnQkFBZ0IiLCJmaWxlIjoiNjQzNi5qcyIsInNvdXJjZXNDb250ZW50IjpbIid1c2Ugc3RyaWN0JztcblxudmFyIGFzc2VydCA9IHJlcXVpcmUoJ21pbmltYWxpc3RpYy1hc3NlcnQnKTtcbnZhciBpbmhlcml0cyA9IHJlcXVpcmUoJ2luaGVyaXRzJyk7XG5cbmV4cG9ydHMuaW5oZXJpdHMgPSBpbmhlcml0cztcblxuZnVuY3Rpb24gaXNTdXJyb2dhdGVQYWlyKG1zZywgaSkge1xuICBpZiAoKG1zZy5jaGFyQ29kZUF0KGkpICYgMHhGQzAwKSAhPT0gMHhEODAwKSB7XG4gICAgcmV0dXJuIGZhbHNlO1xuICB9XG4gIGlmIChpIDwgMCB8fCBpICsgMSA+PSBtc2cubGVuZ3RoKSB7XG4gICAgcmV0dXJuIGZhbHNlO1xuICB9XG4gIHJldHVybiAobXNnLmNoYXJDb2RlQXQoaSArIDEpICYgMHhGQzAwKSA9PT0gMHhEQzAwO1xufVxuXG5mdW5jdGlvbiB0b0FycmF5KG1zZywgZW5jKSB7XG4gIGlmIChBcnJheS5pc0FycmF5KG1zZykpXG4gICAgcmV0dXJuIG1zZy5zbGljZSgpO1xuICBpZiAoIW1zZylcbiAgICByZXR1cm4gW107XG4gIHZhciByZXMgPSBbXTtcbiAgaWYgKHR5cGVvZiBtc2cgPT09ICdzdHJpbmcnKSB7XG4gICAgaWYgKCFlbmMpIHtcbiAgICAgIC8vIEluc3BpcmVkIGJ5IHN0cmluZ1RvVXRmOEJ5dGVBcnJheSgpIGluIGNsb3N1cmUtbGlicmFyeSBieSBHb29nbGVcbiAgICAgIC8vIGh0dHBzOi8vZ2l0aHViLmNvbS9nb29nbGUvY2xvc3VyZS1saWJyYXJ5L2Jsb2IvODU5OGQ4NzI0MmFmNTlhYWMyMzMyNzA3NDJjODk4NGUyYjJiZGJlMC9jbG9zdXJlL2dvb2cvY3J5cHQvY3J5cHQuanMjTDExNy1MMTQzXG4gICAgICAvLyBBcGFjaGUgTGljZW5zZSAyLjBcbiAgICAgIC8vIGh0dHBzOi8vZ2l0aHViLmNvbS9nb29nbGUvY2xvc3VyZS1saWJyYXJ5L2Jsb2IvbWFzdGVyL0xJQ0VOU0VcbiAgICAgIHZhciBwID0gMDtcbiAgICAgIGZvciAodmFyIGkgPSAwOyBpIDwgbXNnLmxlbmd0aDsgaSsrKSB7XG4gICAgICAgIHZhciBjID0gbXNnLmNoYXJDb2RlQXQoaSk7XG4gICAgICAgIGlmIChjIDwgMTI4KSB7XG4gICAgICAgICAgcmVzW3ArK10gPSBjO1xuICAgICAgICB9IGVsc2UgaWYgKGMgPCAyMDQ4KSB7XG4gICAgICAgICAgcmVzW3ArK10gPSAoYyA+PiA2KSB8IDE5MjtcbiAgICAgICAgICByZXNbcCsrXSA9IChjICYgNjMpIHwgMTI4O1xuICAgICAgICB9IGVsc2UgaWYgKGlzU3Vycm9nYXRlUGFpcihtc2csIGkpKSB7XG4gICAgICAgICAgYyA9IDB4MTAwMDAgKyAoKGMgJiAweDAzRkYpIDw8IDEwKSArIChtc2cuY2hhckNvZGVBdCgrK2kpICYgMHgwM0ZGKTtcbiAgICAgICAgICByZXNbcCsrXSA9IChjID4+IDE4KSB8IDI0MDtcbiAgICAgICAgICByZXNbcCsrXSA9ICgoYyA+PiAxMikgJiA2MykgfCAxMjg7XG4gICAgICAgICAgcmVzW3ArK10gPSAoKGMgPj4gNikgJiA2MykgfCAxMjg7XG4gICAgICAgICAgcmVzW3ArK10gPSAoYyAmIDYzKSB8IDEyODtcbiAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICByZXNbcCsrXSA9IChjID4+IDEyKSB8IDIyNDtcbiAgICAgICAgICByZXNbcCsrXSA9ICgoYyA+PiA2KSAmIDYzKSB8IDEyODtcbiAgICAgICAgICByZXNbcCsrXSA9IChjICYgNjMpIHwgMTI4O1xuICAgICAgICB9XG4gICAgICB9XG4gICAgfSBlbHNlIGlmIChlbmMgPT09ICdoZXgnKSB7XG4gICAgICBtc2cgPSBtc2cucmVwbGFjZSgvW15hLXowLTldKy9pZywgJycpO1xuICAgICAgaWYgKG1zZy5sZW5ndGggJSAyICE9PSAwKVxuICAgICAgICBtc2cgPSAnMCcgKyBtc2c7XG4gICAgICBmb3IgKGkgPSAwOyBpIDwgbXNnLmxlbmd0aDsgaSArPSAyKVxuICAgICAgICByZXMucHVzaChwYXJzZUludChtc2dbaV0gKyBtc2dbaSArIDFdLCAxNikpO1xuICAgIH1cbiAgfSBlbHNlIHtcbiAgICBmb3IgKGkgPSAwOyBpIDwgbXNnLmxlbmd0aDsgaSsrKVxuICAgICAgcmVzW2ldID0gbXNnW2ldIHwgMDtcbiAgfVxuICByZXR1cm4gcmVzO1xufVxuZXhwb3J0cy50b0FycmF5ID0gdG9BcnJheTtcblxuZnVuY3Rpb24gdG9IZXgobXNnKSB7XG4gIHZhciByZXMgPSAnJztcbiAgZm9yICh2YXIgaSA9IDA7IGkgPCBtc2cubGVuZ3RoOyBpKyspXG4gICAgcmVzICs9IHplcm8yKG1zZ1tpXS50b1N0cmluZygxNikpO1xuICByZXR1cm4gcmVzO1xufVxuZXhwb3J0cy50b0hleCA9IHRvSGV4O1xuXG5mdW5jdGlvbiBodG9ubCh3KSB7XG4gIHZhciByZXMgPSAodyA+Pj4gMjQpIHxcbiAgICAgICAgICAgICgodyA+Pj4gOCkgJiAweGZmMDApIHxcbiAgICAgICAgICAgICgodyA8PCA4KSAmIDB4ZmYwMDAwKSB8XG4gICAgICAgICAgICAoKHcgJiAweGZmKSA8PCAyNCk7XG4gIHJldHVybiByZXMgPj4+IDA7XG59XG5leHBvcnRzLmh0b25sID0gaHRvbmw7XG5cbmZ1bmN0aW9uIHRvSGV4MzIobXNnLCBlbmRpYW4pIHtcbiAgdmFyIHJlcyA9ICcnO1xuICBmb3IgKHZhciBpID0gMDsgaSA8IG1zZy5sZW5ndGg7IGkrKykge1xuICAgIHZhciB3ID0gbXNnW2ldO1xuICAgIGlmIChlbmRpYW4gPT09ICdsaXR0bGUnKVxuICAgICAgdyA9IGh0b25sKHcpO1xuICAgIHJlcyArPSB6ZXJvOCh3LnRvU3RyaW5nKDE2KSk7XG4gIH1cbiAgcmV0dXJuIHJlcztcbn1cbmV4cG9ydHMudG9IZXgzMiA9IHRvSGV4MzI7XG5cbmZ1bmN0aW9uIHplcm8yKHdvcmQpIHtcbiAgaWYgKHdvcmQubGVuZ3RoID09PSAxKVxuICAgIHJldHVybiAnMCcgKyB3b3JkO1xuICBlbHNlXG4gICAgcmV0dXJuIHdvcmQ7XG59XG5leHBvcnRzLnplcm8yID0gemVybzI7XG5cbmZ1bmN0aW9uIHplcm84KHdvcmQpIHtcbiAgaWYgKHdvcmQubGVuZ3RoID09PSA3KVxuICAgIHJldHVybiAnMCcgKyB3b3JkO1xuICBlbHNlIGlmICh3b3JkLmxlbmd0aCA9PT0gNilcbiAgICByZXR1cm4gJzAwJyArIHdvcmQ7XG4gIGVsc2UgaWYgKHdvcmQubGVuZ3RoID09PSA1KVxuICAgIHJldHVybiAnMDAwJyArIHdvcmQ7XG4gIGVsc2UgaWYgKHdvcmQubGVuZ3RoID09PSA0KVxuICAgIHJldHVybiAnMDAwMCcgKyB3b3JkO1xuICBlbHNlIGlmICh3b3JkLmxlbmd0aCA9PT0gMylcbiAgICByZXR1cm4gJzAwMDAwJyArIHdvcmQ7XG4gIGVsc2UgaWYgKHdvcmQubGVuZ3RoID09PSAyKVxuICAgIHJldHVybiAnMDAwMDAwJyArIHdvcmQ7XG4gIGVsc2UgaWYgKHdvcmQubGVuZ3RoID09PSAxKVxuICAgIHJldHVybiAnMDAwMDAwMCcgKyB3b3JkO1xuICBlbHNlXG4gICAgcmV0dXJuIHdvcmQ7XG59XG5leHBvcnRzLnplcm84ID0gemVybzg7XG5cbmZ1bmN0aW9uIGpvaW4zMihtc2csIHN0YXJ0LCBlbmQsIGVuZGlhbikge1xuICB2YXIgbGVuID0gZW5kIC0gc3RhcnQ7XG4gIGFzc2VydChsZW4gJSA0ID09PSAwKTtcbiAgdmFyIHJlcyA9IG5ldyBBcnJheShsZW4gLyA0KTtcbiAgZm9yICh2YXIgaSA9IDAsIGsgPSBzdGFydDsgaSA8IHJlcy5sZW5ndGg7IGkrKywgayArPSA0KSB7XG4gICAgdmFyIHc7XG4gICAgaWYgKGVuZGlhbiA9PT0gJ2JpZycpXG4gICAgICB3ID0gKG1zZ1trXSA8PCAyNCkgfCAobXNnW2sgKyAxXSA8PCAxNikgfCAobXNnW2sgKyAyXSA8PCA4KSB8IG1zZ1trICsgM107XG4gICAgZWxzZVxuICAgICAgdyA9IChtc2dbayArIDNdIDw8IDI0KSB8IChtc2dbayArIDJdIDw8IDE2KSB8IChtc2dbayArIDFdIDw8IDgpIHwgbXNnW2tdO1xuICAgIHJlc1tpXSA9IHcgPj4+IDA7XG4gIH1cbiAgcmV0dXJuIHJlcztcbn1cbmV4cG9ydHMuam9pbjMyID0gam9pbjMyO1xuXG5mdW5jdGlvbiBzcGxpdDMyKG1zZywgZW5kaWFuKSB7XG4gIHZhciByZXMgPSBuZXcgQXJyYXkobXNnLmxlbmd0aCAqIDQpO1xuICBmb3IgKHZhciBpID0gMCwgayA9IDA7IGkgPCBtc2cubGVuZ3RoOyBpKyssIGsgKz0gNCkge1xuICAgIHZhciBtID0gbXNnW2ldO1xuICAgIGlmIChlbmRpYW4gPT09ICdiaWcnKSB7XG4gICAgICByZXNba10gPSBtID4+PiAyNDtcbiAgICAgIHJlc1trICsgMV0gPSAobSA+Pj4gMTYpICYgMHhmZjtcbiAgICAgIHJlc1trICsgMl0gPSAobSA+Pj4gOCkgJiAweGZmO1xuICAgICAgcmVzW2sgKyAzXSA9IG0gJiAweGZmO1xuICAgIH0gZWxzZSB7XG4gICAgICByZXNbayArIDNdID0gbSA+Pj4gMjQ7XG4gICAgICByZXNbayArIDJdID0gKG0gPj4+IDE2KSAmIDB4ZmY7XG4gICAgICByZXNbayArIDFdID0gKG0gPj4+IDgpICYgMHhmZjtcbiAgICAgIHJlc1trXSA9IG0gJiAweGZmO1xuICAgIH1cbiAgfVxuICByZXR1cm4gcmVzO1xufVxuZXhwb3J0cy5zcGxpdDMyID0gc3BsaXQzMjtcblxuZnVuY3Rpb24gcm90cjMyKHcsIGIpIHtcbiAgcmV0dXJuICh3ID4+PiBiKSB8ICh3IDw8ICgzMiAtIGIpKTtcbn1cbmV4cG9ydHMucm90cjMyID0gcm90cjMyO1xuXG5mdW5jdGlvbiByb3RsMzIodywgYikge1xuICByZXR1cm4gKHcgPDwgYikgfCAodyA+Pj4gKDMyIC0gYikpO1xufVxuZXhwb3J0cy5yb3RsMzIgPSByb3RsMzI7XG5cbmZ1bmN0aW9uIHN1bTMyKGEsIGIpIHtcbiAgcmV0dXJuIChhICsgYikgPj4+IDA7XG59XG5leHBvcnRzLnN1bTMyID0gc3VtMzI7XG5cbmZ1bmN0aW9uIHN1bTMyXzMoYSwgYiwgYykge1xuICByZXR1cm4gKGEgKyBiICsgYykgPj4+IDA7XG59XG5leHBvcnRzLnN1bTMyXzMgPSBzdW0zMl8zO1xuXG5mdW5jdGlvbiBzdW0zMl80KGEsIGIsIGMsIGQpIHtcbiAgcmV0dXJuIChhICsgYiArIGMgKyBkKSA+Pj4gMDtcbn1cbmV4cG9ydHMuc3VtMzJfNCA9IHN1bTMyXzQ7XG5cbmZ1bmN0aW9uIHN1bTMyXzUoYSwgYiwgYywgZCwgZSkge1xuICByZXR1cm4gKGEgKyBiICsgYyArIGQgKyBlKSA+Pj4gMDtcbn1cbmV4cG9ydHMuc3VtMzJfNSA9IHN1bTMyXzU7XG5cbmZ1bmN0aW9uIHN1bTY0KGJ1ZiwgcG9zLCBhaCwgYWwpIHtcbiAgdmFyIGJoID0gYnVmW3Bvc107XG4gIHZhciBibCA9IGJ1Zltwb3MgKyAxXTtcblxuICB2YXIgbG8gPSAoYWwgKyBibCkgPj4+IDA7XG4gIHZhciBoaSA9IChsbyA8IGFsID8gMSA6IDApICsgYWggKyBiaDtcbiAgYnVmW3Bvc10gPSBoaSA+Pj4gMDtcbiAgYnVmW3BvcyArIDFdID0gbG87XG59XG5leHBvcnRzLnN1bTY0ID0gc3VtNjQ7XG5cbmZ1bmN0aW9uIHN1bTY0X2hpKGFoLCBhbCwgYmgsIGJsKSB7XG4gIHZhciBsbyA9IChhbCArIGJsKSA+Pj4gMDtcbiAgdmFyIGhpID0gKGxvIDwgYWwgPyAxIDogMCkgKyBhaCArIGJoO1xuICByZXR1cm4gaGkgPj4+IDA7XG59XG5leHBvcnRzLnN1bTY0X2hpID0gc3VtNjRfaGk7XG5cbmZ1bmN0aW9uIHN1bTY0X2xvKGFoLCBhbCwgYmgsIGJsKSB7XG4gIHZhciBsbyA9IGFsICsgYmw7XG4gIHJldHVybiBsbyA+Pj4gMDtcbn1cbmV4cG9ydHMuc3VtNjRfbG8gPSBzdW02NF9sbztcblxuZnVuY3Rpb24gc3VtNjRfNF9oaShhaCwgYWwsIGJoLCBibCwgY2gsIGNsLCBkaCwgZGwpIHtcbiAgdmFyIGNhcnJ5ID0gMDtcbiAgdmFyIGxvID0gYWw7XG4gIGxvID0gKGxvICsgYmwpID4+PiAwO1xuICBjYXJyeSArPSBsbyA8IGFsID8gMSA6IDA7XG4gIGxvID0gKGxvICsgY2wpID4+PiAwO1xuICBjYXJyeSArPSBsbyA8IGNsID8gMSA6IDA7XG4gIGxvID0gKGxvICsgZGwpID4+PiAwO1xuICBjYXJyeSArPSBsbyA8IGRsID8gMSA6IDA7XG5cbiAgdmFyIGhpID0gYWggKyBiaCArIGNoICsgZGggKyBjYXJyeTtcbiAgcmV0dXJuIGhpID4+PiAwO1xufVxuZXhwb3J0cy5zdW02NF80X2hpID0gc3VtNjRfNF9oaTtcblxuZnVuY3Rpb24gc3VtNjRfNF9sbyhhaCwgYWwsIGJoLCBibCwgY2gsIGNsLCBkaCwgZGwpIHtcbiAgdmFyIGxvID0gYWwgKyBibCArIGNsICsgZGw7XG4gIHJldHVybiBsbyA+Pj4gMDtcbn1cbmV4cG9ydHMuc3VtNjRfNF9sbyA9IHN1bTY0XzRfbG87XG5cbmZ1bmN0aW9uIHN1bTY0XzVfaGkoYWgsIGFsLCBiaCwgYmwsIGNoLCBjbCwgZGgsIGRsLCBlaCwgZWwpIHtcbiAgdmFyIGNhcnJ5ID0gMDtcbiAgdmFyIGxvID0gYWw7XG4gIGxvID0gKGxvICsgYmwpID4+PiAwO1xuICBjYXJyeSArPSBsbyA8IGFsID8gMSA6IDA7XG4gIGxvID0gKGxvICsgY2wpID4+PiAwO1xuICBjYXJyeSArPSBsbyA8IGNsID8gMSA6IDA7XG4gIGxvID0gKGxvICsgZGwpID4+PiAwO1xuICBjYXJyeSArPSBsbyA8IGRsID8gMSA6IDA7XG4gIGxvID0gKGxvICsgZWwpID4+PiAwO1xuICBjYXJyeSArPSBsbyA8IGVsID8gMSA6IDA7XG5cbiAgdmFyIGhpID0gYWggKyBiaCArIGNoICsgZGggKyBlaCArIGNhcnJ5O1xuICByZXR1cm4gaGkgPj4+IDA7XG59XG5leHBvcnRzLnN1bTY0XzVfaGkgPSBzdW02NF81X2hpO1xuXG5mdW5jdGlvbiBzdW02NF81X2xvKGFoLCBhbCwgYmgsIGJsLCBjaCwgY2wsIGRoLCBkbCwgZWgsIGVsKSB7XG4gIHZhciBsbyA9IGFsICsgYmwgKyBjbCArIGRsICsgZWw7XG5cbiAgcmV0dXJuIGxvID4+PiAwO1xufVxuZXhwb3J0cy5zdW02NF81X2xvID0gc3VtNjRfNV9sbztcblxuZnVuY3Rpb24gcm90cjY0X2hpKGFoLCBhbCwgbnVtKSB7XG4gIHZhciByID0gKGFsIDw8ICgzMiAtIG51bSkpIHwgKGFoID4+PiBudW0pO1xuICByZXR1cm4gciA+Pj4gMDtcbn1cbmV4cG9ydHMucm90cjY0X2hpID0gcm90cjY0X2hpO1xuXG5mdW5jdGlvbiByb3RyNjRfbG8oYWgsIGFsLCBudW0pIHtcbiAgdmFyIHIgPSAoYWggPDwgKDMyIC0gbnVtKSkgfCAoYWwgPj4+IG51bSk7XG4gIHJldHVybiByID4+PiAwO1xufVxuZXhwb3J0cy5yb3RyNjRfbG8gPSByb3RyNjRfbG87XG5cbmZ1bmN0aW9uIHNocjY0X2hpKGFoLCBhbCwgbnVtKSB7XG4gIHJldHVybiBhaCA+Pj4gbnVtO1xufVxuZXhwb3J0cy5zaHI2NF9oaSA9IHNocjY0X2hpO1xuXG5mdW5jdGlvbiBzaHI2NF9sbyhhaCwgYWwsIG51bSkge1xuICB2YXIgciA9IChhaCA8PCAoMzIgLSBudW0pKSB8IChhbCA+Pj4gbnVtKTtcbiAgcmV0dXJuIHIgPj4+IDA7XG59XG5leHBvcnRzLnNocjY0X2xvID0gc2hyNjRfbG87XG4iXSwic291cmNlUm9vdCI6IiJ9\n//# sourceURL=webpack-internal:///6436\n")},5717:module=>{eval("if (typeof Object.create === 'function') {\n  // implementation from standard node.js 'util' module\n  module.exports = function inherits(ctor, superCtor) {\n    if (superCtor) {\n      ctor.super_ = superCtor\n      ctor.prototype = Object.create(superCtor.prototype, {\n        constructor: {\n          value: ctor,\n          enumerable: false,\n          writable: true,\n          configurable: true\n        }\n      })\n    }\n  };\n} else {\n  // old school shim for old browsers\n  module.exports = function inherits(ctor, superCtor) {\n    if (superCtor) {\n      ctor.super_ = superCtor\n      var TempCtor = function () {}\n      TempCtor.prototype = superCtor.prototype\n      ctor.prototype = new TempCtor()\n      ctor.prototype.constructor = ctor\n    }\n  }\n}\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2luaGVyaXRzL2luaGVyaXRzX2Jyb3dzZXIuanM/M2ZiNSJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQSxPQUFPO0FBQ1A7QUFDQTtBQUNBLENBQUM7QUFDRDtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBIiwiZmlsZSI6IjU3MTcuanMiLCJzb3VyY2VzQ29udGVudCI6WyJpZiAodHlwZW9mIE9iamVjdC5jcmVhdGUgPT09ICdmdW5jdGlvbicpIHtcbiAgLy8gaW1wbGVtZW50YXRpb24gZnJvbSBzdGFuZGFyZCBub2RlLmpzICd1dGlsJyBtb2R1bGVcbiAgbW9kdWxlLmV4cG9ydHMgPSBmdW5jdGlvbiBpbmhlcml0cyhjdG9yLCBzdXBlckN0b3IpIHtcbiAgICBpZiAoc3VwZXJDdG9yKSB7XG4gICAgICBjdG9yLnN1cGVyXyA9IHN1cGVyQ3RvclxuICAgICAgY3Rvci5wcm90b3R5cGUgPSBPYmplY3QuY3JlYXRlKHN1cGVyQ3Rvci5wcm90b3R5cGUsIHtcbiAgICAgICAgY29uc3RydWN0b3I6IHtcbiAgICAgICAgICB2YWx1ZTogY3RvcixcbiAgICAgICAgICBlbnVtZXJhYmxlOiBmYWxzZSxcbiAgICAgICAgICB3cml0YWJsZTogdHJ1ZSxcbiAgICAgICAgICBjb25maWd1cmFibGU6IHRydWVcbiAgICAgICAgfVxuICAgICAgfSlcbiAgICB9XG4gIH07XG59IGVsc2Uge1xuICAvLyBvbGQgc2Nob29sIHNoaW0gZm9yIG9sZCBicm93c2Vyc1xuICBtb2R1bGUuZXhwb3J0cyA9IGZ1bmN0aW9uIGluaGVyaXRzKGN0b3IsIHN1cGVyQ3Rvcikge1xuICAgIGlmIChzdXBlckN0b3IpIHtcbiAgICAgIGN0b3Iuc3VwZXJfID0gc3VwZXJDdG9yXG4gICAgICB2YXIgVGVtcEN0b3IgPSBmdW5jdGlvbiAoKSB7fVxuICAgICAgVGVtcEN0b3IucHJvdG90eXBlID0gc3VwZXJDdG9yLnByb3RvdHlwZVxuICAgICAgY3Rvci5wcm90b3R5cGUgPSBuZXcgVGVtcEN0b3IoKVxuICAgICAgY3Rvci5wcm90b3R5cGUuY29uc3RydWN0b3IgPSBjdG9yXG4gICAgfVxuICB9XG59XG4iXSwic291cmNlUm9vdCI6IiJ9\n//# sourceURL=webpack-internal:///5717\n")},9496:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\nvar has = __webpack_require__(7642);\nvar channel = __webpack_require__(7478)();\n\nvar $TypeError = GetIntrinsic('%TypeError%');\n\nvar SLOT = {\n\tassert: function (O, slot) {\n\t\tif (!O || (typeof O !== 'object' && typeof O !== 'function')) {\n\t\t\tthrow new $TypeError('`O` is not an object');\n\t\t}\n\t\tif (typeof slot !== 'string') {\n\t\t\tthrow new $TypeError('`slot` must be a string');\n\t\t}\n\t\tchannel.assert(O);\n\t},\n\tget: function (O, slot) {\n\t\tif (!O || (typeof O !== 'object' && typeof O !== 'function')) {\n\t\t\tthrow new $TypeError('`O` is not an object');\n\t\t}\n\t\tif (typeof slot !== 'string') {\n\t\t\tthrow new $TypeError('`slot` must be a string');\n\t\t}\n\t\tvar slots = channel.get(O);\n\t\treturn slots && slots['$' + slot];\n\t},\n\thas: function (O, slot) {\n\t\tif (!O || (typeof O !== 'object' && typeof O !== 'function')) {\n\t\t\tthrow new $TypeError('`O` is not an object');\n\t\t}\n\t\tif (typeof slot !== 'string') {\n\t\t\tthrow new $TypeError('`slot` must be a string');\n\t\t}\n\t\tvar slots = channel.get(O);\n\t\treturn !!slots && has(slots, '$' + slot);\n\t},\n\tset: function (O, slot, V) {\n\t\tif (!O || (typeof O !== 'object' && typeof O !== 'function')) {\n\t\t\tthrow new $TypeError('`O` is not an object');\n\t\t}\n\t\tif (typeof slot !== 'string') {\n\t\t\tthrow new $TypeError('`slot` must be a string');\n\t\t}\n\t\tvar slots = channel.get(O);\n\t\tif (!slots) {\n\t\t\tslots = {};\n\t\t\tchannel.set(O, slots);\n\t\t}\n\t\tslots['$' + slot] = V;\n\t}\n};\n\nif (Object.freeze) {\n\tObject.freeze(SLOT);\n}\n\nmodule.exports = SLOT;\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///9496\n")},5320:module=>{"use strict";eval("\n\nvar fnToStr = Function.prototype.toString;\nvar reflectApply = typeof Reflect === 'object' && Reflect !== null && Reflect.apply;\nvar badArrayLike;\nvar isCallableMarker;\nif (typeof reflectApply === 'function' && typeof Object.defineProperty === 'function') {\n\ttry {\n\t\tbadArrayLike = Object.defineProperty({}, 'length', {\n\t\t\tget: function () {\n\t\t\t\tthrow isCallableMarker;\n\t\t\t}\n\t\t});\n\t\tisCallableMarker = {};\n\t\t// eslint-disable-next-line no-throw-literal\n\t\treflectApply(function () { throw 42; }, null, badArrayLike);\n\t} catch (_) {\n\t\tif (_ !== isCallableMarker) {\n\t\t\treflectApply = null;\n\t\t}\n\t}\n} else {\n\treflectApply = null;\n}\n\nvar constructorRegex = /^\\s*class\\b/;\nvar isES6ClassFn = function isES6ClassFunction(value) {\n\ttry {\n\t\tvar fnStr = fnToStr.call(value);\n\t\treturn constructorRegex.test(fnStr);\n\t} catch (e) {\n\t\treturn false; // not a function\n\t}\n};\n\nvar tryFunctionObject = function tryFunctionToStr(value) {\n\ttry {\n\t\tif (isES6ClassFn(value)) { return false; }\n\t\tfnToStr.call(value);\n\t\treturn true;\n\t} catch (e) {\n\t\treturn false;\n\t}\n};\nvar toStr = Object.prototype.toString;\nvar fnClass = '[object Function]';\nvar genClass = '[object GeneratorFunction]';\nvar hasToStringTag = typeof Symbol === 'function' && !!Symbol.toStringTag; // better: use `has-tostringtag`\n/* globals document: false */\nvar documentDotAll = typeof document === 'object' && typeof document.all === 'undefined' && document.all !== undefined ? document.all : {};\n\nmodule.exports = reflectApply\n\t? function isCallable(value) {\n\t\tif (value === documentDotAll) { return true; }\n\t\tif (!value) { return false; }\n\t\tif (typeof value !== 'function' && typeof value !== 'object') { return false; }\n\t\tif (typeof value === 'function' && !value.prototype) { return true; }\n\t\ttry {\n\t\t\treflectApply(value, null, badArrayLike);\n\t\t} catch (e) {\n\t\t\tif (e !== isCallableMarker) { return false; }\n\t\t}\n\t\treturn !isES6ClassFn(value);\n\t}\n\t: function isCallable(value) {\n\t\tif (value === documentDotAll) { return true; }\n\t\tif (!value) { return false; }\n\t\tif (typeof value !== 'function' && typeof value !== 'object') { return false; }\n\t\tif (typeof value === 'function' && !value.prototype) { return true; }\n\t\tif (hasToStringTag) { return tryFunctionObject(value); }\n\t\tif (isES6ClassFn(value)) { return false; }\n\t\tvar strClass = toStr.call(value);\n\t\treturn strClass === fnClass || strClass === genClass;\n\t};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///5320\n")},8923:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar getDay = Date.prototype.getDay;\nvar tryDateObject = function tryDateGetDayCall(value) {\n\ttry {\n\t\tgetDay.call(value);\n\t\treturn true;\n\t} catch (e) {\n\t\treturn false;\n\t}\n};\n\nvar toStr = Object.prototype.toString;\nvar dateClass = '[object Date]';\nvar hasToStringTag = __webpack_require__(6410)();\n\nmodule.exports = function isDateObject(value) {\n\tif (typeof value !== 'object' || value === null) {\n\t\treturn false;\n\t}\n\treturn hasToStringTag ? tryDateObject(value) : toStr.call(value) === dateClass;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2lzLWRhdGUtb2JqZWN0L2luZGV4LmpzPzBlNjUiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQWE7O0FBRWI7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBLEVBQUU7QUFDRjtBQUNBO0FBQ0E7O0FBRUE7QUFDQTtBQUNBLHFCQUFxQixtQkFBTyxDQUFDLElBQXVCOztBQUVwRDtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0EiLCJmaWxlIjoiODkyMy5qcyIsInNvdXJjZXNDb250ZW50IjpbIid1c2Ugc3RyaWN0JztcblxudmFyIGdldERheSA9IERhdGUucHJvdG90eXBlLmdldERheTtcbnZhciB0cnlEYXRlT2JqZWN0ID0gZnVuY3Rpb24gdHJ5RGF0ZUdldERheUNhbGwodmFsdWUpIHtcblx0dHJ5IHtcblx0XHRnZXREYXkuY2FsbCh2YWx1ZSk7XG5cdFx0cmV0dXJuIHRydWU7XG5cdH0gY2F0Y2ggKGUpIHtcblx0XHRyZXR1cm4gZmFsc2U7XG5cdH1cbn07XG5cbnZhciB0b1N0ciA9IE9iamVjdC5wcm90b3R5cGUudG9TdHJpbmc7XG52YXIgZGF0ZUNsYXNzID0gJ1tvYmplY3QgRGF0ZV0nO1xudmFyIGhhc1RvU3RyaW5nVGFnID0gcmVxdWlyZSgnaGFzLXRvc3RyaW5ndGFnL3NoYW1zJykoKTtcblxubW9kdWxlLmV4cG9ydHMgPSBmdW5jdGlvbiBpc0RhdGVPYmplY3QodmFsdWUpIHtcblx0aWYgKHR5cGVvZiB2YWx1ZSAhPT0gJ29iamVjdCcgfHwgdmFsdWUgPT09IG51bGwpIHtcblx0XHRyZXR1cm4gZmFsc2U7XG5cdH1cblx0cmV0dXJuIGhhc1RvU3RyaW5nVGFnID8gdHJ5RGF0ZU9iamVjdCh2YWx1ZSkgOiB0b1N0ci5jYWxsKHZhbHVlKSA9PT0gZGF0ZUNsYXNzO1xufTtcbiJdLCJzb3VyY2VSb290IjoiIn0=\n//# sourceURL=webpack-internal:///8923\n")},8420:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar callBound = __webpack_require__(1924);\nvar hasToStringTag = __webpack_require__(6410)();\nvar has;\nvar $exec;\nvar isRegexMarker;\nvar badStringifier;\n\nif (hasToStringTag) {\n\thas = callBound('Object.prototype.hasOwnProperty');\n\t$exec = callBound('RegExp.prototype.exec');\n\tisRegexMarker = {};\n\n\tvar throwRegexMarker = function () {\n\t\tthrow isRegexMarker;\n\t};\n\tbadStringifier = {\n\t\ttoString: throwRegexMarker,\n\t\tvalueOf: throwRegexMarker\n\t};\n\n\tif (typeof Symbol.toPrimitive === 'symbol') {\n\t\tbadStringifier[Symbol.toPrimitive] = throwRegexMarker;\n\t}\n}\n\nvar $toString = callBound('Object.prototype.toString');\nvar gOPD = Object.getOwnPropertyDescriptor;\nvar regexClass = '[object RegExp]';\n\nmodule.exports = hasToStringTag\n\t// eslint-disable-next-line consistent-return\n\t? function isRegex(value) {\n\t\tif (!value || typeof value !== 'object') {\n\t\t\treturn false;\n\t\t}\n\n\t\tvar descriptor = gOPD(value, 'lastIndex');\n\t\tvar hasLastIndexDataProperty = descriptor && has(descriptor, 'value');\n\t\tif (!hasLastIndexDataProperty) {\n\t\t\treturn false;\n\t\t}\n\n\t\ttry {\n\t\t\t$exec(value, badStringifier);\n\t\t} catch (e) {\n\t\t\treturn e === isRegexMarker;\n\t\t}\n\t}\n\t: function isRegex(value) {\n\t\t// In older browsers, typeof regex incorrectly returns 'function'\n\t\tif (!value || (typeof value !== 'object' && typeof value !== 'function')) {\n\t\t\treturn false;\n\t\t}\n\n\t\treturn $toString(value) === regexClass;\n\t};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///8420\n")},2636:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar toStr = Object.prototype.toString;\nvar hasSymbols = __webpack_require__(1405)();\n\nif (hasSymbols) {\n\tvar symToStr = Symbol.prototype.toString;\n\tvar symStringRegex = /^Symbol\\(.*\\)$/;\n\tvar isSymbolObject = function isRealSymbolObject(value) {\n\t\tif (typeof value.valueOf() !== 'symbol') {\n\t\t\treturn false;\n\t\t}\n\t\treturn symStringRegex.test(symToStr.call(value));\n\t};\n\n\tmodule.exports = function isSymbol(value) {\n\t\tif (typeof value === 'symbol') {\n\t\t\treturn true;\n\t\t}\n\t\tif (toStr.call(value) !== '[object Symbol]') {\n\t\t\treturn false;\n\t\t}\n\t\ttry {\n\t\t\treturn isSymbolObject(value);\n\t\t} catch (e) {\n\t\t\treturn false;\n\t\t}\n\t};\n} else {\n\n\tmodule.exports = function isSymbol(value) {\n\t\t// this environment does not support Symbols.\n\t\treturn  false && 0;\n\t};\n}\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2lzLXN5bWJvbC9pbmRleC5qcz9mZWM1Il0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFhOztBQUViO0FBQ0EsaUJBQWlCLG1CQUFPLENBQUMsSUFBYTs7QUFFdEM7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBOztBQUVBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBLEdBQUc7QUFDSDtBQUNBO0FBQ0E7QUFDQSxDQUFDOztBQUVEO0FBQ0E7QUFDQSxTQUFTLE1BQUssSUFBSSxDQUFLO0FBQ3ZCO0FBQ0EiLCJmaWxlIjoiMjYzNi5qcyIsInNvdXJjZXNDb250ZW50IjpbIid1c2Ugc3RyaWN0JztcblxudmFyIHRvU3RyID0gT2JqZWN0LnByb3RvdHlwZS50b1N0cmluZztcbnZhciBoYXNTeW1ib2xzID0gcmVxdWlyZSgnaGFzLXN5bWJvbHMnKSgpO1xuXG5pZiAoaGFzU3ltYm9scykge1xuXHR2YXIgc3ltVG9TdHIgPSBTeW1ib2wucHJvdG90eXBlLnRvU3RyaW5nO1xuXHR2YXIgc3ltU3RyaW5nUmVnZXggPSAvXlN5bWJvbFxcKC4qXFwpJC87XG5cdHZhciBpc1N5bWJvbE9iamVjdCA9IGZ1bmN0aW9uIGlzUmVhbFN5bWJvbE9iamVjdCh2YWx1ZSkge1xuXHRcdGlmICh0eXBlb2YgdmFsdWUudmFsdWVPZigpICE9PSAnc3ltYm9sJykge1xuXHRcdFx0cmV0dXJuIGZhbHNlO1xuXHRcdH1cblx0XHRyZXR1cm4gc3ltU3RyaW5nUmVnZXgudGVzdChzeW1Ub1N0ci5jYWxsKHZhbHVlKSk7XG5cdH07XG5cblx0bW9kdWxlLmV4cG9ydHMgPSBmdW5jdGlvbiBpc1N5bWJvbCh2YWx1ZSkge1xuXHRcdGlmICh0eXBlb2YgdmFsdWUgPT09ICdzeW1ib2wnKSB7XG5cdFx0XHRyZXR1cm4gdHJ1ZTtcblx0XHR9XG5cdFx0aWYgKHRvU3RyLmNhbGwodmFsdWUpICE9PSAnW29iamVjdCBTeW1ib2xdJykge1xuXHRcdFx0cmV0dXJuIGZhbHNlO1xuXHRcdH1cblx0XHR0cnkge1xuXHRcdFx0cmV0dXJuIGlzU3ltYm9sT2JqZWN0KHZhbHVlKTtcblx0XHR9IGNhdGNoIChlKSB7XG5cdFx0XHRyZXR1cm4gZmFsc2U7XG5cdFx0fVxuXHR9O1xufSBlbHNlIHtcblxuXHRtb2R1bGUuZXhwb3J0cyA9IGZ1bmN0aW9uIGlzU3ltYm9sKHZhbHVlKSB7XG5cdFx0Ly8gdGhpcyBlbnZpcm9ubWVudCBkb2VzIG5vdCBzdXBwb3J0IFN5bWJvbHMuXG5cdFx0cmV0dXJuIGZhbHNlICYmIHZhbHVlO1xuXHR9O1xufVxuIl0sInNvdXJjZVJvb3QiOiIifQ==\n//# sourceURL=webpack-internal:///2636\n")},9746:module=>{eval("module.exports = assert;\n\nfunction assert(val, msg) {\n  if (!val)\n    throw new Error(msg || 'Assertion failed');\n}\n\nassert.equal = function assertEqual(l, r, msg) {\n  if (l != r)\n    throw new Error(msg || ('Assertion failed: ' + l + ' != ' + r));\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL21pbmltYWxpc3RpYy1hc3NlcnQvaW5kZXguanM/ZGEzZSJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBQTs7QUFFQTtBQUNBO0FBQ0E7QUFDQTs7QUFFQTtBQUNBO0FBQ0E7QUFDQSIsImZpbGUiOiI5NzQ2LmpzIiwic291cmNlc0NvbnRlbnQiOlsibW9kdWxlLmV4cG9ydHMgPSBhc3NlcnQ7XG5cbmZ1bmN0aW9uIGFzc2VydCh2YWwsIG1zZykge1xuICBpZiAoIXZhbClcbiAgICB0aHJvdyBuZXcgRXJyb3IobXNnIHx8ICdBc3NlcnRpb24gZmFpbGVkJyk7XG59XG5cbmFzc2VydC5lcXVhbCA9IGZ1bmN0aW9uIGFzc2VydEVxdWFsKGwsIHIsIG1zZykge1xuICBpZiAobCAhPSByKVxuICAgIHRocm93IG5ldyBFcnJvcihtc2cgfHwgKCdBc3NlcnRpb24gZmFpbGVkOiAnICsgbCArICcgIT0gJyArIHIpKTtcbn07XG4iXSwic291cmNlUm9vdCI6IiJ9\n//# sourceURL=webpack-internal:///9746\n")},631:(module,__unused_webpack_exports,__webpack_require__)=>{eval("var hasMap = typeof Map === 'function' && Map.prototype;\nvar mapSizeDescriptor = Object.getOwnPropertyDescriptor && hasMap ? Object.getOwnPropertyDescriptor(Map.prototype, 'size') : null;\nvar mapSize = hasMap && mapSizeDescriptor && typeof mapSizeDescriptor.get === 'function' ? mapSizeDescriptor.get : null;\nvar mapForEach = hasMap && Map.prototype.forEach;\nvar hasSet = typeof Set === 'function' && Set.prototype;\nvar setSizeDescriptor = Object.getOwnPropertyDescriptor && hasSet ? Object.getOwnPropertyDescriptor(Set.prototype, 'size') : null;\nvar setSize = hasSet && setSizeDescriptor && typeof setSizeDescriptor.get === 'function' ? setSizeDescriptor.get : null;\nvar setForEach = hasSet && Set.prototype.forEach;\nvar hasWeakMap = typeof WeakMap === 'function' && WeakMap.prototype;\nvar weakMapHas = hasWeakMap ? WeakMap.prototype.has : null;\nvar hasWeakSet = typeof WeakSet === 'function' && WeakSet.prototype;\nvar weakSetHas = hasWeakSet ? WeakSet.prototype.has : null;\nvar hasWeakRef = typeof WeakRef === 'function' && WeakRef.prototype;\nvar weakRefDeref = hasWeakRef ? WeakRef.prototype.deref : null;\nvar booleanValueOf = Boolean.prototype.valueOf;\nvar objectToString = Object.prototype.toString;\nvar functionToString = Function.prototype.toString;\nvar match = String.prototype.match;\nvar bigIntValueOf = typeof BigInt === 'function' ? BigInt.prototype.valueOf : null;\nvar gOPS = Object.getOwnPropertySymbols;\nvar symToString = typeof Symbol === 'function' && typeof Symbol.iterator === 'symbol' ? Symbol.prototype.toString : null;\nvar hasShammedSymbols = typeof Symbol === 'function' && typeof Symbol.iterator === 'object';\nvar isEnumerable = Object.prototype.propertyIsEnumerable;\n\nvar gPO = (typeof Reflect === 'function' ? Reflect.getPrototypeOf : Object.getPrototypeOf) || (\n    [].__proto__ === Array.prototype // eslint-disable-line no-proto\n        ? function (O) {\n            return O.__proto__; // eslint-disable-line no-proto\n        }\n        : null\n);\n\nvar inspectCustom = __webpack_require__(4654).custom;\nvar inspectSymbol = inspectCustom && isSymbol(inspectCustom) ? inspectCustom : null;\nvar toStringTag = typeof Symbol === 'function' && typeof Symbol.toStringTag !== 'undefined' ? Symbol.toStringTag : null;\n\nmodule.exports = function inspect_(obj, options, depth, seen) {\n    var opts = options || {};\n\n    if (has(opts, 'quoteStyle') && (opts.quoteStyle !== 'single' && opts.quoteStyle !== 'double')) {\n        throw new TypeError('option \"quoteStyle\" must be \"single\" or \"double\"');\n    }\n    if (\n        has(opts, 'maxStringLength') && (typeof opts.maxStringLength === 'number'\n            ? opts.maxStringLength < 0 && opts.maxStringLength !== Infinity\n            : opts.maxStringLength !== null\n        )\n    ) {\n        throw new TypeError('option \"maxStringLength\", if provided, must be a positive integer, Infinity, or `null`');\n    }\n    var customInspect = has(opts, 'customInspect') ? opts.customInspect : true;\n    if (typeof customInspect !== 'boolean' && customInspect !== 'symbol') {\n        throw new TypeError('option \"customInspect\", if provided, must be `true`, `false`, or `\\'symbol\\'`');\n    }\n\n    if (\n        has(opts, 'indent')\n        && opts.indent !== null\n        && opts.indent !== '\\t'\n        && !(parseInt(opts.indent, 10) === opts.indent && opts.indent > 0)\n    ) {\n        throw new TypeError('options \"indent\" must be \"\\\\t\", an integer > 0, or `null`');\n    }\n\n    if (typeof obj === 'undefined') {\n        return 'undefined';\n    }\n    if (obj === null) {\n        return 'null';\n    }\n    if (typeof obj === 'boolean') {\n        return obj ? 'true' : 'false';\n    }\n\n    if (typeof obj === 'string') {\n        return inspectString(obj, opts);\n    }\n    if (typeof obj === 'number') {\n        if (obj === 0) {\n            return Infinity / obj > 0 ? '0' : '-0';\n        }\n        return String(obj);\n    }\n    if (typeof obj === 'bigint') {\n        return String(obj) + 'n';\n    }\n\n    var maxDepth = typeof opts.depth === 'undefined' ? 5 : opts.depth;\n    if (typeof depth === 'undefined') { depth = 0; }\n    if (depth >= maxDepth && maxDepth > 0 && typeof obj === 'object') {\n        return isArray(obj) ? '[Array]' : '[Object]';\n    }\n\n    var indent = getIndent(opts, depth);\n\n    if (typeof seen === 'undefined') {\n        seen = [];\n    } else if (indexOf(seen, obj) >= 0) {\n        return '[Circular]';\n    }\n\n    function inspect(value, from, noIndent) {\n        if (from) {\n            seen = seen.slice();\n            seen.push(from);\n        }\n        if (noIndent) {\n            var newOpts = {\n                depth: opts.depth\n            };\n            if (has(opts, 'quoteStyle')) {\n                newOpts.quoteStyle = opts.quoteStyle;\n            }\n            return inspect_(value, newOpts, depth + 1, seen);\n        }\n        return inspect_(value, opts, depth + 1, seen);\n    }\n\n    if (typeof obj === 'function') {\n        var name = nameOf(obj);\n        var keys = arrObjKeys(obj, inspect);\n        return '[Function' + (name ? ': ' + name : ' (anonymous)') + ']' + (keys.length > 0 ? ' { ' + keys.join(', ') + ' }' : '');\n    }\n    if (isSymbol(obj)) {\n        var symString = hasShammedSymbols ? String(obj).replace(/^(Symbol\\(.*\\))_[^)]*$/, '$1') : symToString.call(obj);\n        return typeof obj === 'object' && !hasShammedSymbols ? markBoxed(symString) : symString;\n    }\n    if (isElement(obj)) {\n        var s = '<' + String(obj.nodeName).toLowerCase();\n        var attrs = obj.attributes || [];\n        for (var i = 0; i < attrs.length; i++) {\n            s += ' ' + attrs[i].name + '=' + wrapQuotes(quote(attrs[i].value), 'double', opts);\n        }\n        s += '>';\n        if (obj.childNodes && obj.childNodes.length) { s += '...'; }\n        s += '</' + String(obj.nodeName).toLowerCase() + '>';\n        return s;\n    }\n    if (isArray(obj)) {\n        if (obj.length === 0) { return '[]'; }\n        var xs = arrObjKeys(obj, inspect);\n        if (indent && !singleLineValues(xs)) {\n            return '[' + indentedJoin(xs, indent) + ']';\n        }\n        return '[ ' + xs.join(', ') + ' ]';\n    }\n    if (isError(obj)) {\n        var parts = arrObjKeys(obj, inspect);\n        if (parts.length === 0) { return '[' + String(obj) + ']'; }\n        return '{ [' + String(obj) + '] ' + parts.join(', ') + ' }';\n    }\n    if (typeof obj === 'object' && customInspect) {\n        if (inspectSymbol && typeof obj[inspectSymbol] === 'function') {\n            return obj[inspectSymbol]();\n        } else if (customInspect !== 'symbol' && typeof obj.inspect === 'function') {\n            return obj.inspect();\n        }\n    }\n    if (isMap(obj)) {\n        var mapParts = [];\n        mapForEach.call(obj, function (value, key) {\n            mapParts.push(inspect(key, obj, true) + ' => ' + inspect(value, obj));\n        });\n        return collectionOf('Map', mapSize.call(obj), mapParts, indent);\n    }\n    if (isSet(obj)) {\n        var setParts = [];\n        setForEach.call(obj, function (value) {\n            setParts.push(inspect(value, obj));\n        });\n        return collectionOf('Set', setSize.call(obj), setParts, indent);\n    }\n    if (isWeakMap(obj)) {\n        return weakCollectionOf('WeakMap');\n    }\n    if (isWeakSet(obj)) {\n        return weakCollectionOf('WeakSet');\n    }\n    if (isWeakRef(obj)) {\n        return weakCollectionOf('WeakRef');\n    }\n    if (isNumber(obj)) {\n        return markBoxed(inspect(Number(obj)));\n    }\n    if (isBigInt(obj)) {\n        return markBoxed(inspect(bigIntValueOf.call(obj)));\n    }\n    if (isBoolean(obj)) {\n        return markBoxed(booleanValueOf.call(obj));\n    }\n    if (isString(obj)) {\n        return markBoxed(inspect(String(obj)));\n    }\n    if (!isDate(obj) && !isRegExp(obj)) {\n        var ys = arrObjKeys(obj, inspect);\n        var isPlainObject = gPO ? gPO(obj) === Object.prototype : obj instanceof Object || obj.constructor === Object;\n        var protoTag = obj instanceof Object ? '' : 'null prototype';\n        var stringTag = !isPlainObject && toStringTag && Object(obj) === obj && toStringTag in obj ? toStr(obj).slice(8, -1) : protoTag ? 'Object' : '';\n        var constructorTag = isPlainObject || typeof obj.constructor !== 'function' ? '' : obj.constructor.name ? obj.constructor.name + ' ' : '';\n        var tag = constructorTag + (stringTag || protoTag ? '[' + [].concat(stringTag || [], protoTag || []).join(': ') + '] ' : '');\n        if (ys.length === 0) { return tag + '{}'; }\n        if (indent) {\n            return tag + '{' + indentedJoin(ys, indent) + '}';\n        }\n        return tag + '{ ' + ys.join(', ') + ' }';\n    }\n    return String(obj);\n};\n\nfunction wrapQuotes(s, defaultStyle, opts) {\n    var quoteChar = (opts.quoteStyle || defaultStyle) === 'double' ? '\"' : \"'\";\n    return quoteChar + s + quoteChar;\n}\n\nfunction quote(s) {\n    return String(s).replace(/\"/g, '&quot;');\n}\n\nfunction isArray(obj) { return toStr(obj) === '[object Array]' && (!toStringTag || !(typeof obj === 'object' && toStringTag in obj)); }\nfunction isDate(obj) { return toStr(obj) === '[object Date]' && (!toStringTag || !(typeof obj === 'object' && toStringTag in obj)); }\nfunction isRegExp(obj) { return toStr(obj) === '[object RegExp]' && (!toStringTag || !(typeof obj === 'object' && toStringTag in obj)); }\nfunction isError(obj) { return toStr(obj) === '[object Error]' && (!toStringTag || !(typeof obj === 'object' && toStringTag in obj)); }\nfunction isString(obj) { return toStr(obj) === '[object String]' && (!toStringTag || !(typeof obj === 'object' && toStringTag in obj)); }\nfunction isNumber(obj) { return toStr(obj) === '[object Number]' && (!toStringTag || !(typeof obj === 'object' && toStringTag in obj)); }\nfunction isBoolean(obj) { return toStr(obj) === '[object Boolean]' && (!toStringTag || !(typeof obj === 'object' && toStringTag in obj)); }\n\n// Symbol and BigInt do have Symbol.toStringTag by spec, so that can't be used to eliminate false positives\nfunction isSymbol(obj) {\n    if (hasShammedSymbols) {\n        return obj && typeof obj === 'object' && obj instanceof Symbol;\n    }\n    if (typeof obj === 'symbol') {\n        return true;\n    }\n    if (!obj || typeof obj !== 'object' || !symToString) {\n        return false;\n    }\n    try {\n        symToString.call(obj);\n        return true;\n    } catch (e) {}\n    return false;\n}\n\nfunction isBigInt(obj) {\n    if (!obj || typeof obj !== 'object' || !bigIntValueOf) {\n        return false;\n    }\n    try {\n        bigIntValueOf.call(obj);\n        return true;\n    } catch (e) {}\n    return false;\n}\n\nvar hasOwn = Object.prototype.hasOwnProperty || function (key) { return key in this; };\nfunction has(obj, key) {\n    return hasOwn.call(obj, key);\n}\n\nfunction toStr(obj) {\n    return objectToString.call(obj);\n}\n\nfunction nameOf(f) {\n    if (f.name) { return f.name; }\n    var m = match.call(functionToString.call(f), /^function\\s*([\\w$]+)/);\n    if (m) { return m[1]; }\n    return null;\n}\n\nfunction indexOf(xs, x) {\n    if (xs.indexOf) { return xs.indexOf(x); }\n    for (var i = 0, l = xs.length; i < l; i++) {\n        if (xs[i] === x) { return i; }\n    }\n    return -1;\n}\n\nfunction isMap(x) {\n    if (!mapSize || !x || typeof x !== 'object') {\n        return false;\n    }\n    try {\n        mapSize.call(x);\n        try {\n            setSize.call(x);\n        } catch (s) {\n            return true;\n        }\n        return x instanceof Map; // core-js workaround, pre-v2.5.0\n    } catch (e) {}\n    return false;\n}\n\nfunction isWeakMap(x) {\n    if (!weakMapHas || !x || typeof x !== 'object') {\n        return false;\n    }\n    try {\n        weakMapHas.call(x, weakMapHas);\n        try {\n            weakSetHas.call(x, weakSetHas);\n        } catch (s) {\n            return true;\n        }\n        return x instanceof WeakMap; // core-js workaround, pre-v2.5.0\n    } catch (e) {}\n    return false;\n}\n\nfunction isWeakRef(x) {\n    if (!weakRefDeref || !x || typeof x !== 'object') {\n        return false;\n    }\n    try {\n        weakRefDeref.call(x);\n        return true;\n    } catch (e) {}\n    return false;\n}\n\nfunction isSet(x) {\n    if (!setSize || !x || typeof x !== 'object') {\n        return false;\n    }\n    try {\n        setSize.call(x);\n        try {\n            mapSize.call(x);\n        } catch (m) {\n            return true;\n        }\n        return x instanceof Set; // core-js workaround, pre-v2.5.0\n    } catch (e) {}\n    return false;\n}\n\nfunction isWeakSet(x) {\n    if (!weakSetHas || !x || typeof x !== 'object') {\n        return false;\n    }\n    try {\n        weakSetHas.call(x, weakSetHas);\n        try {\n            weakMapHas.call(x, weakMapHas);\n        } catch (s) {\n            return true;\n        }\n        return x instanceof WeakSet; // core-js workaround, pre-v2.5.0\n    } catch (e) {}\n    return false;\n}\n\nfunction isElement(x) {\n    if (!x || typeof x !== 'object') { return false; }\n    if (typeof HTMLElement !== 'undefined' && x instanceof HTMLElement) {\n        return true;\n    }\n    return typeof x.nodeName === 'string' && typeof x.getAttribute === 'function';\n}\n\nfunction inspectString(str, opts) {\n    if (str.length > opts.maxStringLength) {\n        var remaining = str.length - opts.maxStringLength;\n        var trailer = '... ' + remaining + ' more character' + (remaining > 1 ? 's' : '');\n        return inspectString(str.slice(0, opts.maxStringLength), opts) + trailer;\n    }\n    // eslint-disable-next-line no-control-regex\n    var s = str.replace(/(['\\\\])/g, '\\\\$1').replace(/[\\x00-\\x1f]/g, lowbyte);\n    return wrapQuotes(s, 'single', opts);\n}\n\nfunction lowbyte(c) {\n    var n = c.charCodeAt(0);\n    var x = {\n        8: 'b',\n        9: 't',\n        10: 'n',\n        12: 'f',\n        13: 'r'\n    }[n];\n    if (x) { return '\\\\' + x; }\n    return '\\\\x' + (n < 0x10 ? '0' : '') + n.toString(16).toUpperCase();\n}\n\nfunction markBoxed(str) {\n    return 'Object(' + str + ')';\n}\n\nfunction weakCollectionOf(type) {\n    return type + ' { ? }';\n}\n\nfunction collectionOf(type, size, entries, indent) {\n    var joinedEntries = indent ? indentedJoin(entries, indent) : entries.join(', ');\n    return type + ' (' + size + ') {' + joinedEntries + '}';\n}\n\nfunction singleLineValues(xs) {\n    for (var i = 0; i < xs.length; i++) {\n        if (indexOf(xs[i], '\\n') >= 0) {\n            return false;\n        }\n    }\n    return true;\n}\n\nfunction getIndent(opts, depth) {\n    var baseIndent;\n    if (opts.indent === '\\t') {\n        baseIndent = '\\t';\n    } else if (typeof opts.indent === 'number' && opts.indent > 0) {\n        baseIndent = Array(opts.indent + 1).join(' ');\n    } else {\n        return null;\n    }\n    return {\n        base: baseIndent,\n        prev: Array(depth + 1).join(baseIndent)\n    };\n}\n\nfunction indentedJoin(xs, indent) {\n    if (xs.length === 0) { return ''; }\n    var lineJoiner = '\\n' + indent.prev + indent.base;\n    return lineJoiner + xs.join(',' + lineJoiner) + '\\n' + indent.prev;\n}\n\nfunction arrObjKeys(obj, inspect) {\n    var isArr = isArray(obj);\n    var xs = [];\n    if (isArr) {\n        xs.length = obj.length;\n        for (var i = 0; i < obj.length; i++) {\n            xs[i] = has(obj, i) ? inspect(obj[i], obj) : '';\n        }\n    }\n    var syms = typeof gOPS === 'function' ? gOPS(obj) : [];\n    var symMap;\n    if (hasShammedSymbols) {\n        symMap = {};\n        for (var k = 0; k < syms.length; k++) {\n            symMap['$' + syms[k]] = syms[k];\n        }\n    }\n\n    for (var key in obj) { // eslint-disable-line no-restricted-syntax\n        if (!has(obj, key)) { continue; } // eslint-disable-line no-restricted-syntax, no-continue\n        if (isArr && String(Number(key)) === key && key < obj.length) { continue; } // eslint-disable-line no-restricted-syntax, no-continue\n        if (hasShammedSymbols && symMap['$' + key] instanceof Symbol) {\n            // this is to prevent shammed Symbols, which are stored as strings, from being included in the string key section\n            continue; // eslint-disable-line no-restricted-syntax, no-continue\n        } else if ((/[^\\w$]/).test(key)) {\n            xs.push(inspect(key, obj) + ': ' + inspect(obj[key], obj));\n        } else {\n            xs.push(key + ': ' + inspect(obj[key], obj));\n        }\n    }\n    if (typeof gOPS === 'function') {\n        for (var j = 0; j < syms.length; j++) {\n            if (isEnumerable.call(obj, syms[j])) {\n                xs.push('[' + inspect(syms[j]) + ']: ' + inspect(obj[syms[j]], obj));\n            }\n        }\n    }\n    return xs;\n}\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///631\n")},8987:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar keysShim;\nif (!Object.keys) {\n\t// modified from https://github.com/es-shims/es5-shim\n\tvar has = Object.prototype.hasOwnProperty;\n\tvar toStr = Object.prototype.toString;\n\tvar isArgs = __webpack_require__(1414); // eslint-disable-line global-require\n\tvar isEnumerable = Object.prototype.propertyIsEnumerable;\n\tvar hasDontEnumBug = !isEnumerable.call({ toString: null }, 'toString');\n\tvar hasProtoEnumBug = isEnumerable.call(function () {}, 'prototype');\n\tvar dontEnums = [\n\t\t'toString',\n\t\t'toLocaleString',\n\t\t'valueOf',\n\t\t'hasOwnProperty',\n\t\t'isPrototypeOf',\n\t\t'propertyIsEnumerable',\n\t\t'constructor'\n\t];\n\tvar equalsConstructorPrototype = function (o) {\n\t\tvar ctor = o.constructor;\n\t\treturn ctor && ctor.prototype === o;\n\t};\n\tvar excludedKeys = {\n\t\t$applicationCache: true,\n\t\t$console: true,\n\t\t$external: true,\n\t\t$frame: true,\n\t\t$frameElement: true,\n\t\t$frames: true,\n\t\t$innerHeight: true,\n\t\t$innerWidth: true,\n\t\t$onmozfullscreenchange: true,\n\t\t$onmozfullscreenerror: true,\n\t\t$outerHeight: true,\n\t\t$outerWidth: true,\n\t\t$pageXOffset: true,\n\t\t$pageYOffset: true,\n\t\t$parent: true,\n\t\t$scrollLeft: true,\n\t\t$scrollTop: true,\n\t\t$scrollX: true,\n\t\t$scrollY: true,\n\t\t$self: true,\n\t\t$webkitIndexedDB: true,\n\t\t$webkitStorageInfo: true,\n\t\t$window: true\n\t};\n\tvar hasAutomationEqualityBug = (function () {\n\t\t/* global window */\n\t\tif (typeof window === 'undefined') { return false; }\n\t\tfor (var k in window) {\n\t\t\ttry {\n\t\t\t\tif (!excludedKeys['$' + k] && has.call(window, k) && window[k] !== null && typeof window[k] === 'object') {\n\t\t\t\t\ttry {\n\t\t\t\t\t\tequalsConstructorPrototype(window[k]);\n\t\t\t\t\t} catch (e) {\n\t\t\t\t\t\treturn true;\n\t\t\t\t\t}\n\t\t\t\t}\n\t\t\t} catch (e) {\n\t\t\t\treturn true;\n\t\t\t}\n\t\t}\n\t\treturn false;\n\t}());\n\tvar equalsConstructorPrototypeIfNotBuggy = function (o) {\n\t\t/* global window */\n\t\tif (typeof window === 'undefined' || !hasAutomationEqualityBug) {\n\t\t\treturn equalsConstructorPrototype(o);\n\t\t}\n\t\ttry {\n\t\t\treturn equalsConstructorPrototype(o);\n\t\t} catch (e) {\n\t\t\treturn false;\n\t\t}\n\t};\n\n\tkeysShim = function keys(object) {\n\t\tvar isObject = object !== null && typeof object === 'object';\n\t\tvar isFunction = toStr.call(object) === '[object Function]';\n\t\tvar isArguments = isArgs(object);\n\t\tvar isString = isObject && toStr.call(object) === '[object String]';\n\t\tvar theKeys = [];\n\n\t\tif (!isObject && !isFunction && !isArguments) {\n\t\t\tthrow new TypeError('Object.keys called on a non-object');\n\t\t}\n\n\t\tvar skipProto = hasProtoEnumBug && isFunction;\n\t\tif (isString && object.length > 0 && !has.call(object, 0)) {\n\t\t\tfor (var i = 0; i < object.length; ++i) {\n\t\t\t\ttheKeys.push(String(i));\n\t\t\t}\n\t\t}\n\n\t\tif (isArguments && object.length > 0) {\n\t\t\tfor (var j = 0; j < object.length; ++j) {\n\t\t\t\ttheKeys.push(String(j));\n\t\t\t}\n\t\t} else {\n\t\t\tfor (var name in object) {\n\t\t\t\tif (!(skipProto && name === 'prototype') && has.call(object, name)) {\n\t\t\t\t\ttheKeys.push(String(name));\n\t\t\t\t}\n\t\t\t}\n\t\t}\n\n\t\tif (hasDontEnumBug) {\n\t\t\tvar skipConstructor = equalsConstructorPrototypeIfNotBuggy(object);\n\n\t\t\tfor (var k = 0; k < dontEnums.length; ++k) {\n\t\t\t\tif (!(skipConstructor && dontEnums[k] === 'constructor') && has.call(object, dontEnums[k])) {\n\t\t\t\t\ttheKeys.push(dontEnums[k]);\n\t\t\t\t}\n\t\t\t}\n\t\t}\n\t\treturn theKeys;\n\t};\n}\nmodule.exports = keysShim;\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///8987\n")},2215:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar slice = Array.prototype.slice;\nvar isArgs = __webpack_require__(1414);\n\nvar origKeys = Object.keys;\nvar keysShim = origKeys ? function keys(o) { return origKeys(o); } : __webpack_require__(8987);\n\nvar originalKeys = Object.keys;\n\nkeysShim.shim = function shimObjectKeys() {\n\tif (Object.keys) {\n\t\tvar keysWorksWithArguments = (function () {\n\t\t\t// Safari 5.0 bug\n\t\t\tvar args = Object.keys(arguments);\n\t\t\treturn args && args.length === arguments.length;\n\t\t}(1, 2));\n\t\tif (!keysWorksWithArguments) {\n\t\t\tObject.keys = function keys(object) { // eslint-disable-line func-name-matching\n\t\t\t\tif (isArgs(object)) {\n\t\t\t\t\treturn originalKeys(slice.call(object));\n\t\t\t\t}\n\t\t\t\treturn originalKeys(object);\n\t\t\t};\n\t\t}\n\t} else {\n\t\tObject.keys = keysShim;\n\t}\n\treturn Object.keys || keysShim;\n};\n\nmodule.exports = keysShim;\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL29iamVjdC1rZXlzL2luZGV4LmpzP2Q2YzciXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQWE7O0FBRWI7QUFDQSxhQUFhLG1CQUFPLENBQUMsSUFBZTs7QUFFcEM7QUFDQSw0Q0FBNEMsb0JBQW9CLEVBQUUsR0FBRyxtQkFBTyxDQUFDLElBQWtCOztBQUUvRjs7QUFFQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQSxHQUFHO0FBQ0g7QUFDQSx3Q0FBd0M7QUFDeEM7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0EsRUFBRTtBQUNGO0FBQ0E7QUFDQTtBQUNBOztBQUVBIiwiZmlsZSI6IjIyMTUuanMiLCJzb3VyY2VzQ29udGVudCI6WyIndXNlIHN0cmljdCc7XG5cbnZhciBzbGljZSA9IEFycmF5LnByb3RvdHlwZS5zbGljZTtcbnZhciBpc0FyZ3MgPSByZXF1aXJlKCcuL2lzQXJndW1lbnRzJyk7XG5cbnZhciBvcmlnS2V5cyA9IE9iamVjdC5rZXlzO1xudmFyIGtleXNTaGltID0gb3JpZ0tleXMgPyBmdW5jdGlvbiBrZXlzKG8pIHsgcmV0dXJuIG9yaWdLZXlzKG8pOyB9IDogcmVxdWlyZSgnLi9pbXBsZW1lbnRhdGlvbicpO1xuXG52YXIgb3JpZ2luYWxLZXlzID0gT2JqZWN0LmtleXM7XG5cbmtleXNTaGltLnNoaW0gPSBmdW5jdGlvbiBzaGltT2JqZWN0S2V5cygpIHtcblx0aWYgKE9iamVjdC5rZXlzKSB7XG5cdFx0dmFyIGtleXNXb3Jrc1dpdGhBcmd1bWVudHMgPSAoZnVuY3Rpb24gKCkge1xuXHRcdFx0Ly8gU2FmYXJpIDUuMCBidWdcblx0XHRcdHZhciBhcmdzID0gT2JqZWN0LmtleXMoYXJndW1lbnRzKTtcblx0XHRcdHJldHVybiBhcmdzICYmIGFyZ3MubGVuZ3RoID09PSBhcmd1bWVudHMubGVuZ3RoO1xuXHRcdH0oMSwgMikpO1xuXHRcdGlmICgha2V5c1dvcmtzV2l0aEFyZ3VtZW50cykge1xuXHRcdFx0T2JqZWN0LmtleXMgPSBmdW5jdGlvbiBrZXlzKG9iamVjdCkgeyAvLyBlc2xpbnQtZGlzYWJsZS1saW5lIGZ1bmMtbmFtZS1tYXRjaGluZ1xuXHRcdFx0XHRpZiAoaXNBcmdzKG9iamVjdCkpIHtcblx0XHRcdFx0XHRyZXR1cm4gb3JpZ2luYWxLZXlzKHNsaWNlLmNhbGwob2JqZWN0KSk7XG5cdFx0XHRcdH1cblx0XHRcdFx0cmV0dXJuIG9yaWdpbmFsS2V5cyhvYmplY3QpO1xuXHRcdFx0fTtcblx0XHR9XG5cdH0gZWxzZSB7XG5cdFx0T2JqZWN0LmtleXMgPSBrZXlzU2hpbTtcblx0fVxuXHRyZXR1cm4gT2JqZWN0LmtleXMgfHwga2V5c1NoaW07XG59O1xuXG5tb2R1bGUuZXhwb3J0cyA9IGtleXNTaGltO1xuIl0sInNvdXJjZVJvb3QiOiIifQ==\n//# sourceURL=webpack-internal:///2215\n")},1414:module=>{"use strict";eval("\n\nvar toStr = Object.prototype.toString;\n\nmodule.exports = function isArguments(value) {\n\tvar str = toStr.call(value);\n\tvar isArgs = str === '[object Arguments]';\n\tif (!isArgs) {\n\t\tisArgs = str !== '[object Array]' &&\n\t\t\tvalue !== null &&\n\t\t\ttypeof value === 'object' &&\n\t\t\ttypeof value.length === 'number' &&\n\t\t\tvalue.length >= 0 &&\n\t\t\ttoStr.call(value.callee) === '[object Function]';\n\t}\n\treturn isArgs;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL29iamVjdC1rZXlzL2lzQXJndW1lbnRzLmpzP2Q0YWIiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQWE7O0FBRWI7O0FBRUE7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0EiLCJmaWxlIjoiMTQxNC5qcyIsInNvdXJjZXNDb250ZW50IjpbIid1c2Ugc3RyaWN0JztcblxudmFyIHRvU3RyID0gT2JqZWN0LnByb3RvdHlwZS50b1N0cmluZztcblxubW9kdWxlLmV4cG9ydHMgPSBmdW5jdGlvbiBpc0FyZ3VtZW50cyh2YWx1ZSkge1xuXHR2YXIgc3RyID0gdG9TdHIuY2FsbCh2YWx1ZSk7XG5cdHZhciBpc0FyZ3MgPSBzdHIgPT09ICdbb2JqZWN0IEFyZ3VtZW50c10nO1xuXHRpZiAoIWlzQXJncykge1xuXHRcdGlzQXJncyA9IHN0ciAhPT0gJ1tvYmplY3QgQXJyYXldJyAmJlxuXHRcdFx0dmFsdWUgIT09IG51bGwgJiZcblx0XHRcdHR5cGVvZiB2YWx1ZSA9PT0gJ29iamVjdCcgJiZcblx0XHRcdHR5cGVvZiB2YWx1ZS5sZW5ndGggPT09ICdudW1iZXInICYmXG5cdFx0XHR2YWx1ZS5sZW5ndGggPj0gMCAmJlxuXHRcdFx0dG9TdHIuY2FsbCh2YWx1ZS5jYWxsZWUpID09PSAnW29iamVjdCBGdW5jdGlvbl0nO1xuXHR9XG5cdHJldHVybiBpc0FyZ3M7XG59O1xuIl0sInNvdXJjZVJvb3QiOiIifQ==\n//# sourceURL=webpack-internal:///1414\n")},3697:module=>{"use strict";eval("\n\nvar $Object = Object;\nvar $TypeError = TypeError;\n\nmodule.exports = function flags() {\n\tif (this != null && this !== $Object(this)) {\n\t\tthrow new $TypeError('RegExp.prototype.flags getter called on non-object');\n\t}\n\tvar result = '';\n\tif (this.global) {\n\t\tresult += 'g';\n\t}\n\tif (this.ignoreCase) {\n\t\tresult += 'i';\n\t}\n\tif (this.multiline) {\n\t\tresult += 'm';\n\t}\n\tif (this.dotAll) {\n\t\tresult += 's';\n\t}\n\tif (this.unicode) {\n\t\tresult += 'u';\n\t}\n\tif (this.sticky) {\n\t\tresult += 'y';\n\t}\n\treturn result;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL3JlZ2V4cC5wcm90b3R5cGUuZmxhZ3MvaW1wbGVtZW50YXRpb24uanM/NTcwOCJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBYTs7QUFFYjtBQUNBOztBQUVBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBIiwiZmlsZSI6IjM2OTcuanMiLCJzb3VyY2VzQ29udGVudCI6WyIndXNlIHN0cmljdCc7XG5cbnZhciAkT2JqZWN0ID0gT2JqZWN0O1xudmFyICRUeXBlRXJyb3IgPSBUeXBlRXJyb3I7XG5cbm1vZHVsZS5leHBvcnRzID0gZnVuY3Rpb24gZmxhZ3MoKSB7XG5cdGlmICh0aGlzICE9IG51bGwgJiYgdGhpcyAhPT0gJE9iamVjdCh0aGlzKSkge1xuXHRcdHRocm93IG5ldyAkVHlwZUVycm9yKCdSZWdFeHAucHJvdG90eXBlLmZsYWdzIGdldHRlciBjYWxsZWQgb24gbm9uLW9iamVjdCcpO1xuXHR9XG5cdHZhciByZXN1bHQgPSAnJztcblx0aWYgKHRoaXMuZ2xvYmFsKSB7XG5cdFx0cmVzdWx0ICs9ICdnJztcblx0fVxuXHRpZiAodGhpcy5pZ25vcmVDYXNlKSB7XG5cdFx0cmVzdWx0ICs9ICdpJztcblx0fVxuXHRpZiAodGhpcy5tdWx0aWxpbmUpIHtcblx0XHRyZXN1bHQgKz0gJ20nO1xuXHR9XG5cdGlmICh0aGlzLmRvdEFsbCkge1xuXHRcdHJlc3VsdCArPSAncyc7XG5cdH1cblx0aWYgKHRoaXMudW5pY29kZSkge1xuXHRcdHJlc3VsdCArPSAndSc7XG5cdH1cblx0aWYgKHRoaXMuc3RpY2t5KSB7XG5cdFx0cmVzdWx0ICs9ICd5Jztcblx0fVxuXHRyZXR1cm4gcmVzdWx0O1xufTtcbiJdLCJzb3VyY2VSb290IjoiIn0=\n//# sourceURL=webpack-internal:///3697\n")},2847:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar define = __webpack_require__(4289);\nvar callBind = __webpack_require__(5559);\n\nvar implementation = __webpack_require__(3697);\nvar getPolyfill = __webpack_require__(1721);\nvar shim = __webpack_require__(2753);\n\nvar flagsBound = callBind(implementation);\n\ndefine(flagsBound, {\n\tgetPolyfill: getPolyfill,\n\timplementation: implementation,\n\tshim: shim\n});\n\nmodule.exports = flagsBound;\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL3JlZ2V4cC5wcm90b3R5cGUuZmxhZ3MvaW5kZXguanM/ZTcxMCJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBYTs7QUFFYixhQUFhLG1CQUFPLENBQUMsSUFBbUI7QUFDeEMsZUFBZSxtQkFBTyxDQUFDLElBQVc7O0FBRWxDLHFCQUFxQixtQkFBTyxDQUFDLElBQWtCO0FBQy9DLGtCQUFrQixtQkFBTyxDQUFDLElBQVk7QUFDdEMsV0FBVyxtQkFBTyxDQUFDLElBQVE7O0FBRTNCOztBQUVBO0FBQ0E7QUFDQTtBQUNBO0FBQ0EsQ0FBQzs7QUFFRCIsImZpbGUiOiIyODQ3LmpzIiwic291cmNlc0NvbnRlbnQiOlsiJ3VzZSBzdHJpY3QnO1xuXG52YXIgZGVmaW5lID0gcmVxdWlyZSgnZGVmaW5lLXByb3BlcnRpZXMnKTtcbnZhciBjYWxsQmluZCA9IHJlcXVpcmUoJ2NhbGwtYmluZCcpO1xuXG52YXIgaW1wbGVtZW50YXRpb24gPSByZXF1aXJlKCcuL2ltcGxlbWVudGF0aW9uJyk7XG52YXIgZ2V0UG9seWZpbGwgPSByZXF1aXJlKCcuL3BvbHlmaWxsJyk7XG52YXIgc2hpbSA9IHJlcXVpcmUoJy4vc2hpbScpO1xuXG52YXIgZmxhZ3NCb3VuZCA9IGNhbGxCaW5kKGltcGxlbWVudGF0aW9uKTtcblxuZGVmaW5lKGZsYWdzQm91bmQsIHtcblx0Z2V0UG9seWZpbGw6IGdldFBvbHlmaWxsLFxuXHRpbXBsZW1lbnRhdGlvbjogaW1wbGVtZW50YXRpb24sXG5cdHNoaW06IHNoaW1cbn0pO1xuXG5tb2R1bGUuZXhwb3J0cyA9IGZsYWdzQm91bmQ7XG4iXSwic291cmNlUm9vdCI6IiJ9\n//# sourceURL=webpack-internal:///2847\n")},1721:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar implementation = __webpack_require__(3697);\n\nvar supportsDescriptors = __webpack_require__(4289).supportsDescriptors;\nvar $gOPD = Object.getOwnPropertyDescriptor;\nvar $TypeError = TypeError;\n\nmodule.exports = function getPolyfill() {\n\tif (!supportsDescriptors) {\n\t\tthrow new $TypeError('RegExp.prototype.flags requires a true ES5 environment that supports property descriptors');\n\t}\n\tif ((/a/mig).flags === 'gim') {\n\t\tvar descriptor = $gOPD(RegExp.prototype, 'flags');\n\t\tif (descriptor && typeof descriptor.get === 'function' && typeof (/a/).dotAll === 'boolean') {\n\t\t\treturn descriptor.get;\n\t\t}\n\t}\n\treturn implementation;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL3JlZ2V4cC5wcm90b3R5cGUuZmxhZ3MvcG9seWZpbGwuanM/NTdlYyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBYTs7QUFFYixxQkFBcUIsbUJBQU8sQ0FBQyxJQUFrQjs7QUFFL0MsMEJBQTBCLDZDQUFnRDtBQUMxRTtBQUNBOztBQUVBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQSIsImZpbGUiOiIxNzIxLmpzIiwic291cmNlc0NvbnRlbnQiOlsiJ3VzZSBzdHJpY3QnO1xuXG52YXIgaW1wbGVtZW50YXRpb24gPSByZXF1aXJlKCcuL2ltcGxlbWVudGF0aW9uJyk7XG5cbnZhciBzdXBwb3J0c0Rlc2NyaXB0b3JzID0gcmVxdWlyZSgnZGVmaW5lLXByb3BlcnRpZXMnKS5zdXBwb3J0c0Rlc2NyaXB0b3JzO1xudmFyICRnT1BEID0gT2JqZWN0LmdldE93blByb3BlcnR5RGVzY3JpcHRvcjtcbnZhciAkVHlwZUVycm9yID0gVHlwZUVycm9yO1xuXG5tb2R1bGUuZXhwb3J0cyA9IGZ1bmN0aW9uIGdldFBvbHlmaWxsKCkge1xuXHRpZiAoIXN1cHBvcnRzRGVzY3JpcHRvcnMpIHtcblx0XHR0aHJvdyBuZXcgJFR5cGVFcnJvcignUmVnRXhwLnByb3RvdHlwZS5mbGFncyByZXF1aXJlcyBhIHRydWUgRVM1IGVudmlyb25tZW50IHRoYXQgc3VwcG9ydHMgcHJvcGVydHkgZGVzY3JpcHRvcnMnKTtcblx0fVxuXHRpZiAoKC9hL21pZykuZmxhZ3MgPT09ICdnaW0nKSB7XG5cdFx0dmFyIGRlc2NyaXB0b3IgPSAkZ09QRChSZWdFeHAucHJvdG90eXBlLCAnZmxhZ3MnKTtcblx0XHRpZiAoZGVzY3JpcHRvciAmJiB0eXBlb2YgZGVzY3JpcHRvci5nZXQgPT09ICdmdW5jdGlvbicgJiYgdHlwZW9mICgvYS8pLmRvdEFsbCA9PT0gJ2Jvb2xlYW4nKSB7XG5cdFx0XHRyZXR1cm4gZGVzY3JpcHRvci5nZXQ7XG5cdFx0fVxuXHR9XG5cdHJldHVybiBpbXBsZW1lbnRhdGlvbjtcbn07XG4iXSwic291cmNlUm9vdCI6IiJ9\n//# sourceURL=webpack-internal:///1721\n")},2753:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar supportsDescriptors = __webpack_require__(4289).supportsDescriptors;\nvar getPolyfill = __webpack_require__(1721);\nvar gOPD = Object.getOwnPropertyDescriptor;\nvar defineProperty = Object.defineProperty;\nvar TypeErr = TypeError;\nvar getProto = Object.getPrototypeOf;\nvar regex = /a/;\n\nmodule.exports = function shimFlags() {\n\tif (!supportsDescriptors || !getProto) {\n\t\tthrow new TypeErr('RegExp.prototype.flags requires a true ES5 environment that supports property descriptors');\n\t}\n\tvar polyfill = getPolyfill();\n\tvar proto = getProto(regex);\n\tvar descriptor = gOPD(proto, 'flags');\n\tif (!descriptor || descriptor.get !== polyfill) {\n\t\tdefineProperty(proto, 'flags', {\n\t\t\tconfigurable: true,\n\t\t\tenumerable: false,\n\t\t\tget: polyfill\n\t\t});\n\t}\n\treturn polyfill;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL3JlZ2V4cC5wcm90b3R5cGUuZmxhZ3Mvc2hpbS5qcz8xYzdlIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFhOztBQUViLDBCQUEwQiw2Q0FBZ0Q7QUFDMUUsa0JBQWtCLG1CQUFPLENBQUMsSUFBWTtBQUN0QztBQUNBO0FBQ0E7QUFDQTtBQUNBOztBQUVBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBLEdBQUc7QUFDSDtBQUNBO0FBQ0EiLCJmaWxlIjoiMjc1My5qcyIsInNvdXJjZXNDb250ZW50IjpbIid1c2Ugc3RyaWN0JztcblxudmFyIHN1cHBvcnRzRGVzY3JpcHRvcnMgPSByZXF1aXJlKCdkZWZpbmUtcHJvcGVydGllcycpLnN1cHBvcnRzRGVzY3JpcHRvcnM7XG52YXIgZ2V0UG9seWZpbGwgPSByZXF1aXJlKCcuL3BvbHlmaWxsJyk7XG52YXIgZ09QRCA9IE9iamVjdC5nZXRPd25Qcm9wZXJ0eURlc2NyaXB0b3I7XG52YXIgZGVmaW5lUHJvcGVydHkgPSBPYmplY3QuZGVmaW5lUHJvcGVydHk7XG52YXIgVHlwZUVyciA9IFR5cGVFcnJvcjtcbnZhciBnZXRQcm90byA9IE9iamVjdC5nZXRQcm90b3R5cGVPZjtcbnZhciByZWdleCA9IC9hLztcblxubW9kdWxlLmV4cG9ydHMgPSBmdW5jdGlvbiBzaGltRmxhZ3MoKSB7XG5cdGlmICghc3VwcG9ydHNEZXNjcmlwdG9ycyB8fCAhZ2V0UHJvdG8pIHtcblx0XHR0aHJvdyBuZXcgVHlwZUVycignUmVnRXhwLnByb3RvdHlwZS5mbGFncyByZXF1aXJlcyBhIHRydWUgRVM1IGVudmlyb25tZW50IHRoYXQgc3VwcG9ydHMgcHJvcGVydHkgZGVzY3JpcHRvcnMnKTtcblx0fVxuXHR2YXIgcG9seWZpbGwgPSBnZXRQb2x5ZmlsbCgpO1xuXHR2YXIgcHJvdG8gPSBnZXRQcm90byhyZWdleCk7XG5cdHZhciBkZXNjcmlwdG9yID0gZ09QRChwcm90bywgJ2ZsYWdzJyk7XG5cdGlmICghZGVzY3JpcHRvciB8fCBkZXNjcmlwdG9yLmdldCAhPT0gcG9seWZpbGwpIHtcblx0XHRkZWZpbmVQcm9wZXJ0eShwcm90bywgJ2ZsYWdzJywge1xuXHRcdFx0Y29uZmlndXJhYmxlOiB0cnVlLFxuXHRcdFx0ZW51bWVyYWJsZTogZmFsc2UsXG5cdFx0XHRnZXQ6IHBvbHlmaWxsXG5cdFx0fSk7XG5cdH1cblx0cmV0dXJuIHBvbHlmaWxsO1xufTtcbiJdLCJzb3VyY2VSb290IjoiIn0=\n//# sourceURL=webpack-internal:///2753\n")},7478:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\nvar callBound = __webpack_require__(1924);\nvar inspect = __webpack_require__(631);\n\nvar $TypeError = GetIntrinsic('%TypeError%');\nvar $WeakMap = GetIntrinsic('%WeakMap%', true);\nvar $Map = GetIntrinsic('%Map%', true);\n\nvar $weakMapGet = callBound('WeakMap.prototype.get', true);\nvar $weakMapSet = callBound('WeakMap.prototype.set', true);\nvar $weakMapHas = callBound('WeakMap.prototype.has', true);\nvar $mapGet = callBound('Map.prototype.get', true);\nvar $mapSet = callBound('Map.prototype.set', true);\nvar $mapHas = callBound('Map.prototype.has', true);\n\n/*\n * This function traverses the list returning the node corresponding to the\n * given key.\n *\n * That node is also moved to the head of the list, so that if it's accessed\n * again we don't need to traverse the whole list. By doing so, all the recently\n * used nodes can be accessed relatively quickly.\n */\nvar listGetNode = function (list, key) { // eslint-disable-line consistent-return\n\tfor (var prev = list, curr; (curr = prev.next) !== null; prev = curr) {\n\t\tif (curr.key === key) {\n\t\t\tprev.next = curr.next;\n\t\t\tcurr.next = list.next;\n\t\t\tlist.next = curr; // eslint-disable-line no-param-reassign\n\t\t\treturn curr;\n\t\t}\n\t}\n};\n\nvar listGet = function (objects, key) {\n\tvar node = listGetNode(objects, key);\n\treturn node && node.value;\n};\nvar listSet = function (objects, key, value) {\n\tvar node = listGetNode(objects, key);\n\tif (node) {\n\t\tnode.value = value;\n\t} else {\n\t\t// Prepend the new node to the beginning of the list\n\t\tobjects.next = { // eslint-disable-line no-param-reassign\n\t\t\tkey: key,\n\t\t\tnext: objects.next,\n\t\t\tvalue: value\n\t\t};\n\t}\n};\nvar listHas = function (objects, key) {\n\treturn !!listGetNode(objects, key);\n};\n\nmodule.exports = function getSideChannel() {\n\tvar $wm;\n\tvar $m;\n\tvar $o;\n\tvar channel = {\n\t\tassert: function (key) {\n\t\t\tif (!channel.has(key)) {\n\t\t\t\tthrow new $TypeError('Side channel does not contain ' + inspect(key));\n\t\t\t}\n\t\t},\n\t\tget: function (key) { // eslint-disable-line consistent-return\n\t\t\tif ($WeakMap && key && (typeof key === 'object' || typeof key === 'function')) {\n\t\t\t\tif ($wm) {\n\t\t\t\t\treturn $weakMapGet($wm, key);\n\t\t\t\t}\n\t\t\t} else if ($Map) {\n\t\t\t\tif ($m) {\n\t\t\t\t\treturn $mapGet($m, key);\n\t\t\t\t}\n\t\t\t} else {\n\t\t\t\tif ($o) { // eslint-disable-line no-lonely-if\n\t\t\t\t\treturn listGet($o, key);\n\t\t\t\t}\n\t\t\t}\n\t\t},\n\t\thas: function (key) {\n\t\t\tif ($WeakMap && key && (typeof key === 'object' || typeof key === 'function')) {\n\t\t\t\tif ($wm) {\n\t\t\t\t\treturn $weakMapHas($wm, key);\n\t\t\t\t}\n\t\t\t} else if ($Map) {\n\t\t\t\tif ($m) {\n\t\t\t\t\treturn $mapHas($m, key);\n\t\t\t\t}\n\t\t\t} else {\n\t\t\t\tif ($o) { // eslint-disable-line no-lonely-if\n\t\t\t\t\treturn listHas($o, key);\n\t\t\t\t}\n\t\t\t}\n\t\t\treturn false;\n\t\t},\n\t\tset: function (key, value) {\n\t\t\tif ($WeakMap && key && (typeof key === 'object' || typeof key === 'function')) {\n\t\t\t\tif (!$wm) {\n\t\t\t\t\t$wm = new $WeakMap();\n\t\t\t\t}\n\t\t\t\t$weakMapSet($wm, key, value);\n\t\t\t} else if ($Map) {\n\t\t\t\tif (!$m) {\n\t\t\t\t\t$m = new $Map();\n\t\t\t\t}\n\t\t\t\t$mapSet($m, key, value);\n\t\t\t} else {\n\t\t\t\tif (!$o) {\n\t\t\t\t\t/*\n\t\t\t\t\t * Initialize the linked list as an empty node, so that we don't have\n\t\t\t\t\t * to special-case handling of the first node: we can always refer to\n\t\t\t\t\t * it as (previous node).next, instead of something like (list).head\n\t\t\t\t\t */\n\t\t\t\t\t$o = { key: {}, next: null };\n\t\t\t\t}\n\t\t\t\tlistSet($o, key, value);\n\t\t\t}\n\t\t}\n\t};\n\treturn channel;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///7478\n")},1066:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar define = __webpack_require__(4289);\nvar AdvanceStringIndex = __webpack_require__(2672);\nvar CreateIterResultObject = __webpack_require__(3674);\nvar Get = __webpack_require__(5588);\nvar GetIntrinsic = __webpack_require__(210);\nvar OrdinaryObjectCreate = __webpack_require__(4091);\nvar RegExpExec = __webpack_require__(356);\nvar Set = __webpack_require__(2946);\nvar ToLength = __webpack_require__(8502);\nvar ToString = __webpack_require__(7308);\nvar Type = __webpack_require__(1915);\nvar hasSymbols = __webpack_require__(1405)();\n\nvar SLOT = __webpack_require__(9496);\nvar undefined;\n\nvar RegExpStringIterator = function RegExpStringIterator(R, S, global, fullUnicode) {\n\tif (Type(S) !== 'String') {\n\t\tthrow new TypeError('S must be a string');\n\t}\n\tif (Type(global) !== 'Boolean') {\n\t\tthrow new TypeError('global must be a boolean');\n\t}\n\tif (Type(fullUnicode) !== 'Boolean') {\n\t\tthrow new TypeError('fullUnicode must be a boolean');\n\t}\n\tSLOT.set(this, '[[IteratingRegExp]]', R);\n\tSLOT.set(this, '[[IteratedString]]', S);\n\tSLOT.set(this, '[[Global]]', global);\n\tSLOT.set(this, '[[Unicode]]', fullUnicode);\n\tSLOT.set(this, '[[Done]]', false);\n};\n\nvar IteratorPrototype = GetIntrinsic('%IteratorPrototype%', true);\nif (IteratorPrototype) {\n\tRegExpStringIterator.prototype = OrdinaryObjectCreate(IteratorPrototype);\n}\n\ndefine(RegExpStringIterator.prototype, {\n\tnext: function next() {\n\t\tvar O = this;\n\t\tif (Type(O) !== 'Object') {\n\t\t\tthrow new TypeError('receiver must be an object');\n\t\t}\n\t\tif (\n\t\t\t!(O instanceof RegExpStringIterator)\n\t\t\t|| !SLOT.has(O, '[[IteratingRegExp]]')\n\t\t\t|| !SLOT.has(O, '[[IteratedString]]')\n\t\t\t|| !SLOT.has(O, '[[Global]]')\n\t\t\t|| !SLOT.has(O, '[[Unicode]]')\n\t\t\t|| !SLOT.has(O, '[[Done]]')\n\t\t) {\n\t\t\tthrow new TypeError('\"this\" value must be a RegExpStringIterator instance');\n\t\t}\n\t\tif (SLOT.get(O, '[[Done]]')) {\n\t\t\treturn CreateIterResultObject(undefined, true);\n\t\t}\n\t\tvar R = SLOT.get(O, '[[IteratingRegExp]]');\n\t\tvar S = SLOT.get(O, '[[IteratedString]]');\n\t\tvar global = SLOT.get(O, '[[Global]]');\n\t\tvar fullUnicode = SLOT.get(O, '[[Unicode]]');\n\t\tvar match = RegExpExec(R, S);\n\t\tif (match === null) {\n\t\t\tSLOT.set(O, '[[Done]]', true);\n\t\t\treturn CreateIterResultObject(undefined, true);\n\t\t}\n\t\tif (global) {\n\t\t\tvar matchStr = ToString(Get(match, '0'));\n\t\t\tif (matchStr === '') {\n\t\t\t\tvar thisIndex = ToLength(Get(R, 'lastIndex'));\n\t\t\t\tvar nextIndex = AdvanceStringIndex(S, thisIndex, fullUnicode);\n\t\t\t\tSet(R, 'lastIndex', nextIndex, true);\n\t\t\t}\n\t\t\treturn CreateIterResultObject(match, false);\n\t\t}\n\t\tSLOT.set(O, '[[Done]]', true);\n\t\treturn CreateIterResultObject(match, false);\n\t}\n});\nif (hasSymbols) {\n\tvar defineP = Object.defineProperty;\n\tif (Symbol.toStringTag) {\n\t\tif (defineP) {\n\t\t\tdefineP(RegExpStringIterator.prototype, Symbol.toStringTag, {\n\t\t\t\tconfigurable: true,\n\t\t\t\tenumerable: false,\n\t\t\t\tvalue: 'RegExp String Iterator',\n\t\t\t\twritable: false\n\t\t\t});\n\t\t} else {\n\t\t\tRegExpStringIterator.prototype[Symbol.toStringTag] = 'RegExp String Iterator';\n\t\t}\n\t}\n\n\tif (!IteratorPrototype && Symbol.iterator) {\n\t\tvar func = {};\n\t\tfunc[Symbol.iterator] = RegExpStringIterator.prototype[Symbol.iterator] || function SymbolIterator() {\n\t\t\treturn this;\n\t\t};\n\t\tvar predicate = {};\n\t\tpredicate[Symbol.iterator] = function () {\n\t\t\treturn RegExpStringIterator.prototype[Symbol.iterator] !== func[Symbol.iterator];\n\t\t};\n\t\tdefine(RegExpStringIterator.prototype, func, predicate);\n\t}\n}\n\nmodule.exports = RegExpStringIterator;\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///1066\n")},9505:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar Call = __webpack_require__(7800);\nvar Get = __webpack_require__(5588);\nvar GetMethod = __webpack_require__(3081);\nvar IsRegExp = __webpack_require__(1137);\nvar ToString = __webpack_require__(7308);\nvar RequireObjectCoercible = __webpack_require__(3733);\nvar callBound = __webpack_require__(1924);\nvar hasSymbols = __webpack_require__(1405)();\nvar flagsGetter = __webpack_require__(2847);\n\nvar $indexOf = callBound('String.prototype.indexOf');\n\nvar regexpMatchAllPolyfill = __webpack_require__(6966);\n\nvar getMatcher = function getMatcher(regexp) { // eslint-disable-line consistent-return\n\tvar matcherPolyfill = regexpMatchAllPolyfill();\n\tif (hasSymbols && typeof Symbol.matchAll === 'symbol') {\n\t\tvar matcher = GetMethod(regexp, Symbol.matchAll);\n\t\tif (matcher === RegExp.prototype[Symbol.matchAll] && matcher !== matcherPolyfill) {\n\t\t\treturn matcherPolyfill;\n\t\t}\n\t\treturn matcher;\n\t}\n\t// fallback for pre-Symbol.matchAll environments\n\tif (IsRegExp(regexp)) {\n\t\treturn matcherPolyfill;\n\t}\n};\n\nmodule.exports = function matchAll(regexp) {\n\tvar O = RequireObjectCoercible(this);\n\n\tif (typeof regexp !== 'undefined' && regexp !== null) {\n\t\tvar isRegExp = IsRegExp(regexp);\n\t\tif (isRegExp) {\n\t\t\t// workaround for older engines that lack RegExp.prototype.flags\n\t\t\tvar flags = 'flags' in regexp ? Get(regexp, 'flags') : flagsGetter(regexp);\n\t\t\tRequireObjectCoercible(flags);\n\t\t\tif ($indexOf(ToString(flags), 'g') < 0) {\n\t\t\t\tthrow new TypeError('matchAll requires a global regular expression');\n\t\t\t}\n\t\t}\n\n\t\tvar matcher = getMatcher(regexp);\n\t\tif (typeof matcher !== 'undefined') {\n\t\t\treturn Call(matcher, regexp, [O]);\n\t\t}\n\t}\n\n\tvar S = ToString(O);\n\t// var rx = RegExpCreate(regexp, 'g');\n\tvar rx = new RegExp(regexp, 'g');\n\treturn Call(getMatcher(rx), rx, [S]);\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///9505\n")},4956:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar callBind = __webpack_require__(5559);\nvar define = __webpack_require__(4289);\n\nvar implementation = __webpack_require__(9505);\nvar getPolyfill = __webpack_require__(3447);\nvar shim = __webpack_require__(2376);\n\nvar boundMatchAll = callBind(implementation);\n\ndefine(boundMatchAll, {\n\tgetPolyfill: getPolyfill,\n\timplementation: implementation,\n\tshim: shim\n});\n\nmodule.exports = boundMatchAll;\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL3N0cmluZy5wcm90b3R5cGUubWF0Y2hhbGwvaW5kZXguanM/YjFjYyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBYTs7QUFFYixlQUFlLG1CQUFPLENBQUMsSUFBVztBQUNsQyxhQUFhLG1CQUFPLENBQUMsSUFBbUI7O0FBRXhDLHFCQUFxQixtQkFBTyxDQUFDLElBQWtCO0FBQy9DLGtCQUFrQixtQkFBTyxDQUFDLElBQVk7QUFDdEMsV0FBVyxtQkFBTyxDQUFDLElBQVE7O0FBRTNCOztBQUVBO0FBQ0E7QUFDQTtBQUNBO0FBQ0EsQ0FBQzs7QUFFRCIsImZpbGUiOiI0OTU2LmpzIiwic291cmNlc0NvbnRlbnQiOlsiJ3VzZSBzdHJpY3QnO1xuXG52YXIgY2FsbEJpbmQgPSByZXF1aXJlKCdjYWxsLWJpbmQnKTtcbnZhciBkZWZpbmUgPSByZXF1aXJlKCdkZWZpbmUtcHJvcGVydGllcycpO1xuXG52YXIgaW1wbGVtZW50YXRpb24gPSByZXF1aXJlKCcuL2ltcGxlbWVudGF0aW9uJyk7XG52YXIgZ2V0UG9seWZpbGwgPSByZXF1aXJlKCcuL3BvbHlmaWxsJyk7XG52YXIgc2hpbSA9IHJlcXVpcmUoJy4vc2hpbScpO1xuXG52YXIgYm91bmRNYXRjaEFsbCA9IGNhbGxCaW5kKGltcGxlbWVudGF0aW9uKTtcblxuZGVmaW5lKGJvdW5kTWF0Y2hBbGwsIHtcblx0Z2V0UG9seWZpbGw6IGdldFBvbHlmaWxsLFxuXHRpbXBsZW1lbnRhdGlvbjogaW1wbGVtZW50YXRpb24sXG5cdHNoaW06IHNoaW1cbn0pO1xuXG5tb2R1bGUuZXhwb3J0cyA9IGJvdW5kTWF0Y2hBbGw7XG4iXSwic291cmNlUm9vdCI6IiJ9\n//# sourceURL=webpack-internal:///4956\n")},6966:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar hasSymbols = __webpack_require__(1405)();\nvar regexpMatchAll = __webpack_require__(7201);\n\nmodule.exports = function getRegExpMatchAllPolyfill() {\n\tif (!hasSymbols || typeof Symbol.matchAll !== 'symbol' || typeof RegExp.prototype[Symbol.matchAll] !== 'function') {\n\t\treturn regexpMatchAll;\n\t}\n\treturn RegExp.prototype[Symbol.matchAll];\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL3N0cmluZy5wcm90b3R5cGUubWF0Y2hhbGwvcG9seWZpbGwtcmVnZXhwLW1hdGNoYWxsLmpzPzZjMTgiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQWE7O0FBRWIsaUJBQWlCLG1CQUFPLENBQUMsSUFBYTtBQUN0QyxxQkFBcUIsbUJBQU8sQ0FBQyxJQUFtQjs7QUFFaEQ7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBIiwiZmlsZSI6IjY5NjYuanMiLCJzb3VyY2VzQ29udGVudCI6WyIndXNlIHN0cmljdCc7XG5cbnZhciBoYXNTeW1ib2xzID0gcmVxdWlyZSgnaGFzLXN5bWJvbHMnKSgpO1xudmFyIHJlZ2V4cE1hdGNoQWxsID0gcmVxdWlyZSgnLi9yZWdleHAtbWF0Y2hhbGwnKTtcblxubW9kdWxlLmV4cG9ydHMgPSBmdW5jdGlvbiBnZXRSZWdFeHBNYXRjaEFsbFBvbHlmaWxsKCkge1xuXHRpZiAoIWhhc1N5bWJvbHMgfHwgdHlwZW9mIFN5bWJvbC5tYXRjaEFsbCAhPT0gJ3N5bWJvbCcgfHwgdHlwZW9mIFJlZ0V4cC5wcm90b3R5cGVbU3ltYm9sLm1hdGNoQWxsXSAhPT0gJ2Z1bmN0aW9uJykge1xuXHRcdHJldHVybiByZWdleHBNYXRjaEFsbDtcblx0fVxuXHRyZXR1cm4gUmVnRXhwLnByb3RvdHlwZVtTeW1ib2wubWF0Y2hBbGxdO1xufTtcbiJdLCJzb3VyY2VSb290IjoiIn0=\n//# sourceURL=webpack-internal:///6966\n")},3447:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar implementation = __webpack_require__(9505);\n\nmodule.exports = function getPolyfill() {\n\tif (String.prototype.matchAll) {\n\t\ttry {\n\t\t\t''.matchAll(RegExp.prototype);\n\t\t} catch (e) {\n\t\t\treturn String.prototype.matchAll;\n\t\t}\n\t}\n\treturn implementation;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL3N0cmluZy5wcm90b3R5cGUubWF0Y2hhbGwvcG9seWZpbGwuanM/YjhhMSJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBYTs7QUFFYixxQkFBcUIsbUJBQU8sQ0FBQyxJQUFrQjs7QUFFL0M7QUFDQTtBQUNBO0FBQ0E7QUFDQSxHQUFHO0FBQ0g7QUFDQTtBQUNBO0FBQ0E7QUFDQSIsImZpbGUiOiIzNDQ3LmpzIiwic291cmNlc0NvbnRlbnQiOlsiJ3VzZSBzdHJpY3QnO1xuXG52YXIgaW1wbGVtZW50YXRpb24gPSByZXF1aXJlKCcuL2ltcGxlbWVudGF0aW9uJyk7XG5cbm1vZHVsZS5leHBvcnRzID0gZnVuY3Rpb24gZ2V0UG9seWZpbGwoKSB7XG5cdGlmIChTdHJpbmcucHJvdG90eXBlLm1hdGNoQWxsKSB7XG5cdFx0dHJ5IHtcblx0XHRcdCcnLm1hdGNoQWxsKFJlZ0V4cC5wcm90b3R5cGUpO1xuXHRcdH0gY2F0Y2ggKGUpIHtcblx0XHRcdHJldHVybiBTdHJpbmcucHJvdG90eXBlLm1hdGNoQWxsO1xuXHRcdH1cblx0fVxuXHRyZXR1cm4gaW1wbGVtZW50YXRpb247XG59O1xuIl0sInNvdXJjZVJvb3QiOiIifQ==\n//# sourceURL=webpack-internal:///3447\n")},7201:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\n// var Construct = require('es-abstract/2020/Construct');\nvar Get = __webpack_require__(5588);\nvar Set = __webpack_require__(2946);\nvar SpeciesConstructor = __webpack_require__(303);\nvar ToLength = __webpack_require__(8502);\nvar ToString = __webpack_require__(7308);\nvar Type = __webpack_require__(1915);\nvar flagsGetter = __webpack_require__(2847);\n\nvar RegExpStringIterator = __webpack_require__(1066);\nvar OrigRegExp = RegExp;\n\nvar CreateRegExpStringIterator = function CreateRegExpStringIterator(R, S, global, fullUnicode) {\n\tif (Type(S) !== 'String') {\n\t\tthrow new TypeError('\"S\" value must be a String');\n\t}\n\tif (Type(global) !== 'Boolean') {\n\t\tthrow new TypeError('\"global\" value must be a Boolean');\n\t}\n\tif (Type(fullUnicode) !== 'Boolean') {\n\t\tthrow new TypeError('\"fullUnicode\" value must be a Boolean');\n\t}\n\n\tvar iterator = new RegExpStringIterator(R, S, global, fullUnicode);\n\treturn iterator;\n};\n\nvar supportsConstructingWithFlags = 'flags' in RegExp.prototype;\n\nvar constructRegexWithFlags = function constructRegex(C, R) {\n\tvar matcher;\n\t// workaround for older engines that lack RegExp.prototype.flags\n\tvar flags = 'flags' in R ? Get(R, 'flags') : ToString(flagsGetter(R));\n\tif (supportsConstructingWithFlags && typeof flags === 'string') {\n\t\tmatcher = new C(R, flags);\n\t} else if (C === OrigRegExp) {\n\t\t// workaround for older engines that can not construct a RegExp with flags\n\t\tmatcher = new C(R.source, flags);\n\t} else {\n\t\tmatcher = new C(R, flags);\n\t}\n\treturn { flags: flags, matcher: matcher };\n};\n\nvar regexMatchAll = function SymbolMatchAll(string) {\n\tvar R = this;\n\tif (Type(R) !== 'Object') {\n\t\tthrow new TypeError('\"this\" value must be an Object');\n\t}\n\tvar S = ToString(string);\n\tvar C = SpeciesConstructor(R, OrigRegExp);\n\n\tvar tmp = constructRegexWithFlags(C, R);\n\t// var flags = ToString(Get(R, 'flags'));\n\tvar flags = tmp.flags;\n\t// var matcher = Construct(C, [R, flags]);\n\tvar matcher = tmp.matcher;\n\n\tvar lastIndex = ToLength(Get(R, 'lastIndex'));\n\tSet(matcher, 'lastIndex', lastIndex, true);\n\tvar global = flags.indexOf('g') > -1;\n\tvar fullUnicode = flags.indexOf('u') > -1;\n\treturn CreateRegExpStringIterator(matcher, S, global, fullUnicode);\n};\n\nvar defineP = Object.defineProperty;\nvar gOPD = Object.getOwnPropertyDescriptor;\n\nif (defineP && gOPD) {\n\tvar desc = gOPD(regexMatchAll, 'name');\n\tif (desc && desc.configurable) {\n\t\tdefineP(regexMatchAll, 'name', { value: '[Symbol.matchAll]' });\n\t}\n}\n\nmodule.exports = regexMatchAll;\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///7201\n")},2376:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar define = __webpack_require__(4289);\nvar hasSymbols = __webpack_require__(1405)();\nvar getPolyfill = __webpack_require__(3447);\nvar regexpMatchAllPolyfill = __webpack_require__(6966);\n\nvar defineP = Object.defineProperty;\nvar gOPD = Object.getOwnPropertyDescriptor;\n\nmodule.exports = function shimMatchAll() {\n\tvar polyfill = getPolyfill();\n\tdefine(\n\t\tString.prototype,\n\t\t{ matchAll: polyfill },\n\t\t{ matchAll: function () { return String.prototype.matchAll !== polyfill; } }\n\t);\n\tif (hasSymbols) {\n\t\t// eslint-disable-next-line no-restricted-properties\n\t\tvar symbol = Symbol.matchAll || (Symbol['for'] ? Symbol['for']('Symbol.matchAll') : Symbol('Symbol.matchAll'));\n\t\tdefine(\n\t\t\tSymbol,\n\t\t\t{ matchAll: symbol },\n\t\t\t{ matchAll: function () { return Symbol.matchAll !== symbol; } }\n\t\t);\n\n\t\tif (defineP && gOPD) {\n\t\t\tvar desc = gOPD(Symbol, symbol);\n\t\t\tif (!desc || desc.configurable) {\n\t\t\t\tdefineP(Symbol, symbol, {\n\t\t\t\t\tconfigurable: false,\n\t\t\t\t\tenumerable: false,\n\t\t\t\t\tvalue: symbol,\n\t\t\t\t\twritable: false\n\t\t\t\t});\n\t\t\t}\n\t\t}\n\n\t\tvar regexpMatchAll = regexpMatchAllPolyfill();\n\t\tvar func = {};\n\t\tfunc[symbol] = regexpMatchAll;\n\t\tvar predicate = {};\n\t\tpredicate[symbol] = function () {\n\t\t\treturn RegExp.prototype[symbol] !== regexpMatchAll;\n\t\t};\n\t\tdefine(RegExp.prototype, func, predicate);\n\t}\n\treturn polyfill;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///2376\n")},7640:(__unused_webpack_module,__unused_webpack___webpack_exports__,__webpack_require__)=>{"use strict";eval('\n// EXTERNAL MODULE: ./node_modules/approx-string-match/dist/index.js\nvar dist = __webpack_require__(3089);\n;// CONCATENATED MODULE: ./src/vendor/hypothesis/anchoring/match-quote.js\n\n\n/**\n * @typedef {import(\'approx-string-match\').Match} StringMatch\n */\n\n/**\n * @typedef Match\n * @prop {number} start - Start offset of match in text\n * @prop {number} end - End offset of match in text\n * @prop {number} score -\n *   Score for the match between 0 and 1.0, where 1.0 indicates a perfect match\n *   for the quote and context.\n */\n\n/**\n * Find the best approximate matches for `str` in `text` allowing up to `maxErrors` errors.\n *\n * @param {string} text\n * @param {string} str\n * @param {number} maxErrors\n * @return {StringMatch[]}\n */\nfunction search(text, str, maxErrors) {\n  // Do a fast search for exact matches. The `approx-string-match` library\n  // doesn\'t currently incorporate this optimization itself.\n  let matchPos = 0;\n  let exactMatches = [];\n  while (matchPos !== -1) {\n    matchPos = text.indexOf(str, matchPos);\n    if (matchPos !== -1) {\n      exactMatches.push({\n        start: matchPos,\n        end: matchPos + str.length,\n        errors: 0,\n      });\n      matchPos += 1;\n    }\n  }\n  if (exactMatches.length > 0) {\n    return exactMatches;\n  }\n\n  // If there are no exact matches, do a more expensive search for matches\n  // with errors.\n  return (0,dist/* default */.Z)(text, str, maxErrors);\n}\n\n/**\n * Compute a score between 0 and 1.0 for the similarity between `text` and `str`.\n *\n * @param {string} text\n * @param {string} str\n */\nfunction textMatchScore(text, str) {\n  /* istanbul ignore next - `scoreMatch` will never pass an empty string */\n  if (str.length === 0 || text.length === 0) {\n    return 0.0;\n  }\n  const matches = search(text, str, str.length);\n\n  // prettier-ignore\n  return 1 - (matches[0].errors / str.length);\n}\n\n/**\n * Find the best approximate match for `quote` in `text`.\n *\n * Returns `null` if no match exceeding the minimum quality threshold was found.\n *\n * @param {string} text - Document text to search\n * @param {string} quote - String to find within `text`\n * @param {Object} context -\n *   Context in which the quote originally appeared. This is used to choose the\n *   best match.\n *   @param {string} [context.prefix] - Expected text before the quote\n *   @param {string} [context.suffix] - Expected text after the quote\n *   @param {number} [context.hint] - Expected offset of match within text\n * @return {Match|null}\n */\nfunction matchQuote(text, quote, context = {}) {\n  if (quote.length === 0) {\n    return null;\n  }\n\n  // Choose the maximum number of errors to allow for the initial search.\n  // This choice involves a tradeoff between:\n  //\n  //  - Recall (proportion of "good" matches found)\n  //  - Precision (proportion of matches found which are "good")\n  //  - Cost of the initial search and of processing the candidate matches [1]\n  //\n  // [1] Specifically, the expected-time complexity of the initial search is\n  //     `O((maxErrors / 32) * text.length)`. See `approx-string-match` docs.\n  const maxErrors = Math.min(256, quote.length / 2);\n\n  // Find closest matches for `quote` in `text` based on edit distance.\n  const matches = search(text, quote, maxErrors);\n\n  if (matches.length === 0) {\n    return null;\n  }\n\n  /**\n   * Compute a score between 0 and 1.0 for a match candidate.\n   *\n   * @param {StringMatch} match\n   */\n  const scoreMatch = match => {\n    const quoteWeight = 50; // Similarity of matched text to quote.\n    const prefixWeight = 20; // Similarity of text before matched text to `context.prefix`.\n    const suffixWeight = 20; // Similarity of text after matched text to `context.suffix`.\n    const posWeight = 2; // Proximity to expected location. Used as a tie-breaker.\n\n    const quoteScore = 1 - match.errors / quote.length;\n\n    const prefixScore = context.prefix\n      ? textMatchScore(\n          text.slice(Math.max(0, match.start - context.prefix.length), match.start),\n          context.prefix\n        )\n      : 1.0;\n    const suffixScore = context.suffix\n      ? textMatchScore(\n          text.slice(match.end, match.end + context.suffix.length),\n          context.suffix\n        )\n      : 1.0;\n\n    let posScore = 1.0;\n    if (typeof context.hint === \'number\') {\n      const offset = Math.abs(match.start - context.hint);\n      posScore = 1.0 - offset / text.length;\n    }\n\n    const rawScore =\n      quoteWeight * quoteScore +\n      prefixWeight * prefixScore +\n      suffixWeight * suffixScore +\n      posWeight * posScore;\n    const maxScore = quoteWeight + prefixWeight + suffixWeight + posWeight;\n    const normalizedScore = rawScore / maxScore;\n\n    return normalizedScore;\n  };\n\n  // Rank matches based on similarity of actual and expected surrounding text\n  // and actual/expected offset in the document text.\n  const scoredMatches = matches.map(m => ({\n    start: m.start,\n    end: m.end,\n    score: scoreMatch(m),\n  }));\n\n  // Choose match with highest score.\n  scoredMatches.sort((a, b) => b.score - a.score);\n  return scoredMatches[0];\n}\n\n;// CONCATENATED MODULE: ./src/vendor/hypothesis/anchoring/text-range.js\n/**\n * Return the combined length of text nodes contained in `node`.\n *\n * @param {Node} node\n */\nfunction nodeTextLength(node) {\n  switch (node.nodeType) {\n    case Node.ELEMENT_NODE:\n    case Node.TEXT_NODE:\n      // nb. `textContent` excludes text in comments and processing instructions\n      // when called on a parent element, so we don\'t need to subtract that here.\n\n      return /** @type {string} */ (node.textContent).length;\n    default:\n      return 0;\n  }\n}\n\n/**\n * Return the total length of the text of all previous siblings of `node`.\n *\n * @param {Node} node\n */\nfunction previousSiblingsTextLength(node) {\n  let sibling = node.previousSibling;\n  let length = 0;\n  while (sibling) {\n    length += nodeTextLength(sibling);\n    sibling = sibling.previousSibling;\n  }\n  return length;\n}\n\n/**\n * Resolve one or more character offsets within an element to (text node, position)\n * pairs.\n *\n * @param {Element} element\n * @param {number[]} offsets - Offsets, which must be sorted in ascending order\n * @return {{ node: Text, offset: number }[]}\n */\nfunction resolveOffsets(element, ...offsets) {\n  let nextOffset = offsets.shift();\n  const nodeIter = /** @type {Document} */ (\n    element.ownerDocument\n  ).createNodeIterator(element, NodeFilter.SHOW_TEXT);\n  const results = [];\n\n  let currentNode = nodeIter.nextNode();\n  let textNode;\n  let length = 0;\n\n  // Find the text node containing the `nextOffset`th character from the start\n  // of `element`.\n  while (nextOffset !== undefined && currentNode) {\n    textNode = /** @type {Text} */ (currentNode);\n    if (length + textNode.data.length > nextOffset) {\n      results.push({ node: textNode, offset: nextOffset - length });\n      nextOffset = offsets.shift();\n    } else {\n      currentNode = nodeIter.nextNode();\n      length += textNode.data.length;\n    }\n  }\n\n  // Boundary case.\n  while (nextOffset !== undefined && textNode && length === nextOffset) {\n    results.push({ node: textNode, offset: textNode.data.length });\n    nextOffset = offsets.shift();\n  }\n\n  if (nextOffset !== undefined) {\n    throw new RangeError(\'Offset exceeds text length\');\n  }\n\n  return results;\n}\n\nlet RESOLVE_FORWARDS = 1;\nlet RESOLVE_BACKWARDS = 2;\n\n/**\n * Represents an offset within the text content of an element.\n *\n * This position can be resolved to a specific descendant node in the current\n * DOM subtree of the element using the `resolve` method.\n */\nclass text_range_TextPosition {\n  /**\n   * Construct a `TextPosition` that refers to the text position `offset` within\n   * the text content of `element`.\n   *\n   * @param {Element} element\n   * @param {number} offset\n   */\n  constructor(element, offset) {\n    if (offset < 0) {\n      throw new Error(\'Offset is invalid\');\n    }\n\n    /** Element that `offset` is relative to. */\n    this.element = element;\n\n    /** Character offset from the start of the element\'s `textContent`. */\n    this.offset = offset;\n  }\n\n  /**\n   * Return a copy of this position with offset relative to a given ancestor\n   * element.\n   *\n   * @param {Element} parent - Ancestor of `this.element`\n   * @return {TextPosition}\n   */\n  relativeTo(parent) {\n    if (!parent.contains(this.element)) {\n      throw new Error(\'Parent is not an ancestor of current element\');\n    }\n\n    let el = this.element;\n    let offset = this.offset;\n    while (el !== parent) {\n      offset += previousSiblingsTextLength(el);\n      el = /** @type {Element} */ (el.parentElement);\n    }\n\n    return new text_range_TextPosition(el, offset);\n  }\n\n  /**\n   * Resolve the position to a specific text node and offset within that node.\n   *\n   * Throws if `this.offset` exceeds the length of the element\'s text. In the\n   * case where the element has no text and `this.offset` is 0, the `direction`\n   * option determines what happens.\n   *\n   * Offsets at the boundary between two nodes are resolved to the start of the\n   * node that begins at the boundary.\n   *\n   * @param {Object} [options]\n   *   @param {RESOLVE_FORWARDS|RESOLVE_BACKWARDS} [options.direction] -\n   *     Specifies in which direction to search for the nearest text node if\n   *     `this.offset` is `0` and `this.element` has no text. If not specified\n   *     an error is thrown.\n   * @return {{ node: Text, offset: number }}\n   * @throws {RangeError}\n   */\n  resolve(options = {}) {\n    try {\n      return resolveOffsets(this.element, this.offset)[0];\n    } catch (err) {\n      if (this.offset === 0 && options.direction !== undefined) {\n        const tw = document.createTreeWalker(\n          this.element.getRootNode(),\n          NodeFilter.SHOW_TEXT\n        );\n        tw.currentNode = this.element;\n        const forwards = options.direction === RESOLVE_FORWARDS;\n        const text = /** @type {Text|null} */ (\n          forwards ? tw.nextNode() : tw.previousNode()\n        );\n        if (!text) {\n          throw err;\n        }\n        return { node: text, offset: forwards ? 0 : text.data.length };\n      } else {\n        throw err;\n      }\n    }\n  }\n\n  /**\n   * Construct a `TextPosition` that refers to the `offset`th character within\n   * `node`.\n   *\n   * @param {Node} node\n   * @param {number} offset\n   * @return {TextPosition}\n   */\n  static fromCharOffset(node, offset) {\n    switch (node.nodeType) {\n      case Node.TEXT_NODE:\n        return text_range_TextPosition.fromPoint(node, offset);\n      case Node.ELEMENT_NODE:\n        return new text_range_TextPosition(/** @type {Element} */ (node), offset);\n      default:\n        throw new Error(\'Node is not an element or text node\');\n    }\n  }\n\n  /**\n   * Construct a `TextPosition` representing the range start or end point (node, offset).\n   *\n   * @param {Node} node - Text or Element node\n   * @param {number} offset - Offset within the node.\n   * @return {TextPosition}\n   */\n  static fromPoint(node, offset) {\n    switch (node.nodeType) {\n      case Node.TEXT_NODE: {\n        if (offset < 0 || offset > /** @type {Text} */ (node).data.length) {\n          throw new Error(\'Text node offset is out of range\');\n        }\n\n        if (!node.parentElement) {\n          throw new Error(\'Text node has no parent\');\n        }\n\n        // Get the offset from the start of the parent element.\n        const textOffset = previousSiblingsTextLength(node) + offset;\n\n        return new text_range_TextPosition(node.parentElement, textOffset);\n      }\n      case Node.ELEMENT_NODE: {\n        if (offset < 0 || offset > node.childNodes.length) {\n          throw new Error(\'Child node offset is out of range\');\n        }\n\n        // Get the text length before the `offset`th child of element.\n        let textOffset = 0;\n        for (let i = 0; i < offset; i++) {\n          textOffset += nodeTextLength(node.childNodes[i]);\n        }\n\n        return new text_range_TextPosition(/** @type {Element} */ (node), textOffset);\n      }\n      default:\n        throw new Error(\'Point is not in an element or text node\');\n    }\n  }\n}\n\n/**\n * Represents a region of a document as a (start, end) pair of `TextPosition` points.\n *\n * Representing a range in this way allows for changes in the DOM content of the\n * range which don\'t affect its text content, without affecting the text content\n * of the range itself.\n */\nclass text_range_TextRange {\n  /**\n   * Construct an immutable `TextRange` from a `start` and `end` point.\n   *\n   * @param {TextPosition} start\n   * @param {TextPosition} end\n   */\n  constructor(start, end) {\n    this.start = start;\n    this.end = end;\n  }\n\n  /**\n   * Return a copy of this range with start and end positions relative to a\n   * given ancestor. See `TextPosition.relativeTo`.\n   *\n   * @param {Element} element\n   */\n  relativeTo(element) {\n    return new text_range_TextRange(\n      this.start.relativeTo(element),\n      this.end.relativeTo(element)\n    );\n  }\n\n  /**\n   * Resolve the `TextRange` to a DOM range.\n   *\n   * The resulting DOM Range will always start and end in a `Text` node.\n   * Hence `TextRange.fromRange(range).toRange()` can be used to "shrink" a\n   * range to the text it contains.\n   *\n   * May throw if the `start` or `end` positions cannot be resolved to a range.\n   *\n   * @return {Range}\n   */\n  toRange() {\n    let start;\n    let end;\n\n    if (\n      this.start.element === this.end.element &&\n      this.start.offset <= this.end.offset\n    ) {\n      // Fast path for start and end points in same element.\n      [start, end] = resolveOffsets(\n        this.start.element,\n        this.start.offset,\n        this.end.offset\n      );\n    } else {\n      start = this.start.resolve({ direction: RESOLVE_FORWARDS });\n      end = this.end.resolve({ direction: RESOLVE_BACKWARDS });\n    }\n\n    const range = new Range();\n    range.setStart(start.node, start.offset);\n    range.setEnd(end.node, end.offset);\n    return range;\n  }\n\n  /**\n   * Convert an existing DOM `Range` to a `TextRange`\n   *\n   * @param {Range} range\n   * @return {TextRange}\n   */\n  static fromRange(range) {\n    const start = text_range_TextPosition.fromPoint(\n      range.startContainer,\n      range.startOffset\n    );\n    const end = text_range_TextPosition.fromPoint(range.endContainer, range.endOffset);\n    return new text_range_TextRange(start, end);\n  }\n\n  /**\n   * Return a `TextRange` from the `start`th to `end`th characters in `root`.\n   *\n   * @param {Element} root\n   * @param {number} start\n   * @param {number} end\n   */\n  static fromOffsets(root, start, end) {\n    return new text_range_TextRange(\n      new text_range_TextPosition(root, start),\n      new text_range_TextPosition(root, end)\n    );\n  }\n}\n\n;// CONCATENATED MODULE: ./src/vendor/hypothesis/anchoring/types.js\n/**\n * This module exports a set of classes for converting between DOM `Range`\n * objects and different types of selectors. It is mostly a thin wrapper around a\n * set of anchoring libraries. It serves two main purposes:\n *\n *  1. Providing a consistent interface across different types of anchors.\n *  2. Insulating the rest of the code from API changes in the underlying anchoring\n *     libraries.\n */\n\n\n\n\n\n/**\n * @typedef {import(\'../../types/api\').RangeSelector} RangeSelector\n * @typedef {import(\'../../types/api\').TextPositionSelector} TextPositionSelector\n * @typedef {import(\'../../types/api\').TextQuoteSelector} TextQuoteSelector\n */\n\n/**\n * Converts between `RangeSelector` selectors and `Range` objects.\n */\nclass RangeAnchor {\n  /**\n   * @param {Node} root - A root element from which to anchor.\n   * @param {Range} range -  A range describing the anchor.\n   */\n  constructor(root, range) {\n    this.root = root;\n    this.range = range;\n  }\n\n  /**\n   * @param {Node} root -  A root element from which to anchor.\n   * @param {Range} range -  A range describing the anchor.\n   */\n  static fromRange(root, range) {\n    return new RangeAnchor(root, range);\n  }\n\n  /**\n   * Create an anchor from a serialized `RangeSelector` selector.\n   *\n   * @param {Element} root -  A root element from which to anchor.\n   * @param {RangeSelector} selector\n   */\n  static fromSelector(root, selector) {\n    const startContainer = nodeFromXPath(selector.startContainer, root);\n    if (!startContainer) {\n      throw new Error(\'Failed to resolve startContainer XPath\');\n    }\n\n    const endContainer = nodeFromXPath(selector.endContainer, root);\n    if (!endContainer) {\n      throw new Error(\'Failed to resolve endContainer XPath\');\n    }\n\n    const startPos = TextPosition.fromCharOffset(\n      startContainer,\n      selector.startOffset\n    );\n    const endPos = TextPosition.fromCharOffset(\n      endContainer,\n      selector.endOffset\n    );\n\n    const range = new TextRange(startPos, endPos).toRange();\n    return new RangeAnchor(root, range);\n  }\n\n  toRange() {\n    return this.range;\n  }\n\n  /**\n   * @return {RangeSelector}\n   */\n  toSelector() {\n    // "Shrink" the range so that it tightly wraps its text. This ensures more\n    // predictable output for a given text selection.\n    const normalizedRange = TextRange.fromRange(this.range).toRange();\n\n    const textRange = TextRange.fromRange(normalizedRange);\n    const startContainer = xpathFromNode(textRange.start.element, this.root);\n    const endContainer = xpathFromNode(textRange.end.element, this.root);\n\n    return {\n      type: \'RangeSelector\',\n      startContainer,\n      startOffset: textRange.start.offset,\n      endContainer,\n      endOffset: textRange.end.offset,\n    };\n  }\n}\n\n/**\n * Converts between `TextPositionSelector` selectors and `Range` objects.\n */\nclass TextPositionAnchor {\n  /**\n   * @param {Element} root\n   * @param {number} start\n   * @param {number} end\n   */\n  constructor(root, start, end) {\n    this.root = root;\n    this.start = start;\n    this.end = end;\n  }\n\n  /**\n   * @param {Element} root\n   * @param {Range} range\n   */\n  static fromRange(root, range) {\n    const textRange = text_range_TextRange.fromRange(range).relativeTo(root);\n    return new TextPositionAnchor(\n      root,\n      textRange.start.offset,\n      textRange.end.offset\n    );\n  }\n  /**\n   * @param {Element} root\n   * @param {TextPositionSelector} selector\n   */\n  static fromSelector(root, selector) {\n    return new TextPositionAnchor(root, selector.start, selector.end);\n  }\n\n  /**\n   * @return {TextPositionSelector}\n   */\n  toSelector() {\n    return {\n      type: \'TextPositionSelector\',\n      start: this.start,\n      end: this.end,\n    };\n  }\n\n  toRange() {\n    return text_range_TextRange.fromOffsets(this.root, this.start, this.end).toRange();\n  }\n}\n\n/**\n * @typedef QuoteMatchOptions\n * @prop {number} [hint] - Expected position of match in text. See `matchQuote`.\n */\n\n/**\n * Converts between `TextQuoteSelector` selectors and `Range` objects.\n */\nclass TextQuoteAnchor {\n  /**\n   * @param {Element} root - A root element from which to anchor.\n   * @param {string} exact\n   * @param {Object} context\n   *   @param {string} [context.prefix]\n   *   @param {string} [context.suffix]\n   */\n  constructor(root, exact, context = {}) {\n    this.root = root;\n    this.exact = exact;\n    this.context = context;\n  }\n\n  /**\n   * Create a `TextQuoteAnchor` from a range.\n   *\n   * Will throw if `range` does not contain any text nodes.\n   *\n   * @param {Element} root\n   * @param {Range} range\n   */\n  static fromRange(root, range) {\n    const text = /** @type {string} */ (root.textContent);\n    const textRange = text_range_TextRange.fromRange(range).relativeTo(root);\n\n    const start = textRange.start.offset;\n    const end = textRange.end.offset;\n\n    // Number of characters around the quote to capture as context. We currently\n    // always use a fixed amount, but it would be better if this code was aware\n    // of logical boundaries in the document (paragraph, article etc.) to avoid\n    // capturing text unrelated to the quote.\n    //\n    // In regular prose the ideal content would often be the surrounding sentence.\n    // This is a natural unit of meaning which enables displaying quotes in\n    // context even when the document is not available. We could use `Intl.Segmenter`\n    // for this when available.\n    const contextLen = 32;\n\n    return new TextQuoteAnchor(root, text.slice(start, end), {\n      prefix: text.slice(Math.max(0, start - contextLen), start),\n      suffix: text.slice(end, Math.min(text.length, end + contextLen)),\n    });\n  }\n\n  /**\n   * @param {Element} root\n   * @param {TextQuoteSelector} selector\n   */\n  static fromSelector(root, selector) {\n    const { prefix, suffix } = selector;\n    return new TextQuoteAnchor(root, selector.exact, { prefix, suffix });\n  }\n\n  /**\n   * @return {TextQuoteSelector}\n   */\n  toSelector() {\n    return {\n      type: \'TextQuoteSelector\',\n      exact: this.exact,\n      prefix: this.context.prefix,\n      suffix: this.context.suffix,\n    };\n  }\n\n  /**\n   * @param {QuoteMatchOptions} [options]\n   */\n  toRange(options = {}) {\n    return this.toPositionAnchor(options).toRange();\n  }\n\n  /**\n   * @param {QuoteMatchOptions} [options]\n   */\n  toPositionAnchor(options = {}) {\n    const text = /** @type {string} */ (this.root.textContent);\n    const match = matchQuote(text, this.exact, {\n      ...this.context,\n      hint: options.hint,\n    });\n    if (!match) {\n      throw new Error(\'Quote not found\');\n    }\n    return new TextPositionAnchor(this.root, match.start, match.end);\n  }\n}\n\n;// CONCATENATED MODULE: ./src/utils.js\n//\n//  Copyright 2021 Readium Foundation. All rights reserved.\n//  Use of this source code is governed by the BSD-style license\n//  available in the top-level LICENSE file of the project.\n//\n\n\n\n// Catch JS errors to log them in the app.\nwindow.addEventListener(\n  "error",\n  function (event) {\n    Android.logError(event.message, event.filename, event.lineno);\n  },\n  false\n);\n\n// Notify native code that the page has loaded.\nwindow.addEventListener(\n  "load",\n  function () {\n    window.addEventListener("orientationchange", function () {\n      onViewportWidthChanged();\n      snapCurrentOffset();\n    });\n    onViewportWidthChanged();\n  },\n  false\n);\n\nvar pageWidth = 1;\n\nfunction onViewportWidthChanged() {\n  // We can\'t rely on window.innerWidth for the pageWidth on Android, because if the\n  // device pixel ratio is not an integer, we get rounding issues offsetting the pages.\n  //\n  // See https://github.com/readium/readium-css/issues/97\n  // and https://github.com/readium/r2-navigator-kotlin/issues/146\n  var width = Android.getViewportWidth();\n  pageWidth = width / window.devicePixelRatio;\n  setProperty(\n    "--RS__viewportWidth",\n    "calc(" + width + "px / " + window.devicePixelRatio + ")"\n  );\n}\n\nfunction isScrollModeEnabled() {\n  return (\n    document.documentElement.style\n      .getPropertyValue("--USER__scroll")\n      .toString()\n      .trim() == "readium-scroll-on"\n  );\n}\n\nfunction isRTL() {\n  return document.body.dir.toLowerCase() == "rtl";\n}\n\n// Scroll to the given TagId in document and snap.\nfunction scrollToId(id) {\n  var element = document.getElementById(id);\n  if (!element) {\n    return false;\n  }\n\n  return scrollToRect(element.getBoundingClientRect());\n}\n\n// Position must be in the range [0 - 1], 0-100%.\nfunction scrollToPosition(position) {\n  //        Android.log("scrollToPosition " + position);\n  if (position < 0 || position > 1) {\n    throw "scrollToPosition() must be given a position from 0.0 to  1.0";\n  }\n\n  let offset;\n  if (isScrollModeEnabled()) {\n    offset = document.scrollingElement.scrollHeight * position;\n    document.scrollingElement.scrollTop = offset;\n    // window.scrollTo(0, offset);\n  } else {\n    var documentWidth = document.scrollingElement.scrollWidth;\n    var factor = isRTL() ? -1 : 1;\n    offset = documentWidth * position * factor;\n    document.scrollingElement.scrollLeft = snapOffset(offset);\n  }\n}\n\n// Scrolls to the first occurrence of the given text snippet.\n//\n// The expected text argument is a Locator Text object, as defined here:\n// https://readium.org/architecture/models/locators/\nfunction scrollToText(text) {\n  let range = rangeFromLocator({ text });\n  if (!range) {\n    return false;\n  }\n  scrollToRange(range);\n  return true;\n}\n\nfunction scrollToRange(range) {\n  scrollToRect(range.getBoundingClientRect());\n}\n\nfunction scrollToRect(rect) {\n  if (isScrollModeEnabled()) {\n    document.scrollingElement.scrollTop =\n      rect.top + window.scrollY - window.innerHeight / 2;\n  } else {\n    document.scrollingElement.scrollLeft = snapOffset(\n      rect.left + window.scrollX\n    );\n  }\n}\n\nfunction scrollToStart() {\n  //        Android.log("scrollToStart");\n  if (!isScrollModeEnabled()) {\n    document.scrollingElement.scrollLeft = 0;\n  } else {\n    document.scrollingElement.scrollTop = 0;\n    window.scrollTo(0, 0);\n  }\n}\n\nfunction scrollToEnd() {\n  //        Android.log("scrollToEnd");\n  if (!isScrollModeEnabled()) {\n    var factor = isRTL() ? -1 : 1;\n    document.scrollingElement.scrollLeft = snapOffset(\n      document.scrollingElement.scrollWidth * factor\n    );\n  } else {\n    document.scrollingElement.scrollTop = document.body.scrollHeight;\n    window.scrollTo(0, document.body.scrollHeight);\n  }\n}\n\n// Returns false if the page is already at the left-most scroll offset.\nfunction scrollLeft() {\n  var documentWidth = document.scrollingElement.scrollWidth;\n  var offset = window.scrollX - pageWidth;\n  var minOffset = isRTL() ? -(documentWidth - pageWidth) : 0;\n  return scrollToOffset(Math.max(offset, minOffset));\n}\n\n// Returns false if the page is already at the right-most scroll offset.\nfunction scrollRight() {\n  var documentWidth = document.scrollingElement.scrollWidth;\n  var offset = window.scrollX + pageWidth;\n  var maxOffset = isRTL() ? 0 : documentWidth - pageWidth;\n  return scrollToOffset(Math.min(offset, maxOffset));\n}\n\n// Scrolls to the given left offset.\n// Returns false if the page scroll position is already close enough to the given offset.\nfunction scrollToOffset(offset) {\n  //        Android.log("scrollToOffset " + offset);\n  if (isScrollModeEnabled()) {\n    throw "Called scrollToOffset() with scroll mode enabled. This can only be used in paginated mode.";\n  }\n\n  var currentOffset = window.scrollX;\n  document.scrollingElement.scrollLeft = snapOffset(offset);\n  // In some case the scrollX cannot reach the position respecting to innerWidth\n  var diff = Math.abs(currentOffset - offset) / pageWidth;\n  return diff > 0.01;\n}\n\n// Snap the offset to the screen width (page width).\nfunction snapOffset(offset) {\n  var value = offset + (isRTL() ? -1 : 1);\n  return value - (value % pageWidth);\n}\n\n// Snaps the current offset to the page width.\nfunction snapCurrentOffset() {\n  //        Android.log("snapCurrentOffset");\n  if (isScrollModeEnabled()) {\n    return;\n  }\n  var currentOffset = window.scrollX;\n  // Adds half a page to make sure we don\'t snap to the previous page.\n  var factor = isRTL() ? -1 : 1;\n  var delta = factor * (pageWidth / 2);\n  document.scrollingElement.scrollLeft = snapOffset(currentOffset + delta);\n}\n\nfunction rangeFromLocator(locator) {\n  let text = locator.text;\n  if (!text || !text.highlight) {\n    return null;\n  }\n  try {\n    let anchor = new TextQuoteAnchor(document.body, text.highlight, {\n      prefix: text.before,\n      suffix: text.after,\n    });\n    return anchor.toRange();\n  } catch (e) {\n    logError(e);\n    return null;\n  }\n}\n\n/// User Settings.\n\n// For setting user setting.\nfunction setProperty(key, value) {\n  var root = document.documentElement;\n\n  root.style.setProperty(key, value);\n}\n\n// For removing user setting.\nfunction removeProperty(key) {\n  var root = document.documentElement;\n\n  root.style.removeProperty(key);\n}\n\n/// Toolkit\n\nfunction log() {\n  var message = Array.prototype.slice.call(arguments).join(" ");\n  Android.log(message);\n}\n\nfunction logError(message) {\n  Android.logError(message, "", 0);\n}\n\n;// CONCATENATED MODULE: ./src/rect.js\n//\n//  Copyright 2021 Readium Foundation. All rights reserved.\n//  Use of this source code is governed by the BSD-style license\n//  available in the top-level LICENSE file of the project.\n//\n\n\n\nconst debug = false;\n\n/**\n * Converts a DOMRect into a JSON object understandable by the native side.\n */\nfunction toNativeRect(rect) {\n  const pixelRatio = window.devicePixelRatio;\n  const width = rect.width * pixelRatio;\n  const height = rect.height * pixelRatio;\n  const left = rect.left * pixelRatio;\n  const top = rect.top * pixelRatio;\n  const right = left + width;\n  const bottom = top + height;\n  return { width, height, left, top, right, bottom };\n}\n\nfunction getClientRectsNoOverlap(\n  range,\n  doNotMergeHorizontallyAlignedRects\n) {\n  let clientRects = range.getClientRects();\n\n  const tolerance = 1;\n  const originalRects = [];\n  for (const rangeClientRect of clientRects) {\n    originalRects.push({\n      bottom: rangeClientRect.bottom,\n      height: rangeClientRect.height,\n      left: rangeClientRect.left,\n      right: rangeClientRect.right,\n      top: rangeClientRect.top,\n      width: rangeClientRect.width,\n    });\n  }\n  const mergedRects = mergeTouchingRects(\n    originalRects,\n    tolerance,\n    doNotMergeHorizontallyAlignedRects\n  );\n  const noContainedRects = removeContainedRects(mergedRects, tolerance);\n  const newRects = replaceOverlapingRects(noContainedRects);\n  const minArea = 2 * 2;\n  for (let j = newRects.length - 1; j >= 0; j--) {\n    const rect = newRects[j];\n    const bigEnough = rect.width * rect.height > minArea;\n    if (!bigEnough) {\n      if (newRects.length > 1) {\n        rect_log("CLIENT RECT: remove small");\n        newRects.splice(j, 1);\n      } else {\n        rect_log("CLIENT RECT: remove small, but keep otherwise empty!");\n        break;\n      }\n    }\n  }\n  rect_log(`CLIENT RECT: reduced ${originalRects.length} --\x3e ${newRects.length}`);\n  return newRects;\n}\n\nfunction mergeTouchingRects(\n  rects,\n  tolerance,\n  doNotMergeHorizontallyAlignedRects\n) {\n  for (let i = 0; i < rects.length; i++) {\n    for (let j = i + 1; j < rects.length; j++) {\n      const rect1 = rects[i];\n      const rect2 = rects[j];\n      if (rect1 === rect2) {\n        rect_log("mergeTouchingRects rect1 === rect2 ??!");\n        continue;\n      }\n      const rectsLineUpVertically =\n        almostEqual(rect1.top, rect2.top, tolerance) &&\n        almostEqual(rect1.bottom, rect2.bottom, tolerance);\n      const rectsLineUpHorizontally =\n        almostEqual(rect1.left, rect2.left, tolerance) &&\n        almostEqual(rect1.right, rect2.right, tolerance);\n      const horizontalAllowed = !doNotMergeHorizontallyAlignedRects;\n      const aligned =\n        (rectsLineUpHorizontally && horizontalAllowed) ||\n        (rectsLineUpVertically && !rectsLineUpHorizontally);\n      const canMerge = aligned && rectsTouchOrOverlap(rect1, rect2, tolerance);\n      if (canMerge) {\n        rect_log(\n          `CLIENT RECT: merging two into one, VERTICAL: ${rectsLineUpVertically} HORIZONTAL: ${rectsLineUpHorizontally} (${doNotMergeHorizontallyAlignedRects})`\n        );\n        const newRects = rects.filter((rect) => {\n          return rect !== rect1 && rect !== rect2;\n        });\n        const replacementClientRect = getBoundingRect(rect1, rect2);\n        newRects.push(replacementClientRect);\n        return mergeTouchingRects(\n          newRects,\n          tolerance,\n          doNotMergeHorizontallyAlignedRects\n        );\n      }\n    }\n  }\n  return rects;\n}\n\nfunction getBoundingRect(rect1, rect2) {\n  const left = Math.min(rect1.left, rect2.left);\n  const right = Math.max(rect1.right, rect2.right);\n  const top = Math.min(rect1.top, rect2.top);\n  const bottom = Math.max(rect1.bottom, rect2.bottom);\n  return {\n    bottom,\n    height: bottom - top,\n    left,\n    right,\n    top,\n    width: right - left,\n  };\n}\n\nfunction removeContainedRects(rects, tolerance) {\n  const rectsToKeep = new Set(rects);\n  for (const rect of rects) {\n    const bigEnough = rect.width > 1 && rect.height > 1;\n    if (!bigEnough) {\n      rect_log("CLIENT RECT: remove tiny");\n      rectsToKeep.delete(rect);\n      continue;\n    }\n    for (const possiblyContainingRect of rects) {\n      if (rect === possiblyContainingRect) {\n        continue;\n      }\n      if (!rectsToKeep.has(possiblyContainingRect)) {\n        continue;\n      }\n      if (rectContains(possiblyContainingRect, rect, tolerance)) {\n        rect_log("CLIENT RECT: remove contained");\n        rectsToKeep.delete(rect);\n        break;\n      }\n    }\n  }\n  return Array.from(rectsToKeep);\n}\n\nfunction rectContains(rect1, rect2, tolerance) {\n  return (\n    rectContainsPoint(rect1, rect2.left, rect2.top, tolerance) &&\n    rectContainsPoint(rect1, rect2.right, rect2.top, tolerance) &&\n    rectContainsPoint(rect1, rect2.left, rect2.bottom, tolerance) &&\n    rectContainsPoint(rect1, rect2.right, rect2.bottom, tolerance)\n  );\n}\n\nfunction rectContainsPoint(rect, x, y, tolerance) {\n  return (\n    (rect.left < x || almostEqual(rect.left, x, tolerance)) &&\n    (rect.right > x || almostEqual(rect.right, x, tolerance)) &&\n    (rect.top < y || almostEqual(rect.top, y, tolerance)) &&\n    (rect.bottom > y || almostEqual(rect.bottom, y, tolerance))\n  );\n}\n\nfunction replaceOverlapingRects(rects) {\n  for (let i = 0; i < rects.length; i++) {\n    for (let j = i + 1; j < rects.length; j++) {\n      const rect1 = rects[i];\n      const rect2 = rects[j];\n      if (rect1 === rect2) {\n        rect_log("replaceOverlapingRects rect1 === rect2 ??!");\n        continue;\n      }\n      if (rectsTouchOrOverlap(rect1, rect2, -1)) {\n        let toAdd = [];\n        let toRemove;\n        const subtractRects1 = rectSubtract(rect1, rect2);\n        if (subtractRects1.length === 1) {\n          toAdd = subtractRects1;\n          toRemove = rect1;\n        } else {\n          const subtractRects2 = rectSubtract(rect2, rect1);\n          if (subtractRects1.length < subtractRects2.length) {\n            toAdd = subtractRects1;\n            toRemove = rect1;\n          } else {\n            toAdd = subtractRects2;\n            toRemove = rect2;\n          }\n        }\n        rect_log(`CLIENT RECT: overlap, cut one rect into ${toAdd.length}`);\n        const newRects = rects.filter((rect) => {\n          return rect !== toRemove;\n        });\n        Array.prototype.push.apply(newRects, toAdd);\n        return replaceOverlapingRects(newRects);\n      }\n    }\n  }\n  return rects;\n}\n\nfunction rectSubtract(rect1, rect2) {\n  const rectIntersected = rectIntersect(rect2, rect1);\n  if (rectIntersected.height === 0 || rectIntersected.width === 0) {\n    return [rect1];\n  }\n  const rects = [];\n  {\n    const rectA = {\n      bottom: rect1.bottom,\n      height: 0,\n      left: rect1.left,\n      right: rectIntersected.left,\n      top: rect1.top,\n      width: 0,\n    };\n    rectA.width = rectA.right - rectA.left;\n    rectA.height = rectA.bottom - rectA.top;\n    if (rectA.height !== 0 && rectA.width !== 0) {\n      rects.push(rectA);\n    }\n  }\n  {\n    const rectB = {\n      bottom: rectIntersected.top,\n      height: 0,\n      left: rectIntersected.left,\n      right: rectIntersected.right,\n      top: rect1.top,\n      width: 0,\n    };\n    rectB.width = rectB.right - rectB.left;\n    rectB.height = rectB.bottom - rectB.top;\n    if (rectB.height !== 0 && rectB.width !== 0) {\n      rects.push(rectB);\n    }\n  }\n  {\n    const rectC = {\n      bottom: rect1.bottom,\n      height: 0,\n      left: rectIntersected.left,\n      right: rectIntersected.right,\n      top: rectIntersected.bottom,\n      width: 0,\n    };\n    rectC.width = rectC.right - rectC.left;\n    rectC.height = rectC.bottom - rectC.top;\n    if (rectC.height !== 0 && rectC.width !== 0) {\n      rects.push(rectC);\n    }\n  }\n  {\n    const rectD = {\n      bottom: rect1.bottom,\n      height: 0,\n      left: rectIntersected.right,\n      right: rect1.right,\n      top: rect1.top,\n      width: 0,\n    };\n    rectD.width = rectD.right - rectD.left;\n    rectD.height = rectD.bottom - rectD.top;\n    if (rectD.height !== 0 && rectD.width !== 0) {\n      rects.push(rectD);\n    }\n  }\n  return rects;\n}\n\nfunction rectIntersect(rect1, rect2) {\n  const maxLeft = Math.max(rect1.left, rect2.left);\n  const minRight = Math.min(rect1.right, rect2.right);\n  const maxTop = Math.max(rect1.top, rect2.top);\n  const minBottom = Math.min(rect1.bottom, rect2.bottom);\n  return {\n    bottom: minBottom,\n    height: Math.max(0, minBottom - maxTop),\n    left: maxLeft,\n    right: minRight,\n    top: maxTop,\n    width: Math.max(0, minRight - maxLeft),\n  };\n}\n\nfunction rectsTouchOrOverlap(rect1, rect2, tolerance) {\n  return (\n    (rect1.left < rect2.right ||\n      (tolerance >= 0 && almostEqual(rect1.left, rect2.right, tolerance))) &&\n    (rect2.left < rect1.right ||\n      (tolerance >= 0 && almostEqual(rect2.left, rect1.right, tolerance))) &&\n    (rect1.top < rect2.bottom ||\n      (tolerance >= 0 && almostEqual(rect1.top, rect2.bottom, tolerance))) &&\n    (rect2.top < rect1.bottom ||\n      (tolerance >= 0 && almostEqual(rect2.top, rect1.bottom, tolerance)))\n  );\n}\n\nfunction almostEqual(a, b, tolerance) {\n  return Math.abs(a - b) <= tolerance;\n}\n\nfunction rect_log() {\n  if (debug) {\n    log.apply(null, arguments);\n  }\n}\n\n;// CONCATENATED MODULE: ./src/decorator.js\n//\n//  Copyright 2021 Readium Foundation. All rights reserved.\n//  Use of this source code is governed by the BSD-style license\n//  available in the top-level LICENSE file of the project.\n//\n\n\n\n\nlet styles = new Map();\nlet groups = new Map();\nvar lastGroupId = 0;\n\n/**\n * Registers a list of additional supported Decoration Templates.\n *\n * Each template object is indexed by the style ID.\n */\nfunction registerTemplates(newStyles) {\n  var stylesheet = "";\n\n  for (const [id, style] of Object.entries(newStyles)) {\n    styles.set(id, style);\n    if (style.stylesheet) {\n      stylesheet += style.stylesheet + "\\n";\n    }\n  }\n\n  if (stylesheet) {\n    let styleElement = document.createElement("style");\n    styleElement.innerHTML = stylesheet;\n    document.getElementsByTagName("head")[0].appendChild(styleElement);\n  }\n}\n\n/**\n * Returns an instance of DecorationGroup for the given group name.\n */\nfunction getDecorations(groupName) {\n  var group = groups.get(groupName);\n  if (!group) {\n    let id = "r2-decoration-" + lastGroupId++;\n    group = DecorationGroup(id, groupName);\n    groups.set(groupName, group);\n  }\n  return group;\n}\n\n/**\n * Handles click events on a Decoration.\n * Returns whether a decoration matched this event.\n */\nfunction handleDecorationClickEvent(event, clickEvent) {\n  if (groups.size === 0) {\n    return false;\n  }\n\n  function findTarget() {\n    for (const [group, groupContent] of groups) {\n      for (const item of groupContent.items.reverse()) {\n        if (!item.clickableElements) {\n          continue;\n        }\n        for (const element of item.clickableElements) {\n          let rect = element.getBoundingClientRect().toJSON();\n          if (rectContainsPoint(rect, event.clientX, event.clientY, 1)) {\n            return { group, item, element, rect };\n          }\n        }\n      }\n    }\n  }\n\n  let target = findTarget();\n  if (!target) {\n    return false;\n  }\n\n  return Android.onDecorationActivated(\n    JSON.stringify({\n      id: target.item.decoration.id,\n      group: target.group,\n      rect: toNativeRect(target.item.range.getBoundingClientRect()),\n      click: clickEvent,\n    })\n  );\n}\n\n/**\n * Creates a DecorationGroup object from a unique HTML ID and its name.\n */\nfunction DecorationGroup(groupId, groupName) {\n  var items = [];\n  var lastItemId = 0;\n  var container = null;\n\n  /**\n   * Adds a new decoration to the group.\n   */\n  function add(decoration) {\n    let id = groupId + "-" + lastItemId++;\n\n    let range = rangeFromLocator(decoration.locator);\n    if (!range) {\n      log("Can\'t locate DOM range for decoration", decoration);\n      return;\n    }\n\n    let item = { id, decoration, range };\n    items.push(item);\n    layout(item);\n  }\n\n  /**\n   * Removes the decoration with given ID from the group.\n   */\n  function remove(decorationId) {\n    let index = items.findIndex((i) => i.decoration.id === decorationId);\n    if (index === -1) {\n      return;\n    }\n\n    let item = items[index];\n    items.splice(index, 1);\n    item.clickableElements = null;\n    if (item.container) {\n      item.container.remove();\n      item.container = null;\n    }\n  }\n\n  /**\n   * Notifies that the given decoration was modified and needs to be updated.\n   */\n  function update(decoration) {\n    remove(decoration.id);\n    add(decoration);\n  }\n\n  /**\n   * Removes all decorations from this group.\n   */\n  function clear() {\n    clearContainer();\n    items.length = 0;\n  }\n\n  /**\n   * Recreates the decoration elements.\n   *\n   * To be called after reflowing the resource, for example.\n   */\n  function requestLayout() {\n    clearContainer();\n    items.forEach((item) => layout(item));\n  }\n\n  /**\n   * Layouts a single Decoration item.\n   */\n  function layout(item) {\n    let groupContainer = requireContainer();\n\n    let style = styles.get(item.decoration.style);\n    if (!style) {\n      logError(`Unknown decoration style: ${item.decoration.style}`);\n      return;\n    }\n\n    let itemContainer = document.createElement("div");\n    itemContainer.setAttribute("id", item.id);\n    itemContainer.setAttribute("data-style", item.decoration.style);\n    itemContainer.style.setProperty("pointer-events", "none");\n\n    let viewportWidth = window.innerWidth;\n    let columnCount = parseInt(\n      getComputedStyle(document.documentElement).getPropertyValue(\n        "column-count"\n      )\n    );\n    let pageWidth = viewportWidth / (columnCount || 1);\n    let scrollingElement = document.scrollingElement;\n    let xOffset = scrollingElement.scrollLeft;\n    let yOffset = scrollingElement.scrollTop;\n\n    function positionElement(element, rect, boundingRect) {\n      element.style.position = "absolute";\n\n      if (style.width === "wrap") {\n        element.style.width = `${rect.width}px`;\n        element.style.height = `${rect.height}px`;\n        element.style.left = `${rect.left + xOffset}px`;\n        element.style.top = `${rect.top + yOffset}px`;\n      } else if (style.width === "viewport") {\n        element.style.width = `${viewportWidth}px`;\n        element.style.height = `${rect.height}px`;\n        let left = Math.floor(rect.left / viewportWidth) * viewportWidth;\n        element.style.left = `${left + xOffset}px`;\n        element.style.top = `${rect.top + yOffset}px`;\n      } else if (style.width === "bounds") {\n        element.style.width = `${boundingRect.width}px`;\n        element.style.height = `${rect.height}px`;\n        element.style.left = `${boundingRect.left + xOffset}px`;\n        element.style.top = `${rect.top + yOffset}px`;\n      } else if (style.width === "page") {\n        element.style.width = `${pageWidth}px`;\n        element.style.height = `${rect.height}px`;\n        let left = Math.floor(rect.left / pageWidth) * pageWidth;\n        element.style.left = `${left + xOffset}px`;\n        element.style.top = `${rect.top + yOffset}px`;\n      }\n    }\n\n    let boundingRect = item.range.getBoundingClientRect();\n\n    let elementTemplate;\n    try {\n      let template = document.createElement("template");\n      template.innerHTML = item.decoration.element.trim();\n      elementTemplate = template.content.firstElementChild;\n    } catch (error) {\n      logError(\n        `Invalid decoration element "${item.decoration.element}": ${error.message}`\n      );\n      return;\n    }\n\n    if (style.layout === "boxes") {\n      let doNotMergeHorizontallyAlignedRects = true;\n      let clientRects = getClientRectsNoOverlap(\n        item.range,\n        doNotMergeHorizontallyAlignedRects\n      );\n\n      clientRects = clientRects.sort((r1, r2) => {\n        if (r1.top < r2.top) {\n          return -1;\n        } else if (r1.top > r2.top) {\n          return 1;\n        } else {\n          return 0;\n        }\n      });\n\n      for (let clientRect of clientRects) {\n        const line = elementTemplate.cloneNode(true);\n        line.style.setProperty("pointer-events", "none");\n        positionElement(line, clientRect, boundingRect);\n        itemContainer.append(line);\n      }\n    } else if (style.layout === "bounds") {\n      const bounds = elementTemplate.cloneNode(true);\n      bounds.style.setProperty("pointer-events", "none");\n      positionElement(bounds, boundingRect, boundingRect);\n\n      itemContainer.append(bounds);\n    }\n\n    groupContainer.append(itemContainer);\n    item.container = itemContainer;\n    item.clickableElements = Array.from(\n      itemContainer.querySelectorAll("[data-activable=\'1\']")\n    );\n    if (item.clickableElements.length === 0) {\n      item.clickableElements = Array.from(itemContainer.children);\n    }\n  }\n\n  /**\n   * Returns the group container element, after making sure it exists.\n   */\n  function requireContainer() {\n    if (!container) {\n      container = document.createElement("div");\n      container.setAttribute("id", groupId);\n      container.setAttribute("data-group", groupName);\n      container.style.setProperty("pointer-events", "none");\n      document.body.append(container);\n    }\n    return container;\n  }\n\n  /**\n   * Removes the group container.\n   */\n  function clearContainer() {\n    if (container) {\n      container.remove();\n      container = null;\n    }\n  }\n\n  return { add, remove, update, clear, items, requestLayout };\n}\n\nwindow.addEventListener(\n  "load",\n  function () {\n    // Will relayout all the decorations when the document body is resized.\n    const body = document.body;\n    var lastSize = { width: 0, height: 0 };\n    const observer = new ResizeObserver(() => {\n      if (\n        lastSize.width === body.clientWidth &&\n        lastSize.height === body.clientHeight\n      ) {\n        return;\n      }\n      lastSize = {\n        width: body.clientWidth,\n        height: body.clientHeight,\n      };\n\n      groups.forEach(function (group) {\n        group.requestLayout();\n      });\n    });\n    observer.observe(body);\n  },\n  false\n);\n\n;// CONCATENATED MODULE: ./src/gestures.js\n/*\n * Copyright 2021 Readium Foundation. All rights reserved.\n * Use of this source code is governed by the BSD-style license\n * available in the top-level LICENSE file of the project.\n */\n\n\n\nwindow.addEventListener("DOMContentLoaded", function () {\n  document.addEventListener("click", onClick, false);\n});\n\nfunction onClick(event) {\n  if (!window.getSelection().isCollapsed) {\n    // There\'s an on-going selection, the tap will dismiss it so we don\'t forward it.\n    return;\n  }\n\n  var pixelRatio = window.devicePixelRatio;\n  let clickEvent = {\n    defaultPrevented: event.defaultPrevented,\n    x: event.clientX * pixelRatio,\n    y: event.clientY * pixelRatio,\n    targetElement: event.target.outerHTML,\n    interactiveElement: nearestInteractiveElement(event.target),\n  };\n\n  if (handleDecorationClickEvent(event, clickEvent)) {\n    return;\n  }\n\n  // Send the tap data over the JS bridge even if it\'s been handled within the web view, so that\n  // it can be preserved and used by the toolkit if needed.\n  var shouldPreventDefault = Android.onTap(JSON.stringify(clickEvent));\n\n  if (shouldPreventDefault) {\n    event.stopPropagation();\n    event.preventDefault();\n  }\n}\n\n// See. https://github.com/JayPanoz/architecture/tree/touch-handling/misc/touch-handling\nfunction nearestInteractiveElement(element) {\n  var interactiveTags = [\n    "a",\n    "audio",\n    "button",\n    "canvas",\n    "details",\n    "input",\n    "label",\n    "option",\n    "select",\n    "submit",\n    "textarea",\n    "video",\n  ];\n  if (interactiveTags.indexOf(element.nodeName.toLowerCase()) != -1) {\n    return element.outerHTML;\n  }\n\n  // Checks whether the element is editable by the user.\n  if (\n    element.hasAttribute("contenteditable") &&\n    element.getAttribute("contenteditable").toLowerCase() != "false"\n  ) {\n    return element.outerHTML;\n  }\n\n  // Checks parents recursively because the touch might be for example on an <em> inside a <a>.\n  if (element.parentElement) {\n    return nearestInteractiveElement(element.parentElement);\n  }\n\n  return null;\n}\n\n;// CONCATENATED MODULE: ./src/highlight.js\n/* eslint-disable */\n//\n//  highlight.js\n//  r2-navigator-kotlin\n//\n//  Organized by Taehyun Kim on 6/27/19 from r2-navigator-js.\n//\n//  Copyright 2019 Readium Foundation. All rights reserved.\n//  Use of this source code is governed by a BSD-style license which is detailed\n//  in the LICENSE file present in the project repository where this source code is maintained.\n//\n\nconst ROOT_CLASS_REDUCE_MOTION = "r2-reduce-motion";\nconst ROOT_CLASS_NO_FOOTNOTES = "r2-no-popup-foonotes";\nconst POPUP_DIALOG_CLASS = "r2-popup-dialog";\nconst FOOTNOTES_CONTAINER_CLASS = "r2-footnote-container";\nconst FOOTNOTES_CLOSE_BUTTON_CLASS = "r2-footnote-close";\nconst FOOTNOTE_FORCE_SHOW = "r2-footnote-force-show";\nconst TTS_ID_PREVIOUS = "r2-tts-previous";\nconst TTS_ID_NEXT = "r2-tts-next";\nconst TTS_ID_SLIDER = "r2-tts-slider";\nconst TTS_ID_ACTIVE_WORD = "r2-tts-active-word";\nconst TTS_ID_CONTAINER = "r2-tts-txt";\nconst TTS_ID_INFO = "r2-tts-info";\nconst TTS_NAV_BUTTON_CLASS = "r2-tts-button";\nconst TTS_ID_SPEAKING_DOC_ELEMENT = "r2-tts-speaking-el";\nconst TTS_CLASS_INJECTED_SPAN = "r2-tts-speaking-txt";\nconst TTS_CLASS_INJECTED_SUBSPAN = "r2-tts-speaking-word";\nconst TTS_ID_INJECTED_PARENT = "r2-tts-speaking-txt-parent";\nconst ID_HIGHLIGHTS_CONTAINER = "R2_ID_HIGHLIGHTS_CONTAINER";\nconst ID_ANNOTATION_CONTAINER = "R2_ID_ANNOTATION_CONTAINER";\nconst CLASS_HIGHLIGHT_CONTAINER = "R2_CLASS_HIGHLIGHT_CONTAINER";\nconst CLASS_ANNOTATION_CONTAINER = "R2_CLASS_ANNOTATION_CONTAINER";\nconst CLASS_HIGHLIGHT_AREA = "R2_CLASS_HIGHLIGHT_AREA";\nconst CLASS_ANNOTATION_AREA = "R2_CLASS_ANNOTATION_AREA";\nconst CLASS_HIGHLIGHT_BOUNDING_AREA = "R2_CLASS_HIGHLIGHT_BOUNDING_AREA";\nconst CLASS_ANNOTATION_BOUNDING_AREA = "R2_CLASS_ANNOTATION_BOUNDING_AREA";\n// tslint:disable-next-line:max-line-length\nconst _blacklistIdClassForCFI = [\n  POPUP_DIALOG_CLASS,\n  TTS_CLASS_INJECTED_SPAN,\n  TTS_CLASS_INJECTED_SUBSPAN,\n  ID_HIGHLIGHTS_CONTAINER,\n  CLASS_HIGHLIGHT_CONTAINER,\n  CLASS_HIGHLIGHT_AREA,\n  CLASS_HIGHLIGHT_BOUNDING_AREA,\n  "resize-sensor",\n];\nconst CLASS_PAGINATED = "r2-css-paginated";\n\n//const IS_DEV = (process.env.NODE_ENV === "development" || process.env.NODE_ENV === "dev");\nconst IS_DEV = false;\nconst _highlights = [];\n\nlet _highlightsContainer;\nlet _annotationContainer;\nlet lastMouseDownX = -1;\nlet lastMouseDownY = -1;\nlet bodyEventListenersSet = false;\n\nconst USE_SVG = false;\nconst DEFAULT_BACKGROUND_COLOR_OPACITY = 0.3;\nconst ALT_BACKGROUND_COLOR_OPACITY = 0.45;\n\n//const DEBUG_VISUALS = false;\nconst DEBUG_VISUALS = false;\nconst DEFAULT_BACKGROUND_COLOR = {\n  blue: 100,\n  green: 50,\n  red: 230,\n};\n\nconst ANNOTATION_WIDTH = 15;\n\nfunction resetHighlightBoundingStyle(_win, highlightBounding) {\n  if (\n    highlightBounding.getAttribute("class") == CLASS_ANNOTATION_BOUNDING_AREA\n  ) {\n    return;\n  }\n  highlightBounding.style.outline = "none";\n  highlightBounding.style.setProperty(\n    "background-color",\n    "transparent",\n    "important"\n  );\n}\n\nfunction setHighlightAreaStyle(win, highlightAreas, highlight) {\n  const useSVG = !DEBUG_VISUALS && USE_SVG;\n  for (const highlightArea of highlightAreas) {\n    const isSVG = useSVG && highlightArea.namespaceURI === SVG_XML_NAMESPACE;\n    const opacity = ALT_BACKGROUND_COLOR_OPACITY;\n    if (isSVG) {\n      highlightArea.style.setProperty(\n        "fill",\n        `rgb(${highlight.color.red}, ${highlight.color.green}, ${highlight.color.blue})`,\n        "important"\n      );\n      highlightArea.style.setProperty(\n        "fill-opacity",\n        `${opacity}`,\n        "important"\n      );\n      highlightArea.style.setProperty(\n        "stroke",\n        `rgb(${highlight.color.red}, ${highlight.color.green}, ${highlight.color.blue})`,\n        "important"\n      );\n      highlightArea.style.setProperty(\n        "stroke-opacity",\n        `${opacity}`,\n        "important"\n      );\n    } else {\n      highlightArea.style.setProperty(\n        "background-color",\n        `rgba(${highlight.color.red}, ${highlight.color.green}, ${highlight.color.blue}, ${opacity})`,\n        "important"\n      );\n    }\n  }\n}\n\nfunction resetHighlightAreaStyle(win, highlightArea) {\n  const useSVG = !DEBUG_VISUALS && USE_SVG;\n  //const useSVG = USE_SVG;\n  const isSVG = useSVG && highlightArea.namespaceURI === SVG_XML_NAMESPACE;\n  const id = isSVG\n    ? highlightArea.parentNode &&\n      highlightArea.parentNode.parentNode &&\n      highlightArea.parentNode.parentNode.nodeType === Node.ELEMENT_NODE &&\n      highlightArea.parentNode.parentNode.getAttribute\n      ? highlightArea.parentNode.parentNode.getAttribute("id")\n      : undefined\n    : highlightArea.parentNode &&\n      highlightArea.parentNode.nodeType === Node.ELEMENT_NODE &&\n      highlightArea.parentNode.getAttribute\n    ? highlightArea.parentNode.getAttribute("id")\n    : undefined;\n  if (id) {\n    const highlight = _highlights.find((h) => {\n      return h.id === id;\n    });\n    if (highlight) {\n      const opacity = DEFAULT_BACKGROUND_COLOR_OPACITY;\n      if (isSVG) {\n        highlightArea.style.setProperty(\n          "fill",\n          `rgb(${highlight.color.red}, ${highlight.color.green}, ${highlight.color.blue})`,\n          "important"\n        );\n        highlightArea.style.setProperty(\n          "fill-opacity",\n          `${opacity}`,\n          "important"\n        );\n        highlightArea.style.setProperty(\n          "stroke",\n          `rgb(${highlight.color.red}, ${highlight.color.green}, ${highlight.color.blue})`,\n          "important"\n        );\n        highlightArea.style.setProperty(\n          "stroke-opacity",\n          `${opacity}`,\n          "important"\n        );\n      } else {\n        highlightArea.style.setProperty(\n          "background-color",\n          `rgba(${highlight.color.red}, ${highlight.color.green}, ${highlight.color.blue}, ${opacity})`,\n          "important"\n        );\n      }\n    }\n  }\n}\nfunction processTouchEvent(win, ev) {\n  const document = win.document;\n  const scrollElement = getScrollingElement(document);\n  const x = ev.changedTouches[0].clientX;\n  const y = ev.changedTouches[0].clientY;\n  if (!_highlightsContainer) {\n    return;\n  }\n  const paginated = isPaginated(document);\n  const bodyRect = document.body.getBoundingClientRect();\n  let xOffset;\n  let yOffset;\n  if (navigator.userAgent.match(/Android/i)) {\n    xOffset = paginated ? -scrollElement.scrollLeft : bodyRect.left;\n    yOffset = paginated ? -scrollElement.scrollTop : bodyRect.top;\n  } else if (navigator.userAgent.match(/iPhone|iPad|iPod/i)) {\n    xOffset = paginated ? 0 : -scrollElement.scrollLeft;\n    yOffset = paginated ? 0 : bodyRect.top;\n  }\n  let foundHighlight;\n  let foundElement;\n  let foundRect;\n  //    _highlights.sort(function(a, b) {\n  //        console.log(JSON.stringify(a.selectionInfo))\n  //        return a.selectionInfo.cleanText.length < b.selectionInfo.cleanText.length\n  //    })\n  for (let i = _highlights.length - 1; i >= 0; i--) {\n    const highlight = _highlights[i];\n    let highlightParent = document.getElementById(`${highlight.id}`);\n    if (!highlightParent) {\n      highlightParent = _highlightsContainer.querySelector(`#${highlight.id}`);\n    }\n    if (!highlightParent) {\n      continue;\n    }\n    let hit = false;\n    const highlightFragments = highlightParent.querySelectorAll(\n      `.${CLASS_HIGHLIGHT_AREA}`\n    );\n    for (const highlightFragment of highlightFragments) {\n      const withRect = highlightFragment;\n      const left = withRect.rect.left + xOffset;\n      const top = withRect.rect.top + yOffset;\n      foundRect = withRect.rect;\n      if (\n        x >= left &&\n        x < left + withRect.rect.width &&\n        y >= top &&\n        y < top + withRect.rect.height\n      ) {\n        hit = true;\n        break;\n      }\n    }\n    if (hit) {\n      foundHighlight = highlight;\n      foundElement = highlightParent;\n      break;\n    }\n  }\n  if (!foundHighlight || !foundElement) {\n    const highlightBoundings = _highlightsContainer.querySelectorAll(\n      `.${CLASS_HIGHLIGHT_BOUNDING_AREA}`\n    );\n    for (const highlightBounding of highlightBoundings) {\n      resetHighlightBoundingStyle(win, highlightBounding);\n    }\n    const allHighlightAreas = Array.from(\n      _highlightsContainer.querySelectorAll(`.${CLASS_HIGHLIGHT_AREA}`)\n    );\n    for (const highlightArea of allHighlightAreas) {\n      resetHighlightAreaStyle(win, highlightArea);\n    }\n    return;\n  }\n\n  if (foundElement.getAttribute("data-click")) {\n    if (ev.type === "mousemove") {\n      const foundElementHighlightAreas = Array.from(\n        foundElement.querySelectorAll(`.${CLASS_HIGHLIGHT_AREA}`)\n      );\n      const allHighlightAreas = _highlightsContainer.querySelectorAll(\n        `.${CLASS_HIGHLIGHT_AREA}`\n      );\n      for (const highlightArea of allHighlightAreas) {\n        if (foundElementHighlightAreas.indexOf(highlightArea) < 0) {\n          resetHighlightAreaStyle(win, highlightArea);\n        }\n      }\n      setHighlightAreaStyle(win, foundElementHighlightAreas, foundHighlight);\n      const foundElementHighlightBounding = foundElement.querySelector(\n        `.${CLASS_HIGHLIGHT_BOUNDING_AREA}`\n      );\n      const allHighlightBoundings = _highlightsContainer.querySelectorAll(\n        `.${CLASS_HIGHLIGHT_BOUNDING_AREA}`\n      );\n      for (const highlightBounding of allHighlightBoundings) {\n        if (\n          !foundElementHighlightBounding ||\n          highlightBounding !== foundElementHighlightBounding\n        ) {\n          resetHighlightBoundingStyle(win, highlightBounding);\n        }\n      }\n      if (foundElementHighlightBounding) {\n        if (DEBUG_VISUALS) {\n          setHighlightBoundingStyle(\n            win,\n            foundElementHighlightBounding,\n            foundHighlight\n          );\n        }\n      }\n    } else if (ev.type === "touchstart" || ev.type === "touchend") {\n      const size = {\n        screenWidth: window.outerWidth,\n        screenHeight: window.outerHeight,\n        left: foundRect.left,\n        width: foundRect.width,\n        top: foundRect.top,\n        height: foundRect.height,\n      };\n      const payload = {\n        highlight: foundHighlight.id,\n        size: size,\n      };\n\n      if (\n        typeof window !== "undefined" &&\n        typeof window.process === "object" &&\n        window.process.type === "renderer"\n      ) {\n        electron_1.ipcRenderer.sendToHost(R2_EVENT_HIGHLIGHT_CLICK, payload);\n      } else if (window.webkitURL) {\n        console.log(foundHighlight.id.includes("R2_ANNOTATION_"));\n        if (foundHighlight.id.search("R2_ANNOTATION_") >= 0) {\n          if (navigator.userAgent.match(/Android/i)) {\n            Android.highlightAnnotationMarkActivated(foundHighlight.id);\n          } else if (navigator.userAgent.match(/iPhone|iPad|iPod/i)) {\n            webkit.messageHandlers.highlightAnnotationMarkActivated.postMessage(\n              foundHighlight.id\n            );\n          }\n        } else if (foundHighlight.id.search("R2_HIGHLIGHT_") >= 0) {\n          if (navigator.userAgent.match(/Android/i)) {\n            Android.highlightActivated(foundHighlight.id);\n          } else if (navigator.userAgent.match(/iPhone|iPad|iPod/i)) {\n            webkit.messageHandlers.highlightActivated.postMessage(\n              foundHighlight.id\n            );\n          }\n        }\n      }\n\n      ev.stopPropagation();\n      ev.preventDefault();\n    }\n  }\n}\n\nfunction processMouseEvent(win, ev) {\n  const document = win.document;\n  const scrollElement = getScrollingElement(document);\n  const x = ev.clientX;\n  const y = ev.clientY;\n  if (!_highlightsContainer) {\n    return;\n  }\n\n  const paginated = isPaginated(document);\n  const bodyRect = document.body.getBoundingClientRect();\n  let xOffset;\n  let yOffset;\n  if (navigator.userAgent.match(/Android/i)) {\n    xOffset = paginated ? -scrollElement.scrollLeft : bodyRect.left;\n    yOffset = paginated ? -scrollElement.scrollTop : bodyRect.top;\n  } else if (navigator.userAgent.match(/iPhone|iPad|iPod/i)) {\n    xOffset = paginated ? 0 : -scrollElement.scrollLeft;\n    yOffset = paginated ? 0 : bodyRect.top;\n  }\n  let foundHighlight;\n  let foundElement;\n  let foundRect;\n  for (let i = _highlights.length - 1; i >= 0; i--) {\n    const highlight = _highlights[i];\n    let highlightParent = document.getElementById(`${highlight.id}`);\n    if (!highlightParent) {\n      highlightParent = _highlightsContainer.querySelector(`#${highlight.id}`);\n    }\n    if (!highlightParent) {\n      continue;\n    }\n    let hit = false;\n    const highlightFragments = highlightParent.querySelectorAll(\n      `.${CLASS_HIGHLIGHT_AREA}`\n    );\n    for (const highlightFragment of highlightFragments) {\n      const withRect = highlightFragment;\n      const left = withRect.rect.left + xOffset;\n      const top = withRect.rect.top + yOffset;\n      foundRect = withRect.rect;\n      if (\n        x >= left &&\n        x < left + withRect.rect.width &&\n        y >= top &&\n        y < top + withRect.rect.height\n      ) {\n        hit = true;\n        break;\n      }\n    }\n    if (hit) {\n      foundHighlight = highlight;\n      foundElement = highlightParent;\n      break;\n    }\n  }\n\n  if (!foundHighlight || !foundElement) {\n    const highlightBoundings = _highlightsContainer.querySelectorAll(\n      `.${CLASS_HIGHLIGHT_BOUNDING_AREA}`\n    );\n    for (const highlightBounding of highlightBoundings) {\n      resetHighlightBoundingStyle(win, highlightBounding);\n    }\n    const allHighlightAreas = Array.from(\n      _highlightsContainer.querySelectorAll(`.${CLASS_HIGHLIGHT_AREA}`)\n    );\n    for (const highlightArea of allHighlightAreas) {\n      resetHighlightAreaStyle(win, highlightArea);\n    }\n    return;\n  }\n\n  if (foundElement.getAttribute("data-click")) {\n    if (ev.type === "mousemove") {\n      const foundElementHighlightAreas = Array.from(\n        foundElement.querySelectorAll(`.${CLASS_HIGHLIGHT_AREA}`)\n      );\n      const allHighlightAreas = _highlightsContainer.querySelectorAll(\n        `.${CLASS_HIGHLIGHT_AREA}`\n      );\n      for (const highlightArea of allHighlightAreas) {\n        if (foundElementHighlightAreas.indexOf(highlightArea) < 0) {\n          resetHighlightAreaStyle(win, highlightArea);\n        }\n      }\n      setHighlightAreaStyle(win, foundElementHighlightAreas, foundHighlight);\n      const foundElementHighlightBounding = foundElement.querySelector(\n        `.${CLASS_HIGHLIGHT_BOUNDING_AREA}`\n      );\n      const allHighlightBoundings = _highlightsContainer.querySelectorAll(\n        `.${CLASS_HIGHLIGHT_BOUNDING_AREA}`\n      );\n      for (const highlightBounding of allHighlightBoundings) {\n        if (\n          !foundElementHighlightBounding ||\n          highlightBounding !== foundElementHighlightBounding\n        ) {\n          resetHighlightBoundingStyle(win, highlightBounding);\n        }\n      }\n      if (foundElementHighlightBounding) {\n        if (DEBUG_VISUALS) {\n          setHighlightBoundingStyle(\n            win,\n            foundElementHighlightBounding,\n            foundHighlight\n          );\n        }\n      }\n    } else if (ev.type === "mouseup" || ev.type === "touchend") {\n      const touchedPosition = {\n        screenWidth: window.outerWidth,\n        screenHeight: window.innerHeight,\n        left: foundRect.left,\n        width: foundRect.width,\n        top: foundRect.top,\n        height: foundRect.height,\n      };\n\n      const payload = {\n        highlight: foundHighlight,\n        position: touchedPosition,\n      };\n\n      if (\n        typeof window !== "undefined" &&\n        typeof window.process === "object" &&\n        window.process.type === "renderer"\n      ) {\n        electron_1.ipcRenderer.sendToHost(R2_EVENT_HIGHLIGHT_CLICK, payload);\n      } else if (window.webkitURL) {\n        if (foundHighlight.id.search("R2_ANNOTATION_") >= 0) {\n          if (navigator.userAgent.match(/Android/i)) {\n            Android.highlightAnnotationMarkActivated(foundHighlight.id);\n          } else if (navigator.userAgent.match(/iPhone|iPad|iPod/i)) {\n            webkit.messageHandlers.highlightAnnotationMarkActivated.postMessage(\n              foundHighlight.id\n            );\n          }\n        } else if (foundHighlight.id.search("R2_HIGHLIGHT_") >= 0) {\n          if (navigator.userAgent.match(/Android/i)) {\n            Android.highlightActivated(foundHighlight.id);\n          } else if (navigator.userAgent.match(/iPhone|iPad|iPod/i)) {\n            webkit.messageHandlers.highlightActivated.postMessage(\n              foundHighlight.id\n            );\n          }\n        }\n      }\n\n      ev.stopPropagation();\n    }\n  }\n}\n\nfunction highlight_rectsTouchOrOverlap(rect1, rect2, tolerance) {\n  return (\n    (rect1.left < rect2.right ||\n      (tolerance >= 0 && highlight_almostEqual(rect1.left, rect2.right, tolerance))) &&\n    (rect2.left < rect1.right ||\n      (tolerance >= 0 && highlight_almostEqual(rect2.left, rect1.right, tolerance))) &&\n    (rect1.top < rect2.bottom ||\n      (tolerance >= 0 && highlight_almostEqual(rect1.top, rect2.bottom, tolerance))) &&\n    (rect2.top < rect1.bottom ||\n      (tolerance >= 0 && highlight_almostEqual(rect2.top, rect1.bottom, tolerance)))\n  );\n}\n\nfunction highlight_replaceOverlapingRects(rects) {\n  for (let i = 0; i < rects.length; i++) {\n    for (let j = i + 1; j < rects.length; j++) {\n      const rect1 = rects[i];\n      const rect2 = rects[j];\n      if (rect1 === rect2) {\n        if (IS_DEV) {\n          console.log("replaceOverlapingRects rect1 === rect2 ??!");\n        }\n        continue;\n      }\n      if (highlight_rectsTouchOrOverlap(rect1, rect2, -1)) {\n        let toAdd = [];\n        let toRemove;\n        let toPreserve;\n        const subtractRects1 = highlight_rectSubtract(rect1, rect2);\n        if (subtractRects1.length === 1) {\n          toAdd = subtractRects1;\n          toRemove = rect1;\n          toPreserve = rect2;\n        } else {\n          const subtractRects2 = highlight_rectSubtract(rect2, rect1);\n          if (subtractRects1.length < subtractRects2.length) {\n            toAdd = subtractRects1;\n            toRemove = rect1;\n            toPreserve = rect2;\n          } else {\n            toAdd = subtractRects2;\n            toRemove = rect2;\n            toPreserve = rect1;\n          }\n        }\n        if (IS_DEV) {\n          const toCheck = [];\n          toCheck.push(toPreserve);\n          Array.prototype.push.apply(toCheck, toAdd);\n          checkOverlaps(toCheck);\n        }\n        if (IS_DEV) {\n          console.log(\n            `CLIENT RECT: overlap, cut one rect into ${toAdd.length}`\n          );\n        }\n        const newRects = rects.filter((rect) => {\n          return rect !== toRemove;\n        });\n        Array.prototype.push.apply(newRects, toAdd);\n        return highlight_replaceOverlapingRects(newRects);\n      }\n    }\n  }\n  return rects;\n}\n\nfunction checkOverlaps(rects) {\n  const stillOverlapingRects = [];\n  for (const rect1 of rects) {\n    for (const rect2 of rects) {\n      if (rect1 === rect2) {\n        continue;\n      }\n      const has1 = stillOverlapingRects.indexOf(rect1) >= 0;\n      const has2 = stillOverlapingRects.indexOf(rect2) >= 0;\n      if (!has1 || !has2) {\n        if (highlight_rectsTouchOrOverlap(rect1, rect2, -1)) {\n          if (!has1) {\n            stillOverlapingRects.push(rect1);\n          }\n          if (!has2) {\n            stillOverlapingRects.push(rect2);\n          }\n          console.log("CLIENT RECT: overlap ---");\n          console.log(\n            `#1 TOP:${rect1.top} BOTTOM:${rect1.bottom} LEFT:${rect1.left} RIGHT:${rect1.right} WIDTH:${rect1.width} HEIGHT:${rect1.height}`\n          );\n          console.log(\n            `#2 TOP:${rect2.top} BOTTOM:${rect2.bottom} LEFT:${rect2.left} RIGHT:${rect2.right} WIDTH:${rect2.width} HEIGHT:${rect2.height}`\n          );\n          const xOverlap = getRectOverlapX(rect1, rect2);\n          console.log(`xOverlap: ${xOverlap}`);\n          const yOverlap = getRectOverlapY(rect1, rect2);\n          console.log(`yOverlap: ${yOverlap}`);\n        }\n      }\n    }\n  }\n  if (stillOverlapingRects.length) {\n    console.log(`CLIENT RECT: overlaps ${stillOverlapingRects.length}`);\n  }\n}\n\nfunction highlight_removeContainedRects(rects, tolerance) {\n  const rectsToKeep = new Set(rects);\n  for (const rect of rects) {\n    const bigEnough = rect.width > 1 && rect.height > 1;\n    if (!bigEnough) {\n      if (IS_DEV) {\n        console.log("CLIENT RECT: remove tiny");\n      }\n      rectsToKeep.delete(rect);\n      continue;\n    }\n    for (const possiblyContainingRect of rects) {\n      if (rect === possiblyContainingRect) {\n        continue;\n      }\n      if (!rectsToKeep.has(possiblyContainingRect)) {\n        continue;\n      }\n      if (highlight_rectContains(possiblyContainingRect, rect, tolerance)) {\n        if (IS_DEV) {\n          console.log("CLIENT RECT: remove contained");\n        }\n        rectsToKeep.delete(rect);\n        break;\n      }\n    }\n  }\n  return Array.from(rectsToKeep);\n}\n\nfunction highlight_almostEqual(a, b, tolerance) {\n  return Math.abs(a - b) <= tolerance;\n}\n\nfunction highlight_rectIntersect(rect1, rect2) {\n  const maxLeft = Math.max(rect1.left, rect2.left);\n  const minRight = Math.min(rect1.right, rect2.right);\n  const maxTop = Math.max(rect1.top, rect2.top);\n  const minBottom = Math.min(rect1.bottom, rect2.bottom);\n  const rect = {\n    bottom: minBottom,\n    height: Math.max(0, minBottom - maxTop),\n    left: maxLeft,\n    right: minRight,\n    top: maxTop,\n    width: Math.max(0, minRight - maxLeft),\n  };\n  return rect;\n}\n\nfunction highlight_rectSubtract(rect1, rect2) {\n  const rectIntersected = highlight_rectIntersect(rect2, rect1);\n  if (rectIntersected.height === 0 || rectIntersected.width === 0) {\n    return [rect1];\n  }\n  const rects = [];\n  {\n    const rectA = {\n      bottom: rect1.bottom,\n      height: 0,\n      left: rect1.left,\n      right: rectIntersected.left,\n      top: rect1.top,\n      width: 0,\n    };\n    rectA.width = rectA.right - rectA.left;\n    rectA.height = rectA.bottom - rectA.top;\n    if (rectA.height !== 0 && rectA.width !== 0) {\n      rects.push(rectA);\n    }\n  }\n  {\n    const rectB = {\n      bottom: rectIntersected.top,\n      height: 0,\n      left: rectIntersected.left,\n      right: rectIntersected.right,\n      top: rect1.top,\n      width: 0,\n    };\n    rectB.width = rectB.right - rectB.left;\n    rectB.height = rectB.bottom - rectB.top;\n    if (rectB.height !== 0 && rectB.width !== 0) {\n      rects.push(rectB);\n    }\n  }\n  {\n    const rectC = {\n      bottom: rect1.bottom,\n      height: 0,\n      left: rectIntersected.left,\n      right: rectIntersected.right,\n      top: rectIntersected.bottom,\n      width: 0,\n    };\n    rectC.width = rectC.right - rectC.left;\n    rectC.height = rectC.bottom - rectC.top;\n    if (rectC.height !== 0 && rectC.width !== 0) {\n      rects.push(rectC);\n    }\n  }\n  {\n    const rectD = {\n      bottom: rect1.bottom,\n      height: 0,\n      left: rectIntersected.right,\n      right: rect1.right,\n      top: rect1.top,\n      width: 0,\n    };\n    rectD.width = rectD.right - rectD.left;\n    rectD.height = rectD.bottom - rectD.top;\n    if (rectD.height !== 0 && rectD.width !== 0) {\n      rects.push(rectD);\n    }\n  }\n  return rects;\n}\n\nfunction highlight_rectContainsPoint(rect, x, y, tolerance) {\n  return (\n    (rect.left < x || highlight_almostEqual(rect.left, x, tolerance)) &&\n    (rect.right > x || highlight_almostEqual(rect.right, x, tolerance)) &&\n    (rect.top < y || highlight_almostEqual(rect.top, y, tolerance)) &&\n    (rect.bottom > y || highlight_almostEqual(rect.bottom, y, tolerance))\n  );\n}\n\nfunction highlight_rectContains(rect1, rect2, tolerance) {\n  return (\n    highlight_rectContainsPoint(rect1, rect2.left, rect2.top, tolerance) &&\n    highlight_rectContainsPoint(rect1, rect2.right, rect2.top, tolerance) &&\n    highlight_rectContainsPoint(rect1, rect2.left, rect2.bottom, tolerance) &&\n    highlight_rectContainsPoint(rect1, rect2.right, rect2.bottom, tolerance)\n  );\n}\n\nfunction highlight_getBoundingRect(rect1, rect2) {\n  const left = Math.min(rect1.left, rect2.left);\n  const right = Math.max(rect1.right, rect2.right);\n  const top = Math.min(rect1.top, rect2.top);\n  const bottom = Math.max(rect1.bottom, rect2.bottom);\n  return {\n    bottom,\n    height: bottom - top,\n    left,\n    right,\n    top,\n    width: right - left,\n  };\n}\n\nfunction highlight_mergeTouchingRects(\n  rects,\n  tolerance,\n  doNotMergeHorizontallyAlignedRects\n) {\n  for (let i = 0; i < rects.length; i++) {\n    for (let j = i + 1; j < rects.length; j++) {\n      const rect1 = rects[i];\n      const rect2 = rects[j];\n      if (rect1 === rect2) {\n        if (IS_DEV) {\n          console.log("mergeTouchingRects rect1 === rect2 ??!");\n        }\n        continue;\n      }\n      const rectsLineUpVertically =\n        highlight_almostEqual(rect1.top, rect2.top, tolerance) &&\n        highlight_almostEqual(rect1.bottom, rect2.bottom, tolerance);\n      const rectsLineUpHorizontally =\n        highlight_almostEqual(rect1.left, rect2.left, tolerance) &&\n        highlight_almostEqual(rect1.right, rect2.right, tolerance);\n      const horizontalAllowed = !doNotMergeHorizontallyAlignedRects;\n      const aligned =\n        (rectsLineUpHorizontally && horizontalAllowed) ||\n        (rectsLineUpVertically && !rectsLineUpHorizontally);\n      const canMerge = aligned && highlight_rectsTouchOrOverlap(rect1, rect2, tolerance);\n      if (canMerge) {\n        if (IS_DEV) {\n          console.log(\n            `CLIENT RECT: merging two into one, VERTICAL: ${rectsLineUpVertically} HORIZONTAL: ${rectsLineUpHorizontally} (${doNotMergeHorizontallyAlignedRects})`\n          );\n        }\n        const newRects = rects.filter((rect) => {\n          return rect !== rect1 && rect !== rect2;\n        });\n        const replacementClientRect = highlight_getBoundingRect(rect1, rect2);\n        newRects.push(replacementClientRect);\n        return highlight_mergeTouchingRects(\n          newRects,\n          tolerance,\n          doNotMergeHorizontallyAlignedRects\n        );\n      }\n    }\n  }\n  return rects;\n}\n\nfunction highlight_getClientRectsNoOverlap(range, doNotMergeHorizontallyAlignedRects) {\n  const rangeClientRects = range.getClientRects();\n  return getClientRectsNoOverlap_(\n    rangeClientRects,\n    doNotMergeHorizontallyAlignedRects\n  );\n}\n\nfunction getClientRectsNoOverlap_(\n  clientRects,\n  doNotMergeHorizontallyAlignedRects\n) {\n  const tolerance = 1;\n  const originalRects = [];\n  for (const rangeClientRect of clientRects) {\n    originalRects.push({\n      bottom: rangeClientRect.bottom,\n      height: rangeClientRect.height,\n      left: rangeClientRect.left,\n      right: rangeClientRect.right,\n      top: rangeClientRect.top,\n      width: rangeClientRect.width,\n    });\n  }\n  const mergedRects = highlight_mergeTouchingRects(\n    originalRects,\n    tolerance,\n    doNotMergeHorizontallyAlignedRects\n  );\n  const noContainedRects = highlight_removeContainedRects(mergedRects, tolerance);\n  const newRects = highlight_replaceOverlapingRects(noContainedRects);\n  const minArea = 2 * 2;\n  for (let j = newRects.length - 1; j >= 0; j--) {\n    const rect = newRects[j];\n    const bigEnough = rect.width * rect.height > minArea;\n    if (!bigEnough) {\n      if (newRects.length > 1) {\n        if (IS_DEV) {\n          console.log("CLIENT RECT: remove small");\n        }\n        newRects.splice(j, 1);\n      } else {\n        if (IS_DEV) {\n          console.log("CLIENT RECT: remove small, but keep otherwise empty!");\n        }\n        break;\n      }\n    }\n  }\n  if (IS_DEV) {\n    checkOverlaps(newRects);\n  }\n  if (IS_DEV) {\n    console.log(\n      `CLIENT RECT: reduced ${originalRects.length} --\x3e ${newRects.length}`\n    );\n  }\n  return newRects;\n}\n\nfunction isPaginated(document) {\n  return (\n    document &&\n    document.documentElement &&\n    document.documentElement.classList.contains(CLASS_PAGINATED)\n  );\n}\n\nfunction getScrollingElement(document) {\n  if (document.scrollingElement) {\n    return document.scrollingElement;\n  }\n  return document.body;\n}\n\nfunction ensureContainer(win, annotationFlag) {\n  const document = win.document;\n\n  if (!_highlightsContainer) {\n    if (!bodyEventListenersSet) {\n      bodyEventListenersSet = true;\n      document.body.addEventListener(\n        "mousedown",\n        (ev) => {\n          lastMouseDownX = ev.clientX;\n          lastMouseDownY = ev.clientY;\n        },\n        false\n      );\n      document.body.addEventListener(\n        "mouseup",\n        (ev) => {\n          if (\n            Math.abs(lastMouseDownX - ev.clientX) < 3 &&\n            Math.abs(lastMouseDownY - ev.clientY) < 3\n          ) {\n            processMouseEvent(win, ev);\n          }\n        },\n        false\n      );\n      document.body.addEventListener(\n        "mousemove",\n        (ev) => {\n          processMouseEvent(win, ev);\n        },\n        false\n      );\n\n      document.body.addEventListener(\n        "touchend",\n        function touchEnd(e) {\n          processTouchEvent(win, e);\n        },\n        false\n      );\n    }\n    _highlightsContainer = document.createElement("div");\n    _highlightsContainer.setAttribute("id", ID_HIGHLIGHTS_CONTAINER);\n\n    _highlightsContainer.style.setProperty("pointer-events", "none");\n    document.body.append(_highlightsContainer);\n  }\n\n  return _highlightsContainer;\n}\n\nfunction hideAllhighlights() {\n  if (_highlightsContainer) {\n    _highlightsContainer.remove();\n    _highlightsContainer = null;\n  }\n}\n\nfunction destroyAllhighlights() {\n  hideAllhighlights();\n  _highlights.splice(0, _highlights.length);\n}\n\nfunction destroyHighlight(id) {\n  let i = -1;\n  let _document = window.document;\n  const highlight = _highlights.find((h, j) => {\n    i = j;\n    return h.id === id;\n  });\n  if (highlight && i >= 0 && i < _highlights.length) {\n    _highlights.splice(i, 1);\n  }\n  const highlightContainer = _document.getElementById(id);\n  if (highlightContainer) {\n    highlightContainer.remove();\n  }\n}\n\nfunction isCfiTextNode(node) {\n  return node.nodeType !== Node.ELEMENT_NODE;\n}\n\nfunction getChildTextNodeCfiIndex(element, child) {\n  let found = -1;\n  let textNodeIndex = -1;\n  let previousWasElement = false;\n  for (let i = 0; i < element.childNodes.length; i++) {\n    const childNode = element.childNodes[i];\n    const isText = isCfiTextNode(childNode);\n    if (isText || previousWasElement) {\n      textNodeIndex += 2;\n    }\n    if (isText) {\n      if (childNode === child) {\n        found = textNodeIndex;\n        break;\n      }\n    }\n    previousWasElement = childNode.nodeType === Node.ELEMENT_NODE;\n  }\n  return found;\n}\n\nfunction getCommonAncestorElement(node1, node2) {\n  if (node1.nodeType === Node.ELEMENT_NODE && node1 === node2) {\n    return node1;\n  }\n  if (node1.nodeType === Node.ELEMENT_NODE && node1.contains(node2)) {\n    return node1;\n  }\n  if (node2.nodeType === Node.ELEMENT_NODE && node2.contains(node1)) {\n    return node2;\n  }\n  const node1ElementAncestorChain = [];\n  let parent = node1.parentNode;\n  while (parent && parent.nodeType === Node.ELEMENT_NODE) {\n    node1ElementAncestorChain.push(parent);\n    parent = parent.parentNode;\n  }\n  const node2ElementAncestorChain = [];\n  parent = node2.parentNode;\n  while (parent && parent.nodeType === Node.ELEMENT_NODE) {\n    node2ElementAncestorChain.push(parent);\n    parent = parent.parentNode;\n  }\n  let commonAncestor = node1ElementAncestorChain.find(\n    (node1ElementAncestor) => {\n      return node2ElementAncestorChain.indexOf(node1ElementAncestor) >= 0;\n    }\n  );\n  if (!commonAncestor) {\n    commonAncestor = node2ElementAncestorChain.find((node2ElementAncestor) => {\n      return node1ElementAncestorChain.indexOf(node2ElementAncestor) >= 0;\n    });\n  }\n  return commonAncestor;\n}\n\nfunction fullQualifiedSelector(node) {\n  if (node.nodeType !== Node.ELEMENT_NODE) {\n    const lowerCaseName =\n      (node.localName && node.localName.toLowerCase()) ||\n      node.nodeName.toLowerCase();\n    return lowerCaseName;\n  }\n  //return cssPath(node, justSelector);\n  return cssPath(node, true);\n}\n\nfunction getCurrentSelectionInfo() {\n  const selection = window.getSelection();\n  if (!selection) {\n    return undefined;\n  }\n  if (selection.isCollapsed) {\n    console.log("^^^ SELECTION COLLAPSED.");\n    return undefined;\n  }\n  const rawText = selection.toString();\n  const cleanText = rawText.trim().replace(/\\n/g, " ").replace(/\\s\\s+/g, " ");\n  if (cleanText.length === 0) {\n    console.log("^^^ SELECTION TEXT EMPTY.");\n    return undefined;\n  }\n  if (!selection.anchorNode || !selection.focusNode) {\n    return undefined;\n  }\n  const range =\n    selection.rangeCount === 1\n      ? selection.getRangeAt(0)\n      : createOrderedRange(\n          selection.anchorNode,\n          selection.anchorOffset,\n          selection.focusNode,\n          selection.focusOffset\n        );\n  if (!range || range.collapsed) {\n    console.log("$$$$$$$$$$$$$$$$$ CANNOT GET NON-COLLAPSED SELECTION RANGE?!");\n    return undefined;\n  }\n  const rangeInfo = convertRange(range, fullQualifiedSelector, computeCFI);\n  if (!rangeInfo) {\n    console.log("^^^ SELECTION RANGE INFO FAIL?!");\n    return undefined;\n  }\n\n  if (IS_DEV && DEBUG_VISUALS) {\n    const restoredRange = convertRangeInfo(win.document, rangeInfo);\n    if (restoredRange) {\n      if (\n        restoredRange.startOffset === range.startOffset &&\n        restoredRange.endOffset === range.endOffset &&\n        restoredRange.startContainer === range.startContainer &&\n        restoredRange.endContainer === range.endContainer\n      ) {\n        console.log("SELECTION RANGE RESTORED OKAY (dev check).");\n      } else {\n        console.log("SELECTION RANGE RESTORE FAIL (dev check).");\n        dumpDebug(\n          "SELECTION",\n          selection.anchorNode,\n          selection.anchorOffset,\n          selection.focusNode,\n          selection.focusOffset,\n          getCssSelector\n        );\n        dumpDebug(\n          "ORDERED RANGE FROM SELECTION",\n          range.startContainer,\n          range.startOffset,\n          range.endContainer,\n          range.endOffset,\n          getCssSelector\n        );\n        dumpDebug(\n          "RESTORED RANGE",\n          restoredRange.startContainer,\n          restoredRange.startOffset,\n          restoredRange.endContainer,\n          restoredRange.endOffset,\n          getCssSelector\n        );\n      }\n    } else {\n      console.log("CANNOT RESTORE SELECTION RANGE ??!");\n    }\n  } else {\n  }\n\n  return {\n    locations: rangeInfo2Location(rangeInfo),\n    text: {\n      highlight: rawText,\n    },\n  };\n}\n\nfunction checkBlacklisted(el) {\n  let blacklistedId;\n  const id = el.getAttribute("id");\n  if (id && _blacklistIdClassForCFI.indexOf(id) >= 0) {\n    console.log("checkBlacklisted ID: " + id);\n    blacklistedId = id;\n  }\n  let blacklistedClass;\n  for (const item of _blacklistIdClassForCFI) {\n    if (el.classList.contains(item)) {\n      console.log("checkBlacklisted CLASS: " + item);\n      blacklistedClass = item;\n      break;\n    }\n  }\n  if (blacklistedId || blacklistedClass) {\n    return true;\n  }\n\n  return false;\n}\n\nfunction cssPath(node, optimized) {\n  if (node.nodeType !== Node.ELEMENT_NODE) {\n    return "";\n  }\n\n  const steps = [];\n  let contextNode = node;\n  while (contextNode) {\n    const step = _cssPathStep(contextNode, !!optimized, contextNode === node);\n    if (!step) {\n      break; // Error - bail out early.\n    }\n    steps.push(step.value);\n    if (step.optimized) {\n      break;\n    }\n    contextNode = contextNode.parentNode;\n  }\n  steps.reverse();\n  return steps.join(" > ");\n}\n// tslint:disable-next-line:max-line-length\n// https://chromium.googlesource.com/chromium/blink/+/master/Source/devtools/front_end/components/DOMPresentationUtils.js#316\nfunction _cssPathStep(node, optimized, isTargetNode) {\n  function prefixedElementClassNames(nd) {\n    const classAttribute = nd.getAttribute("class");\n    if (!classAttribute) {\n      return [];\n    }\n\n    return classAttribute\n      .split(/\\s+/g)\n      .filter(Boolean)\n      .map((nm) => {\n        // The prefix is required to store "__proto__" in a object-based map.\n        return "$" + nm;\n      });\n  }\n\n  function idSelector(idd) {\n    return "#" + escapeIdentifierIfNeeded(idd);\n  }\n\n  function escapeIdentifierIfNeeded(ident) {\n    if (isCSSIdentifier(ident)) {\n      return ident;\n    }\n\n    const shouldEscapeFirst = /^(?:[0-9]|-[0-9-]?)/.test(ident);\n    const lastIndex = ident.length - 1;\n    return ident.replace(/./g, function (c, ii) {\n      return (shouldEscapeFirst && ii === 0) || !isCSSIdentChar(c)\n        ? escapeAsciiChar(c, ii === lastIndex)\n        : c;\n    });\n  }\n\n  function escapeAsciiChar(c, isLast) {\n    return "\\\\" + toHexByte(c) + (isLast ? "" : " ");\n  }\n\n  function toHexByte(c) {\n    let hexByte = c.charCodeAt(0).toString(16);\n    if (hexByte.length === 1) {\n      hexByte = "0" + hexByte;\n    }\n    return hexByte;\n  }\n\n  function isCSSIdentChar(c) {\n    if (/[a-zA-Z0-9_-]/.test(c)) {\n      return true;\n    }\n    return c.charCodeAt(0) >= 0xa0;\n  }\n\n  function isCSSIdentifier(value) {\n    return /^-?[a-zA-Z_][a-zA-Z0-9_-]*$/.test(value);\n  }\n\n  if (node.nodeType !== Node.ELEMENT_NODE) {\n    return undefined;\n  }\n  const lowerCaseName =\n    (node.localName && node.localName.toLowerCase()) ||\n    node.nodeName.toLowerCase();\n\n  const element = node;\n\n  const id = element.getAttribute("id");\n\n  if (optimized) {\n    if (id) {\n      return {\n        optimized: true,\n        value: idSelector(id),\n      };\n    }\n    if (\n      lowerCaseName === "body" ||\n      lowerCaseName === "head" ||\n      lowerCaseName === "html"\n    ) {\n      return {\n        optimized: true,\n        value: lowerCaseName, // node.nodeNameInCorrectCase(),\n      };\n    }\n  }\n\n  const nodeName = lowerCaseName; // node.nodeNameInCorrectCase();\n  if (id) {\n    return {\n      optimized: true,\n      value: nodeName + idSelector(id),\n    };\n  }\n\n  const parent = node.parentNode;\n\n  if (!parent || parent.nodeType === Node.DOCUMENT_NODE) {\n    return {\n      optimized: true,\n      value: nodeName,\n    };\n  }\n\n  const prefixedOwnClassNamesArray_ = prefixedElementClassNames(element);\n\n  const prefixedOwnClassNamesArray = []; // .keySet()\n  prefixedOwnClassNamesArray_.forEach((arrItem) => {\n    if (prefixedOwnClassNamesArray.indexOf(arrItem) < 0) {\n      prefixedOwnClassNamesArray.push(arrItem);\n    }\n  });\n\n  let needsClassNames = false;\n  let needsNthChild = false;\n  let ownIndex = -1;\n  let elementIndex = -1;\n  const siblings = parent.children;\n\n  for (\n    let i = 0;\n    (ownIndex === -1 || !needsNthChild) && i < siblings.length;\n    ++i\n  ) {\n    const sibling = siblings[i];\n    if (sibling.nodeType !== Node.ELEMENT_NODE) {\n      continue;\n    }\n    elementIndex += 1;\n    if (sibling === node) {\n      ownIndex = elementIndex;\n      continue;\n    }\n    if (needsNthChild) {\n      continue;\n    }\n\n    // sibling.nodeNameInCorrectCase()\n    const siblingName =\n      (sibling.localName && sibling.localName.toLowerCase()) ||\n      sibling.nodeName.toLowerCase();\n    if (siblingName !== nodeName) {\n      continue;\n    }\n    needsClassNames = true;\n\n    const ownClassNames = [];\n    prefixedOwnClassNamesArray.forEach((arrItem) => {\n      ownClassNames.push(arrItem);\n    });\n    let ownClassNameCount = ownClassNames.length;\n\n    if (ownClassNameCount === 0) {\n      needsNthChild = true;\n      continue;\n    }\n    const siblingClassNamesArray_ = prefixedElementClassNames(sibling);\n    const siblingClassNamesArray = []; // .keySet()\n    siblingClassNamesArray_.forEach((arrItem) => {\n      if (siblingClassNamesArray.indexOf(arrItem) < 0) {\n        siblingClassNamesArray.push(arrItem);\n      }\n    });\n\n    for (const siblingClass of siblingClassNamesArray) {\n      const ind = ownClassNames.indexOf(siblingClass);\n      if (ind < 0) {\n        continue;\n      }\n\n      ownClassNames.splice(ind, 1); // delete ownClassNames[siblingClass];\n\n      if (!--ownClassNameCount) {\n        needsNthChild = true;\n        break;\n      }\n    }\n  }\n\n  let result = nodeName;\n  if (\n    isTargetNode &&\n    nodeName === "input" &&\n    element.getAttribute("type") &&\n    !element.getAttribute("id") &&\n    !element.getAttribute("class")\n  ) {\n    result += \'[type="\' + element.getAttribute("type") + \'"]\';\n  }\n  if (needsNthChild) {\n    result += ":nth-child(" + (ownIndex + 1) + ")";\n  } else if (needsClassNames) {\n    for (const prefixedName of prefixedOwnClassNamesArray) {\n      result += "." + escapeIdentifierIfNeeded(prefixedName.substr(1));\n    }\n  }\n\n  return {\n    optimized: false,\n    value: result,\n  };\n}\n\nfunction computeCFI(node) {\n  // TODO: handle character position inside text node\n  if (node.nodeType !== Node.ELEMENT_NODE) {\n    return undefined;\n  }\n\n  let cfi = "";\n\n  let currentElement = node;\n  while (\n    currentElement.parentNode &&\n    currentElement.parentNode.nodeType === Node.ELEMENT_NODE\n  ) {\n    const blacklisted = checkBlacklisted(currentElement);\n    if (!blacklisted) {\n      const currentElementParentChildren = currentElement.parentNode.children;\n      let currentElementIndex = -1;\n      for (let i = 0; i < currentElementParentChildren.length; i++) {\n        if (currentElement === currentElementParentChildren[i]) {\n          currentElementIndex = i;\n          break;\n        }\n      }\n      if (currentElementIndex >= 0) {\n        const cfiIndex = (currentElementIndex + 1) * 2;\n        cfi =\n          cfiIndex +\n          (currentElement.id ? "[" + currentElement.id + "]" : "") +\n          (cfi.length ? "/" + cfi : "");\n      }\n    }\n    currentElement = currentElement.parentNode;\n  }\n\n  return "/" + cfi;\n}\n\nfunction _createHighlight(locations, color, pointerInteraction, type) {\n  const rangeInfo = location2RangeInfo(locations);\n  const uniqueStr = `${rangeInfo.cfi}${rangeInfo.startContainerElementCssSelector}${rangeInfo.startContainerChildTextNodeIndex}${rangeInfo.startOffset}${rangeInfo.endContainerElementCssSelector}${rangeInfo.endContainerChildTextNodeIndex}${rangeInfo.endOffset}`;\n\n  const hash = __webpack_require__(3715);\n  const sha256Hex = hash.sha256().update(uniqueStr).digest("hex");\n\n  var id;\n  if (type == ID_HIGHLIGHTS_CONTAINER) {\n    id = "R2_HIGHLIGHT_" + sha256Hex;\n  } else {\n    id = "R2_ANNOTATION_" + sha256Hex;\n  }\n\n  destroyHighlight(id);\n\n  const highlight = {\n    color: color ? color : DEFAULT_BACKGROUND_COLOR,\n    id,\n    pointerInteraction,\n    rangeInfo,\n  };\n  _highlights.push(highlight);\n  createHighlightDom(\n    window,\n    highlight,\n    type == ID_ANNOTATION_CONTAINER ? true : false\n  );\n\n  return highlight;\n}\n\nfunction createHighlight(selectionInfo, color, pointerInteraction) {\n  return _createHighlight(\n    selectionInfo,\n    color,\n    pointerInteraction,\n    ID_HIGHLIGHTS_CONTAINER\n  );\n}\n\nfunction createAnnotation(id) {\n  let i = -1;\n\n  const highlight = _highlights.find((h, j) => {\n    i = j;\n    return h.id === id;\n  });\n  if (i == _highlights.length) return;\n\n  var locations = {\n    locations: rangeInfo2Location(highlight.rangeInfo),\n  };\n\n  return _createHighlight(\n    locations,\n    highlight.color,\n    true,\n    ID_ANNOTATION_CONTAINER\n  );\n}\n\nfunction createHighlightDom(win, highlight, annotationFlag) {\n  const document = win.document;\n\n  const scale =\n    1 /\n    (win.READIUM2 && win.READIUM2.isFixedLayout\n      ? win.READIUM2.fxlViewportScale\n      : 1);\n\n  const scrollElement = getScrollingElement(document);\n\n  const range = convertRangeInfo(document, highlight.rangeInfo);\n  if (!range) {\n    return undefined;\n  }\n\n  const paginated = isPaginated(document);\n  const highlightsContainer = ensureContainer(win, annotationFlag);\n  const highlightParent = document.createElement("div");\n\n  highlightParent.setAttribute("id", highlight.id);\n  highlightParent.setAttribute("class", CLASS_HIGHLIGHT_CONTAINER);\n\n  document.body.style.position = "relative";\n  highlightParent.style.setProperty("pointer-events", "none");\n  if (highlight.pointerInteraction) {\n    highlightParent.setAttribute("data-click", "1");\n  }\n\n  const bodyRect = document.body.getBoundingClientRect();\n  const useSVG = !DEBUG_VISUALS && USE_SVG;\n  //const useSVG = USE_SVG;\n  const drawUnderline = false;\n  const drawStrikeThrough = false;\n  const doNotMergeHorizontallyAlignedRects = drawUnderline || drawStrikeThrough;\n  //const clientRects = DEBUG_VISUALS ? range.getClientRects() :\n  const clientRects = highlight_getClientRectsNoOverlap(\n    range,\n    doNotMergeHorizontallyAlignedRects\n  );\n  let highlightAreaSVGDocFrag;\n  const roundedCorner = 3;\n  const underlineThickness = 2;\n  const strikeThroughLineThickness = 3;\n  const opacity = DEFAULT_BACKGROUND_COLOR_OPACITY;\n  let extra = "";\n  const rangeAnnotationBoundingClientRect =\n    frameForHighlightAnnotationMarkWithID(win, highlight.id);\n\n  let xOffset;\n  let yOffset;\n  let annotationOffset;\n\n  if (navigator.userAgent.match(/Android/i)) {\n    xOffset = paginated ? -scrollElement.scrollLeft : bodyRect.left;\n    yOffset = paginated ? -scrollElement.scrollTop : bodyRect.top;\n    annotationOffset =\n      parseInt(\n        (rangeAnnotationBoundingClientRect.right - xOffset) / window.innerWidth\n      ) + 1;\n  } else if (navigator.userAgent.match(/iPhone|iPad|iPod/i)) {\n    xOffset = paginated ? 0 : -scrollElement.scrollLeft;\n    yOffset = paginated ? 0 : bodyRect.top;\n    annotationOffset = parseInt(\n      rangeAnnotationBoundingClientRect.right / window.innerWidth + 1\n    );\n  }\n\n  for (const clientRect of clientRects) {\n    if (useSVG) {\n      const borderThickness = 0;\n      if (!highlightAreaSVGDocFrag) {\n        highlightAreaSVGDocFrag = document.createDocumentFragment();\n      }\n      const highlightAreaSVGRect = document.createElementNS(\n        SVG_XML_NAMESPACE,\n        "rect"\n      );\n\n      highlightAreaSVGRect.setAttribute("class", CLASS_HIGHLIGHT_AREA);\n      highlightAreaSVGRect.setAttribute(\n        "style",\n        `fill: rgb(${highlight.color.red}, ${highlight.color.green}, ${highlight.color.blue}) !important; fill-opacity: ${opacity} !important; stroke-width: 0;`\n      );\n      highlightAreaSVGRect.scale = scale;\n\n      /*\n             highlightAreaSVGRect.rect = {\n             height: clientRect.height,\n             left: clientRect.left - xOffset,\n             top: clientRect.top - yOffset,\n             width: clientRect.width,\n             };\n             */\n\n      if (annotationFlag) {\n        highlightAreaSVGRect.rect = {\n          height: ANNOTATION_WIDTH, //rangeAnnotationBoundingClientRect.height - rangeAnnotationBoundingClientRect.height/4,\n          left: window.innerWidth * annotationOffset - ANNOTATION_WIDTH,\n          top: rangeAnnotationBoundingClientRect.top - yOffset,\n          width: ANNOTATION_WIDTH,\n        };\n      } else {\n        highlightAreaSVGRect.rect = {\n          height: clientRect.height,\n          left: clientRect.left - xOffset,\n          top: clientRect.top - yOffset,\n          width: clientRect.width,\n        };\n      }\n\n      highlightAreaSVGRect.setAttribute("rx", `${roundedCorner * scale}`);\n      highlightAreaSVGRect.setAttribute("ry", `${roundedCorner * scale}`);\n      highlightAreaSVGRect.setAttribute(\n        "x",\n        `${(highlightAreaSVGRect.rect.left - borderThickness) * scale}`\n      );\n      highlightAreaSVGRect.setAttribute(\n        "y",\n        `${(highlightAreaSVGRect.rect.top - borderThickness) * scale}`\n      );\n      highlightAreaSVGRect.setAttribute(\n        "height",\n        `${(highlightAreaSVGRect.rect.height + borderThickness * 2) * scale}`\n      );\n      highlightAreaSVGRect.setAttribute(\n        "width",\n        `${(highlightAreaSVGRect.rect.width + borderThickness * 2) * scale}`\n      );\n      highlightAreaSVGDocFrag.appendChild(highlightAreaSVGRect);\n      if (drawUnderline) {\n        const highlightAreaSVGLine = document.createElementNS(\n          SVG_XML_NAMESPACE,\n          "line"\n        );\n        highlightAreaSVGRect.setAttribute("class", CLASS_HIGHLIGHT_AREA);\n        highlightAreaSVGLine.setAttribute(\n          "style",\n          `stroke-linecap: round; stroke-width: ${\n            underlineThickness * scale\n          }; stroke: rgb(${highlight.color.red}, ${highlight.color.green}, ${\n            highlight.color.blue\n          }) !important; stroke-opacity: ${opacity} !important`\n        );\n        highlightAreaSVGLine.scale = scale;\n        /*\n                 highlightAreaSVGLine.rect = {\n                 height: clientRect.height,\n                 left: clientRect.left - xOffset,\n                 top: clientRect.top - yOffset,\n                 width: clientRect.width,\n                 };\n                 */\n        if (annotationFlag) {\n          highlightAreaSVGLine.rect = {\n            height: ANNOTATION_WIDTH, //rangeAnnotationBoundingClientRect.height - rangeAnnotationBoundingClientRect.height/4,\n            left: window.innerWidth * annotationOffset - ANNOTATION_WIDTH,\n            top: rangeAnnotationBoundingClientRect.top - yOffset,\n            width: ANNOTATION_WIDTH,\n          };\n        } else {\n          highlightAreaSVGLine.rect = {\n            height: clientRect.height,\n            left: clientRect.left - xOffset,\n            top: clientRect.top - yOffset,\n            width: clientRect.width,\n          };\n        }\n\n        const lineOffset =\n          highlightAreaSVGLine.rect.width > roundedCorner ? roundedCorner : 0;\n        highlightAreaSVGLine.setAttribute(\n          "x1",\n          `${(highlightAreaSVGLine.rect.left + lineOffset) * scale}`\n        );\n        highlightAreaSVGLine.setAttribute(\n          "x2",\n          `${\n            (highlightAreaSVGLine.rect.left +\n              highlightAreaSVGLine.rect.width -\n              lineOffset) *\n            scale\n          }`\n        );\n        const y =\n          (highlightAreaSVGLine.rect.top +\n            highlightAreaSVGLine.rect.height -\n            underlineThickness / 2) *\n          scale;\n        highlightAreaSVGLine.setAttribute("y1", `${y}`);\n        highlightAreaSVGLine.setAttribute("y2", `${y}`);\n        highlightAreaSVGLine.setAttribute(\n          "height",\n          `${highlightAreaSVGLine.rect.height * scale}`\n        );\n        highlightAreaSVGLine.setAttribute(\n          "width",\n          `${highlightAreaSVGLine.rect.width * scale}`\n        );\n        highlightAreaSVGDocFrag.appendChild(highlightAreaSVGLine);\n      }\n      if (drawStrikeThrough) {\n        const highlightAreaSVGLine = document.createElementNS(\n          SVG_XML_NAMESPACE,\n          "line"\n        );\n\n        highlightAreaSVGRect.setAttribute("class", CLASS_HIGHLIGHT_AREA);\n        highlightAreaSVGLine.setAttribute(\n          "style",\n          `stroke-linecap: butt; stroke-width: ${\n            strikeThroughLineThickness * scale\n          }; stroke: rgb(${highlight.color.red}, ${highlight.color.green}, ${\n            highlight.color.blue\n          }) !important; stroke-opacity: ${opacity} !important`\n        );\n        highlightAreaSVGLine.scale = scale;\n\n        /*\n                 highlightAreaSVGLine.rect = {\n                 height: clientRect.height,\n                 left: clientRect.left - xOffset,\n                 top: clientRect.top - yOffset,\n                 width: clientRect.width,\n                 };\n                 */\n\n        if (annotationFlag) {\n          highlightAreaSVGLine.rect = {\n            height: ANNOTATION_WIDTH, //rangeAnnotationBoundingClientRect.height - rangeAnnotationBoundingClientRect.height/4,\n            left: window.innerWidth * annotationOffset - ANNOTATION_WIDTH,\n            top: rangeAnnotationBoundingClientRect.top - yOffset,\n            width: ANNOTATION_WIDTH,\n          };\n        } else {\n          highlightAreaSVGLine.rect = {\n            height: clientRect.height,\n            left: clientRect.left - xOffset,\n            top: clientRect.top - yOffset,\n            width: clientRect.width,\n          };\n        }\n\n        highlightAreaSVGLine.setAttribute(\n          "x1",\n          `${highlightAreaSVGLine.rect.left * scale}`\n        );\n        highlightAreaSVGLine.setAttribute(\n          "x2",\n          `${\n            (highlightAreaSVGLine.rect.left + highlightAreaSVGLine.rect.width) *\n            scale\n          }`\n        );\n        const lineOffset = highlightAreaSVGLine.rect.height / 2;\n        const y = (highlightAreaSVGLine.rect.top + lineOffset) * scale;\n        highlightAreaSVGLine.setAttribute("y1", `${y}`);\n        highlightAreaSVGLine.setAttribute("y2", `${y}`);\n        highlightAreaSVGLine.setAttribute(\n          "height",\n          `${highlightAreaSVGLine.rect.height * scale}`\n        );\n        highlightAreaSVGLine.setAttribute(\n          "width",\n          `${highlightAreaSVGLine.rect.width * scale}`\n        );\n        highlightAreaSVGDocFrag.appendChild(highlightAreaSVGLine);\n      }\n    } else {\n      const highlightArea = document.createElement("div");\n\n      highlightArea.setAttribute("class", CLASS_HIGHLIGHT_AREA);\n\n      if (DEBUG_VISUALS) {\n        const rgb = Math.round(0xffffff * Math.random());\n        const r = rgb >> 16;\n        const g = (rgb >> 8) & 255;\n        const b = rgb & 255;\n        extra = `outline-color: rgb(${r}, ${g}, ${b}); outline-style: solid; outline-width: 1px; outline-offset: -1px;`;\n      } else {\n        if (drawUnderline) {\n          extra += `border-bottom: ${underlineThickness * scale}px solid rgba(${\n            highlight.color.red\n          }, ${highlight.color.green}, ${\n            highlight.color.blue\n          }, ${opacity}) !important`;\n        }\n      }\n      highlightArea.setAttribute(\n        "style",\n        `border-radius: ${roundedCorner}px !important; background-color: rgba(${highlight.color.red}, ${highlight.color.green}, ${highlight.color.blue}, ${opacity}) !important; ${extra}`\n      );\n      highlightArea.style.setProperty("pointer-events", "none");\n      highlightArea.style.position = paginated ? "fixed" : "absolute";\n      highlightArea.scale = scale;\n      /*\n             highlightArea.rect = {\n             height: clientRect.height,\n             left: clientRect.left - xOffset,\n             top: clientRect.top - yOffset,\n             width: clientRect.width,\n             };\n             */\n      if (annotationFlag) {\n        highlightArea.rect = {\n          height: ANNOTATION_WIDTH, //rangeAnnotationBoundingClientRect.height - rangeAnnotationBoundingClientRect.height/4,\n          left: window.innerWidth * annotationOffset - ANNOTATION_WIDTH,\n          top: rangeAnnotationBoundingClientRect.top - yOffset,\n          width: ANNOTATION_WIDTH,\n        };\n      } else {\n        highlightArea.rect = {\n          height: clientRect.height,\n          left: clientRect.left - xOffset,\n          top: clientRect.top - yOffset,\n          width: clientRect.width,\n        };\n      }\n\n      highlightArea.style.width = `${highlightArea.rect.width * scale}px`;\n      highlightArea.style.height = `${highlightArea.rect.height * scale}px`;\n      highlightArea.style.left = `${highlightArea.rect.left * scale}px`;\n      highlightArea.style.top = `${highlightArea.rect.top * scale}px`;\n      highlightParent.append(highlightArea);\n      if (!DEBUG_VISUALS && drawStrikeThrough) {\n        //if (drawStrikeThrough) {\n        const highlightAreaLine = document.createElement("div");\n        highlightAreaLine.setAttribute("class", CLASS_HIGHLIGHT_AREA);\n\n        highlightAreaLine.setAttribute(\n          "style",\n          `background-color: rgba(${highlight.color.red}, ${highlight.color.green}, ${highlight.color.blue}, ${opacity}) !important;`\n        );\n        highlightAreaLine.style.setProperty("pointer-events", "none");\n        highlightAreaLine.style.position = paginated ? "fixed" : "absolute";\n        highlightAreaLine.scale = scale;\n        /*\n                 highlightAreaLine.rect = {\n                 height: clientRect.height,\n                 left: clientRect.left - xOffset,\n                 top: clientRect.top - yOffset,\n                 width: clientRect.width,\n                 };\n                 */\n\n        if (annotationFlag) {\n          highlightAreaLine.rect = {\n            height: ANNOTATION_WIDTH, //rangeAnnotationBoundingClientRect.height - rangeAnnotationBoundingClientRect.height/4,\n            left: window.innerWidth * annotationOffset - ANNOTATION_WIDTH,\n            top: rangeAnnotationBoundingClientRect.top - yOffset,\n            width: ANNOTATION_WIDTH,\n          };\n        } else {\n          highlightAreaLine.rect = {\n            height: clientRect.height,\n            left: clientRect.left - xOffset,\n            top: clientRect.top - yOffset,\n            width: clientRect.width,\n          };\n        }\n\n        highlightAreaLine.style.width = `${\n          highlightAreaLine.rect.width * scale\n        }px`;\n        highlightAreaLine.style.height = `${\n          strikeThroughLineThickness * scale\n        }px`;\n        highlightAreaLine.style.left = `${\n          highlightAreaLine.rect.left * scale\n        }px`;\n        highlightAreaLine.style.top = `${\n          (highlightAreaLine.rect.top +\n            highlightAreaLine.rect.height / 2 -\n            strikeThroughLineThickness / 2) *\n          scale\n        }px`;\n        highlightParent.append(highlightAreaLine);\n      }\n    }\n\n    if (annotationFlag) {\n      break;\n    }\n  }\n\n  if (useSVG && highlightAreaSVGDocFrag) {\n    const highlightAreaSVG = document.createElementNS(SVG_XML_NAMESPACE, "svg");\n    highlightAreaSVG.setAttribute("pointer-events", "none");\n    highlightAreaSVG.style.position = paginated ? "fixed" : "absolute";\n    highlightAreaSVG.style.overflow = "visible";\n    highlightAreaSVG.style.left = "0";\n    highlightAreaSVG.style.top = "0";\n    highlightAreaSVG.append(highlightAreaSVGDocFrag);\n    highlightParent.append(highlightAreaSVG);\n  }\n\n  const highlightBounding = document.createElement("div");\n\n  if (annotationFlag) {\n    highlightBounding.setAttribute("class", CLASS_ANNOTATION_BOUNDING_AREA);\n    highlightBounding.setAttribute(\n      "style",\n      `border-radius: ${roundedCorner}px !important; background-color: rgba(${highlight.color.red}, ${highlight.color.green}, ${highlight.color.blue}, ${opacity}) !important; ${extra}`\n    );\n  } else {\n    highlightBounding.setAttribute("class", CLASS_HIGHLIGHT_BOUNDING_AREA);\n  }\n\n  highlightBounding.style.setProperty("pointer-events", "none");\n  highlightBounding.style.position = paginated ? "fixed" : "absolute";\n  highlightBounding.scale = scale;\n\n  if (DEBUG_VISUALS) {\n    highlightBounding.setAttribute(\n      "style",\n      `outline-color: magenta; outline-style: solid; outline-width: 1px; outline-offset: -1px;`\n    );\n  }\n\n  if (annotationFlag) {\n    highlightBounding.rect = {\n      height: ANNOTATION_WIDTH, //rangeAnnotationBoundingClientRect.height - rangeAnnotationBoundingClientRect.height/4,\n      left: window.innerWidth * annotationOffset - ANNOTATION_WIDTH,\n      top: rangeAnnotationBoundingClientRect.top - yOffset,\n      width: ANNOTATION_WIDTH,\n    };\n  } else {\n    const rangeBoundingClientRect = range.getBoundingClientRect();\n    highlightBounding.rect = {\n      height: rangeBoundingClientRect.height,\n      left: rangeBoundingClientRect.left - xOffset,\n      top: rangeBoundingClientRect.top - yOffset,\n      width: rangeBoundingClientRect.width,\n    };\n  }\n\n  highlightBounding.style.width = `${highlightBounding.rect.width * scale}px`;\n  highlightBounding.style.height = `${highlightBounding.rect.height * scale}px`;\n  highlightBounding.style.left = `${highlightBounding.rect.left * scale}px`;\n  highlightBounding.style.top = `${highlightBounding.rect.top * scale}px`;\n\n  highlightParent.append(highlightBounding);\n  highlightsContainer.append(highlightParent);\n\n  return highlightParent;\n}\n\nfunction createOrderedRange(startNode, startOffset, endNode, endOffset) {\n  const range = new Range();\n  range.setStart(startNode, startOffset);\n  range.setEnd(endNode, endOffset);\n  if (!range.collapsed) {\n    return range;\n  }\n  console.log(">>> createOrderedRange COLLAPSED ... RANGE REVERSE?");\n  const rangeReverse = new Range();\n  rangeReverse.setStart(endNode, endOffset);\n  rangeReverse.setEnd(startNode, startOffset);\n  if (!rangeReverse.collapsed) {\n    console.log(">>> createOrderedRange RANGE REVERSE OK.");\n    return range;\n  }\n  console.log(">>> createOrderedRange RANGE REVERSE ALSO COLLAPSED?!");\n  return undefined;\n}\n\nfunction convertRange(range, getCssSelector, computeElementCFI) {\n  const startIsElement = range.startContainer.nodeType === Node.ELEMENT_NODE;\n  const startContainerElement = startIsElement\n    ? range.startContainer\n    : range.startContainer.parentNode &&\n      range.startContainer.parentNode.nodeType === Node.ELEMENT_NODE\n    ? range.startContainer.parentNode\n    : undefined;\n  if (!startContainerElement) {\n    return undefined;\n  }\n  const startContainerChildTextNodeIndex = startIsElement\n    ? -1\n    : Array.from(startContainerElement.childNodes).indexOf(\n        range.startContainer\n      );\n  if (startContainerChildTextNodeIndex < -1) {\n    return undefined;\n  }\n  const startContainerElementCssSelector = getCssSelector(\n    startContainerElement\n  );\n  const endIsElement = range.endContainer.nodeType === Node.ELEMENT_NODE;\n  const endContainerElement = endIsElement\n    ? range.endContainer\n    : range.endContainer.parentNode &&\n      range.endContainer.parentNode.nodeType === Node.ELEMENT_NODE\n    ? range.endContainer.parentNode\n    : undefined;\n  if (!endContainerElement) {\n    return undefined;\n  }\n  const endContainerChildTextNodeIndex = endIsElement\n    ? -1\n    : Array.from(endContainerElement.childNodes).indexOf(range.endContainer);\n  if (endContainerChildTextNodeIndex < -1) {\n    return undefined;\n  }\n  const endContainerElementCssSelector = getCssSelector(endContainerElement);\n  const commonElementAncestor = getCommonAncestorElement(\n    range.startContainer,\n    range.endContainer\n  );\n  if (!commonElementAncestor) {\n    console.log("^^^ NO RANGE COMMON ANCESTOR?!");\n    return undefined;\n  }\n  if (range.commonAncestorContainer) {\n    const rangeCommonAncestorElement =\n      range.commonAncestorContainer.nodeType === Node.ELEMENT_NODE\n        ? range.commonAncestorContainer\n        : range.commonAncestorContainer.parentNode;\n    if (\n      rangeCommonAncestorElement &&\n      rangeCommonAncestorElement.nodeType === Node.ELEMENT_NODE\n    ) {\n      if (commonElementAncestor !== rangeCommonAncestorElement) {\n        console.log(">>>>>> COMMON ANCESTOR CONTAINER DIFF??!");\n        console.log(getCssSelector(commonElementAncestor));\n        console.log(getCssSelector(rangeCommonAncestorElement));\n      }\n    }\n  }\n  const rootElementCfi = computeElementCFI(commonElementAncestor);\n  const startElementCfi = computeElementCFI(startContainerElement);\n  const endElementCfi = computeElementCFI(endContainerElement);\n  let cfi;\n  if (rootElementCfi && startElementCfi && endElementCfi) {\n    let startElementOrTextCfi = startElementCfi;\n    if (!startIsElement) {\n      const startContainerChildTextNodeIndexForCfi = getChildTextNodeCfiIndex(\n        startContainerElement,\n        range.startContainer\n      );\n      startElementOrTextCfi =\n        startElementCfi +\n        "/" +\n        startContainerChildTextNodeIndexForCfi +\n        ":" +\n        range.startOffset;\n    } else {\n      if (\n        range.startOffset >= 0 &&\n        range.startOffset < startContainerElement.childNodes.length\n      ) {\n        const childNode = startContainerElement.childNodes[range.startOffset];\n        if (childNode.nodeType === Node.ELEMENT_NODE) {\n          startElementOrTextCfi =\n            startElementCfi + "/" + (range.startOffset + 1) * 2;\n        } else {\n          const cfiTextNodeIndex = getChildTextNodeCfiIndex(\n            startContainerElement,\n            childNode\n          );\n          startElementOrTextCfi = startElementCfi + "/" + cfiTextNodeIndex;\n        }\n      } else {\n        const cfiIndexOfLastElement =\n          startContainerElement.childElementCount * 2;\n        const lastChildNode =\n          startContainerElement.childNodes[\n            startContainerElement.childNodes.length - 1\n          ];\n        if (lastChildNode.nodeType === Node.ELEMENT_NODE) {\n          startElementOrTextCfi =\n            startElementCfi + "/" + (cfiIndexOfLastElement + 1);\n        } else {\n          startElementOrTextCfi =\n            startElementCfi + "/" + (cfiIndexOfLastElement + 2);\n        }\n      }\n    }\n    let endElementOrTextCfi = endElementCfi;\n    if (!endIsElement) {\n      const endContainerChildTextNodeIndexForCfi = getChildTextNodeCfiIndex(\n        endContainerElement,\n        range.endContainer\n      );\n      endElementOrTextCfi =\n        endElementCfi +\n        "/" +\n        endContainerChildTextNodeIndexForCfi +\n        ":" +\n        range.endOffset;\n    } else {\n      if (\n        range.endOffset >= 0 &&\n        range.endOffset < endContainerElement.childNodes.length\n      ) {\n        const childNode = endContainerElement.childNodes[range.endOffset];\n        if (childNode.nodeType === Node.ELEMENT_NODE) {\n          endElementOrTextCfi = endElementCfi + "/" + (range.endOffset + 1) * 2;\n        } else {\n          const cfiTextNodeIndex = getChildTextNodeCfiIndex(\n            endContainerElement,\n            childNode\n          );\n          endElementOrTextCfi = endElementCfi + "/" + cfiTextNodeIndex;\n        }\n      } else {\n        const cfiIndexOfLastElement = endContainerElement.childElementCount * 2;\n        const lastChildNode =\n          endContainerElement.childNodes[\n            endContainerElement.childNodes.length - 1\n          ];\n        if (lastChildNode.nodeType === Node.ELEMENT_NODE) {\n          endElementOrTextCfi =\n            endElementCfi + "/" + (cfiIndexOfLastElement + 1);\n        } else {\n          endElementOrTextCfi =\n            endElementCfi + "/" + (cfiIndexOfLastElement + 2);\n        }\n      }\n    }\n    cfi =\n      rootElementCfi +\n      "," +\n      startElementOrTextCfi.replace(rootElementCfi, "") +\n      "," +\n      endElementOrTextCfi.replace(rootElementCfi, "");\n  }\n  return {\n    cfi,\n    endContainerChildTextNodeIndex,\n    endContainerElementCssSelector,\n    endOffset: range.endOffset,\n    startContainerChildTextNodeIndex,\n    startContainerElementCssSelector,\n    startOffset: range.startOffset,\n  };\n}\n\nfunction convertRangeInfo(document, rangeInfo) {\n  const startElement = document.querySelector(\n    rangeInfo.startContainerElementCssSelector\n  );\n  if (!startElement) {\n    console.log("^^^ convertRangeInfo NO START ELEMENT CSS SELECTOR?!");\n    return undefined;\n  }\n  let startContainer = startElement;\n  if (rangeInfo.startContainerChildTextNodeIndex >= 0) {\n    if (\n      rangeInfo.startContainerChildTextNodeIndex >=\n      startElement.childNodes.length\n    ) {\n      console.log(\n        "^^^ convertRangeInfo rangeInfo.startContainerChildTextNodeIndex >= startElement.childNodes.length?!"\n      );\n      return undefined;\n    }\n    startContainer =\n      startElement.childNodes[rangeInfo.startContainerChildTextNodeIndex];\n    if (startContainer.nodeType !== Node.TEXT_NODE) {\n      console.log(\n        "^^^ convertRangeInfo startContainer.nodeType !== Node.TEXT_NODE?!"\n      );\n      return undefined;\n    }\n  }\n  const endElement = document.querySelector(\n    rangeInfo.endContainerElementCssSelector\n  );\n  if (!endElement) {\n    console.log("^^^ convertRangeInfo NO END ELEMENT CSS SELECTOR?!");\n    return undefined;\n  }\n  let endContainer = endElement;\n  if (rangeInfo.endContainerChildTextNodeIndex >= 0) {\n    if (\n      rangeInfo.endContainerChildTextNodeIndex >= endElement.childNodes.length\n    ) {\n      console.log(\n        "^^^ convertRangeInfo rangeInfo.endContainerChildTextNodeIndex >= endElement.childNodes.length?!"\n      );\n      return undefined;\n    }\n    endContainer =\n      endElement.childNodes[rangeInfo.endContainerChildTextNodeIndex];\n    if (endContainer.nodeType !== Node.TEXT_NODE) {\n      console.log(\n        "^^^ convertRangeInfo endContainer.nodeType !== Node.TEXT_NODE?!"\n      );\n      return undefined;\n    }\n  }\n  return createOrderedRange(\n    startContainer,\n    rangeInfo.startOffset,\n    endContainer,\n    rangeInfo.endOffset\n  );\n}\n\nfunction frameForHighlightAnnotationMarkWithID(win, id) {\n  let clientRects = frameForHighlightWithID(id);\n  if (!clientRects) return;\n\n  var topClientRect = clientRects[0];\n  var maxHeight = topClientRect.height;\n  for (const clientRect of clientRects) {\n    if (clientRect.top < topClientRect.top) topClientRect = clientRect;\n    if (clientRect.height > maxHeight) maxHeight = clientRect.height;\n  }\n\n  const document = win.document;\n\n  const scrollElement = getScrollingElement(document);\n  const paginated = isPaginated(document);\n  const bodyRect = document.body.getBoundingClientRect();\n  let yOffset;\n  if (navigator.userAgent.match(/Android/i)) {\n    yOffset = paginated ? -scrollElement.scrollTop : bodyRect.top;\n  } else if (navigator.userAgent.match(/iPhone|iPad|iPod/i)) {\n    yOffset = paginated ? 0 : bodyRect.top;\n  }\n  var newTop = topClientRect.top;\n\n  if (_highlightsContainer) {\n    do {\n      var boundingAreas = document.getElementsByClassName(\n        CLASS_ANNOTATION_BOUNDING_AREA\n      );\n      var found = false;\n      //for (let i = 0, length = boundingAreas.snapshotLength; i < length; ++i) {\n      for (\n        var i = 0, len = boundingAreas.length | 0;\n        i < len;\n        i = (i + 1) | 0\n      ) {\n        var boundingArea = boundingAreas[i];\n        if (Math.abs(boundingArea.rect.top - (newTop - yOffset)) < 3) {\n          newTop += boundingArea.rect.height;\n          found = true;\n          break;\n        }\n      }\n    } while (found);\n  }\n\n  topClientRect.top = newTop;\n  topClientRect.height = maxHeight;\n\n  return topClientRect;\n}\n\nfunction highlightWithID(id) {\n  let i = -1;\n  const highlight = _highlights.find((h, j) => {\n    i = j;\n    return h.id === id;\n  });\n  return highlight;\n}\n\nfunction frameForHighlightWithID(id) {\n  const highlight = highlightWithID(id);\n  if (!highlight) return;\n\n  const document = window.document;\n  const scrollElement = getScrollingElement(document);\n  const range = convertRangeInfo(document, highlight.rangeInfo);\n  if (!range) {\n    return undefined;\n  }\n\n  const drawUnderline = false;\n  const drawStrikeThrough = false;\n  const doNotMergeHorizontallyAlignedRects = drawUnderline || drawStrikeThrough;\n  //const clientRects = DEBUG_VISUALS ? range.getClientRects() :\n  const clientRects = highlight_getClientRectsNoOverlap(\n    range,\n    doNotMergeHorizontallyAlignedRects\n  );\n\n  return clientRects;\n}\n\nfunction rangeInfo2Location(rangeInfo) {\n  return {\n    cssSelector: rangeInfo.startContainerElementCssSelector,\n    partialCfi: rangeInfo.cfi,\n    domRange: {\n      start: {\n        cssSelector: rangeInfo.startContainerElementCssSelector,\n        textNodeIndex: rangeInfo.startContainerChildTextNodeIndex,\n        offset: rangeInfo.startOffset,\n      },\n      end: {\n        cssSelector: rangeInfo.endContainerElementCssSelector,\n        textNodeIndex: rangeInfo.endContainerChildTextNodeIndex,\n        offset: rangeInfo.endOffset,\n      },\n    },\n  };\n}\n\nfunction location2RangeInfo(location) {\n  const locations = location.locations;\n  const domRange = locations.domRange;\n  const start = domRange.start;\n  const end = domRange.end;\n\n  return {\n    cfi: location.partialCfi,\n    endContainerChildTextNodeIndex: end.textNodeIndex,\n    endContainerElementCssSelector: end.cssSelector,\n    endOffset: end.offset,\n    startContainerChildTextNodeIndex: start.textNodeIndex,\n    startContainerElementCssSelector: start.cssSelector,\n    startOffset: start.offset,\n  };\n}\n\nfunction rectangleForHighlightWithID(id) {\n  const highlight = highlightWithID(id);\n  if (!highlight) return;\n\n  const document = window.document;\n  const scrollElement = getScrollingElement(document);\n  const range = convertRangeInfo(document, highlight.rangeInfo);\n  if (!range) {\n    return undefined;\n  }\n\n  const drawUnderline = false;\n  const drawStrikeThrough = false;\n  const doNotMergeHorizontallyAlignedRects = drawUnderline || drawStrikeThrough;\n  //const clientRects = DEBUG_VISUALS ? range.getClientRects() :\n  const clientRects = highlight_getClientRectsNoOverlap(\n    range,\n    doNotMergeHorizontallyAlignedRects\n  );\n  var size = {\n    screenWidth: window.outerWidth,\n    screenHeight: window.outerHeight,\n    left: clientRects[0].left,\n    width: clientRects[0].width,\n    top: clientRects[0].top,\n    height: clientRects[0].height,\n  };\n\n  return size;\n}\n\nfunction getSelectionRect() {\n  try {\n    var sel = window.getSelection();\n    if (!sel) {\n      return;\n    }\n    var range = sel.getRangeAt(0);\n\n    const clientRect = range.getBoundingClientRect();\n\n    var handleBounds = {\n      screenWidth: window.outerWidth,\n      screenHeight: window.outerHeight,\n      left: clientRect.left,\n      width: clientRect.width,\n      top: clientRect.top,\n      height: clientRect.height,\n    };\n    return handleBounds;\n  } catch (e) {\n    return null;\n  }\n}\n\nfunction setScrollMode(flag) {\n  if (!flag) {\n    document.documentElement.classList.add(CLASS_PAGINATED);\n  } else {\n    document.documentElement.classList.remove(CLASS_PAGINATED);\n  }\n}\n\n/*\n if (document.addEventListener) { // IE >= 9; other browsers\n        document.addEventListener(\'contextmenu\', function(e) {\n            //alert("You\'ve tried to open context menu"); //here you draw your own menu\n            //e.preventDefault();\n            //let getCssSelector = fullQualifiedSelector;\n            \n\t\t\tlet str = window.getSelection();\n\t\t\tlet selectionInfo = getCurrentSelectionInfo();\n\t\t\tlet pos = createHighlight(selectionInfo,{red:10,green:50,blue:230},true);\n\t\t\tlet ret2 = createAnnotation(pos.id);\n\t\t\t\n  }, false);\n    } else { // IE < 9\n        document.attachEvent(\'oncontextmenu\', function() {\n            alert("You\'ve tried to open context menu");\n            window.event.returnValue = false;\n        });\n    }\n*/\n\n// EXTERNAL MODULE: ./node_modules/string.prototype.matchall/index.js\nvar string_prototype_matchall = __webpack_require__(4956);\nvar string_prototype_matchall_default = /*#__PURE__*/__webpack_require__.n(string_prototype_matchall);\n;// CONCATENATED MODULE: ./src/selection.js\n//\n//  Copyright 2021 Readium Foundation. All rights reserved.\n//  Use of this source code is governed by the BSD-style license\n//  available in the top-level LICENSE file of the project.\n//\n\n\n\n\n\n// Polyfill for Android API 26\n\nstring_prototype_matchall_default().shim();\n\nconst selection_debug = true;\n\nfunction getCurrentSelection() {\n  const text = getCurrentSelectionText();\n  if (!text) {\n    return null;\n  }\n  const rect = selection_getSelectionRect();\n  return { text, rect };\n}\n\nfunction selection_getSelectionRect() {\n  try {\n    let sel = window.getSelection();\n    if (!sel) {\n      return;\n    }\n    let range = sel.getRangeAt(0);\n\n    return toNativeRect(range.getBoundingClientRect());\n  } catch (e) {\n    logError(e);\n    return null;\n  }\n}\n\nfunction getCurrentSelectionText() {\n  const selection = window.getSelection();\n  if (!selection) {\n    return undefined;\n  }\n  if (selection.isCollapsed) {\n    return undefined;\n  }\n  const highlight = selection.toString();\n  const cleanHighlight = highlight\n    .trim()\n    .replace(/\\n/g, " ")\n    .replace(/\\s\\s+/g, " ");\n  if (cleanHighlight.length === 0) {\n    return undefined;\n  }\n  if (!selection.anchorNode || !selection.focusNode) {\n    return undefined;\n  }\n  const range =\n    selection.rangeCount === 1\n      ? selection.getRangeAt(0)\n      : selection_createOrderedRange(\n          selection.anchorNode,\n          selection.anchorOffset,\n          selection.focusNode,\n          selection.focusOffset\n        );\n  if (!range || range.collapsed) {\n    selection_log("$$$$$$$$$$$$$$$$$ CANNOT GET NON-COLLAPSED SELECTION RANGE?!");\n    return undefined;\n  }\n\n  const text = document.body.textContent;\n  const textRange = text_range_TextRange.fromRange(range).relativeTo(document.body);\n  const start = textRange.start.offset;\n  const end = textRange.end.offset;\n\n  const snippetLength = 200;\n\n  // Compute the text before the highlight, ignoring the first "word", which might be cut.\n  let before = text.slice(Math.max(0, start - snippetLength), start);\n  let firstWordStart = before.search(/\\P{L}\\p{L}/gu);\n  if (firstWordStart !== -1) {\n    before = before.slice(firstWordStart + 1);\n  }\n\n  // Compute the text after the highlight, ignoring the last "word", which might be cut.\n  let after = text.slice(end, Math.min(text.length, end + snippetLength));\n  let lastWordEnd = Array.from(after.matchAll(/\\p{L}\\P{L}/gu)).pop();\n  if (lastWordEnd !== undefined && lastWordEnd.index > 1) {\n    after = after.slice(0, lastWordEnd.index + 1);\n  }\n\n  return { highlight, before, after };\n}\n\nfunction selection_createOrderedRange(startNode, startOffset, endNode, endOffset) {\n  const range = new Range();\n  range.setStart(startNode, startOffset);\n  range.setEnd(endNode, endOffset);\n  if (!range.collapsed) {\n    return range;\n  }\n  selection_log(">>> createOrderedRange COLLAPSED ... RANGE REVERSE?");\n  const rangeReverse = new Range();\n  rangeReverse.setStart(endNode, endOffset);\n  rangeReverse.setEnd(startNode, startOffset);\n  if (!rangeReverse.collapsed) {\n    selection_log(">>> createOrderedRange RANGE REVERSE OK.");\n    return range;\n  }\n  selection_log(">>> createOrderedRange RANGE REVERSE ALSO COLLAPSED?!");\n  return undefined;\n}\n\nfunction selection_convertRangeInfo(document, rangeInfo) {\n  const startElement = document.querySelector(\n    rangeInfo.startContainerElementCssSelector\n  );\n  if (!startElement) {\n    selection_log("^^^ convertRangeInfo NO START ELEMENT CSS SELECTOR?!");\n    return undefined;\n  }\n  let startContainer = startElement;\n  if (rangeInfo.startContainerChildTextNodeIndex >= 0) {\n    if (\n      rangeInfo.startContainerChildTextNodeIndex >=\n      startElement.childNodes.length\n    ) {\n      selection_log(\n        "^^^ convertRangeInfo rangeInfo.startContainerChildTextNodeIndex >= startElement.childNodes.length?!"\n      );\n      return undefined;\n    }\n    startContainer =\n      startElement.childNodes[rangeInfo.startContainerChildTextNodeIndex];\n    if (startContainer.nodeType !== Node.TEXT_NODE) {\n      selection_log("^^^ convertRangeInfo startContainer.nodeType !== Node.TEXT_NODE?!");\n      return undefined;\n    }\n  }\n  const endElement = document.querySelector(\n    rangeInfo.endContainerElementCssSelector\n  );\n  if (!endElement) {\n    selection_log("^^^ convertRangeInfo NO END ELEMENT CSS SELECTOR?!");\n    return undefined;\n  }\n  let endContainer = endElement;\n  if (rangeInfo.endContainerChildTextNodeIndex >= 0) {\n    if (\n      rangeInfo.endContainerChildTextNodeIndex >= endElement.childNodes.length\n    ) {\n      selection_log(\n        "^^^ convertRangeInfo rangeInfo.endContainerChildTextNodeIndex >= endElement.childNodes.length?!"\n      );\n      return undefined;\n    }\n    endContainer =\n      endElement.childNodes[rangeInfo.endContainerChildTextNodeIndex];\n    if (endContainer.nodeType !== Node.TEXT_NODE) {\n      selection_log("^^^ convertRangeInfo endContainer.nodeType !== Node.TEXT_NODE?!");\n      return undefined;\n    }\n  }\n  return selection_createOrderedRange(\n    startContainer,\n    rangeInfo.startOffset,\n    endContainer,\n    rangeInfo.endOffset\n  );\n}\n\nfunction selection_location2RangeInfo(location) {\n  const locations = location.locations;\n  const domRange = locations.domRange;\n  const start = domRange.start;\n  const end = domRange.end;\n\n  return {\n    endContainerChildTextNodeIndex: end.textNodeIndex,\n    endContainerElementCssSelector: end.cssSelector,\n    endOffset: end.offset,\n    startContainerChildTextNodeIndex: start.textNodeIndex,\n    startContainerElementCssSelector: start.cssSelector,\n    startOffset: start.offset,\n  };\n}\n\nfunction selection_log() {\n  if (selection_debug) {\n    log.apply(null, arguments);\n  }\n}\n\n;// CONCATENATED MODULE: ./src/index.js\n//\n//  Copyright 2021 Readium Foundation. All rights reserved.\n//  Use of this source code is governed by the BSD-style license\n//  available in the top-level LICENSE file of the project.\n//\n\n// Base script used by both reflowable and fixed layout resources.\n\n\n\n\n\n\n\n// Public API used by the navigator.\nwindow.readium = {\n  // utils\n  scrollToId: scrollToId,\n  scrollToPosition: scrollToPosition,\n  scrollToText: scrollToText,\n  scrollLeft: scrollLeft,\n  scrollRight: scrollRight,\n  scrollToStart: scrollToStart,\n  scrollToEnd: scrollToEnd,\n  setProperty: setProperty,\n  removeProperty: removeProperty,\n\n  // selection\n  getCurrentSelection: getCurrentSelection,\n\n  // decoration\n  registerDecorationTemplates: registerTemplates,\n  getDecorations: getDecorations,\n};\n\n// Legacy highlights API.\nwindow.createAnnotation = createAnnotation;\nwindow.createHighlight = createHighlight;\nwindow.destroyHighlight = destroyHighlight;\nwindow.getCurrentSelectionInfo = getCurrentSelectionInfo;\nwindow.getSelectionRect = getSelectionRect;\nwindow.rectangleForHighlightWithID = rectangleForHighlightWithID;\nwindow.setScrollMode = setScrollMode;\n\n;// CONCATENATED MODULE: ./src/index-reflowable.js\n//\n//  Copyright 2021 Readium Foundation. All rights reserved.\n//  Use of this source code is governed by the BSD-style license\n//  available in the top-level LICENSE file of the project.\n//\n\n// Script used for reflowable resources.\n\n\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///7640\n')},4654:()=>{}},__webpack_module_cache__={};function __webpack_require__(n){var t=__webpack_module_cache__[n];if(void 0!==t)return t.exports;var e=__webpack_module_cache__[n]={exports:{}};return __webpack_modules__[n](e,e.exports,__webpack_require__),e.exports}__webpack_require__.n=n=>{var t=n&&n.__esModule?()=>n.default:()=>n;return __webpack_require__.d(t,{a:t}),t},__webpack_require__.d=(n,t)=>{for(var e in t)__webpack_require__.o(t,e)&&!__webpack_require__.o(n,e)&&Object.defineProperty(n,e,{enumerable:!0,get:t[e]})},__webpack_require__.o=(n,t)=>Object.prototype.hasOwnProperty.call(n,t);var __webpack_exports__=__webpack_require__(7640)})();