(()=>{var __webpack_modules__={3089:(__unused_webpack_module,exports)=>{"use strict";eval('var __webpack_unused_export__;\n\n/**\n * Implementation of Myers\' online approximate string matching algorithm [1],\n * with additional optimizations suggested by [2].\n *\n * This has O((k/w) * n) complexity where `n` is the length of the text, `k` is\n * the maximum number of errors allowed (always <= the pattern length) and `w`\n * is the word size. Because JS only supports bitwise operations on 32 bit\n * integers, `w` is 32.\n *\n * As far as I am aware, there aren\'t any online algorithms which are\n * significantly better for a wide range of input parameters. The problem can be\n * solved faster using "filter then verify" approaches which first filter out\n * regions of the text that cannot match using a "cheap" check and then verify\n * the remaining potential matches. The verify step requires an algorithm such\n * as this one however.\n *\n * The algorithm\'s approach is essentially to optimize the classic dynamic\n * programming solution to the problem by computing columns of the matrix in\n * word-sized chunks (ie. dealing with 32 chars of the pattern at a time) and\n * avoiding calculating regions of the matrix where the minimum error count is\n * guaranteed to exceed the input threshold.\n *\n * The paper consists of two parts, the first describes the core algorithm for\n * matching patterns <= the size of a word (implemented by `advanceBlock` here).\n * The second uses the core algorithm as part of a larger block-based algorithm\n * to handle longer patterns.\n *\n * [1] G. Myers, “A Fast Bit-Vector Algorithm for Approximate String Matching\n * Based on Dynamic Programming,” vol. 46, no. 3, pp. 395–415, 1999.\n *\n * [2] Šošić, M. (2014). An simd dynamic programming c/c++ library (Doctoral\n * dissertation, Fakultet Elektrotehnike i računarstva, Sveučilište u Zagrebu).\n */\n__webpack_unused_export__ = ({ value: true });\nfunction reverse(s) {\n    return s\n        .split("")\n        .reverse()\n        .join("");\n}\n/**\n * Given the ends of approximate matches for `pattern` in `text`, find\n * the start of the matches.\n *\n * @param findEndFn - Function for finding the end of matches in\n * text.\n * @return Matches with the `start` property set.\n */\nfunction findMatchStarts(text, pattern, matches) {\n    var patRev = reverse(pattern);\n    return matches.map(function (m) {\n        // Find start of each match by reversing the pattern and matching segment\n        // of text and searching for an approx match with the same number of\n        // errors.\n        var minStart = Math.max(0, m.end - pattern.length - m.errors);\n        var textRev = reverse(text.slice(minStart, m.end));\n        // If there are multiple possible start points, choose the one that\n        // maximizes the length of the match.\n        var start = findMatchEnds(textRev, patRev, m.errors).reduce(function (min, rm) {\n            if (m.end - rm.end < min) {\n                return m.end - rm.end;\n            }\n            return min;\n        }, m.end);\n        return {\n            start: start,\n            end: m.end,\n            errors: m.errors\n        };\n    });\n}\n/**\n * Return 1 if a number is non-zero or zero otherwise, without using\n * conditional operators.\n *\n * This should get inlined into `advanceBlock` below by the JIT.\n *\n * Adapted from https://stackoverflow.com/a/3912218/434243\n */\nfunction oneIfNotZero(n) {\n    return ((n | -n) >> 31) & 1;\n}\n/**\n * Block calculation step of the algorithm.\n *\n * From Fig 8. on p. 408 of [1], additionally optimized to replace conditional\n * checks with bitwise operations as per Section 4.2.3 of [2].\n *\n * @param ctx - The pattern context object\n * @param peq - The `peq` array for the current character (`ctx.peq.get(ch)`)\n * @param b - The block level\n * @param hIn - Horizontal input delta ∈ {1,0,-1}\n * @return Horizontal output delta ∈ {1,0,-1}\n */\nfunction advanceBlock(ctx, peq, b, hIn) {\n    var pV = ctx.P[b];\n    var mV = ctx.M[b];\n    var hInIsNegative = hIn >>> 31; // 1 if hIn < 0 or 0 otherwise.\n    var eq = peq[b] | hInIsNegative;\n    // Step 1: Compute horizontal deltas.\n    var xV = eq | mV;\n    var xH = (((eq & pV) + pV) ^ pV) | eq;\n    var pH = mV | ~(xH | pV);\n    var mH = pV & xH;\n    // Step 2: Update score (value of last row of this block).\n    var hOut = oneIfNotZero(pH & ctx.lastRowMask[b]) -\n        oneIfNotZero(mH & ctx.lastRowMask[b]);\n    // Step 3: Update vertical deltas for use when processing next char.\n    pH <<= 1;\n    mH <<= 1;\n    mH |= hInIsNegative;\n    pH |= oneIfNotZero(hIn) - hInIsNegative; // set pH[0] if hIn > 0\n    pV = mH | ~(xV | pH);\n    mV = pH & xV;\n    ctx.P[b] = pV;\n    ctx.M[b] = mV;\n    return hOut;\n}\n/**\n * Find the ends and error counts for matches of `pattern` in `text`.\n *\n * Only the matches with the lowest error count are reported. Other matches\n * with error counts <= maxErrors are discarded.\n *\n * This is the block-based search algorithm from Fig. 9 on p.410 of [1].\n */\nfunction findMatchEnds(text, pattern, maxErrors) {\n    if (pattern.length === 0) {\n        return [];\n    }\n    // Clamp error count so we can rely on the `maxErrors` and `pattern.length`\n    // rows being in the same block below.\n    maxErrors = Math.min(maxErrors, pattern.length);\n    var matches = [];\n    // Word size.\n    var w = 32;\n    // Index of maximum block level.\n    var bMax = Math.ceil(pattern.length / w) - 1;\n    // Context used across block calculations.\n    var ctx = {\n        P: new Uint32Array(bMax + 1),\n        M: new Uint32Array(bMax + 1),\n        lastRowMask: new Uint32Array(bMax + 1)\n    };\n    ctx.lastRowMask.fill(1 << 31);\n    ctx.lastRowMask[bMax] = 1 << (pattern.length - 1) % w;\n    // Dummy "peq" array for chars in the text which do not occur in the pattern.\n    var emptyPeq = new Uint32Array(bMax + 1);\n    // Map of UTF-16 character code to bit vector indicating positions in the\n    // pattern that equal that character.\n    var peq = new Map();\n    // Version of `peq` that only stores mappings for small characters. This\n    // allows faster lookups when iterating through the text because a simple\n    // array lookup can be done instead of a hash table lookup.\n    var asciiPeq = [];\n    for (var i = 0; i < 256; i++) {\n        asciiPeq.push(emptyPeq);\n    }\n    // Calculate `ctx.peq` - a map of character values to bitmasks indicating\n    // positions of that character within the pattern, where each bit represents\n    // a position in the pattern.\n    for (var c = 0; c < pattern.length; c += 1) {\n        var val = pattern.charCodeAt(c);\n        if (peq.has(val)) {\n            // Duplicate char in pattern.\n            continue;\n        }\n        var charPeq = new Uint32Array(bMax + 1);\n        peq.set(val, charPeq);\n        if (val < asciiPeq.length) {\n            asciiPeq[val] = charPeq;\n        }\n        for (var b = 0; b <= bMax; b += 1) {\n            charPeq[b] = 0;\n            // Set all the bits where the pattern matches the current char (ch).\n            // For indexes beyond the end of the pattern, always set the bit as if the\n            // pattern contained a wildcard char in that position.\n            for (var r = 0; r < w; r += 1) {\n                var idx = b * w + r;\n                if (idx >= pattern.length) {\n                    continue;\n                }\n                var match = pattern.charCodeAt(idx) === val;\n                if (match) {\n                    charPeq[b] |= 1 << r;\n                }\n            }\n        }\n    }\n    // Index of last-active block level in the column.\n    var y = Math.max(0, Math.ceil(maxErrors / w) - 1);\n    // Initialize maximum error count at bottom of each block.\n    var score = new Uint32Array(bMax + 1);\n    for (var b = 0; b <= y; b += 1) {\n        score[b] = (b + 1) * w;\n    }\n    score[bMax] = pattern.length;\n    // Initialize vertical deltas for each block.\n    for (var b = 0; b <= y; b += 1) {\n        ctx.P[b] = ~0;\n        ctx.M[b] = 0;\n    }\n    // Process each char of the text, computing the error count for `w` chars of\n    // the pattern at a time.\n    for (var j = 0; j < text.length; j += 1) {\n        // Lookup the bitmask representing the positions of the current char from\n        // the text within the pattern.\n        var charCode = text.charCodeAt(j);\n        var charPeq = void 0;\n        if (charCode < asciiPeq.length) {\n            // Fast array lookup.\n            charPeq = asciiPeq[charCode];\n        }\n        else {\n            // Slower hash table lookup.\n            charPeq = peq.get(charCode);\n            if (typeof charPeq === "undefined") {\n                charPeq = emptyPeq;\n            }\n        }\n        // Calculate error count for blocks that we definitely have to process for\n        // this column.\n        var carry = 0;\n        for (var b = 0; b <= y; b += 1) {\n            carry = advanceBlock(ctx, charPeq, b, carry);\n            score[b] += carry;\n        }\n        // Check if we also need to compute an additional block, or if we can reduce\n        // the number of blocks processed for the next column.\n        if (score[y] - carry <= maxErrors &&\n            y < bMax &&\n            (charPeq[y + 1] & 1 || carry < 0)) {\n            // Error count for bottom block is under threshold, increase the number of\n            // blocks processed for this column & next by 1.\n            y += 1;\n            ctx.P[y] = ~0;\n            ctx.M[y] = 0;\n            var maxBlockScore = y === bMax ? pattern.length % w : w;\n            score[y] =\n                score[y - 1] +\n                    maxBlockScore -\n                    carry +\n                    advanceBlock(ctx, charPeq, y, carry);\n        }\n        else {\n            // Error count for bottom block exceeds threshold, reduce the number of\n            // blocks processed for the next column.\n            while (y > 0 && score[y] >= maxErrors + w) {\n                y -= 1;\n            }\n        }\n        // If error count is under threshold, report a match.\n        if (y === bMax && score[y] <= maxErrors) {\n            if (score[y] < maxErrors) {\n                // Discard any earlier, worse matches.\n                matches.splice(0, matches.length);\n            }\n            matches.push({\n                start: -1,\n                end: j + 1,\n                errors: score[y]\n            });\n            // Because `search` only reports the matches with the lowest error count,\n            // we can "ratchet down" the max error threshold whenever a match is\n            // encountered and thereby save a small amount of work for the remainder\n            // of the text.\n            maxErrors = score[y];\n        }\n    }\n    return matches;\n}\n/**\n * Search for matches for `pattern` in `text` allowing up to `maxErrors` errors.\n *\n * Returns the start, and end positions and error counts for each lowest-cost\n * match. Only the "best" matches are returned.\n */\nfunction search(text, pattern, maxErrors) {\n    var matches = findMatchEnds(text, pattern, maxErrors);\n    return findMatchStarts(text, pattern, matches);\n}\nexports.Z = search;\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///3089\n')},1924:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\n\nvar callBind = __webpack_require__(5559);\n\nvar $indexOf = callBind(GetIntrinsic('String.prototype.indexOf'));\n\nmodule.exports = function callBoundIntrinsic(name, allowMissing) {\n\tvar intrinsic = GetIntrinsic(name, !!allowMissing);\n\tif (typeof intrinsic === 'function' && $indexOf(name, '.prototype.') > -1) {\n\t\treturn callBind(intrinsic);\n\t}\n\treturn intrinsic;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2NhbGwtYmluZC9jYWxsQm91bmQuanM/NTQ1ZSJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBYTs7QUFFYixtQkFBbUIsbUJBQU8sQ0FBQyxHQUFlOztBQUUxQyxlQUFlLG1CQUFPLENBQUMsSUFBSTs7QUFFM0I7O0FBRUE7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0EiLCJmaWxlIjoiMTkyNC5qcyIsInNvdXJjZXNDb250ZW50IjpbIid1c2Ugc3RyaWN0JztcblxudmFyIEdldEludHJpbnNpYyA9IHJlcXVpcmUoJ2dldC1pbnRyaW5zaWMnKTtcblxudmFyIGNhbGxCaW5kID0gcmVxdWlyZSgnLi8nKTtcblxudmFyICRpbmRleE9mID0gY2FsbEJpbmQoR2V0SW50cmluc2ljKCdTdHJpbmcucHJvdG90eXBlLmluZGV4T2YnKSk7XG5cbm1vZHVsZS5leHBvcnRzID0gZnVuY3Rpb24gY2FsbEJvdW5kSW50cmluc2ljKG5hbWUsIGFsbG93TWlzc2luZykge1xuXHR2YXIgaW50cmluc2ljID0gR2V0SW50cmluc2ljKG5hbWUsICEhYWxsb3dNaXNzaW5nKTtcblx0aWYgKHR5cGVvZiBpbnRyaW5zaWMgPT09ICdmdW5jdGlvbicgJiYgJGluZGV4T2YobmFtZSwgJy5wcm90b3R5cGUuJykgPiAtMSkge1xuXHRcdHJldHVybiBjYWxsQmluZChpbnRyaW5zaWMpO1xuXHR9XG5cdHJldHVybiBpbnRyaW5zaWM7XG59O1xuIl0sInNvdXJjZVJvb3QiOiIifQ==\n//# sourceURL=webpack-internal:///1924\n")},5559:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar bind = __webpack_require__(8612);\nvar GetIntrinsic = __webpack_require__(210);\n\nvar $apply = GetIntrinsic('%Function.prototype.apply%');\nvar $call = GetIntrinsic('%Function.prototype.call%');\nvar $reflectApply = GetIntrinsic('%Reflect.apply%', true) || bind.call($call, $apply);\n\nvar $gOPD = GetIntrinsic('%Object.getOwnPropertyDescriptor%', true);\nvar $defineProperty = GetIntrinsic('%Object.defineProperty%', true);\nvar $max = GetIntrinsic('%Math.max%');\n\nif ($defineProperty) {\n\ttry {\n\t\t$defineProperty({}, 'a', { value: 1 });\n\t} catch (e) {\n\t\t// IE 8 has a broken defineProperty\n\t\t$defineProperty = null;\n\t}\n}\n\nmodule.exports = function callBind(originalFunction) {\n\tvar func = $reflectApply(bind, $call, arguments);\n\tif ($gOPD && $defineProperty) {\n\t\tvar desc = $gOPD(func, 'length');\n\t\tif (desc.configurable) {\n\t\t\t// original length, plus the receiver, minus any additional arguments (after the receiver)\n\t\t\t$defineProperty(\n\t\t\t\tfunc,\n\t\t\t\t'length',\n\t\t\t\t{ value: 1 + $max(0, originalFunction.length - (arguments.length - 1)) }\n\t\t\t);\n\t\t}\n\t}\n\treturn func;\n};\n\nvar applyBind = function applyBind() {\n\treturn $reflectApply(bind, $apply, arguments);\n};\n\nif ($defineProperty) {\n\t$defineProperty(module.exports, 'apply', { value: applyBind });\n} else {\n\tmodule.exports.apply = applyBind;\n}\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///5559\n")},4289:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar keys = __webpack_require__(2215);\nvar hasSymbols = typeof Symbol === 'function' && typeof Symbol('foo') === 'symbol';\n\nvar toStr = Object.prototype.toString;\nvar concat = Array.prototype.concat;\nvar origDefineProperty = Object.defineProperty;\n\nvar isFunction = function (fn) {\n\treturn typeof fn === 'function' && toStr.call(fn) === '[object Function]';\n};\n\nvar arePropertyDescriptorsSupported = function () {\n\tvar obj = {};\n\ttry {\n\t\torigDefineProperty(obj, 'x', { enumerable: false, value: obj });\n\t\t// eslint-disable-next-line no-unused-vars, no-restricted-syntax\n\t\tfor (var _ in obj) { // jscs:ignore disallowUnusedVariables\n\t\t\treturn false;\n\t\t}\n\t\treturn obj.x === obj;\n\t} catch (e) { /* this is IE 8. */\n\t\treturn false;\n\t}\n};\nvar supportsDescriptors = origDefineProperty && arePropertyDescriptorsSupported();\n\nvar defineProperty = function (object, name, value, predicate) {\n\tif (name in object && (!isFunction(predicate) || !predicate())) {\n\t\treturn;\n\t}\n\tif (supportsDescriptors) {\n\t\torigDefineProperty(object, name, {\n\t\t\tconfigurable: true,\n\t\t\tenumerable: false,\n\t\t\tvalue: value,\n\t\t\twritable: true\n\t\t});\n\t} else {\n\t\tobject[name] = value;\n\t}\n};\n\nvar defineProperties = function (object, map) {\n\tvar predicates = arguments.length > 2 ? arguments[2] : {};\n\tvar props = keys(map);\n\tif (hasSymbols) {\n\t\tprops = concat.call(props, Object.getOwnPropertySymbols(map));\n\t}\n\tfor (var i = 0; i < props.length; i += 1) {\n\t\tdefineProperty(object, props[i], map[props[i]], predicates[props[i]]);\n\t}\n};\n\ndefineProperties.supportsDescriptors = !!supportsDescriptors;\n\nmodule.exports = defineProperties;\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///4289\n")},2672:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\n\nvar CodePointAt = __webpack_require__(9044);\nvar IsInteger = __webpack_require__(7868);\nvar Type = __webpack_require__(1915);\n\nvar MAX_SAFE_INTEGER = __webpack_require__(5376);\n\nvar $TypeError = GetIntrinsic('%TypeError%');\n\n// https://262.ecma-international.org/6.0/#sec-advancestringindex\n\nmodule.exports = function AdvanceStringIndex(S, index, unicode) {\n\tif (Type(S) !== 'String') {\n\t\tthrow new $TypeError('Assertion failed: `S` must be a String');\n\t}\n\tif (!IsInteger(index) || index < 0 || index > MAX_SAFE_INTEGER) {\n\t\tthrow new $TypeError('Assertion failed: `length` must be an integer >= 0 and <= 2**53');\n\t}\n\tif (Type(unicode) !== 'Boolean') {\n\t\tthrow new $TypeError('Assertion failed: `unicode` must be a Boolean');\n\t}\n\tif (!unicode) {\n\t\treturn index + 1;\n\t}\n\tvar length = S.length;\n\tif ((index + 1) >= length) {\n\t\treturn index + 1;\n\t}\n\tvar cp = CodePointAt(S, index);\n\treturn index + cp['[[CodeUnitCount]]'];\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvQWR2YW5jZVN0cmluZ0luZGV4LmpzP2Q1ZmMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQWE7O0FBRWIsbUJBQW1CLG1CQUFPLENBQUMsR0FBZTs7QUFFMUMsa0JBQWtCLG1CQUFPLENBQUMsSUFBZTtBQUN6QyxnQkFBZ0IsbUJBQU8sQ0FBQyxJQUFhO0FBQ3JDLFdBQVcsbUJBQU8sQ0FBQyxJQUFROztBQUUzQix1QkFBdUIsbUJBQU8sQ0FBQyxJQUEyQjs7QUFFMUQ7O0FBRUE7O0FBRUE7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQSIsImZpbGUiOiIyNjcyLmpzIiwic291cmNlc0NvbnRlbnQiOlsiJ3VzZSBzdHJpY3QnO1xuXG52YXIgR2V0SW50cmluc2ljID0gcmVxdWlyZSgnZ2V0LWludHJpbnNpYycpO1xuXG52YXIgQ29kZVBvaW50QXQgPSByZXF1aXJlKCcuL0NvZGVQb2ludEF0Jyk7XG52YXIgSXNJbnRlZ2VyID0gcmVxdWlyZSgnLi9Jc0ludGVnZXInKTtcbnZhciBUeXBlID0gcmVxdWlyZSgnLi9UeXBlJyk7XG5cbnZhciBNQVhfU0FGRV9JTlRFR0VSID0gcmVxdWlyZSgnLi4vaGVscGVycy9tYXhTYWZlSW50ZWdlcicpO1xuXG52YXIgJFR5cGVFcnJvciA9IEdldEludHJpbnNpYygnJVR5cGVFcnJvciUnKTtcblxuLy8gaHR0cHM6Ly8yNjIuZWNtYS1pbnRlcm5hdGlvbmFsLm9yZy82LjAvI3NlYy1hZHZhbmNlc3RyaW5naW5kZXhcblxubW9kdWxlLmV4cG9ydHMgPSBmdW5jdGlvbiBBZHZhbmNlU3RyaW5nSW5kZXgoUywgaW5kZXgsIHVuaWNvZGUpIHtcblx0aWYgKFR5cGUoUykgIT09ICdTdHJpbmcnKSB7XG5cdFx0dGhyb3cgbmV3ICRUeXBlRXJyb3IoJ0Fzc2VydGlvbiBmYWlsZWQ6IGBTYCBtdXN0IGJlIGEgU3RyaW5nJyk7XG5cdH1cblx0aWYgKCFJc0ludGVnZXIoaW5kZXgpIHx8IGluZGV4IDwgMCB8fCBpbmRleCA+IE1BWF9TQUZFX0lOVEVHRVIpIHtcblx0XHR0aHJvdyBuZXcgJFR5cGVFcnJvcignQXNzZXJ0aW9uIGZhaWxlZDogYGxlbmd0aGAgbXVzdCBiZSBhbiBpbnRlZ2VyID49IDAgYW5kIDw9IDIqKjUzJyk7XG5cdH1cblx0aWYgKFR5cGUodW5pY29kZSkgIT09ICdCb29sZWFuJykge1xuXHRcdHRocm93IG5ldyAkVHlwZUVycm9yKCdBc3NlcnRpb24gZmFpbGVkOiBgdW5pY29kZWAgbXVzdCBiZSBhIEJvb2xlYW4nKTtcblx0fVxuXHRpZiAoIXVuaWNvZGUpIHtcblx0XHRyZXR1cm4gaW5kZXggKyAxO1xuXHR9XG5cdHZhciBsZW5ndGggPSBTLmxlbmd0aDtcblx0aWYgKChpbmRleCArIDEpID49IGxlbmd0aCkge1xuXHRcdHJldHVybiBpbmRleCArIDE7XG5cdH1cblx0dmFyIGNwID0gQ29kZVBvaW50QXQoUywgaW5kZXgpO1xuXHRyZXR1cm4gaW5kZXggKyBjcFsnW1tDb2RlVW5pdENvdW50XV0nXTtcbn07XG4iXSwic291cmNlUm9vdCI6IiJ9\n//# sourceURL=webpack-internal:///2672\n")},7800:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\nvar callBound = __webpack_require__(1924);\n\nvar $TypeError = GetIntrinsic('%TypeError%');\n\nvar IsArray = __webpack_require__(4400);\n\nvar $apply = GetIntrinsic('%Reflect.apply%', true) || callBound('%Function.prototype.apply%');\n\n// https://ecma-international.org/ecma-262/6.0/#sec-call\n\nmodule.exports = function Call(F, V) {\n\tvar argumentsList = arguments.length > 2 ? arguments[2] : [];\n\tif (!IsArray(argumentsList)) {\n\t\tthrow new $TypeError('Assertion failed: optional `argumentsList`, if provided, must be a List');\n\t}\n\treturn $apply(F, V, argumentsList);\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvQ2FsbC5qcz80OTA4Il0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFhOztBQUViLG1CQUFtQixtQkFBTyxDQUFDLEdBQWU7QUFDMUMsZ0JBQWdCLG1CQUFPLENBQUMsSUFBcUI7O0FBRTdDOztBQUVBLGNBQWMsbUJBQU8sQ0FBQyxJQUFXOztBQUVqQzs7QUFFQTs7QUFFQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQSIsImZpbGUiOiI3ODAwLmpzIiwic291cmNlc0NvbnRlbnQiOlsiJ3VzZSBzdHJpY3QnO1xuXG52YXIgR2V0SW50cmluc2ljID0gcmVxdWlyZSgnZ2V0LWludHJpbnNpYycpO1xudmFyIGNhbGxCb3VuZCA9IHJlcXVpcmUoJ2NhbGwtYmluZC9jYWxsQm91bmQnKTtcblxudmFyICRUeXBlRXJyb3IgPSBHZXRJbnRyaW5zaWMoJyVUeXBlRXJyb3IlJyk7XG5cbnZhciBJc0FycmF5ID0gcmVxdWlyZSgnLi9Jc0FycmF5Jyk7XG5cbnZhciAkYXBwbHkgPSBHZXRJbnRyaW5zaWMoJyVSZWZsZWN0LmFwcGx5JScsIHRydWUpIHx8IGNhbGxCb3VuZCgnJUZ1bmN0aW9uLnByb3RvdHlwZS5hcHBseSUnKTtcblxuLy8gaHR0cHM6Ly9lY21hLWludGVybmF0aW9uYWwub3JnL2VjbWEtMjYyLzYuMC8jc2VjLWNhbGxcblxubW9kdWxlLmV4cG9ydHMgPSBmdW5jdGlvbiBDYWxsKEYsIFYpIHtcblx0dmFyIGFyZ3VtZW50c0xpc3QgPSBhcmd1bWVudHMubGVuZ3RoID4gMiA/IGFyZ3VtZW50c1syXSA6IFtdO1xuXHRpZiAoIUlzQXJyYXkoYXJndW1lbnRzTGlzdCkpIHtcblx0XHR0aHJvdyBuZXcgJFR5cGVFcnJvcignQXNzZXJ0aW9uIGZhaWxlZDogb3B0aW9uYWwgYGFyZ3VtZW50c0xpc3RgLCBpZiBwcm92aWRlZCwgbXVzdCBiZSBhIExpc3QnKTtcblx0fVxuXHRyZXR1cm4gJGFwcGx5KEYsIFYsIGFyZ3VtZW50c0xpc3QpO1xufTtcbiJdLCJzb3VyY2VSb290IjoiIn0=\n//# sourceURL=webpack-internal:///7800\n")},9044:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\n\nvar $TypeError = GetIntrinsic('%TypeError%');\nvar callBound = __webpack_require__(1924);\nvar isLeadingSurrogate = __webpack_require__(1996);\nvar isTrailingSurrogate = __webpack_require__(6935);\n\nvar Type = __webpack_require__(1915);\nvar UTF16DecodeSurrogatePair = __webpack_require__(4969);\n\nvar $charAt = callBound('String.prototype.charAt');\nvar $charCodeAt = callBound('String.prototype.charCodeAt');\n\n// https://262.ecma-international.org/11.0/#sec-codepointat\n\nmodule.exports = function CodePointAt(string, position) {\n\tif (Type(string) !== 'String') {\n\t\tthrow new $TypeError('Assertion failed: `string` must be a String');\n\t}\n\tvar size = string.length;\n\tif (position < 0 || position >= size) {\n\t\tthrow new $TypeError('Assertion failed: `position` must be >= 0, and < the length of `string`');\n\t}\n\tvar first = $charCodeAt(string, position);\n\tvar cp = $charAt(string, position);\n\tvar firstIsLeading = isLeadingSurrogate(first);\n\tvar firstIsTrailing = isTrailingSurrogate(first);\n\tif (!firstIsLeading && !firstIsTrailing) {\n\t\treturn {\n\t\t\t'[[CodePoint]]': cp,\n\t\t\t'[[CodeUnitCount]]': 1,\n\t\t\t'[[IsUnpairedSurrogate]]': false\n\t\t};\n\t}\n\tif (firstIsTrailing || (position + 1 === size)) {\n\t\treturn {\n\t\t\t'[[CodePoint]]': cp,\n\t\t\t'[[CodeUnitCount]]': 1,\n\t\t\t'[[IsUnpairedSurrogate]]': true\n\t\t};\n\t}\n\tvar second = $charCodeAt(string, position + 1);\n\tif (!isTrailingSurrogate(second)) {\n\t\treturn {\n\t\t\t'[[CodePoint]]': cp,\n\t\t\t'[[CodeUnitCount]]': 1,\n\t\t\t'[[IsUnpairedSurrogate]]': true\n\t\t};\n\t}\n\n\treturn {\n\t\t'[[CodePoint]]': UTF16DecodeSurrogatePair(first, second),\n\t\t'[[CodeUnitCount]]': 2,\n\t\t'[[IsUnpairedSurrogate]]': false\n\t};\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///9044\n")},3674:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\n\nvar $TypeError = GetIntrinsic('%TypeError%');\n\nvar Type = __webpack_require__(1915);\n\n// https://ecma-international.org/ecma-262/6.0/#sec-createiterresultobject\n\nmodule.exports = function CreateIterResultObject(value, done) {\n\tif (Type(done) !== 'Boolean') {\n\t\tthrow new $TypeError('Assertion failed: Type(done) is not Boolean');\n\t}\n\treturn {\n\t\tvalue: value,\n\t\tdone: done\n\t};\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvQ3JlYXRlSXRlclJlc3VsdE9iamVjdC5qcz9lYTY3Il0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFhOztBQUViLG1CQUFtQixtQkFBTyxDQUFDLEdBQWU7O0FBRTFDOztBQUVBLFdBQVcsbUJBQU8sQ0FBQyxJQUFROztBQUUzQjs7QUFFQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0EiLCJmaWxlIjoiMzY3NC5qcyIsInNvdXJjZXNDb250ZW50IjpbIid1c2Ugc3RyaWN0JztcblxudmFyIEdldEludHJpbnNpYyA9IHJlcXVpcmUoJ2dldC1pbnRyaW5zaWMnKTtcblxudmFyICRUeXBlRXJyb3IgPSBHZXRJbnRyaW5zaWMoJyVUeXBlRXJyb3IlJyk7XG5cbnZhciBUeXBlID0gcmVxdWlyZSgnLi9UeXBlJyk7XG5cbi8vIGh0dHBzOi8vZWNtYS1pbnRlcm5hdGlvbmFsLm9yZy9lY21hLTI2Mi82LjAvI3NlYy1jcmVhdGVpdGVycmVzdWx0b2JqZWN0XG5cbm1vZHVsZS5leHBvcnRzID0gZnVuY3Rpb24gQ3JlYXRlSXRlclJlc3VsdE9iamVjdCh2YWx1ZSwgZG9uZSkge1xuXHRpZiAoVHlwZShkb25lKSAhPT0gJ0Jvb2xlYW4nKSB7XG5cdFx0dGhyb3cgbmV3ICRUeXBlRXJyb3IoJ0Fzc2VydGlvbiBmYWlsZWQ6IFR5cGUoZG9uZSkgaXMgbm90IEJvb2xlYW4nKTtcblx0fVxuXHRyZXR1cm4ge1xuXHRcdHZhbHVlOiB2YWx1ZSxcblx0XHRkb25lOiBkb25lXG5cdH07XG59O1xuIl0sInNvdXJjZVJvb3QiOiIifQ==\n//# sourceURL=webpack-internal:///3674\n")},7942:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\n\nvar $TypeError = GetIntrinsic('%TypeError%');\n\nvar isPropertyDescriptor = __webpack_require__(9146);\nvar DefineOwnProperty = __webpack_require__(509);\n\nvar FromPropertyDescriptor = __webpack_require__(1922);\nvar IsAccessorDescriptor = __webpack_require__(1876);\nvar IsDataDescriptor = __webpack_require__(4307);\nvar IsPropertyKey = __webpack_require__(1388);\nvar SameValue = __webpack_require__(1221);\nvar ToPropertyDescriptor = __webpack_require__(2449);\nvar Type = __webpack_require__(1915);\n\n// https://ecma-international.org/ecma-262/6.0/#sec-definepropertyorthrow\n\nmodule.exports = function DefinePropertyOrThrow(O, P, desc) {\n\tif (Type(O) !== 'Object') {\n\t\tthrow new $TypeError('Assertion failed: Type(O) is not Object');\n\t}\n\n\tif (!IsPropertyKey(P)) {\n\t\tthrow new $TypeError('Assertion failed: IsPropertyKey(P) is not true');\n\t}\n\n\tvar Desc = isPropertyDescriptor({\n\t\tType: Type,\n\t\tIsDataDescriptor: IsDataDescriptor,\n\t\tIsAccessorDescriptor: IsAccessorDescriptor\n\t}, desc) ? desc : ToPropertyDescriptor(desc);\n\tif (!isPropertyDescriptor({\n\t\tType: Type,\n\t\tIsDataDescriptor: IsDataDescriptor,\n\t\tIsAccessorDescriptor: IsAccessorDescriptor\n\t}, Desc)) {\n\t\tthrow new $TypeError('Assertion failed: Desc is not a valid Property Descriptor');\n\t}\n\n\treturn DefineOwnProperty(\n\t\tIsDataDescriptor,\n\t\tSameValue,\n\t\tFromPropertyDescriptor,\n\t\tO,\n\t\tP,\n\t\tDesc\n\t);\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///7942\n")},1922:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar assertRecord = __webpack_require__(4179);\n\nvar Type = __webpack_require__(1915);\n\n// https://ecma-international.org/ecma-262/6.0/#sec-frompropertydescriptor\n\nmodule.exports = function FromPropertyDescriptor(Desc) {\n\tif (typeof Desc === 'undefined') {\n\t\treturn Desc;\n\t}\n\n\tassertRecord(Type, 'Property Descriptor', 'Desc', Desc);\n\n\tvar obj = {};\n\tif ('[[Value]]' in Desc) {\n\t\tobj.value = Desc['[[Value]]'];\n\t}\n\tif ('[[Writable]]' in Desc) {\n\t\tobj.writable = Desc['[[Writable]]'];\n\t}\n\tif ('[[Get]]' in Desc) {\n\t\tobj.get = Desc['[[Get]]'];\n\t}\n\tif ('[[Set]]' in Desc) {\n\t\tobj.set = Desc['[[Set]]'];\n\t}\n\tif ('[[Enumerable]]' in Desc) {\n\t\tobj.enumerable = Desc['[[Enumerable]]'];\n\t}\n\tif ('[[Configurable]]' in Desc) {\n\t\tobj.configurable = Desc['[[Configurable]]'];\n\t}\n\treturn obj;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvRnJvbVByb3BlcnR5RGVzY3JpcHRvci5qcz80OWYwIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFhOztBQUViLG1CQUFtQixtQkFBTyxDQUFDLElBQXlCOztBQUVwRCxXQUFXLG1CQUFPLENBQUMsSUFBUTs7QUFFM0I7O0FBRUE7QUFDQTtBQUNBO0FBQ0E7O0FBRUE7O0FBRUE7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBIiwiZmlsZSI6IjE5MjIuanMiLCJzb3VyY2VzQ29udGVudCI6WyIndXNlIHN0cmljdCc7XG5cbnZhciBhc3NlcnRSZWNvcmQgPSByZXF1aXJlKCcuLi9oZWxwZXJzL2Fzc2VydFJlY29yZCcpO1xuXG52YXIgVHlwZSA9IHJlcXVpcmUoJy4vVHlwZScpO1xuXG4vLyBodHRwczovL2VjbWEtaW50ZXJuYXRpb25hbC5vcmcvZWNtYS0yNjIvNi4wLyNzZWMtZnJvbXByb3BlcnR5ZGVzY3JpcHRvclxuXG5tb2R1bGUuZXhwb3J0cyA9IGZ1bmN0aW9uIEZyb21Qcm9wZXJ0eURlc2NyaXB0b3IoRGVzYykge1xuXHRpZiAodHlwZW9mIERlc2MgPT09ICd1bmRlZmluZWQnKSB7XG5cdFx0cmV0dXJuIERlc2M7XG5cdH1cblxuXHRhc3NlcnRSZWNvcmQoVHlwZSwgJ1Byb3BlcnR5IERlc2NyaXB0b3InLCAnRGVzYycsIERlc2MpO1xuXG5cdHZhciBvYmogPSB7fTtcblx0aWYgKCdbW1ZhbHVlXV0nIGluIERlc2MpIHtcblx0XHRvYmoudmFsdWUgPSBEZXNjWydbW1ZhbHVlXV0nXTtcblx0fVxuXHRpZiAoJ1tbV3JpdGFibGVdXScgaW4gRGVzYykge1xuXHRcdG9iai53cml0YWJsZSA9IERlc2NbJ1tbV3JpdGFibGVdXSddO1xuXHR9XG5cdGlmICgnW1tHZXRdXScgaW4gRGVzYykge1xuXHRcdG9iai5nZXQgPSBEZXNjWydbW0dldF1dJ107XG5cdH1cblx0aWYgKCdbW1NldF1dJyBpbiBEZXNjKSB7XG5cdFx0b2JqLnNldCA9IERlc2NbJ1tbU2V0XV0nXTtcblx0fVxuXHRpZiAoJ1tbRW51bWVyYWJsZV1dJyBpbiBEZXNjKSB7XG5cdFx0b2JqLmVudW1lcmFibGUgPSBEZXNjWydbW0VudW1lcmFibGVdXSddO1xuXHR9XG5cdGlmICgnW1tDb25maWd1cmFibGVdXScgaW4gRGVzYykge1xuXHRcdG9iai5jb25maWd1cmFibGUgPSBEZXNjWydbW0NvbmZpZ3VyYWJsZV1dJ107XG5cdH1cblx0cmV0dXJuIG9iajtcbn07XG4iXSwic291cmNlUm9vdCI6IiJ9\n//# sourceURL=webpack-internal:///1922\n")},5588:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\n\nvar $TypeError = GetIntrinsic('%TypeError%');\n\nvar inspect = __webpack_require__(631);\n\nvar IsPropertyKey = __webpack_require__(1388);\nvar Type = __webpack_require__(1915);\n\n/**\n * 7.3.1 Get (O, P) - https://ecma-international.org/ecma-262/6.0/#sec-get-o-p\n * 1. Assert: Type(O) is Object.\n * 2. Assert: IsPropertyKey(P) is true.\n * 3. Return O.[[Get]](P, O).\n */\n\nmodule.exports = function Get(O, P) {\n\t// 7.3.1.1\n\tif (Type(O) !== 'Object') {\n\t\tthrow new $TypeError('Assertion failed: Type(O) is not Object');\n\t}\n\t// 7.3.1.2\n\tif (!IsPropertyKey(P)) {\n\t\tthrow new $TypeError('Assertion failed: IsPropertyKey(P) is not true, got ' + inspect(P));\n\t}\n\t// 7.3.1.3\n\treturn O[P];\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvR2V0LmpzPzdjZTMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQWE7O0FBRWIsbUJBQW1CLG1CQUFPLENBQUMsR0FBZTs7QUFFMUM7O0FBRUEsY0FBYyxtQkFBTyxDQUFDLEdBQWdCOztBQUV0QyxvQkFBb0IsbUJBQU8sQ0FBQyxJQUFpQjtBQUM3QyxXQUFXLG1CQUFPLENBQUMsSUFBUTs7QUFFM0I7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBOztBQUVBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQSIsImZpbGUiOiI1NTg4LmpzIiwic291cmNlc0NvbnRlbnQiOlsiJ3VzZSBzdHJpY3QnO1xuXG52YXIgR2V0SW50cmluc2ljID0gcmVxdWlyZSgnZ2V0LWludHJpbnNpYycpO1xuXG52YXIgJFR5cGVFcnJvciA9IEdldEludHJpbnNpYygnJVR5cGVFcnJvciUnKTtcblxudmFyIGluc3BlY3QgPSByZXF1aXJlKCdvYmplY3QtaW5zcGVjdCcpO1xuXG52YXIgSXNQcm9wZXJ0eUtleSA9IHJlcXVpcmUoJy4vSXNQcm9wZXJ0eUtleScpO1xudmFyIFR5cGUgPSByZXF1aXJlKCcuL1R5cGUnKTtcblxuLyoqXG4gKiA3LjMuMSBHZXQgKE8sIFApIC0gaHR0cHM6Ly9lY21hLWludGVybmF0aW9uYWwub3JnL2VjbWEtMjYyLzYuMC8jc2VjLWdldC1vLXBcbiAqIDEuIEFzc2VydDogVHlwZShPKSBpcyBPYmplY3QuXG4gKiAyLiBBc3NlcnQ6IElzUHJvcGVydHlLZXkoUCkgaXMgdHJ1ZS5cbiAqIDMuIFJldHVybiBPLltbR2V0XV0oUCwgTykuXG4gKi9cblxubW9kdWxlLmV4cG9ydHMgPSBmdW5jdGlvbiBHZXQoTywgUCkge1xuXHQvLyA3LjMuMS4xXG5cdGlmIChUeXBlKE8pICE9PSAnT2JqZWN0Jykge1xuXHRcdHRocm93IG5ldyAkVHlwZUVycm9yKCdBc3NlcnRpb24gZmFpbGVkOiBUeXBlKE8pIGlzIG5vdCBPYmplY3QnKTtcblx0fVxuXHQvLyA3LjMuMS4yXG5cdGlmICghSXNQcm9wZXJ0eUtleShQKSkge1xuXHRcdHRocm93IG5ldyAkVHlwZUVycm9yKCdBc3NlcnRpb24gZmFpbGVkOiBJc1Byb3BlcnR5S2V5KFApIGlzIG5vdCB0cnVlLCBnb3QgJyArIGluc3BlY3QoUCkpO1xuXHR9XG5cdC8vIDcuMy4xLjNcblx0cmV0dXJuIE9bUF07XG59O1xuIl0sInNvdXJjZVJvb3QiOiIifQ==\n//# sourceURL=webpack-internal:///5588\n")},3081:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\n\nvar $TypeError = GetIntrinsic('%TypeError%');\n\nvar GetV = __webpack_require__(5423);\nvar IsCallable = __webpack_require__(590);\nvar IsPropertyKey = __webpack_require__(1388);\n\n/**\n * 7.3.9 - https://ecma-international.org/ecma-262/6.0/#sec-getmethod\n * 1. Assert: IsPropertyKey(P) is true.\n * 2. Let func be GetV(O, P).\n * 3. ReturnIfAbrupt(func).\n * 4. If func is either undefined or null, return undefined.\n * 5. If IsCallable(func) is false, throw a TypeError exception.\n * 6. Return func.\n */\n\nmodule.exports = function GetMethod(O, P) {\n\t// 7.3.9.1\n\tif (!IsPropertyKey(P)) {\n\t\tthrow new $TypeError('Assertion failed: IsPropertyKey(P) is not true');\n\t}\n\n\t// 7.3.9.2\n\tvar func = GetV(O, P);\n\n\t// 7.3.9.4\n\tif (func == null) {\n\t\treturn void 0;\n\t}\n\n\t// 7.3.9.5\n\tif (!IsCallable(func)) {\n\t\tthrow new $TypeError(P + 'is not a function');\n\t}\n\n\t// 7.3.9.6\n\treturn func;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvR2V0TWV0aG9kLmpzPzRmYTUiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQWE7O0FBRWIsbUJBQW1CLG1CQUFPLENBQUMsR0FBZTs7QUFFMUM7O0FBRUEsV0FBVyxtQkFBTyxDQUFDLElBQVE7QUFDM0IsaUJBQWlCLG1CQUFPLENBQUMsR0FBYztBQUN2QyxvQkFBb0IsbUJBQU8sQ0FBQyxJQUFpQjs7QUFFN0M7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBOztBQUVBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7O0FBRUE7QUFDQTs7QUFFQTtBQUNBO0FBQ0E7QUFDQTs7QUFFQTtBQUNBO0FBQ0E7QUFDQTs7QUFFQTtBQUNBO0FBQ0EiLCJmaWxlIjoiMzA4MS5qcyIsInNvdXJjZXNDb250ZW50IjpbIid1c2Ugc3RyaWN0JztcblxudmFyIEdldEludHJpbnNpYyA9IHJlcXVpcmUoJ2dldC1pbnRyaW5zaWMnKTtcblxudmFyICRUeXBlRXJyb3IgPSBHZXRJbnRyaW5zaWMoJyVUeXBlRXJyb3IlJyk7XG5cbnZhciBHZXRWID0gcmVxdWlyZSgnLi9HZXRWJyk7XG52YXIgSXNDYWxsYWJsZSA9IHJlcXVpcmUoJy4vSXNDYWxsYWJsZScpO1xudmFyIElzUHJvcGVydHlLZXkgPSByZXF1aXJlKCcuL0lzUHJvcGVydHlLZXknKTtcblxuLyoqXG4gKiA3LjMuOSAtIGh0dHBzOi8vZWNtYS1pbnRlcm5hdGlvbmFsLm9yZy9lY21hLTI2Mi82LjAvI3NlYy1nZXRtZXRob2RcbiAqIDEuIEFzc2VydDogSXNQcm9wZXJ0eUtleShQKSBpcyB0cnVlLlxuICogMi4gTGV0IGZ1bmMgYmUgR2V0VihPLCBQKS5cbiAqIDMuIFJldHVybklmQWJydXB0KGZ1bmMpLlxuICogNC4gSWYgZnVuYyBpcyBlaXRoZXIgdW5kZWZpbmVkIG9yIG51bGwsIHJldHVybiB1bmRlZmluZWQuXG4gKiA1LiBJZiBJc0NhbGxhYmxlKGZ1bmMpIGlzIGZhbHNlLCB0aHJvdyBhIFR5cGVFcnJvciBleGNlcHRpb24uXG4gKiA2LiBSZXR1cm4gZnVuYy5cbiAqL1xuXG5tb2R1bGUuZXhwb3J0cyA9IGZ1bmN0aW9uIEdldE1ldGhvZChPLCBQKSB7XG5cdC8vIDcuMy45LjFcblx0aWYgKCFJc1Byb3BlcnR5S2V5KFApKSB7XG5cdFx0dGhyb3cgbmV3ICRUeXBlRXJyb3IoJ0Fzc2VydGlvbiBmYWlsZWQ6IElzUHJvcGVydHlLZXkoUCkgaXMgbm90IHRydWUnKTtcblx0fVxuXG5cdC8vIDcuMy45LjJcblx0dmFyIGZ1bmMgPSBHZXRWKE8sIFApO1xuXG5cdC8vIDcuMy45LjRcblx0aWYgKGZ1bmMgPT0gbnVsbCkge1xuXHRcdHJldHVybiB2b2lkIDA7XG5cdH1cblxuXHQvLyA3LjMuOS41XG5cdGlmICghSXNDYWxsYWJsZShmdW5jKSkge1xuXHRcdHRocm93IG5ldyAkVHlwZUVycm9yKFAgKyAnaXMgbm90IGEgZnVuY3Rpb24nKTtcblx0fVxuXG5cdC8vIDcuMy45LjZcblx0cmV0dXJuIGZ1bmM7XG59O1xuIl0sInNvdXJjZVJvb3QiOiIifQ==\n//# sourceURL=webpack-internal:///3081\n")},5423:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\n\nvar $TypeError = GetIntrinsic('%TypeError%');\n\nvar IsPropertyKey = __webpack_require__(1388);\nvar ToObject = __webpack_require__(2093);\n\n/**\n * 7.3.2 GetV (V, P)\n * 1. Assert: IsPropertyKey(P) is true.\n * 2. Let O be ToObject(V).\n * 3. ReturnIfAbrupt(O).\n * 4. Return O.[[Get]](P, V).\n */\n\nmodule.exports = function GetV(V, P) {\n\t// 7.3.2.1\n\tif (!IsPropertyKey(P)) {\n\t\tthrow new $TypeError('Assertion failed: IsPropertyKey(P) is not true');\n\t}\n\n\t// 7.3.2.2-3\n\tvar O = ToObject(V);\n\n\t// 7.3.2.4\n\treturn O[P];\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvR2V0Vi5qcz82Y2E0Il0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFhOztBQUViLG1CQUFtQixtQkFBTyxDQUFDLEdBQWU7O0FBRTFDOztBQUVBLG9CQUFvQixtQkFBTyxDQUFDLElBQWlCO0FBQzdDLGVBQWUsbUJBQU8sQ0FBQyxJQUFZOztBQUVuQztBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTs7QUFFQTtBQUNBO0FBQ0E7QUFDQTtBQUNBOztBQUVBO0FBQ0E7O0FBRUE7QUFDQTtBQUNBIiwiZmlsZSI6IjU0MjMuanMiLCJzb3VyY2VzQ29udGVudCI6WyIndXNlIHN0cmljdCc7XG5cbnZhciBHZXRJbnRyaW5zaWMgPSByZXF1aXJlKCdnZXQtaW50cmluc2ljJyk7XG5cbnZhciAkVHlwZUVycm9yID0gR2V0SW50cmluc2ljKCclVHlwZUVycm9yJScpO1xuXG52YXIgSXNQcm9wZXJ0eUtleSA9IHJlcXVpcmUoJy4vSXNQcm9wZXJ0eUtleScpO1xudmFyIFRvT2JqZWN0ID0gcmVxdWlyZSgnLi9Ub09iamVjdCcpO1xuXG4vKipcbiAqIDcuMy4yIEdldFYgKFYsIFApXG4gKiAxLiBBc3NlcnQ6IElzUHJvcGVydHlLZXkoUCkgaXMgdHJ1ZS5cbiAqIDIuIExldCBPIGJlIFRvT2JqZWN0KFYpLlxuICogMy4gUmV0dXJuSWZBYnJ1cHQoTykuXG4gKiA0LiBSZXR1cm4gTy5bW0dldF1dKFAsIFYpLlxuICovXG5cbm1vZHVsZS5leHBvcnRzID0gZnVuY3Rpb24gR2V0VihWLCBQKSB7XG5cdC8vIDcuMy4yLjFcblx0aWYgKCFJc1Byb3BlcnR5S2V5KFApKSB7XG5cdFx0dGhyb3cgbmV3ICRUeXBlRXJyb3IoJ0Fzc2VydGlvbiBmYWlsZWQ6IElzUHJvcGVydHlLZXkoUCkgaXMgbm90IHRydWUnKTtcblx0fVxuXG5cdC8vIDcuMy4yLjItM1xuXHR2YXIgTyA9IFRvT2JqZWN0KFYpO1xuXG5cdC8vIDcuMy4yLjRcblx0cmV0dXJuIE9bUF07XG59O1xuIl0sInNvdXJjZVJvb3QiOiIifQ==\n//# sourceURL=webpack-internal:///5423\n")},1876:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar has = __webpack_require__(7642);\n\nvar assertRecord = __webpack_require__(4179);\n\nvar Type = __webpack_require__(1915);\n\n// https://ecma-international.org/ecma-262/6.0/#sec-isaccessordescriptor\n\nmodule.exports = function IsAccessorDescriptor(Desc) {\n\tif (typeof Desc === 'undefined') {\n\t\treturn false;\n\t}\n\n\tassertRecord(Type, 'Property Descriptor', 'Desc', Desc);\n\n\tif (!has(Desc, '[[Get]]') && !has(Desc, '[[Set]]')) {\n\t\treturn false;\n\t}\n\n\treturn true;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvSXNBY2Nlc3NvckRlc2NyaXB0b3IuanM/NjEzZCJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBYTs7QUFFYixVQUFVLG1CQUFPLENBQUMsSUFBSzs7QUFFdkIsbUJBQW1CLG1CQUFPLENBQUMsSUFBeUI7O0FBRXBELFdBQVcsbUJBQU8sQ0FBQyxJQUFROztBQUUzQjs7QUFFQTtBQUNBO0FBQ0E7QUFDQTs7QUFFQTs7QUFFQTtBQUNBO0FBQ0E7O0FBRUE7QUFDQSIsImZpbGUiOiIxODc2LmpzIiwic291cmNlc0NvbnRlbnQiOlsiJ3VzZSBzdHJpY3QnO1xuXG52YXIgaGFzID0gcmVxdWlyZSgnaGFzJyk7XG5cbnZhciBhc3NlcnRSZWNvcmQgPSByZXF1aXJlKCcuLi9oZWxwZXJzL2Fzc2VydFJlY29yZCcpO1xuXG52YXIgVHlwZSA9IHJlcXVpcmUoJy4vVHlwZScpO1xuXG4vLyBodHRwczovL2VjbWEtaW50ZXJuYXRpb25hbC5vcmcvZWNtYS0yNjIvNi4wLyNzZWMtaXNhY2Nlc3NvcmRlc2NyaXB0b3JcblxubW9kdWxlLmV4cG9ydHMgPSBmdW5jdGlvbiBJc0FjY2Vzc29yRGVzY3JpcHRvcihEZXNjKSB7XG5cdGlmICh0eXBlb2YgRGVzYyA9PT0gJ3VuZGVmaW5lZCcpIHtcblx0XHRyZXR1cm4gZmFsc2U7XG5cdH1cblxuXHRhc3NlcnRSZWNvcmQoVHlwZSwgJ1Byb3BlcnR5IERlc2NyaXB0b3InLCAnRGVzYycsIERlc2MpO1xuXG5cdGlmICghaGFzKERlc2MsICdbW0dldF1dJykgJiYgIWhhcyhEZXNjLCAnW1tTZXRdXScpKSB7XG5cdFx0cmV0dXJuIGZhbHNlO1xuXHR9XG5cblx0cmV0dXJuIHRydWU7XG59O1xuIl0sInNvdXJjZVJvb3QiOiIifQ==\n//# sourceURL=webpack-internal:///1876\n")},4400:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\n\nvar $Array = GetIntrinsic('%Array%');\n\n// eslint-disable-next-line global-require\nvar toStr = !$Array.isArray && __webpack_require__(1924)('Object.prototype.toString');\n\n// https://ecma-international.org/ecma-262/6.0/#sec-isarray\n\nmodule.exports = $Array.isArray || function IsArray(argument) {\n\treturn toStr(argument) === '[object Array]';\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvSXNBcnJheS5qcz8yN2IxIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFhOztBQUViLG1CQUFtQixtQkFBTyxDQUFDLEdBQWU7O0FBRTFDOztBQUVBO0FBQ0EsK0JBQStCLG1CQUFPLENBQUMsSUFBcUI7O0FBRTVEOztBQUVBO0FBQ0E7QUFDQSIsImZpbGUiOiI0NDAwLmpzIiwic291cmNlc0NvbnRlbnQiOlsiJ3VzZSBzdHJpY3QnO1xuXG52YXIgR2V0SW50cmluc2ljID0gcmVxdWlyZSgnZ2V0LWludHJpbnNpYycpO1xuXG52YXIgJEFycmF5ID0gR2V0SW50cmluc2ljKCclQXJyYXklJyk7XG5cbi8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBnbG9iYWwtcmVxdWlyZVxudmFyIHRvU3RyID0gISRBcnJheS5pc0FycmF5ICYmIHJlcXVpcmUoJ2NhbGwtYmluZC9jYWxsQm91bmQnKSgnT2JqZWN0LnByb3RvdHlwZS50b1N0cmluZycpO1xuXG4vLyBodHRwczovL2VjbWEtaW50ZXJuYXRpb25hbC5vcmcvZWNtYS0yNjIvNi4wLyNzZWMtaXNhcnJheVxuXG5tb2R1bGUuZXhwb3J0cyA9ICRBcnJheS5pc0FycmF5IHx8IGZ1bmN0aW9uIElzQXJyYXkoYXJndW1lbnQpIHtcblx0cmV0dXJuIHRvU3RyKGFyZ3VtZW50KSA9PT0gJ1tvYmplY3QgQXJyYXldJztcbn07XG4iXSwic291cmNlUm9vdCI6IiJ9\n//# sourceURL=webpack-internal:///4400\n")},590:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\n// http://262.ecma-international.org/5.1/#sec-9.11\n\nmodule.exports = __webpack_require__(5320);\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvSXNDYWxsYWJsZS5qcz8xNjI5Il0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFhOztBQUViOztBQUVBLDBDQUF1QyIsImZpbGUiOiI1OTAuanMiLCJzb3VyY2VzQ29udGVudCI6WyIndXNlIHN0cmljdCc7XG5cbi8vIGh0dHA6Ly8yNjIuZWNtYS1pbnRlcm5hdGlvbmFsLm9yZy81LjEvI3NlYy05LjExXG5cbm1vZHVsZS5leHBvcnRzID0gcmVxdWlyZSgnaXMtY2FsbGFibGUnKTtcbiJdLCJzb3VyY2VSb290IjoiIn0=\n//# sourceURL=webpack-internal:///590\n")},970:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(8769);\n\nvar $construct = GetIntrinsic('%Reflect.construct%', true);\n\nvar DefinePropertyOrThrow = __webpack_require__(7942);\ntry {\n\tDefinePropertyOrThrow({}, '', { '[[Get]]': function () {} });\n} catch (e) {\n\t// Accessor properties aren't supported\n\tDefinePropertyOrThrow = null;\n}\n\n// https://ecma-international.org/ecma-262/6.0/#sec-isconstructor\n\nif (DefinePropertyOrThrow && $construct) {\n\tvar isConstructorMarker = {};\n\tvar badArrayLike = {};\n\tDefinePropertyOrThrow(badArrayLike, 'length', {\n\t\t'[[Get]]': function () {\n\t\t\tthrow isConstructorMarker;\n\t\t},\n\t\t'[[Enumerable]]': true\n\t});\n\n\tmodule.exports = function IsConstructor(argument) {\n\t\ttry {\n\t\t\t// `Reflect.construct` invokes `IsConstructor(target)` before `Get(args, 'length')`:\n\t\t\t$construct(argument, badArrayLike);\n\t\t} catch (err) {\n\t\t\treturn err === isConstructorMarker;\n\t\t}\n\t};\n} else {\n\tmodule.exports = function IsConstructor(argument) {\n\t\t// unfortunately there's no way to truly check this without try/catch `new argument` in old environments\n\t\treturn typeof argument === 'function' && !!argument.prototype;\n\t};\n}\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///970\n")},4307:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar has = __webpack_require__(7642);\n\nvar assertRecord = __webpack_require__(4179);\n\nvar Type = __webpack_require__(1915);\n\n// https://ecma-international.org/ecma-262/6.0/#sec-isdatadescriptor\n\nmodule.exports = function IsDataDescriptor(Desc) {\n\tif (typeof Desc === 'undefined') {\n\t\treturn false;\n\t}\n\n\tassertRecord(Type, 'Property Descriptor', 'Desc', Desc);\n\n\tif (!has(Desc, '[[Value]]') && !has(Desc, '[[Writable]]')) {\n\t\treturn false;\n\t}\n\n\treturn true;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvSXNEYXRhRGVzY3JpcHRvci5qcz8wNDUyIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFhOztBQUViLFVBQVUsbUJBQU8sQ0FBQyxJQUFLOztBQUV2QixtQkFBbUIsbUJBQU8sQ0FBQyxJQUF5Qjs7QUFFcEQsV0FBVyxtQkFBTyxDQUFDLElBQVE7O0FBRTNCOztBQUVBO0FBQ0E7QUFDQTtBQUNBOztBQUVBOztBQUVBO0FBQ0E7QUFDQTs7QUFFQTtBQUNBIiwiZmlsZSI6IjQzMDcuanMiLCJzb3VyY2VzQ29udGVudCI6WyIndXNlIHN0cmljdCc7XG5cbnZhciBoYXMgPSByZXF1aXJlKCdoYXMnKTtcblxudmFyIGFzc2VydFJlY29yZCA9IHJlcXVpcmUoJy4uL2hlbHBlcnMvYXNzZXJ0UmVjb3JkJyk7XG5cbnZhciBUeXBlID0gcmVxdWlyZSgnLi9UeXBlJyk7XG5cbi8vIGh0dHBzOi8vZWNtYS1pbnRlcm5hdGlvbmFsLm9yZy9lY21hLTI2Mi82LjAvI3NlYy1pc2RhdGFkZXNjcmlwdG9yXG5cbm1vZHVsZS5leHBvcnRzID0gZnVuY3Rpb24gSXNEYXRhRGVzY3JpcHRvcihEZXNjKSB7XG5cdGlmICh0eXBlb2YgRGVzYyA9PT0gJ3VuZGVmaW5lZCcpIHtcblx0XHRyZXR1cm4gZmFsc2U7XG5cdH1cblxuXHRhc3NlcnRSZWNvcmQoVHlwZSwgJ1Byb3BlcnR5IERlc2NyaXB0b3InLCAnRGVzYycsIERlc2MpO1xuXG5cdGlmICghaGFzKERlc2MsICdbW1ZhbHVlXV0nKSAmJiAhaGFzKERlc2MsICdbW1dyaXRhYmxlXV0nKSkge1xuXHRcdHJldHVybiBmYWxzZTtcblx0fVxuXG5cdHJldHVybiB0cnVlO1xufTtcbiJdLCJzb3VyY2VSb290IjoiIn0=\n//# sourceURL=webpack-internal:///4307\n")},7868:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar abs = __webpack_require__(1717);\nvar floor = __webpack_require__(9202);\n\nvar $isNaN = __webpack_require__(1214);\nvar $isFinite = __webpack_require__(3060);\n\n// https://ecma-international.org/ecma-262/6.0/#sec-isinteger\n\nmodule.exports = function IsInteger(argument) {\n\tif (typeof argument !== 'number' || $isNaN(argument) || !$isFinite(argument)) {\n\t\treturn false;\n\t}\n\tvar absValue = abs(argument);\n\treturn floor(absValue) === absValue;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvSXNJbnRlZ2VyLmpzPzkyNzciXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQWE7O0FBRWIsVUFBVSxtQkFBTyxDQUFDLElBQU87QUFDekIsWUFBWSxtQkFBTyxDQUFDLElBQVM7O0FBRTdCLGFBQWEsbUJBQU8sQ0FBQyxJQUFrQjtBQUN2QyxnQkFBZ0IsbUJBQU8sQ0FBQyxJQUFxQjs7QUFFN0M7O0FBRUE7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0EiLCJmaWxlIjoiNzg2OC5qcyIsInNvdXJjZXNDb250ZW50IjpbIid1c2Ugc3RyaWN0JztcblxudmFyIGFicyA9IHJlcXVpcmUoJy4vYWJzJyk7XG52YXIgZmxvb3IgPSByZXF1aXJlKCcuL2Zsb29yJyk7XG5cbnZhciAkaXNOYU4gPSByZXF1aXJlKCcuLi9oZWxwZXJzL2lzTmFOJyk7XG52YXIgJGlzRmluaXRlID0gcmVxdWlyZSgnLi4vaGVscGVycy9pc0Zpbml0ZScpO1xuXG4vLyBodHRwczovL2VjbWEtaW50ZXJuYXRpb25hbC5vcmcvZWNtYS0yNjIvNi4wLyNzZWMtaXNpbnRlZ2VyXG5cbm1vZHVsZS5leHBvcnRzID0gZnVuY3Rpb24gSXNJbnRlZ2VyKGFyZ3VtZW50KSB7XG5cdGlmICh0eXBlb2YgYXJndW1lbnQgIT09ICdudW1iZXInIHx8ICRpc05hTihhcmd1bWVudCkgfHwgISRpc0Zpbml0ZShhcmd1bWVudCkpIHtcblx0XHRyZXR1cm4gZmFsc2U7XG5cdH1cblx0dmFyIGFic1ZhbHVlID0gYWJzKGFyZ3VtZW50KTtcblx0cmV0dXJuIGZsb29yKGFic1ZhbHVlKSA9PT0gYWJzVmFsdWU7XG59O1xuIl0sInNvdXJjZVJvb3QiOiIifQ==\n//# sourceURL=webpack-internal:///7868\n")},1388:module=>{"use strict";eval("\n\n// https://ecma-international.org/ecma-262/6.0/#sec-ispropertykey\n\nmodule.exports = function IsPropertyKey(argument) {\n\treturn typeof argument === 'string' || typeof argument === 'symbol';\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvSXNQcm9wZXJ0eUtleS5qcz83YTIwIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFhOztBQUViOztBQUVBO0FBQ0E7QUFDQSIsImZpbGUiOiIxMzg4LmpzIiwic291cmNlc0NvbnRlbnQiOlsiJ3VzZSBzdHJpY3QnO1xuXG4vLyBodHRwczovL2VjbWEtaW50ZXJuYXRpb25hbC5vcmcvZWNtYS0yNjIvNi4wLyNzZWMtaXNwcm9wZXJ0eWtleVxuXG5tb2R1bGUuZXhwb3J0cyA9IGZ1bmN0aW9uIElzUHJvcGVydHlLZXkoYXJndW1lbnQpIHtcblx0cmV0dXJuIHR5cGVvZiBhcmd1bWVudCA9PT0gJ3N0cmluZycgfHwgdHlwZW9mIGFyZ3VtZW50ID09PSAnc3ltYm9sJztcbn07XG4iXSwic291cmNlUm9vdCI6IiJ9\n//# sourceURL=webpack-internal:///1388\n")},1137:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\n\nvar $match = GetIntrinsic('%Symbol.match%', true);\n\nvar hasRegExpMatcher = __webpack_require__(8420);\n\nvar ToBoolean = __webpack_require__(3683);\n\n// https://ecma-international.org/ecma-262/6.0/#sec-isregexp\n\nmodule.exports = function IsRegExp(argument) {\n\tif (!argument || typeof argument !== 'object') {\n\t\treturn false;\n\t}\n\tif ($match) {\n\t\tvar isRegExp = argument[$match];\n\t\tif (typeof isRegExp !== 'undefined') {\n\t\t\treturn ToBoolean(isRegExp);\n\t\t}\n\t}\n\treturn hasRegExpMatcher(argument);\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvSXNSZWdFeHAuanM/YjJjMCJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBYTs7QUFFYixtQkFBbUIsbUJBQU8sQ0FBQyxHQUFlOztBQUUxQzs7QUFFQSx1QkFBdUIsbUJBQU8sQ0FBQyxJQUFVOztBQUV6QyxnQkFBZ0IsbUJBQU8sQ0FBQyxJQUFhOztBQUVyQzs7QUFFQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0EiLCJmaWxlIjoiMTEzNy5qcyIsInNvdXJjZXNDb250ZW50IjpbIid1c2Ugc3RyaWN0JztcblxudmFyIEdldEludHJpbnNpYyA9IHJlcXVpcmUoJ2dldC1pbnRyaW5zaWMnKTtcblxudmFyICRtYXRjaCA9IEdldEludHJpbnNpYygnJVN5bWJvbC5tYXRjaCUnLCB0cnVlKTtcblxudmFyIGhhc1JlZ0V4cE1hdGNoZXIgPSByZXF1aXJlKCdpcy1yZWdleCcpO1xuXG52YXIgVG9Cb29sZWFuID0gcmVxdWlyZSgnLi9Ub0Jvb2xlYW4nKTtcblxuLy8gaHR0cHM6Ly9lY21hLWludGVybmF0aW9uYWwub3JnL2VjbWEtMjYyLzYuMC8jc2VjLWlzcmVnZXhwXG5cbm1vZHVsZS5leHBvcnRzID0gZnVuY3Rpb24gSXNSZWdFeHAoYXJndW1lbnQpIHtcblx0aWYgKCFhcmd1bWVudCB8fCB0eXBlb2YgYXJndW1lbnQgIT09ICdvYmplY3QnKSB7XG5cdFx0cmV0dXJuIGZhbHNlO1xuXHR9XG5cdGlmICgkbWF0Y2gpIHtcblx0XHR2YXIgaXNSZWdFeHAgPSBhcmd1bWVudFskbWF0Y2hdO1xuXHRcdGlmICh0eXBlb2YgaXNSZWdFeHAgIT09ICd1bmRlZmluZWQnKSB7XG5cdFx0XHRyZXR1cm4gVG9Cb29sZWFuKGlzUmVnRXhwKTtcblx0XHR9XG5cdH1cblx0cmV0dXJuIGhhc1JlZ0V4cE1hdGNoZXIoYXJndW1lbnQpO1xufTtcbiJdLCJzb3VyY2VSb290IjoiIn0=\n//# sourceURL=webpack-internal:///1137\n")},4091:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\n\nvar $ObjectCreate = GetIntrinsic('%Object.create%', true);\nvar $TypeError = GetIntrinsic('%TypeError%');\nvar $SyntaxError = GetIntrinsic('%SyntaxError%');\n\nvar IsArray = __webpack_require__(4400);\nvar Type = __webpack_require__(1915);\n\nvar hasProto = !({ __proto__: null } instanceof Object);\n\n// https://262.ecma-international.org/6.0/#sec-objectcreate\n\nmodule.exports = function OrdinaryObjectCreate(proto) {\n\tif (proto !== null && Type(proto) !== 'Object') {\n\t\tthrow new $TypeError('Assertion failed: `proto` must be null or an object');\n\t}\n\tvar additionalInternalSlotsList = arguments.length < 2 ? [] : arguments[1];\n\tif (!IsArray(additionalInternalSlotsList)) {\n\t\tthrow new $TypeError('Assertion failed: `additionalInternalSlotsList` must be an Array');\n\t}\n\t// var internalSlotsList = ['[[Prototype]]', '[[Extensible]]'];\n\tif (additionalInternalSlotsList.length > 0) {\n\t\tthrow new $SyntaxError('es-abstract does not yet support internal slots');\n\t\t// internalSlotsList.push(...additionalInternalSlotsList);\n\t}\n\t// var O = MakeBasicObject(internalSlotsList);\n\t// setProto(O, proto);\n\t// return O;\n\n\tif ($ObjectCreate) {\n\t\treturn $ObjectCreate(proto);\n\t}\n\tif (hasProto) {\n\t\treturn { __proto__: proto };\n\t}\n\n\tif (proto === null) {\n\t\tthrow new $SyntaxError('native Object.create support is required to create null objects');\n\t}\n\tvar T = function T() {};\n\tT.prototype = proto;\n\treturn new T();\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///4091\n")},356:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\n\nvar $TypeError = GetIntrinsic('%TypeError%');\n\nvar regexExec = __webpack_require__(1924)('RegExp.prototype.exec');\n\nvar Call = __webpack_require__(7800);\nvar Get = __webpack_require__(5588);\nvar IsCallable = __webpack_require__(590);\nvar Type = __webpack_require__(1915);\n\n// https://ecma-international.org/ecma-262/6.0/#sec-regexpexec\n\nmodule.exports = function RegExpExec(R, S) {\n\tif (Type(R) !== 'Object') {\n\t\tthrow new $TypeError('Assertion failed: `R` must be an Object');\n\t}\n\tif (Type(S) !== 'String') {\n\t\tthrow new $TypeError('Assertion failed: `S` must be a String');\n\t}\n\tvar exec = Get(R, 'exec');\n\tif (IsCallable(exec)) {\n\t\tvar result = Call(exec, R, [S]);\n\t\tif (result === null || Type(result) === 'Object') {\n\t\t\treturn result;\n\t\t}\n\t\tthrow new $TypeError('\"exec\" method must return `null` or an Object');\n\t}\n\treturn regexExec(R, S);\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvUmVnRXhwRXhlYy5qcz8zMzc2Il0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFhOztBQUViLG1CQUFtQixtQkFBTyxDQUFDLEdBQWU7O0FBRTFDOztBQUVBLGdCQUFnQixtQkFBTyxDQUFDLElBQXFCOztBQUU3QyxXQUFXLG1CQUFPLENBQUMsSUFBUTtBQUMzQixVQUFVLG1CQUFPLENBQUMsSUFBTztBQUN6QixpQkFBaUIsbUJBQU8sQ0FBQyxHQUFjO0FBQ3ZDLFdBQVcsbUJBQU8sQ0FBQyxJQUFROztBQUUzQjs7QUFFQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBIiwiZmlsZSI6IjM1Ni5qcyIsInNvdXJjZXNDb250ZW50IjpbIid1c2Ugc3RyaWN0JztcblxudmFyIEdldEludHJpbnNpYyA9IHJlcXVpcmUoJ2dldC1pbnRyaW5zaWMnKTtcblxudmFyICRUeXBlRXJyb3IgPSBHZXRJbnRyaW5zaWMoJyVUeXBlRXJyb3IlJyk7XG5cbnZhciByZWdleEV4ZWMgPSByZXF1aXJlKCdjYWxsLWJpbmQvY2FsbEJvdW5kJykoJ1JlZ0V4cC5wcm90b3R5cGUuZXhlYycpO1xuXG52YXIgQ2FsbCA9IHJlcXVpcmUoJy4vQ2FsbCcpO1xudmFyIEdldCA9IHJlcXVpcmUoJy4vR2V0Jyk7XG52YXIgSXNDYWxsYWJsZSA9IHJlcXVpcmUoJy4vSXNDYWxsYWJsZScpO1xudmFyIFR5cGUgPSByZXF1aXJlKCcuL1R5cGUnKTtcblxuLy8gaHR0cHM6Ly9lY21hLWludGVybmF0aW9uYWwub3JnL2VjbWEtMjYyLzYuMC8jc2VjLXJlZ2V4cGV4ZWNcblxubW9kdWxlLmV4cG9ydHMgPSBmdW5jdGlvbiBSZWdFeHBFeGVjKFIsIFMpIHtcblx0aWYgKFR5cGUoUikgIT09ICdPYmplY3QnKSB7XG5cdFx0dGhyb3cgbmV3ICRUeXBlRXJyb3IoJ0Fzc2VydGlvbiBmYWlsZWQ6IGBSYCBtdXN0IGJlIGFuIE9iamVjdCcpO1xuXHR9XG5cdGlmIChUeXBlKFMpICE9PSAnU3RyaW5nJykge1xuXHRcdHRocm93IG5ldyAkVHlwZUVycm9yKCdBc3NlcnRpb24gZmFpbGVkOiBgU2AgbXVzdCBiZSBhIFN0cmluZycpO1xuXHR9XG5cdHZhciBleGVjID0gR2V0KFIsICdleGVjJyk7XG5cdGlmIChJc0NhbGxhYmxlKGV4ZWMpKSB7XG5cdFx0dmFyIHJlc3VsdCA9IENhbGwoZXhlYywgUiwgW1NdKTtcblx0XHRpZiAocmVzdWx0ID09PSBudWxsIHx8IFR5cGUocmVzdWx0KSA9PT0gJ09iamVjdCcpIHtcblx0XHRcdHJldHVybiByZXN1bHQ7XG5cdFx0fVxuXHRcdHRocm93IG5ldyAkVHlwZUVycm9yKCdcImV4ZWNcIiBtZXRob2QgbXVzdCByZXR1cm4gYG51bGxgIG9yIGFuIE9iamVjdCcpO1xuXHR9XG5cdHJldHVybiByZWdleEV4ZWMoUiwgUyk7XG59O1xuIl0sInNvdXJjZVJvb3QiOiIifQ==\n//# sourceURL=webpack-internal:///356\n")},3733:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nmodule.exports = __webpack_require__(8631);\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvUmVxdWlyZU9iamVjdENvZXJjaWJsZS5qcz8wNDI3Il0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFhOztBQUViLDBDQUFxRCIsImZpbGUiOiIzNzMzLmpzIiwic291cmNlc0NvbnRlbnQiOlsiJ3VzZSBzdHJpY3QnO1xuXG5tb2R1bGUuZXhwb3J0cyA9IHJlcXVpcmUoJy4uLzUvQ2hlY2tPYmplY3RDb2VyY2libGUnKTtcbiJdLCJzb3VyY2VSb290IjoiIn0=\n//# sourceURL=webpack-internal:///3733\n")},1221:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar $isNaN = __webpack_require__(1214);\n\n// http://262.ecma-international.org/5.1/#sec-9.12\n\nmodule.exports = function SameValue(x, y) {\n\tif (x === y) { // 0 === -0, but they are not identical.\n\t\tif (x === 0) { return 1 / x === 1 / y; }\n\t\treturn true;\n\t}\n\treturn $isNaN(x) && $isNaN(y);\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvU2FtZVZhbHVlLmpzPzk3OTkiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQWE7O0FBRWIsYUFBYSxtQkFBTyxDQUFDLElBQWtCOztBQUV2Qzs7QUFFQTtBQUNBLGVBQWU7QUFDZixnQkFBZ0Isd0JBQXdCO0FBQ3hDO0FBQ0E7QUFDQTtBQUNBIiwiZmlsZSI6IjEyMjEuanMiLCJzb3VyY2VzQ29udGVudCI6WyIndXNlIHN0cmljdCc7XG5cbnZhciAkaXNOYU4gPSByZXF1aXJlKCcuLi9oZWxwZXJzL2lzTmFOJyk7XG5cbi8vIGh0dHA6Ly8yNjIuZWNtYS1pbnRlcm5hdGlvbmFsLm9yZy81LjEvI3NlYy05LjEyXG5cbm1vZHVsZS5leHBvcnRzID0gZnVuY3Rpb24gU2FtZVZhbHVlKHgsIHkpIHtcblx0aWYgKHggPT09IHkpIHsgLy8gMCA9PT0gLTAsIGJ1dCB0aGV5IGFyZSBub3QgaWRlbnRpY2FsLlxuXHRcdGlmICh4ID09PSAwKSB7IHJldHVybiAxIC8geCA9PT0gMSAvIHk7IH1cblx0XHRyZXR1cm4gdHJ1ZTtcblx0fVxuXHRyZXR1cm4gJGlzTmFOKHgpICYmICRpc05hTih5KTtcbn07XG4iXSwic291cmNlUm9vdCI6IiJ9\n//# sourceURL=webpack-internal:///1221\n")},2946:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\n\nvar $TypeError = GetIntrinsic('%TypeError%');\n\nvar IsPropertyKey = __webpack_require__(1388);\nvar SameValue = __webpack_require__(1221);\nvar Type = __webpack_require__(1915);\n\n// IE 9 does not throw in strict mode when writability/configurability/extensibility is violated\nvar noThrowOnStrictViolation = (function () {\n\ttry {\n\t\tdelete [].length;\n\t\treturn true;\n\t} catch (e) {\n\t\treturn false;\n\t}\n}());\n\n// https://ecma-international.org/ecma-262/6.0/#sec-set-o-p-v-throw\n\nmodule.exports = function Set(O, P, V, Throw) {\n\tif (Type(O) !== 'Object') {\n\t\tthrow new $TypeError('Assertion failed: `O` must be an Object');\n\t}\n\tif (!IsPropertyKey(P)) {\n\t\tthrow new $TypeError('Assertion failed: `P` must be a Property Key');\n\t}\n\tif (Type(Throw) !== 'Boolean') {\n\t\tthrow new $TypeError('Assertion failed: `Throw` must be a Boolean');\n\t}\n\tif (Throw) {\n\t\tO[P] = V; // eslint-disable-line no-param-reassign\n\t\tif (noThrowOnStrictViolation && !SameValue(O[P], V)) {\n\t\t\tthrow new $TypeError('Attempted to assign to readonly property.');\n\t\t}\n\t\treturn true;\n\t}\n\ttry {\n\t\tO[P] = V; // eslint-disable-line no-param-reassign\n\t\treturn noThrowOnStrictViolation ? SameValue(O[P], V) : true;\n\t} catch (e) {\n\t\treturn false;\n\t}\n\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///2946\n")},303:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\n\nvar $species = GetIntrinsic('%Symbol.species%', true);\nvar $TypeError = GetIntrinsic('%TypeError%');\n\nvar IsConstructor = __webpack_require__(970);\nvar Type = __webpack_require__(1915);\n\n// https://ecma-international.org/ecma-262/6.0/#sec-speciesconstructor\n\nmodule.exports = function SpeciesConstructor(O, defaultConstructor) {\n\tif (Type(O) !== 'Object') {\n\t\tthrow new $TypeError('Assertion failed: Type(O) is not Object');\n\t}\n\tvar C = O.constructor;\n\tif (typeof C === 'undefined') {\n\t\treturn defaultConstructor;\n\t}\n\tif (Type(C) !== 'Object') {\n\t\tthrow new $TypeError('O.constructor is not an Object');\n\t}\n\tvar S = $species ? C[$species] : void 0;\n\tif (S == null) {\n\t\treturn defaultConstructor;\n\t}\n\tif (IsConstructor(S)) {\n\t\treturn S;\n\t}\n\tthrow new $TypeError('no constructor found');\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvU3BlY2llc0NvbnN0cnVjdG9yLmpzP2U2NjYiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQWE7O0FBRWIsbUJBQW1CLG1CQUFPLENBQUMsR0FBZTs7QUFFMUM7QUFDQTs7QUFFQSxvQkFBb0IsbUJBQU8sQ0FBQyxHQUFpQjtBQUM3QyxXQUFXLG1CQUFPLENBQUMsSUFBUTs7QUFFM0I7O0FBRUE7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQSIsImZpbGUiOiIzMDMuanMiLCJzb3VyY2VzQ29udGVudCI6WyIndXNlIHN0cmljdCc7XG5cbnZhciBHZXRJbnRyaW5zaWMgPSByZXF1aXJlKCdnZXQtaW50cmluc2ljJyk7XG5cbnZhciAkc3BlY2llcyA9IEdldEludHJpbnNpYygnJVN5bWJvbC5zcGVjaWVzJScsIHRydWUpO1xudmFyICRUeXBlRXJyb3IgPSBHZXRJbnRyaW5zaWMoJyVUeXBlRXJyb3IlJyk7XG5cbnZhciBJc0NvbnN0cnVjdG9yID0gcmVxdWlyZSgnLi9Jc0NvbnN0cnVjdG9yJyk7XG52YXIgVHlwZSA9IHJlcXVpcmUoJy4vVHlwZScpO1xuXG4vLyBodHRwczovL2VjbWEtaW50ZXJuYXRpb25hbC5vcmcvZWNtYS0yNjIvNi4wLyNzZWMtc3BlY2llc2NvbnN0cnVjdG9yXG5cbm1vZHVsZS5leHBvcnRzID0gZnVuY3Rpb24gU3BlY2llc0NvbnN0cnVjdG9yKE8sIGRlZmF1bHRDb25zdHJ1Y3Rvcikge1xuXHRpZiAoVHlwZShPKSAhPT0gJ09iamVjdCcpIHtcblx0XHR0aHJvdyBuZXcgJFR5cGVFcnJvcignQXNzZXJ0aW9uIGZhaWxlZDogVHlwZShPKSBpcyBub3QgT2JqZWN0Jyk7XG5cdH1cblx0dmFyIEMgPSBPLmNvbnN0cnVjdG9yO1xuXHRpZiAodHlwZW9mIEMgPT09ICd1bmRlZmluZWQnKSB7XG5cdFx0cmV0dXJuIGRlZmF1bHRDb25zdHJ1Y3Rvcjtcblx0fVxuXHRpZiAoVHlwZShDKSAhPT0gJ09iamVjdCcpIHtcblx0XHR0aHJvdyBuZXcgJFR5cGVFcnJvcignTy5jb25zdHJ1Y3RvciBpcyBub3QgYW4gT2JqZWN0Jyk7XG5cdH1cblx0dmFyIFMgPSAkc3BlY2llcyA/IENbJHNwZWNpZXNdIDogdm9pZCAwO1xuXHRpZiAoUyA9PSBudWxsKSB7XG5cdFx0cmV0dXJuIGRlZmF1bHRDb25zdHJ1Y3Rvcjtcblx0fVxuXHRpZiAoSXNDb25zdHJ1Y3RvcihTKSkge1xuXHRcdHJldHVybiBTO1xuXHR9XG5cdHRocm93IG5ldyAkVHlwZUVycm9yKCdubyBjb25zdHJ1Y3RvciBmb3VuZCcpO1xufTtcbiJdLCJzb3VyY2VSb290IjoiIn0=\n//# sourceURL=webpack-internal:///303\n")},3683:module=>{"use strict";eval("\n\n// http://262.ecma-international.org/5.1/#sec-9.2\n\nmodule.exports = function ToBoolean(value) { return !!value; };\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvVG9Cb29sZWFuLmpzPzhiMTEiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQWE7O0FBRWI7O0FBRUEsNENBQTRDLGdCQUFnQiIsImZpbGUiOiIzNjgzLmpzIiwic291cmNlc0NvbnRlbnQiOlsiJ3VzZSBzdHJpY3QnO1xuXG4vLyBodHRwOi8vMjYyLmVjbWEtaW50ZXJuYXRpb25hbC5vcmcvNS4xLyNzZWMtOS4yXG5cbm1vZHVsZS5leHBvcnRzID0gZnVuY3Rpb24gVG9Cb29sZWFuKHZhbHVlKSB7IHJldHVybiAhIXZhbHVlOyB9O1xuIl0sInNvdXJjZVJvb3QiOiIifQ==\n//# sourceURL=webpack-internal:///3683\n")},5912:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar ES5ToInteger = __webpack_require__(7195);\n\nvar ToNumber = __webpack_require__(7622);\n\n// https://262.ecma-international.org/11.0/#sec-tointeger\n\nmodule.exports = function ToInteger(value) {\n\tvar number = ToNumber(value);\n\tif (number !== 0) {\n\t\tnumber = ES5ToInteger(number);\n\t}\n\treturn number === 0 ? 0 : number;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvVG9JbnRlZ2VyLmpzPzQyYmYiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQWE7O0FBRWIsbUJBQW1CLG1CQUFPLENBQUMsSUFBZ0I7O0FBRTNDLGVBQWUsbUJBQU8sQ0FBQyxJQUFZOztBQUVuQzs7QUFFQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQSIsImZpbGUiOiI1OTEyLmpzIiwic291cmNlc0NvbnRlbnQiOlsiJ3VzZSBzdHJpY3QnO1xuXG52YXIgRVM1VG9JbnRlZ2VyID0gcmVxdWlyZSgnLi4vNS9Ub0ludGVnZXInKTtcblxudmFyIFRvTnVtYmVyID0gcmVxdWlyZSgnLi9Ub051bWJlcicpO1xuXG4vLyBodHRwczovLzI2Mi5lY21hLWludGVybmF0aW9uYWwub3JnLzExLjAvI3NlYy10b2ludGVnZXJcblxubW9kdWxlLmV4cG9ydHMgPSBmdW5jdGlvbiBUb0ludGVnZXIodmFsdWUpIHtcblx0dmFyIG51bWJlciA9IFRvTnVtYmVyKHZhbHVlKTtcblx0aWYgKG51bWJlciAhPT0gMCkge1xuXHRcdG51bWJlciA9IEVTNVRvSW50ZWdlcihudW1iZXIpO1xuXHR9XG5cdHJldHVybiBudW1iZXIgPT09IDAgPyAwIDogbnVtYmVyO1xufTtcbiJdLCJzb3VyY2VSb290IjoiIn0=\n//# sourceURL=webpack-internal:///5912\n")},8502:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar MAX_SAFE_INTEGER = __webpack_require__(5376);\n\nvar ToInteger = __webpack_require__(5912);\n\nmodule.exports = function ToLength(argument) {\n\tvar len = ToInteger(argument);\n\tif (len <= 0) { return 0; } // includes converting -0 to +0\n\tif (len > MAX_SAFE_INTEGER) { return MAX_SAFE_INTEGER; }\n\treturn len;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvVG9MZW5ndGguanM/OTk5OSJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBYTs7QUFFYix1QkFBdUIsbUJBQU8sQ0FBQyxJQUEyQjs7QUFFMUQsZ0JBQWdCLG1CQUFPLENBQUMsSUFBYTs7QUFFckM7QUFDQTtBQUNBLGdCQUFnQixVQUFVLEVBQUU7QUFDNUIsOEJBQThCLHlCQUF5QjtBQUN2RDtBQUNBIiwiZmlsZSI6Ijg1MDIuanMiLCJzb3VyY2VzQ29udGVudCI6WyIndXNlIHN0cmljdCc7XG5cbnZhciBNQVhfU0FGRV9JTlRFR0VSID0gcmVxdWlyZSgnLi4vaGVscGVycy9tYXhTYWZlSW50ZWdlcicpO1xuXG52YXIgVG9JbnRlZ2VyID0gcmVxdWlyZSgnLi9Ub0ludGVnZXInKTtcblxubW9kdWxlLmV4cG9ydHMgPSBmdW5jdGlvbiBUb0xlbmd0aChhcmd1bWVudCkge1xuXHR2YXIgbGVuID0gVG9JbnRlZ2VyKGFyZ3VtZW50KTtcblx0aWYgKGxlbiA8PSAwKSB7IHJldHVybiAwOyB9IC8vIGluY2x1ZGVzIGNvbnZlcnRpbmcgLTAgdG8gKzBcblx0aWYgKGxlbiA+IE1BWF9TQUZFX0lOVEVHRVIpIHsgcmV0dXJuIE1BWF9TQUZFX0lOVEVHRVI7IH1cblx0cmV0dXJuIGxlbjtcbn07XG4iXSwic291cmNlUm9vdCI6IiJ9\n//# sourceURL=webpack-internal:///8502\n")},7622:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\n\nvar $TypeError = GetIntrinsic('%TypeError%');\nvar $Number = GetIntrinsic('%Number%');\nvar $RegExp = GetIntrinsic('%RegExp%');\nvar $parseInteger = GetIntrinsic('%parseInt%');\n\nvar callBound = __webpack_require__(1924);\nvar regexTester = __webpack_require__(1652);\nvar isPrimitive = __webpack_require__(3126);\n\nvar $strSlice = callBound('String.prototype.slice');\nvar isBinary = regexTester(/^0b[01]+$/i);\nvar isOctal = regexTester(/^0o[0-7]+$/i);\nvar isInvalidHexLiteral = regexTester(/^[-+]0x[0-9a-f]+$/i);\nvar nonWS = ['\\u0085', '\\u200b', '\\ufffe'].join('');\nvar nonWSregex = new $RegExp('[' + nonWS + ']', 'g');\nvar hasNonWS = regexTester(nonWSregex);\n\n// whitespace from: https://es5.github.io/#x15.5.4.20\n// implementation from https://github.com/es-shims/es5-shim/blob/v3.4.0/es5-shim.js#L1304-L1324\nvar ws = [\n\t'\\x09\\x0A\\x0B\\x0C\\x0D\\x20\\xA0\\u1680\\u180E\\u2000\\u2001\\u2002\\u2003',\n\t'\\u2004\\u2005\\u2006\\u2007\\u2008\\u2009\\u200A\\u202F\\u205F\\u3000\\u2028',\n\t'\\u2029\\uFEFF'\n].join('');\nvar trimRegex = new RegExp('(^[' + ws + ']+)|([' + ws + ']+$)', 'g');\nvar $replace = callBound('String.prototype.replace');\nvar $trim = function (value) {\n\treturn $replace(value, trimRegex, '');\n};\n\nvar ToPrimitive = __webpack_require__(8842);\n\n// https://ecma-international.org/ecma-262/6.0/#sec-tonumber\n\nmodule.exports = function ToNumber(argument) {\n\tvar value = isPrimitive(argument) ? argument : ToPrimitive(argument, $Number);\n\tif (typeof value === 'symbol') {\n\t\tthrow new $TypeError('Cannot convert a Symbol value to a number');\n\t}\n\tif (typeof value === 'bigint') {\n\t\tthrow new $TypeError('Conversion from \\'BigInt\\' to \\'number\\' is not allowed.');\n\t}\n\tif (typeof value === 'string') {\n\t\tif (isBinary(value)) {\n\t\t\treturn ToNumber($parseInteger($strSlice(value, 2), 2));\n\t\t} else if (isOctal(value)) {\n\t\t\treturn ToNumber($parseInteger($strSlice(value, 2), 8));\n\t\t} else if (hasNonWS(value) || isInvalidHexLiteral(value)) {\n\t\t\treturn NaN;\n\t\t}\n\t\tvar trimmed = $trim(value);\n\t\tif (trimmed !== value) {\n\t\t\treturn ToNumber(trimmed);\n\t\t}\n\n\t}\n\treturn $Number(value);\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///7622\n")},2093:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\n\nvar $Object = GetIntrinsic('%Object%');\n\nvar RequireObjectCoercible = __webpack_require__(3733);\n\n// https://ecma-international.org/ecma-262/6.0/#sec-toobject\n\nmodule.exports = function ToObject(value) {\n\tRequireObjectCoercible(value);\n\treturn $Object(value);\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvVG9PYmplY3QuanM/ZWY3MCJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBYTs7QUFFYixtQkFBbUIsbUJBQU8sQ0FBQyxHQUFlOztBQUUxQzs7QUFFQSw2QkFBNkIsbUJBQU8sQ0FBQyxJQUEwQjs7QUFFL0Q7O0FBRUE7QUFDQTtBQUNBO0FBQ0EiLCJmaWxlIjoiMjA5My5qcyIsInNvdXJjZXNDb250ZW50IjpbIid1c2Ugc3RyaWN0JztcblxudmFyIEdldEludHJpbnNpYyA9IHJlcXVpcmUoJ2dldC1pbnRyaW5zaWMnKTtcblxudmFyICRPYmplY3QgPSBHZXRJbnRyaW5zaWMoJyVPYmplY3QlJyk7XG5cbnZhciBSZXF1aXJlT2JqZWN0Q29lcmNpYmxlID0gcmVxdWlyZSgnLi9SZXF1aXJlT2JqZWN0Q29lcmNpYmxlJyk7XG5cbi8vIGh0dHBzOi8vZWNtYS1pbnRlcm5hdGlvbmFsLm9yZy9lY21hLTI2Mi82LjAvI3NlYy10b29iamVjdFxuXG5tb2R1bGUuZXhwb3J0cyA9IGZ1bmN0aW9uIFRvT2JqZWN0KHZhbHVlKSB7XG5cdFJlcXVpcmVPYmplY3RDb2VyY2libGUodmFsdWUpO1xuXHRyZXR1cm4gJE9iamVjdCh2YWx1ZSk7XG59O1xuIl0sInNvdXJjZVJvb3QiOiIifQ==\n//# sourceURL=webpack-internal:///2093\n")},8842:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar toPrimitive = __webpack_require__(1503);\n\n// https://ecma-international.org/ecma-262/6.0/#sec-toprimitive\n\nmodule.exports = function ToPrimitive(input) {\n\tif (arguments.length > 1) {\n\t\treturn toPrimitive(input, arguments[1]);\n\t}\n\treturn toPrimitive(input);\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvVG9QcmltaXRpdmUuanM/NGZmOCJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBYTs7QUFFYixrQkFBa0IsbUJBQU8sQ0FBQyxJQUF3Qjs7QUFFbEQ7O0FBRUE7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBIiwiZmlsZSI6Ijg4NDIuanMiLCJzb3VyY2VzQ29udGVudCI6WyIndXNlIHN0cmljdCc7XG5cbnZhciB0b1ByaW1pdGl2ZSA9IHJlcXVpcmUoJ2VzLXRvLXByaW1pdGl2ZS9lczIwMTUnKTtcblxuLy8gaHR0cHM6Ly9lY21hLWludGVybmF0aW9uYWwub3JnL2VjbWEtMjYyLzYuMC8jc2VjLXRvcHJpbWl0aXZlXG5cbm1vZHVsZS5leHBvcnRzID0gZnVuY3Rpb24gVG9QcmltaXRpdmUoaW5wdXQpIHtcblx0aWYgKGFyZ3VtZW50cy5sZW5ndGggPiAxKSB7XG5cdFx0cmV0dXJuIHRvUHJpbWl0aXZlKGlucHV0LCBhcmd1bWVudHNbMV0pO1xuXHR9XG5cdHJldHVybiB0b1ByaW1pdGl2ZShpbnB1dCk7XG59O1xuIl0sInNvdXJjZVJvb3QiOiIifQ==\n//# sourceURL=webpack-internal:///8842\n")},2449:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar has = __webpack_require__(7642);\n\nvar GetIntrinsic = __webpack_require__(210);\n\nvar $TypeError = GetIntrinsic('%TypeError%');\n\nvar Type = __webpack_require__(1915);\nvar ToBoolean = __webpack_require__(3683);\nvar IsCallable = __webpack_require__(590);\n\n// https://262.ecma-international.org/5.1/#sec-8.10.5\n\nmodule.exports = function ToPropertyDescriptor(Obj) {\n\tif (Type(Obj) !== 'Object') {\n\t\tthrow new $TypeError('ToPropertyDescriptor requires an object');\n\t}\n\n\tvar desc = {};\n\tif (has(Obj, 'enumerable')) {\n\t\tdesc['[[Enumerable]]'] = ToBoolean(Obj.enumerable);\n\t}\n\tif (has(Obj, 'configurable')) {\n\t\tdesc['[[Configurable]]'] = ToBoolean(Obj.configurable);\n\t}\n\tif (has(Obj, 'value')) {\n\t\tdesc['[[Value]]'] = Obj.value;\n\t}\n\tif (has(Obj, 'writable')) {\n\t\tdesc['[[Writable]]'] = ToBoolean(Obj.writable);\n\t}\n\tif (has(Obj, 'get')) {\n\t\tvar getter = Obj.get;\n\t\tif (typeof getter !== 'undefined' && !IsCallable(getter)) {\n\t\t\tthrow new $TypeError('getter must be a function');\n\t\t}\n\t\tdesc['[[Get]]'] = getter;\n\t}\n\tif (has(Obj, 'set')) {\n\t\tvar setter = Obj.set;\n\t\tif (typeof setter !== 'undefined' && !IsCallable(setter)) {\n\t\t\tthrow new $TypeError('setter must be a function');\n\t\t}\n\t\tdesc['[[Set]]'] = setter;\n\t}\n\n\tif ((has(desc, '[[Get]]') || has(desc, '[[Set]]')) && (has(desc, '[[Value]]') || has(desc, '[[Writable]]'))) {\n\t\tthrow new $TypeError('Invalid property descriptor. Cannot both specify accessors and a value or writable attribute');\n\t}\n\treturn desc;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///2449\n")},7308:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\n\nvar $String = GetIntrinsic('%String%');\nvar $TypeError = GetIntrinsic('%TypeError%');\n\n// https://ecma-international.org/ecma-262/6.0/#sec-tostring\n\nmodule.exports = function ToString(argument) {\n\tif (typeof argument === 'symbol') {\n\t\tthrow new $TypeError('Cannot convert a Symbol value to a string');\n\t}\n\treturn $String(argument);\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvVG9TdHJpbmcuanM/ZmUxZiJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBYTs7QUFFYixtQkFBbUIsbUJBQU8sQ0FBQyxHQUFlOztBQUUxQztBQUNBOztBQUVBOztBQUVBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQSIsImZpbGUiOiI3MzA4LmpzIiwic291cmNlc0NvbnRlbnQiOlsiJ3VzZSBzdHJpY3QnO1xuXG52YXIgR2V0SW50cmluc2ljID0gcmVxdWlyZSgnZ2V0LWludHJpbnNpYycpO1xuXG52YXIgJFN0cmluZyA9IEdldEludHJpbnNpYygnJVN0cmluZyUnKTtcbnZhciAkVHlwZUVycm9yID0gR2V0SW50cmluc2ljKCclVHlwZUVycm9yJScpO1xuXG4vLyBodHRwczovL2VjbWEtaW50ZXJuYXRpb25hbC5vcmcvZWNtYS0yNjIvNi4wLyNzZWMtdG9zdHJpbmdcblxubW9kdWxlLmV4cG9ydHMgPSBmdW5jdGlvbiBUb1N0cmluZyhhcmd1bWVudCkge1xuXHRpZiAodHlwZW9mIGFyZ3VtZW50ID09PSAnc3ltYm9sJykge1xuXHRcdHRocm93IG5ldyAkVHlwZUVycm9yKCdDYW5ub3QgY29udmVydCBhIFN5bWJvbCB2YWx1ZSB0byBhIHN0cmluZycpO1xuXHR9XG5cdHJldHVybiAkU3RyaW5nKGFyZ3VtZW50KTtcbn07XG4iXSwic291cmNlUm9vdCI6IiJ9\n//# sourceURL=webpack-internal:///7308\n")},1915:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar ES5Type = __webpack_require__(1276);\n\n// https://262.ecma-international.org/11.0/#sec-ecmascript-data-types-and-values\n\nmodule.exports = function Type(x) {\n\tif (typeof x === 'symbol') {\n\t\treturn 'Symbol';\n\t}\n\tif (typeof x === 'bigint') {\n\t\treturn 'BigInt';\n\t}\n\treturn ES5Type(x);\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvVHlwZS5qcz82NzYwIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFhOztBQUViLGNBQWMsbUJBQU8sQ0FBQyxJQUFXOztBQUVqQzs7QUFFQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0EiLCJmaWxlIjoiMTkxNS5qcyIsInNvdXJjZXNDb250ZW50IjpbIid1c2Ugc3RyaWN0JztcblxudmFyIEVTNVR5cGUgPSByZXF1aXJlKCcuLi81L1R5cGUnKTtcblxuLy8gaHR0cHM6Ly8yNjIuZWNtYS1pbnRlcm5hdGlvbmFsLm9yZy8xMS4wLyNzZWMtZWNtYXNjcmlwdC1kYXRhLXR5cGVzLWFuZC12YWx1ZXNcblxubW9kdWxlLmV4cG9ydHMgPSBmdW5jdGlvbiBUeXBlKHgpIHtcblx0aWYgKHR5cGVvZiB4ID09PSAnc3ltYm9sJykge1xuXHRcdHJldHVybiAnU3ltYm9sJztcblx0fVxuXHRpZiAodHlwZW9mIHggPT09ICdiaWdpbnQnKSB7XG5cdFx0cmV0dXJuICdCaWdJbnQnO1xuXHR9XG5cdHJldHVybiBFUzVUeXBlKHgpO1xufTtcbiJdLCJzb3VyY2VSb290IjoiIn0=\n//# sourceURL=webpack-internal:///1915\n")},4969:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\n\nvar $TypeError = GetIntrinsic('%TypeError%');\nvar $fromCharCode = GetIntrinsic('%String.fromCharCode%');\n\nvar isLeadingSurrogate = __webpack_require__(1996);\nvar isTrailingSurrogate = __webpack_require__(6935);\n\n// https://262.ecma-international.org/11.0/#sec-utf16decodesurrogatepair\n\nmodule.exports = function UTF16DecodeSurrogatePair(lead, trail) {\n\tif (!isLeadingSurrogate(lead) || !isTrailingSurrogate(trail)) {\n\t\tthrow new $TypeError('Assertion failed: `lead` must be a leading surrogate char code, and `trail` must be a trailing surrogate char code');\n\t}\n\t// var cp = (lead - 0xD800) * 0x400 + (trail - 0xDC00) + 0x10000;\n\treturn $fromCharCode(lead) + $fromCharCode(trail);\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvVVRGMTZEZWNvZGVTdXJyb2dhdGVQYWlyLmpzP2E0MzIiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQWE7O0FBRWIsbUJBQW1CLG1CQUFPLENBQUMsR0FBZTs7QUFFMUM7QUFDQTs7QUFFQSx5QkFBeUIsbUJBQU8sQ0FBQyxJQUErQjtBQUNoRSwwQkFBMEIsbUJBQU8sQ0FBQyxJQUFnQzs7QUFFbEU7O0FBRUE7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0EiLCJmaWxlIjoiNDk2OS5qcyIsInNvdXJjZXNDb250ZW50IjpbIid1c2Ugc3RyaWN0JztcblxudmFyIEdldEludHJpbnNpYyA9IHJlcXVpcmUoJ2dldC1pbnRyaW5zaWMnKTtcblxudmFyICRUeXBlRXJyb3IgPSBHZXRJbnRyaW5zaWMoJyVUeXBlRXJyb3IlJyk7XG52YXIgJGZyb21DaGFyQ29kZSA9IEdldEludHJpbnNpYygnJVN0cmluZy5mcm9tQ2hhckNvZGUlJyk7XG5cbnZhciBpc0xlYWRpbmdTdXJyb2dhdGUgPSByZXF1aXJlKCcuLi9oZWxwZXJzL2lzTGVhZGluZ1N1cnJvZ2F0ZScpO1xudmFyIGlzVHJhaWxpbmdTdXJyb2dhdGUgPSByZXF1aXJlKCcuLi9oZWxwZXJzL2lzVHJhaWxpbmdTdXJyb2dhdGUnKTtcblxuLy8gaHR0cHM6Ly8yNjIuZWNtYS1pbnRlcm5hdGlvbmFsLm9yZy8xMS4wLyNzZWMtdXRmMTZkZWNvZGVzdXJyb2dhdGVwYWlyXG5cbm1vZHVsZS5leHBvcnRzID0gZnVuY3Rpb24gVVRGMTZEZWNvZGVTdXJyb2dhdGVQYWlyKGxlYWQsIHRyYWlsKSB7XG5cdGlmICghaXNMZWFkaW5nU3Vycm9nYXRlKGxlYWQpIHx8ICFpc1RyYWlsaW5nU3Vycm9nYXRlKHRyYWlsKSkge1xuXHRcdHRocm93IG5ldyAkVHlwZUVycm9yKCdBc3NlcnRpb24gZmFpbGVkOiBgbGVhZGAgbXVzdCBiZSBhIGxlYWRpbmcgc3Vycm9nYXRlIGNoYXIgY29kZSwgYW5kIGB0cmFpbGAgbXVzdCBiZSBhIHRyYWlsaW5nIHN1cnJvZ2F0ZSBjaGFyIGNvZGUnKTtcblx0fVxuXHQvLyB2YXIgY3AgPSAobGVhZCAtIDB4RDgwMCkgKiAweDQwMCArICh0cmFpbCAtIDB4REMwMCkgKyAweDEwMDAwO1xuXHRyZXR1cm4gJGZyb21DaGFyQ29kZShsZWFkKSArICRmcm9tQ2hhckNvZGUodHJhaWwpO1xufTtcbiJdLCJzb3VyY2VSb290IjoiIn0=\n//# sourceURL=webpack-internal:///4969\n")},1717:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\n\nvar $abs = GetIntrinsic('%Math.abs%');\n\n// http://262.ecma-international.org/5.1/#sec-5.2\n\nmodule.exports = function abs(x) {\n\treturn $abs(x);\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvYWJzLmpzPzQ1NjAiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQWE7O0FBRWIsbUJBQW1CLG1CQUFPLENBQUMsR0FBZTs7QUFFMUM7O0FBRUE7O0FBRUE7QUFDQTtBQUNBIiwiZmlsZSI6IjE3MTcuanMiLCJzb3VyY2VzQ29udGVudCI6WyIndXNlIHN0cmljdCc7XG5cbnZhciBHZXRJbnRyaW5zaWMgPSByZXF1aXJlKCdnZXQtaW50cmluc2ljJyk7XG5cbnZhciAkYWJzID0gR2V0SW50cmluc2ljKCclTWF0aC5hYnMlJyk7XG5cbi8vIGh0dHA6Ly8yNjIuZWNtYS1pbnRlcm5hdGlvbmFsLm9yZy81LjEvI3NlYy01LjJcblxubW9kdWxlLmV4cG9ydHMgPSBmdW5jdGlvbiBhYnMoeCkge1xuXHRyZXR1cm4gJGFicyh4KTtcbn07XG4iXSwic291cmNlUm9vdCI6IiJ9\n//# sourceURL=webpack-internal:///1717\n")},9202:module=>{"use strict";eval("\n\n// var modulo = require('./modulo');\nvar $floor = Math.floor;\n\n// http://262.ecma-international.org/5.1/#sec-5.2\n\nmodule.exports = function floor(x) {\n\t// return x - modulo(x, 1);\n\treturn $floor(x);\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzIwMjAvZmxvb3IuanM/MTVkNSJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBYTs7QUFFYjtBQUNBOztBQUVBOztBQUVBO0FBQ0E7QUFDQTtBQUNBIiwiZmlsZSI6IjkyMDIuanMiLCJzb3VyY2VzQ29udGVudCI6WyIndXNlIHN0cmljdCc7XG5cbi8vIHZhciBtb2R1bG8gPSByZXF1aXJlKCcuL21vZHVsbycpO1xudmFyICRmbG9vciA9IE1hdGguZmxvb3I7XG5cbi8vIGh0dHA6Ly8yNjIuZWNtYS1pbnRlcm5hdGlvbmFsLm9yZy81LjEvI3NlYy01LjJcblxubW9kdWxlLmV4cG9ydHMgPSBmdW5jdGlvbiBmbG9vcih4KSB7XG5cdC8vIHJldHVybiB4IC0gbW9kdWxvKHgsIDEpO1xuXHRyZXR1cm4gJGZsb29yKHgpO1xufTtcbiJdLCJzb3VyY2VSb290IjoiIn0=\n//# sourceURL=webpack-internal:///9202\n")},8631:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\n\nvar $TypeError = GetIntrinsic('%TypeError%');\n\n// http://262.ecma-international.org/5.1/#sec-9.10\n\nmodule.exports = function CheckObjectCoercible(value, optMessage) {\n\tif (value == null) {\n\t\tthrow new $TypeError(optMessage || ('Cannot call method on ' + value));\n\t}\n\treturn value;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzUvQ2hlY2tPYmplY3RDb2VyY2libGUuanM/NGQ1MiJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBYTs7QUFFYixtQkFBbUIsbUJBQU8sQ0FBQyxHQUFlOztBQUUxQzs7QUFFQTs7QUFFQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0EiLCJmaWxlIjoiODYzMS5qcyIsInNvdXJjZXNDb250ZW50IjpbIid1c2Ugc3RyaWN0JztcblxudmFyIEdldEludHJpbnNpYyA9IHJlcXVpcmUoJ2dldC1pbnRyaW5zaWMnKTtcblxudmFyICRUeXBlRXJyb3IgPSBHZXRJbnRyaW5zaWMoJyVUeXBlRXJyb3IlJyk7XG5cbi8vIGh0dHA6Ly8yNjIuZWNtYS1pbnRlcm5hdGlvbmFsLm9yZy81LjEvI3NlYy05LjEwXG5cbm1vZHVsZS5leHBvcnRzID0gZnVuY3Rpb24gQ2hlY2tPYmplY3RDb2VyY2libGUodmFsdWUsIG9wdE1lc3NhZ2UpIHtcblx0aWYgKHZhbHVlID09IG51bGwpIHtcblx0XHR0aHJvdyBuZXcgJFR5cGVFcnJvcihvcHRNZXNzYWdlIHx8ICgnQ2Fubm90IGNhbGwgbWV0aG9kIG9uICcgKyB2YWx1ZSkpO1xuXHR9XG5cdHJldHVybiB2YWx1ZTtcbn07XG4iXSwic291cmNlUm9vdCI6IiJ9\n//# sourceURL=webpack-internal:///8631\n")},7195:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar abs = __webpack_require__(2683);\nvar floor = __webpack_require__(9711);\nvar ToNumber = __webpack_require__(7196);\n\nvar $isNaN = __webpack_require__(1214);\nvar $isFinite = __webpack_require__(3060);\nvar $sign = __webpack_require__(4099);\n\n// http://262.ecma-international.org/5.1/#sec-9.4\n\nmodule.exports = function ToInteger(value) {\n\tvar number = ToNumber(value);\n\tif ($isNaN(number)) { return 0; }\n\tif (number === 0 || !$isFinite(number)) { return number; }\n\treturn $sign(number) * floor(abs(number));\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzUvVG9JbnRlZ2VyLmpzP2QzMjciXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQWE7O0FBRWIsVUFBVSxtQkFBTyxDQUFDLElBQU87QUFDekIsWUFBWSxtQkFBTyxDQUFDLElBQVM7QUFDN0IsZUFBZSxtQkFBTyxDQUFDLElBQVk7O0FBRW5DLGFBQWEsbUJBQU8sQ0FBQyxJQUFrQjtBQUN2QyxnQkFBZ0IsbUJBQU8sQ0FBQyxJQUFxQjtBQUM3QyxZQUFZLG1CQUFPLENBQUMsSUFBaUI7O0FBRXJDOztBQUVBO0FBQ0E7QUFDQSxzQkFBc0IsVUFBVTtBQUNoQywwQ0FBMEMsZUFBZTtBQUN6RDtBQUNBIiwiZmlsZSI6IjcxOTUuanMiLCJzb3VyY2VzQ29udGVudCI6WyIndXNlIHN0cmljdCc7XG5cbnZhciBhYnMgPSByZXF1aXJlKCcuL2FicycpO1xudmFyIGZsb29yID0gcmVxdWlyZSgnLi9mbG9vcicpO1xudmFyIFRvTnVtYmVyID0gcmVxdWlyZSgnLi9Ub051bWJlcicpO1xuXG52YXIgJGlzTmFOID0gcmVxdWlyZSgnLi4vaGVscGVycy9pc05hTicpO1xudmFyICRpc0Zpbml0ZSA9IHJlcXVpcmUoJy4uL2hlbHBlcnMvaXNGaW5pdGUnKTtcbnZhciAkc2lnbiA9IHJlcXVpcmUoJy4uL2hlbHBlcnMvc2lnbicpO1xuXG4vLyBodHRwOi8vMjYyLmVjbWEtaW50ZXJuYXRpb25hbC5vcmcvNS4xLyNzZWMtOS40XG5cbm1vZHVsZS5leHBvcnRzID0gZnVuY3Rpb24gVG9JbnRlZ2VyKHZhbHVlKSB7XG5cdHZhciBudW1iZXIgPSBUb051bWJlcih2YWx1ZSk7XG5cdGlmICgkaXNOYU4obnVtYmVyKSkgeyByZXR1cm4gMDsgfVxuXHRpZiAobnVtYmVyID09PSAwIHx8ICEkaXNGaW5pdGUobnVtYmVyKSkgeyByZXR1cm4gbnVtYmVyOyB9XG5cdHJldHVybiAkc2lnbihudW1iZXIpICogZmxvb3IoYWJzKG51bWJlcikpO1xufTtcbiJdLCJzb3VyY2VSb290IjoiIn0=\n//# sourceURL=webpack-internal:///7195\n")},7196:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar ToPrimitive = __webpack_require__(1318);\n\n// http://262.ecma-international.org/5.1/#sec-9.3\n\nmodule.exports = function ToNumber(value) {\n\tvar prim = ToPrimitive(value, Number);\n\tif (typeof prim !== 'string') {\n\t\treturn +prim; // eslint-disable-line no-implicit-coercion\n\t}\n\n\t// eslint-disable-next-line no-control-regex\n\tvar trimmed = prim.replace(/^[ \\t\\x0b\\f\\xa0\\ufeff\\n\\r\\u2028\\u2029\\u1680\\u180e\\u2000\\u2001\\u2002\\u2003\\u2004\\u2005\\u2006\\u2007\\u2008\\u2009\\u200a\\u202f\\u205f\\u3000\\u0085]+|[ \\t\\x0b\\f\\xa0\\ufeff\\n\\r\\u2028\\u2029\\u1680\\u180e\\u2000\\u2001\\u2002\\u2003\\u2004\\u2005\\u2006\\u2007\\u2008\\u2009\\u200a\\u202f\\u205f\\u3000\\u0085]+$/g, '');\n\tif ((/^0[ob]|^[+-]0x/).test(trimmed)) {\n\t\treturn NaN;\n\t}\n\n\treturn +trimmed; // eslint-disable-line no-implicit-coercion\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzUvVG9OdW1iZXIuanM/YmI1YyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBYTs7QUFFYixrQkFBa0IsbUJBQU8sQ0FBQyxJQUFlOztBQUV6Qzs7QUFFQTtBQUNBO0FBQ0E7QUFDQSxlQUFlO0FBQ2Y7O0FBRUE7QUFDQTtBQUNBO0FBQ0E7QUFDQTs7QUFFQSxpQkFBaUI7QUFDakIiLCJmaWxlIjoiNzE5Ni5qcyIsInNvdXJjZXNDb250ZW50IjpbIid1c2Ugc3RyaWN0JztcblxudmFyIFRvUHJpbWl0aXZlID0gcmVxdWlyZSgnLi9Ub1ByaW1pdGl2ZScpO1xuXG4vLyBodHRwOi8vMjYyLmVjbWEtaW50ZXJuYXRpb25hbC5vcmcvNS4xLyNzZWMtOS4zXG5cbm1vZHVsZS5leHBvcnRzID0gZnVuY3Rpb24gVG9OdW1iZXIodmFsdWUpIHtcblx0dmFyIHByaW0gPSBUb1ByaW1pdGl2ZSh2YWx1ZSwgTnVtYmVyKTtcblx0aWYgKHR5cGVvZiBwcmltICE9PSAnc3RyaW5nJykge1xuXHRcdHJldHVybiArcHJpbTsgLy8gZXNsaW50LWRpc2FibGUtbGluZSBuby1pbXBsaWNpdC1jb2VyY2lvblxuXHR9XG5cblx0Ly8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIG5vLWNvbnRyb2wtcmVnZXhcblx0dmFyIHRyaW1tZWQgPSBwcmltLnJlcGxhY2UoL15bIFxcdFxceDBiXFxmXFx4YTBcXHVmZWZmXFxuXFxyXFx1MjAyOFxcdTIwMjlcXHUxNjgwXFx1MTgwZVxcdTIwMDBcXHUyMDAxXFx1MjAwMlxcdTIwMDNcXHUyMDA0XFx1MjAwNVxcdTIwMDZcXHUyMDA3XFx1MjAwOFxcdTIwMDlcXHUyMDBhXFx1MjAyZlxcdTIwNWZcXHUzMDAwXFx1MDA4NV0rfFsgXFx0XFx4MGJcXGZcXHhhMFxcdWZlZmZcXG5cXHJcXHUyMDI4XFx1MjAyOVxcdTE2ODBcXHUxODBlXFx1MjAwMFxcdTIwMDFcXHUyMDAyXFx1MjAwM1xcdTIwMDRcXHUyMDA1XFx1MjAwNlxcdTIwMDdcXHUyMDA4XFx1MjAwOVxcdTIwMGFcXHUyMDJmXFx1MjA1ZlxcdTMwMDBcXHUwMDg1XSskL2csICcnKTtcblx0aWYgKCgvXjBbb2JdfF5bKy1dMHgvKS50ZXN0KHRyaW1tZWQpKSB7XG5cdFx0cmV0dXJuIE5hTjtcblx0fVxuXG5cdHJldHVybiArdHJpbW1lZDsgLy8gZXNsaW50LWRpc2FibGUtbGluZSBuby1pbXBsaWNpdC1jb2VyY2lvblxufTtcbiJdLCJzb3VyY2VSb290IjoiIn0=\n//# sourceURL=webpack-internal:///7196\n")},1318:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\n// http://262.ecma-international.org/5.1/#sec-9.1\n\nmodule.exports = __webpack_require__(2116);\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzUvVG9QcmltaXRpdmUuanM/MDExYyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBYTs7QUFFYjs7QUFFQSwwQ0FBK0MiLCJmaWxlIjoiMTMxOC5qcyIsInNvdXJjZXNDb250ZW50IjpbIid1c2Ugc3RyaWN0JztcblxuLy8gaHR0cDovLzI2Mi5lY21hLWludGVybmF0aW9uYWwub3JnLzUuMS8jc2VjLTkuMVxuXG5tb2R1bGUuZXhwb3J0cyA9IHJlcXVpcmUoJ2VzLXRvLXByaW1pdGl2ZS9lczUnKTtcbiJdLCJzb3VyY2VSb290IjoiIn0=\n//# sourceURL=webpack-internal:///1318\n")},1276:module=>{"use strict";eval("\n\n// https://262.ecma-international.org/5.1/#sec-8\n\nmodule.exports = function Type(x) {\n\tif (x === null) {\n\t\treturn 'Null';\n\t}\n\tif (typeof x === 'undefined') {\n\t\treturn 'Undefined';\n\t}\n\tif (typeof x === 'function' || typeof x === 'object') {\n\t\treturn 'Object';\n\t}\n\tif (typeof x === 'number') {\n\t\treturn 'Number';\n\t}\n\tif (typeof x === 'boolean') {\n\t\treturn 'Boolean';\n\t}\n\tif (typeof x === 'string') {\n\t\treturn 'String';\n\t}\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzUvVHlwZS5qcz81MTgzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFhOztBQUViOztBQUVBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0EiLCJmaWxlIjoiMTI3Ni5qcyIsInNvdXJjZXNDb250ZW50IjpbIid1c2Ugc3RyaWN0JztcblxuLy8gaHR0cHM6Ly8yNjIuZWNtYS1pbnRlcm5hdGlvbmFsLm9yZy81LjEvI3NlYy04XG5cbm1vZHVsZS5leHBvcnRzID0gZnVuY3Rpb24gVHlwZSh4KSB7XG5cdGlmICh4ID09PSBudWxsKSB7XG5cdFx0cmV0dXJuICdOdWxsJztcblx0fVxuXHRpZiAodHlwZW9mIHggPT09ICd1bmRlZmluZWQnKSB7XG5cdFx0cmV0dXJuICdVbmRlZmluZWQnO1xuXHR9XG5cdGlmICh0eXBlb2YgeCA9PT0gJ2Z1bmN0aW9uJyB8fCB0eXBlb2YgeCA9PT0gJ29iamVjdCcpIHtcblx0XHRyZXR1cm4gJ09iamVjdCc7XG5cdH1cblx0aWYgKHR5cGVvZiB4ID09PSAnbnVtYmVyJykge1xuXHRcdHJldHVybiAnTnVtYmVyJztcblx0fVxuXHRpZiAodHlwZW9mIHggPT09ICdib29sZWFuJykge1xuXHRcdHJldHVybiAnQm9vbGVhbic7XG5cdH1cblx0aWYgKHR5cGVvZiB4ID09PSAnc3RyaW5nJykge1xuXHRcdHJldHVybiAnU3RyaW5nJztcblx0fVxufTtcbiJdLCJzb3VyY2VSb290IjoiIn0=\n//# sourceURL=webpack-internal:///1276\n")},2683:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\n\nvar $abs = GetIntrinsic('%Math.abs%');\n\n// http://262.ecma-international.org/5.1/#sec-5.2\n\nmodule.exports = function abs(x) {\n\treturn $abs(x);\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzUvYWJzLmpzP2JkMTAiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQWE7O0FBRWIsbUJBQW1CLG1CQUFPLENBQUMsR0FBZTs7QUFFMUM7O0FBRUE7O0FBRUE7QUFDQTtBQUNBIiwiZmlsZSI6IjI2ODMuanMiLCJzb3VyY2VzQ29udGVudCI6WyIndXNlIHN0cmljdCc7XG5cbnZhciBHZXRJbnRyaW5zaWMgPSByZXF1aXJlKCdnZXQtaW50cmluc2ljJyk7XG5cbnZhciAkYWJzID0gR2V0SW50cmluc2ljKCclTWF0aC5hYnMlJyk7XG5cbi8vIGh0dHA6Ly8yNjIuZWNtYS1pbnRlcm5hdGlvbmFsLm9yZy81LjEvI3NlYy01LjJcblxubW9kdWxlLmV4cG9ydHMgPSBmdW5jdGlvbiBhYnMoeCkge1xuXHRyZXR1cm4gJGFicyh4KTtcbn07XG4iXSwic291cmNlUm9vdCI6IiJ9\n//# sourceURL=webpack-internal:///2683\n")},9711:module=>{"use strict";eval("\n\n// var modulo = require('./modulo');\nvar $floor = Math.floor;\n\n// http://262.ecma-international.org/5.1/#sec-5.2\n\nmodule.exports = function floor(x) {\n\t// return x - modulo(x, 1);\n\treturn $floor(x);\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0LzUvZmxvb3IuanM/YjVhNiJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBYTs7QUFFYjtBQUNBOztBQUVBOztBQUVBO0FBQ0E7QUFDQTtBQUNBIiwiZmlsZSI6Ijk3MTEuanMiLCJzb3VyY2VzQ29udGVudCI6WyIndXNlIHN0cmljdCc7XG5cbi8vIHZhciBtb2R1bG8gPSByZXF1aXJlKCcuL21vZHVsbycpO1xudmFyICRmbG9vciA9IE1hdGguZmxvb3I7XG5cbi8vIGh0dHA6Ly8yNjIuZWNtYS1pbnRlcm5hdGlvbmFsLm9yZy81LjEvI3NlYy01LjJcblxubW9kdWxlLmV4cG9ydHMgPSBmdW5jdGlvbiBmbG9vcih4KSB7XG5cdC8vIHJldHVybiB4IC0gbW9kdWxvKHgsIDEpO1xuXHRyZXR1cm4gJGZsb29yKHgpO1xufTtcbiJdLCJzb3VyY2VSb290IjoiIn0=\n//# sourceURL=webpack-internal:///9711\n")},8769:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\n// TODO: remove, semver-major\n\nmodule.exports = __webpack_require__(210);\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0L0dldEludHJpbnNpYy5qcz9lOWFjIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFhOztBQUViOztBQUVBLHlDQUF5QyIsImZpbGUiOiI4NzY5LmpzIiwic291cmNlc0NvbnRlbnQiOlsiJ3VzZSBzdHJpY3QnO1xuXG4vLyBUT0RPOiByZW1vdmUsIHNlbXZlci1tYWpvclxuXG5tb2R1bGUuZXhwb3J0cyA9IHJlcXVpcmUoJ2dldC1pbnRyaW5zaWMnKTtcbiJdLCJzb3VyY2VSb290IjoiIn0=\n//# sourceURL=webpack-internal:///8769\n")},509:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\n\nvar $defineProperty = GetIntrinsic('%Object.defineProperty%', true);\n\nif ($defineProperty) {\n\ttry {\n\t\t$defineProperty({}, 'a', { value: 1 });\n\t} catch (e) {\n\t\t// IE 8 has a broken defineProperty\n\t\t$defineProperty = null;\n\t}\n}\n\nvar callBound = __webpack_require__(1924);\n\nvar $isEnumerable = callBound('Object.prototype.propertyIsEnumerable');\n\n// eslint-disable-next-line max-params\nmodule.exports = function DefineOwnProperty(IsDataDescriptor, SameValue, FromPropertyDescriptor, O, P, desc) {\n\tif (!$defineProperty) {\n\t\tif (!IsDataDescriptor(desc)) {\n\t\t\t// ES3 does not support getters/setters\n\t\t\treturn false;\n\t\t}\n\t\tif (!desc['[[Configurable]]'] || !desc['[[Writable]]']) {\n\t\t\treturn false;\n\t\t}\n\n\t\t// fallback for ES3\n\t\tif (P in O && $isEnumerable(O, P) !== !!desc['[[Enumerable]]']) {\n\t\t\t// a non-enumerable existing property\n\t\t\treturn false;\n\t\t}\n\n\t\t// property does not exist at all, or exists but is enumerable\n\t\tvar V = desc['[[Value]]'];\n\t\t// eslint-disable-next-line no-param-reassign\n\t\tO[P] = V; // will use [[Define]]\n\t\treturn SameValue(O[P], V);\n\t}\n\t$defineProperty(O, P, FromPropertyDescriptor(desc));\n\treturn true;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///509\n")},4179:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\n\nvar $TypeError = GetIntrinsic('%TypeError%');\nvar $SyntaxError = GetIntrinsic('%SyntaxError%');\n\nvar has = __webpack_require__(7642);\n\nvar predicates = {\n\t// https://262.ecma-international.org/6.0/#sec-property-descriptor-specification-type\n\t'Property Descriptor': function isPropertyDescriptor(Type, Desc) {\n\t\tif (Type(Desc) !== 'Object') {\n\t\t\treturn false;\n\t\t}\n\t\tvar allowed = {\n\t\t\t'[[Configurable]]': true,\n\t\t\t'[[Enumerable]]': true,\n\t\t\t'[[Get]]': true,\n\t\t\t'[[Set]]': true,\n\t\t\t'[[Value]]': true,\n\t\t\t'[[Writable]]': true\n\t\t};\n\n\t\tfor (var key in Desc) { // eslint-disable-line\n\t\t\tif (has(Desc, key) && !allowed[key]) {\n\t\t\t\treturn false;\n\t\t\t}\n\t\t}\n\n\t\tvar isData = has(Desc, '[[Value]]');\n\t\tvar IsAccessor = has(Desc, '[[Get]]') || has(Desc, '[[Set]]');\n\t\tif (isData && IsAccessor) {\n\t\t\tthrow new $TypeError('Property Descriptors may not be both accessor and data descriptors');\n\t\t}\n\t\treturn true;\n\t}\n};\n\nmodule.exports = function assertRecord(Type, recordType, argumentName, value) {\n\tvar predicate = predicates[recordType];\n\tif (typeof predicate !== 'function') {\n\t\tthrow new $SyntaxError('unknown record type: ' + recordType);\n\t}\n\tif (!predicate(Type, value)) {\n\t\tthrow new $TypeError(argumentName + ' must be a ' + recordType);\n\t}\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0L2hlbHBlcnMvYXNzZXJ0UmVjb3JkLmpzP2M0NmQiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQWE7O0FBRWIsbUJBQW1CLG1CQUFPLENBQUMsR0FBZTs7QUFFMUM7QUFDQTs7QUFFQSxVQUFVLG1CQUFPLENBQUMsSUFBSzs7QUFFdkI7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTs7QUFFQSx5QkFBeUI7QUFDekI7QUFDQTtBQUNBO0FBQ0E7O0FBRUE7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTs7QUFFQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0EiLCJmaWxlIjoiNDE3OS5qcyIsInNvdXJjZXNDb250ZW50IjpbIid1c2Ugc3RyaWN0JztcblxudmFyIEdldEludHJpbnNpYyA9IHJlcXVpcmUoJ2dldC1pbnRyaW5zaWMnKTtcblxudmFyICRUeXBlRXJyb3IgPSBHZXRJbnRyaW5zaWMoJyVUeXBlRXJyb3IlJyk7XG52YXIgJFN5bnRheEVycm9yID0gR2V0SW50cmluc2ljKCclU3ludGF4RXJyb3IlJyk7XG5cbnZhciBoYXMgPSByZXF1aXJlKCdoYXMnKTtcblxudmFyIHByZWRpY2F0ZXMgPSB7XG5cdC8vIGh0dHBzOi8vMjYyLmVjbWEtaW50ZXJuYXRpb25hbC5vcmcvNi4wLyNzZWMtcHJvcGVydHktZGVzY3JpcHRvci1zcGVjaWZpY2F0aW9uLXR5cGVcblx0J1Byb3BlcnR5IERlc2NyaXB0b3InOiBmdW5jdGlvbiBpc1Byb3BlcnR5RGVzY3JpcHRvcihUeXBlLCBEZXNjKSB7XG5cdFx0aWYgKFR5cGUoRGVzYykgIT09ICdPYmplY3QnKSB7XG5cdFx0XHRyZXR1cm4gZmFsc2U7XG5cdFx0fVxuXHRcdHZhciBhbGxvd2VkID0ge1xuXHRcdFx0J1tbQ29uZmlndXJhYmxlXV0nOiB0cnVlLFxuXHRcdFx0J1tbRW51bWVyYWJsZV1dJzogdHJ1ZSxcblx0XHRcdCdbW0dldF1dJzogdHJ1ZSxcblx0XHRcdCdbW1NldF1dJzogdHJ1ZSxcblx0XHRcdCdbW1ZhbHVlXV0nOiB0cnVlLFxuXHRcdFx0J1tbV3JpdGFibGVdXSc6IHRydWVcblx0XHR9O1xuXG5cdFx0Zm9yICh2YXIga2V5IGluIERlc2MpIHsgLy8gZXNsaW50LWRpc2FibGUtbGluZVxuXHRcdFx0aWYgKGhhcyhEZXNjLCBrZXkpICYmICFhbGxvd2VkW2tleV0pIHtcblx0XHRcdFx0cmV0dXJuIGZhbHNlO1xuXHRcdFx0fVxuXHRcdH1cblxuXHRcdHZhciBpc0RhdGEgPSBoYXMoRGVzYywgJ1tbVmFsdWVdXScpO1xuXHRcdHZhciBJc0FjY2Vzc29yID0gaGFzKERlc2MsICdbW0dldF1dJykgfHwgaGFzKERlc2MsICdbW1NldF1dJyk7XG5cdFx0aWYgKGlzRGF0YSAmJiBJc0FjY2Vzc29yKSB7XG5cdFx0XHR0aHJvdyBuZXcgJFR5cGVFcnJvcignUHJvcGVydHkgRGVzY3JpcHRvcnMgbWF5IG5vdCBiZSBib3RoIGFjY2Vzc29yIGFuZCBkYXRhIGRlc2NyaXB0b3JzJyk7XG5cdFx0fVxuXHRcdHJldHVybiB0cnVlO1xuXHR9XG59O1xuXG5tb2R1bGUuZXhwb3J0cyA9IGZ1bmN0aW9uIGFzc2VydFJlY29yZChUeXBlLCByZWNvcmRUeXBlLCBhcmd1bWVudE5hbWUsIHZhbHVlKSB7XG5cdHZhciBwcmVkaWNhdGUgPSBwcmVkaWNhdGVzW3JlY29yZFR5cGVdO1xuXHRpZiAodHlwZW9mIHByZWRpY2F0ZSAhPT0gJ2Z1bmN0aW9uJykge1xuXHRcdHRocm93IG5ldyAkU3ludGF4RXJyb3IoJ3Vua25vd24gcmVjb3JkIHR5cGU6ICcgKyByZWNvcmRUeXBlKTtcblx0fVxuXHRpZiAoIXByZWRpY2F0ZShUeXBlLCB2YWx1ZSkpIHtcblx0XHR0aHJvdyBuZXcgJFR5cGVFcnJvcihhcmd1bWVudE5hbWUgKyAnIG11c3QgYmUgYSAnICsgcmVjb3JkVHlwZSk7XG5cdH1cbn07XG4iXSwic291cmNlUm9vdCI6IiJ9\n//# sourceURL=webpack-internal:///4179\n")},3060:module=>{"use strict";eval("\n\nvar $isNaN = Number.isNaN || function (a) { return a !== a; };\n\nmodule.exports = Number.isFinite || function (x) { return typeof x === 'number' && !$isNaN(x) && x !== Infinity && x !== -Infinity; };\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0L2hlbHBlcnMvaXNGaW5pdGUuanM/YzYxMiJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBYTs7QUFFYiwyQ0FBMkMsZ0JBQWdCOztBQUUzRCxrREFBa0QsaUZBQWlGIiwiZmlsZSI6IjMwNjAuanMiLCJzb3VyY2VzQ29udGVudCI6WyIndXNlIHN0cmljdCc7XG5cbnZhciAkaXNOYU4gPSBOdW1iZXIuaXNOYU4gfHwgZnVuY3Rpb24gKGEpIHsgcmV0dXJuIGEgIT09IGE7IH07XG5cbm1vZHVsZS5leHBvcnRzID0gTnVtYmVyLmlzRmluaXRlIHx8IGZ1bmN0aW9uICh4KSB7IHJldHVybiB0eXBlb2YgeCA9PT0gJ251bWJlcicgJiYgISRpc05hTih4KSAmJiB4ICE9PSBJbmZpbml0eSAmJiB4ICE9PSAtSW5maW5pdHk7IH07XG4iXSwic291cmNlUm9vdCI6IiJ9\n//# sourceURL=webpack-internal:///3060\n")},1996:module=>{"use strict";eval("\n\nmodule.exports = function isLeadingSurrogate(charCode) {\n\treturn typeof charCode === 'number' && charCode >= 0xD800 && charCode <= 0xDBFF;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0L2hlbHBlcnMvaXNMZWFkaW5nU3Vycm9nYXRlLmpzPzYzYmUiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQWE7O0FBRWI7QUFDQTtBQUNBIiwiZmlsZSI6IjE5OTYuanMiLCJzb3VyY2VzQ29udGVudCI6WyIndXNlIHN0cmljdCc7XG5cbm1vZHVsZS5leHBvcnRzID0gZnVuY3Rpb24gaXNMZWFkaW5nU3Vycm9nYXRlKGNoYXJDb2RlKSB7XG5cdHJldHVybiB0eXBlb2YgY2hhckNvZGUgPT09ICdudW1iZXInICYmIGNoYXJDb2RlID49IDB4RDgwMCAmJiBjaGFyQ29kZSA8PSAweERCRkY7XG59O1xuIl0sInNvdXJjZVJvb3QiOiIifQ==\n//# sourceURL=webpack-internal:///1996\n")},1214:module=>{"use strict";eval("\n\nmodule.exports = Number.isNaN || function isNaN(a) {\n\treturn a !== a;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0L2hlbHBlcnMvaXNOYU4uanM/MjA1NyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBYTs7QUFFYjtBQUNBO0FBQ0EiLCJmaWxlIjoiMTIxNC5qcyIsInNvdXJjZXNDb250ZW50IjpbIid1c2Ugc3RyaWN0JztcblxubW9kdWxlLmV4cG9ydHMgPSBOdW1iZXIuaXNOYU4gfHwgZnVuY3Rpb24gaXNOYU4oYSkge1xuXHRyZXR1cm4gYSAhPT0gYTtcbn07XG4iXSwic291cmNlUm9vdCI6IiJ9\n//# sourceURL=webpack-internal:///1214\n")},3126:module=>{"use strict";eval("\n\nmodule.exports = function isPrimitive(value) {\n\treturn value === null || (typeof value !== 'function' && typeof value !== 'object');\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0L2hlbHBlcnMvaXNQcmltaXRpdmUuanM/MTM1OCJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBYTs7QUFFYjtBQUNBO0FBQ0EiLCJmaWxlIjoiMzEyNi5qcyIsInNvdXJjZXNDb250ZW50IjpbIid1c2Ugc3RyaWN0JztcblxubW9kdWxlLmV4cG9ydHMgPSBmdW5jdGlvbiBpc1ByaW1pdGl2ZSh2YWx1ZSkge1xuXHRyZXR1cm4gdmFsdWUgPT09IG51bGwgfHwgKHR5cGVvZiB2YWx1ZSAhPT0gJ2Z1bmN0aW9uJyAmJiB0eXBlb2YgdmFsdWUgIT09ICdvYmplY3QnKTtcbn07XG4iXSwic291cmNlUm9vdCI6IiJ9\n//# sourceURL=webpack-internal:///3126\n")},9146:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\n\nvar has = __webpack_require__(7642);\nvar $TypeError = GetIntrinsic('%TypeError%');\n\nmodule.exports = function IsPropertyDescriptor(ES, Desc) {\n\tif (ES.Type(Desc) !== 'Object') {\n\t\treturn false;\n\t}\n\tvar allowed = {\n\t\t'[[Configurable]]': true,\n\t\t'[[Enumerable]]': true,\n\t\t'[[Get]]': true,\n\t\t'[[Set]]': true,\n\t\t'[[Value]]': true,\n\t\t'[[Writable]]': true\n\t};\n\n\tfor (var key in Desc) { // eslint-disable-line no-restricted-syntax\n\t\tif (has(Desc, key) && !allowed[key]) {\n\t\t\treturn false;\n\t\t}\n\t}\n\n\tif (ES.IsDataDescriptor(Desc) && ES.IsAccessorDescriptor(Desc)) {\n\t\tthrow new $TypeError('Property Descriptors may not be both accessor and data descriptors');\n\t}\n\treturn true;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0L2hlbHBlcnMvaXNQcm9wZXJ0eURlc2NyaXB0b3IuanM/ZmZmZCJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBYTs7QUFFYixtQkFBbUIsbUJBQU8sQ0FBQyxHQUFlOztBQUUxQyxVQUFVLG1CQUFPLENBQUMsSUFBSztBQUN2Qjs7QUFFQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7O0FBRUEsd0JBQXdCO0FBQ3hCO0FBQ0E7QUFDQTtBQUNBOztBQUVBO0FBQ0E7QUFDQTtBQUNBO0FBQ0EiLCJmaWxlIjoiOTE0Ni5qcyIsInNvdXJjZXNDb250ZW50IjpbIid1c2Ugc3RyaWN0JztcblxudmFyIEdldEludHJpbnNpYyA9IHJlcXVpcmUoJ2dldC1pbnRyaW5zaWMnKTtcblxudmFyIGhhcyA9IHJlcXVpcmUoJ2hhcycpO1xudmFyICRUeXBlRXJyb3IgPSBHZXRJbnRyaW5zaWMoJyVUeXBlRXJyb3IlJyk7XG5cbm1vZHVsZS5leHBvcnRzID0gZnVuY3Rpb24gSXNQcm9wZXJ0eURlc2NyaXB0b3IoRVMsIERlc2MpIHtcblx0aWYgKEVTLlR5cGUoRGVzYykgIT09ICdPYmplY3QnKSB7XG5cdFx0cmV0dXJuIGZhbHNlO1xuXHR9XG5cdHZhciBhbGxvd2VkID0ge1xuXHRcdCdbW0NvbmZpZ3VyYWJsZV1dJzogdHJ1ZSxcblx0XHQnW1tFbnVtZXJhYmxlXV0nOiB0cnVlLFxuXHRcdCdbW0dldF1dJzogdHJ1ZSxcblx0XHQnW1tTZXRdXSc6IHRydWUsXG5cdFx0J1tbVmFsdWVdXSc6IHRydWUsXG5cdFx0J1tbV3JpdGFibGVdXSc6IHRydWVcblx0fTtcblxuXHRmb3IgKHZhciBrZXkgaW4gRGVzYykgeyAvLyBlc2xpbnQtZGlzYWJsZS1saW5lIG5vLXJlc3RyaWN0ZWQtc3ludGF4XG5cdFx0aWYgKGhhcyhEZXNjLCBrZXkpICYmICFhbGxvd2VkW2tleV0pIHtcblx0XHRcdHJldHVybiBmYWxzZTtcblx0XHR9XG5cdH1cblxuXHRpZiAoRVMuSXNEYXRhRGVzY3JpcHRvcihEZXNjKSAmJiBFUy5Jc0FjY2Vzc29yRGVzY3JpcHRvcihEZXNjKSkge1xuXHRcdHRocm93IG5ldyAkVHlwZUVycm9yKCdQcm9wZXJ0eSBEZXNjcmlwdG9ycyBtYXkgbm90IGJlIGJvdGggYWNjZXNzb3IgYW5kIGRhdGEgZGVzY3JpcHRvcnMnKTtcblx0fVxuXHRyZXR1cm4gdHJ1ZTtcbn07XG4iXSwic291cmNlUm9vdCI6IiJ9\n//# sourceURL=webpack-internal:///9146\n")},6935:module=>{"use strict";eval("\n\nmodule.exports = function isTrailingSurrogate(charCode) {\n\treturn typeof charCode === 'number' && charCode >= 0xDC00 && charCode <= 0xDFFF;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0L2hlbHBlcnMvaXNUcmFpbGluZ1N1cnJvZ2F0ZS5qcz8wMTM1Il0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFhOztBQUViO0FBQ0E7QUFDQSIsImZpbGUiOiI2OTM1LmpzIiwic291cmNlc0NvbnRlbnQiOlsiJ3VzZSBzdHJpY3QnO1xuXG5tb2R1bGUuZXhwb3J0cyA9IGZ1bmN0aW9uIGlzVHJhaWxpbmdTdXJyb2dhdGUoY2hhckNvZGUpIHtcblx0cmV0dXJuIHR5cGVvZiBjaGFyQ29kZSA9PT0gJ251bWJlcicgJiYgY2hhckNvZGUgPj0gMHhEQzAwICYmIGNoYXJDb2RlIDw9IDB4REZGRjtcbn07XG4iXSwic291cmNlUm9vdCI6IiJ9\n//# sourceURL=webpack-internal:///6935\n")},5376:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\n\nvar $Math = GetIntrinsic('%Math%');\nvar $Number = GetIntrinsic('%Number%');\n\nmodule.exports = $Number.MAX_SAFE_INTEGER || $Math.pow(2, 53) - 1;\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0L2hlbHBlcnMvbWF4U2FmZUludGVnZXIuanM/YjJhNSJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBYTs7QUFFYixtQkFBbUIsbUJBQU8sQ0FBQyxHQUFlOztBQUUxQztBQUNBOztBQUVBIiwiZmlsZSI6IjUzNzYuanMiLCJzb3VyY2VzQ29udGVudCI6WyIndXNlIHN0cmljdCc7XG5cbnZhciBHZXRJbnRyaW5zaWMgPSByZXF1aXJlKCdnZXQtaW50cmluc2ljJyk7XG5cbnZhciAkTWF0aCA9IEdldEludHJpbnNpYygnJU1hdGglJyk7XG52YXIgJE51bWJlciA9IEdldEludHJpbnNpYygnJU51bWJlciUnKTtcblxubW9kdWxlLmV4cG9ydHMgPSAkTnVtYmVyLk1BWF9TQUZFX0lOVEVHRVIgfHwgJE1hdGgucG93KDIsIDUzKSAtIDE7XG4iXSwic291cmNlUm9vdCI6IiJ9\n//# sourceURL=webpack-internal:///5376\n")},1652:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\n\nvar $test = GetIntrinsic('RegExp.prototype.test');\n\nvar callBind = __webpack_require__(5559);\n\nmodule.exports = function regexTester(regex) {\n\treturn callBind($test, regex);\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0L2hlbHBlcnMvcmVnZXhUZXN0ZXIuanM/MGZhYSJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBYTs7QUFFYixtQkFBbUIsbUJBQU8sQ0FBQyxHQUFlOztBQUUxQzs7QUFFQSxlQUFlLG1CQUFPLENBQUMsSUFBVzs7QUFFbEM7QUFDQTtBQUNBIiwiZmlsZSI6IjE2NTIuanMiLCJzb3VyY2VzQ29udGVudCI6WyIndXNlIHN0cmljdCc7XG5cbnZhciBHZXRJbnRyaW5zaWMgPSByZXF1aXJlKCdnZXQtaW50cmluc2ljJyk7XG5cbnZhciAkdGVzdCA9IEdldEludHJpbnNpYygnUmVnRXhwLnByb3RvdHlwZS50ZXN0Jyk7XG5cbnZhciBjYWxsQmluZCA9IHJlcXVpcmUoJ2NhbGwtYmluZCcpO1xuXG5tb2R1bGUuZXhwb3J0cyA9IGZ1bmN0aW9uIHJlZ2V4VGVzdGVyKHJlZ2V4KSB7XG5cdHJldHVybiBjYWxsQmluZCgkdGVzdCwgcmVnZXgpO1xufTtcbiJdLCJzb3VyY2VSb290IjoiIn0=\n//# sourceURL=webpack-internal:///1652\n")},4099:module=>{"use strict";eval("\n\nmodule.exports = function sign(number) {\n\treturn number >= 0 ? 1 : -1;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLWFic3RyYWN0L2hlbHBlcnMvc2lnbi5qcz81OTc1Il0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFhOztBQUViO0FBQ0E7QUFDQSIsImZpbGUiOiI0MDk5LmpzIiwic291cmNlc0NvbnRlbnQiOlsiJ3VzZSBzdHJpY3QnO1xuXG5tb2R1bGUuZXhwb3J0cyA9IGZ1bmN0aW9uIHNpZ24obnVtYmVyKSB7XG5cdHJldHVybiBudW1iZXIgPj0gMCA/IDEgOiAtMTtcbn07XG4iXSwic291cmNlUm9vdCI6IiJ9\n//# sourceURL=webpack-internal:///4099\n")},1503:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar hasSymbols = typeof Symbol === 'function' && typeof Symbol.iterator === 'symbol';\n\nvar isPrimitive = __webpack_require__(4149);\nvar isCallable = __webpack_require__(5320);\nvar isDate = __webpack_require__(8923);\nvar isSymbol = __webpack_require__(2636);\n\nvar ordinaryToPrimitive = function OrdinaryToPrimitive(O, hint) {\n\tif (typeof O === 'undefined' || O === null) {\n\t\tthrow new TypeError('Cannot call method on ' + O);\n\t}\n\tif (typeof hint !== 'string' || (hint !== 'number' && hint !== 'string')) {\n\t\tthrow new TypeError('hint must be \"string\" or \"number\"');\n\t}\n\tvar methodNames = hint === 'string' ? ['toString', 'valueOf'] : ['valueOf', 'toString'];\n\tvar method, result, i;\n\tfor (i = 0; i < methodNames.length; ++i) {\n\t\tmethod = O[methodNames[i]];\n\t\tif (isCallable(method)) {\n\t\t\tresult = method.call(O);\n\t\t\tif (isPrimitive(result)) {\n\t\t\t\treturn result;\n\t\t\t}\n\t\t}\n\t}\n\tthrow new TypeError('No default value');\n};\n\nvar GetMethod = function GetMethod(O, P) {\n\tvar func = O[P];\n\tif (func !== null && typeof func !== 'undefined') {\n\t\tif (!isCallable(func)) {\n\t\t\tthrow new TypeError(func + ' returned for property ' + P + ' of object ' + O + ' is not a function');\n\t\t}\n\t\treturn func;\n\t}\n\treturn void 0;\n};\n\n// http://www.ecma-international.org/ecma-262/6.0/#sec-toprimitive\nmodule.exports = function ToPrimitive(input) {\n\tif (isPrimitive(input)) {\n\t\treturn input;\n\t}\n\tvar hint = 'default';\n\tif (arguments.length > 1) {\n\t\tif (arguments[1] === String) {\n\t\t\thint = 'string';\n\t\t} else if (arguments[1] === Number) {\n\t\t\thint = 'number';\n\t\t}\n\t}\n\n\tvar exoticToPrim;\n\tif (hasSymbols) {\n\t\tif (Symbol.toPrimitive) {\n\t\t\texoticToPrim = GetMethod(input, Symbol.toPrimitive);\n\t\t} else if (isSymbol(input)) {\n\t\t\texoticToPrim = Symbol.prototype.valueOf;\n\t\t}\n\t}\n\tif (typeof exoticToPrim !== 'undefined') {\n\t\tvar result = exoticToPrim.call(input, hint);\n\t\tif (isPrimitive(result)) {\n\t\t\treturn result;\n\t\t}\n\t\tthrow new TypeError('unable to convert exotic object to primitive');\n\t}\n\tif (hint === 'default' && (isDate(input) || isSymbol(input))) {\n\t\thint = 'string';\n\t}\n\treturn ordinaryToPrimitive(input, hint === 'default' ? 'number' : hint);\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///1503\n")},2116:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar toStr = Object.prototype.toString;\n\nvar isPrimitive = __webpack_require__(4149);\n\nvar isCallable = __webpack_require__(5320);\n\n// http://ecma-international.org/ecma-262/5.1/#sec-8.12.8\nvar ES5internalSlots = {\n\t'[[DefaultValue]]': function (O) {\n\t\tvar actualHint;\n\t\tif (arguments.length > 1) {\n\t\t\tactualHint = arguments[1];\n\t\t} else {\n\t\t\tactualHint = toStr.call(O) === '[object Date]' ? String : Number;\n\t\t}\n\n\t\tif (actualHint === String || actualHint === Number) {\n\t\t\tvar methods = actualHint === String ? ['toString', 'valueOf'] : ['valueOf', 'toString'];\n\t\t\tvar value, i;\n\t\t\tfor (i = 0; i < methods.length; ++i) {\n\t\t\t\tif (isCallable(O[methods[i]])) {\n\t\t\t\t\tvalue = O[methods[i]]();\n\t\t\t\t\tif (isPrimitive(value)) {\n\t\t\t\t\t\treturn value;\n\t\t\t\t\t}\n\t\t\t\t}\n\t\t\t}\n\t\t\tthrow new TypeError('No default value');\n\t\t}\n\t\tthrow new TypeError('invalid [[DefaultValue]] hint supplied');\n\t}\n};\n\n// http://ecma-international.org/ecma-262/5.1/#sec-9.1\nmodule.exports = function ToPrimitive(input) {\n\tif (isPrimitive(input)) {\n\t\treturn input;\n\t}\n\tif (arguments.length > 1) {\n\t\treturn ES5internalSlots['[[DefaultValue]]'](input, arguments[1]);\n\t}\n\treturn ES5internalSlots['[[DefaultValue]]'](input);\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLXRvLXByaW1pdGl2ZS9lczUuanM/MmYxNyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBYTs7QUFFYjs7QUFFQSxrQkFBa0IsbUJBQU8sQ0FBQyxJQUF1Qjs7QUFFakQsaUJBQWlCLG1CQUFPLENBQUMsSUFBYTs7QUFFdEM7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0EsR0FBRztBQUNIO0FBQ0E7O0FBRUE7QUFDQTtBQUNBO0FBQ0EsY0FBYyxvQkFBb0I7QUFDbEM7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBOztBQUVBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBIiwiZmlsZSI6IjIxMTYuanMiLCJzb3VyY2VzQ29udGVudCI6WyIndXNlIHN0cmljdCc7XG5cbnZhciB0b1N0ciA9IE9iamVjdC5wcm90b3R5cGUudG9TdHJpbmc7XG5cbnZhciBpc1ByaW1pdGl2ZSA9IHJlcXVpcmUoJy4vaGVscGVycy9pc1ByaW1pdGl2ZScpO1xuXG52YXIgaXNDYWxsYWJsZSA9IHJlcXVpcmUoJ2lzLWNhbGxhYmxlJyk7XG5cbi8vIGh0dHA6Ly9lY21hLWludGVybmF0aW9uYWwub3JnL2VjbWEtMjYyLzUuMS8jc2VjLTguMTIuOFxudmFyIEVTNWludGVybmFsU2xvdHMgPSB7XG5cdCdbW0RlZmF1bHRWYWx1ZV1dJzogZnVuY3Rpb24gKE8pIHtcblx0XHR2YXIgYWN0dWFsSGludDtcblx0XHRpZiAoYXJndW1lbnRzLmxlbmd0aCA+IDEpIHtcblx0XHRcdGFjdHVhbEhpbnQgPSBhcmd1bWVudHNbMV07XG5cdFx0fSBlbHNlIHtcblx0XHRcdGFjdHVhbEhpbnQgPSB0b1N0ci5jYWxsKE8pID09PSAnW29iamVjdCBEYXRlXScgPyBTdHJpbmcgOiBOdW1iZXI7XG5cdFx0fVxuXG5cdFx0aWYgKGFjdHVhbEhpbnQgPT09IFN0cmluZyB8fCBhY3R1YWxIaW50ID09PSBOdW1iZXIpIHtcblx0XHRcdHZhciBtZXRob2RzID0gYWN0dWFsSGludCA9PT0gU3RyaW5nID8gWyd0b1N0cmluZycsICd2YWx1ZU9mJ10gOiBbJ3ZhbHVlT2YnLCAndG9TdHJpbmcnXTtcblx0XHRcdHZhciB2YWx1ZSwgaTtcblx0XHRcdGZvciAoaSA9IDA7IGkgPCBtZXRob2RzLmxlbmd0aDsgKytpKSB7XG5cdFx0XHRcdGlmIChpc0NhbGxhYmxlKE9bbWV0aG9kc1tpXV0pKSB7XG5cdFx0XHRcdFx0dmFsdWUgPSBPW21ldGhvZHNbaV1dKCk7XG5cdFx0XHRcdFx0aWYgKGlzUHJpbWl0aXZlKHZhbHVlKSkge1xuXHRcdFx0XHRcdFx0cmV0dXJuIHZhbHVlO1xuXHRcdFx0XHRcdH1cblx0XHRcdFx0fVxuXHRcdFx0fVxuXHRcdFx0dGhyb3cgbmV3IFR5cGVFcnJvcignTm8gZGVmYXVsdCB2YWx1ZScpO1xuXHRcdH1cblx0XHR0aHJvdyBuZXcgVHlwZUVycm9yKCdpbnZhbGlkIFtbRGVmYXVsdFZhbHVlXV0gaGludCBzdXBwbGllZCcpO1xuXHR9XG59O1xuXG4vLyBodHRwOi8vZWNtYS1pbnRlcm5hdGlvbmFsLm9yZy9lY21hLTI2Mi81LjEvI3NlYy05LjFcbm1vZHVsZS5leHBvcnRzID0gZnVuY3Rpb24gVG9QcmltaXRpdmUoaW5wdXQpIHtcblx0aWYgKGlzUHJpbWl0aXZlKGlucHV0KSkge1xuXHRcdHJldHVybiBpbnB1dDtcblx0fVxuXHRpZiAoYXJndW1lbnRzLmxlbmd0aCA+IDEpIHtcblx0XHRyZXR1cm4gRVM1aW50ZXJuYWxTbG90c1snW1tEZWZhdWx0VmFsdWVdXSddKGlucHV0LCBhcmd1bWVudHNbMV0pO1xuXHR9XG5cdHJldHVybiBFUzVpbnRlcm5hbFNsb3RzWydbW0RlZmF1bHRWYWx1ZV1dJ10oaW5wdXQpO1xufTtcbiJdLCJzb3VyY2VSb290IjoiIn0=\n//# sourceURL=webpack-internal:///2116\n")},4149:module=>{"use strict";eval("\n\nmodule.exports = function isPrimitive(value) {\n\treturn value === null || (typeof value !== 'function' && typeof value !== 'object');\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2VzLXRvLXByaW1pdGl2ZS9oZWxwZXJzL2lzUHJpbWl0aXZlLmpzPzRkZTgiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQWE7O0FBRWI7QUFDQTtBQUNBIiwiZmlsZSI6IjQxNDkuanMiLCJzb3VyY2VzQ29udGVudCI6WyIndXNlIHN0cmljdCc7XG5cbm1vZHVsZS5leHBvcnRzID0gZnVuY3Rpb24gaXNQcmltaXRpdmUodmFsdWUpIHtcblx0cmV0dXJuIHZhbHVlID09PSBudWxsIHx8ICh0eXBlb2YgdmFsdWUgIT09ICdmdW5jdGlvbicgJiYgdHlwZW9mIHZhbHVlICE9PSAnb2JqZWN0Jyk7XG59O1xuIl0sInNvdXJjZVJvb3QiOiIifQ==\n//# sourceURL=webpack-internal:///4149\n")},7648:module=>{"use strict";eval("\n\n/* eslint no-invalid-this: 1 */\n\nvar ERROR_MESSAGE = 'Function.prototype.bind called on incompatible ';\nvar slice = Array.prototype.slice;\nvar toStr = Object.prototype.toString;\nvar funcType = '[object Function]';\n\nmodule.exports = function bind(that) {\n    var target = this;\n    if (typeof target !== 'function' || toStr.call(target) !== funcType) {\n        throw new TypeError(ERROR_MESSAGE + target);\n    }\n    var args = slice.call(arguments, 1);\n\n    var bound;\n    var binder = function () {\n        if (this instanceof bound) {\n            var result = target.apply(\n                this,\n                args.concat(slice.call(arguments))\n            );\n            if (Object(result) === result) {\n                return result;\n            }\n            return this;\n        } else {\n            return target.apply(\n                that,\n                args.concat(slice.call(arguments))\n            );\n        }\n    };\n\n    var boundLength = Math.max(0, target.length - args.length);\n    var boundArgs = [];\n    for (var i = 0; i < boundLength; i++) {\n        boundArgs.push('$' + i);\n    }\n\n    bound = Function('binder', 'return function (' + boundArgs.join(',') + '){ return binder.apply(this,arguments); }')(binder);\n\n    if (target.prototype) {\n        var Empty = function Empty() {};\n        Empty.prototype = target.prototype;\n        bound.prototype = new Empty();\n        Empty.prototype = null;\n    }\n\n    return bound;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///7648\n")},8612:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar implementation = __webpack_require__(7648);\n\nmodule.exports = Function.prototype.bind || implementation;\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2Z1bmN0aW9uLWJpbmQvaW5kZXguanM/MGY3YyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBYTs7QUFFYixxQkFBcUIsbUJBQU8sQ0FBQyxJQUFrQjs7QUFFL0MiLCJmaWxlIjoiODYxMi5qcyIsInNvdXJjZXNDb250ZW50IjpbIid1c2Ugc3RyaWN0JztcblxudmFyIGltcGxlbWVudGF0aW9uID0gcmVxdWlyZSgnLi9pbXBsZW1lbnRhdGlvbicpO1xuXG5tb2R1bGUuZXhwb3J0cyA9IEZ1bmN0aW9uLnByb3RvdHlwZS5iaW5kIHx8IGltcGxlbWVudGF0aW9uO1xuIl0sInNvdXJjZVJvb3QiOiIifQ==\n//# sourceURL=webpack-internal:///8612\n")},210:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar undefined;\n\nvar $SyntaxError = SyntaxError;\nvar $Function = Function;\nvar $TypeError = TypeError;\n\n// eslint-disable-next-line consistent-return\nvar getEvalledConstructor = function (expressionSyntax) {\n\ttry {\n\t\treturn $Function('\"use strict\"; return (' + expressionSyntax + ').constructor;')();\n\t} catch (e) {}\n};\n\nvar $gOPD = Object.getOwnPropertyDescriptor;\nif ($gOPD) {\n\ttry {\n\t\t$gOPD({}, '');\n\t} catch (e) {\n\t\t$gOPD = null; // this is IE 8, which has a broken gOPD\n\t}\n}\n\nvar throwTypeError = function () {\n\tthrow new $TypeError();\n};\nvar ThrowTypeError = $gOPD\n\t? (function () {\n\t\ttry {\n\t\t\t// eslint-disable-next-line no-unused-expressions, no-caller, no-restricted-properties\n\t\t\targuments.callee; // IE 8 does not throw here\n\t\t\treturn throwTypeError;\n\t\t} catch (calleeThrows) {\n\t\t\ttry {\n\t\t\t\t// IE 8 throws on Object.getOwnPropertyDescriptor(arguments, '')\n\t\t\t\treturn $gOPD(arguments, 'callee').get;\n\t\t\t} catch (gOPDthrows) {\n\t\t\t\treturn throwTypeError;\n\t\t\t}\n\t\t}\n\t}())\n\t: throwTypeError;\n\nvar hasSymbols = __webpack_require__(1405)();\n\nvar getProto = Object.getPrototypeOf || function (x) { return x.__proto__; }; // eslint-disable-line no-proto\n\nvar needsEval = {};\n\nvar TypedArray = typeof Uint8Array === 'undefined' ? undefined : getProto(Uint8Array);\n\nvar INTRINSICS = {\n\t'%AggregateError%': typeof AggregateError === 'undefined' ? undefined : AggregateError,\n\t'%Array%': Array,\n\t'%ArrayBuffer%': typeof ArrayBuffer === 'undefined' ? undefined : ArrayBuffer,\n\t'%ArrayIteratorPrototype%': hasSymbols ? getProto([][Symbol.iterator]()) : undefined,\n\t'%AsyncFromSyncIteratorPrototype%': undefined,\n\t'%AsyncFunction%': needsEval,\n\t'%AsyncGenerator%': needsEval,\n\t'%AsyncGeneratorFunction%': needsEval,\n\t'%AsyncIteratorPrototype%': needsEval,\n\t'%Atomics%': typeof Atomics === 'undefined' ? undefined : Atomics,\n\t'%BigInt%': typeof BigInt === 'undefined' ? undefined : BigInt,\n\t'%Boolean%': Boolean,\n\t'%DataView%': typeof DataView === 'undefined' ? undefined : DataView,\n\t'%Date%': Date,\n\t'%decodeURI%': decodeURI,\n\t'%decodeURIComponent%': decodeURIComponent,\n\t'%encodeURI%': encodeURI,\n\t'%encodeURIComponent%': encodeURIComponent,\n\t'%Error%': Error,\n\t'%eval%': eval, // eslint-disable-line no-eval\n\t'%EvalError%': EvalError,\n\t'%Float32Array%': typeof Float32Array === 'undefined' ? undefined : Float32Array,\n\t'%Float64Array%': typeof Float64Array === 'undefined' ? undefined : Float64Array,\n\t'%FinalizationRegistry%': typeof FinalizationRegistry === 'undefined' ? undefined : FinalizationRegistry,\n\t'%Function%': $Function,\n\t'%GeneratorFunction%': needsEval,\n\t'%Int8Array%': typeof Int8Array === 'undefined' ? undefined : Int8Array,\n\t'%Int16Array%': typeof Int16Array === 'undefined' ? undefined : Int16Array,\n\t'%Int32Array%': typeof Int32Array === 'undefined' ? undefined : Int32Array,\n\t'%isFinite%': isFinite,\n\t'%isNaN%': isNaN,\n\t'%IteratorPrototype%': hasSymbols ? getProto(getProto([][Symbol.iterator]())) : undefined,\n\t'%JSON%': typeof JSON === 'object' ? JSON : undefined,\n\t'%Map%': typeof Map === 'undefined' ? undefined : Map,\n\t'%MapIteratorPrototype%': typeof Map === 'undefined' || !hasSymbols ? undefined : getProto(new Map()[Symbol.iterator]()),\n\t'%Math%': Math,\n\t'%Number%': Number,\n\t'%Object%': Object,\n\t'%parseFloat%': parseFloat,\n\t'%parseInt%': parseInt,\n\t'%Promise%': typeof Promise === 'undefined' ? undefined : Promise,\n\t'%Proxy%': typeof Proxy === 'undefined' ? undefined : Proxy,\n\t'%RangeError%': RangeError,\n\t'%ReferenceError%': ReferenceError,\n\t'%Reflect%': typeof Reflect === 'undefined' ? undefined : Reflect,\n\t'%RegExp%': RegExp,\n\t'%Set%': typeof Set === 'undefined' ? undefined : Set,\n\t'%SetIteratorPrototype%': typeof Set === 'undefined' || !hasSymbols ? undefined : getProto(new Set()[Symbol.iterator]()),\n\t'%SharedArrayBuffer%': typeof SharedArrayBuffer === 'undefined' ? undefined : SharedArrayBuffer,\n\t'%String%': String,\n\t'%StringIteratorPrototype%': hasSymbols ? getProto(''[Symbol.iterator]()) : undefined,\n\t'%Symbol%': hasSymbols ? Symbol : undefined,\n\t'%SyntaxError%': $SyntaxError,\n\t'%ThrowTypeError%': ThrowTypeError,\n\t'%TypedArray%': TypedArray,\n\t'%TypeError%': $TypeError,\n\t'%Uint8Array%': typeof Uint8Array === 'undefined' ? undefined : Uint8Array,\n\t'%Uint8ClampedArray%': typeof Uint8ClampedArray === 'undefined' ? undefined : Uint8ClampedArray,\n\t'%Uint16Array%': typeof Uint16Array === 'undefined' ? undefined : Uint16Array,\n\t'%Uint32Array%': typeof Uint32Array === 'undefined' ? undefined : Uint32Array,\n\t'%URIError%': URIError,\n\t'%WeakMap%': typeof WeakMap === 'undefined' ? undefined : WeakMap,\n\t'%WeakRef%': typeof WeakRef === 'undefined' ? undefined : WeakRef,\n\t'%WeakSet%': typeof WeakSet === 'undefined' ? undefined : WeakSet\n};\n\nvar doEval = function doEval(name) {\n\tvar value;\n\tif (name === '%AsyncFunction%') {\n\t\tvalue = getEvalledConstructor('async function () {}');\n\t} else if (name === '%GeneratorFunction%') {\n\t\tvalue = getEvalledConstructor('function* () {}');\n\t} else if (name === '%AsyncGeneratorFunction%') {\n\t\tvalue = getEvalledConstructor('async function* () {}');\n\t} else if (name === '%AsyncGenerator%') {\n\t\tvar fn = doEval('%AsyncGeneratorFunction%');\n\t\tif (fn) {\n\t\t\tvalue = fn.prototype;\n\t\t}\n\t} else if (name === '%AsyncIteratorPrototype%') {\n\t\tvar gen = doEval('%AsyncGenerator%');\n\t\tif (gen) {\n\t\t\tvalue = getProto(gen.prototype);\n\t\t}\n\t}\n\n\tINTRINSICS[name] = value;\n\n\treturn value;\n};\n\nvar LEGACY_ALIASES = {\n\t'%ArrayBufferPrototype%': ['ArrayBuffer', 'prototype'],\n\t'%ArrayPrototype%': ['Array', 'prototype'],\n\t'%ArrayProto_entries%': ['Array', 'prototype', 'entries'],\n\t'%ArrayProto_forEach%': ['Array', 'prototype', 'forEach'],\n\t'%ArrayProto_keys%': ['Array', 'prototype', 'keys'],\n\t'%ArrayProto_values%': ['Array', 'prototype', 'values'],\n\t'%AsyncFunctionPrototype%': ['AsyncFunction', 'prototype'],\n\t'%AsyncGenerator%': ['AsyncGeneratorFunction', 'prototype'],\n\t'%AsyncGeneratorPrototype%': ['AsyncGeneratorFunction', 'prototype', 'prototype'],\n\t'%BooleanPrototype%': ['Boolean', 'prototype'],\n\t'%DataViewPrototype%': ['DataView', 'prototype'],\n\t'%DatePrototype%': ['Date', 'prototype'],\n\t'%ErrorPrototype%': ['Error', 'prototype'],\n\t'%EvalErrorPrototype%': ['EvalError', 'prototype'],\n\t'%Float32ArrayPrototype%': ['Float32Array', 'prototype'],\n\t'%Float64ArrayPrototype%': ['Float64Array', 'prototype'],\n\t'%FunctionPrototype%': ['Function', 'prototype'],\n\t'%Generator%': ['GeneratorFunction', 'prototype'],\n\t'%GeneratorPrototype%': ['GeneratorFunction', 'prototype', 'prototype'],\n\t'%Int8ArrayPrototype%': ['Int8Array', 'prototype'],\n\t'%Int16ArrayPrototype%': ['Int16Array', 'prototype'],\n\t'%Int32ArrayPrototype%': ['Int32Array', 'prototype'],\n\t'%JSONParse%': ['JSON', 'parse'],\n\t'%JSONStringify%': ['JSON', 'stringify'],\n\t'%MapPrototype%': ['Map', 'prototype'],\n\t'%NumberPrototype%': ['Number', 'prototype'],\n\t'%ObjectPrototype%': ['Object', 'prototype'],\n\t'%ObjProto_toString%': ['Object', 'prototype', 'toString'],\n\t'%ObjProto_valueOf%': ['Object', 'prototype', 'valueOf'],\n\t'%PromisePrototype%': ['Promise', 'prototype'],\n\t'%PromiseProto_then%': ['Promise', 'prototype', 'then'],\n\t'%Promise_all%': ['Promise', 'all'],\n\t'%Promise_reject%': ['Promise', 'reject'],\n\t'%Promise_resolve%': ['Promise', 'resolve'],\n\t'%RangeErrorPrototype%': ['RangeError', 'prototype'],\n\t'%ReferenceErrorPrototype%': ['ReferenceError', 'prototype'],\n\t'%RegExpPrototype%': ['RegExp', 'prototype'],\n\t'%SetPrototype%': ['Set', 'prototype'],\n\t'%SharedArrayBufferPrototype%': ['SharedArrayBuffer', 'prototype'],\n\t'%StringPrototype%': ['String', 'prototype'],\n\t'%SymbolPrototype%': ['Symbol', 'prototype'],\n\t'%SyntaxErrorPrototype%': ['SyntaxError', 'prototype'],\n\t'%TypedArrayPrototype%': ['TypedArray', 'prototype'],\n\t'%TypeErrorPrototype%': ['TypeError', 'prototype'],\n\t'%Uint8ArrayPrototype%': ['Uint8Array', 'prototype'],\n\t'%Uint8ClampedArrayPrototype%': ['Uint8ClampedArray', 'prototype'],\n\t'%Uint16ArrayPrototype%': ['Uint16Array', 'prototype'],\n\t'%Uint32ArrayPrototype%': ['Uint32Array', 'prototype'],\n\t'%URIErrorPrototype%': ['URIError', 'prototype'],\n\t'%WeakMapPrototype%': ['WeakMap', 'prototype'],\n\t'%WeakSetPrototype%': ['WeakSet', 'prototype']\n};\n\nvar bind = __webpack_require__(8612);\nvar hasOwn = __webpack_require__(7642);\nvar $concat = bind.call(Function.call, Array.prototype.concat);\nvar $spliceApply = bind.call(Function.apply, Array.prototype.splice);\nvar $replace = bind.call(Function.call, String.prototype.replace);\nvar $strSlice = bind.call(Function.call, String.prototype.slice);\n\n/* adapted from https://github.com/lodash/lodash/blob/4.17.15/dist/lodash.js#L6735-L6744 */\nvar rePropName = /[^%.[\\]]+|\\[(?:(-?\\d+(?:\\.\\d+)?)|([\"'])((?:(?!\\2)[^\\\\]|\\\\.)*?)\\2)\\]|(?=(?:\\.|\\[\\])(?:\\.|\\[\\]|%$))/g;\nvar reEscapeChar = /\\\\(\\\\)?/g; /** Used to match backslashes in property paths. */\nvar stringToPath = function stringToPath(string) {\n\tvar first = $strSlice(string, 0, 1);\n\tvar last = $strSlice(string, -1);\n\tif (first === '%' && last !== '%') {\n\t\tthrow new $SyntaxError('invalid intrinsic syntax, expected closing `%`');\n\t} else if (last === '%' && first !== '%') {\n\t\tthrow new $SyntaxError('invalid intrinsic syntax, expected opening `%`');\n\t}\n\tvar result = [];\n\t$replace(string, rePropName, function (match, number, quote, subString) {\n\t\tresult[result.length] = quote ? $replace(subString, reEscapeChar, '$1') : number || match;\n\t});\n\treturn result;\n};\n/* end adaptation */\n\nvar getBaseIntrinsic = function getBaseIntrinsic(name, allowMissing) {\n\tvar intrinsicName = name;\n\tvar alias;\n\tif (hasOwn(LEGACY_ALIASES, intrinsicName)) {\n\t\talias = LEGACY_ALIASES[intrinsicName];\n\t\tintrinsicName = '%' + alias[0] + '%';\n\t}\n\n\tif (hasOwn(INTRINSICS, intrinsicName)) {\n\t\tvar value = INTRINSICS[intrinsicName];\n\t\tif (value === needsEval) {\n\t\t\tvalue = doEval(intrinsicName);\n\t\t}\n\t\tif (typeof value === 'undefined' && !allowMissing) {\n\t\t\tthrow new $TypeError('intrinsic ' + name + ' exists, but is not available. Please file an issue!');\n\t\t}\n\n\t\treturn {\n\t\t\talias: alias,\n\t\t\tname: intrinsicName,\n\t\t\tvalue: value\n\t\t};\n\t}\n\n\tthrow new $SyntaxError('intrinsic ' + name + ' does not exist!');\n};\n\nmodule.exports = function GetIntrinsic(name, allowMissing) {\n\tif (typeof name !== 'string' || name.length === 0) {\n\t\tthrow new $TypeError('intrinsic name must be a non-empty string');\n\t}\n\tif (arguments.length > 1 && typeof allowMissing !== 'boolean') {\n\t\tthrow new $TypeError('\"allowMissing\" argument must be a boolean');\n\t}\n\n\tvar parts = stringToPath(name);\n\tvar intrinsicBaseName = parts.length > 0 ? parts[0] : '';\n\n\tvar intrinsic = getBaseIntrinsic('%' + intrinsicBaseName + '%', allowMissing);\n\tvar intrinsicRealName = intrinsic.name;\n\tvar value = intrinsic.value;\n\tvar skipFurtherCaching = false;\n\n\tvar alias = intrinsic.alias;\n\tif (alias) {\n\t\tintrinsicBaseName = alias[0];\n\t\t$spliceApply(parts, $concat([0, 1], alias));\n\t}\n\n\tfor (var i = 1, isOwn = true; i < parts.length; i += 1) {\n\t\tvar part = parts[i];\n\t\tvar first = $strSlice(part, 0, 1);\n\t\tvar last = $strSlice(part, -1);\n\t\tif (\n\t\t\t(\n\t\t\t\t(first === '\"' || first === \"'\" || first === '`')\n\t\t\t\t|| (last === '\"' || last === \"'\" || last === '`')\n\t\t\t)\n\t\t\t&& first !== last\n\t\t) {\n\t\t\tthrow new $SyntaxError('property names with quotes must have matching quotes');\n\t\t}\n\t\tif (part === 'constructor' || !isOwn) {\n\t\t\tskipFurtherCaching = true;\n\t\t}\n\n\t\tintrinsicBaseName += '.' + part;\n\t\tintrinsicRealName = '%' + intrinsicBaseName + '%';\n\n\t\tif (hasOwn(INTRINSICS, intrinsicRealName)) {\n\t\t\tvalue = INTRINSICS[intrinsicRealName];\n\t\t} else if (value != null) {\n\t\t\tif (!(part in value)) {\n\t\t\t\tif (!allowMissing) {\n\t\t\t\t\tthrow new $TypeError('base intrinsic for ' + name + ' exists, but the property is not available.');\n\t\t\t\t}\n\t\t\t\treturn void undefined;\n\t\t\t}\n\t\t\tif ($gOPD && (i + 1) >= parts.length) {\n\t\t\t\tvar desc = $gOPD(value, part);\n\t\t\t\tisOwn = !!desc;\n\n\t\t\t\t// By convention, when a data property is converted to an accessor\n\t\t\t\t// property to emulate a data property that does not suffer from\n\t\t\t\t// the override mistake, that accessor's getter is marked with\n\t\t\t\t// an `originalValue` property. Here, when we detect this, we\n\t\t\t\t// uphold the illusion by pretending to see that original data\n\t\t\t\t// property, i.e., returning the value rather than the getter\n\t\t\t\t// itself.\n\t\t\t\tif (isOwn && 'get' in desc && !('originalValue' in desc.get)) {\n\t\t\t\t\tvalue = desc.get;\n\t\t\t\t} else {\n\t\t\t\t\tvalue = value[part];\n\t\t\t\t}\n\t\t\t} else {\n\t\t\t\tisOwn = hasOwn(value, part);\n\t\t\t\tvalue = value[part];\n\t\t\t}\n\n\t\t\tif (isOwn && !skipFurtherCaching) {\n\t\t\t\tINTRINSICS[intrinsicRealName] = value;\n\t\t\t}\n\t\t}\n\t}\n\treturn value;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///210\n")},1405:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar origSymbol = typeof Symbol !== 'undefined' && Symbol;\nvar hasSymbolSham = __webpack_require__(5419);\n\nmodule.exports = function hasNativeSymbols() {\n\tif (typeof origSymbol !== 'function') { return false; }\n\tif (typeof Symbol !== 'function') { return false; }\n\tif (typeof origSymbol('foo') !== 'symbol') { return false; }\n\tif (typeof Symbol('bar') !== 'symbol') { return false; }\n\n\treturn hasSymbolSham();\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2hhcy1zeW1ib2xzL2luZGV4LmpzPzUxNTYiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQWE7O0FBRWI7QUFDQSxvQkFBb0IsbUJBQU8sQ0FBQyxJQUFTOztBQUVyQztBQUNBLHdDQUF3QyxjQUFjO0FBQ3RELG9DQUFvQyxjQUFjO0FBQ2xELDZDQUE2QyxjQUFjO0FBQzNELHlDQUF5QyxjQUFjOztBQUV2RDtBQUNBIiwiZmlsZSI6IjE0MDUuanMiLCJzb3VyY2VzQ29udGVudCI6WyIndXNlIHN0cmljdCc7XG5cbnZhciBvcmlnU3ltYm9sID0gdHlwZW9mIFN5bWJvbCAhPT0gJ3VuZGVmaW5lZCcgJiYgU3ltYm9sO1xudmFyIGhhc1N5bWJvbFNoYW0gPSByZXF1aXJlKCcuL3NoYW1zJyk7XG5cbm1vZHVsZS5leHBvcnRzID0gZnVuY3Rpb24gaGFzTmF0aXZlU3ltYm9scygpIHtcblx0aWYgKHR5cGVvZiBvcmlnU3ltYm9sICE9PSAnZnVuY3Rpb24nKSB7IHJldHVybiBmYWxzZTsgfVxuXHRpZiAodHlwZW9mIFN5bWJvbCAhPT0gJ2Z1bmN0aW9uJykgeyByZXR1cm4gZmFsc2U7IH1cblx0aWYgKHR5cGVvZiBvcmlnU3ltYm9sKCdmb28nKSAhPT0gJ3N5bWJvbCcpIHsgcmV0dXJuIGZhbHNlOyB9XG5cdGlmICh0eXBlb2YgU3ltYm9sKCdiYXInKSAhPT0gJ3N5bWJvbCcpIHsgcmV0dXJuIGZhbHNlOyB9XG5cblx0cmV0dXJuIGhhc1N5bWJvbFNoYW0oKTtcbn07XG4iXSwic291cmNlUm9vdCI6IiJ9\n//# sourceURL=webpack-internal:///1405\n")},5419:module=>{"use strict";eval("\n\n/* eslint complexity: [2, 18], max-statements: [2, 33] */\nmodule.exports = function hasSymbols() {\n\tif (typeof Symbol !== 'function' || typeof Object.getOwnPropertySymbols !== 'function') { return false; }\n\tif (typeof Symbol.iterator === 'symbol') { return true; }\n\n\tvar obj = {};\n\tvar sym = Symbol('test');\n\tvar symObj = Object(sym);\n\tif (typeof sym === 'string') { return false; }\n\n\tif (Object.prototype.toString.call(sym) !== '[object Symbol]') { return false; }\n\tif (Object.prototype.toString.call(symObj) !== '[object Symbol]') { return false; }\n\n\t// temp disabled per https://github.com/ljharb/object.assign/issues/17\n\t// if (sym instanceof Symbol) { return false; }\n\t// temp disabled per https://github.com/WebReflection/get-own-property-symbols/issues/4\n\t// if (!(symObj instanceof Symbol)) { return false; }\n\n\t// if (typeof Symbol.prototype.toString !== 'function') { return false; }\n\t// if (String(sym) !== Symbol.prototype.toString.call(sym)) { return false; }\n\n\tvar symVal = 42;\n\tobj[sym] = symVal;\n\tfor (sym in obj) { return false; } // eslint-disable-line no-restricted-syntax, no-unreachable-loop\n\tif (typeof Object.keys === 'function' && Object.keys(obj).length !== 0) { return false; }\n\n\tif (typeof Object.getOwnPropertyNames === 'function' && Object.getOwnPropertyNames(obj).length !== 0) { return false; }\n\n\tvar syms = Object.getOwnPropertySymbols(obj);\n\tif (syms.length !== 1 || syms[0] !== sym) { return false; }\n\n\tif (!Object.prototype.propertyIsEnumerable.call(obj, sym)) { return false; }\n\n\tif (typeof Object.getOwnPropertyDescriptor === 'function') {\n\t\tvar descriptor = Object.getOwnPropertyDescriptor(obj, sym);\n\t\tif (descriptor.value !== symVal || descriptor.enumerable !== true) { return false; }\n\t}\n\n\treturn true;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2hhcy1zeW1ib2xzL3NoYW1zLmpzPzE2OTYiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQWE7O0FBRWI7QUFDQTtBQUNBLDBGQUEwRixjQUFjO0FBQ3hHLDJDQUEyQyxhQUFhOztBQUV4RDtBQUNBO0FBQ0E7QUFDQSwrQkFBK0IsY0FBYzs7QUFFN0MsaUVBQWlFLGNBQWM7QUFDL0Usb0VBQW9FLGNBQWM7O0FBRWxGO0FBQ0EsZ0NBQWdDLGNBQWM7QUFDOUM7QUFDQSxzQ0FBc0MsY0FBYzs7QUFFcEQsMERBQTBELGNBQWM7QUFDeEUsOERBQThELGNBQWM7O0FBRTVFO0FBQ0E7QUFDQSxtQkFBbUIsY0FBYyxFQUFFO0FBQ25DLDBFQUEwRSxjQUFjOztBQUV4Rix3R0FBd0csY0FBYzs7QUFFdEg7QUFDQSw0Q0FBNEMsY0FBYzs7QUFFMUQsNkRBQTZELGNBQWM7O0FBRTNFO0FBQ0E7QUFDQSxzRUFBc0UsY0FBYztBQUNwRjs7QUFFQTtBQUNBIiwiZmlsZSI6IjU0MTkuanMiLCJzb3VyY2VzQ29udGVudCI6WyIndXNlIHN0cmljdCc7XG5cbi8qIGVzbGludCBjb21wbGV4aXR5OiBbMiwgMThdLCBtYXgtc3RhdGVtZW50czogWzIsIDMzXSAqL1xubW9kdWxlLmV4cG9ydHMgPSBmdW5jdGlvbiBoYXNTeW1ib2xzKCkge1xuXHRpZiAodHlwZW9mIFN5bWJvbCAhPT0gJ2Z1bmN0aW9uJyB8fCB0eXBlb2YgT2JqZWN0LmdldE93blByb3BlcnR5U3ltYm9scyAhPT0gJ2Z1bmN0aW9uJykgeyByZXR1cm4gZmFsc2U7IH1cblx0aWYgKHR5cGVvZiBTeW1ib2wuaXRlcmF0b3IgPT09ICdzeW1ib2wnKSB7IHJldHVybiB0cnVlOyB9XG5cblx0dmFyIG9iaiA9IHt9O1xuXHR2YXIgc3ltID0gU3ltYm9sKCd0ZXN0Jyk7XG5cdHZhciBzeW1PYmogPSBPYmplY3Qoc3ltKTtcblx0aWYgKHR5cGVvZiBzeW0gPT09ICdzdHJpbmcnKSB7IHJldHVybiBmYWxzZTsgfVxuXG5cdGlmIChPYmplY3QucHJvdG90eXBlLnRvU3RyaW5nLmNhbGwoc3ltKSAhPT0gJ1tvYmplY3QgU3ltYm9sXScpIHsgcmV0dXJuIGZhbHNlOyB9XG5cdGlmIChPYmplY3QucHJvdG90eXBlLnRvU3RyaW5nLmNhbGwoc3ltT2JqKSAhPT0gJ1tvYmplY3QgU3ltYm9sXScpIHsgcmV0dXJuIGZhbHNlOyB9XG5cblx0Ly8gdGVtcCBkaXNhYmxlZCBwZXIgaHR0cHM6Ly9naXRodWIuY29tL2xqaGFyYi9vYmplY3QuYXNzaWduL2lzc3Vlcy8xN1xuXHQvLyBpZiAoc3ltIGluc3RhbmNlb2YgU3ltYm9sKSB7IHJldHVybiBmYWxzZTsgfVxuXHQvLyB0ZW1wIGRpc2FibGVkIHBlciBodHRwczovL2dpdGh1Yi5jb20vV2ViUmVmbGVjdGlvbi9nZXQtb3duLXByb3BlcnR5LXN5bWJvbHMvaXNzdWVzLzRcblx0Ly8gaWYgKCEoc3ltT2JqIGluc3RhbmNlb2YgU3ltYm9sKSkgeyByZXR1cm4gZmFsc2U7IH1cblxuXHQvLyBpZiAodHlwZW9mIFN5bWJvbC5wcm90b3R5cGUudG9TdHJpbmcgIT09ICdmdW5jdGlvbicpIHsgcmV0dXJuIGZhbHNlOyB9XG5cdC8vIGlmIChTdHJpbmcoc3ltKSAhPT0gU3ltYm9sLnByb3RvdHlwZS50b1N0cmluZy5jYWxsKHN5bSkpIHsgcmV0dXJuIGZhbHNlOyB9XG5cblx0dmFyIHN5bVZhbCA9IDQyO1xuXHRvYmpbc3ltXSA9IHN5bVZhbDtcblx0Zm9yIChzeW0gaW4gb2JqKSB7IHJldHVybiBmYWxzZTsgfSAvLyBlc2xpbnQtZGlzYWJsZS1saW5lIG5vLXJlc3RyaWN0ZWQtc3ludGF4LCBuby11bnJlYWNoYWJsZS1sb29wXG5cdGlmICh0eXBlb2YgT2JqZWN0LmtleXMgPT09ICdmdW5jdGlvbicgJiYgT2JqZWN0LmtleXMob2JqKS5sZW5ndGggIT09IDApIHsgcmV0dXJuIGZhbHNlOyB9XG5cblx0aWYgKHR5cGVvZiBPYmplY3QuZ2V0T3duUHJvcGVydHlOYW1lcyA9PT0gJ2Z1bmN0aW9uJyAmJiBPYmplY3QuZ2V0T3duUHJvcGVydHlOYW1lcyhvYmopLmxlbmd0aCAhPT0gMCkgeyByZXR1cm4gZmFsc2U7IH1cblxuXHR2YXIgc3ltcyA9IE9iamVjdC5nZXRPd25Qcm9wZXJ0eVN5bWJvbHMob2JqKTtcblx0aWYgKHN5bXMubGVuZ3RoICE9PSAxIHx8IHN5bXNbMF0gIT09IHN5bSkgeyByZXR1cm4gZmFsc2U7IH1cblxuXHRpZiAoIU9iamVjdC5wcm90b3R5cGUucHJvcGVydHlJc0VudW1lcmFibGUuY2FsbChvYmosIHN5bSkpIHsgcmV0dXJuIGZhbHNlOyB9XG5cblx0aWYgKHR5cGVvZiBPYmplY3QuZ2V0T3duUHJvcGVydHlEZXNjcmlwdG9yID09PSAnZnVuY3Rpb24nKSB7XG5cdFx0dmFyIGRlc2NyaXB0b3IgPSBPYmplY3QuZ2V0T3duUHJvcGVydHlEZXNjcmlwdG9yKG9iaiwgc3ltKTtcblx0XHRpZiAoZGVzY3JpcHRvci52YWx1ZSAhPT0gc3ltVmFsIHx8IGRlc2NyaXB0b3IuZW51bWVyYWJsZSAhPT0gdHJ1ZSkgeyByZXR1cm4gZmFsc2U7IH1cblx0fVxuXG5cdHJldHVybiB0cnVlO1xufTtcbiJdLCJzb3VyY2VSb290IjoiIn0=\n//# sourceURL=webpack-internal:///5419\n")},6410:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar hasSymbols = __webpack_require__(5419);\n\nmodule.exports = function hasToStringTagShams() {\n\treturn hasSymbols() && !!Symbol.toStringTag;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2hhcy10b3N0cmluZ3RhZy9zaGFtcy5qcz8wN2E0Il0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFhOztBQUViLGlCQUFpQixtQkFBTyxDQUFDLElBQW1COztBQUU1QztBQUNBO0FBQ0EiLCJmaWxlIjoiNjQxMC5qcyIsInNvdXJjZXNDb250ZW50IjpbIid1c2Ugc3RyaWN0JztcblxudmFyIGhhc1N5bWJvbHMgPSByZXF1aXJlKCdoYXMtc3ltYm9scy9zaGFtcycpO1xuXG5tb2R1bGUuZXhwb3J0cyA9IGZ1bmN0aW9uIGhhc1RvU3RyaW5nVGFnU2hhbXMoKSB7XG5cdHJldHVybiBoYXNTeW1ib2xzKCkgJiYgISFTeW1ib2wudG9TdHJpbmdUYWc7XG59O1xuIl0sInNvdXJjZVJvb3QiOiIifQ==\n//# sourceURL=webpack-internal:///6410\n")},7642:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar bind = __webpack_require__(8612);\n\nmodule.exports = bind.call(Function.call, Object.prototype.hasOwnProperty);\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2hhcy9zcmMvaW5kZXguanM/YTBkMyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBYTs7QUFFYixXQUFXLG1CQUFPLENBQUMsSUFBZTs7QUFFbEMiLCJmaWxlIjoiNzY0Mi5qcyIsInNvdXJjZXNDb250ZW50IjpbIid1c2Ugc3RyaWN0JztcblxudmFyIGJpbmQgPSByZXF1aXJlKCdmdW5jdGlvbi1iaW5kJyk7XG5cbm1vZHVsZS5leHBvcnRzID0gYmluZC5jYWxsKEZ1bmN0aW9uLmNhbGwsIE9iamVjdC5wcm90b3R5cGUuaGFzT3duUHJvcGVydHkpO1xuIl0sInNvdXJjZVJvb3QiOiIifQ==\n//# sourceURL=webpack-internal:///7642\n")},3715:(__unused_webpack_module,exports,__webpack_require__)=>{eval("var hash = exports;\n\nhash.utils = __webpack_require__(6436);\nhash.common = __webpack_require__(5772);\nhash.sha = __webpack_require__(9041);\nhash.ripemd = __webpack_require__(2949);\nhash.hmac = __webpack_require__(2344);\n\n// Proxy hash functions to the main object\nhash.sha1 = hash.sha.sha1;\nhash.sha256 = hash.sha.sha256;\nhash.sha224 = hash.sha.sha224;\nhash.sha384 = hash.sha.sha384;\nhash.sha512 = hash.sha.sha512;\nhash.ripemd160 = hash.ripemd.ripemd160;\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2hhc2guanMvbGliL2hhc2guanM/N2Q5MiJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBQTs7QUFFQSxhQUFhLG1CQUFPLENBQUMsSUFBYztBQUNuQyxjQUFjLG1CQUFPLENBQUMsSUFBZTtBQUNyQyxXQUFXLG1CQUFPLENBQUMsSUFBWTtBQUMvQixjQUFjLG1CQUFPLENBQUMsSUFBZTtBQUNyQyxZQUFZLG1CQUFPLENBQUMsSUFBYTs7QUFFakM7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0EiLCJmaWxlIjoiMzcxNS5qcyIsInNvdXJjZXNDb250ZW50IjpbInZhciBoYXNoID0gZXhwb3J0cztcblxuaGFzaC51dGlscyA9IHJlcXVpcmUoJy4vaGFzaC91dGlscycpO1xuaGFzaC5jb21tb24gPSByZXF1aXJlKCcuL2hhc2gvY29tbW9uJyk7XG5oYXNoLnNoYSA9IHJlcXVpcmUoJy4vaGFzaC9zaGEnKTtcbmhhc2gucmlwZW1kID0gcmVxdWlyZSgnLi9oYXNoL3JpcGVtZCcpO1xuaGFzaC5obWFjID0gcmVxdWlyZSgnLi9oYXNoL2htYWMnKTtcblxuLy8gUHJveHkgaGFzaCBmdW5jdGlvbnMgdG8gdGhlIG1haW4gb2JqZWN0XG5oYXNoLnNoYTEgPSBoYXNoLnNoYS5zaGExO1xuaGFzaC5zaGEyNTYgPSBoYXNoLnNoYS5zaGEyNTY7XG5oYXNoLnNoYTIyNCA9IGhhc2guc2hhLnNoYTIyNDtcbmhhc2guc2hhMzg0ID0gaGFzaC5zaGEuc2hhMzg0O1xuaGFzaC5zaGE1MTIgPSBoYXNoLnNoYS5zaGE1MTI7XG5oYXNoLnJpcGVtZDE2MCA9IGhhc2gucmlwZW1kLnJpcGVtZDE2MDtcbiJdLCJzb3VyY2VSb290IjoiIn0=\n//# sourceURL=webpack-internal:///3715\n")},5772:(__unused_webpack_module,exports,__webpack_require__)=>{"use strict";eval("\n\nvar utils = __webpack_require__(6436);\nvar assert = __webpack_require__(9746);\n\nfunction BlockHash() {\n  this.pending = null;\n  this.pendingTotal = 0;\n  this.blockSize = this.constructor.blockSize;\n  this.outSize = this.constructor.outSize;\n  this.hmacStrength = this.constructor.hmacStrength;\n  this.padLength = this.constructor.padLength / 8;\n  this.endian = 'big';\n\n  this._delta8 = this.blockSize / 8;\n  this._delta32 = this.blockSize / 32;\n}\nexports.BlockHash = BlockHash;\n\nBlockHash.prototype.update = function update(msg, enc) {\n  // Convert message to array, pad it, and join into 32bit blocks\n  msg = utils.toArray(msg, enc);\n  if (!this.pending)\n    this.pending = msg;\n  else\n    this.pending = this.pending.concat(msg);\n  this.pendingTotal += msg.length;\n\n  // Enough data, try updating\n  if (this.pending.length >= this._delta8) {\n    msg = this.pending;\n\n    // Process pending data in blocks\n    var r = msg.length % this._delta8;\n    this.pending = msg.slice(msg.length - r, msg.length);\n    if (this.pending.length === 0)\n      this.pending = null;\n\n    msg = utils.join32(msg, 0, msg.length - r, this.endian);\n    for (var i = 0; i < msg.length; i += this._delta32)\n      this._update(msg, i, i + this._delta32);\n  }\n\n  return this;\n};\n\nBlockHash.prototype.digest = function digest(enc) {\n  this.update(this._pad());\n  assert(this.pending === null);\n\n  return this._digest(enc);\n};\n\nBlockHash.prototype._pad = function pad() {\n  var len = this.pendingTotal;\n  var bytes = this._delta8;\n  var k = bytes - ((len + this.padLength) % bytes);\n  var res = new Array(k + this.padLength);\n  res[0] = 0x80;\n  for (var i = 1; i < k; i++)\n    res[i] = 0;\n\n  // Append length\n  len <<= 3;\n  if (this.endian === 'big') {\n    for (var t = 8; t < this.padLength; t++)\n      res[i++] = 0;\n\n    res[i++] = 0;\n    res[i++] = 0;\n    res[i++] = 0;\n    res[i++] = 0;\n    res[i++] = (len >>> 24) & 0xff;\n    res[i++] = (len >>> 16) & 0xff;\n    res[i++] = (len >>> 8) & 0xff;\n    res[i++] = len & 0xff;\n  } else {\n    res[i++] = len & 0xff;\n    res[i++] = (len >>> 8) & 0xff;\n    res[i++] = (len >>> 16) & 0xff;\n    res[i++] = (len >>> 24) & 0xff;\n    res[i++] = 0;\n    res[i++] = 0;\n    res[i++] = 0;\n    res[i++] = 0;\n\n    for (t = 8; t < this.padLength; t++)\n      res[i++] = 0;\n  }\n\n  return res;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///5772\n")},2344:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar utils = __webpack_require__(6436);\nvar assert = __webpack_require__(9746);\n\nfunction Hmac(hash, key, enc) {\n  if (!(this instanceof Hmac))\n    return new Hmac(hash, key, enc);\n  this.Hash = hash;\n  this.blockSize = hash.blockSize / 8;\n  this.outSize = hash.outSize / 8;\n  this.inner = null;\n  this.outer = null;\n\n  this._init(utils.toArray(key, enc));\n}\nmodule.exports = Hmac;\n\nHmac.prototype._init = function init(key) {\n  // Shorten key, if needed\n  if (key.length > this.blockSize)\n    key = new this.Hash().update(key).digest();\n  assert(key.length <= this.blockSize);\n\n  // Add padding to key\n  for (var i = key.length; i < this.blockSize; i++)\n    key.push(0);\n\n  for (i = 0; i < key.length; i++)\n    key[i] ^= 0x36;\n  this.inner = new this.Hash().update(key);\n\n  // 0x36 ^ 0x5c = 0x6a\n  for (i = 0; i < key.length; i++)\n    key[i] ^= 0x6a;\n  this.outer = new this.Hash().update(key);\n};\n\nHmac.prototype.update = function update(msg, enc) {\n  this.inner.update(msg, enc);\n  return this;\n};\n\nHmac.prototype.digest = function digest(enc) {\n  this.outer.update(this.inner.digest());\n  return this.outer.digest(enc);\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///2344\n")},2949:(__unused_webpack_module,exports,__webpack_require__)=>{"use strict";eval("\n\nvar utils = __webpack_require__(6436);\nvar common = __webpack_require__(5772);\n\nvar rotl32 = utils.rotl32;\nvar sum32 = utils.sum32;\nvar sum32_3 = utils.sum32_3;\nvar sum32_4 = utils.sum32_4;\nvar BlockHash = common.BlockHash;\n\nfunction RIPEMD160() {\n  if (!(this instanceof RIPEMD160))\n    return new RIPEMD160();\n\n  BlockHash.call(this);\n\n  this.h = [ 0x67452301, 0xefcdab89, 0x98badcfe, 0x10325476, 0xc3d2e1f0 ];\n  this.endian = 'little';\n}\nutils.inherits(RIPEMD160, BlockHash);\nexports.ripemd160 = RIPEMD160;\n\nRIPEMD160.blockSize = 512;\nRIPEMD160.outSize = 160;\nRIPEMD160.hmacStrength = 192;\nRIPEMD160.padLength = 64;\n\nRIPEMD160.prototype._update = function update(msg, start) {\n  var A = this.h[0];\n  var B = this.h[1];\n  var C = this.h[2];\n  var D = this.h[3];\n  var E = this.h[4];\n  var Ah = A;\n  var Bh = B;\n  var Ch = C;\n  var Dh = D;\n  var Eh = E;\n  for (var j = 0; j < 80; j++) {\n    var T = sum32(\n      rotl32(\n        sum32_4(A, f(j, B, C, D), msg[r[j] + start], K(j)),\n        s[j]),\n      E);\n    A = E;\n    E = D;\n    D = rotl32(C, 10);\n    C = B;\n    B = T;\n    T = sum32(\n      rotl32(\n        sum32_4(Ah, f(79 - j, Bh, Ch, Dh), msg[rh[j] + start], Kh(j)),\n        sh[j]),\n      Eh);\n    Ah = Eh;\n    Eh = Dh;\n    Dh = rotl32(Ch, 10);\n    Ch = Bh;\n    Bh = T;\n  }\n  T = sum32_3(this.h[1], C, Dh);\n  this.h[1] = sum32_3(this.h[2], D, Eh);\n  this.h[2] = sum32_3(this.h[3], E, Ah);\n  this.h[3] = sum32_3(this.h[4], A, Bh);\n  this.h[4] = sum32_3(this.h[0], B, Ch);\n  this.h[0] = T;\n};\n\nRIPEMD160.prototype._digest = function digest(enc) {\n  if (enc === 'hex')\n    return utils.toHex32(this.h, 'little');\n  else\n    return utils.split32(this.h, 'little');\n};\n\nfunction f(j, x, y, z) {\n  if (j <= 15)\n    return x ^ y ^ z;\n  else if (j <= 31)\n    return (x & y) | ((~x) & z);\n  else if (j <= 47)\n    return (x | (~y)) ^ z;\n  else if (j <= 63)\n    return (x & z) | (y & (~z));\n  else\n    return x ^ (y | (~z));\n}\n\nfunction K(j) {\n  if (j <= 15)\n    return 0x00000000;\n  else if (j <= 31)\n    return 0x5a827999;\n  else if (j <= 47)\n    return 0x6ed9eba1;\n  else if (j <= 63)\n    return 0x8f1bbcdc;\n  else\n    return 0xa953fd4e;\n}\n\nfunction Kh(j) {\n  if (j <= 15)\n    return 0x50a28be6;\n  else if (j <= 31)\n    return 0x5c4dd124;\n  else if (j <= 47)\n    return 0x6d703ef3;\n  else if (j <= 63)\n    return 0x7a6d76e9;\n  else\n    return 0x00000000;\n}\n\nvar r = [\n  0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15,\n  7, 4, 13, 1, 10, 6, 15, 3, 12, 0, 9, 5, 2, 14, 11, 8,\n  3, 10, 14, 4, 9, 15, 8, 1, 2, 7, 0, 6, 13, 11, 5, 12,\n  1, 9, 11, 10, 0, 8, 12, 4, 13, 3, 7, 15, 14, 5, 6, 2,\n  4, 0, 5, 9, 7, 12, 2, 10, 14, 1, 3, 8, 11, 6, 15, 13\n];\n\nvar rh = [\n  5, 14, 7, 0, 9, 2, 11, 4, 13, 6, 15, 8, 1, 10, 3, 12,\n  6, 11, 3, 7, 0, 13, 5, 10, 14, 15, 8, 12, 4, 9, 1, 2,\n  15, 5, 1, 3, 7, 14, 6, 9, 11, 8, 12, 2, 10, 0, 4, 13,\n  8, 6, 4, 1, 3, 11, 15, 0, 5, 12, 2, 13, 9, 7, 10, 14,\n  12, 15, 10, 4, 1, 5, 8, 7, 6, 2, 13, 14, 0, 3, 9, 11\n];\n\nvar s = [\n  11, 14, 15, 12, 5, 8, 7, 9, 11, 13, 14, 15, 6, 7, 9, 8,\n  7, 6, 8, 13, 11, 9, 7, 15, 7, 12, 15, 9, 11, 7, 13, 12,\n  11, 13, 6, 7, 14, 9, 13, 15, 14, 8, 13, 6, 5, 12, 7, 5,\n  11, 12, 14, 15, 14, 15, 9, 8, 9, 14, 5, 6, 8, 6, 5, 12,\n  9, 15, 5, 11, 6, 8, 13, 12, 5, 12, 13, 14, 11, 8, 5, 6\n];\n\nvar sh = [\n  8, 9, 9, 11, 13, 15, 15, 5, 7, 7, 8, 11, 14, 14, 12, 6,\n  9, 13, 15, 7, 12, 8, 9, 11, 7, 7, 12, 7, 6, 15, 13, 11,\n  9, 7, 15, 11, 8, 6, 6, 14, 12, 13, 5, 14, 13, 13, 7, 5,\n  15, 5, 8, 11, 14, 14, 6, 14, 6, 9, 12, 9, 12, 5, 15, 8,\n  8, 5, 12, 9, 12, 5, 14, 6, 8, 13, 6, 5, 15, 13, 11, 11\n];\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///2949\n")},9041:(__unused_webpack_module,exports,__webpack_require__)=>{"use strict";eval("\n\nexports.sha1 = __webpack_require__(4761);\nexports.sha224 = __webpack_require__(799);\nexports.sha256 = __webpack_require__(9344);\nexports.sha384 = __webpack_require__(772);\nexports.sha512 = __webpack_require__(5900);\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2hhc2guanMvbGliL2hhc2gvc2hhLmpzPzU5MTkiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQWE7O0FBRWIsd0NBQWlDO0FBQ2pDLHlDQUFxQztBQUNyQywwQ0FBcUM7QUFDckMseUNBQXFDO0FBQ3JDLDBDQUFxQyIsImZpbGUiOiI5MDQxLmpzIiwic291cmNlc0NvbnRlbnQiOlsiJ3VzZSBzdHJpY3QnO1xuXG5leHBvcnRzLnNoYTEgPSByZXF1aXJlKCcuL3NoYS8xJyk7XG5leHBvcnRzLnNoYTIyNCA9IHJlcXVpcmUoJy4vc2hhLzIyNCcpO1xuZXhwb3J0cy5zaGEyNTYgPSByZXF1aXJlKCcuL3NoYS8yNTYnKTtcbmV4cG9ydHMuc2hhMzg0ID0gcmVxdWlyZSgnLi9zaGEvMzg0Jyk7XG5leHBvcnRzLnNoYTUxMiA9IHJlcXVpcmUoJy4vc2hhLzUxMicpO1xuIl0sInNvdXJjZVJvb3QiOiIifQ==\n//# sourceURL=webpack-internal:///9041\n")},4761:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar utils = __webpack_require__(6436);\nvar common = __webpack_require__(5772);\nvar shaCommon = __webpack_require__(7038);\n\nvar rotl32 = utils.rotl32;\nvar sum32 = utils.sum32;\nvar sum32_5 = utils.sum32_5;\nvar ft_1 = shaCommon.ft_1;\nvar BlockHash = common.BlockHash;\n\nvar sha1_K = [\n  0x5A827999, 0x6ED9EBA1,\n  0x8F1BBCDC, 0xCA62C1D6\n];\n\nfunction SHA1() {\n  if (!(this instanceof SHA1))\n    return new SHA1();\n\n  BlockHash.call(this);\n  this.h = [\n    0x67452301, 0xefcdab89, 0x98badcfe,\n    0x10325476, 0xc3d2e1f0 ];\n  this.W = new Array(80);\n}\n\nutils.inherits(SHA1, BlockHash);\nmodule.exports = SHA1;\n\nSHA1.blockSize = 512;\nSHA1.outSize = 160;\nSHA1.hmacStrength = 80;\nSHA1.padLength = 64;\n\nSHA1.prototype._update = function _update(msg, start) {\n  var W = this.W;\n\n  for (var i = 0; i < 16; i++)\n    W[i] = msg[start + i];\n\n  for(; i < W.length; i++)\n    W[i] = rotl32(W[i - 3] ^ W[i - 8] ^ W[i - 14] ^ W[i - 16], 1);\n\n  var a = this.h[0];\n  var b = this.h[1];\n  var c = this.h[2];\n  var d = this.h[3];\n  var e = this.h[4];\n\n  for (i = 0; i < W.length; i++) {\n    var s = ~~(i / 20);\n    var t = sum32_5(rotl32(a, 5), ft_1(s, b, c, d), e, W[i], sha1_K[s]);\n    e = d;\n    d = c;\n    c = rotl32(b, 30);\n    b = a;\n    a = t;\n  }\n\n  this.h[0] = sum32(this.h[0], a);\n  this.h[1] = sum32(this.h[1], b);\n  this.h[2] = sum32(this.h[2], c);\n  this.h[3] = sum32(this.h[3], d);\n  this.h[4] = sum32(this.h[4], e);\n};\n\nSHA1.prototype._digest = function digest(enc) {\n  if (enc === 'hex')\n    return utils.toHex32(this.h, 'big');\n  else\n    return utils.split32(this.h, 'big');\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///4761\n")},799:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar utils = __webpack_require__(6436);\nvar SHA256 = __webpack_require__(9344);\n\nfunction SHA224() {\n  if (!(this instanceof SHA224))\n    return new SHA224();\n\n  SHA256.call(this);\n  this.h = [\n    0xc1059ed8, 0x367cd507, 0x3070dd17, 0xf70e5939,\n    0xffc00b31, 0x68581511, 0x64f98fa7, 0xbefa4fa4 ];\n}\nutils.inherits(SHA224, SHA256);\nmodule.exports = SHA224;\n\nSHA224.blockSize = 512;\nSHA224.outSize = 224;\nSHA224.hmacStrength = 192;\nSHA224.padLength = 64;\n\nSHA224.prototype._digest = function digest(enc) {\n  // Just truncate output\n  if (enc === 'hex')\n    return utils.toHex32(this.h.slice(0, 7), 'big');\n  else\n    return utils.split32(this.h.slice(0, 7), 'big');\n};\n\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2hhc2guanMvbGliL2hhc2gvc2hhLzIyNC5qcz8wN2YyIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFhOztBQUViLFlBQVksbUJBQU8sQ0FBQyxJQUFVO0FBQzlCLGFBQWEsbUJBQU8sQ0FBQyxJQUFPOztBQUU1QjtBQUNBO0FBQ0E7O0FBRUE7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7O0FBRUE7QUFDQTtBQUNBO0FBQ0E7O0FBRUE7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0EiLCJmaWxlIjoiNzk5LmpzIiwic291cmNlc0NvbnRlbnQiOlsiJ3VzZSBzdHJpY3QnO1xuXG52YXIgdXRpbHMgPSByZXF1aXJlKCcuLi91dGlscycpO1xudmFyIFNIQTI1NiA9IHJlcXVpcmUoJy4vMjU2Jyk7XG5cbmZ1bmN0aW9uIFNIQTIyNCgpIHtcbiAgaWYgKCEodGhpcyBpbnN0YW5jZW9mIFNIQTIyNCkpXG4gICAgcmV0dXJuIG5ldyBTSEEyMjQoKTtcblxuICBTSEEyNTYuY2FsbCh0aGlzKTtcbiAgdGhpcy5oID0gW1xuICAgIDB4YzEwNTllZDgsIDB4MzY3Y2Q1MDcsIDB4MzA3MGRkMTcsIDB4ZjcwZTU5MzksXG4gICAgMHhmZmMwMGIzMSwgMHg2ODU4MTUxMSwgMHg2NGY5OGZhNywgMHhiZWZhNGZhNCBdO1xufVxudXRpbHMuaW5oZXJpdHMoU0hBMjI0LCBTSEEyNTYpO1xubW9kdWxlLmV4cG9ydHMgPSBTSEEyMjQ7XG5cblNIQTIyNC5ibG9ja1NpemUgPSA1MTI7XG5TSEEyMjQub3V0U2l6ZSA9IDIyNDtcblNIQTIyNC5obWFjU3RyZW5ndGggPSAxOTI7XG5TSEEyMjQucGFkTGVuZ3RoID0gNjQ7XG5cblNIQTIyNC5wcm90b3R5cGUuX2RpZ2VzdCA9IGZ1bmN0aW9uIGRpZ2VzdChlbmMpIHtcbiAgLy8gSnVzdCB0cnVuY2F0ZSBvdXRwdXRcbiAgaWYgKGVuYyA9PT0gJ2hleCcpXG4gICAgcmV0dXJuIHV0aWxzLnRvSGV4MzIodGhpcy5oLnNsaWNlKDAsIDcpLCAnYmlnJyk7XG4gIGVsc2VcbiAgICByZXR1cm4gdXRpbHMuc3BsaXQzMih0aGlzLmguc2xpY2UoMCwgNyksICdiaWcnKTtcbn07XG5cbiJdLCJzb3VyY2VSb290IjoiIn0=\n//# sourceURL=webpack-internal:///799\n")},9344:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar utils = __webpack_require__(6436);\nvar common = __webpack_require__(5772);\nvar shaCommon = __webpack_require__(7038);\nvar assert = __webpack_require__(9746);\n\nvar sum32 = utils.sum32;\nvar sum32_4 = utils.sum32_4;\nvar sum32_5 = utils.sum32_5;\nvar ch32 = shaCommon.ch32;\nvar maj32 = shaCommon.maj32;\nvar s0_256 = shaCommon.s0_256;\nvar s1_256 = shaCommon.s1_256;\nvar g0_256 = shaCommon.g0_256;\nvar g1_256 = shaCommon.g1_256;\n\nvar BlockHash = common.BlockHash;\n\nvar sha256_K = [\n  0x428a2f98, 0x71374491, 0xb5c0fbcf, 0xe9b5dba5,\n  0x3956c25b, 0x59f111f1, 0x923f82a4, 0xab1c5ed5,\n  0xd807aa98, 0x12835b01, 0x243185be, 0x550c7dc3,\n  0x72be5d74, 0x80deb1fe, 0x9bdc06a7, 0xc19bf174,\n  0xe49b69c1, 0xefbe4786, 0x0fc19dc6, 0x240ca1cc,\n  0x2de92c6f, 0x4a7484aa, 0x5cb0a9dc, 0x76f988da,\n  0x983e5152, 0xa831c66d, 0xb00327c8, 0xbf597fc7,\n  0xc6e00bf3, 0xd5a79147, 0x06ca6351, 0x14292967,\n  0x27b70a85, 0x2e1b2138, 0x4d2c6dfc, 0x53380d13,\n  0x650a7354, 0x766a0abb, 0x81c2c92e, 0x92722c85,\n  0xa2bfe8a1, 0xa81a664b, 0xc24b8b70, 0xc76c51a3,\n  0xd192e819, 0xd6990624, 0xf40e3585, 0x106aa070,\n  0x19a4c116, 0x1e376c08, 0x2748774c, 0x34b0bcb5,\n  0x391c0cb3, 0x4ed8aa4a, 0x5b9cca4f, 0x682e6ff3,\n  0x748f82ee, 0x78a5636f, 0x84c87814, 0x8cc70208,\n  0x90befffa, 0xa4506ceb, 0xbef9a3f7, 0xc67178f2\n];\n\nfunction SHA256() {\n  if (!(this instanceof SHA256))\n    return new SHA256();\n\n  BlockHash.call(this);\n  this.h = [\n    0x6a09e667, 0xbb67ae85, 0x3c6ef372, 0xa54ff53a,\n    0x510e527f, 0x9b05688c, 0x1f83d9ab, 0x5be0cd19\n  ];\n  this.k = sha256_K;\n  this.W = new Array(64);\n}\nutils.inherits(SHA256, BlockHash);\nmodule.exports = SHA256;\n\nSHA256.blockSize = 512;\nSHA256.outSize = 256;\nSHA256.hmacStrength = 192;\nSHA256.padLength = 64;\n\nSHA256.prototype._update = function _update(msg, start) {\n  var W = this.W;\n\n  for (var i = 0; i < 16; i++)\n    W[i] = msg[start + i];\n  for (; i < W.length; i++)\n    W[i] = sum32_4(g1_256(W[i - 2]), W[i - 7], g0_256(W[i - 15]), W[i - 16]);\n\n  var a = this.h[0];\n  var b = this.h[1];\n  var c = this.h[2];\n  var d = this.h[3];\n  var e = this.h[4];\n  var f = this.h[5];\n  var g = this.h[6];\n  var h = this.h[7];\n\n  assert(this.k.length === W.length);\n  for (i = 0; i < W.length; i++) {\n    var T1 = sum32_5(h, s1_256(e), ch32(e, f, g), this.k[i], W[i]);\n    var T2 = sum32(s0_256(a), maj32(a, b, c));\n    h = g;\n    g = f;\n    f = e;\n    e = sum32(d, T1);\n    d = c;\n    c = b;\n    b = a;\n    a = sum32(T1, T2);\n  }\n\n  this.h[0] = sum32(this.h[0], a);\n  this.h[1] = sum32(this.h[1], b);\n  this.h[2] = sum32(this.h[2], c);\n  this.h[3] = sum32(this.h[3], d);\n  this.h[4] = sum32(this.h[4], e);\n  this.h[5] = sum32(this.h[5], f);\n  this.h[6] = sum32(this.h[6], g);\n  this.h[7] = sum32(this.h[7], h);\n};\n\nSHA256.prototype._digest = function digest(enc) {\n  if (enc === 'hex')\n    return utils.toHex32(this.h, 'big');\n  else\n    return utils.split32(this.h, 'big');\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///9344\n")},772:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar utils = __webpack_require__(6436);\n\nvar SHA512 = __webpack_require__(5900);\n\nfunction SHA384() {\n  if (!(this instanceof SHA384))\n    return new SHA384();\n\n  SHA512.call(this);\n  this.h = [\n    0xcbbb9d5d, 0xc1059ed8,\n    0x629a292a, 0x367cd507,\n    0x9159015a, 0x3070dd17,\n    0x152fecd8, 0xf70e5939,\n    0x67332667, 0xffc00b31,\n    0x8eb44a87, 0x68581511,\n    0xdb0c2e0d, 0x64f98fa7,\n    0x47b5481d, 0xbefa4fa4 ];\n}\nutils.inherits(SHA384, SHA512);\nmodule.exports = SHA384;\n\nSHA384.blockSize = 1024;\nSHA384.outSize = 384;\nSHA384.hmacStrength = 192;\nSHA384.padLength = 128;\n\nSHA384.prototype._digest = function digest(enc) {\n  if (enc === 'hex')\n    return utils.toHex32(this.h.slice(0, 12), 'big');\n  else\n    return utils.split32(this.h.slice(0, 12), 'big');\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2hhc2guanMvbGliL2hhc2gvc2hhLzM4NC5qcz84Yjk1Il0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFhOztBQUViLFlBQVksbUJBQU8sQ0FBQyxJQUFVOztBQUU5QixhQUFhLG1CQUFPLENBQUMsSUFBTzs7QUFFNUI7QUFDQTtBQUNBOztBQUVBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBOztBQUVBO0FBQ0E7QUFDQTtBQUNBOztBQUVBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQSIsImZpbGUiOiI3NzIuanMiLCJzb3VyY2VzQ29udGVudCI6WyIndXNlIHN0cmljdCc7XG5cbnZhciB1dGlscyA9IHJlcXVpcmUoJy4uL3V0aWxzJyk7XG5cbnZhciBTSEE1MTIgPSByZXF1aXJlKCcuLzUxMicpO1xuXG5mdW5jdGlvbiBTSEEzODQoKSB7XG4gIGlmICghKHRoaXMgaW5zdGFuY2VvZiBTSEEzODQpKVxuICAgIHJldHVybiBuZXcgU0hBMzg0KCk7XG5cbiAgU0hBNTEyLmNhbGwodGhpcyk7XG4gIHRoaXMuaCA9IFtcbiAgICAweGNiYmI5ZDVkLCAweGMxMDU5ZWQ4LFxuICAgIDB4NjI5YTI5MmEsIDB4MzY3Y2Q1MDcsXG4gICAgMHg5MTU5MDE1YSwgMHgzMDcwZGQxNyxcbiAgICAweDE1MmZlY2Q4LCAweGY3MGU1OTM5LFxuICAgIDB4NjczMzI2NjcsIDB4ZmZjMDBiMzEsXG4gICAgMHg4ZWI0NGE4NywgMHg2ODU4MTUxMSxcbiAgICAweGRiMGMyZTBkLCAweDY0Zjk4ZmE3LFxuICAgIDB4NDdiNTQ4MWQsIDB4YmVmYTRmYTQgXTtcbn1cbnV0aWxzLmluaGVyaXRzKFNIQTM4NCwgU0hBNTEyKTtcbm1vZHVsZS5leHBvcnRzID0gU0hBMzg0O1xuXG5TSEEzODQuYmxvY2tTaXplID0gMTAyNDtcblNIQTM4NC5vdXRTaXplID0gMzg0O1xuU0hBMzg0LmhtYWNTdHJlbmd0aCA9IDE5MjtcblNIQTM4NC5wYWRMZW5ndGggPSAxMjg7XG5cblNIQTM4NC5wcm90b3R5cGUuX2RpZ2VzdCA9IGZ1bmN0aW9uIGRpZ2VzdChlbmMpIHtcbiAgaWYgKGVuYyA9PT0gJ2hleCcpXG4gICAgcmV0dXJuIHV0aWxzLnRvSGV4MzIodGhpcy5oLnNsaWNlKDAsIDEyKSwgJ2JpZycpO1xuICBlbHNlXG4gICAgcmV0dXJuIHV0aWxzLnNwbGl0MzIodGhpcy5oLnNsaWNlKDAsIDEyKSwgJ2JpZycpO1xufTtcbiJdLCJzb3VyY2VSb290IjoiIn0=\n//# sourceURL=webpack-internal:///772\n")},5900:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar utils = __webpack_require__(6436);\nvar common = __webpack_require__(5772);\nvar assert = __webpack_require__(9746);\n\nvar rotr64_hi = utils.rotr64_hi;\nvar rotr64_lo = utils.rotr64_lo;\nvar shr64_hi = utils.shr64_hi;\nvar shr64_lo = utils.shr64_lo;\nvar sum64 = utils.sum64;\nvar sum64_hi = utils.sum64_hi;\nvar sum64_lo = utils.sum64_lo;\nvar sum64_4_hi = utils.sum64_4_hi;\nvar sum64_4_lo = utils.sum64_4_lo;\nvar sum64_5_hi = utils.sum64_5_hi;\nvar sum64_5_lo = utils.sum64_5_lo;\n\nvar BlockHash = common.BlockHash;\n\nvar sha512_K = [\n  0x428a2f98, 0xd728ae22, 0x71374491, 0x23ef65cd,\n  0xb5c0fbcf, 0xec4d3b2f, 0xe9b5dba5, 0x8189dbbc,\n  0x3956c25b, 0xf348b538, 0x59f111f1, 0xb605d019,\n  0x923f82a4, 0xaf194f9b, 0xab1c5ed5, 0xda6d8118,\n  0xd807aa98, 0xa3030242, 0x12835b01, 0x45706fbe,\n  0x243185be, 0x4ee4b28c, 0x550c7dc3, 0xd5ffb4e2,\n  0x72be5d74, 0xf27b896f, 0x80deb1fe, 0x3b1696b1,\n  0x9bdc06a7, 0x25c71235, 0xc19bf174, 0xcf692694,\n  0xe49b69c1, 0x9ef14ad2, 0xefbe4786, 0x384f25e3,\n  0x0fc19dc6, 0x8b8cd5b5, 0x240ca1cc, 0x77ac9c65,\n  0x2de92c6f, 0x592b0275, 0x4a7484aa, 0x6ea6e483,\n  0x5cb0a9dc, 0xbd41fbd4, 0x76f988da, 0x831153b5,\n  0x983e5152, 0xee66dfab, 0xa831c66d, 0x2db43210,\n  0xb00327c8, 0x98fb213f, 0xbf597fc7, 0xbeef0ee4,\n  0xc6e00bf3, 0x3da88fc2, 0xd5a79147, 0x930aa725,\n  0x06ca6351, 0xe003826f, 0x14292967, 0x0a0e6e70,\n  0x27b70a85, 0x46d22ffc, 0x2e1b2138, 0x5c26c926,\n  0x4d2c6dfc, 0x5ac42aed, 0x53380d13, 0x9d95b3df,\n  0x650a7354, 0x8baf63de, 0x766a0abb, 0x3c77b2a8,\n  0x81c2c92e, 0x47edaee6, 0x92722c85, 0x1482353b,\n  0xa2bfe8a1, 0x4cf10364, 0xa81a664b, 0xbc423001,\n  0xc24b8b70, 0xd0f89791, 0xc76c51a3, 0x0654be30,\n  0xd192e819, 0xd6ef5218, 0xd6990624, 0x5565a910,\n  0xf40e3585, 0x5771202a, 0x106aa070, 0x32bbd1b8,\n  0x19a4c116, 0xb8d2d0c8, 0x1e376c08, 0x5141ab53,\n  0x2748774c, 0xdf8eeb99, 0x34b0bcb5, 0xe19b48a8,\n  0x391c0cb3, 0xc5c95a63, 0x4ed8aa4a, 0xe3418acb,\n  0x5b9cca4f, 0x7763e373, 0x682e6ff3, 0xd6b2b8a3,\n  0x748f82ee, 0x5defb2fc, 0x78a5636f, 0x43172f60,\n  0x84c87814, 0xa1f0ab72, 0x8cc70208, 0x1a6439ec,\n  0x90befffa, 0x23631e28, 0xa4506ceb, 0xde82bde9,\n  0xbef9a3f7, 0xb2c67915, 0xc67178f2, 0xe372532b,\n  0xca273ece, 0xea26619c, 0xd186b8c7, 0x21c0c207,\n  0xeada7dd6, 0xcde0eb1e, 0xf57d4f7f, 0xee6ed178,\n  0x06f067aa, 0x72176fba, 0x0a637dc5, 0xa2c898a6,\n  0x113f9804, 0xbef90dae, 0x1b710b35, 0x131c471b,\n  0x28db77f5, 0x23047d84, 0x32caab7b, 0x40c72493,\n  0x3c9ebe0a, 0x15c9bebc, 0x431d67c4, 0x9c100d4c,\n  0x4cc5d4be, 0xcb3e42b6, 0x597f299c, 0xfc657e2a,\n  0x5fcb6fab, 0x3ad6faec, 0x6c44198c, 0x4a475817\n];\n\nfunction SHA512() {\n  if (!(this instanceof SHA512))\n    return new SHA512();\n\n  BlockHash.call(this);\n  this.h = [\n    0x6a09e667, 0xf3bcc908,\n    0xbb67ae85, 0x84caa73b,\n    0x3c6ef372, 0xfe94f82b,\n    0xa54ff53a, 0x5f1d36f1,\n    0x510e527f, 0xade682d1,\n    0x9b05688c, 0x2b3e6c1f,\n    0x1f83d9ab, 0xfb41bd6b,\n    0x5be0cd19, 0x137e2179 ];\n  this.k = sha512_K;\n  this.W = new Array(160);\n}\nutils.inherits(SHA512, BlockHash);\nmodule.exports = SHA512;\n\nSHA512.blockSize = 1024;\nSHA512.outSize = 512;\nSHA512.hmacStrength = 192;\nSHA512.padLength = 128;\n\nSHA512.prototype._prepareBlock = function _prepareBlock(msg, start) {\n  var W = this.W;\n\n  // 32 x 32bit words\n  for (var i = 0; i < 32; i++)\n    W[i] = msg[start + i];\n  for (; i < W.length; i += 2) {\n    var c0_hi = g1_512_hi(W[i - 4], W[i - 3]);  // i - 2\n    var c0_lo = g1_512_lo(W[i - 4], W[i - 3]);\n    var c1_hi = W[i - 14];  // i - 7\n    var c1_lo = W[i - 13];\n    var c2_hi = g0_512_hi(W[i - 30], W[i - 29]);  // i - 15\n    var c2_lo = g0_512_lo(W[i - 30], W[i - 29]);\n    var c3_hi = W[i - 32];  // i - 16\n    var c3_lo = W[i - 31];\n\n    W[i] = sum64_4_hi(\n      c0_hi, c0_lo,\n      c1_hi, c1_lo,\n      c2_hi, c2_lo,\n      c3_hi, c3_lo);\n    W[i + 1] = sum64_4_lo(\n      c0_hi, c0_lo,\n      c1_hi, c1_lo,\n      c2_hi, c2_lo,\n      c3_hi, c3_lo);\n  }\n};\n\nSHA512.prototype._update = function _update(msg, start) {\n  this._prepareBlock(msg, start);\n\n  var W = this.W;\n\n  var ah = this.h[0];\n  var al = this.h[1];\n  var bh = this.h[2];\n  var bl = this.h[3];\n  var ch = this.h[4];\n  var cl = this.h[5];\n  var dh = this.h[6];\n  var dl = this.h[7];\n  var eh = this.h[8];\n  var el = this.h[9];\n  var fh = this.h[10];\n  var fl = this.h[11];\n  var gh = this.h[12];\n  var gl = this.h[13];\n  var hh = this.h[14];\n  var hl = this.h[15];\n\n  assert(this.k.length === W.length);\n  for (var i = 0; i < W.length; i += 2) {\n    var c0_hi = hh;\n    var c0_lo = hl;\n    var c1_hi = s1_512_hi(eh, el);\n    var c1_lo = s1_512_lo(eh, el);\n    var c2_hi = ch64_hi(eh, el, fh, fl, gh, gl);\n    var c2_lo = ch64_lo(eh, el, fh, fl, gh, gl);\n    var c3_hi = this.k[i];\n    var c3_lo = this.k[i + 1];\n    var c4_hi = W[i];\n    var c4_lo = W[i + 1];\n\n    var T1_hi = sum64_5_hi(\n      c0_hi, c0_lo,\n      c1_hi, c1_lo,\n      c2_hi, c2_lo,\n      c3_hi, c3_lo,\n      c4_hi, c4_lo);\n    var T1_lo = sum64_5_lo(\n      c0_hi, c0_lo,\n      c1_hi, c1_lo,\n      c2_hi, c2_lo,\n      c3_hi, c3_lo,\n      c4_hi, c4_lo);\n\n    c0_hi = s0_512_hi(ah, al);\n    c0_lo = s0_512_lo(ah, al);\n    c1_hi = maj64_hi(ah, al, bh, bl, ch, cl);\n    c1_lo = maj64_lo(ah, al, bh, bl, ch, cl);\n\n    var T2_hi = sum64_hi(c0_hi, c0_lo, c1_hi, c1_lo);\n    var T2_lo = sum64_lo(c0_hi, c0_lo, c1_hi, c1_lo);\n\n    hh = gh;\n    hl = gl;\n\n    gh = fh;\n    gl = fl;\n\n    fh = eh;\n    fl = el;\n\n    eh = sum64_hi(dh, dl, T1_hi, T1_lo);\n    el = sum64_lo(dl, dl, T1_hi, T1_lo);\n\n    dh = ch;\n    dl = cl;\n\n    ch = bh;\n    cl = bl;\n\n    bh = ah;\n    bl = al;\n\n    ah = sum64_hi(T1_hi, T1_lo, T2_hi, T2_lo);\n    al = sum64_lo(T1_hi, T1_lo, T2_hi, T2_lo);\n  }\n\n  sum64(this.h, 0, ah, al);\n  sum64(this.h, 2, bh, bl);\n  sum64(this.h, 4, ch, cl);\n  sum64(this.h, 6, dh, dl);\n  sum64(this.h, 8, eh, el);\n  sum64(this.h, 10, fh, fl);\n  sum64(this.h, 12, gh, gl);\n  sum64(this.h, 14, hh, hl);\n};\n\nSHA512.prototype._digest = function digest(enc) {\n  if (enc === 'hex')\n    return utils.toHex32(this.h, 'big');\n  else\n    return utils.split32(this.h, 'big');\n};\n\nfunction ch64_hi(xh, xl, yh, yl, zh) {\n  var r = (xh & yh) ^ ((~xh) & zh);\n  if (r < 0)\n    r += 0x100000000;\n  return r;\n}\n\nfunction ch64_lo(xh, xl, yh, yl, zh, zl) {\n  var r = (xl & yl) ^ ((~xl) & zl);\n  if (r < 0)\n    r += 0x100000000;\n  return r;\n}\n\nfunction maj64_hi(xh, xl, yh, yl, zh) {\n  var r = (xh & yh) ^ (xh & zh) ^ (yh & zh);\n  if (r < 0)\n    r += 0x100000000;\n  return r;\n}\n\nfunction maj64_lo(xh, xl, yh, yl, zh, zl) {\n  var r = (xl & yl) ^ (xl & zl) ^ (yl & zl);\n  if (r < 0)\n    r += 0x100000000;\n  return r;\n}\n\nfunction s0_512_hi(xh, xl) {\n  var c0_hi = rotr64_hi(xh, xl, 28);\n  var c1_hi = rotr64_hi(xl, xh, 2);  // 34\n  var c2_hi = rotr64_hi(xl, xh, 7);  // 39\n\n  var r = c0_hi ^ c1_hi ^ c2_hi;\n  if (r < 0)\n    r += 0x100000000;\n  return r;\n}\n\nfunction s0_512_lo(xh, xl) {\n  var c0_lo = rotr64_lo(xh, xl, 28);\n  var c1_lo = rotr64_lo(xl, xh, 2);  // 34\n  var c2_lo = rotr64_lo(xl, xh, 7);  // 39\n\n  var r = c0_lo ^ c1_lo ^ c2_lo;\n  if (r < 0)\n    r += 0x100000000;\n  return r;\n}\n\nfunction s1_512_hi(xh, xl) {\n  var c0_hi = rotr64_hi(xh, xl, 14);\n  var c1_hi = rotr64_hi(xh, xl, 18);\n  var c2_hi = rotr64_hi(xl, xh, 9);  // 41\n\n  var r = c0_hi ^ c1_hi ^ c2_hi;\n  if (r < 0)\n    r += 0x100000000;\n  return r;\n}\n\nfunction s1_512_lo(xh, xl) {\n  var c0_lo = rotr64_lo(xh, xl, 14);\n  var c1_lo = rotr64_lo(xh, xl, 18);\n  var c2_lo = rotr64_lo(xl, xh, 9);  // 41\n\n  var r = c0_lo ^ c1_lo ^ c2_lo;\n  if (r < 0)\n    r += 0x100000000;\n  return r;\n}\n\nfunction g0_512_hi(xh, xl) {\n  var c0_hi = rotr64_hi(xh, xl, 1);\n  var c1_hi = rotr64_hi(xh, xl, 8);\n  var c2_hi = shr64_hi(xh, xl, 7);\n\n  var r = c0_hi ^ c1_hi ^ c2_hi;\n  if (r < 0)\n    r += 0x100000000;\n  return r;\n}\n\nfunction g0_512_lo(xh, xl) {\n  var c0_lo = rotr64_lo(xh, xl, 1);\n  var c1_lo = rotr64_lo(xh, xl, 8);\n  var c2_lo = shr64_lo(xh, xl, 7);\n\n  var r = c0_lo ^ c1_lo ^ c2_lo;\n  if (r < 0)\n    r += 0x100000000;\n  return r;\n}\n\nfunction g1_512_hi(xh, xl) {\n  var c0_hi = rotr64_hi(xh, xl, 19);\n  var c1_hi = rotr64_hi(xl, xh, 29);  // 61\n  var c2_hi = shr64_hi(xh, xl, 6);\n\n  var r = c0_hi ^ c1_hi ^ c2_hi;\n  if (r < 0)\n    r += 0x100000000;\n  return r;\n}\n\nfunction g1_512_lo(xh, xl) {\n  var c0_lo = rotr64_lo(xh, xl, 19);\n  var c1_lo = rotr64_lo(xl, xh, 29);  // 61\n  var c2_lo = shr64_lo(xh, xl, 6);\n\n  var r = c0_lo ^ c1_lo ^ c2_lo;\n  if (r < 0)\n    r += 0x100000000;\n  return r;\n}\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///5900\n")},7038:(__unused_webpack_module,exports,__webpack_require__)=>{"use strict";eval("\n\nvar utils = __webpack_require__(6436);\nvar rotr32 = utils.rotr32;\n\nfunction ft_1(s, x, y, z) {\n  if (s === 0)\n    return ch32(x, y, z);\n  if (s === 1 || s === 3)\n    return p32(x, y, z);\n  if (s === 2)\n    return maj32(x, y, z);\n}\nexports.ft_1 = ft_1;\n\nfunction ch32(x, y, z) {\n  return (x & y) ^ ((~x) & z);\n}\nexports.ch32 = ch32;\n\nfunction maj32(x, y, z) {\n  return (x & y) ^ (x & z) ^ (y & z);\n}\nexports.maj32 = maj32;\n\nfunction p32(x, y, z) {\n  return x ^ y ^ z;\n}\nexports.p32 = p32;\n\nfunction s0_256(x) {\n  return rotr32(x, 2) ^ rotr32(x, 13) ^ rotr32(x, 22);\n}\nexports.s0_256 = s0_256;\n\nfunction s1_256(x) {\n  return rotr32(x, 6) ^ rotr32(x, 11) ^ rotr32(x, 25);\n}\nexports.s1_256 = s1_256;\n\nfunction g0_256(x) {\n  return rotr32(x, 7) ^ rotr32(x, 18) ^ (x >>> 3);\n}\nexports.g0_256 = g0_256;\n\nfunction g1_256(x) {\n  return rotr32(x, 17) ^ rotr32(x, 19) ^ (x >>> 10);\n}\nexports.g1_256 = g1_256;\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2hhc2guanMvbGliL2hhc2gvc2hhL2NvbW1vbi5qcz9hYTU2Il0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFhOztBQUViLFlBQVksbUJBQU8sQ0FBQyxJQUFVO0FBQzlCOztBQUVBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQSxZQUFZOztBQUVaO0FBQ0E7QUFDQTtBQUNBLFlBQVk7O0FBRVo7QUFDQTtBQUNBO0FBQ0EsYUFBYTs7QUFFYjtBQUNBO0FBQ0E7QUFDQSxXQUFXOztBQUVYO0FBQ0E7QUFDQTtBQUNBLGNBQWM7O0FBRWQ7QUFDQTtBQUNBO0FBQ0EsY0FBYzs7QUFFZDtBQUNBO0FBQ0E7QUFDQSxjQUFjOztBQUVkO0FBQ0E7QUFDQTtBQUNBLGNBQWMiLCJmaWxlIjoiNzAzOC5qcyIsInNvdXJjZXNDb250ZW50IjpbIid1c2Ugc3RyaWN0JztcblxudmFyIHV0aWxzID0gcmVxdWlyZSgnLi4vdXRpbHMnKTtcbnZhciByb3RyMzIgPSB1dGlscy5yb3RyMzI7XG5cbmZ1bmN0aW9uIGZ0XzEocywgeCwgeSwgeikge1xuICBpZiAocyA9PT0gMClcbiAgICByZXR1cm4gY2gzMih4LCB5LCB6KTtcbiAgaWYgKHMgPT09IDEgfHwgcyA9PT0gMylcbiAgICByZXR1cm4gcDMyKHgsIHksIHopO1xuICBpZiAocyA9PT0gMilcbiAgICByZXR1cm4gbWFqMzIoeCwgeSwgeik7XG59XG5leHBvcnRzLmZ0XzEgPSBmdF8xO1xuXG5mdW5jdGlvbiBjaDMyKHgsIHksIHopIHtcbiAgcmV0dXJuICh4ICYgeSkgXiAoKH54KSAmIHopO1xufVxuZXhwb3J0cy5jaDMyID0gY2gzMjtcblxuZnVuY3Rpb24gbWFqMzIoeCwgeSwgeikge1xuICByZXR1cm4gKHggJiB5KSBeICh4ICYgeikgXiAoeSAmIHopO1xufVxuZXhwb3J0cy5tYWozMiA9IG1hajMyO1xuXG5mdW5jdGlvbiBwMzIoeCwgeSwgeikge1xuICByZXR1cm4geCBeIHkgXiB6O1xufVxuZXhwb3J0cy5wMzIgPSBwMzI7XG5cbmZ1bmN0aW9uIHMwXzI1Nih4KSB7XG4gIHJldHVybiByb3RyMzIoeCwgMikgXiByb3RyMzIoeCwgMTMpIF4gcm90cjMyKHgsIDIyKTtcbn1cbmV4cG9ydHMuczBfMjU2ID0gczBfMjU2O1xuXG5mdW5jdGlvbiBzMV8yNTYoeCkge1xuICByZXR1cm4gcm90cjMyKHgsIDYpIF4gcm90cjMyKHgsIDExKSBeIHJvdHIzMih4LCAyNSk7XG59XG5leHBvcnRzLnMxXzI1NiA9IHMxXzI1NjtcblxuZnVuY3Rpb24gZzBfMjU2KHgpIHtcbiAgcmV0dXJuIHJvdHIzMih4LCA3KSBeIHJvdHIzMih4LCAxOCkgXiAoeCA+Pj4gMyk7XG59XG5leHBvcnRzLmcwXzI1NiA9IGcwXzI1NjtcblxuZnVuY3Rpb24gZzFfMjU2KHgpIHtcbiAgcmV0dXJuIHJvdHIzMih4LCAxNykgXiByb3RyMzIoeCwgMTkpIF4gKHggPj4+IDEwKTtcbn1cbmV4cG9ydHMuZzFfMjU2ID0gZzFfMjU2O1xuIl0sInNvdXJjZVJvb3QiOiIifQ==\n//# sourceURL=webpack-internal:///7038\n")},6436:(__unused_webpack_module,exports,__webpack_require__)=>{"use strict";eval("\n\nvar assert = __webpack_require__(9746);\nvar inherits = __webpack_require__(5717);\n\nexports.inherits = inherits;\n\nfunction isSurrogatePair(msg, i) {\n  if ((msg.charCodeAt(i) & 0xFC00) !== 0xD800) {\n    return false;\n  }\n  if (i < 0 || i + 1 >= msg.length) {\n    return false;\n  }\n  return (msg.charCodeAt(i + 1) & 0xFC00) === 0xDC00;\n}\n\nfunction toArray(msg, enc) {\n  if (Array.isArray(msg))\n    return msg.slice();\n  if (!msg)\n    return [];\n  var res = [];\n  if (typeof msg === 'string') {\n    if (!enc) {\n      // Inspired by stringToUtf8ByteArray() in closure-library by Google\n      // https://github.com/google/closure-library/blob/8598d87242af59aac233270742c8984e2b2bdbe0/closure/goog/crypt/crypt.js#L117-L143\n      // Apache License 2.0\n      // https://github.com/google/closure-library/blob/master/LICENSE\n      var p = 0;\n      for (var i = 0; i < msg.length; i++) {\n        var c = msg.charCodeAt(i);\n        if (c < 128) {\n          res[p++] = c;\n        } else if (c < 2048) {\n          res[p++] = (c >> 6) | 192;\n          res[p++] = (c & 63) | 128;\n        } else if (isSurrogatePair(msg, i)) {\n          c = 0x10000 + ((c & 0x03FF) << 10) + (msg.charCodeAt(++i) & 0x03FF);\n          res[p++] = (c >> 18) | 240;\n          res[p++] = ((c >> 12) & 63) | 128;\n          res[p++] = ((c >> 6) & 63) | 128;\n          res[p++] = (c & 63) | 128;\n        } else {\n          res[p++] = (c >> 12) | 224;\n          res[p++] = ((c >> 6) & 63) | 128;\n          res[p++] = (c & 63) | 128;\n        }\n      }\n    } else if (enc === 'hex') {\n      msg = msg.replace(/[^a-z0-9]+/ig, '');\n      if (msg.length % 2 !== 0)\n        msg = '0' + msg;\n      for (i = 0; i < msg.length; i += 2)\n        res.push(parseInt(msg[i] + msg[i + 1], 16));\n    }\n  } else {\n    for (i = 0; i < msg.length; i++)\n      res[i] = msg[i] | 0;\n  }\n  return res;\n}\nexports.toArray = toArray;\n\nfunction toHex(msg) {\n  var res = '';\n  for (var i = 0; i < msg.length; i++)\n    res += zero2(msg[i].toString(16));\n  return res;\n}\nexports.toHex = toHex;\n\nfunction htonl(w) {\n  var res = (w >>> 24) |\n            ((w >>> 8) & 0xff00) |\n            ((w << 8) & 0xff0000) |\n            ((w & 0xff) << 24);\n  return res >>> 0;\n}\nexports.htonl = htonl;\n\nfunction toHex32(msg, endian) {\n  var res = '';\n  for (var i = 0; i < msg.length; i++) {\n    var w = msg[i];\n    if (endian === 'little')\n      w = htonl(w);\n    res += zero8(w.toString(16));\n  }\n  return res;\n}\nexports.toHex32 = toHex32;\n\nfunction zero2(word) {\n  if (word.length === 1)\n    return '0' + word;\n  else\n    return word;\n}\nexports.zero2 = zero2;\n\nfunction zero8(word) {\n  if (word.length === 7)\n    return '0' + word;\n  else if (word.length === 6)\n    return '00' + word;\n  else if (word.length === 5)\n    return '000' + word;\n  else if (word.length === 4)\n    return '0000' + word;\n  else if (word.length === 3)\n    return '00000' + word;\n  else if (word.length === 2)\n    return '000000' + word;\n  else if (word.length === 1)\n    return '0000000' + word;\n  else\n    return word;\n}\nexports.zero8 = zero8;\n\nfunction join32(msg, start, end, endian) {\n  var len = end - start;\n  assert(len % 4 === 0);\n  var res = new Array(len / 4);\n  for (var i = 0, k = start; i < res.length; i++, k += 4) {\n    var w;\n    if (endian === 'big')\n      w = (msg[k] << 24) | (msg[k + 1] << 16) | (msg[k + 2] << 8) | msg[k + 3];\n    else\n      w = (msg[k + 3] << 24) | (msg[k + 2] << 16) | (msg[k + 1] << 8) | msg[k];\n    res[i] = w >>> 0;\n  }\n  return res;\n}\nexports.join32 = join32;\n\nfunction split32(msg, endian) {\n  var res = new Array(msg.length * 4);\n  for (var i = 0, k = 0; i < msg.length; i++, k += 4) {\n    var m = msg[i];\n    if (endian === 'big') {\n      res[k] = m >>> 24;\n      res[k + 1] = (m >>> 16) & 0xff;\n      res[k + 2] = (m >>> 8) & 0xff;\n      res[k + 3] = m & 0xff;\n    } else {\n      res[k + 3] = m >>> 24;\n      res[k + 2] = (m >>> 16) & 0xff;\n      res[k + 1] = (m >>> 8) & 0xff;\n      res[k] = m & 0xff;\n    }\n  }\n  return res;\n}\nexports.split32 = split32;\n\nfunction rotr32(w, b) {\n  return (w >>> b) | (w << (32 - b));\n}\nexports.rotr32 = rotr32;\n\nfunction rotl32(w, b) {\n  return (w << b) | (w >>> (32 - b));\n}\nexports.rotl32 = rotl32;\n\nfunction sum32(a, b) {\n  return (a + b) >>> 0;\n}\nexports.sum32 = sum32;\n\nfunction sum32_3(a, b, c) {\n  return (a + b + c) >>> 0;\n}\nexports.sum32_3 = sum32_3;\n\nfunction sum32_4(a, b, c, d) {\n  return (a + b + c + d) >>> 0;\n}\nexports.sum32_4 = sum32_4;\n\nfunction sum32_5(a, b, c, d, e) {\n  return (a + b + c + d + e) >>> 0;\n}\nexports.sum32_5 = sum32_5;\n\nfunction sum64(buf, pos, ah, al) {\n  var bh = buf[pos];\n  var bl = buf[pos + 1];\n\n  var lo = (al + bl) >>> 0;\n  var hi = (lo < al ? 1 : 0) + ah + bh;\n  buf[pos] = hi >>> 0;\n  buf[pos + 1] = lo;\n}\nexports.sum64 = sum64;\n\nfunction sum64_hi(ah, al, bh, bl) {\n  var lo = (al + bl) >>> 0;\n  var hi = (lo < al ? 1 : 0) + ah + bh;\n  return hi >>> 0;\n}\nexports.sum64_hi = sum64_hi;\n\nfunction sum64_lo(ah, al, bh, bl) {\n  var lo = al + bl;\n  return lo >>> 0;\n}\nexports.sum64_lo = sum64_lo;\n\nfunction sum64_4_hi(ah, al, bh, bl, ch, cl, dh, dl) {\n  var carry = 0;\n  var lo = al;\n  lo = (lo + bl) >>> 0;\n  carry += lo < al ? 1 : 0;\n  lo = (lo + cl) >>> 0;\n  carry += lo < cl ? 1 : 0;\n  lo = (lo + dl) >>> 0;\n  carry += lo < dl ? 1 : 0;\n\n  var hi = ah + bh + ch + dh + carry;\n  return hi >>> 0;\n}\nexports.sum64_4_hi = sum64_4_hi;\n\nfunction sum64_4_lo(ah, al, bh, bl, ch, cl, dh, dl) {\n  var lo = al + bl + cl + dl;\n  return lo >>> 0;\n}\nexports.sum64_4_lo = sum64_4_lo;\n\nfunction sum64_5_hi(ah, al, bh, bl, ch, cl, dh, dl, eh, el) {\n  var carry = 0;\n  var lo = al;\n  lo = (lo + bl) >>> 0;\n  carry += lo < al ? 1 : 0;\n  lo = (lo + cl) >>> 0;\n  carry += lo < cl ? 1 : 0;\n  lo = (lo + dl) >>> 0;\n  carry += lo < dl ? 1 : 0;\n  lo = (lo + el) >>> 0;\n  carry += lo < el ? 1 : 0;\n\n  var hi = ah + bh + ch + dh + eh + carry;\n  return hi >>> 0;\n}\nexports.sum64_5_hi = sum64_5_hi;\n\nfunction sum64_5_lo(ah, al, bh, bl, ch, cl, dh, dl, eh, el) {\n  var lo = al + bl + cl + dl + el;\n\n  return lo >>> 0;\n}\nexports.sum64_5_lo = sum64_5_lo;\n\nfunction rotr64_hi(ah, al, num) {\n  var r = (al << (32 - num)) | (ah >>> num);\n  return r >>> 0;\n}\nexports.rotr64_hi = rotr64_hi;\n\nfunction rotr64_lo(ah, al, num) {\n  var r = (ah << (32 - num)) | (al >>> num);\n  return r >>> 0;\n}\nexports.rotr64_lo = rotr64_lo;\n\nfunction shr64_hi(ah, al, num) {\n  return ah >>> num;\n}\nexports.shr64_hi = shr64_hi;\n\nfunction shr64_lo(ah, al, num) {\n  var r = (ah << (32 - num)) | (al >>> num);\n  return r >>> 0;\n}\nexports.shr64_lo = shr64_lo;\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///6436\n")},5717:module=>{eval("if (typeof Object.create === 'function') {\n  // implementation from standard node.js 'util' module\n  module.exports = function inherits(ctor, superCtor) {\n    if (superCtor) {\n      ctor.super_ = superCtor\n      ctor.prototype = Object.create(superCtor.prototype, {\n        constructor: {\n          value: ctor,\n          enumerable: false,\n          writable: true,\n          configurable: true\n        }\n      })\n    }\n  };\n} else {\n  // old school shim for old browsers\n  module.exports = function inherits(ctor, superCtor) {\n    if (superCtor) {\n      ctor.super_ = superCtor\n      var TempCtor = function () {}\n      TempCtor.prototype = superCtor.prototype\n      ctor.prototype = new TempCtor()\n      ctor.prototype.constructor = ctor\n    }\n  }\n}\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2luaGVyaXRzL2luaGVyaXRzX2Jyb3dzZXIuanM/M2ZiNSJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQSxPQUFPO0FBQ1A7QUFDQTtBQUNBLENBQUM7QUFDRDtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBIiwiZmlsZSI6IjU3MTcuanMiLCJzb3VyY2VzQ29udGVudCI6WyJpZiAodHlwZW9mIE9iamVjdC5jcmVhdGUgPT09ICdmdW5jdGlvbicpIHtcbiAgLy8gaW1wbGVtZW50YXRpb24gZnJvbSBzdGFuZGFyZCBub2RlLmpzICd1dGlsJyBtb2R1bGVcbiAgbW9kdWxlLmV4cG9ydHMgPSBmdW5jdGlvbiBpbmhlcml0cyhjdG9yLCBzdXBlckN0b3IpIHtcbiAgICBpZiAoc3VwZXJDdG9yKSB7XG4gICAgICBjdG9yLnN1cGVyXyA9IHN1cGVyQ3RvclxuICAgICAgY3Rvci5wcm90b3R5cGUgPSBPYmplY3QuY3JlYXRlKHN1cGVyQ3Rvci5wcm90b3R5cGUsIHtcbiAgICAgICAgY29uc3RydWN0b3I6IHtcbiAgICAgICAgICB2YWx1ZTogY3RvcixcbiAgICAgICAgICBlbnVtZXJhYmxlOiBmYWxzZSxcbiAgICAgICAgICB3cml0YWJsZTogdHJ1ZSxcbiAgICAgICAgICBjb25maWd1cmFibGU6IHRydWVcbiAgICAgICAgfVxuICAgICAgfSlcbiAgICB9XG4gIH07XG59IGVsc2Uge1xuICAvLyBvbGQgc2Nob29sIHNoaW0gZm9yIG9sZCBicm93c2Vyc1xuICBtb2R1bGUuZXhwb3J0cyA9IGZ1bmN0aW9uIGluaGVyaXRzKGN0b3IsIHN1cGVyQ3Rvcikge1xuICAgIGlmIChzdXBlckN0b3IpIHtcbiAgICAgIGN0b3Iuc3VwZXJfID0gc3VwZXJDdG9yXG4gICAgICB2YXIgVGVtcEN0b3IgPSBmdW5jdGlvbiAoKSB7fVxuICAgICAgVGVtcEN0b3IucHJvdG90eXBlID0gc3VwZXJDdG9yLnByb3RvdHlwZVxuICAgICAgY3Rvci5wcm90b3R5cGUgPSBuZXcgVGVtcEN0b3IoKVxuICAgICAgY3Rvci5wcm90b3R5cGUuY29uc3RydWN0b3IgPSBjdG9yXG4gICAgfVxuICB9XG59XG4iXSwic291cmNlUm9vdCI6IiJ9\n//# sourceURL=webpack-internal:///5717\n")},9496:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\nvar has = __webpack_require__(7642);\nvar channel = __webpack_require__(7478)();\n\nvar $TypeError = GetIntrinsic('%TypeError%');\n\nvar SLOT = {\n\tassert: function (O, slot) {\n\t\tif (!O || (typeof O !== 'object' && typeof O !== 'function')) {\n\t\t\tthrow new $TypeError('`O` is not an object');\n\t\t}\n\t\tif (typeof slot !== 'string') {\n\t\t\tthrow new $TypeError('`slot` must be a string');\n\t\t}\n\t\tchannel.assert(O);\n\t},\n\tget: function (O, slot) {\n\t\tif (!O || (typeof O !== 'object' && typeof O !== 'function')) {\n\t\t\tthrow new $TypeError('`O` is not an object');\n\t\t}\n\t\tif (typeof slot !== 'string') {\n\t\t\tthrow new $TypeError('`slot` must be a string');\n\t\t}\n\t\tvar slots = channel.get(O);\n\t\treturn slots && slots['$' + slot];\n\t},\n\thas: function (O, slot) {\n\t\tif (!O || (typeof O !== 'object' && typeof O !== 'function')) {\n\t\t\tthrow new $TypeError('`O` is not an object');\n\t\t}\n\t\tif (typeof slot !== 'string') {\n\t\t\tthrow new $TypeError('`slot` must be a string');\n\t\t}\n\t\tvar slots = channel.get(O);\n\t\treturn !!slots && has(slots, '$' + slot);\n\t},\n\tset: function (O, slot, V) {\n\t\tif (!O || (typeof O !== 'object' && typeof O !== 'function')) {\n\t\t\tthrow new $TypeError('`O` is not an object');\n\t\t}\n\t\tif (typeof slot !== 'string') {\n\t\t\tthrow new $TypeError('`slot` must be a string');\n\t\t}\n\t\tvar slots = channel.get(O);\n\t\tif (!slots) {\n\t\t\tslots = {};\n\t\t\tchannel.set(O, slots);\n\t\t}\n\t\tslots['$' + slot] = V;\n\t}\n};\n\nif (Object.freeze) {\n\tObject.freeze(SLOT);\n}\n\nmodule.exports = SLOT;\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///9496\n")},5320:module=>{"use strict";eval("\n\nvar fnToStr = Function.prototype.toString;\nvar reflectApply = typeof Reflect === 'object' && Reflect !== null && Reflect.apply;\nvar badArrayLike;\nvar isCallableMarker;\nif (typeof reflectApply === 'function' && typeof Object.defineProperty === 'function') {\n\ttry {\n\t\tbadArrayLike = Object.defineProperty({}, 'length', {\n\t\t\tget: function () {\n\t\t\t\tthrow isCallableMarker;\n\t\t\t}\n\t\t});\n\t\tisCallableMarker = {};\n\t\t// eslint-disable-next-line no-throw-literal\n\t\treflectApply(function () { throw 42; }, null, badArrayLike);\n\t} catch (_) {\n\t\tif (_ !== isCallableMarker) {\n\t\t\treflectApply = null;\n\t\t}\n\t}\n} else {\n\treflectApply = null;\n}\n\nvar constructorRegex = /^\\s*class\\b/;\nvar isES6ClassFn = function isES6ClassFunction(value) {\n\ttry {\n\t\tvar fnStr = fnToStr.call(value);\n\t\treturn constructorRegex.test(fnStr);\n\t} catch (e) {\n\t\treturn false; // not a function\n\t}\n};\n\nvar tryFunctionObject = function tryFunctionToStr(value) {\n\ttry {\n\t\tif (isES6ClassFn(value)) { return false; }\n\t\tfnToStr.call(value);\n\t\treturn true;\n\t} catch (e) {\n\t\treturn false;\n\t}\n};\nvar toStr = Object.prototype.toString;\nvar fnClass = '[object Function]';\nvar genClass = '[object GeneratorFunction]';\nvar hasToStringTag = typeof Symbol === 'function' && !!Symbol.toStringTag; // better: use `has-tostringtag`\n/* globals document: false */\nvar documentDotAll = typeof document === 'object' && typeof document.all === 'undefined' && document.all !== undefined ? document.all : {};\n\nmodule.exports = reflectApply\n\t? function isCallable(value) {\n\t\tif (value === documentDotAll) { return true; }\n\t\tif (!value) { return false; }\n\t\tif (typeof value !== 'function' && typeof value !== 'object') { return false; }\n\t\tif (typeof value === 'function' && !value.prototype) { return true; }\n\t\ttry {\n\t\t\treflectApply(value, null, badArrayLike);\n\t\t} catch (e) {\n\t\t\tif (e !== isCallableMarker) { return false; }\n\t\t}\n\t\treturn !isES6ClassFn(value);\n\t}\n\t: function isCallable(value) {\n\t\tif (value === documentDotAll) { return true; }\n\t\tif (!value) { return false; }\n\t\tif (typeof value !== 'function' && typeof value !== 'object') { return false; }\n\t\tif (typeof value === 'function' && !value.prototype) { return true; }\n\t\tif (hasToStringTag) { return tryFunctionObject(value); }\n\t\tif (isES6ClassFn(value)) { return false; }\n\t\tvar strClass = toStr.call(value);\n\t\treturn strClass === fnClass || strClass === genClass;\n\t};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///5320\n")},8923:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar getDay = Date.prototype.getDay;\nvar tryDateObject = function tryDateGetDayCall(value) {\n\ttry {\n\t\tgetDay.call(value);\n\t\treturn true;\n\t} catch (e) {\n\t\treturn false;\n\t}\n};\n\nvar toStr = Object.prototype.toString;\nvar dateClass = '[object Date]';\nvar hasToStringTag = __webpack_require__(6410)();\n\nmodule.exports = function isDateObject(value) {\n\tif (typeof value !== 'object' || value === null) {\n\t\treturn false;\n\t}\n\treturn hasToStringTag ? tryDateObject(value) : toStr.call(value) === dateClass;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2lzLWRhdGUtb2JqZWN0L2luZGV4LmpzPzBlNjUiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQWE7O0FBRWI7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBLEVBQUU7QUFDRjtBQUNBO0FBQ0E7O0FBRUE7QUFDQTtBQUNBLHFCQUFxQixtQkFBTyxDQUFDLElBQXVCOztBQUVwRDtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0EiLCJmaWxlIjoiODkyMy5qcyIsInNvdXJjZXNDb250ZW50IjpbIid1c2Ugc3RyaWN0JztcblxudmFyIGdldERheSA9IERhdGUucHJvdG90eXBlLmdldERheTtcbnZhciB0cnlEYXRlT2JqZWN0ID0gZnVuY3Rpb24gdHJ5RGF0ZUdldERheUNhbGwodmFsdWUpIHtcblx0dHJ5IHtcblx0XHRnZXREYXkuY2FsbCh2YWx1ZSk7XG5cdFx0cmV0dXJuIHRydWU7XG5cdH0gY2F0Y2ggKGUpIHtcblx0XHRyZXR1cm4gZmFsc2U7XG5cdH1cbn07XG5cbnZhciB0b1N0ciA9IE9iamVjdC5wcm90b3R5cGUudG9TdHJpbmc7XG52YXIgZGF0ZUNsYXNzID0gJ1tvYmplY3QgRGF0ZV0nO1xudmFyIGhhc1RvU3RyaW5nVGFnID0gcmVxdWlyZSgnaGFzLXRvc3RyaW5ndGFnL3NoYW1zJykoKTtcblxubW9kdWxlLmV4cG9ydHMgPSBmdW5jdGlvbiBpc0RhdGVPYmplY3QodmFsdWUpIHtcblx0aWYgKHR5cGVvZiB2YWx1ZSAhPT0gJ29iamVjdCcgfHwgdmFsdWUgPT09IG51bGwpIHtcblx0XHRyZXR1cm4gZmFsc2U7XG5cdH1cblx0cmV0dXJuIGhhc1RvU3RyaW5nVGFnID8gdHJ5RGF0ZU9iamVjdCh2YWx1ZSkgOiB0b1N0ci5jYWxsKHZhbHVlKSA9PT0gZGF0ZUNsYXNzO1xufTtcbiJdLCJzb3VyY2VSb290IjoiIn0=\n//# sourceURL=webpack-internal:///8923\n")},8420:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar callBound = __webpack_require__(1924);\nvar hasToStringTag = __webpack_require__(6410)();\nvar has;\nvar $exec;\nvar isRegexMarker;\nvar badStringifier;\n\nif (hasToStringTag) {\n\thas = callBound('Object.prototype.hasOwnProperty');\n\t$exec = callBound('RegExp.prototype.exec');\n\tisRegexMarker = {};\n\n\tvar throwRegexMarker = function () {\n\t\tthrow isRegexMarker;\n\t};\n\tbadStringifier = {\n\t\ttoString: throwRegexMarker,\n\t\tvalueOf: throwRegexMarker\n\t};\n\n\tif (typeof Symbol.toPrimitive === 'symbol') {\n\t\tbadStringifier[Symbol.toPrimitive] = throwRegexMarker;\n\t}\n}\n\nvar $toString = callBound('Object.prototype.toString');\nvar gOPD = Object.getOwnPropertyDescriptor;\nvar regexClass = '[object RegExp]';\n\nmodule.exports = hasToStringTag\n\t// eslint-disable-next-line consistent-return\n\t? function isRegex(value) {\n\t\tif (!value || typeof value !== 'object') {\n\t\t\treturn false;\n\t\t}\n\n\t\tvar descriptor = gOPD(value, 'lastIndex');\n\t\tvar hasLastIndexDataProperty = descriptor && has(descriptor, 'value');\n\t\tif (!hasLastIndexDataProperty) {\n\t\t\treturn false;\n\t\t}\n\n\t\ttry {\n\t\t\t$exec(value, badStringifier);\n\t\t} catch (e) {\n\t\t\treturn e === isRegexMarker;\n\t\t}\n\t}\n\t: function isRegex(value) {\n\t\t// In older browsers, typeof regex incorrectly returns 'function'\n\t\tif (!value || (typeof value !== 'object' && typeof value !== 'function')) {\n\t\t\treturn false;\n\t\t}\n\n\t\treturn $toString(value) === regexClass;\n\t};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///8420\n")},2636:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar toStr = Object.prototype.toString;\nvar hasSymbols = __webpack_require__(1405)();\n\nif (hasSymbols) {\n\tvar symToStr = Symbol.prototype.toString;\n\tvar symStringRegex = /^Symbol\\(.*\\)$/;\n\tvar isSymbolObject = function isRealSymbolObject(value) {\n\t\tif (typeof value.valueOf() !== 'symbol') {\n\t\t\treturn false;\n\t\t}\n\t\treturn symStringRegex.test(symToStr.call(value));\n\t};\n\n\tmodule.exports = function isSymbol(value) {\n\t\tif (typeof value === 'symbol') {\n\t\t\treturn true;\n\t\t}\n\t\tif (toStr.call(value) !== '[object Symbol]') {\n\t\t\treturn false;\n\t\t}\n\t\ttry {\n\t\t\treturn isSymbolObject(value);\n\t\t} catch (e) {\n\t\t\treturn false;\n\t\t}\n\t};\n} else {\n\n\tmodule.exports = function isSymbol(value) {\n\t\t// this environment does not support Symbols.\n\t\treturn  false && 0;\n\t};\n}\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL2lzLXN5bWJvbC9pbmRleC5qcz9mZWM1Il0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFhOztBQUViO0FBQ0EsaUJBQWlCLG1CQUFPLENBQUMsSUFBYTs7QUFFdEM7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBOztBQUVBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBLEdBQUc7QUFDSDtBQUNBO0FBQ0E7QUFDQSxDQUFDOztBQUVEO0FBQ0E7QUFDQSxTQUFTLE1BQUssSUFBSSxDQUFLO0FBQ3ZCO0FBQ0EiLCJmaWxlIjoiMjYzNi5qcyIsInNvdXJjZXNDb250ZW50IjpbIid1c2Ugc3RyaWN0JztcblxudmFyIHRvU3RyID0gT2JqZWN0LnByb3RvdHlwZS50b1N0cmluZztcbnZhciBoYXNTeW1ib2xzID0gcmVxdWlyZSgnaGFzLXN5bWJvbHMnKSgpO1xuXG5pZiAoaGFzU3ltYm9scykge1xuXHR2YXIgc3ltVG9TdHIgPSBTeW1ib2wucHJvdG90eXBlLnRvU3RyaW5nO1xuXHR2YXIgc3ltU3RyaW5nUmVnZXggPSAvXlN5bWJvbFxcKC4qXFwpJC87XG5cdHZhciBpc1N5bWJvbE9iamVjdCA9IGZ1bmN0aW9uIGlzUmVhbFN5bWJvbE9iamVjdCh2YWx1ZSkge1xuXHRcdGlmICh0eXBlb2YgdmFsdWUudmFsdWVPZigpICE9PSAnc3ltYm9sJykge1xuXHRcdFx0cmV0dXJuIGZhbHNlO1xuXHRcdH1cblx0XHRyZXR1cm4gc3ltU3RyaW5nUmVnZXgudGVzdChzeW1Ub1N0ci5jYWxsKHZhbHVlKSk7XG5cdH07XG5cblx0bW9kdWxlLmV4cG9ydHMgPSBmdW5jdGlvbiBpc1N5bWJvbCh2YWx1ZSkge1xuXHRcdGlmICh0eXBlb2YgdmFsdWUgPT09ICdzeW1ib2wnKSB7XG5cdFx0XHRyZXR1cm4gdHJ1ZTtcblx0XHR9XG5cdFx0aWYgKHRvU3RyLmNhbGwodmFsdWUpICE9PSAnW29iamVjdCBTeW1ib2xdJykge1xuXHRcdFx0cmV0dXJuIGZhbHNlO1xuXHRcdH1cblx0XHR0cnkge1xuXHRcdFx0cmV0dXJuIGlzU3ltYm9sT2JqZWN0KHZhbHVlKTtcblx0XHR9IGNhdGNoIChlKSB7XG5cdFx0XHRyZXR1cm4gZmFsc2U7XG5cdFx0fVxuXHR9O1xufSBlbHNlIHtcblxuXHRtb2R1bGUuZXhwb3J0cyA9IGZ1bmN0aW9uIGlzU3ltYm9sKHZhbHVlKSB7XG5cdFx0Ly8gdGhpcyBlbnZpcm9ubWVudCBkb2VzIG5vdCBzdXBwb3J0IFN5bWJvbHMuXG5cdFx0cmV0dXJuIGZhbHNlICYmIHZhbHVlO1xuXHR9O1xufVxuIl0sInNvdXJjZVJvb3QiOiIifQ==\n//# sourceURL=webpack-internal:///2636\n")},9746:module=>{eval("module.exports = assert;\n\nfunction assert(val, msg) {\n  if (!val)\n    throw new Error(msg || 'Assertion failed');\n}\n\nassert.equal = function assertEqual(l, r, msg) {\n  if (l != r)\n    throw new Error(msg || ('Assertion failed: ' + l + ' != ' + r));\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL21pbmltYWxpc3RpYy1hc3NlcnQvaW5kZXguanM/ZGEzZSJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBQTs7QUFFQTtBQUNBO0FBQ0E7QUFDQTs7QUFFQTtBQUNBO0FBQ0E7QUFDQSIsImZpbGUiOiI5NzQ2LmpzIiwic291cmNlc0NvbnRlbnQiOlsibW9kdWxlLmV4cG9ydHMgPSBhc3NlcnQ7XG5cbmZ1bmN0aW9uIGFzc2VydCh2YWwsIG1zZykge1xuICBpZiAoIXZhbClcbiAgICB0aHJvdyBuZXcgRXJyb3IobXNnIHx8ICdBc3NlcnRpb24gZmFpbGVkJyk7XG59XG5cbmFzc2VydC5lcXVhbCA9IGZ1bmN0aW9uIGFzc2VydEVxdWFsKGwsIHIsIG1zZykge1xuICBpZiAobCAhPSByKVxuICAgIHRocm93IG5ldyBFcnJvcihtc2cgfHwgKCdBc3NlcnRpb24gZmFpbGVkOiAnICsgbCArICcgIT0gJyArIHIpKTtcbn07XG4iXSwic291cmNlUm9vdCI6IiJ9\n//# sourceURL=webpack-internal:///9746\n")},631:(module,__unused_webpack_exports,__webpack_require__)=>{eval("var hasMap = typeof Map === 'function' && Map.prototype;\nvar mapSizeDescriptor = Object.getOwnPropertyDescriptor && hasMap ? Object.getOwnPropertyDescriptor(Map.prototype, 'size') : null;\nvar mapSize = hasMap && mapSizeDescriptor && typeof mapSizeDescriptor.get === 'function' ? mapSizeDescriptor.get : null;\nvar mapForEach = hasMap && Map.prototype.forEach;\nvar hasSet = typeof Set === 'function' && Set.prototype;\nvar setSizeDescriptor = Object.getOwnPropertyDescriptor && hasSet ? Object.getOwnPropertyDescriptor(Set.prototype, 'size') : null;\nvar setSize = hasSet && setSizeDescriptor && typeof setSizeDescriptor.get === 'function' ? setSizeDescriptor.get : null;\nvar setForEach = hasSet && Set.prototype.forEach;\nvar hasWeakMap = typeof WeakMap === 'function' && WeakMap.prototype;\nvar weakMapHas = hasWeakMap ? WeakMap.prototype.has : null;\nvar hasWeakSet = typeof WeakSet === 'function' && WeakSet.prototype;\nvar weakSetHas = hasWeakSet ? WeakSet.prototype.has : null;\nvar hasWeakRef = typeof WeakRef === 'function' && WeakRef.prototype;\nvar weakRefDeref = hasWeakRef ? WeakRef.prototype.deref : null;\nvar booleanValueOf = Boolean.prototype.valueOf;\nvar objectToString = Object.prototype.toString;\nvar functionToString = Function.prototype.toString;\nvar match = String.prototype.match;\nvar bigIntValueOf = typeof BigInt === 'function' ? BigInt.prototype.valueOf : null;\nvar gOPS = Object.getOwnPropertySymbols;\nvar symToString = typeof Symbol === 'function' && typeof Symbol.iterator === 'symbol' ? Symbol.prototype.toString : null;\nvar hasShammedSymbols = typeof Symbol === 'function' && typeof Symbol.iterator === 'object';\nvar isEnumerable = Object.prototype.propertyIsEnumerable;\n\nvar gPO = (typeof Reflect === 'function' ? Reflect.getPrototypeOf : Object.getPrototypeOf) || (\n    [].__proto__ === Array.prototype // eslint-disable-line no-proto\n        ? function (O) {\n            return O.__proto__; // eslint-disable-line no-proto\n        }\n        : null\n);\n\nvar inspectCustom = __webpack_require__(4654).custom;\nvar inspectSymbol = inspectCustom && isSymbol(inspectCustom) ? inspectCustom : null;\nvar toStringTag = typeof Symbol === 'function' && typeof Symbol.toStringTag !== 'undefined' ? Symbol.toStringTag : null;\n\nmodule.exports = function inspect_(obj, options, depth, seen) {\n    var opts = options || {};\n\n    if (has(opts, 'quoteStyle') && (opts.quoteStyle !== 'single' && opts.quoteStyle !== 'double')) {\n        throw new TypeError('option \"quoteStyle\" must be \"single\" or \"double\"');\n    }\n    if (\n        has(opts, 'maxStringLength') && (typeof opts.maxStringLength === 'number'\n            ? opts.maxStringLength < 0 && opts.maxStringLength !== Infinity\n            : opts.maxStringLength !== null\n        )\n    ) {\n        throw new TypeError('option \"maxStringLength\", if provided, must be a positive integer, Infinity, or `null`');\n    }\n    var customInspect = has(opts, 'customInspect') ? opts.customInspect : true;\n    if (typeof customInspect !== 'boolean' && customInspect !== 'symbol') {\n        throw new TypeError('option \"customInspect\", if provided, must be `true`, `false`, or `\\'symbol\\'`');\n    }\n\n    if (\n        has(opts, 'indent')\n        && opts.indent !== null\n        && opts.indent !== '\\t'\n        && !(parseInt(opts.indent, 10) === opts.indent && opts.indent > 0)\n    ) {\n        throw new TypeError('options \"indent\" must be \"\\\\t\", an integer > 0, or `null`');\n    }\n\n    if (typeof obj === 'undefined') {\n        return 'undefined';\n    }\n    if (obj === null) {\n        return 'null';\n    }\n    if (typeof obj === 'boolean') {\n        return obj ? 'true' : 'false';\n    }\n\n    if (typeof obj === 'string') {\n        return inspectString(obj, opts);\n    }\n    if (typeof obj === 'number') {\n        if (obj === 0) {\n            return Infinity / obj > 0 ? '0' : '-0';\n        }\n        return String(obj);\n    }\n    if (typeof obj === 'bigint') {\n        return String(obj) + 'n';\n    }\n\n    var maxDepth = typeof opts.depth === 'undefined' ? 5 : opts.depth;\n    if (typeof depth === 'undefined') { depth = 0; }\n    if (depth >= maxDepth && maxDepth > 0 && typeof obj === 'object') {\n        return isArray(obj) ? '[Array]' : '[Object]';\n    }\n\n    var indent = getIndent(opts, depth);\n\n    if (typeof seen === 'undefined') {\n        seen = [];\n    } else if (indexOf(seen, obj) >= 0) {\n        return '[Circular]';\n    }\n\n    function inspect(value, from, noIndent) {\n        if (from) {\n            seen = seen.slice();\n            seen.push(from);\n        }\n        if (noIndent) {\n            var newOpts = {\n                depth: opts.depth\n            };\n            if (has(opts, 'quoteStyle')) {\n                newOpts.quoteStyle = opts.quoteStyle;\n            }\n            return inspect_(value, newOpts, depth + 1, seen);\n        }\n        return inspect_(value, opts, depth + 1, seen);\n    }\n\n    if (typeof obj === 'function') {\n        var name = nameOf(obj);\n        var keys = arrObjKeys(obj, inspect);\n        return '[Function' + (name ? ': ' + name : ' (anonymous)') + ']' + (keys.length > 0 ? ' { ' + keys.join(', ') + ' }' : '');\n    }\n    if (isSymbol(obj)) {\n        var symString = hasShammedSymbols ? String(obj).replace(/^(Symbol\\(.*\\))_[^)]*$/, '$1') : symToString.call(obj);\n        return typeof obj === 'object' && !hasShammedSymbols ? markBoxed(symString) : symString;\n    }\n    if (isElement(obj)) {\n        var s = '<' + String(obj.nodeName).toLowerCase();\n        var attrs = obj.attributes || [];\n        for (var i = 0; i < attrs.length; i++) {\n            s += ' ' + attrs[i].name + '=' + wrapQuotes(quote(attrs[i].value), 'double', opts);\n        }\n        s += '>';\n        if (obj.childNodes && obj.childNodes.length) { s += '...'; }\n        s += '</' + String(obj.nodeName).toLowerCase() + '>';\n        return s;\n    }\n    if (isArray(obj)) {\n        if (obj.length === 0) { return '[]'; }\n        var xs = arrObjKeys(obj, inspect);\n        if (indent && !singleLineValues(xs)) {\n            return '[' + indentedJoin(xs, indent) + ']';\n        }\n        return '[ ' + xs.join(', ') + ' ]';\n    }\n    if (isError(obj)) {\n        var parts = arrObjKeys(obj, inspect);\n        if (parts.length === 0) { return '[' + String(obj) + ']'; }\n        return '{ [' + String(obj) + '] ' + parts.join(', ') + ' }';\n    }\n    if (typeof obj === 'object' && customInspect) {\n        if (inspectSymbol && typeof obj[inspectSymbol] === 'function') {\n            return obj[inspectSymbol]();\n        } else if (customInspect !== 'symbol' && typeof obj.inspect === 'function') {\n            return obj.inspect();\n        }\n    }\n    if (isMap(obj)) {\n        var mapParts = [];\n        mapForEach.call(obj, function (value, key) {\n            mapParts.push(inspect(key, obj, true) + ' => ' + inspect(value, obj));\n        });\n        return collectionOf('Map', mapSize.call(obj), mapParts, indent);\n    }\n    if (isSet(obj)) {\n        var setParts = [];\n        setForEach.call(obj, function (value) {\n            setParts.push(inspect(value, obj));\n        });\n        return collectionOf('Set', setSize.call(obj), setParts, indent);\n    }\n    if (isWeakMap(obj)) {\n        return weakCollectionOf('WeakMap');\n    }\n    if (isWeakSet(obj)) {\n        return weakCollectionOf('WeakSet');\n    }\n    if (isWeakRef(obj)) {\n        return weakCollectionOf('WeakRef');\n    }\n    if (isNumber(obj)) {\n        return markBoxed(inspect(Number(obj)));\n    }\n    if (isBigInt(obj)) {\n        return markBoxed(inspect(bigIntValueOf.call(obj)));\n    }\n    if (isBoolean(obj)) {\n        return markBoxed(booleanValueOf.call(obj));\n    }\n    if (isString(obj)) {\n        return markBoxed(inspect(String(obj)));\n    }\n    if (!isDate(obj) && !isRegExp(obj)) {\n        var ys = arrObjKeys(obj, inspect);\n        var isPlainObject = gPO ? gPO(obj) === Object.prototype : obj instanceof Object || obj.constructor === Object;\n        var protoTag = obj instanceof Object ? '' : 'null prototype';\n        var stringTag = !isPlainObject && toStringTag && Object(obj) === obj && toStringTag in obj ? toStr(obj).slice(8, -1) : protoTag ? 'Object' : '';\n        var constructorTag = isPlainObject || typeof obj.constructor !== 'function' ? '' : obj.constructor.name ? obj.constructor.name + ' ' : '';\n        var tag = constructorTag + (stringTag || protoTag ? '[' + [].concat(stringTag || [], protoTag || []).join(': ') + '] ' : '');\n        if (ys.length === 0) { return tag + '{}'; }\n        if (indent) {\n            return tag + '{' + indentedJoin(ys, indent) + '}';\n        }\n        return tag + '{ ' + ys.join(', ') + ' }';\n    }\n    return String(obj);\n};\n\nfunction wrapQuotes(s, defaultStyle, opts) {\n    var quoteChar = (opts.quoteStyle || defaultStyle) === 'double' ? '\"' : \"'\";\n    return quoteChar + s + quoteChar;\n}\n\nfunction quote(s) {\n    return String(s).replace(/\"/g, '&quot;');\n}\n\nfunction isArray(obj) { return toStr(obj) === '[object Array]' && (!toStringTag || !(typeof obj === 'object' && toStringTag in obj)); }\nfunction isDate(obj) { return toStr(obj) === '[object Date]' && (!toStringTag || !(typeof obj === 'object' && toStringTag in obj)); }\nfunction isRegExp(obj) { return toStr(obj) === '[object RegExp]' && (!toStringTag || !(typeof obj === 'object' && toStringTag in obj)); }\nfunction isError(obj) { return toStr(obj) === '[object Error]' && (!toStringTag || !(typeof obj === 'object' && toStringTag in obj)); }\nfunction isString(obj) { return toStr(obj) === '[object String]' && (!toStringTag || !(typeof obj === 'object' && toStringTag in obj)); }\nfunction isNumber(obj) { return toStr(obj) === '[object Number]' && (!toStringTag || !(typeof obj === 'object' && toStringTag in obj)); }\nfunction isBoolean(obj) { return toStr(obj) === '[object Boolean]' && (!toStringTag || !(typeof obj === 'object' && toStringTag in obj)); }\n\n// Symbol and BigInt do have Symbol.toStringTag by spec, so that can't be used to eliminate false positives\nfunction isSymbol(obj) {\n    if (hasShammedSymbols) {\n        return obj && typeof obj === 'object' && obj instanceof Symbol;\n    }\n    if (typeof obj === 'symbol') {\n        return true;\n    }\n    if (!obj || typeof obj !== 'object' || !symToString) {\n        return false;\n    }\n    try {\n        symToString.call(obj);\n        return true;\n    } catch (e) {}\n    return false;\n}\n\nfunction isBigInt(obj) {\n    if (!obj || typeof obj !== 'object' || !bigIntValueOf) {\n        return false;\n    }\n    try {\n        bigIntValueOf.call(obj);\n        return true;\n    } catch (e) {}\n    return false;\n}\n\nvar hasOwn = Object.prototype.hasOwnProperty || function (key) { return key in this; };\nfunction has(obj, key) {\n    return hasOwn.call(obj, key);\n}\n\nfunction toStr(obj) {\n    return objectToString.call(obj);\n}\n\nfunction nameOf(f) {\n    if (f.name) { return f.name; }\n    var m = match.call(functionToString.call(f), /^function\\s*([\\w$]+)/);\n    if (m) { return m[1]; }\n    return null;\n}\n\nfunction indexOf(xs, x) {\n    if (xs.indexOf) { return xs.indexOf(x); }\n    for (var i = 0, l = xs.length; i < l; i++) {\n        if (xs[i] === x) { return i; }\n    }\n    return -1;\n}\n\nfunction isMap(x) {\n    if (!mapSize || !x || typeof x !== 'object') {\n        return false;\n    }\n    try {\n        mapSize.call(x);\n        try {\n            setSize.call(x);\n        } catch (s) {\n            return true;\n        }\n        return x instanceof Map; // core-js workaround, pre-v2.5.0\n    } catch (e) {}\n    return false;\n}\n\nfunction isWeakMap(x) {\n    if (!weakMapHas || !x || typeof x !== 'object') {\n        return false;\n    }\n    try {\n        weakMapHas.call(x, weakMapHas);\n        try {\n            weakSetHas.call(x, weakSetHas);\n        } catch (s) {\n            return true;\n        }\n        return x instanceof WeakMap; // core-js workaround, pre-v2.5.0\n    } catch (e) {}\n    return false;\n}\n\nfunction isWeakRef(x) {\n    if (!weakRefDeref || !x || typeof x !== 'object') {\n        return false;\n    }\n    try {\n        weakRefDeref.call(x);\n        return true;\n    } catch (e) {}\n    return false;\n}\n\nfunction isSet(x) {\n    if (!setSize || !x || typeof x !== 'object') {\n        return false;\n    }\n    try {\n        setSize.call(x);\n        try {\n            mapSize.call(x);\n        } catch (m) {\n            return true;\n        }\n        return x instanceof Set; // core-js workaround, pre-v2.5.0\n    } catch (e) {}\n    return false;\n}\n\nfunction isWeakSet(x) {\n    if (!weakSetHas || !x || typeof x !== 'object') {\n        return false;\n    }\n    try {\n        weakSetHas.call(x, weakSetHas);\n        try {\n            weakMapHas.call(x, weakMapHas);\n        } catch (s) {\n            return true;\n        }\n        return x instanceof WeakSet; // core-js workaround, pre-v2.5.0\n    } catch (e) {}\n    return false;\n}\n\nfunction isElement(x) {\n    if (!x || typeof x !== 'object') { return false; }\n    if (typeof HTMLElement !== 'undefined' && x instanceof HTMLElement) {\n        return true;\n    }\n    return typeof x.nodeName === 'string' && typeof x.getAttribute === 'function';\n}\n\nfunction inspectString(str, opts) {\n    if (str.length > opts.maxStringLength) {\n        var remaining = str.length - opts.maxStringLength;\n        var trailer = '... ' + remaining + ' more character' + (remaining > 1 ? 's' : '');\n        return inspectString(str.slice(0, opts.maxStringLength), opts) + trailer;\n    }\n    // eslint-disable-next-line no-control-regex\n    var s = str.replace(/(['\\\\])/g, '\\\\$1').replace(/[\\x00-\\x1f]/g, lowbyte);\n    return wrapQuotes(s, 'single', opts);\n}\n\nfunction lowbyte(c) {\n    var n = c.charCodeAt(0);\n    var x = {\n        8: 'b',\n        9: 't',\n        10: 'n',\n        12: 'f',\n        13: 'r'\n    }[n];\n    if (x) { return '\\\\' + x; }\n    return '\\\\x' + (n < 0x10 ? '0' : '') + n.toString(16).toUpperCase();\n}\n\nfunction markBoxed(str) {\n    return 'Object(' + str + ')';\n}\n\nfunction weakCollectionOf(type) {\n    return type + ' { ? }';\n}\n\nfunction collectionOf(type, size, entries, indent) {\n    var joinedEntries = indent ? indentedJoin(entries, indent) : entries.join(', ');\n    return type + ' (' + size + ') {' + joinedEntries + '}';\n}\n\nfunction singleLineValues(xs) {\n    for (var i = 0; i < xs.length; i++) {\n        if (indexOf(xs[i], '\\n') >= 0) {\n            return false;\n        }\n    }\n    return true;\n}\n\nfunction getIndent(opts, depth) {\n    var baseIndent;\n    if (opts.indent === '\\t') {\n        baseIndent = '\\t';\n    } else if (typeof opts.indent === 'number' && opts.indent > 0) {\n        baseIndent = Array(opts.indent + 1).join(' ');\n    } else {\n        return null;\n    }\n    return {\n        base: baseIndent,\n        prev: Array(depth + 1).join(baseIndent)\n    };\n}\n\nfunction indentedJoin(xs, indent) {\n    if (xs.length === 0) { return ''; }\n    var lineJoiner = '\\n' + indent.prev + indent.base;\n    return lineJoiner + xs.join(',' + lineJoiner) + '\\n' + indent.prev;\n}\n\nfunction arrObjKeys(obj, inspect) {\n    var isArr = isArray(obj);\n    var xs = [];\n    if (isArr) {\n        xs.length = obj.length;\n        for (var i = 0; i < obj.length; i++) {\n            xs[i] = has(obj, i) ? inspect(obj[i], obj) : '';\n        }\n    }\n    var syms = typeof gOPS === 'function' ? gOPS(obj) : [];\n    var symMap;\n    if (hasShammedSymbols) {\n        symMap = {};\n        for (var k = 0; k < syms.length; k++) {\n            symMap['$' + syms[k]] = syms[k];\n        }\n    }\n\n    for (var key in obj) { // eslint-disable-line no-restricted-syntax\n        if (!has(obj, key)) { continue; } // eslint-disable-line no-restricted-syntax, no-continue\n        if (isArr && String(Number(key)) === key && key < obj.length) { continue; } // eslint-disable-line no-restricted-syntax, no-continue\n        if (hasShammedSymbols && symMap['$' + key] instanceof Symbol) {\n            // this is to prevent shammed Symbols, which are stored as strings, from being included in the string key section\n            continue; // eslint-disable-line no-restricted-syntax, no-continue\n        } else if ((/[^\\w$]/).test(key)) {\n            xs.push(inspect(key, obj) + ': ' + inspect(obj[key], obj));\n        } else {\n            xs.push(key + ': ' + inspect(obj[key], obj));\n        }\n    }\n    if (typeof gOPS === 'function') {\n        for (var j = 0; j < syms.length; j++) {\n            if (isEnumerable.call(obj, syms[j])) {\n                xs.push('[' + inspect(syms[j]) + ']: ' + inspect(obj[syms[j]], obj));\n            }\n        }\n    }\n    return xs;\n}\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///631\n")},8987:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar keysShim;\nif (!Object.keys) {\n\t// modified from https://github.com/es-shims/es5-shim\n\tvar has = Object.prototype.hasOwnProperty;\n\tvar toStr = Object.prototype.toString;\n\tvar isArgs = __webpack_require__(1414); // eslint-disable-line global-require\n\tvar isEnumerable = Object.prototype.propertyIsEnumerable;\n\tvar hasDontEnumBug = !isEnumerable.call({ toString: null }, 'toString');\n\tvar hasProtoEnumBug = isEnumerable.call(function () {}, 'prototype');\n\tvar dontEnums = [\n\t\t'toString',\n\t\t'toLocaleString',\n\t\t'valueOf',\n\t\t'hasOwnProperty',\n\t\t'isPrototypeOf',\n\t\t'propertyIsEnumerable',\n\t\t'constructor'\n\t];\n\tvar equalsConstructorPrototype = function (o) {\n\t\tvar ctor = o.constructor;\n\t\treturn ctor && ctor.prototype === o;\n\t};\n\tvar excludedKeys = {\n\t\t$applicationCache: true,\n\t\t$console: true,\n\t\t$external: true,\n\t\t$frame: true,\n\t\t$frameElement: true,\n\t\t$frames: true,\n\t\t$innerHeight: true,\n\t\t$innerWidth: true,\n\t\t$onmozfullscreenchange: true,\n\t\t$onmozfullscreenerror: true,\n\t\t$outerHeight: true,\n\t\t$outerWidth: true,\n\t\t$pageXOffset: true,\n\t\t$pageYOffset: true,\n\t\t$parent: true,\n\t\t$scrollLeft: true,\n\t\t$scrollTop: true,\n\t\t$scrollX: true,\n\t\t$scrollY: true,\n\t\t$self: true,\n\t\t$webkitIndexedDB: true,\n\t\t$webkitStorageInfo: true,\n\t\t$window: true\n\t};\n\tvar hasAutomationEqualityBug = (function () {\n\t\t/* global window */\n\t\tif (typeof window === 'undefined') { return false; }\n\t\tfor (var k in window) {\n\t\t\ttry {\n\t\t\t\tif (!excludedKeys['$' + k] && has.call(window, k) && window[k] !== null && typeof window[k] === 'object') {\n\t\t\t\t\ttry {\n\t\t\t\t\t\tequalsConstructorPrototype(window[k]);\n\t\t\t\t\t} catch (e) {\n\t\t\t\t\t\treturn true;\n\t\t\t\t\t}\n\t\t\t\t}\n\t\t\t} catch (e) {\n\t\t\t\treturn true;\n\t\t\t}\n\t\t}\n\t\treturn false;\n\t}());\n\tvar equalsConstructorPrototypeIfNotBuggy = function (o) {\n\t\t/* global window */\n\t\tif (typeof window === 'undefined' || !hasAutomationEqualityBug) {\n\t\t\treturn equalsConstructorPrototype(o);\n\t\t}\n\t\ttry {\n\t\t\treturn equalsConstructorPrototype(o);\n\t\t} catch (e) {\n\t\t\treturn false;\n\t\t}\n\t};\n\n\tkeysShim = function keys(object) {\n\t\tvar isObject = object !== null && typeof object === 'object';\n\t\tvar isFunction = toStr.call(object) === '[object Function]';\n\t\tvar isArguments = isArgs(object);\n\t\tvar isString = isObject && toStr.call(object) === '[object String]';\n\t\tvar theKeys = [];\n\n\t\tif (!isObject && !isFunction && !isArguments) {\n\t\t\tthrow new TypeError('Object.keys called on a non-object');\n\t\t}\n\n\t\tvar skipProto = hasProtoEnumBug && isFunction;\n\t\tif (isString && object.length > 0 && !has.call(object, 0)) {\n\t\t\tfor (var i = 0; i < object.length; ++i) {\n\t\t\t\ttheKeys.push(String(i));\n\t\t\t}\n\t\t}\n\n\t\tif (isArguments && object.length > 0) {\n\t\t\tfor (var j = 0; j < object.length; ++j) {\n\t\t\t\ttheKeys.push(String(j));\n\t\t\t}\n\t\t} else {\n\t\t\tfor (var name in object) {\n\t\t\t\tif (!(skipProto && name === 'prototype') && has.call(object, name)) {\n\t\t\t\t\ttheKeys.push(String(name));\n\t\t\t\t}\n\t\t\t}\n\t\t}\n\n\t\tif (hasDontEnumBug) {\n\t\t\tvar skipConstructor = equalsConstructorPrototypeIfNotBuggy(object);\n\n\t\t\tfor (var k = 0; k < dontEnums.length; ++k) {\n\t\t\t\tif (!(skipConstructor && dontEnums[k] === 'constructor') && has.call(object, dontEnums[k])) {\n\t\t\t\t\ttheKeys.push(dontEnums[k]);\n\t\t\t\t}\n\t\t\t}\n\t\t}\n\t\treturn theKeys;\n\t};\n}\nmodule.exports = keysShim;\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///8987\n")},2215:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar slice = Array.prototype.slice;\nvar isArgs = __webpack_require__(1414);\n\nvar origKeys = Object.keys;\nvar keysShim = origKeys ? function keys(o) { return origKeys(o); } : __webpack_require__(8987);\n\nvar originalKeys = Object.keys;\n\nkeysShim.shim = function shimObjectKeys() {\n\tif (Object.keys) {\n\t\tvar keysWorksWithArguments = (function () {\n\t\t\t// Safari 5.0 bug\n\t\t\tvar args = Object.keys(arguments);\n\t\t\treturn args && args.length === arguments.length;\n\t\t}(1, 2));\n\t\tif (!keysWorksWithArguments) {\n\t\t\tObject.keys = function keys(object) { // eslint-disable-line func-name-matching\n\t\t\t\tif (isArgs(object)) {\n\t\t\t\t\treturn originalKeys(slice.call(object));\n\t\t\t\t}\n\t\t\t\treturn originalKeys(object);\n\t\t\t};\n\t\t}\n\t} else {\n\t\tObject.keys = keysShim;\n\t}\n\treturn Object.keys || keysShim;\n};\n\nmodule.exports = keysShim;\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL29iamVjdC1rZXlzL2luZGV4LmpzP2Q2YzciXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQWE7O0FBRWI7QUFDQSxhQUFhLG1CQUFPLENBQUMsSUFBZTs7QUFFcEM7QUFDQSw0Q0FBNEMsb0JBQW9CLEVBQUUsR0FBRyxtQkFBTyxDQUFDLElBQWtCOztBQUUvRjs7QUFFQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQSxHQUFHO0FBQ0g7QUFDQSx3Q0FBd0M7QUFDeEM7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0EsRUFBRTtBQUNGO0FBQ0E7QUFDQTtBQUNBOztBQUVBIiwiZmlsZSI6IjIyMTUuanMiLCJzb3VyY2VzQ29udGVudCI6WyIndXNlIHN0cmljdCc7XG5cbnZhciBzbGljZSA9IEFycmF5LnByb3RvdHlwZS5zbGljZTtcbnZhciBpc0FyZ3MgPSByZXF1aXJlKCcuL2lzQXJndW1lbnRzJyk7XG5cbnZhciBvcmlnS2V5cyA9IE9iamVjdC5rZXlzO1xudmFyIGtleXNTaGltID0gb3JpZ0tleXMgPyBmdW5jdGlvbiBrZXlzKG8pIHsgcmV0dXJuIG9yaWdLZXlzKG8pOyB9IDogcmVxdWlyZSgnLi9pbXBsZW1lbnRhdGlvbicpO1xuXG52YXIgb3JpZ2luYWxLZXlzID0gT2JqZWN0LmtleXM7XG5cbmtleXNTaGltLnNoaW0gPSBmdW5jdGlvbiBzaGltT2JqZWN0S2V5cygpIHtcblx0aWYgKE9iamVjdC5rZXlzKSB7XG5cdFx0dmFyIGtleXNXb3Jrc1dpdGhBcmd1bWVudHMgPSAoZnVuY3Rpb24gKCkge1xuXHRcdFx0Ly8gU2FmYXJpIDUuMCBidWdcblx0XHRcdHZhciBhcmdzID0gT2JqZWN0LmtleXMoYXJndW1lbnRzKTtcblx0XHRcdHJldHVybiBhcmdzICYmIGFyZ3MubGVuZ3RoID09PSBhcmd1bWVudHMubGVuZ3RoO1xuXHRcdH0oMSwgMikpO1xuXHRcdGlmICgha2V5c1dvcmtzV2l0aEFyZ3VtZW50cykge1xuXHRcdFx0T2JqZWN0LmtleXMgPSBmdW5jdGlvbiBrZXlzKG9iamVjdCkgeyAvLyBlc2xpbnQtZGlzYWJsZS1saW5lIGZ1bmMtbmFtZS1tYXRjaGluZ1xuXHRcdFx0XHRpZiAoaXNBcmdzKG9iamVjdCkpIHtcblx0XHRcdFx0XHRyZXR1cm4gb3JpZ2luYWxLZXlzKHNsaWNlLmNhbGwob2JqZWN0KSk7XG5cdFx0XHRcdH1cblx0XHRcdFx0cmV0dXJuIG9yaWdpbmFsS2V5cyhvYmplY3QpO1xuXHRcdFx0fTtcblx0XHR9XG5cdH0gZWxzZSB7XG5cdFx0T2JqZWN0LmtleXMgPSBrZXlzU2hpbTtcblx0fVxuXHRyZXR1cm4gT2JqZWN0LmtleXMgfHwga2V5c1NoaW07XG59O1xuXG5tb2R1bGUuZXhwb3J0cyA9IGtleXNTaGltO1xuIl0sInNvdXJjZVJvb3QiOiIifQ==\n//# sourceURL=webpack-internal:///2215\n")},1414:module=>{"use strict";eval("\n\nvar toStr = Object.prototype.toString;\n\nmodule.exports = function isArguments(value) {\n\tvar str = toStr.call(value);\n\tvar isArgs = str === '[object Arguments]';\n\tif (!isArgs) {\n\t\tisArgs = str !== '[object Array]' &&\n\t\t\tvalue !== null &&\n\t\t\ttypeof value === 'object' &&\n\t\t\ttypeof value.length === 'number' &&\n\t\t\tvalue.length >= 0 &&\n\t\t\ttoStr.call(value.callee) === '[object Function]';\n\t}\n\treturn isArgs;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL29iamVjdC1rZXlzL2lzQXJndW1lbnRzLmpzP2Q0YWIiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQWE7O0FBRWI7O0FBRUE7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0EiLCJmaWxlIjoiMTQxNC5qcyIsInNvdXJjZXNDb250ZW50IjpbIid1c2Ugc3RyaWN0JztcblxudmFyIHRvU3RyID0gT2JqZWN0LnByb3RvdHlwZS50b1N0cmluZztcblxubW9kdWxlLmV4cG9ydHMgPSBmdW5jdGlvbiBpc0FyZ3VtZW50cyh2YWx1ZSkge1xuXHR2YXIgc3RyID0gdG9TdHIuY2FsbCh2YWx1ZSk7XG5cdHZhciBpc0FyZ3MgPSBzdHIgPT09ICdbb2JqZWN0IEFyZ3VtZW50c10nO1xuXHRpZiAoIWlzQXJncykge1xuXHRcdGlzQXJncyA9IHN0ciAhPT0gJ1tvYmplY3QgQXJyYXldJyAmJlxuXHRcdFx0dmFsdWUgIT09IG51bGwgJiZcblx0XHRcdHR5cGVvZiB2YWx1ZSA9PT0gJ29iamVjdCcgJiZcblx0XHRcdHR5cGVvZiB2YWx1ZS5sZW5ndGggPT09ICdudW1iZXInICYmXG5cdFx0XHR2YWx1ZS5sZW5ndGggPj0gMCAmJlxuXHRcdFx0dG9TdHIuY2FsbCh2YWx1ZS5jYWxsZWUpID09PSAnW29iamVjdCBGdW5jdGlvbl0nO1xuXHR9XG5cdHJldHVybiBpc0FyZ3M7XG59O1xuIl0sInNvdXJjZVJvb3QiOiIifQ==\n//# sourceURL=webpack-internal:///1414\n")},3697:module=>{"use strict";eval("\n\nvar $Object = Object;\nvar $TypeError = TypeError;\n\nmodule.exports = function flags() {\n\tif (this != null && this !== $Object(this)) {\n\t\tthrow new $TypeError('RegExp.prototype.flags getter called on non-object');\n\t}\n\tvar result = '';\n\tif (this.global) {\n\t\tresult += 'g';\n\t}\n\tif (this.ignoreCase) {\n\t\tresult += 'i';\n\t}\n\tif (this.multiline) {\n\t\tresult += 'm';\n\t}\n\tif (this.dotAll) {\n\t\tresult += 's';\n\t}\n\tif (this.unicode) {\n\t\tresult += 'u';\n\t}\n\tif (this.sticky) {\n\t\tresult += 'y';\n\t}\n\treturn result;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL3JlZ2V4cC5wcm90b3R5cGUuZmxhZ3MvaW1wbGVtZW50YXRpb24uanM/NTcwOCJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBYTs7QUFFYjtBQUNBOztBQUVBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBIiwiZmlsZSI6IjM2OTcuanMiLCJzb3VyY2VzQ29udGVudCI6WyIndXNlIHN0cmljdCc7XG5cbnZhciAkT2JqZWN0ID0gT2JqZWN0O1xudmFyICRUeXBlRXJyb3IgPSBUeXBlRXJyb3I7XG5cbm1vZHVsZS5leHBvcnRzID0gZnVuY3Rpb24gZmxhZ3MoKSB7XG5cdGlmICh0aGlzICE9IG51bGwgJiYgdGhpcyAhPT0gJE9iamVjdCh0aGlzKSkge1xuXHRcdHRocm93IG5ldyAkVHlwZUVycm9yKCdSZWdFeHAucHJvdG90eXBlLmZsYWdzIGdldHRlciBjYWxsZWQgb24gbm9uLW9iamVjdCcpO1xuXHR9XG5cdHZhciByZXN1bHQgPSAnJztcblx0aWYgKHRoaXMuZ2xvYmFsKSB7XG5cdFx0cmVzdWx0ICs9ICdnJztcblx0fVxuXHRpZiAodGhpcy5pZ25vcmVDYXNlKSB7XG5cdFx0cmVzdWx0ICs9ICdpJztcblx0fVxuXHRpZiAodGhpcy5tdWx0aWxpbmUpIHtcblx0XHRyZXN1bHQgKz0gJ20nO1xuXHR9XG5cdGlmICh0aGlzLmRvdEFsbCkge1xuXHRcdHJlc3VsdCArPSAncyc7XG5cdH1cblx0aWYgKHRoaXMudW5pY29kZSkge1xuXHRcdHJlc3VsdCArPSAndSc7XG5cdH1cblx0aWYgKHRoaXMuc3RpY2t5KSB7XG5cdFx0cmVzdWx0ICs9ICd5Jztcblx0fVxuXHRyZXR1cm4gcmVzdWx0O1xufTtcbiJdLCJzb3VyY2VSb290IjoiIn0=\n//# sourceURL=webpack-internal:///3697\n")},2847:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar define = __webpack_require__(4289);\nvar callBind = __webpack_require__(5559);\n\nvar implementation = __webpack_require__(3697);\nvar getPolyfill = __webpack_require__(1721);\nvar shim = __webpack_require__(2753);\n\nvar flagsBound = callBind(implementation);\n\ndefine(flagsBound, {\n\tgetPolyfill: getPolyfill,\n\timplementation: implementation,\n\tshim: shim\n});\n\nmodule.exports = flagsBound;\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL3JlZ2V4cC5wcm90b3R5cGUuZmxhZ3MvaW5kZXguanM/ZTcxMCJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBYTs7QUFFYixhQUFhLG1CQUFPLENBQUMsSUFBbUI7QUFDeEMsZUFBZSxtQkFBTyxDQUFDLElBQVc7O0FBRWxDLHFCQUFxQixtQkFBTyxDQUFDLElBQWtCO0FBQy9DLGtCQUFrQixtQkFBTyxDQUFDLElBQVk7QUFDdEMsV0FBVyxtQkFBTyxDQUFDLElBQVE7O0FBRTNCOztBQUVBO0FBQ0E7QUFDQTtBQUNBO0FBQ0EsQ0FBQzs7QUFFRCIsImZpbGUiOiIyODQ3LmpzIiwic291cmNlc0NvbnRlbnQiOlsiJ3VzZSBzdHJpY3QnO1xuXG52YXIgZGVmaW5lID0gcmVxdWlyZSgnZGVmaW5lLXByb3BlcnRpZXMnKTtcbnZhciBjYWxsQmluZCA9IHJlcXVpcmUoJ2NhbGwtYmluZCcpO1xuXG52YXIgaW1wbGVtZW50YXRpb24gPSByZXF1aXJlKCcuL2ltcGxlbWVudGF0aW9uJyk7XG52YXIgZ2V0UG9seWZpbGwgPSByZXF1aXJlKCcuL3BvbHlmaWxsJyk7XG52YXIgc2hpbSA9IHJlcXVpcmUoJy4vc2hpbScpO1xuXG52YXIgZmxhZ3NCb3VuZCA9IGNhbGxCaW5kKGltcGxlbWVudGF0aW9uKTtcblxuZGVmaW5lKGZsYWdzQm91bmQsIHtcblx0Z2V0UG9seWZpbGw6IGdldFBvbHlmaWxsLFxuXHRpbXBsZW1lbnRhdGlvbjogaW1wbGVtZW50YXRpb24sXG5cdHNoaW06IHNoaW1cbn0pO1xuXG5tb2R1bGUuZXhwb3J0cyA9IGZsYWdzQm91bmQ7XG4iXSwic291cmNlUm9vdCI6IiJ9\n//# sourceURL=webpack-internal:///2847\n")},1721:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar implementation = __webpack_require__(3697);\n\nvar supportsDescriptors = __webpack_require__(4289).supportsDescriptors;\nvar $gOPD = Object.getOwnPropertyDescriptor;\nvar $TypeError = TypeError;\n\nmodule.exports = function getPolyfill() {\n\tif (!supportsDescriptors) {\n\t\tthrow new $TypeError('RegExp.prototype.flags requires a true ES5 environment that supports property descriptors');\n\t}\n\tif ((/a/mig).flags === 'gim') {\n\t\tvar descriptor = $gOPD(RegExp.prototype, 'flags');\n\t\tif (descriptor && typeof descriptor.get === 'function' && typeof (/a/).dotAll === 'boolean') {\n\t\t\treturn descriptor.get;\n\t\t}\n\t}\n\treturn implementation;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL3JlZ2V4cC5wcm90b3R5cGUuZmxhZ3MvcG9seWZpbGwuanM/NTdlYyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBYTs7QUFFYixxQkFBcUIsbUJBQU8sQ0FBQyxJQUFrQjs7QUFFL0MsMEJBQTBCLDZDQUFnRDtBQUMxRTtBQUNBOztBQUVBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQSIsImZpbGUiOiIxNzIxLmpzIiwic291cmNlc0NvbnRlbnQiOlsiJ3VzZSBzdHJpY3QnO1xuXG52YXIgaW1wbGVtZW50YXRpb24gPSByZXF1aXJlKCcuL2ltcGxlbWVudGF0aW9uJyk7XG5cbnZhciBzdXBwb3J0c0Rlc2NyaXB0b3JzID0gcmVxdWlyZSgnZGVmaW5lLXByb3BlcnRpZXMnKS5zdXBwb3J0c0Rlc2NyaXB0b3JzO1xudmFyICRnT1BEID0gT2JqZWN0LmdldE93blByb3BlcnR5RGVzY3JpcHRvcjtcbnZhciAkVHlwZUVycm9yID0gVHlwZUVycm9yO1xuXG5tb2R1bGUuZXhwb3J0cyA9IGZ1bmN0aW9uIGdldFBvbHlmaWxsKCkge1xuXHRpZiAoIXN1cHBvcnRzRGVzY3JpcHRvcnMpIHtcblx0XHR0aHJvdyBuZXcgJFR5cGVFcnJvcignUmVnRXhwLnByb3RvdHlwZS5mbGFncyByZXF1aXJlcyBhIHRydWUgRVM1IGVudmlyb25tZW50IHRoYXQgc3VwcG9ydHMgcHJvcGVydHkgZGVzY3JpcHRvcnMnKTtcblx0fVxuXHRpZiAoKC9hL21pZykuZmxhZ3MgPT09ICdnaW0nKSB7XG5cdFx0dmFyIGRlc2NyaXB0b3IgPSAkZ09QRChSZWdFeHAucHJvdG90eXBlLCAnZmxhZ3MnKTtcblx0XHRpZiAoZGVzY3JpcHRvciAmJiB0eXBlb2YgZGVzY3JpcHRvci5nZXQgPT09ICdmdW5jdGlvbicgJiYgdHlwZW9mICgvYS8pLmRvdEFsbCA9PT0gJ2Jvb2xlYW4nKSB7XG5cdFx0XHRyZXR1cm4gZGVzY3JpcHRvci5nZXQ7XG5cdFx0fVxuXHR9XG5cdHJldHVybiBpbXBsZW1lbnRhdGlvbjtcbn07XG4iXSwic291cmNlUm9vdCI6IiJ9\n//# sourceURL=webpack-internal:///1721\n")},2753:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar supportsDescriptors = __webpack_require__(4289).supportsDescriptors;\nvar getPolyfill = __webpack_require__(1721);\nvar gOPD = Object.getOwnPropertyDescriptor;\nvar defineProperty = Object.defineProperty;\nvar TypeErr = TypeError;\nvar getProto = Object.getPrototypeOf;\nvar regex = /a/;\n\nmodule.exports = function shimFlags() {\n\tif (!supportsDescriptors || !getProto) {\n\t\tthrow new TypeErr('RegExp.prototype.flags requires a true ES5 environment that supports property descriptors');\n\t}\n\tvar polyfill = getPolyfill();\n\tvar proto = getProto(regex);\n\tvar descriptor = gOPD(proto, 'flags');\n\tif (!descriptor || descriptor.get !== polyfill) {\n\t\tdefineProperty(proto, 'flags', {\n\t\t\tconfigurable: true,\n\t\t\tenumerable: false,\n\t\t\tget: polyfill\n\t\t});\n\t}\n\treturn polyfill;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL3JlZ2V4cC5wcm90b3R5cGUuZmxhZ3Mvc2hpbS5qcz8xYzdlIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFhOztBQUViLDBCQUEwQiw2Q0FBZ0Q7QUFDMUUsa0JBQWtCLG1CQUFPLENBQUMsSUFBWTtBQUN0QztBQUNBO0FBQ0E7QUFDQTtBQUNBOztBQUVBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBLEdBQUc7QUFDSDtBQUNBO0FBQ0EiLCJmaWxlIjoiMjc1My5qcyIsInNvdXJjZXNDb250ZW50IjpbIid1c2Ugc3RyaWN0JztcblxudmFyIHN1cHBvcnRzRGVzY3JpcHRvcnMgPSByZXF1aXJlKCdkZWZpbmUtcHJvcGVydGllcycpLnN1cHBvcnRzRGVzY3JpcHRvcnM7XG52YXIgZ2V0UG9seWZpbGwgPSByZXF1aXJlKCcuL3BvbHlmaWxsJyk7XG52YXIgZ09QRCA9IE9iamVjdC5nZXRPd25Qcm9wZXJ0eURlc2NyaXB0b3I7XG52YXIgZGVmaW5lUHJvcGVydHkgPSBPYmplY3QuZGVmaW5lUHJvcGVydHk7XG52YXIgVHlwZUVyciA9IFR5cGVFcnJvcjtcbnZhciBnZXRQcm90byA9IE9iamVjdC5nZXRQcm90b3R5cGVPZjtcbnZhciByZWdleCA9IC9hLztcblxubW9kdWxlLmV4cG9ydHMgPSBmdW5jdGlvbiBzaGltRmxhZ3MoKSB7XG5cdGlmICghc3VwcG9ydHNEZXNjcmlwdG9ycyB8fCAhZ2V0UHJvdG8pIHtcblx0XHR0aHJvdyBuZXcgVHlwZUVycignUmVnRXhwLnByb3RvdHlwZS5mbGFncyByZXF1aXJlcyBhIHRydWUgRVM1IGVudmlyb25tZW50IHRoYXQgc3VwcG9ydHMgcHJvcGVydHkgZGVzY3JpcHRvcnMnKTtcblx0fVxuXHR2YXIgcG9seWZpbGwgPSBnZXRQb2x5ZmlsbCgpO1xuXHR2YXIgcHJvdG8gPSBnZXRQcm90byhyZWdleCk7XG5cdHZhciBkZXNjcmlwdG9yID0gZ09QRChwcm90bywgJ2ZsYWdzJyk7XG5cdGlmICghZGVzY3JpcHRvciB8fCBkZXNjcmlwdG9yLmdldCAhPT0gcG9seWZpbGwpIHtcblx0XHRkZWZpbmVQcm9wZXJ0eShwcm90bywgJ2ZsYWdzJywge1xuXHRcdFx0Y29uZmlndXJhYmxlOiB0cnVlLFxuXHRcdFx0ZW51bWVyYWJsZTogZmFsc2UsXG5cdFx0XHRnZXQ6IHBvbHlmaWxsXG5cdFx0fSk7XG5cdH1cblx0cmV0dXJuIHBvbHlmaWxsO1xufTtcbiJdLCJzb3VyY2VSb290IjoiIn0=\n//# sourceURL=webpack-internal:///2753\n")},7478:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar GetIntrinsic = __webpack_require__(210);\nvar callBound = __webpack_require__(1924);\nvar inspect = __webpack_require__(631);\n\nvar $TypeError = GetIntrinsic('%TypeError%');\nvar $WeakMap = GetIntrinsic('%WeakMap%', true);\nvar $Map = GetIntrinsic('%Map%', true);\n\nvar $weakMapGet = callBound('WeakMap.prototype.get', true);\nvar $weakMapSet = callBound('WeakMap.prototype.set', true);\nvar $weakMapHas = callBound('WeakMap.prototype.has', true);\nvar $mapGet = callBound('Map.prototype.get', true);\nvar $mapSet = callBound('Map.prototype.set', true);\nvar $mapHas = callBound('Map.prototype.has', true);\n\n/*\n * This function traverses the list returning the node corresponding to the\n * given key.\n *\n * That node is also moved to the head of the list, so that if it's accessed\n * again we don't need to traverse the whole list. By doing so, all the recently\n * used nodes can be accessed relatively quickly.\n */\nvar listGetNode = function (list, key) { // eslint-disable-line consistent-return\n\tfor (var prev = list, curr; (curr = prev.next) !== null; prev = curr) {\n\t\tif (curr.key === key) {\n\t\t\tprev.next = curr.next;\n\t\t\tcurr.next = list.next;\n\t\t\tlist.next = curr; // eslint-disable-line no-param-reassign\n\t\t\treturn curr;\n\t\t}\n\t}\n};\n\nvar listGet = function (objects, key) {\n\tvar node = listGetNode(objects, key);\n\treturn node && node.value;\n};\nvar listSet = function (objects, key, value) {\n\tvar node = listGetNode(objects, key);\n\tif (node) {\n\t\tnode.value = value;\n\t} else {\n\t\t// Prepend the new node to the beginning of the list\n\t\tobjects.next = { // eslint-disable-line no-param-reassign\n\t\t\tkey: key,\n\t\t\tnext: objects.next,\n\t\t\tvalue: value\n\t\t};\n\t}\n};\nvar listHas = function (objects, key) {\n\treturn !!listGetNode(objects, key);\n};\n\nmodule.exports = function getSideChannel() {\n\tvar $wm;\n\tvar $m;\n\tvar $o;\n\tvar channel = {\n\t\tassert: function (key) {\n\t\t\tif (!channel.has(key)) {\n\t\t\t\tthrow new $TypeError('Side channel does not contain ' + inspect(key));\n\t\t\t}\n\t\t},\n\t\tget: function (key) { // eslint-disable-line consistent-return\n\t\t\tif ($WeakMap && key && (typeof key === 'object' || typeof key === 'function')) {\n\t\t\t\tif ($wm) {\n\t\t\t\t\treturn $weakMapGet($wm, key);\n\t\t\t\t}\n\t\t\t} else if ($Map) {\n\t\t\t\tif ($m) {\n\t\t\t\t\treturn $mapGet($m, key);\n\t\t\t\t}\n\t\t\t} else {\n\t\t\t\tif ($o) { // eslint-disable-line no-lonely-if\n\t\t\t\t\treturn listGet($o, key);\n\t\t\t\t}\n\t\t\t}\n\t\t},\n\t\thas: function (key) {\n\t\t\tif ($WeakMap && key && (typeof key === 'object' || typeof key === 'function')) {\n\t\t\t\tif ($wm) {\n\t\t\t\t\treturn $weakMapHas($wm, key);\n\t\t\t\t}\n\t\t\t} else if ($Map) {\n\t\t\t\tif ($m) {\n\t\t\t\t\treturn $mapHas($m, key);\n\t\t\t\t}\n\t\t\t} else {\n\t\t\t\tif ($o) { // eslint-disable-line no-lonely-if\n\t\t\t\t\treturn listHas($o, key);\n\t\t\t\t}\n\t\t\t}\n\t\t\treturn false;\n\t\t},\n\t\tset: function (key, value) {\n\t\t\tif ($WeakMap && key && (typeof key === 'object' || typeof key === 'function')) {\n\t\t\t\tif (!$wm) {\n\t\t\t\t\t$wm = new $WeakMap();\n\t\t\t\t}\n\t\t\t\t$weakMapSet($wm, key, value);\n\t\t\t} else if ($Map) {\n\t\t\t\tif (!$m) {\n\t\t\t\t\t$m = new $Map();\n\t\t\t\t}\n\t\t\t\t$mapSet($m, key, value);\n\t\t\t} else {\n\t\t\t\tif (!$o) {\n\t\t\t\t\t/*\n\t\t\t\t\t * Initialize the linked list as an empty node, so that we don't have\n\t\t\t\t\t * to special-case handling of the first node: we can always refer to\n\t\t\t\t\t * it as (previous node).next, instead of something like (list).head\n\t\t\t\t\t */\n\t\t\t\t\t$o = { key: {}, next: null };\n\t\t\t\t}\n\t\t\t\tlistSet($o, key, value);\n\t\t\t}\n\t\t}\n\t};\n\treturn channel;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///7478\n")},1066:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar define = __webpack_require__(4289);\nvar AdvanceStringIndex = __webpack_require__(2672);\nvar CreateIterResultObject = __webpack_require__(3674);\nvar Get = __webpack_require__(5588);\nvar GetIntrinsic = __webpack_require__(210);\nvar OrdinaryObjectCreate = __webpack_require__(4091);\nvar RegExpExec = __webpack_require__(356);\nvar Set = __webpack_require__(2946);\nvar ToLength = __webpack_require__(8502);\nvar ToString = __webpack_require__(7308);\nvar Type = __webpack_require__(1915);\nvar hasSymbols = __webpack_require__(1405)();\n\nvar SLOT = __webpack_require__(9496);\nvar undefined;\n\nvar RegExpStringIterator = function RegExpStringIterator(R, S, global, fullUnicode) {\n\tif (Type(S) !== 'String') {\n\t\tthrow new TypeError('S must be a string');\n\t}\n\tif (Type(global) !== 'Boolean') {\n\t\tthrow new TypeError('global must be a boolean');\n\t}\n\tif (Type(fullUnicode) !== 'Boolean') {\n\t\tthrow new TypeError('fullUnicode must be a boolean');\n\t}\n\tSLOT.set(this, '[[IteratingRegExp]]', R);\n\tSLOT.set(this, '[[IteratedString]]', S);\n\tSLOT.set(this, '[[Global]]', global);\n\tSLOT.set(this, '[[Unicode]]', fullUnicode);\n\tSLOT.set(this, '[[Done]]', false);\n};\n\nvar IteratorPrototype = GetIntrinsic('%IteratorPrototype%', true);\nif (IteratorPrototype) {\n\tRegExpStringIterator.prototype = OrdinaryObjectCreate(IteratorPrototype);\n}\n\ndefine(RegExpStringIterator.prototype, {\n\tnext: function next() {\n\t\tvar O = this;\n\t\tif (Type(O) !== 'Object') {\n\t\t\tthrow new TypeError('receiver must be an object');\n\t\t}\n\t\tif (\n\t\t\t!(O instanceof RegExpStringIterator)\n\t\t\t|| !SLOT.has(O, '[[IteratingRegExp]]')\n\t\t\t|| !SLOT.has(O, '[[IteratedString]]')\n\t\t\t|| !SLOT.has(O, '[[Global]]')\n\t\t\t|| !SLOT.has(O, '[[Unicode]]')\n\t\t\t|| !SLOT.has(O, '[[Done]]')\n\t\t) {\n\t\t\tthrow new TypeError('\"this\" value must be a RegExpStringIterator instance');\n\t\t}\n\t\tif (SLOT.get(O, '[[Done]]')) {\n\t\t\treturn CreateIterResultObject(undefined, true);\n\t\t}\n\t\tvar R = SLOT.get(O, '[[IteratingRegExp]]');\n\t\tvar S = SLOT.get(O, '[[IteratedString]]');\n\t\tvar global = SLOT.get(O, '[[Global]]');\n\t\tvar fullUnicode = SLOT.get(O, '[[Unicode]]');\n\t\tvar match = RegExpExec(R, S);\n\t\tif (match === null) {\n\t\t\tSLOT.set(O, '[[Done]]', true);\n\t\t\treturn CreateIterResultObject(undefined, true);\n\t\t}\n\t\tif (global) {\n\t\t\tvar matchStr = ToString(Get(match, '0'));\n\t\t\tif (matchStr === '') {\n\t\t\t\tvar thisIndex = ToLength(Get(R, 'lastIndex'));\n\t\t\t\tvar nextIndex = AdvanceStringIndex(S, thisIndex, fullUnicode);\n\t\t\t\tSet(R, 'lastIndex', nextIndex, true);\n\t\t\t}\n\t\t\treturn CreateIterResultObject(match, false);\n\t\t}\n\t\tSLOT.set(O, '[[Done]]', true);\n\t\treturn CreateIterResultObject(match, false);\n\t}\n});\nif (hasSymbols) {\n\tvar defineP = Object.defineProperty;\n\tif (Symbol.toStringTag) {\n\t\tif (defineP) {\n\t\t\tdefineP(RegExpStringIterator.prototype, Symbol.toStringTag, {\n\t\t\t\tconfigurable: true,\n\t\t\t\tenumerable: false,\n\t\t\t\tvalue: 'RegExp String Iterator',\n\t\t\t\twritable: false\n\t\t\t});\n\t\t} else {\n\t\t\tRegExpStringIterator.prototype[Symbol.toStringTag] = 'RegExp String Iterator';\n\t\t}\n\t}\n\n\tif (!IteratorPrototype && Symbol.iterator) {\n\t\tvar func = {};\n\t\tfunc[Symbol.iterator] = RegExpStringIterator.prototype[Symbol.iterator] || function SymbolIterator() {\n\t\t\treturn this;\n\t\t};\n\t\tvar predicate = {};\n\t\tpredicate[Symbol.iterator] = function () {\n\t\t\treturn RegExpStringIterator.prototype[Symbol.iterator] !== func[Symbol.iterator];\n\t\t};\n\t\tdefine(RegExpStringIterator.prototype, func, predicate);\n\t}\n}\n\nmodule.exports = RegExpStringIterator;\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///1066\n")},9505:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar Call = __webpack_require__(7800);\nvar Get = __webpack_require__(5588);\nvar GetMethod = __webpack_require__(3081);\nvar IsRegExp = __webpack_require__(1137);\nvar ToString = __webpack_require__(7308);\nvar RequireObjectCoercible = __webpack_require__(3733);\nvar callBound = __webpack_require__(1924);\nvar hasSymbols = __webpack_require__(1405)();\nvar flagsGetter = __webpack_require__(2847);\n\nvar $indexOf = callBound('String.prototype.indexOf');\n\nvar regexpMatchAllPolyfill = __webpack_require__(6966);\n\nvar getMatcher = function getMatcher(regexp) { // eslint-disable-line consistent-return\n\tvar matcherPolyfill = regexpMatchAllPolyfill();\n\tif (hasSymbols && typeof Symbol.matchAll === 'symbol') {\n\t\tvar matcher = GetMethod(regexp, Symbol.matchAll);\n\t\tif (matcher === RegExp.prototype[Symbol.matchAll] && matcher !== matcherPolyfill) {\n\t\t\treturn matcherPolyfill;\n\t\t}\n\t\treturn matcher;\n\t}\n\t// fallback for pre-Symbol.matchAll environments\n\tif (IsRegExp(regexp)) {\n\t\treturn matcherPolyfill;\n\t}\n};\n\nmodule.exports = function matchAll(regexp) {\n\tvar O = RequireObjectCoercible(this);\n\n\tif (typeof regexp !== 'undefined' && regexp !== null) {\n\t\tvar isRegExp = IsRegExp(regexp);\n\t\tif (isRegExp) {\n\t\t\t// workaround for older engines that lack RegExp.prototype.flags\n\t\t\tvar flags = 'flags' in regexp ? Get(regexp, 'flags') : flagsGetter(regexp);\n\t\t\tRequireObjectCoercible(flags);\n\t\t\tif ($indexOf(ToString(flags), 'g') < 0) {\n\t\t\t\tthrow new TypeError('matchAll requires a global regular expression');\n\t\t\t}\n\t\t}\n\n\t\tvar matcher = getMatcher(regexp);\n\t\tif (typeof matcher !== 'undefined') {\n\t\t\treturn Call(matcher, regexp, [O]);\n\t\t}\n\t}\n\n\tvar S = ToString(O);\n\t// var rx = RegExpCreate(regexp, 'g');\n\tvar rx = new RegExp(regexp, 'g');\n\treturn Call(getMatcher(rx), rx, [S]);\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL3N0cmluZy5wcm90b3R5cGUubWF0Y2hhbGwvaW1wbGVtZW50YXRpb24uanM/YzE3ZCJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBYTs7QUFFYixXQUFXLG1CQUFPLENBQUMsSUFBdUI7QUFDMUMsVUFBVSxtQkFBTyxDQUFDLElBQXNCO0FBQ3hDLGdCQUFnQixtQkFBTyxDQUFDLElBQTRCO0FBQ3BELGVBQWUsbUJBQU8sQ0FBQyxJQUEyQjtBQUNsRCxlQUFlLG1CQUFPLENBQUMsSUFBMkI7QUFDbEQsNkJBQTZCLG1CQUFPLENBQUMsSUFBeUM7QUFDOUUsZ0JBQWdCLG1CQUFPLENBQUMsSUFBcUI7QUFDN0MsaUJBQWlCLG1CQUFPLENBQUMsSUFBYTtBQUN0QyxrQkFBa0IsbUJBQU8sQ0FBQyxJQUF3Qjs7QUFFbEQ7O0FBRUEsNkJBQTZCLG1CQUFPLENBQUMsSUFBNEI7O0FBRWpFLDhDQUE4QztBQUM5QztBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTs7QUFFQTtBQUNBOztBQUVBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBOztBQUVBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7O0FBRUE7QUFDQTtBQUNBO0FBQ0E7QUFDQSIsImZpbGUiOiI5NTA1LmpzIiwic291cmNlc0NvbnRlbnQiOlsiJ3VzZSBzdHJpY3QnO1xuXG52YXIgQ2FsbCA9IHJlcXVpcmUoJ2VzLWFic3RyYWN0LzIwMjAvQ2FsbCcpO1xudmFyIEdldCA9IHJlcXVpcmUoJ2VzLWFic3RyYWN0LzIwMjAvR2V0Jyk7XG52YXIgR2V0TWV0aG9kID0gcmVxdWlyZSgnZXMtYWJzdHJhY3QvMjAyMC9HZXRNZXRob2QnKTtcbnZhciBJc1JlZ0V4cCA9IHJlcXVpcmUoJ2VzLWFic3RyYWN0LzIwMjAvSXNSZWdFeHAnKTtcbnZhciBUb1N0cmluZyA9IHJlcXVpcmUoJ2VzLWFic3RyYWN0LzIwMjAvVG9TdHJpbmcnKTtcbnZhciBSZXF1aXJlT2JqZWN0Q29lcmNpYmxlID0gcmVxdWlyZSgnZXMtYWJzdHJhY3QvMjAyMC9SZXF1aXJlT2JqZWN0Q29lcmNpYmxlJyk7XG52YXIgY2FsbEJvdW5kID0gcmVxdWlyZSgnY2FsbC1iaW5kL2NhbGxCb3VuZCcpO1xudmFyIGhhc1N5bWJvbHMgPSByZXF1aXJlKCdoYXMtc3ltYm9scycpKCk7XG52YXIgZmxhZ3NHZXR0ZXIgPSByZXF1aXJlKCdyZWdleHAucHJvdG90eXBlLmZsYWdzJyk7XG5cbnZhciAkaW5kZXhPZiA9IGNhbGxCb3VuZCgnU3RyaW5nLnByb3RvdHlwZS5pbmRleE9mJyk7XG5cbnZhciByZWdleHBNYXRjaEFsbFBvbHlmaWxsID0gcmVxdWlyZSgnLi9wb2x5ZmlsbC1yZWdleHAtbWF0Y2hhbGwnKTtcblxudmFyIGdldE1hdGNoZXIgPSBmdW5jdGlvbiBnZXRNYXRjaGVyKHJlZ2V4cCkgeyAvLyBlc2xpbnQtZGlzYWJsZS1saW5lIGNvbnNpc3RlbnQtcmV0dXJuXG5cdHZhciBtYXRjaGVyUG9seWZpbGwgPSByZWdleHBNYXRjaEFsbFBvbHlmaWxsKCk7XG5cdGlmIChoYXNTeW1ib2xzICYmIHR5cGVvZiBTeW1ib2wubWF0Y2hBbGwgPT09ICdzeW1ib2wnKSB7XG5cdFx0dmFyIG1hdGNoZXIgPSBHZXRNZXRob2QocmVnZXhwLCBTeW1ib2wubWF0Y2hBbGwpO1xuXHRcdGlmIChtYXRjaGVyID09PSBSZWdFeHAucHJvdG90eXBlW1N5bWJvbC5tYXRjaEFsbF0gJiYgbWF0Y2hlciAhPT0gbWF0Y2hlclBvbHlmaWxsKSB7XG5cdFx0XHRyZXR1cm4gbWF0Y2hlclBvbHlmaWxsO1xuXHRcdH1cblx0XHRyZXR1cm4gbWF0Y2hlcjtcblx0fVxuXHQvLyBmYWxsYmFjayBmb3IgcHJlLVN5bWJvbC5tYXRjaEFsbCBlbnZpcm9ubWVudHNcblx0aWYgKElzUmVnRXhwKHJlZ2V4cCkpIHtcblx0XHRyZXR1cm4gbWF0Y2hlclBvbHlmaWxsO1xuXHR9XG59O1xuXG5tb2R1bGUuZXhwb3J0cyA9IGZ1bmN0aW9uIG1hdGNoQWxsKHJlZ2V4cCkge1xuXHR2YXIgTyA9IFJlcXVpcmVPYmplY3RDb2VyY2libGUodGhpcyk7XG5cblx0aWYgKHR5cGVvZiByZWdleHAgIT09ICd1bmRlZmluZWQnICYmIHJlZ2V4cCAhPT0gbnVsbCkge1xuXHRcdHZhciBpc1JlZ0V4cCA9IElzUmVnRXhwKHJlZ2V4cCk7XG5cdFx0aWYgKGlzUmVnRXhwKSB7XG5cdFx0XHQvLyB3b3JrYXJvdW5kIGZvciBvbGRlciBlbmdpbmVzIHRoYXQgbGFjayBSZWdFeHAucHJvdG90eXBlLmZsYWdzXG5cdFx0XHR2YXIgZmxhZ3MgPSAnZmxhZ3MnIGluIHJlZ2V4cCA/IEdldChyZWdleHAsICdmbGFncycpIDogZmxhZ3NHZXR0ZXIocmVnZXhwKTtcblx0XHRcdFJlcXVpcmVPYmplY3RDb2VyY2libGUoZmxhZ3MpO1xuXHRcdFx0aWYgKCRpbmRleE9mKFRvU3RyaW5nKGZsYWdzKSwgJ2cnKSA8IDApIHtcblx0XHRcdFx0dGhyb3cgbmV3IFR5cGVFcnJvcignbWF0Y2hBbGwgcmVxdWlyZXMgYSBnbG9iYWwgcmVndWxhciBleHByZXNzaW9uJyk7XG5cdFx0XHR9XG5cdFx0fVxuXG5cdFx0dmFyIG1hdGNoZXIgPSBnZXRNYXRjaGVyKHJlZ2V4cCk7XG5cdFx0aWYgKHR5cGVvZiBtYXRjaGVyICE9PSAndW5kZWZpbmVkJykge1xuXHRcdFx0cmV0dXJuIENhbGwobWF0Y2hlciwgcmVnZXhwLCBbT10pO1xuXHRcdH1cblx0fVxuXG5cdHZhciBTID0gVG9TdHJpbmcoTyk7XG5cdC8vIHZhciByeCA9IFJlZ0V4cENyZWF0ZShyZWdleHAsICdnJyk7XG5cdHZhciByeCA9IG5ldyBSZWdFeHAocmVnZXhwLCAnZycpO1xuXHRyZXR1cm4gQ2FsbChnZXRNYXRjaGVyKHJ4KSwgcngsIFtTXSk7XG59O1xuIl0sInNvdXJjZVJvb3QiOiIifQ==\n//# sourceURL=webpack-internal:///9505\n")},4956:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar callBind = __webpack_require__(5559);\nvar define = __webpack_require__(4289);\n\nvar implementation = __webpack_require__(9505);\nvar getPolyfill = __webpack_require__(3447);\nvar shim = __webpack_require__(2376);\n\nvar boundMatchAll = callBind(implementation);\n\ndefine(boundMatchAll, {\n\tgetPolyfill: getPolyfill,\n\timplementation: implementation,\n\tshim: shim\n});\n\nmodule.exports = boundMatchAll;\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL3N0cmluZy5wcm90b3R5cGUubWF0Y2hhbGwvaW5kZXguanM/YjFjYyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBYTs7QUFFYixlQUFlLG1CQUFPLENBQUMsSUFBVztBQUNsQyxhQUFhLG1CQUFPLENBQUMsSUFBbUI7O0FBRXhDLHFCQUFxQixtQkFBTyxDQUFDLElBQWtCO0FBQy9DLGtCQUFrQixtQkFBTyxDQUFDLElBQVk7QUFDdEMsV0FBVyxtQkFBTyxDQUFDLElBQVE7O0FBRTNCOztBQUVBO0FBQ0E7QUFDQTtBQUNBO0FBQ0EsQ0FBQzs7QUFFRCIsImZpbGUiOiI0OTU2LmpzIiwic291cmNlc0NvbnRlbnQiOlsiJ3VzZSBzdHJpY3QnO1xuXG52YXIgY2FsbEJpbmQgPSByZXF1aXJlKCdjYWxsLWJpbmQnKTtcbnZhciBkZWZpbmUgPSByZXF1aXJlKCdkZWZpbmUtcHJvcGVydGllcycpO1xuXG52YXIgaW1wbGVtZW50YXRpb24gPSByZXF1aXJlKCcuL2ltcGxlbWVudGF0aW9uJyk7XG52YXIgZ2V0UG9seWZpbGwgPSByZXF1aXJlKCcuL3BvbHlmaWxsJyk7XG52YXIgc2hpbSA9IHJlcXVpcmUoJy4vc2hpbScpO1xuXG52YXIgYm91bmRNYXRjaEFsbCA9IGNhbGxCaW5kKGltcGxlbWVudGF0aW9uKTtcblxuZGVmaW5lKGJvdW5kTWF0Y2hBbGwsIHtcblx0Z2V0UG9seWZpbGw6IGdldFBvbHlmaWxsLFxuXHRpbXBsZW1lbnRhdGlvbjogaW1wbGVtZW50YXRpb24sXG5cdHNoaW06IHNoaW1cbn0pO1xuXG5tb2R1bGUuZXhwb3J0cyA9IGJvdW5kTWF0Y2hBbGw7XG4iXSwic291cmNlUm9vdCI6IiJ9\n//# sourceURL=webpack-internal:///4956\n")},6966:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar hasSymbols = __webpack_require__(1405)();\nvar regexpMatchAll = __webpack_require__(7201);\n\nmodule.exports = function getRegExpMatchAllPolyfill() {\n\tif (!hasSymbols || typeof Symbol.matchAll !== 'symbol' || typeof RegExp.prototype[Symbol.matchAll] !== 'function') {\n\t\treturn regexpMatchAll;\n\t}\n\treturn RegExp.prototype[Symbol.matchAll];\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL3N0cmluZy5wcm90b3R5cGUubWF0Y2hhbGwvcG9seWZpbGwtcmVnZXhwLW1hdGNoYWxsLmpzPzZjMTgiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQWE7O0FBRWIsaUJBQWlCLG1CQUFPLENBQUMsSUFBYTtBQUN0QyxxQkFBcUIsbUJBQU8sQ0FBQyxJQUFtQjs7QUFFaEQ7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBIiwiZmlsZSI6IjY5NjYuanMiLCJzb3VyY2VzQ29udGVudCI6WyIndXNlIHN0cmljdCc7XG5cbnZhciBoYXNTeW1ib2xzID0gcmVxdWlyZSgnaGFzLXN5bWJvbHMnKSgpO1xudmFyIHJlZ2V4cE1hdGNoQWxsID0gcmVxdWlyZSgnLi9yZWdleHAtbWF0Y2hhbGwnKTtcblxubW9kdWxlLmV4cG9ydHMgPSBmdW5jdGlvbiBnZXRSZWdFeHBNYXRjaEFsbFBvbHlmaWxsKCkge1xuXHRpZiAoIWhhc1N5bWJvbHMgfHwgdHlwZW9mIFN5bWJvbC5tYXRjaEFsbCAhPT0gJ3N5bWJvbCcgfHwgdHlwZW9mIFJlZ0V4cC5wcm90b3R5cGVbU3ltYm9sLm1hdGNoQWxsXSAhPT0gJ2Z1bmN0aW9uJykge1xuXHRcdHJldHVybiByZWdleHBNYXRjaEFsbDtcblx0fVxuXHRyZXR1cm4gUmVnRXhwLnByb3RvdHlwZVtTeW1ib2wubWF0Y2hBbGxdO1xufTtcbiJdLCJzb3VyY2VSb290IjoiIn0=\n//# sourceURL=webpack-internal:///6966\n")},3447:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar implementation = __webpack_require__(9505);\n\nmodule.exports = function getPolyfill() {\n\tif (String.prototype.matchAll) {\n\t\ttry {\n\t\t\t''.matchAll(RegExp.prototype);\n\t\t} catch (e) {\n\t\t\treturn String.prototype.matchAll;\n\t\t}\n\t}\n\treturn implementation;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL3N0cmluZy5wcm90b3R5cGUubWF0Y2hhbGwvcG9seWZpbGwuanM/YjhhMSJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBYTs7QUFFYixxQkFBcUIsbUJBQU8sQ0FBQyxJQUFrQjs7QUFFL0M7QUFDQTtBQUNBO0FBQ0E7QUFDQSxHQUFHO0FBQ0g7QUFDQTtBQUNBO0FBQ0E7QUFDQSIsImZpbGUiOiIzNDQ3LmpzIiwic291cmNlc0NvbnRlbnQiOlsiJ3VzZSBzdHJpY3QnO1xuXG52YXIgaW1wbGVtZW50YXRpb24gPSByZXF1aXJlKCcuL2ltcGxlbWVudGF0aW9uJyk7XG5cbm1vZHVsZS5leHBvcnRzID0gZnVuY3Rpb24gZ2V0UG9seWZpbGwoKSB7XG5cdGlmIChTdHJpbmcucHJvdG90eXBlLm1hdGNoQWxsKSB7XG5cdFx0dHJ5IHtcblx0XHRcdCcnLm1hdGNoQWxsKFJlZ0V4cC5wcm90b3R5cGUpO1xuXHRcdH0gY2F0Y2ggKGUpIHtcblx0XHRcdHJldHVybiBTdHJpbmcucHJvdG90eXBlLm1hdGNoQWxsO1xuXHRcdH1cblx0fVxuXHRyZXR1cm4gaW1wbGVtZW50YXRpb247XG59O1xuIl0sInNvdXJjZVJvb3QiOiIifQ==\n//# sourceURL=webpack-internal:///3447\n")},7201:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\n// var Construct = require('es-abstract/2020/Construct');\nvar Get = __webpack_require__(5588);\nvar Set = __webpack_require__(2946);\nvar SpeciesConstructor = __webpack_require__(303);\nvar ToLength = __webpack_require__(8502);\nvar ToString = __webpack_require__(7308);\nvar Type = __webpack_require__(1915);\nvar flagsGetter = __webpack_require__(2847);\n\nvar RegExpStringIterator = __webpack_require__(1066);\nvar OrigRegExp = RegExp;\n\nvar CreateRegExpStringIterator = function CreateRegExpStringIterator(R, S, global, fullUnicode) {\n\tif (Type(S) !== 'String') {\n\t\tthrow new TypeError('\"S\" value must be a String');\n\t}\n\tif (Type(global) !== 'Boolean') {\n\t\tthrow new TypeError('\"global\" value must be a Boolean');\n\t}\n\tif (Type(fullUnicode) !== 'Boolean') {\n\t\tthrow new TypeError('\"fullUnicode\" value must be a Boolean');\n\t}\n\n\tvar iterator = new RegExpStringIterator(R, S, global, fullUnicode);\n\treturn iterator;\n};\n\nvar supportsConstructingWithFlags = 'flags' in RegExp.prototype;\n\nvar constructRegexWithFlags = function constructRegex(C, R) {\n\tvar matcher;\n\t// workaround for older engines that lack RegExp.prototype.flags\n\tvar flags = 'flags' in R ? Get(R, 'flags') : ToString(flagsGetter(R));\n\tif (supportsConstructingWithFlags && typeof flags === 'string') {\n\t\tmatcher = new C(R, flags);\n\t} else if (C === OrigRegExp) {\n\t\t// workaround for older engines that can not construct a RegExp with flags\n\t\tmatcher = new C(R.source, flags);\n\t} else {\n\t\tmatcher = new C(R, flags);\n\t}\n\treturn { flags: flags, matcher: matcher };\n};\n\nvar regexMatchAll = function SymbolMatchAll(string) {\n\tvar R = this;\n\tif (Type(R) !== 'Object') {\n\t\tthrow new TypeError('\"this\" value must be an Object');\n\t}\n\tvar S = ToString(string);\n\tvar C = SpeciesConstructor(R, OrigRegExp);\n\n\tvar tmp = constructRegexWithFlags(C, R);\n\t// var flags = ToString(Get(R, 'flags'));\n\tvar flags = tmp.flags;\n\t// var matcher = Construct(C, [R, flags]);\n\tvar matcher = tmp.matcher;\n\n\tvar lastIndex = ToLength(Get(R, 'lastIndex'));\n\tSet(matcher, 'lastIndex', lastIndex, true);\n\tvar global = flags.indexOf('g') > -1;\n\tvar fullUnicode = flags.indexOf('u') > -1;\n\treturn CreateRegExpStringIterator(matcher, S, global, fullUnicode);\n};\n\nvar defineP = Object.defineProperty;\nvar gOPD = Object.getOwnPropertyDescriptor;\n\nif (defineP && gOPD) {\n\tvar desc = gOPD(regexMatchAll, 'name');\n\tif (desc && desc.configurable) {\n\t\tdefineP(regexMatchAll, 'name', { value: '[Symbol.matchAll]' });\n\t}\n}\n\nmodule.exports = regexMatchAll;\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndlYnBhY2s6Ly9yZWFkaXVtLWpzLy4vbm9kZV9tb2R1bGVzL3N0cmluZy5wcm90b3R5cGUubWF0Y2hhbGwvcmVnZXhwLW1hdGNoYWxsLmpzP2ZhODkiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQWE7O0FBRWI7QUFDQSxVQUFVLG1CQUFPLENBQUMsSUFBc0I7QUFDeEMsVUFBVSxtQkFBTyxDQUFDLElBQXNCO0FBQ3hDLHlCQUF5QixtQkFBTyxDQUFDLEdBQXFDO0FBQ3RFLGVBQWUsbUJBQU8sQ0FBQyxJQUEyQjtBQUNsRCxlQUFlLG1CQUFPLENBQUMsSUFBMkI7QUFDbEQsV0FBVyxtQkFBTyxDQUFDLElBQXVCO0FBQzFDLGtCQUFrQixtQkFBTyxDQUFDLElBQXdCOztBQUVsRCwyQkFBMkIsbUJBQU8sQ0FBQyxJQUFnQztBQUNuRTs7QUFFQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTs7QUFFQTtBQUNBO0FBQ0E7O0FBRUE7O0FBRUE7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0EsRUFBRTtBQUNGO0FBQ0E7QUFDQSxFQUFFO0FBQ0Y7QUFDQTtBQUNBLFNBQVM7QUFDVDs7QUFFQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTs7QUFFQTtBQUNBO0FBQ0E7QUFDQTtBQUNBOztBQUVBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTs7QUFFQTtBQUNBOztBQUVBO0FBQ0E7QUFDQTtBQUNBLGtDQUFrQyw2QkFBNkI7QUFDL0Q7QUFDQTs7QUFFQSIsImZpbGUiOiI3MjAxLmpzIiwic291cmNlc0NvbnRlbnQiOlsiJ3VzZSBzdHJpY3QnO1xuXG4vLyB2YXIgQ29uc3RydWN0ID0gcmVxdWlyZSgnZXMtYWJzdHJhY3QvMjAyMC9Db25zdHJ1Y3QnKTtcbnZhciBHZXQgPSByZXF1aXJlKCdlcy1hYnN0cmFjdC8yMDIwL0dldCcpO1xudmFyIFNldCA9IHJlcXVpcmUoJ2VzLWFic3RyYWN0LzIwMjAvU2V0Jyk7XG52YXIgU3BlY2llc0NvbnN0cnVjdG9yID0gcmVxdWlyZSgnZXMtYWJzdHJhY3QvMjAyMC9TcGVjaWVzQ29uc3RydWN0b3InKTtcbnZhciBUb0xlbmd0aCA9IHJlcXVpcmUoJ2VzLWFic3RyYWN0LzIwMjAvVG9MZW5ndGgnKTtcbnZhciBUb1N0cmluZyA9IHJlcXVpcmUoJ2VzLWFic3RyYWN0LzIwMjAvVG9TdHJpbmcnKTtcbnZhciBUeXBlID0gcmVxdWlyZSgnZXMtYWJzdHJhY3QvMjAyMC9UeXBlJyk7XG52YXIgZmxhZ3NHZXR0ZXIgPSByZXF1aXJlKCdyZWdleHAucHJvdG90eXBlLmZsYWdzJyk7XG5cbnZhciBSZWdFeHBTdHJpbmdJdGVyYXRvciA9IHJlcXVpcmUoJy4vaGVscGVycy9SZWdFeHBTdHJpbmdJdGVyYXRvcicpO1xudmFyIE9yaWdSZWdFeHAgPSBSZWdFeHA7XG5cbnZhciBDcmVhdGVSZWdFeHBTdHJpbmdJdGVyYXRvciA9IGZ1bmN0aW9uIENyZWF0ZVJlZ0V4cFN0cmluZ0l0ZXJhdG9yKFIsIFMsIGdsb2JhbCwgZnVsbFVuaWNvZGUpIHtcblx0aWYgKFR5cGUoUykgIT09ICdTdHJpbmcnKSB7XG5cdFx0dGhyb3cgbmV3IFR5cGVFcnJvcignXCJTXCIgdmFsdWUgbXVzdCBiZSBhIFN0cmluZycpO1xuXHR9XG5cdGlmIChUeXBlKGdsb2JhbCkgIT09ICdCb29sZWFuJykge1xuXHRcdHRocm93IG5ldyBUeXBlRXJyb3IoJ1wiZ2xvYmFsXCIgdmFsdWUgbXVzdCBiZSBhIEJvb2xlYW4nKTtcblx0fVxuXHRpZiAoVHlwZShmdWxsVW5pY29kZSkgIT09ICdCb29sZWFuJykge1xuXHRcdHRocm93IG5ldyBUeXBlRXJyb3IoJ1wiZnVsbFVuaWNvZGVcIiB2YWx1ZSBtdXN0IGJlIGEgQm9vbGVhbicpO1xuXHR9XG5cblx0dmFyIGl0ZXJhdG9yID0gbmV3IFJlZ0V4cFN0cmluZ0l0ZXJhdG9yKFIsIFMsIGdsb2JhbCwgZnVsbFVuaWNvZGUpO1xuXHRyZXR1cm4gaXRlcmF0b3I7XG59O1xuXG52YXIgc3VwcG9ydHNDb25zdHJ1Y3RpbmdXaXRoRmxhZ3MgPSAnZmxhZ3MnIGluIFJlZ0V4cC5wcm90b3R5cGU7XG5cbnZhciBjb25zdHJ1Y3RSZWdleFdpdGhGbGFncyA9IGZ1bmN0aW9uIGNvbnN0cnVjdFJlZ2V4KEMsIFIpIHtcblx0dmFyIG1hdGNoZXI7XG5cdC8vIHdvcmthcm91bmQgZm9yIG9sZGVyIGVuZ2luZXMgdGhhdCBsYWNrIFJlZ0V4cC5wcm90b3R5cGUuZmxhZ3Ncblx0dmFyIGZsYWdzID0gJ2ZsYWdzJyBpbiBSID8gR2V0KFIsICdmbGFncycpIDogVG9TdHJpbmcoZmxhZ3NHZXR0ZXIoUikpO1xuXHRpZiAoc3VwcG9ydHNDb25zdHJ1Y3RpbmdXaXRoRmxhZ3MgJiYgdHlwZW9mIGZsYWdzID09PSAnc3RyaW5nJykge1xuXHRcdG1hdGNoZXIgPSBuZXcgQyhSLCBmbGFncyk7XG5cdH0gZWxzZSBpZiAoQyA9PT0gT3JpZ1JlZ0V4cCkge1xuXHRcdC8vIHdvcmthcm91bmQgZm9yIG9sZGVyIGVuZ2luZXMgdGhhdCBjYW4gbm90IGNvbnN0cnVjdCBhIFJlZ0V4cCB3aXRoIGZsYWdzXG5cdFx0bWF0Y2hlciA9IG5ldyBDKFIuc291cmNlLCBmbGFncyk7XG5cdH0gZWxzZSB7XG5cdFx0bWF0Y2hlciA9IG5ldyBDKFIsIGZsYWdzKTtcblx0fVxuXHRyZXR1cm4geyBmbGFnczogZmxhZ3MsIG1hdGNoZXI6IG1hdGNoZXIgfTtcbn07XG5cbnZhciByZWdleE1hdGNoQWxsID0gZnVuY3Rpb24gU3ltYm9sTWF0Y2hBbGwoc3RyaW5nKSB7XG5cdHZhciBSID0gdGhpcztcblx0aWYgKFR5cGUoUikgIT09ICdPYmplY3QnKSB7XG5cdFx0dGhyb3cgbmV3IFR5cGVFcnJvcignXCJ0aGlzXCIgdmFsdWUgbXVzdCBiZSBhbiBPYmplY3QnKTtcblx0fVxuXHR2YXIgUyA9IFRvU3RyaW5nKHN0cmluZyk7XG5cdHZhciBDID0gU3BlY2llc0NvbnN0cnVjdG9yKFIsIE9yaWdSZWdFeHApO1xuXG5cdHZhciB0bXAgPSBjb25zdHJ1Y3RSZWdleFdpdGhGbGFncyhDLCBSKTtcblx0Ly8gdmFyIGZsYWdzID0gVG9TdHJpbmcoR2V0KFIsICdmbGFncycpKTtcblx0dmFyIGZsYWdzID0gdG1wLmZsYWdzO1xuXHQvLyB2YXIgbWF0Y2hlciA9IENvbnN0cnVjdChDLCBbUiwgZmxhZ3NdKTtcblx0dmFyIG1hdGNoZXIgPSB0bXAubWF0Y2hlcjtcblxuXHR2YXIgbGFzdEluZGV4ID0gVG9MZW5ndGgoR2V0KFIsICdsYXN0SW5kZXgnKSk7XG5cdFNldChtYXRjaGVyLCAnbGFzdEluZGV4JywgbGFzdEluZGV4LCB0cnVlKTtcblx0dmFyIGdsb2JhbCA9IGZsYWdzLmluZGV4T2YoJ2cnKSA+IC0xO1xuXHR2YXIgZnVsbFVuaWNvZGUgPSBmbGFncy5pbmRleE9mKCd1JykgPiAtMTtcblx0cmV0dXJuIENyZWF0ZVJlZ0V4cFN0cmluZ0l0ZXJhdG9yKG1hdGNoZXIsIFMsIGdsb2JhbCwgZnVsbFVuaWNvZGUpO1xufTtcblxudmFyIGRlZmluZVAgPSBPYmplY3QuZGVmaW5lUHJvcGVydHk7XG52YXIgZ09QRCA9IE9iamVjdC5nZXRPd25Qcm9wZXJ0eURlc2NyaXB0b3I7XG5cbmlmIChkZWZpbmVQICYmIGdPUEQpIHtcblx0dmFyIGRlc2MgPSBnT1BEKHJlZ2V4TWF0Y2hBbGwsICduYW1lJyk7XG5cdGlmIChkZXNjICYmIGRlc2MuY29uZmlndXJhYmxlKSB7XG5cdFx0ZGVmaW5lUChyZWdleE1hdGNoQWxsLCAnbmFtZScsIHsgdmFsdWU6ICdbU3ltYm9sLm1hdGNoQWxsXScgfSk7XG5cdH1cbn1cblxubW9kdWxlLmV4cG9ydHMgPSByZWdleE1hdGNoQWxsO1xuIl0sInNvdXJjZVJvb3QiOiIifQ==\n//# sourceURL=webpack-internal:///7201\n")},2376:(module,__unused_webpack_exports,__webpack_require__)=>{"use strict";eval("\n\nvar define = __webpack_require__(4289);\nvar hasSymbols = __webpack_require__(1405)();\nvar getPolyfill = __webpack_require__(3447);\nvar regexpMatchAllPolyfill = __webpack_require__(6966);\n\nvar defineP = Object.defineProperty;\nvar gOPD = Object.getOwnPropertyDescriptor;\n\nmodule.exports = function shimMatchAll() {\n\tvar polyfill = getPolyfill();\n\tdefine(\n\t\tString.prototype,\n\t\t{ matchAll: polyfill },\n\t\t{ matchAll: function () { return String.prototype.matchAll !== polyfill; } }\n\t);\n\tif (hasSymbols) {\n\t\t// eslint-disable-next-line no-restricted-properties\n\t\tvar symbol = Symbol.matchAll || (Symbol['for'] ? Symbol['for']('Symbol.matchAll') : Symbol('Symbol.matchAll'));\n\t\tdefine(\n\t\t\tSymbol,\n\t\t\t{ matchAll: symbol },\n\t\t\t{ matchAll: function () { return Symbol.matchAll !== symbol; } }\n\t\t);\n\n\t\tif (defineP && gOPD) {\n\t\t\tvar desc = gOPD(Symbol, symbol);\n\t\t\tif (!desc || desc.configurable) {\n\t\t\t\tdefineP(Symbol, symbol, {\n\t\t\t\t\tconfigurable: false,\n\t\t\t\t\tenumerable: false,\n\t\t\t\t\tvalue: symbol,\n\t\t\t\t\twritable: false\n\t\t\t\t});\n\t\t\t}\n\t\t}\n\n\t\tvar regexpMatchAll = regexpMatchAllPolyfill();\n\t\tvar func = {};\n\t\tfunc[symbol] = regexpMatchAll;\n\t\tvar predicate = {};\n\t\tpredicate[symbol] = function () {\n\t\t\treturn RegExp.prototype[symbol] !== regexpMatchAll;\n\t\t};\n\t\tdefine(RegExp.prototype, func, predicate);\n\t}\n\treturn polyfill;\n};\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///2376\n")},7786:(__unused_webpack_module,__unused_webpack___webpack_exports__,__webpack_require__)=>{"use strict";eval('\n// EXTERNAL MODULE: ./node_modules/approx-string-match/dist/index.js\nvar dist = __webpack_require__(3089);\n;// CONCATENATED MODULE: ./src/vendor/hypothesis/anchoring/match-quote.js\n\n\n/**\n * @typedef {import(\'approx-string-match\').Match} StringMatch\n */\n\n/**\n * @typedef Match\n * @prop {number} start - Start offset of match in text\n * @prop {number} end - End offset of match in text\n * @prop {number} score -\n *   Score for the match between 0 and 1.0, where 1.0 indicates a perfect match\n *   for the quote and context.\n */\n\n/**\n * Find the best approximate matches for `str` in `text` allowing up to `maxErrors` errors.\n *\n * @param {string} text\n * @param {string} str\n * @param {number} maxErrors\n * @return {StringMatch[]}\n */\nfunction search(text, str, maxErrors) {\n  // Do a fast search for exact matches. The `approx-string-match` library\n  // doesn\'t currently incorporate this optimization itself.\n  let matchPos = 0;\n  let exactMatches = [];\n  while (matchPos !== -1) {\n    matchPos = text.indexOf(str, matchPos);\n    if (matchPos !== -1) {\n      exactMatches.push({\n        start: matchPos,\n        end: matchPos + str.length,\n        errors: 0,\n      });\n      matchPos += 1;\n    }\n  }\n  if (exactMatches.length > 0) {\n    return exactMatches;\n  }\n\n  // If there are no exact matches, do a more expensive search for matches\n  // with errors.\n  return (0,dist/* default */.Z)(text, str, maxErrors);\n}\n\n/**\n * Compute a score between 0 and 1.0 for the similarity between `text` and `str`.\n *\n * @param {string} text\n * @param {string} str\n */\nfunction textMatchScore(text, str) {\n  /* istanbul ignore next - `scoreMatch` will never pass an empty string */\n  if (str.length === 0 || text.length === 0) {\n    return 0.0;\n  }\n  const matches = search(text, str, str.length);\n\n  // prettier-ignore\n  return 1 - (matches[0].errors / str.length);\n}\n\n/**\n * Find the best approximate match for `quote` in `text`.\n *\n * Returns `null` if no match exceeding the minimum quality threshold was found.\n *\n * @param {string} text - Document text to search\n * @param {string} quote - String to find within `text`\n * @param {Object} context -\n *   Context in which the quote originally appeared. This is used to choose the\n *   best match.\n *   @param {string} [context.prefix] - Expected text before the quote\n *   @param {string} [context.suffix] - Expected text after the quote\n *   @param {number} [context.hint] - Expected offset of match within text\n * @return {Match|null}\n */\nfunction matchQuote(text, quote, context = {}) {\n  if (quote.length === 0) {\n    return null;\n  }\n\n  // Choose the maximum number of errors to allow for the initial search.\n  // This choice involves a tradeoff between:\n  //\n  //  - Recall (proportion of "good" matches found)\n  //  - Precision (proportion of matches found which are "good")\n  //  - Cost of the initial search and of processing the candidate matches [1]\n  //\n  // [1] Specifically, the expected-time complexity of the initial search is\n  //     `O((maxErrors / 32) * text.length)`. See `approx-string-match` docs.\n  const maxErrors = Math.min(256, quote.length / 2);\n\n  // Find closest matches for `quote` in `text` based on edit distance.\n  const matches = search(text, quote, maxErrors);\n\n  if (matches.length === 0) {\n    return null;\n  }\n\n  /**\n   * Compute a score between 0 and 1.0 for a match candidate.\n   *\n   * @param {StringMatch} match\n   */\n  const scoreMatch = match => {\n    const quoteWeight = 50; // Similarity of matched text to quote.\n    const prefixWeight = 20; // Similarity of text before matched text to `context.prefix`.\n    const suffixWeight = 20; // Similarity of text after matched text to `context.suffix`.\n    const posWeight = 2; // Proximity to expected location. Used as a tie-breaker.\n\n    const quoteScore = 1 - match.errors / quote.length;\n\n    const prefixScore = context.prefix\n      ? textMatchScore(\n          text.slice(Math.max(0, match.start - context.prefix.length), match.start),\n          context.prefix\n        )\n      : 1.0;\n    const suffixScore = context.suffix\n      ? textMatchScore(\n          text.slice(match.end, match.end + context.suffix.length),\n          context.suffix\n        )\n      : 1.0;\n\n    let posScore = 1.0;\n    if (typeof context.hint === \'number\') {\n      const offset = Math.abs(match.start - context.hint);\n      posScore = 1.0 - offset / text.length;\n    }\n\n    const rawScore =\n      quoteWeight * quoteScore +\n      prefixWeight * prefixScore +\n      suffixWeight * suffixScore +\n      posWeight * posScore;\n    const maxScore = quoteWeight + prefixWeight + suffixWeight + posWeight;\n    const normalizedScore = rawScore / maxScore;\n\n    return normalizedScore;\n  };\n\n  // Rank matches based on similarity of actual and expected surrounding text\n  // and actual/expected offset in the document text.\n  const scoredMatches = matches.map(m => ({\n    start: m.start,\n    end: m.end,\n    score: scoreMatch(m),\n  }));\n\n  // Choose match with highest score.\n  scoredMatches.sort((a, b) => b.score - a.score);\n  return scoredMatches[0];\n}\n\n;// CONCATENATED MODULE: ./src/vendor/hypothesis/anchoring/text-range.js\n/**\n * Return the combined length of text nodes contained in `node`.\n *\n * @param {Node} node\n */\nfunction nodeTextLength(node) {\n  switch (node.nodeType) {\n    case Node.ELEMENT_NODE:\n    case Node.TEXT_NODE:\n      // nb. `textContent` excludes text in comments and processing instructions\n      // when called on a parent element, so we don\'t need to subtract that here.\n\n      return /** @type {string} */ (node.textContent).length;\n    default:\n      return 0;\n  }\n}\n\n/**\n * Return the total length of the text of all previous siblings of `node`.\n *\n * @param {Node} node\n */\nfunction previousSiblingsTextLength(node) {\n  let sibling = node.previousSibling;\n  let length = 0;\n  while (sibling) {\n    length += nodeTextLength(sibling);\n    sibling = sibling.previousSibling;\n  }\n  return length;\n}\n\n/**\n * Resolve one or more character offsets within an element to (text node, position)\n * pairs.\n *\n * @param {Element} element\n * @param {number[]} offsets - Offsets, which must be sorted in ascending order\n * @return {{ node: Text, offset: number }[]}\n */\nfunction resolveOffsets(element, ...offsets) {\n  let nextOffset = offsets.shift();\n  const nodeIter = /** @type {Document} */ (\n    element.ownerDocument\n  ).createNodeIterator(element, NodeFilter.SHOW_TEXT);\n  const results = [];\n\n  let currentNode = nodeIter.nextNode();\n  let textNode;\n  let length = 0;\n\n  // Find the text node containing the `nextOffset`th character from the start\n  // of `element`.\n  while (nextOffset !== undefined && currentNode) {\n    textNode = /** @type {Text} */ (currentNode);\n    if (length + textNode.data.length > nextOffset) {\n      results.push({ node: textNode, offset: nextOffset - length });\n      nextOffset = offsets.shift();\n    } else {\n      currentNode = nodeIter.nextNode();\n      length += textNode.data.length;\n    }\n  }\n\n  // Boundary case.\n  while (nextOffset !== undefined && textNode && length === nextOffset) {\n    results.push({ node: textNode, offset: textNode.data.length });\n    nextOffset = offsets.shift();\n  }\n\n  if (nextOffset !== undefined) {\n    throw new RangeError(\'Offset exceeds text length\');\n  }\n\n  return results;\n}\n\nlet RESOLVE_FORWARDS = 1;\nlet RESOLVE_BACKWARDS = 2;\n\n/**\n * Represents an offset within the text content of an element.\n *\n * This position can be resolved to a specific descendant node in the current\n * DOM subtree of the element using the `resolve` method.\n */\nclass text_range_TextPosition {\n  /**\n   * Construct a `TextPosition` that refers to the text position `offset` within\n   * the text content of `element`.\n   *\n   * @param {Element} element\n   * @param {number} offset\n   */\n  constructor(element, offset) {\n    if (offset < 0) {\n      throw new Error(\'Offset is invalid\');\n    }\n\n    /** Element that `offset` is relative to. */\n    this.element = element;\n\n    /** Character offset from the start of the element\'s `textContent`. */\n    this.offset = offset;\n  }\n\n  /**\n   * Return a copy of this position with offset relative to a given ancestor\n   * element.\n   *\n   * @param {Element} parent - Ancestor of `this.element`\n   * @return {TextPosition}\n   */\n  relativeTo(parent) {\n    if (!parent.contains(this.element)) {\n      throw new Error(\'Parent is not an ancestor of current element\');\n    }\n\n    let el = this.element;\n    let offset = this.offset;\n    while (el !== parent) {\n      offset += previousSiblingsTextLength(el);\n      el = /** @type {Element} */ (el.parentElement);\n    }\n\n    return new text_range_TextPosition(el, offset);\n  }\n\n  /**\n   * Resolve the position to a specific text node and offset within that node.\n   *\n   * Throws if `this.offset` exceeds the length of the element\'s text. In the\n   * case where the element has no text and `this.offset` is 0, the `direction`\n   * option determines what happens.\n   *\n   * Offsets at the boundary between two nodes are resolved to the start of the\n   * node that begins at the boundary.\n   *\n   * @param {Object} [options]\n   *   @param {RESOLVE_FORWARDS|RESOLVE_BACKWARDS} [options.direction] -\n   *     Specifies in which direction to search for the nearest text node if\n   *     `this.offset` is `0` and `this.element` has no text. If not specified\n   *     an error is thrown.\n   * @return {{ node: Text, offset: number }}\n   * @throws {RangeError}\n   */\n  resolve(options = {}) {\n    try {\n      return resolveOffsets(this.element, this.offset)[0];\n    } catch (err) {\n      if (this.offset === 0 && options.direction !== undefined) {\n        const tw = document.createTreeWalker(\n          this.element.getRootNode(),\n          NodeFilter.SHOW_TEXT\n        );\n        tw.currentNode = this.element;\n        const forwards = options.direction === RESOLVE_FORWARDS;\n        const text = /** @type {Text|null} */ (\n          forwards ? tw.nextNode() : tw.previousNode()\n        );\n        if (!text) {\n          throw err;\n        }\n        return { node: text, offset: forwards ? 0 : text.data.length };\n      } else {\n        throw err;\n      }\n    }\n  }\n\n  /**\n   * Construct a `TextPosition` that refers to the `offset`th character within\n   * `node`.\n   *\n   * @param {Node} node\n   * @param {number} offset\n   * @return {TextPosition}\n   */\n  static fromCharOffset(node, offset) {\n    switch (node.nodeType) {\n      case Node.TEXT_NODE:\n        return text_range_TextPosition.fromPoint(node, offset);\n      case Node.ELEMENT_NODE:\n        return new text_range_TextPosition(/** @type {Element} */ (node), offset);\n      default:\n        throw new Error(\'Node is not an element or text node\');\n    }\n  }\n\n  /**\n   * Construct a `TextPosition` representing the range start or end point (node, offset).\n   *\n   * @param {Node} node - Text or Element node\n   * @param {number} offset - Offset within the node.\n   * @return {TextPosition}\n   */\n  static fromPoint(node, offset) {\n    switch (node.nodeType) {\n      case Node.TEXT_NODE: {\n        if (offset < 0 || offset > /** @type {Text} */ (node).data.length) {\n          throw new Error(\'Text node offset is out of range\');\n        }\n\n        if (!node.parentElement) {\n          throw new Error(\'Text node has no parent\');\n        }\n\n        // Get the offset from the start of the parent element.\n        const textOffset = previousSiblingsTextLength(node) + offset;\n\n        return new text_range_TextPosition(node.parentElement, textOffset);\n      }\n      case Node.ELEMENT_NODE: {\n        if (offset < 0 || offset > node.childNodes.length) {\n          throw new Error(\'Child node offset is out of range\');\n        }\n\n        // Get the text length before the `offset`th child of element.\n        let textOffset = 0;\n        for (let i = 0; i < offset; i++) {\n          textOffset += nodeTextLength(node.childNodes[i]);\n        }\n\n        return new text_range_TextPosition(/** @type {Element} */ (node), textOffset);\n      }\n      default:\n        throw new Error(\'Point is not in an element or text node\');\n    }\n  }\n}\n\n/**\n * Represents a region of a document as a (start, end) pair of `TextPosition` points.\n *\n * Representing a range in this way allows for changes in the DOM content of the\n * range which don\'t affect its text content, without affecting the text content\n * of the range itself.\n */\nclass text_range_TextRange {\n  /**\n   * Construct an immutable `TextRange` from a `start` and `end` point.\n   *\n   * @param {TextPosition} start\n   * @param {TextPosition} end\n   */\n  constructor(start, end) {\n    this.start = start;\n    this.end = end;\n  }\n\n  /**\n   * Return a copy of this range with start and end positions relative to a\n   * given ancestor. See `TextPosition.relativeTo`.\n   *\n   * @param {Element} element\n   */\n  relativeTo(element) {\n    return new text_range_TextRange(\n      this.start.relativeTo(element),\n      this.end.relativeTo(element)\n    );\n  }\n\n  /**\n   * Resolve the `TextRange` to a DOM range.\n   *\n   * The resulting DOM Range will always start and end in a `Text` node.\n   * Hence `TextRange.fromRange(range).toRange()` can be used to "shrink" a\n   * range to the text it contains.\n   *\n   * May throw if the `start` or `end` positions cannot be resolved to a range.\n   *\n   * @return {Range}\n   */\n  toRange() {\n    let start;\n    let end;\n\n    if (\n      this.start.element === this.end.element &&\n      this.start.offset <= this.end.offset\n    ) {\n      // Fast path for start and end points in same element.\n      [start, end] = resolveOffsets(\n        this.start.element,\n        this.start.offset,\n        this.end.offset\n      );\n    } else {\n      start = this.start.resolve({ direction: RESOLVE_FORWARDS });\n      end = this.end.resolve({ direction: RESOLVE_BACKWARDS });\n    }\n\n    const range = new Range();\n    range.setStart(start.node, start.offset);\n    range.setEnd(end.node, end.offset);\n    return range;\n  }\n\n  /**\n   * Convert an existing DOM `Range` to a `TextRange`\n   *\n   * @param {Range} range\n   * @return {TextRange}\n   */\n  static fromRange(range) {\n    const start = text_range_TextPosition.fromPoint(\n      range.startContainer,\n      range.startOffset\n    );\n    const end = text_range_TextPosition.fromPoint(range.endContainer, range.endOffset);\n    return new text_range_TextRange(start, end);\n  }\n\n  /**\n   * Return a `TextRange` from the `start`th to `end`th characters in `root`.\n   *\n   * @param {Element} root\n   * @param {number} start\n   * @param {number} end\n   */\n  static fromOffsets(root, start, end) {\n    return new text_range_TextRange(\n      new text_range_TextPosition(root, start),\n      new text_range_TextPosition(root, end)\n    );\n  }\n}\n\n;// CONCATENATED MODULE: ./src/vendor/hypothesis/anchoring/types.js\n/**\n * This module exports a set of classes for converting between DOM `Range`\n * objects and different types of selectors. It is mostly a thin wrapper around a\n * set of anchoring libraries. It serves two main purposes:\n *\n *  1. Providing a consistent interface across different types of anchors.\n *  2. Insulating the rest of the code from API changes in the underlying anchoring\n *     libraries.\n */\n\n\n\n\n\n/**\n * @typedef {import(\'../../types/api\').RangeSelector} RangeSelector\n * @typedef {import(\'../../types/api\').TextPositionSelector} TextPositionSelector\n * @typedef {import(\'../../types/api\').TextQuoteSelector} TextQuoteSelector\n */\n\n/**\n * Converts between `RangeSelector` selectors and `Range` objects.\n */\nclass RangeAnchor {\n  /**\n   * @param {Node} root - A root element from which to anchor.\n   * @param {Range} range -  A range describing the anchor.\n   */\n  constructor(root, range) {\n    this.root = root;\n    this.range = range;\n  }\n\n  /**\n   * @param {Node} root -  A root element from which to anchor.\n   * @param {Range} range -  A range describing the anchor.\n   */\n  static fromRange(root, range) {\n    return new RangeAnchor(root, range);\n  }\n\n  /**\n   * Create an anchor from a serialized `RangeSelector` selector.\n   *\n   * @param {Element} root -  A root element from which to anchor.\n   * @param {RangeSelector} selector\n   */\n  static fromSelector(root, selector) {\n    const startContainer = nodeFromXPath(selector.startContainer, root);\n    if (!startContainer) {\n      throw new Error(\'Failed to resolve startContainer XPath\');\n    }\n\n    const endContainer = nodeFromXPath(selector.endContainer, root);\n    if (!endContainer) {\n      throw new Error(\'Failed to resolve endContainer XPath\');\n    }\n\n    const startPos = TextPosition.fromCharOffset(\n      startContainer,\n      selector.startOffset\n    );\n    const endPos = TextPosition.fromCharOffset(\n      endContainer,\n      selector.endOffset\n    );\n\n    const range = new TextRange(startPos, endPos).toRange();\n    return new RangeAnchor(root, range);\n  }\n\n  toRange() {\n    return this.range;\n  }\n\n  /**\n   * @return {RangeSelector}\n   */\n  toSelector() {\n    // "Shrink" the range so that it tightly wraps its text. This ensures more\n    // predictable output for a given text selection.\n    const normalizedRange = TextRange.fromRange(this.range).toRange();\n\n    const textRange = TextRange.fromRange(normalizedRange);\n    const startContainer = xpathFromNode(textRange.start.element, this.root);\n    const endContainer = xpathFromNode(textRange.end.element, this.root);\n\n    return {\n      type: \'RangeSelector\',\n      startContainer,\n      startOffset: textRange.start.offset,\n      endContainer,\n      endOffset: textRange.end.offset,\n    };\n  }\n}\n\n/**\n * Converts between `TextPositionSelector` selectors and `Range` objects.\n */\nclass TextPositionAnchor {\n  /**\n   * @param {Element} root\n   * @param {number} start\n   * @param {number} end\n   */\n  constructor(root, start, end) {\n    this.root = root;\n    this.start = start;\n    this.end = end;\n  }\n\n  /**\n   * @param {Element} root\n   * @param {Range} range\n   */\n  static fromRange(root, range) {\n    const textRange = text_range_TextRange.fromRange(range).relativeTo(root);\n    return new TextPositionAnchor(\n      root,\n      textRange.start.offset,\n      textRange.end.offset\n    );\n  }\n  /**\n   * @param {Element} root\n   * @param {TextPositionSelector} selector\n   */\n  static fromSelector(root, selector) {\n    return new TextPositionAnchor(root, selector.start, selector.end);\n  }\n\n  /**\n   * @return {TextPositionSelector}\n   */\n  toSelector() {\n    return {\n      type: \'TextPositionSelector\',\n      start: this.start,\n      end: this.end,\n    };\n  }\n\n  toRange() {\n    return text_range_TextRange.fromOffsets(this.root, this.start, this.end).toRange();\n  }\n}\n\n/**\n * @typedef QuoteMatchOptions\n * @prop {number} [hint] - Expected position of match in text. See `matchQuote`.\n */\n\n/**\n * Converts between `TextQuoteSelector` selectors and `Range` objects.\n */\nclass TextQuoteAnchor {\n  /**\n   * @param {Element} root - A root element from which to anchor.\n   * @param {string} exact\n   * @param {Object} context\n   *   @param {string} [context.prefix]\n   *   @param {string} [context.suffix]\n   */\n  constructor(root, exact, context = {}) {\n    this.root = root;\n    this.exact = exact;\n    this.context = context;\n  }\n\n  /**\n   * Create a `TextQuoteAnchor` from a range.\n   *\n   * Will throw if `range` does not contain any text nodes.\n   *\n   * @param {Element} root\n   * @param {Range} range\n   */\n  static fromRange(root, range) {\n    const text = /** @type {string} */ (root.textContent);\n    const textRange = text_range_TextRange.fromRange(range).relativeTo(root);\n\n    const start = textRange.start.offset;\n    const end = textRange.end.offset;\n\n    // Number of characters around the quote to capture as context. We currently\n    // always use a fixed amount, but it would be better if this code was aware\n    // of logical boundaries in the document (paragraph, article etc.) to avoid\n    // capturing text unrelated to the quote.\n    //\n    // In regular prose the ideal content would often be the surrounding sentence.\n    // This is a natural unit of meaning which enables displaying quotes in\n    // context even when the document is not available. We could use `Intl.Segmenter`\n    // for this when available.\n    const contextLen = 32;\n\n    return new TextQuoteAnchor(root, text.slice(start, end), {\n      prefix: text.slice(Math.max(0, start - contextLen), start),\n      suffix: text.slice(end, Math.min(text.length, end + contextLen)),\n    });\n  }\n\n  /**\n   * @param {Element} root\n   * @param {TextQuoteSelector} selector\n   */\n  static fromSelector(root, selector) {\n    const { prefix, suffix } = selector;\n    return new TextQuoteAnchor(root, selector.exact, { prefix, suffix });\n  }\n\n  /**\n   * @return {TextQuoteSelector}\n   */\n  toSelector() {\n    return {\n      type: \'TextQuoteSelector\',\n      exact: this.exact,\n      prefix: this.context.prefix,\n      suffix: this.context.suffix,\n    };\n  }\n\n  /**\n   * @param {QuoteMatchOptions} [options]\n   */\n  toRange(options = {}) {\n    return this.toPositionAnchor(options).toRange();\n  }\n\n  /**\n   * @param {QuoteMatchOptions} [options]\n   */\n  toPositionAnchor(options = {}) {\n    const text = /** @type {string} */ (this.root.textContent);\n    const match = matchQuote(text, this.exact, {\n      ...this.context,\n      hint: options.hint,\n    });\n    if (!match) {\n      throw new Error(\'Quote not found\');\n    }\n    return new TextPositionAnchor(this.root, match.start, match.end);\n  }\n}\n\n;// CONCATENATED MODULE: ./src/utils.js\n//\n//  Copyright 2021 Readium Foundation. All rights reserved.\n//  Use of this source code is governed by the BSD-style license\n//  available in the top-level LICENSE file of the project.\n//\n\n\n\n// Catch JS errors to log them in the app.\nwindow.addEventListener(\n  "error",\n  function (event) {\n    Android.logError(event.message, event.filename, event.lineno);\n  },\n  false\n);\n\n// Notify native code that the page has loaded.\nwindow.addEventListener(\n  "load",\n  function () {\n    window.addEventListener("orientationchange", function () {\n      onViewportWidthChanged();\n      snapCurrentOffset();\n    });\n    onViewportWidthChanged();\n  },\n  false\n);\n\nvar pageWidth = 1;\n\nfunction onViewportWidthChanged() {\n  // We can\'t rely on window.innerWidth for the pageWidth on Android, because if the\n  // device pixel ratio is not an integer, we get rounding issues offsetting the pages.\n  //\n  // See https://github.com/readium/readium-css/issues/97\n  // and https://github.com/readium/r2-navigator-kotlin/issues/146\n  var width = Android.getViewportWidth();\n  pageWidth = width / window.devicePixelRatio;\n  setProperty(\n    "--RS__viewportWidth",\n    "calc(" + width + "px / " + window.devicePixelRatio + ")"\n  );\n}\n\nfunction isScrollModeEnabled() {\n  return (\n    document.documentElement.style\n      .getPropertyValue("--USER__scroll")\n      .toString()\n      .trim() == "readium-scroll-on"\n  );\n}\n\nfunction isRTL() {\n  return document.body.dir.toLowerCase() == "rtl";\n}\n\n// Scroll to the given TagId in document and snap.\nfunction scrollToId(id) {\n  var element = document.getElementById(id);\n  if (!element) {\n    return false;\n  }\n\n  return scrollToRect(element.getBoundingClientRect());\n}\n\n// Position must be in the range [0 - 1], 0-100%.\nfunction scrollToPosition(position) {\n  //        Android.log("scrollToPosition " + position);\n  if (position < 0 || position > 1) {\n    throw "scrollToPosition() must be given a position from 0.0 to  1.0";\n  }\n\n  let offset;\n  if (isScrollModeEnabled()) {\n    offset = document.scrollingElement.scrollHeight * position;\n    document.scrollingElement.scrollTop = offset;\n    // window.scrollTo(0, offset);\n  } else {\n    var documentWidth = document.scrollingElement.scrollWidth;\n    var factor = isRTL() ? -1 : 1;\n    offset = documentWidth * position * factor;\n    document.scrollingElement.scrollLeft = snapOffset(offset);\n  }\n}\n\n// Scrolls to the first occurrence of the given text snippet.\n//\n// The expected text argument is a Locator Text object, as defined here:\n// https://readium.org/architecture/models/locators/\nfunction scrollToText(text) {\n  let range = rangeFromLocator({ text });\n  if (!range) {\n    return false;\n  }\n  scrollToRange(range);\n  return true;\n}\n\nfunction scrollToRange(range) {\n  scrollToRect(range.getBoundingClientRect());\n}\n\nfunction scrollToRect(rect) {\n  if (isScrollModeEnabled()) {\n    document.scrollingElement.scrollTop =\n      rect.top + window.scrollY - window.innerHeight / 2;\n  } else {\n    document.scrollingElement.scrollLeft = snapOffset(\n      rect.left + window.scrollX\n    );\n  }\n}\n\nfunction scrollToStart() {\n  //        Android.log("scrollToStart");\n  if (!isScrollModeEnabled()) {\n    document.scrollingElement.scrollLeft = 0;\n  } else {\n    document.scrollingElement.scrollTop = 0;\n    window.scrollTo(0, 0);\n  }\n}\n\nfunction scrollToEnd() {\n  //        Android.log("scrollToEnd");\n  if (!isScrollModeEnabled()) {\n    var factor = isRTL() ? -1 : 1;\n    document.scrollingElement.scrollLeft = snapOffset(\n      document.scrollingElement.scrollWidth * factor\n    );\n  } else {\n    document.scrollingElement.scrollTop = document.body.scrollHeight;\n    window.scrollTo(0, document.body.scrollHeight);\n  }\n}\n\n// Returns false if the page is already at the left-most scroll offset.\nfunction scrollLeft() {\n  var documentWidth = document.scrollingElement.scrollWidth;\n  var offset = window.scrollX - pageWidth;\n  var minOffset = isRTL() ? -(documentWidth - pageWidth) : 0;\n  return scrollToOffset(Math.max(offset, minOffset));\n}\n\n// Returns false if the page is already at the right-most scroll offset.\nfunction scrollRight() {\n  var documentWidth = document.scrollingElement.scrollWidth;\n  var offset = window.scrollX + pageWidth;\n  var maxOffset = isRTL() ? 0 : documentWidth - pageWidth;\n  return scrollToOffset(Math.min(offset, maxOffset));\n}\n\n// Scrolls to the given left offset.\n// Returns false if the page scroll position is already close enough to the given offset.\nfunction scrollToOffset(offset) {\n  //        Android.log("scrollToOffset " + offset);\n  if (isScrollModeEnabled()) {\n    throw "Called scrollToOffset() with scroll mode enabled. This can only be used in paginated mode.";\n  }\n\n  var currentOffset = window.scrollX;\n  document.scrollingElement.scrollLeft = snapOffset(offset);\n  // In some case the scrollX cannot reach the position respecting to innerWidth\n  var diff = Math.abs(currentOffset - offset) / pageWidth;\n  return diff > 0.01;\n}\n\n// Snap the offset to the screen width (page width).\nfunction snapOffset(offset) {\n  var value = offset + (isRTL() ? -1 : 1);\n  return value - (value % pageWidth);\n}\n\n// Snaps the current offset to the page width.\nfunction snapCurrentOffset() {\n  //        Android.log("snapCurrentOffset");\n  if (isScrollModeEnabled()) {\n    return;\n  }\n  var currentOffset = window.scrollX;\n  // Adds half a page to make sure we don\'t snap to the previous page.\n  var factor = isRTL() ? -1 : 1;\n  var delta = factor * (pageWidth / 2);\n  document.scrollingElement.scrollLeft = snapOffset(currentOffset + delta);\n}\n\nfunction rangeFromLocator(locator) {\n  let text = locator.text;\n  if (!text || !text.highlight) {\n    return null;\n  }\n  try {\n    let anchor = new TextQuoteAnchor(document.body, text.highlight, {\n      prefix: text.before,\n      suffix: text.after,\n    });\n    return anchor.toRange();\n  } catch (e) {\n    logError(e);\n    return null;\n  }\n}\n\n/// User Settings.\n\n// For setting user setting.\nfunction setProperty(key, value) {\n  var root = document.documentElement;\n\n  root.style.setProperty(key, value);\n}\n\n// For removing user setting.\nfunction removeProperty(key) {\n  var root = document.documentElement;\n\n  root.style.removeProperty(key);\n}\n\n/// Toolkit\n\nfunction log() {\n  var message = Array.prototype.slice.call(arguments).join(" ");\n  Android.log(message);\n}\n\nfunction logError(message) {\n  Android.logError(message, "", 0);\n}\n\n;// CONCATENATED MODULE: ./src/rect.js\n//\n//  Copyright 2021 Readium Foundation. All rights reserved.\n//  Use of this source code is governed by the BSD-style license\n//  available in the top-level LICENSE file of the project.\n//\n\n\n\nconst debug = false;\n\n/**\n * Converts a DOMRect into a JSON object understandable by the native side.\n */\nfunction toNativeRect(rect) {\n  const pixelRatio = window.devicePixelRatio;\n  const width = rect.width * pixelRatio;\n  const height = rect.height * pixelRatio;\n  const left = rect.left * pixelRatio;\n  const top = rect.top * pixelRatio;\n  const right = left + width;\n  const bottom = top + height;\n  return { width, height, left, top, right, bottom };\n}\n\nfunction getClientRectsNoOverlap(\n  range,\n  doNotMergeHorizontallyAlignedRects\n) {\n  let clientRects = range.getClientRects();\n\n  const tolerance = 1;\n  const originalRects = [];\n  for (const rangeClientRect of clientRects) {\n    originalRects.push({\n      bottom: rangeClientRect.bottom,\n      height: rangeClientRect.height,\n      left: rangeClientRect.left,\n      right: rangeClientRect.right,\n      top: rangeClientRect.top,\n      width: rangeClientRect.width,\n    });\n  }\n  const mergedRects = mergeTouchingRects(\n    originalRects,\n    tolerance,\n    doNotMergeHorizontallyAlignedRects\n  );\n  const noContainedRects = removeContainedRects(mergedRects, tolerance);\n  const newRects = replaceOverlapingRects(noContainedRects);\n  const minArea = 2 * 2;\n  for (let j = newRects.length - 1; j >= 0; j--) {\n    const rect = newRects[j];\n    const bigEnough = rect.width * rect.height > minArea;\n    if (!bigEnough) {\n      if (newRects.length > 1) {\n        rect_log("CLIENT RECT: remove small");\n        newRects.splice(j, 1);\n      } else {\n        rect_log("CLIENT RECT: remove small, but keep otherwise empty!");\n        break;\n      }\n    }\n  }\n  rect_log(`CLIENT RECT: reduced ${originalRects.length} --\x3e ${newRects.length}`);\n  return newRects;\n}\n\nfunction mergeTouchingRects(\n  rects,\n  tolerance,\n  doNotMergeHorizontallyAlignedRects\n) {\n  for (let i = 0; i < rects.length; i++) {\n    for (let j = i + 1; j < rects.length; j++) {\n      const rect1 = rects[i];\n      const rect2 = rects[j];\n      if (rect1 === rect2) {\n        rect_log("mergeTouchingRects rect1 === rect2 ??!");\n        continue;\n      }\n      const rectsLineUpVertically =\n        almostEqual(rect1.top, rect2.top, tolerance) &&\n        almostEqual(rect1.bottom, rect2.bottom, tolerance);\n      const rectsLineUpHorizontally =\n        almostEqual(rect1.left, rect2.left, tolerance) &&\n        almostEqual(rect1.right, rect2.right, tolerance);\n      const horizontalAllowed = !doNotMergeHorizontallyAlignedRects;\n      const aligned =\n        (rectsLineUpHorizontally && horizontalAllowed) ||\n        (rectsLineUpVertically && !rectsLineUpHorizontally);\n      const canMerge = aligned && rectsTouchOrOverlap(rect1, rect2, tolerance);\n      if (canMerge) {\n        rect_log(\n          `CLIENT RECT: merging two into one, VERTICAL: ${rectsLineUpVertically} HORIZONTAL: ${rectsLineUpHorizontally} (${doNotMergeHorizontallyAlignedRects})`\n        );\n        const newRects = rects.filter((rect) => {\n          return rect !== rect1 && rect !== rect2;\n        });\n        const replacementClientRect = getBoundingRect(rect1, rect2);\n        newRects.push(replacementClientRect);\n        return mergeTouchingRects(\n          newRects,\n          tolerance,\n          doNotMergeHorizontallyAlignedRects\n        );\n      }\n    }\n  }\n  return rects;\n}\n\nfunction getBoundingRect(rect1, rect2) {\n  const left = Math.min(rect1.left, rect2.left);\n  const right = Math.max(rect1.right, rect2.right);\n  const top = Math.min(rect1.top, rect2.top);\n  const bottom = Math.max(rect1.bottom, rect2.bottom);\n  return {\n    bottom,\n    height: bottom - top,\n    left,\n    right,\n    top,\n    width: right - left,\n  };\n}\n\nfunction removeContainedRects(rects, tolerance) {\n  const rectsToKeep = new Set(rects);\n  for (const rect of rects) {\n    const bigEnough = rect.width > 1 && rect.height > 1;\n    if (!bigEnough) {\n      rect_log("CLIENT RECT: remove tiny");\n      rectsToKeep.delete(rect);\n      continue;\n    }\n    for (const possiblyContainingRect of rects) {\n      if (rect === possiblyContainingRect) {\n        continue;\n      }\n      if (!rectsToKeep.has(possiblyContainingRect)) {\n        continue;\n      }\n      if (rectContains(possiblyContainingRect, rect, tolerance)) {\n        rect_log("CLIENT RECT: remove contained");\n        rectsToKeep.delete(rect);\n        break;\n      }\n    }\n  }\n  return Array.from(rectsToKeep);\n}\n\nfunction rectContains(rect1, rect2, tolerance) {\n  return (\n    rectContainsPoint(rect1, rect2.left, rect2.top, tolerance) &&\n    rectContainsPoint(rect1, rect2.right, rect2.top, tolerance) &&\n    rectContainsPoint(rect1, rect2.left, rect2.bottom, tolerance) &&\n    rectContainsPoint(rect1, rect2.right, rect2.bottom, tolerance)\n  );\n}\n\nfunction rectContainsPoint(rect, x, y, tolerance) {\n  return (\n    (rect.left < x || almostEqual(rect.left, x, tolerance)) &&\n    (rect.right > x || almostEqual(rect.right, x, tolerance)) &&\n    (rect.top < y || almostEqual(rect.top, y, tolerance)) &&\n    (rect.bottom > y || almostEqual(rect.bottom, y, tolerance))\n  );\n}\n\nfunction replaceOverlapingRects(rects) {\n  for (let i = 0; i < rects.length; i++) {\n    for (let j = i + 1; j < rects.length; j++) {\n      const rect1 = rects[i];\n      const rect2 = rects[j];\n      if (rect1 === rect2) {\n        rect_log("replaceOverlapingRects rect1 === rect2 ??!");\n        continue;\n      }\n      if (rectsTouchOrOverlap(rect1, rect2, -1)) {\n        let toAdd = [];\n        let toRemove;\n        const subtractRects1 = rectSubtract(rect1, rect2);\n        if (subtractRects1.length === 1) {\n          toAdd = subtractRects1;\n          toRemove = rect1;\n        } else {\n          const subtractRects2 = rectSubtract(rect2, rect1);\n          if (subtractRects1.length < subtractRects2.length) {\n            toAdd = subtractRects1;\n            toRemove = rect1;\n          } else {\n            toAdd = subtractRects2;\n            toRemove = rect2;\n          }\n        }\n        rect_log(`CLIENT RECT: overlap, cut one rect into ${toAdd.length}`);\n        const newRects = rects.filter((rect) => {\n          return rect !== toRemove;\n        });\n        Array.prototype.push.apply(newRects, toAdd);\n        return replaceOverlapingRects(newRects);\n      }\n    }\n  }\n  return rects;\n}\n\nfunction rectSubtract(rect1, rect2) {\n  const rectIntersected = rectIntersect(rect2, rect1);\n  if (rectIntersected.height === 0 || rectIntersected.width === 0) {\n    return [rect1];\n  }\n  const rects = [];\n  {\n    const rectA = {\n      bottom: rect1.bottom,\n      height: 0,\n      left: rect1.left,\n      right: rectIntersected.left,\n      top: rect1.top,\n      width: 0,\n    };\n    rectA.width = rectA.right - rectA.left;\n    rectA.height = rectA.bottom - rectA.top;\n    if (rectA.height !== 0 && rectA.width !== 0) {\n      rects.push(rectA);\n    }\n  }\n  {\n    const rectB = {\n      bottom: rectIntersected.top,\n      height: 0,\n      left: rectIntersected.left,\n      right: rectIntersected.right,\n      top: rect1.top,\n      width: 0,\n    };\n    rectB.width = rectB.right - rectB.left;\n    rectB.height = rectB.bottom - rectB.top;\n    if (rectB.height !== 0 && rectB.width !== 0) {\n      rects.push(rectB);\n    }\n  }\n  {\n    const rectC = {\n      bottom: rect1.bottom,\n      height: 0,\n      left: rectIntersected.left,\n      right: rectIntersected.right,\n      top: rectIntersected.bottom,\n      width: 0,\n    };\n    rectC.width = rectC.right - rectC.left;\n    rectC.height = rectC.bottom - rectC.top;\n    if (rectC.height !== 0 && rectC.width !== 0) {\n      rects.push(rectC);\n    }\n  }\n  {\n    const rectD = {\n      bottom: rect1.bottom,\n      height: 0,\n      left: rectIntersected.right,\n      right: rect1.right,\n      top: rect1.top,\n      width: 0,\n    };\n    rectD.width = rectD.right - rectD.left;\n    rectD.height = rectD.bottom - rectD.top;\n    if (rectD.height !== 0 && rectD.width !== 0) {\n      rects.push(rectD);\n    }\n  }\n  return rects;\n}\n\nfunction rectIntersect(rect1, rect2) {\n  const maxLeft = Math.max(rect1.left, rect2.left);\n  const minRight = Math.min(rect1.right, rect2.right);\n  const maxTop = Math.max(rect1.top, rect2.top);\n  const minBottom = Math.min(rect1.bottom, rect2.bottom);\n  return {\n    bottom: minBottom,\n    height: Math.max(0, minBottom - maxTop),\n    left: maxLeft,\n    right: minRight,\n    top: maxTop,\n    width: Math.max(0, minRight - maxLeft),\n  };\n}\n\nfunction rectsTouchOrOverlap(rect1, rect2, tolerance) {\n  return (\n    (rect1.left < rect2.right ||\n      (tolerance >= 0 && almostEqual(rect1.left, rect2.right, tolerance))) &&\n    (rect2.left < rect1.right ||\n      (tolerance >= 0 && almostEqual(rect2.left, rect1.right, tolerance))) &&\n    (rect1.top < rect2.bottom ||\n      (tolerance >= 0 && almostEqual(rect1.top, rect2.bottom, tolerance))) &&\n    (rect2.top < rect1.bottom ||\n      (tolerance >= 0 && almostEqual(rect2.top, rect1.bottom, tolerance)))\n  );\n}\n\nfunction almostEqual(a, b, tolerance) {\n  return Math.abs(a - b) <= tolerance;\n}\n\nfunction rect_log() {\n  if (debug) {\n    log.apply(null, arguments);\n  }\n}\n\n;// CONCATENATED MODULE: ./src/decorator.js\n//\n//  Copyright 2021 Readium Foundation. All rights reserved.\n//  Use of this source code is governed by the BSD-style license\n//  available in the top-level LICENSE file of the project.\n//\n\n\n\n\nlet styles = new Map();\nlet groups = new Map();\nvar lastGroupId = 0;\n\n/**\n * Registers a list of additional supported Decoration Templates.\n *\n * Each template object is indexed by the style ID.\n */\nfunction registerTemplates(newStyles) {\n  var stylesheet = "";\n\n  for (const [id, style] of Object.entries(newStyles)) {\n    styles.set(id, style);\n    if (style.stylesheet) {\n      stylesheet += style.stylesheet + "\\n";\n    }\n  }\n\n  if (stylesheet) {\n    let styleElement = document.createElement("style");\n    styleElement.innerHTML = stylesheet;\n    document.getElementsByTagName("head")[0].appendChild(styleElement);\n  }\n}\n\n/**\n * Returns an instance of DecorationGroup for the given group name.\n */\nfunction getDecorations(groupName) {\n  var group = groups.get(groupName);\n  if (!group) {\n    let id = "r2-decoration-" + lastGroupId++;\n    group = DecorationGroup(id, groupName);\n    groups.set(groupName, group);\n  }\n  return group;\n}\n\n/**\n * Handles click events on a Decoration.\n * Returns whether a decoration matched this event.\n */\nfunction handleDecorationClickEvent(event, clickEvent) {\n  if (groups.size === 0) {\n    return false;\n  }\n\n  function findTarget() {\n    for (const [group, groupContent] of groups) {\n      for (const item of groupContent.items.reverse()) {\n        if (!item.clickableElements) {\n          continue;\n        }\n        for (const element of item.clickableElements) {\n          let rect = element.getBoundingClientRect().toJSON();\n          if (rectContainsPoint(rect, event.clientX, event.clientY, 1)) {\n            return { group, item, element, rect };\n          }\n        }\n      }\n    }\n  }\n\n  let target = findTarget();\n  if (!target) {\n    return false;\n  }\n\n  return Android.onDecorationActivated(\n    JSON.stringify({\n      id: target.item.decoration.id,\n      group: target.group,\n      rect: toNativeRect(target.item.range.getBoundingClientRect()),\n      click: clickEvent,\n    })\n  );\n}\n\n/**\n * Creates a DecorationGroup object from a unique HTML ID and its name.\n */\nfunction DecorationGroup(groupId, groupName) {\n  var items = [];\n  var lastItemId = 0;\n  var container = null;\n\n  /**\n   * Adds a new decoration to the group.\n   */\n  function add(decoration) {\n    let id = groupId + "-" + lastItemId++;\n\n    let range = rangeFromLocator(decoration.locator);\n    if (!range) {\n      log("Can\'t locate DOM range for decoration", decoration);\n      return;\n    }\n\n    let item = { id, decoration, range };\n    items.push(item);\n    layout(item);\n  }\n\n  /**\n   * Removes the decoration with given ID from the group.\n   */\n  function remove(decorationId) {\n    let index = items.findIndex((i) => i.decoration.id === decorationId);\n    if (index === -1) {\n      return;\n    }\n\n    let item = items[index];\n    items.splice(index, 1);\n    item.clickableElements = null;\n    if (item.container) {\n      item.container.remove();\n      item.container = null;\n    }\n  }\n\n  /**\n   * Notifies that the given decoration was modified and needs to be updated.\n   */\n  function update(decoration) {\n    remove(decoration.id);\n    add(decoration);\n  }\n\n  /**\n   * Removes all decorations from this group.\n   */\n  function clear() {\n    clearContainer();\n    items.length = 0;\n  }\n\n  /**\n   * Recreates the decoration elements.\n   *\n   * To be called after reflowing the resource, for example.\n   */\n  function requestLayout() {\n    clearContainer();\n    items.forEach((item) => layout(item));\n  }\n\n  /**\n   * Layouts a single Decoration item.\n   */\n  function layout(item) {\n    let groupContainer = requireContainer();\n\n    let style = styles.get(item.decoration.style);\n    if (!style) {\n      logError(`Unknown decoration style: ${item.decoration.style}`);\n      return;\n    }\n\n    let itemContainer = document.createElement("div");\n    itemContainer.setAttribute("id", item.id);\n    itemContainer.setAttribute("data-style", item.decoration.style);\n    itemContainer.style.setProperty("pointer-events", "none");\n\n    let viewportWidth = window.innerWidth;\n    let columnCount = parseInt(\n      getComputedStyle(document.documentElement).getPropertyValue(\n        "column-count"\n      )\n    );\n    let pageWidth = viewportWidth / (columnCount || 1);\n    let scrollingElement = document.scrollingElement;\n    let xOffset = scrollingElement.scrollLeft;\n    let yOffset = scrollingElement.scrollTop;\n\n    function positionElement(element, rect, boundingRect) {\n      element.style.position = "absolute";\n\n      if (style.width === "wrap") {\n        element.style.width = `${rect.width}px`;\n        element.style.height = `${rect.height}px`;\n        element.style.left = `${rect.left + xOffset}px`;\n        element.style.top = `${rect.top + yOffset}px`;\n      } else if (style.width === "viewport") {\n        element.style.width = `${viewportWidth}px`;\n        element.style.height = `${rect.height}px`;\n        let left = Math.floor(rect.left / viewportWidth) * viewportWidth;\n        element.style.left = `${left + xOffset}px`;\n        element.style.top = `${rect.top + yOffset}px`;\n      } else if (style.width === "bounds") {\n        element.style.width = `${boundingRect.width}px`;\n        element.style.height = `${rect.height}px`;\n        element.style.left = `${boundingRect.left + xOffset}px`;\n        element.style.top = `${rect.top + yOffset}px`;\n      } else if (style.width === "page") {\n        element.style.width = `${pageWidth}px`;\n        element.style.height = `${rect.height}px`;\n        let left = Math.floor(rect.left / pageWidth) * pageWidth;\n        element.style.left = `${left + xOffset}px`;\n        element.style.top = `${rect.top + yOffset}px`;\n      }\n    }\n\n    let boundingRect = item.range.getBoundingClientRect();\n\n    let elementTemplate;\n    try {\n      let template = document.createElement("template");\n      template.innerHTML = item.decoration.element.trim();\n      elementTemplate = template.content.firstElementChild;\n    } catch (error) {\n      logError(\n        `Invalid decoration element "${item.decoration.element}": ${error.message}`\n      );\n      return;\n    }\n\n    if (style.layout === "boxes") {\n      let doNotMergeHorizontallyAlignedRects = true;\n      let clientRects = getClientRectsNoOverlap(\n        item.range,\n        doNotMergeHorizontallyAlignedRects\n      );\n\n      clientRects = clientRects.sort((r1, r2) => {\n        if (r1.top < r2.top) {\n          return -1;\n        } else if (r1.top > r2.top) {\n          return 1;\n        } else {\n          return 0;\n        }\n      });\n\n      for (let clientRect of clientRects) {\n        const line = elementTemplate.cloneNode(true);\n        line.style.setProperty("pointer-events", "none");\n        positionElement(line, clientRect, boundingRect);\n        itemContainer.append(line);\n      }\n    } else if (style.layout === "bounds") {\n      const bounds = elementTemplate.cloneNode(true);\n      bounds.style.setProperty("pointer-events", "none");\n      positionElement(bounds, boundingRect, boundingRect);\n\n      itemContainer.append(bounds);\n    }\n\n    groupContainer.append(itemContainer);\n    item.container = itemContainer;\n    item.clickableElements = Array.from(\n      itemContainer.querySelectorAll("[data-activable=\'1\']")\n    );\n    if (item.clickableElements.length === 0) {\n      item.clickableElements = Array.from(itemContainer.children);\n    }\n  }\n\n  /**\n   * Returns the group container element, after making sure it exists.\n   */\n  function requireContainer() {\n    if (!container) {\n      container = document.createElement("div");\n      container.setAttribute("id", groupId);\n      container.setAttribute("data-group", groupName);\n      container.style.setProperty("pointer-events", "none");\n      document.body.append(container);\n    }\n    return container;\n  }\n\n  /**\n   * Removes the group container.\n   */\n  function clearContainer() {\n    if (container) {\n      container.remove();\n      container = null;\n    }\n  }\n\n  return { add, remove, update, clear, items, requestLayout };\n}\n\nwindow.addEventListener(\n  "load",\n  function () {\n    // Will relayout all the decorations when the document body is resized.\n    const body = document.body;\n    var lastSize = { width: 0, height: 0 };\n    const observer = new ResizeObserver(() => {\n      if (\n        lastSize.width === body.clientWidth &&\n        lastSize.height === body.clientHeight\n      ) {\n        return;\n      }\n      lastSize = {\n        width: body.clientWidth,\n        height: body.clientHeight,\n      };\n\n      groups.forEach(function (group) {\n        group.requestLayout();\n      });\n    });\n    observer.observe(body);\n  },\n  false\n);\n\n;// CONCATENATED MODULE: ./src/gestures.js\n/*\n * Copyright 2021 Readium Foundation. All rights reserved.\n * Use of this source code is governed by the BSD-style license\n * available in the top-level LICENSE file of the project.\n */\n\n\n\nwindow.addEventListener("DOMContentLoaded", function () {\n  document.addEventListener("click", onClick, false);\n});\n\nfunction onClick(event) {\n  if (!window.getSelection().isCollapsed) {\n    // There\'s an on-going selection, the tap will dismiss it so we don\'t forward it.\n    return;\n  }\n\n  var pixelRatio = window.devicePixelRatio;\n  let clickEvent = {\n    defaultPrevented: event.defaultPrevented,\n    x: event.clientX * pixelRatio,\n    y: event.clientY * pixelRatio,\n    targetElement: event.target.outerHTML,\n    interactiveElement: nearestInteractiveElement(event.target),\n  };\n\n  if (handleDecorationClickEvent(event, clickEvent)) {\n    return;\n  }\n\n  // Send the tap data over the JS bridge even if it\'s been handled within the web view, so that\n  // it can be preserved and used by the toolkit if needed.\n  var shouldPreventDefault = Android.onTap(JSON.stringify(clickEvent));\n\n  if (shouldPreventDefault) {\n    event.stopPropagation();\n    event.preventDefault();\n  }\n}\n\n// See. https://github.com/JayPanoz/architecture/tree/touch-handling/misc/touch-handling\nfunction nearestInteractiveElement(element) {\n  var interactiveTags = [\n    "a",\n    "audio",\n    "button",\n    "canvas",\n    "details",\n    "input",\n    "label",\n    "option",\n    "select",\n    "submit",\n    "textarea",\n    "video",\n  ];\n  if (interactiveTags.indexOf(element.nodeName.toLowerCase()) != -1) {\n    return element.outerHTML;\n  }\n\n  // Checks whether the element is editable by the user.\n  if (\n    element.hasAttribute("contenteditable") &&\n    element.getAttribute("contenteditable").toLowerCase() != "false"\n  ) {\n    return element.outerHTML;\n  }\n\n  // Checks parents recursively because the touch might be for example on an <em> inside a <a>.\n  if (element.parentElement) {\n    return nearestInteractiveElement(element.parentElement);\n  }\n\n  return null;\n}\n\n;// CONCATENATED MODULE: ./src/highlight.js\n/* eslint-disable */\n//\n//  highlight.js\n//  r2-navigator-kotlin\n//\n//  Organized by Taehyun Kim on 6/27/19 from r2-navigator-js.\n//\n//  Copyright 2019 Readium Foundation. All rights reserved.\n//  Use of this source code is governed by a BSD-style license which is detailed\n//  in the LICENSE file present in the project repository where this source code is maintained.\n//\n\nconst ROOT_CLASS_REDUCE_MOTION = "r2-reduce-motion";\nconst ROOT_CLASS_NO_FOOTNOTES = "r2-no-popup-foonotes";\nconst POPUP_DIALOG_CLASS = "r2-popup-dialog";\nconst FOOTNOTES_CONTAINER_CLASS = "r2-footnote-container";\nconst FOOTNOTES_CLOSE_BUTTON_CLASS = "r2-footnote-close";\nconst FOOTNOTE_FORCE_SHOW = "r2-footnote-force-show";\nconst TTS_ID_PREVIOUS = "r2-tts-previous";\nconst TTS_ID_NEXT = "r2-tts-next";\nconst TTS_ID_SLIDER = "r2-tts-slider";\nconst TTS_ID_ACTIVE_WORD = "r2-tts-active-word";\nconst TTS_ID_CONTAINER = "r2-tts-txt";\nconst TTS_ID_INFO = "r2-tts-info";\nconst TTS_NAV_BUTTON_CLASS = "r2-tts-button";\nconst TTS_ID_SPEAKING_DOC_ELEMENT = "r2-tts-speaking-el";\nconst TTS_CLASS_INJECTED_SPAN = "r2-tts-speaking-txt";\nconst TTS_CLASS_INJECTED_SUBSPAN = "r2-tts-speaking-word";\nconst TTS_ID_INJECTED_PARENT = "r2-tts-speaking-txt-parent";\nconst ID_HIGHLIGHTS_CONTAINER = "R2_ID_HIGHLIGHTS_CONTAINER";\nconst ID_ANNOTATION_CONTAINER = "R2_ID_ANNOTATION_CONTAINER";\nconst CLASS_HIGHLIGHT_CONTAINER = "R2_CLASS_HIGHLIGHT_CONTAINER";\nconst CLASS_ANNOTATION_CONTAINER = "R2_CLASS_ANNOTATION_CONTAINER";\nconst CLASS_HIGHLIGHT_AREA = "R2_CLASS_HIGHLIGHT_AREA";\nconst CLASS_ANNOTATION_AREA = "R2_CLASS_ANNOTATION_AREA";\nconst CLASS_HIGHLIGHT_BOUNDING_AREA = "R2_CLASS_HIGHLIGHT_BOUNDING_AREA";\nconst CLASS_ANNOTATION_BOUNDING_AREA = "R2_CLASS_ANNOTATION_BOUNDING_AREA";\n// tslint:disable-next-line:max-line-length\nconst _blacklistIdClassForCFI = [\n  POPUP_DIALOG_CLASS,\n  TTS_CLASS_INJECTED_SPAN,\n  TTS_CLASS_INJECTED_SUBSPAN,\n  ID_HIGHLIGHTS_CONTAINER,\n  CLASS_HIGHLIGHT_CONTAINER,\n  CLASS_HIGHLIGHT_AREA,\n  CLASS_HIGHLIGHT_BOUNDING_AREA,\n  "resize-sensor",\n];\nconst CLASS_PAGINATED = "r2-css-paginated";\n\n//const IS_DEV = (process.env.NODE_ENV === "development" || process.env.NODE_ENV === "dev");\nconst IS_DEV = false;\nconst _highlights = [];\n\nlet _highlightsContainer;\nlet _annotationContainer;\nlet lastMouseDownX = -1;\nlet lastMouseDownY = -1;\nlet bodyEventListenersSet = false;\n\nconst USE_SVG = false;\nconst DEFAULT_BACKGROUND_COLOR_OPACITY = 0.3;\nconst ALT_BACKGROUND_COLOR_OPACITY = 0.45;\n\n//const DEBUG_VISUALS = false;\nconst DEBUG_VISUALS = false;\nconst DEFAULT_BACKGROUND_COLOR = {\n  blue: 100,\n  green: 50,\n  red: 230,\n};\n\nconst ANNOTATION_WIDTH = 15;\n\nfunction resetHighlightBoundingStyle(_win, highlightBounding) {\n  if (\n    highlightBounding.getAttribute("class") == CLASS_ANNOTATION_BOUNDING_AREA\n  ) {\n    return;\n  }\n  highlightBounding.style.outline = "none";\n  highlightBounding.style.setProperty(\n    "background-color",\n    "transparent",\n    "important"\n  );\n}\n\nfunction setHighlightAreaStyle(win, highlightAreas, highlight) {\n  const useSVG = !DEBUG_VISUALS && USE_SVG;\n  for (const highlightArea of highlightAreas) {\n    const isSVG = useSVG && highlightArea.namespaceURI === SVG_XML_NAMESPACE;\n    const opacity = ALT_BACKGROUND_COLOR_OPACITY;\n    if (isSVG) {\n      highlightArea.style.setProperty(\n        "fill",\n        `rgb(${highlight.color.red}, ${highlight.color.green}, ${highlight.color.blue})`,\n        "important"\n      );\n      highlightArea.style.setProperty(\n        "fill-opacity",\n        `${opacity}`,\n        "important"\n      );\n      highlightArea.style.setProperty(\n        "stroke",\n        `rgb(${highlight.color.red}, ${highlight.color.green}, ${highlight.color.blue})`,\n        "important"\n      );\n      highlightArea.style.setProperty(\n        "stroke-opacity",\n        `${opacity}`,\n        "important"\n      );\n    } else {\n      highlightArea.style.setProperty(\n        "background-color",\n        `rgba(${highlight.color.red}, ${highlight.color.green}, ${highlight.color.blue}, ${opacity})`,\n        "important"\n      );\n    }\n  }\n}\n\nfunction resetHighlightAreaStyle(win, highlightArea) {\n  const useSVG = !DEBUG_VISUALS && USE_SVG;\n  //const useSVG = USE_SVG;\n  const isSVG = useSVG && highlightArea.namespaceURI === SVG_XML_NAMESPACE;\n  const id = isSVG\n    ? highlightArea.parentNode &&\n      highlightArea.parentNode.parentNode &&\n      highlightArea.parentNode.parentNode.nodeType === Node.ELEMENT_NODE &&\n      highlightArea.parentNode.parentNode.getAttribute\n      ? highlightArea.parentNode.parentNode.getAttribute("id")\n      : undefined\n    : highlightArea.parentNode &&\n      highlightArea.parentNode.nodeType === Node.ELEMENT_NODE &&\n      highlightArea.parentNode.getAttribute\n    ? highlightArea.parentNode.getAttribute("id")\n    : undefined;\n  if (id) {\n    const highlight = _highlights.find((h) => {\n      return h.id === id;\n    });\n    if (highlight) {\n      const opacity = DEFAULT_BACKGROUND_COLOR_OPACITY;\n      if (isSVG) {\n        highlightArea.style.setProperty(\n          "fill",\n          `rgb(${highlight.color.red}, ${highlight.color.green}, ${highlight.color.blue})`,\n          "important"\n        );\n        highlightArea.style.setProperty(\n          "fill-opacity",\n          `${opacity}`,\n          "important"\n        );\n        highlightArea.style.setProperty(\n          "stroke",\n          `rgb(${highlight.color.red}, ${highlight.color.green}, ${highlight.color.blue})`,\n          "important"\n        );\n        highlightArea.style.setProperty(\n          "stroke-opacity",\n          `${opacity}`,\n          "important"\n        );\n      } else {\n        highlightArea.style.setProperty(\n          "background-color",\n          `rgba(${highlight.color.red}, ${highlight.color.green}, ${highlight.color.blue}, ${opacity})`,\n          "important"\n        );\n      }\n    }\n  }\n}\nfunction processTouchEvent(win, ev) {\n  const document = win.document;\n  const scrollElement = getScrollingElement(document);\n  const x = ev.changedTouches[0].clientX;\n  const y = ev.changedTouches[0].clientY;\n  if (!_highlightsContainer) {\n    return;\n  }\n  const paginated = isPaginated(document);\n  const bodyRect = document.body.getBoundingClientRect();\n  let xOffset;\n  let yOffset;\n  if (navigator.userAgent.match(/Android/i)) {\n    xOffset = paginated ? -scrollElement.scrollLeft : bodyRect.left;\n    yOffset = paginated ? -scrollElement.scrollTop : bodyRect.top;\n  } else if (navigator.userAgent.match(/iPhone|iPad|iPod/i)) {\n    xOffset = paginated ? 0 : -scrollElement.scrollLeft;\n    yOffset = paginated ? 0 : bodyRect.top;\n  }\n  let foundHighlight;\n  let foundElement;\n  let foundRect;\n  //    _highlights.sort(function(a, b) {\n  //        console.log(JSON.stringify(a.selectionInfo))\n  //        return a.selectionInfo.cleanText.length < b.selectionInfo.cleanText.length\n  //    })\n  for (let i = _highlights.length - 1; i >= 0; i--) {\n    const highlight = _highlights[i];\n    let highlightParent = document.getElementById(`${highlight.id}`);\n    if (!highlightParent) {\n      highlightParent = _highlightsContainer.querySelector(`#${highlight.id}`);\n    }\n    if (!highlightParent) {\n      continue;\n    }\n    let hit = false;\n    const highlightFragments = highlightParent.querySelectorAll(\n      `.${CLASS_HIGHLIGHT_AREA}`\n    );\n    for (const highlightFragment of highlightFragments) {\n      const withRect = highlightFragment;\n      const left = withRect.rect.left + xOffset;\n      const top = withRect.rect.top + yOffset;\n      foundRect = withRect.rect;\n      if (\n        x >= left &&\n        x < left + withRect.rect.width &&\n        y >= top &&\n        y < top + withRect.rect.height\n      ) {\n        hit = true;\n        break;\n      }\n    }\n    if (hit) {\n      foundHighlight = highlight;\n      foundElement = highlightParent;\n      break;\n    }\n  }\n  if (!foundHighlight || !foundElement) {\n    const highlightBoundings = _highlightsContainer.querySelectorAll(\n      `.${CLASS_HIGHLIGHT_BOUNDING_AREA}`\n    );\n    for (const highlightBounding of highlightBoundings) {\n      resetHighlightBoundingStyle(win, highlightBounding);\n    }\n    const allHighlightAreas = Array.from(\n      _highlightsContainer.querySelectorAll(`.${CLASS_HIGHLIGHT_AREA}`)\n    );\n    for (const highlightArea of allHighlightAreas) {\n      resetHighlightAreaStyle(win, highlightArea);\n    }\n    return;\n  }\n\n  if (foundElement.getAttribute("data-click")) {\n    if (ev.type === "mousemove") {\n      const foundElementHighlightAreas = Array.from(\n        foundElement.querySelectorAll(`.${CLASS_HIGHLIGHT_AREA}`)\n      );\n      const allHighlightAreas = _highlightsContainer.querySelectorAll(\n        `.${CLASS_HIGHLIGHT_AREA}`\n      );\n      for (const highlightArea of allHighlightAreas) {\n        if (foundElementHighlightAreas.indexOf(highlightArea) < 0) {\n          resetHighlightAreaStyle(win, highlightArea);\n        }\n      }\n      setHighlightAreaStyle(win, foundElementHighlightAreas, foundHighlight);\n      const foundElementHighlightBounding = foundElement.querySelector(\n        `.${CLASS_HIGHLIGHT_BOUNDING_AREA}`\n      );\n      const allHighlightBoundings = _highlightsContainer.querySelectorAll(\n        `.${CLASS_HIGHLIGHT_BOUNDING_AREA}`\n      );\n      for (const highlightBounding of allHighlightBoundings) {\n        if (\n          !foundElementHighlightBounding ||\n          highlightBounding !== foundElementHighlightBounding\n        ) {\n          resetHighlightBoundingStyle(win, highlightBounding);\n        }\n      }\n      if (foundElementHighlightBounding) {\n        if (DEBUG_VISUALS) {\n          setHighlightBoundingStyle(\n            win,\n            foundElementHighlightBounding,\n            foundHighlight\n          );\n        }\n      }\n    } else if (ev.type === "touchstart" || ev.type === "touchend") {\n      const size = {\n        screenWidth: window.outerWidth,\n        screenHeight: window.outerHeight,\n        left: foundRect.left,\n        width: foundRect.width,\n        top: foundRect.top,\n        height: foundRect.height,\n      };\n      const payload = {\n        highlight: foundHighlight.id,\n        size: size,\n      };\n\n      if (\n        typeof window !== "undefined" &&\n        typeof window.process === "object" &&\n        window.process.type === "renderer"\n      ) {\n        electron_1.ipcRenderer.sendToHost(R2_EVENT_HIGHLIGHT_CLICK, payload);\n      } else if (window.webkitURL) {\n        console.log(foundHighlight.id.includes("R2_ANNOTATION_"));\n        if (foundHighlight.id.search("R2_ANNOTATION_") >= 0) {\n          if (navigator.userAgent.match(/Android/i)) {\n            Android.highlightAnnotationMarkActivated(foundHighlight.id);\n          } else if (navigator.userAgent.match(/iPhone|iPad|iPod/i)) {\n            webkit.messageHandlers.highlightAnnotationMarkActivated.postMessage(\n              foundHighlight.id\n            );\n          }\n        } else if (foundHighlight.id.search("R2_HIGHLIGHT_") >= 0) {\n          if (navigator.userAgent.match(/Android/i)) {\n            Android.highlightActivated(foundHighlight.id);\n          } else if (navigator.userAgent.match(/iPhone|iPad|iPod/i)) {\n            webkit.messageHandlers.highlightActivated.postMessage(\n              foundHighlight.id\n            );\n          }\n        }\n      }\n\n      ev.stopPropagation();\n      ev.preventDefault();\n    }\n  }\n}\n\nfunction processMouseEvent(win, ev) {\n  const document = win.document;\n  const scrollElement = getScrollingElement(document);\n  const x = ev.clientX;\n  const y = ev.clientY;\n  if (!_highlightsContainer) {\n    return;\n  }\n\n  const paginated = isPaginated(document);\n  const bodyRect = document.body.getBoundingClientRect();\n  let xOffset;\n  let yOffset;\n  if (navigator.userAgent.match(/Android/i)) {\n    xOffset = paginated ? -scrollElement.scrollLeft : bodyRect.left;\n    yOffset = paginated ? -scrollElement.scrollTop : bodyRect.top;\n  } else if (navigator.userAgent.match(/iPhone|iPad|iPod/i)) {\n    xOffset = paginated ? 0 : -scrollElement.scrollLeft;\n    yOffset = paginated ? 0 : bodyRect.top;\n  }\n  let foundHighlight;\n  let foundElement;\n  let foundRect;\n  for (let i = _highlights.length - 1; i >= 0; i--) {\n    const highlight = _highlights[i];\n    let highlightParent = document.getElementById(`${highlight.id}`);\n    if (!highlightParent) {\n      highlightParent = _highlightsContainer.querySelector(`#${highlight.id}`);\n    }\n    if (!highlightParent) {\n      continue;\n    }\n    let hit = false;\n    const highlightFragments = highlightParent.querySelectorAll(\n      `.${CLASS_HIGHLIGHT_AREA}`\n    );\n    for (const highlightFragment of highlightFragments) {\n      const withRect = highlightFragment;\n      const left = withRect.rect.left + xOffset;\n      const top = withRect.rect.top + yOffset;\n      foundRect = withRect.rect;\n      if (\n        x >= left &&\n        x < left + withRect.rect.width &&\n        y >= top &&\n        y < top + withRect.rect.height\n      ) {\n        hit = true;\n        break;\n      }\n    }\n    if (hit) {\n      foundHighlight = highlight;\n      foundElement = highlightParent;\n      break;\n    }\n  }\n\n  if (!foundHighlight || !foundElement) {\n    const highlightBoundings = _highlightsContainer.querySelectorAll(\n      `.${CLASS_HIGHLIGHT_BOUNDING_AREA}`\n    );\n    for (const highlightBounding of highlightBoundings) {\n      resetHighlightBoundingStyle(win, highlightBounding);\n    }\n    const allHighlightAreas = Array.from(\n      _highlightsContainer.querySelectorAll(`.${CLASS_HIGHLIGHT_AREA}`)\n    );\n    for (const highlightArea of allHighlightAreas) {\n      resetHighlightAreaStyle(win, highlightArea);\n    }\n    return;\n  }\n\n  if (foundElement.getAttribute("data-click")) {\n    if (ev.type === "mousemove") {\n      const foundElementHighlightAreas = Array.from(\n        foundElement.querySelectorAll(`.${CLASS_HIGHLIGHT_AREA}`)\n      );\n      const allHighlightAreas = _highlightsContainer.querySelectorAll(\n        `.${CLASS_HIGHLIGHT_AREA}`\n      );\n      for (const highlightArea of allHighlightAreas) {\n        if (foundElementHighlightAreas.indexOf(highlightArea) < 0) {\n          resetHighlightAreaStyle(win, highlightArea);\n        }\n      }\n      setHighlightAreaStyle(win, foundElementHighlightAreas, foundHighlight);\n      const foundElementHighlightBounding = foundElement.querySelector(\n        `.${CLASS_HIGHLIGHT_BOUNDING_AREA}`\n      );\n      const allHighlightBoundings = _highlightsContainer.querySelectorAll(\n        `.${CLASS_HIGHLIGHT_BOUNDING_AREA}`\n      );\n      for (const highlightBounding of allHighlightBoundings) {\n        if (\n          !foundElementHighlightBounding ||\n          highlightBounding !== foundElementHighlightBounding\n        ) {\n          resetHighlightBoundingStyle(win, highlightBounding);\n        }\n      }\n      if (foundElementHighlightBounding) {\n        if (DEBUG_VISUALS) {\n          setHighlightBoundingStyle(\n            win,\n            foundElementHighlightBounding,\n            foundHighlight\n          );\n        }\n      }\n    } else if (ev.type === "mouseup" || ev.type === "touchend") {\n      const touchedPosition = {\n        screenWidth: window.outerWidth,\n        screenHeight: window.innerHeight,\n        left: foundRect.left,\n        width: foundRect.width,\n        top: foundRect.top,\n        height: foundRect.height,\n      };\n\n      const payload = {\n        highlight: foundHighlight,\n        position: touchedPosition,\n      };\n\n      if (\n        typeof window !== "undefined" &&\n        typeof window.process === "object" &&\n        window.process.type === "renderer"\n      ) {\n        electron_1.ipcRenderer.sendToHost(R2_EVENT_HIGHLIGHT_CLICK, payload);\n      } else if (window.webkitURL) {\n        if (foundHighlight.id.search("R2_ANNOTATION_") >= 0) {\n          if (navigator.userAgent.match(/Android/i)) {\n            Android.highlightAnnotationMarkActivated(foundHighlight.id);\n          } else if (navigator.userAgent.match(/iPhone|iPad|iPod/i)) {\n            webkit.messageHandlers.highlightAnnotationMarkActivated.postMessage(\n              foundHighlight.id\n            );\n          }\n        } else if (foundHighlight.id.search("R2_HIGHLIGHT_") >= 0) {\n          if (navigator.userAgent.match(/Android/i)) {\n            Android.highlightActivated(foundHighlight.id);\n          } else if (navigator.userAgent.match(/iPhone|iPad|iPod/i)) {\n            webkit.messageHandlers.highlightActivated.postMessage(\n              foundHighlight.id\n            );\n          }\n        }\n      }\n\n      ev.stopPropagation();\n    }\n  }\n}\n\nfunction highlight_rectsTouchOrOverlap(rect1, rect2, tolerance) {\n  return (\n    (rect1.left < rect2.right ||\n      (tolerance >= 0 && highlight_almostEqual(rect1.left, rect2.right, tolerance))) &&\n    (rect2.left < rect1.right ||\n      (tolerance >= 0 && highlight_almostEqual(rect2.left, rect1.right, tolerance))) &&\n    (rect1.top < rect2.bottom ||\n      (tolerance >= 0 && highlight_almostEqual(rect1.top, rect2.bottom, tolerance))) &&\n    (rect2.top < rect1.bottom ||\n      (tolerance >= 0 && highlight_almostEqual(rect2.top, rect1.bottom, tolerance)))\n  );\n}\n\nfunction highlight_replaceOverlapingRects(rects) {\n  for (let i = 0; i < rects.length; i++) {\n    for (let j = i + 1; j < rects.length; j++) {\n      const rect1 = rects[i];\n      const rect2 = rects[j];\n      if (rect1 === rect2) {\n        if (IS_DEV) {\n          console.log("replaceOverlapingRects rect1 === rect2 ??!");\n        }\n        continue;\n      }\n      if (highlight_rectsTouchOrOverlap(rect1, rect2, -1)) {\n        let toAdd = [];\n        let toRemove;\n        let toPreserve;\n        const subtractRects1 = highlight_rectSubtract(rect1, rect2);\n        if (subtractRects1.length === 1) {\n          toAdd = subtractRects1;\n          toRemove = rect1;\n          toPreserve = rect2;\n        } else {\n          const subtractRects2 = highlight_rectSubtract(rect2, rect1);\n          if (subtractRects1.length < subtractRects2.length) {\n            toAdd = subtractRects1;\n            toRemove = rect1;\n            toPreserve = rect2;\n          } else {\n            toAdd = subtractRects2;\n            toRemove = rect2;\n            toPreserve = rect1;\n          }\n        }\n        if (IS_DEV) {\n          const toCheck = [];\n          toCheck.push(toPreserve);\n          Array.prototype.push.apply(toCheck, toAdd);\n          checkOverlaps(toCheck);\n        }\n        if (IS_DEV) {\n          console.log(\n            `CLIENT RECT: overlap, cut one rect into ${toAdd.length}`\n          );\n        }\n        const newRects = rects.filter((rect) => {\n          return rect !== toRemove;\n        });\n        Array.prototype.push.apply(newRects, toAdd);\n        return highlight_replaceOverlapingRects(newRects);\n      }\n    }\n  }\n  return rects;\n}\n\nfunction checkOverlaps(rects) {\n  const stillOverlapingRects = [];\n  for (const rect1 of rects) {\n    for (const rect2 of rects) {\n      if (rect1 === rect2) {\n        continue;\n      }\n      const has1 = stillOverlapingRects.indexOf(rect1) >= 0;\n      const has2 = stillOverlapingRects.indexOf(rect2) >= 0;\n      if (!has1 || !has2) {\n        if (highlight_rectsTouchOrOverlap(rect1, rect2, -1)) {\n          if (!has1) {\n            stillOverlapingRects.push(rect1);\n          }\n          if (!has2) {\n            stillOverlapingRects.push(rect2);\n          }\n          console.log("CLIENT RECT: overlap ---");\n          console.log(\n            `#1 TOP:${rect1.top} BOTTOM:${rect1.bottom} LEFT:${rect1.left} RIGHT:${rect1.right} WIDTH:${rect1.width} HEIGHT:${rect1.height}`\n          );\n          console.log(\n            `#2 TOP:${rect2.top} BOTTOM:${rect2.bottom} LEFT:${rect2.left} RIGHT:${rect2.right} WIDTH:${rect2.width} HEIGHT:${rect2.height}`\n          );\n          const xOverlap = getRectOverlapX(rect1, rect2);\n          console.log(`xOverlap: ${xOverlap}`);\n          const yOverlap = getRectOverlapY(rect1, rect2);\n          console.log(`yOverlap: ${yOverlap}`);\n        }\n      }\n    }\n  }\n  if (stillOverlapingRects.length) {\n    console.log(`CLIENT RECT: overlaps ${stillOverlapingRects.length}`);\n  }\n}\n\nfunction highlight_removeContainedRects(rects, tolerance) {\n  const rectsToKeep = new Set(rects);\n  for (const rect of rects) {\n    const bigEnough = rect.width > 1 && rect.height > 1;\n    if (!bigEnough) {\n      if (IS_DEV) {\n        console.log("CLIENT RECT: remove tiny");\n      }\n      rectsToKeep.delete(rect);\n      continue;\n    }\n    for (const possiblyContainingRect of rects) {\n      if (rect === possiblyContainingRect) {\n        continue;\n      }\n      if (!rectsToKeep.has(possiblyContainingRect)) {\n        continue;\n      }\n      if (highlight_rectContains(possiblyContainingRect, rect, tolerance)) {\n        if (IS_DEV) {\n          console.log("CLIENT RECT: remove contained");\n        }\n        rectsToKeep.delete(rect);\n        break;\n      }\n    }\n  }\n  return Array.from(rectsToKeep);\n}\n\nfunction highlight_almostEqual(a, b, tolerance) {\n  return Math.abs(a - b) <= tolerance;\n}\n\nfunction highlight_rectIntersect(rect1, rect2) {\n  const maxLeft = Math.max(rect1.left, rect2.left);\n  const minRight = Math.min(rect1.right, rect2.right);\n  const maxTop = Math.max(rect1.top, rect2.top);\n  const minBottom = Math.min(rect1.bottom, rect2.bottom);\n  const rect = {\n    bottom: minBottom,\n    height: Math.max(0, minBottom - maxTop),\n    left: maxLeft,\n    right: minRight,\n    top: maxTop,\n    width: Math.max(0, minRight - maxLeft),\n  };\n  return rect;\n}\n\nfunction highlight_rectSubtract(rect1, rect2) {\n  const rectIntersected = highlight_rectIntersect(rect2, rect1);\n  if (rectIntersected.height === 0 || rectIntersected.width === 0) {\n    return [rect1];\n  }\n  const rects = [];\n  {\n    const rectA = {\n      bottom: rect1.bottom,\n      height: 0,\n      left: rect1.left,\n      right: rectIntersected.left,\n      top: rect1.top,\n      width: 0,\n    };\n    rectA.width = rectA.right - rectA.left;\n    rectA.height = rectA.bottom - rectA.top;\n    if (rectA.height !== 0 && rectA.width !== 0) {\n      rects.push(rectA);\n    }\n  }\n  {\n    const rectB = {\n      bottom: rectIntersected.top,\n      height: 0,\n      left: rectIntersected.left,\n      right: rectIntersected.right,\n      top: rect1.top,\n      width: 0,\n    };\n    rectB.width = rectB.right - rectB.left;\n    rectB.height = rectB.bottom - rectB.top;\n    if (rectB.height !== 0 && rectB.width !== 0) {\n      rects.push(rectB);\n    }\n  }\n  {\n    const rectC = {\n      bottom: rect1.bottom,\n      height: 0,\n      left: rectIntersected.left,\n      right: rectIntersected.right,\n      top: rectIntersected.bottom,\n      width: 0,\n    };\n    rectC.width = rectC.right - rectC.left;\n    rectC.height = rectC.bottom - rectC.top;\n    if (rectC.height !== 0 && rectC.width !== 0) {\n      rects.push(rectC);\n    }\n  }\n  {\n    const rectD = {\n      bottom: rect1.bottom,\n      height: 0,\n      left: rectIntersected.right,\n      right: rect1.right,\n      top: rect1.top,\n      width: 0,\n    };\n    rectD.width = rectD.right - rectD.left;\n    rectD.height = rectD.bottom - rectD.top;\n    if (rectD.height !== 0 && rectD.width !== 0) {\n      rects.push(rectD);\n    }\n  }\n  return rects;\n}\n\nfunction highlight_rectContainsPoint(rect, x, y, tolerance) {\n  return (\n    (rect.left < x || highlight_almostEqual(rect.left, x, tolerance)) &&\n    (rect.right > x || highlight_almostEqual(rect.right, x, tolerance)) &&\n    (rect.top < y || highlight_almostEqual(rect.top, y, tolerance)) &&\n    (rect.bottom > y || highlight_almostEqual(rect.bottom, y, tolerance))\n  );\n}\n\nfunction highlight_rectContains(rect1, rect2, tolerance) {\n  return (\n    highlight_rectContainsPoint(rect1, rect2.left, rect2.top, tolerance) &&\n    highlight_rectContainsPoint(rect1, rect2.right, rect2.top, tolerance) &&\n    highlight_rectContainsPoint(rect1, rect2.left, rect2.bottom, tolerance) &&\n    highlight_rectContainsPoint(rect1, rect2.right, rect2.bottom, tolerance)\n  );\n}\n\nfunction highlight_getBoundingRect(rect1, rect2) {\n  const left = Math.min(rect1.left, rect2.left);\n  const right = Math.max(rect1.right, rect2.right);\n  const top = Math.min(rect1.top, rect2.top);\n  const bottom = Math.max(rect1.bottom, rect2.bottom);\n  return {\n    bottom,\n    height: bottom - top,\n    left,\n    right,\n    top,\n    width: right - left,\n  };\n}\n\nfunction highlight_mergeTouchingRects(\n  rects,\n  tolerance,\n  doNotMergeHorizontallyAlignedRects\n) {\n  for (let i = 0; i < rects.length; i++) {\n    for (let j = i + 1; j < rects.length; j++) {\n      const rect1 = rects[i];\n      const rect2 = rects[j];\n      if (rect1 === rect2) {\n        if (IS_DEV) {\n          console.log("mergeTouchingRects rect1 === rect2 ??!");\n        }\n        continue;\n      }\n      const rectsLineUpVertically =\n        highlight_almostEqual(rect1.top, rect2.top, tolerance) &&\n        highlight_almostEqual(rect1.bottom, rect2.bottom, tolerance);\n      const rectsLineUpHorizontally =\n        highlight_almostEqual(rect1.left, rect2.left, tolerance) &&\n        highlight_almostEqual(rect1.right, rect2.right, tolerance);\n      const horizontalAllowed = !doNotMergeHorizontallyAlignedRects;\n      const aligned =\n        (rectsLineUpHorizontally && horizontalAllowed) ||\n        (rectsLineUpVertically && !rectsLineUpHorizontally);\n      const canMerge = aligned && highlight_rectsTouchOrOverlap(rect1, rect2, tolerance);\n      if (canMerge) {\n        if (IS_DEV) {\n          console.log(\n            `CLIENT RECT: merging two into one, VERTICAL: ${rectsLineUpVertically} HORIZONTAL: ${rectsLineUpHorizontally} (${doNotMergeHorizontallyAlignedRects})`\n          );\n        }\n        const newRects = rects.filter((rect) => {\n          return rect !== rect1 && rect !== rect2;\n        });\n        const replacementClientRect = highlight_getBoundingRect(rect1, rect2);\n        newRects.push(replacementClientRect);\n        return highlight_mergeTouchingRects(\n          newRects,\n          tolerance,\n          doNotMergeHorizontallyAlignedRects\n        );\n      }\n    }\n  }\n  return rects;\n}\n\nfunction highlight_getClientRectsNoOverlap(range, doNotMergeHorizontallyAlignedRects) {\n  const rangeClientRects = range.getClientRects();\n  return getClientRectsNoOverlap_(\n    rangeClientRects,\n    doNotMergeHorizontallyAlignedRects\n  );\n}\n\nfunction getClientRectsNoOverlap_(\n  clientRects,\n  doNotMergeHorizontallyAlignedRects\n) {\n  const tolerance = 1;\n  const originalRects = [];\n  for (const rangeClientRect of clientRects) {\n    originalRects.push({\n      bottom: rangeClientRect.bottom,\n      height: rangeClientRect.height,\n      left: rangeClientRect.left,\n      right: rangeClientRect.right,\n      top: rangeClientRect.top,\n      width: rangeClientRect.width,\n    });\n  }\n  const mergedRects = highlight_mergeTouchingRects(\n    originalRects,\n    tolerance,\n    doNotMergeHorizontallyAlignedRects\n  );\n  const noContainedRects = highlight_removeContainedRects(mergedRects, tolerance);\n  const newRects = highlight_replaceOverlapingRects(noContainedRects);\n  const minArea = 2 * 2;\n  for (let j = newRects.length - 1; j >= 0; j--) {\n    const rect = newRects[j];\n    const bigEnough = rect.width * rect.height > minArea;\n    if (!bigEnough) {\n      if (newRects.length > 1) {\n        if (IS_DEV) {\n          console.log("CLIENT RECT: remove small");\n        }\n        newRects.splice(j, 1);\n      } else {\n        if (IS_DEV) {\n          console.log("CLIENT RECT: remove small, but keep otherwise empty!");\n        }\n        break;\n      }\n    }\n  }\n  if (IS_DEV) {\n    checkOverlaps(newRects);\n  }\n  if (IS_DEV) {\n    console.log(\n      `CLIENT RECT: reduced ${originalRects.length} --\x3e ${newRects.length}`\n    );\n  }\n  return newRects;\n}\n\nfunction isPaginated(document) {\n  return (\n    document &&\n    document.documentElement &&\n    document.documentElement.classList.contains(CLASS_PAGINATED)\n  );\n}\n\nfunction getScrollingElement(document) {\n  if (document.scrollingElement) {\n    return document.scrollingElement;\n  }\n  return document.body;\n}\n\nfunction ensureContainer(win, annotationFlag) {\n  const document = win.document;\n\n  if (!_highlightsContainer) {\n    if (!bodyEventListenersSet) {\n      bodyEventListenersSet = true;\n      document.body.addEventListener(\n        "mousedown",\n        (ev) => {\n          lastMouseDownX = ev.clientX;\n          lastMouseDownY = ev.clientY;\n        },\n        false\n      );\n      document.body.addEventListener(\n        "mouseup",\n        (ev) => {\n          if (\n            Math.abs(lastMouseDownX - ev.clientX) < 3 &&\n            Math.abs(lastMouseDownY - ev.clientY) < 3\n          ) {\n            processMouseEvent(win, ev);\n          }\n        },\n        false\n      );\n      document.body.addEventListener(\n        "mousemove",\n        (ev) => {\n          processMouseEvent(win, ev);\n        },\n        false\n      );\n\n      document.body.addEventListener(\n        "touchend",\n        function touchEnd(e) {\n          processTouchEvent(win, e);\n        },\n        false\n      );\n    }\n    _highlightsContainer = document.createElement("div");\n    _highlightsContainer.setAttribute("id", ID_HIGHLIGHTS_CONTAINER);\n\n    _highlightsContainer.style.setProperty("pointer-events", "none");\n    document.body.append(_highlightsContainer);\n  }\n\n  return _highlightsContainer;\n}\n\nfunction hideAllhighlights() {\n  if (_highlightsContainer) {\n    _highlightsContainer.remove();\n    _highlightsContainer = null;\n  }\n}\n\nfunction destroyAllhighlights() {\n  hideAllhighlights();\n  _highlights.splice(0, _highlights.length);\n}\n\nfunction destroyHighlight(id) {\n  let i = -1;\n  let _document = window.document;\n  const highlight = _highlights.find((h, j) => {\n    i = j;\n    return h.id === id;\n  });\n  if (highlight && i >= 0 && i < _highlights.length) {\n    _highlights.splice(i, 1);\n  }\n  const highlightContainer = _document.getElementById(id);\n  if (highlightContainer) {\n    highlightContainer.remove();\n  }\n}\n\nfunction isCfiTextNode(node) {\n  return node.nodeType !== Node.ELEMENT_NODE;\n}\n\nfunction getChildTextNodeCfiIndex(element, child) {\n  let found = -1;\n  let textNodeIndex = -1;\n  let previousWasElement = false;\n  for (let i = 0; i < element.childNodes.length; i++) {\n    const childNode = element.childNodes[i];\n    const isText = isCfiTextNode(childNode);\n    if (isText || previousWasElement) {\n      textNodeIndex += 2;\n    }\n    if (isText) {\n      if (childNode === child) {\n        found = textNodeIndex;\n        break;\n      }\n    }\n    previousWasElement = childNode.nodeType === Node.ELEMENT_NODE;\n  }\n  return found;\n}\n\nfunction getCommonAncestorElement(node1, node2) {\n  if (node1.nodeType === Node.ELEMENT_NODE && node1 === node2) {\n    return node1;\n  }\n  if (node1.nodeType === Node.ELEMENT_NODE && node1.contains(node2)) {\n    return node1;\n  }\n  if (node2.nodeType === Node.ELEMENT_NODE && node2.contains(node1)) {\n    return node2;\n  }\n  const node1ElementAncestorChain = [];\n  let parent = node1.parentNode;\n  while (parent && parent.nodeType === Node.ELEMENT_NODE) {\n    node1ElementAncestorChain.push(parent);\n    parent = parent.parentNode;\n  }\n  const node2ElementAncestorChain = [];\n  parent = node2.parentNode;\n  while (parent && parent.nodeType === Node.ELEMENT_NODE) {\n    node2ElementAncestorChain.push(parent);\n    parent = parent.parentNode;\n  }\n  let commonAncestor = node1ElementAncestorChain.find(\n    (node1ElementAncestor) => {\n      return node2ElementAncestorChain.indexOf(node1ElementAncestor) >= 0;\n    }\n  );\n  if (!commonAncestor) {\n    commonAncestor = node2ElementAncestorChain.find((node2ElementAncestor) => {\n      return node1ElementAncestorChain.indexOf(node2ElementAncestor) >= 0;\n    });\n  }\n  return commonAncestor;\n}\n\nfunction fullQualifiedSelector(node) {\n  if (node.nodeType !== Node.ELEMENT_NODE) {\n    const lowerCaseName =\n      (node.localName && node.localName.toLowerCase()) ||\n      node.nodeName.toLowerCase();\n    return lowerCaseName;\n  }\n  //return cssPath(node, justSelector);\n  return cssPath(node, true);\n}\n\nfunction getCurrentSelectionInfo() {\n  const selection = window.getSelection();\n  if (!selection) {\n    return undefined;\n  }\n  if (selection.isCollapsed) {\n    console.log("^^^ SELECTION COLLAPSED.");\n    return undefined;\n  }\n  const rawText = selection.toString();\n  const cleanText = rawText.trim().replace(/\\n/g, " ").replace(/\\s\\s+/g, " ");\n  if (cleanText.length === 0) {\n    console.log("^^^ SELECTION TEXT EMPTY.");\n    return undefined;\n  }\n  if (!selection.anchorNode || !selection.focusNode) {\n    return undefined;\n  }\n  const range =\n    selection.rangeCount === 1\n      ? selection.getRangeAt(0)\n      : createOrderedRange(\n          selection.anchorNode,\n          selection.anchorOffset,\n          selection.focusNode,\n          selection.focusOffset\n        );\n  if (!range || range.collapsed) {\n    console.log("$$$$$$$$$$$$$$$$$ CANNOT GET NON-COLLAPSED SELECTION RANGE?!");\n    return undefined;\n  }\n  const rangeInfo = convertRange(range, fullQualifiedSelector, computeCFI);\n  if (!rangeInfo) {\n    console.log("^^^ SELECTION RANGE INFO FAIL?!");\n    return undefined;\n  }\n\n  if (IS_DEV && DEBUG_VISUALS) {\n    const restoredRange = convertRangeInfo(win.document, rangeInfo);\n    if (restoredRange) {\n      if (\n        restoredRange.startOffset === range.startOffset &&\n        restoredRange.endOffset === range.endOffset &&\n        restoredRange.startContainer === range.startContainer &&\n        restoredRange.endContainer === range.endContainer\n      ) {\n        console.log("SELECTION RANGE RESTORED OKAY (dev check).");\n      } else {\n        console.log("SELECTION RANGE RESTORE FAIL (dev check).");\n        dumpDebug(\n          "SELECTION",\n          selection.anchorNode,\n          selection.anchorOffset,\n          selection.focusNode,\n          selection.focusOffset,\n          getCssSelector\n        );\n        dumpDebug(\n          "ORDERED RANGE FROM SELECTION",\n          range.startContainer,\n          range.startOffset,\n          range.endContainer,\n          range.endOffset,\n          getCssSelector\n        );\n        dumpDebug(\n          "RESTORED RANGE",\n          restoredRange.startContainer,\n          restoredRange.startOffset,\n          restoredRange.endContainer,\n          restoredRange.endOffset,\n          getCssSelector\n        );\n      }\n    } else {\n      console.log("CANNOT RESTORE SELECTION RANGE ??!");\n    }\n  } else {\n  }\n\n  return {\n    locations: rangeInfo2Location(rangeInfo),\n    text: {\n      highlight: rawText,\n    },\n  };\n}\n\nfunction checkBlacklisted(el) {\n  let blacklistedId;\n  const id = el.getAttribute("id");\n  if (id && _blacklistIdClassForCFI.indexOf(id) >= 0) {\n    console.log("checkBlacklisted ID: " + id);\n    blacklistedId = id;\n  }\n  let blacklistedClass;\n  for (const item of _blacklistIdClassForCFI) {\n    if (el.classList.contains(item)) {\n      console.log("checkBlacklisted CLASS: " + item);\n      blacklistedClass = item;\n      break;\n    }\n  }\n  if (blacklistedId || blacklistedClass) {\n    return true;\n  }\n\n  return false;\n}\n\nfunction cssPath(node, optimized) {\n  if (node.nodeType !== Node.ELEMENT_NODE) {\n    return "";\n  }\n\n  const steps = [];\n  let contextNode = node;\n  while (contextNode) {\n    const step = _cssPathStep(contextNode, !!optimized, contextNode === node);\n    if (!step) {\n      break; // Error - bail out early.\n    }\n    steps.push(step.value);\n    if (step.optimized) {\n      break;\n    }\n    contextNode = contextNode.parentNode;\n  }\n  steps.reverse();\n  return steps.join(" > ");\n}\n// tslint:disable-next-line:max-line-length\n// https://chromium.googlesource.com/chromium/blink/+/master/Source/devtools/front_end/components/DOMPresentationUtils.js#316\nfunction _cssPathStep(node, optimized, isTargetNode) {\n  function prefixedElementClassNames(nd) {\n    const classAttribute = nd.getAttribute("class");\n    if (!classAttribute) {\n      return [];\n    }\n\n    return classAttribute\n      .split(/\\s+/g)\n      .filter(Boolean)\n      .map((nm) => {\n        // The prefix is required to store "__proto__" in a object-based map.\n        return "$" + nm;\n      });\n  }\n\n  function idSelector(idd) {\n    return "#" + escapeIdentifierIfNeeded(idd);\n  }\n\n  function escapeIdentifierIfNeeded(ident) {\n    if (isCSSIdentifier(ident)) {\n      return ident;\n    }\n\n    const shouldEscapeFirst = /^(?:[0-9]|-[0-9-]?)/.test(ident);\n    const lastIndex = ident.length - 1;\n    return ident.replace(/./g, function (c, ii) {\n      return (shouldEscapeFirst && ii === 0) || !isCSSIdentChar(c)\n        ? escapeAsciiChar(c, ii === lastIndex)\n        : c;\n    });\n  }\n\n  function escapeAsciiChar(c, isLast) {\n    return "\\\\" + toHexByte(c) + (isLast ? "" : " ");\n  }\n\n  function toHexByte(c) {\n    let hexByte = c.charCodeAt(0).toString(16);\n    if (hexByte.length === 1) {\n      hexByte = "0" + hexByte;\n    }\n    return hexByte;\n  }\n\n  function isCSSIdentChar(c) {\n    if (/[a-zA-Z0-9_-]/.test(c)) {\n      return true;\n    }\n    return c.charCodeAt(0) >= 0xa0;\n  }\n\n  function isCSSIdentifier(value) {\n    return /^-?[a-zA-Z_][a-zA-Z0-9_-]*$/.test(value);\n  }\n\n  if (node.nodeType !== Node.ELEMENT_NODE) {\n    return undefined;\n  }\n  const lowerCaseName =\n    (node.localName && node.localName.toLowerCase()) ||\n    node.nodeName.toLowerCase();\n\n  const element = node;\n\n  const id = element.getAttribute("id");\n\n  if (optimized) {\n    if (id) {\n      return {\n        optimized: true,\n        value: idSelector(id),\n      };\n    }\n    if (\n      lowerCaseName === "body" ||\n      lowerCaseName === "head" ||\n      lowerCaseName === "html"\n    ) {\n      return {\n        optimized: true,\n        value: lowerCaseName, // node.nodeNameInCorrectCase(),\n      };\n    }\n  }\n\n  const nodeName = lowerCaseName; // node.nodeNameInCorrectCase();\n  if (id) {\n    return {\n      optimized: true,\n      value: nodeName + idSelector(id),\n    };\n  }\n\n  const parent = node.parentNode;\n\n  if (!parent || parent.nodeType === Node.DOCUMENT_NODE) {\n    return {\n      optimized: true,\n      value: nodeName,\n    };\n  }\n\n  const prefixedOwnClassNamesArray_ = prefixedElementClassNames(element);\n\n  const prefixedOwnClassNamesArray = []; // .keySet()\n  prefixedOwnClassNamesArray_.forEach((arrItem) => {\n    if (prefixedOwnClassNamesArray.indexOf(arrItem) < 0) {\n      prefixedOwnClassNamesArray.push(arrItem);\n    }\n  });\n\n  let needsClassNames = false;\n  let needsNthChild = false;\n  let ownIndex = -1;\n  let elementIndex = -1;\n  const siblings = parent.children;\n\n  for (\n    let i = 0;\n    (ownIndex === -1 || !needsNthChild) && i < siblings.length;\n    ++i\n  ) {\n    const sibling = siblings[i];\n    if (sibling.nodeType !== Node.ELEMENT_NODE) {\n      continue;\n    }\n    elementIndex += 1;\n    if (sibling === node) {\n      ownIndex = elementIndex;\n      continue;\n    }\n    if (needsNthChild) {\n      continue;\n    }\n\n    // sibling.nodeNameInCorrectCase()\n    const siblingName =\n      (sibling.localName && sibling.localName.toLowerCase()) ||\n      sibling.nodeName.toLowerCase();\n    if (siblingName !== nodeName) {\n      continue;\n    }\n    needsClassNames = true;\n\n    const ownClassNames = [];\n    prefixedOwnClassNamesArray.forEach((arrItem) => {\n      ownClassNames.push(arrItem);\n    });\n    let ownClassNameCount = ownClassNames.length;\n\n    if (ownClassNameCount === 0) {\n      needsNthChild = true;\n      continue;\n    }\n    const siblingClassNamesArray_ = prefixedElementClassNames(sibling);\n    const siblingClassNamesArray = []; // .keySet()\n    siblingClassNamesArray_.forEach((arrItem) => {\n      if (siblingClassNamesArray.indexOf(arrItem) < 0) {\n        siblingClassNamesArray.push(arrItem);\n      }\n    });\n\n    for (const siblingClass of siblingClassNamesArray) {\n      const ind = ownClassNames.indexOf(siblingClass);\n      if (ind < 0) {\n        continue;\n      }\n\n      ownClassNames.splice(ind, 1); // delete ownClassNames[siblingClass];\n\n      if (!--ownClassNameCount) {\n        needsNthChild = true;\n        break;\n      }\n    }\n  }\n\n  let result = nodeName;\n  if (\n    isTargetNode &&\n    nodeName === "input" &&\n    element.getAttribute("type") &&\n    !element.getAttribute("id") &&\n    !element.getAttribute("class")\n  ) {\n    result += \'[type="\' + element.getAttribute("type") + \'"]\';\n  }\n  if (needsNthChild) {\n    result += ":nth-child(" + (ownIndex + 1) + ")";\n  } else if (needsClassNames) {\n    for (const prefixedName of prefixedOwnClassNamesArray) {\n      result += "." + escapeIdentifierIfNeeded(prefixedName.substr(1));\n    }\n  }\n\n  return {\n    optimized: false,\n    value: result,\n  };\n}\n\nfunction computeCFI(node) {\n  // TODO: handle character position inside text node\n  if (node.nodeType !== Node.ELEMENT_NODE) {\n    return undefined;\n  }\n\n  let cfi = "";\n\n  let currentElement = node;\n  while (\n    currentElement.parentNode &&\n    currentElement.parentNode.nodeType === Node.ELEMENT_NODE\n  ) {\n    const blacklisted = checkBlacklisted(currentElement);\n    if (!blacklisted) {\n      const currentElementParentChildren = currentElement.parentNode.children;\n      let currentElementIndex = -1;\n      for (let i = 0; i < currentElementParentChildren.length; i++) {\n        if (currentElement === currentElementParentChildren[i]) {\n          currentElementIndex = i;\n          break;\n        }\n      }\n      if (currentElementIndex >= 0) {\n        const cfiIndex = (currentElementIndex + 1) * 2;\n        cfi =\n          cfiIndex +\n          (currentElement.id ? "[" + currentElement.id + "]" : "") +\n          (cfi.length ? "/" + cfi : "");\n      }\n    }\n    currentElement = currentElement.parentNode;\n  }\n\n  return "/" + cfi;\n}\n\nfunction _createHighlight(locations, color, pointerInteraction, type) {\n  const rangeInfo = location2RangeInfo(locations);\n  const uniqueStr = `${rangeInfo.cfi}${rangeInfo.startContainerElementCssSelector}${rangeInfo.startContainerChildTextNodeIndex}${rangeInfo.startOffset}${rangeInfo.endContainerElementCssSelector}${rangeInfo.endContainerChildTextNodeIndex}${rangeInfo.endOffset}`;\n\n  const hash = __webpack_require__(3715);\n  const sha256Hex = hash.sha256().update(uniqueStr).digest("hex");\n\n  var id;\n  if (type == ID_HIGHLIGHTS_CONTAINER) {\n    id = "R2_HIGHLIGHT_" + sha256Hex;\n  } else {\n    id = "R2_ANNOTATION_" + sha256Hex;\n  }\n\n  destroyHighlight(id);\n\n  const highlight = {\n    color: color ? color : DEFAULT_BACKGROUND_COLOR,\n    id,\n    pointerInteraction,\n    rangeInfo,\n  };\n  _highlights.push(highlight);\n  createHighlightDom(\n    window,\n    highlight,\n    type == ID_ANNOTATION_CONTAINER ? true : false\n  );\n\n  return highlight;\n}\n\nfunction createHighlight(selectionInfo, color, pointerInteraction) {\n  return _createHighlight(\n    selectionInfo,\n    color,\n    pointerInteraction,\n    ID_HIGHLIGHTS_CONTAINER\n  );\n}\n\nfunction createAnnotation(id) {\n  let i = -1;\n\n  const highlight = _highlights.find((h, j) => {\n    i = j;\n    return h.id === id;\n  });\n  if (i == _highlights.length) return;\n\n  var locations = {\n    locations: rangeInfo2Location(highlight.rangeInfo),\n  };\n\n  return _createHighlight(\n    locations,\n    highlight.color,\n    true,\n    ID_ANNOTATION_CONTAINER\n  );\n}\n\nfunction createHighlightDom(win, highlight, annotationFlag) {\n  const document = win.document;\n\n  const scale =\n    1 /\n    (win.READIUM2 && win.READIUM2.isFixedLayout\n      ? win.READIUM2.fxlViewportScale\n      : 1);\n\n  const scrollElement = getScrollingElement(document);\n\n  const range = convertRangeInfo(document, highlight.rangeInfo);\n  if (!range) {\n    return undefined;\n  }\n\n  const paginated = isPaginated(document);\n  const highlightsContainer = ensureContainer(win, annotationFlag);\n  const highlightParent = document.createElement("div");\n\n  highlightParent.setAttribute("id", highlight.id);\n  highlightParent.setAttribute("class", CLASS_HIGHLIGHT_CONTAINER);\n\n  document.body.style.position = "relative";\n  highlightParent.style.setProperty("pointer-events", "none");\n  if (highlight.pointerInteraction) {\n    highlightParent.setAttribute("data-click", "1");\n  }\n\n  const bodyRect = document.body.getBoundingClientRect();\n  const useSVG = !DEBUG_VISUALS && USE_SVG;\n  //const useSVG = USE_SVG;\n  const drawUnderline = false;\n  const drawStrikeThrough = false;\n  const doNotMergeHorizontallyAlignedRects = drawUnderline || drawStrikeThrough;\n  //const clientRects = DEBUG_VISUALS ? range.getClientRects() :\n  const clientRects = highlight_getClientRectsNoOverlap(\n    range,\n    doNotMergeHorizontallyAlignedRects\n  );\n  let highlightAreaSVGDocFrag;\n  const roundedCorner = 3;\n  const underlineThickness = 2;\n  const strikeThroughLineThickness = 3;\n  const opacity = DEFAULT_BACKGROUND_COLOR_OPACITY;\n  let extra = "";\n  const rangeAnnotationBoundingClientRect =\n    frameForHighlightAnnotationMarkWithID(win, highlight.id);\n\n  let xOffset;\n  let yOffset;\n  let annotationOffset;\n\n  if (navigator.userAgent.match(/Android/i)) {\n    xOffset = paginated ? -scrollElement.scrollLeft : bodyRect.left;\n    yOffset = paginated ? -scrollElement.scrollTop : bodyRect.top;\n    annotationOffset =\n      parseInt(\n        (rangeAnnotationBoundingClientRect.right - xOffset) / window.innerWidth\n      ) + 1;\n  } else if (navigator.userAgent.match(/iPhone|iPad|iPod/i)) {\n    xOffset = paginated ? 0 : -scrollElement.scrollLeft;\n    yOffset = paginated ? 0 : bodyRect.top;\n    annotationOffset = parseInt(\n      rangeAnnotationBoundingClientRect.right / window.innerWidth + 1\n    );\n  }\n\n  for (const clientRect of clientRects) {\n    if (useSVG) {\n      const borderThickness = 0;\n      if (!highlightAreaSVGDocFrag) {\n        highlightAreaSVGDocFrag = document.createDocumentFragment();\n      }\n      const highlightAreaSVGRect = document.createElementNS(\n        SVG_XML_NAMESPACE,\n        "rect"\n      );\n\n      highlightAreaSVGRect.setAttribute("class", CLASS_HIGHLIGHT_AREA);\n      highlightAreaSVGRect.setAttribute(\n        "style",\n        `fill: rgb(${highlight.color.red}, ${highlight.color.green}, ${highlight.color.blue}) !important; fill-opacity: ${opacity} !important; stroke-width: 0;`\n      );\n      highlightAreaSVGRect.scale = scale;\n\n      /*\n             highlightAreaSVGRect.rect = {\n             height: clientRect.height,\n             left: clientRect.left - xOffset,\n             top: clientRect.top - yOffset,\n             width: clientRect.width,\n             };\n             */\n\n      if (annotationFlag) {\n        highlightAreaSVGRect.rect = {\n          height: ANNOTATION_WIDTH, //rangeAnnotationBoundingClientRect.height - rangeAnnotationBoundingClientRect.height/4,\n          left: window.innerWidth * annotationOffset - ANNOTATION_WIDTH,\n          top: rangeAnnotationBoundingClientRect.top - yOffset,\n          width: ANNOTATION_WIDTH,\n        };\n      } else {\n        highlightAreaSVGRect.rect = {\n          height: clientRect.height,\n          left: clientRect.left - xOffset,\n          top: clientRect.top - yOffset,\n          width: clientRect.width,\n        };\n      }\n\n      highlightAreaSVGRect.setAttribute("rx", `${roundedCorner * scale}`);\n      highlightAreaSVGRect.setAttribute("ry", `${roundedCorner * scale}`);\n      highlightAreaSVGRect.setAttribute(\n        "x",\n        `${(highlightAreaSVGRect.rect.left - borderThickness) * scale}`\n      );\n      highlightAreaSVGRect.setAttribute(\n        "y",\n        `${(highlightAreaSVGRect.rect.top - borderThickness) * scale}`\n      );\n      highlightAreaSVGRect.setAttribute(\n        "height",\n        `${(highlightAreaSVGRect.rect.height + borderThickness * 2) * scale}`\n      );\n      highlightAreaSVGRect.setAttribute(\n        "width",\n        `${(highlightAreaSVGRect.rect.width + borderThickness * 2) * scale}`\n      );\n      highlightAreaSVGDocFrag.appendChild(highlightAreaSVGRect);\n      if (drawUnderline) {\n        const highlightAreaSVGLine = document.createElementNS(\n          SVG_XML_NAMESPACE,\n          "line"\n        );\n        highlightAreaSVGRect.setAttribute("class", CLASS_HIGHLIGHT_AREA);\n        highlightAreaSVGLine.setAttribute(\n          "style",\n          `stroke-linecap: round; stroke-width: ${\n            underlineThickness * scale\n          }; stroke: rgb(${highlight.color.red}, ${highlight.color.green}, ${\n            highlight.color.blue\n          }) !important; stroke-opacity: ${opacity} !important`\n        );\n        highlightAreaSVGLine.scale = scale;\n        /*\n                 highlightAreaSVGLine.rect = {\n                 height: clientRect.height,\n                 left: clientRect.left - xOffset,\n                 top: clientRect.top - yOffset,\n                 width: clientRect.width,\n                 };\n                 */\n        if (annotationFlag) {\n          highlightAreaSVGLine.rect = {\n            height: ANNOTATION_WIDTH, //rangeAnnotationBoundingClientRect.height - rangeAnnotationBoundingClientRect.height/4,\n            left: window.innerWidth * annotationOffset - ANNOTATION_WIDTH,\n            top: rangeAnnotationBoundingClientRect.top - yOffset,\n            width: ANNOTATION_WIDTH,\n          };\n        } else {\n          highlightAreaSVGLine.rect = {\n            height: clientRect.height,\n            left: clientRect.left - xOffset,\n            top: clientRect.top - yOffset,\n            width: clientRect.width,\n          };\n        }\n\n        const lineOffset =\n          highlightAreaSVGLine.rect.width > roundedCorner ? roundedCorner : 0;\n        highlightAreaSVGLine.setAttribute(\n          "x1",\n          `${(highlightAreaSVGLine.rect.left + lineOffset) * scale}`\n        );\n        highlightAreaSVGLine.setAttribute(\n          "x2",\n          `${\n            (highlightAreaSVGLine.rect.left +\n              highlightAreaSVGLine.rect.width -\n              lineOffset) *\n            scale\n          }`\n        );\n        const y =\n          (highlightAreaSVGLine.rect.top +\n            highlightAreaSVGLine.rect.height -\n            underlineThickness / 2) *\n          scale;\n        highlightAreaSVGLine.setAttribute("y1", `${y}`);\n        highlightAreaSVGLine.setAttribute("y2", `${y}`);\n        highlightAreaSVGLine.setAttribute(\n          "height",\n          `${highlightAreaSVGLine.rect.height * scale}`\n        );\n        highlightAreaSVGLine.setAttribute(\n          "width",\n          `${highlightAreaSVGLine.rect.width * scale}`\n        );\n        highlightAreaSVGDocFrag.appendChild(highlightAreaSVGLine);\n      }\n      if (drawStrikeThrough) {\n        const highlightAreaSVGLine = document.createElementNS(\n          SVG_XML_NAMESPACE,\n          "line"\n        );\n\n        highlightAreaSVGRect.setAttribute("class", CLASS_HIGHLIGHT_AREA);\n        highlightAreaSVGLine.setAttribute(\n          "style",\n          `stroke-linecap: butt; stroke-width: ${\n            strikeThroughLineThickness * scale\n          }; stroke: rgb(${highlight.color.red}, ${highlight.color.green}, ${\n            highlight.color.blue\n          }) !important; stroke-opacity: ${opacity} !important`\n        );\n        highlightAreaSVGLine.scale = scale;\n\n        /*\n                 highlightAreaSVGLine.rect = {\n                 height: clientRect.height,\n                 left: clientRect.left - xOffset,\n                 top: clientRect.top - yOffset,\n                 width: clientRect.width,\n                 };\n                 */\n\n        if (annotationFlag) {\n          highlightAreaSVGLine.rect = {\n            height: ANNOTATION_WIDTH, //rangeAnnotationBoundingClientRect.height - rangeAnnotationBoundingClientRect.height/4,\n            left: window.innerWidth * annotationOffset - ANNOTATION_WIDTH,\n            top: rangeAnnotationBoundingClientRect.top - yOffset,\n            width: ANNOTATION_WIDTH,\n          };\n        } else {\n          highlightAreaSVGLine.rect = {\n            height: clientRect.height,\n            left: clientRect.left - xOffset,\n            top: clientRect.top - yOffset,\n            width: clientRect.width,\n          };\n        }\n\n        highlightAreaSVGLine.setAttribute(\n          "x1",\n          `${highlightAreaSVGLine.rect.left * scale}`\n        );\n        highlightAreaSVGLine.setAttribute(\n          "x2",\n          `${\n            (highlightAreaSVGLine.rect.left + highlightAreaSVGLine.rect.width) *\n            scale\n          }`\n        );\n        const lineOffset = highlightAreaSVGLine.rect.height / 2;\n        const y = (highlightAreaSVGLine.rect.top + lineOffset) * scale;\n        highlightAreaSVGLine.setAttribute("y1", `${y}`);\n        highlightAreaSVGLine.setAttribute("y2", `${y}`);\n        highlightAreaSVGLine.setAttribute(\n          "height",\n          `${highlightAreaSVGLine.rect.height * scale}`\n        );\n        highlightAreaSVGLine.setAttribute(\n          "width",\n          `${highlightAreaSVGLine.rect.width * scale}`\n        );\n        highlightAreaSVGDocFrag.appendChild(highlightAreaSVGLine);\n      }\n    } else {\n      const highlightArea = document.createElement("div");\n\n      highlightArea.setAttribute("class", CLASS_HIGHLIGHT_AREA);\n\n      if (DEBUG_VISUALS) {\n        const rgb = Math.round(0xffffff * Math.random());\n        const r = rgb >> 16;\n        const g = (rgb >> 8) & 255;\n        const b = rgb & 255;\n        extra = `outline-color: rgb(${r}, ${g}, ${b}); outline-style: solid; outline-width: 1px; outline-offset: -1px;`;\n      } else {\n        if (drawUnderline) {\n          extra += `border-bottom: ${underlineThickness * scale}px solid rgba(${\n            highlight.color.red\n          }, ${highlight.color.green}, ${\n            highlight.color.blue\n          }, ${opacity}) !important`;\n        }\n      }\n      highlightArea.setAttribute(\n        "style",\n        `border-radius: ${roundedCorner}px !important; background-color: rgba(${highlight.color.red}, ${highlight.color.green}, ${highlight.color.blue}, ${opacity}) !important; ${extra}`\n      );\n      highlightArea.style.setProperty("pointer-events", "none");\n      highlightArea.style.position = paginated ? "fixed" : "absolute";\n      highlightArea.scale = scale;\n      /*\n             highlightArea.rect = {\n             height: clientRect.height,\n             left: clientRect.left - xOffset,\n             top: clientRect.top - yOffset,\n             width: clientRect.width,\n             };\n             */\n      if (annotationFlag) {\n        highlightArea.rect = {\n          height: ANNOTATION_WIDTH, //rangeAnnotationBoundingClientRect.height - rangeAnnotationBoundingClientRect.height/4,\n          left: window.innerWidth * annotationOffset - ANNOTATION_WIDTH,\n          top: rangeAnnotationBoundingClientRect.top - yOffset,\n          width: ANNOTATION_WIDTH,\n        };\n      } else {\n        highlightArea.rect = {\n          height: clientRect.height,\n          left: clientRect.left - xOffset,\n          top: clientRect.top - yOffset,\n          width: clientRect.width,\n        };\n      }\n\n      highlightArea.style.width = `${highlightArea.rect.width * scale}px`;\n      highlightArea.style.height = `${highlightArea.rect.height * scale}px`;\n      highlightArea.style.left = `${highlightArea.rect.left * scale}px`;\n      highlightArea.style.top = `${highlightArea.rect.top * scale}px`;\n      highlightParent.append(highlightArea);\n      if (!DEBUG_VISUALS && drawStrikeThrough) {\n        //if (drawStrikeThrough) {\n        const highlightAreaLine = document.createElement("div");\n        highlightAreaLine.setAttribute("class", CLASS_HIGHLIGHT_AREA);\n\n        highlightAreaLine.setAttribute(\n          "style",\n          `background-color: rgba(${highlight.color.red}, ${highlight.color.green}, ${highlight.color.blue}, ${opacity}) !important;`\n        );\n        highlightAreaLine.style.setProperty("pointer-events", "none");\n        highlightAreaLine.style.position = paginated ? "fixed" : "absolute";\n        highlightAreaLine.scale = scale;\n        /*\n                 highlightAreaLine.rect = {\n                 height: clientRect.height,\n                 left: clientRect.left - xOffset,\n                 top: clientRect.top - yOffset,\n                 width: clientRect.width,\n                 };\n                 */\n\n        if (annotationFlag) {\n          highlightAreaLine.rect = {\n            height: ANNOTATION_WIDTH, //rangeAnnotationBoundingClientRect.height - rangeAnnotationBoundingClientRect.height/4,\n            left: window.innerWidth * annotationOffset - ANNOTATION_WIDTH,\n            top: rangeAnnotationBoundingClientRect.top - yOffset,\n            width: ANNOTATION_WIDTH,\n          };\n        } else {\n          highlightAreaLine.rect = {\n            height: clientRect.height,\n            left: clientRect.left - xOffset,\n            top: clientRect.top - yOffset,\n            width: clientRect.width,\n          };\n        }\n\n        highlightAreaLine.style.width = `${\n          highlightAreaLine.rect.width * scale\n        }px`;\n        highlightAreaLine.style.height = `${\n          strikeThroughLineThickness * scale\n        }px`;\n        highlightAreaLine.style.left = `${\n          highlightAreaLine.rect.left * scale\n        }px`;\n        highlightAreaLine.style.top = `${\n          (highlightAreaLine.rect.top +\n            highlightAreaLine.rect.height / 2 -\n            strikeThroughLineThickness / 2) *\n          scale\n        }px`;\n        highlightParent.append(highlightAreaLine);\n      }\n    }\n\n    if (annotationFlag) {\n      break;\n    }\n  }\n\n  if (useSVG && highlightAreaSVGDocFrag) {\n    const highlightAreaSVG = document.createElementNS(SVG_XML_NAMESPACE, "svg");\n    highlightAreaSVG.setAttribute("pointer-events", "none");\n    highlightAreaSVG.style.position = paginated ? "fixed" : "absolute";\n    highlightAreaSVG.style.overflow = "visible";\n    highlightAreaSVG.style.left = "0";\n    highlightAreaSVG.style.top = "0";\n    highlightAreaSVG.append(highlightAreaSVGDocFrag);\n    highlightParent.append(highlightAreaSVG);\n  }\n\n  const highlightBounding = document.createElement("div");\n\n  if (annotationFlag) {\n    highlightBounding.setAttribute("class", CLASS_ANNOTATION_BOUNDING_AREA);\n    highlightBounding.setAttribute(\n      "style",\n      `border-radius: ${roundedCorner}px !important; background-color: rgba(${highlight.color.red}, ${highlight.color.green}, ${highlight.color.blue}, ${opacity}) !important; ${extra}`\n    );\n  } else {\n    highlightBounding.setAttribute("class", CLASS_HIGHLIGHT_BOUNDING_AREA);\n  }\n\n  highlightBounding.style.setProperty("pointer-events", "none");\n  highlightBounding.style.position = paginated ? "fixed" : "absolute";\n  highlightBounding.scale = scale;\n\n  if (DEBUG_VISUALS) {\n    highlightBounding.setAttribute(\n      "style",\n      `outline-color: magenta; outline-style: solid; outline-width: 1px; outline-offset: -1px;`\n    );\n  }\n\n  if (annotationFlag) {\n    highlightBounding.rect = {\n      height: ANNOTATION_WIDTH, //rangeAnnotationBoundingClientRect.height - rangeAnnotationBoundingClientRect.height/4,\n      left: window.innerWidth * annotationOffset - ANNOTATION_WIDTH,\n      top: rangeAnnotationBoundingClientRect.top - yOffset,\n      width: ANNOTATION_WIDTH,\n    };\n  } else {\n    const rangeBoundingClientRect = range.getBoundingClientRect();\n    highlightBounding.rect = {\n      height: rangeBoundingClientRect.height,\n      left: rangeBoundingClientRect.left - xOffset,\n      top: rangeBoundingClientRect.top - yOffset,\n      width: rangeBoundingClientRect.width,\n    };\n  }\n\n  highlightBounding.style.width = `${highlightBounding.rect.width * scale}px`;\n  highlightBounding.style.height = `${highlightBounding.rect.height * scale}px`;\n  highlightBounding.style.left = `${highlightBounding.rect.left * scale}px`;\n  highlightBounding.style.top = `${highlightBounding.rect.top * scale}px`;\n\n  highlightParent.append(highlightBounding);\n  highlightsContainer.append(highlightParent);\n\n  return highlightParent;\n}\n\nfunction createOrderedRange(startNode, startOffset, endNode, endOffset) {\n  const range = new Range();\n  range.setStart(startNode, startOffset);\n  range.setEnd(endNode, endOffset);\n  if (!range.collapsed) {\n    return range;\n  }\n  console.log(">>> createOrderedRange COLLAPSED ... RANGE REVERSE?");\n  const rangeReverse = new Range();\n  rangeReverse.setStart(endNode, endOffset);\n  rangeReverse.setEnd(startNode, startOffset);\n  if (!rangeReverse.collapsed) {\n    console.log(">>> createOrderedRange RANGE REVERSE OK.");\n    return range;\n  }\n  console.log(">>> createOrderedRange RANGE REVERSE ALSO COLLAPSED?!");\n  return undefined;\n}\n\nfunction convertRange(range, getCssSelector, computeElementCFI) {\n  const startIsElement = range.startContainer.nodeType === Node.ELEMENT_NODE;\n  const startContainerElement = startIsElement\n    ? range.startContainer\n    : range.startContainer.parentNode &&\n      range.startContainer.parentNode.nodeType === Node.ELEMENT_NODE\n    ? range.startContainer.parentNode\n    : undefined;\n  if (!startContainerElement) {\n    return undefined;\n  }\n  const startContainerChildTextNodeIndex = startIsElement\n    ? -1\n    : Array.from(startContainerElement.childNodes).indexOf(\n        range.startContainer\n      );\n  if (startContainerChildTextNodeIndex < -1) {\n    return undefined;\n  }\n  const startContainerElementCssSelector = getCssSelector(\n    startContainerElement\n  );\n  const endIsElement = range.endContainer.nodeType === Node.ELEMENT_NODE;\n  const endContainerElement = endIsElement\n    ? range.endContainer\n    : range.endContainer.parentNode &&\n      range.endContainer.parentNode.nodeType === Node.ELEMENT_NODE\n    ? range.endContainer.parentNode\n    : undefined;\n  if (!endContainerElement) {\n    return undefined;\n  }\n  const endContainerChildTextNodeIndex = endIsElement\n    ? -1\n    : Array.from(endContainerElement.childNodes).indexOf(range.endContainer);\n  if (endContainerChildTextNodeIndex < -1) {\n    return undefined;\n  }\n  const endContainerElementCssSelector = getCssSelector(endContainerElement);\n  const commonElementAncestor = getCommonAncestorElement(\n    range.startContainer,\n    range.endContainer\n  );\n  if (!commonElementAncestor) {\n    console.log("^^^ NO RANGE COMMON ANCESTOR?!");\n    return undefined;\n  }\n  if (range.commonAncestorContainer) {\n    const rangeCommonAncestorElement =\n      range.commonAncestorContainer.nodeType === Node.ELEMENT_NODE\n        ? range.commonAncestorContainer\n        : range.commonAncestorContainer.parentNode;\n    if (\n      rangeCommonAncestorElement &&\n      rangeCommonAncestorElement.nodeType === Node.ELEMENT_NODE\n    ) {\n      if (commonElementAncestor !== rangeCommonAncestorElement) {\n        console.log(">>>>>> COMMON ANCESTOR CONTAINER DIFF??!");\n        console.log(getCssSelector(commonElementAncestor));\n        console.log(getCssSelector(rangeCommonAncestorElement));\n      }\n    }\n  }\n  const rootElementCfi = computeElementCFI(commonElementAncestor);\n  const startElementCfi = computeElementCFI(startContainerElement);\n  const endElementCfi = computeElementCFI(endContainerElement);\n  let cfi;\n  if (rootElementCfi && startElementCfi && endElementCfi) {\n    let startElementOrTextCfi = startElementCfi;\n    if (!startIsElement) {\n      const startContainerChildTextNodeIndexForCfi = getChildTextNodeCfiIndex(\n        startContainerElement,\n        range.startContainer\n      );\n      startElementOrTextCfi =\n        startElementCfi +\n        "/" +\n        startContainerChildTextNodeIndexForCfi +\n        ":" +\n        range.startOffset;\n    } else {\n      if (\n        range.startOffset >= 0 &&\n        range.startOffset < startContainerElement.childNodes.length\n      ) {\n        const childNode = startContainerElement.childNodes[range.startOffset];\n        if (childNode.nodeType === Node.ELEMENT_NODE) {\n          startElementOrTextCfi =\n            startElementCfi + "/" + (range.startOffset + 1) * 2;\n        } else {\n          const cfiTextNodeIndex = getChildTextNodeCfiIndex(\n            startContainerElement,\n            childNode\n          );\n          startElementOrTextCfi = startElementCfi + "/" + cfiTextNodeIndex;\n        }\n      } else {\n        const cfiIndexOfLastElement =\n          startContainerElement.childElementCount * 2;\n        const lastChildNode =\n          startContainerElement.childNodes[\n            startContainerElement.childNodes.length - 1\n          ];\n        if (lastChildNode.nodeType === Node.ELEMENT_NODE) {\n          startElementOrTextCfi =\n            startElementCfi + "/" + (cfiIndexOfLastElement + 1);\n        } else {\n          startElementOrTextCfi =\n            startElementCfi + "/" + (cfiIndexOfLastElement + 2);\n        }\n      }\n    }\n    let endElementOrTextCfi = endElementCfi;\n    if (!endIsElement) {\n      const endContainerChildTextNodeIndexForCfi = getChildTextNodeCfiIndex(\n        endContainerElement,\n        range.endContainer\n      );\n      endElementOrTextCfi =\n        endElementCfi +\n        "/" +\n        endContainerChildTextNodeIndexForCfi +\n        ":" +\n        range.endOffset;\n    } else {\n      if (\n        range.endOffset >= 0 &&\n        range.endOffset < endContainerElement.childNodes.length\n      ) {\n        const childNode = endContainerElement.childNodes[range.endOffset];\n        if (childNode.nodeType === Node.ELEMENT_NODE) {\n          endElementOrTextCfi = endElementCfi + "/" + (range.endOffset + 1) * 2;\n        } else {\n          const cfiTextNodeIndex = getChildTextNodeCfiIndex(\n            endContainerElement,\n            childNode\n          );\n          endElementOrTextCfi = endElementCfi + "/" + cfiTextNodeIndex;\n        }\n      } else {\n        const cfiIndexOfLastElement = endContainerElement.childElementCount * 2;\n        const lastChildNode =\n          endContainerElement.childNodes[\n            endContainerElement.childNodes.length - 1\n          ];\n        if (lastChildNode.nodeType === Node.ELEMENT_NODE) {\n          endElementOrTextCfi =\n            endElementCfi + "/" + (cfiIndexOfLastElement + 1);\n        } else {\n          endElementOrTextCfi =\n            endElementCfi + "/" + (cfiIndexOfLastElement + 2);\n        }\n      }\n    }\n    cfi =\n      rootElementCfi +\n      "," +\n      startElementOrTextCfi.replace(rootElementCfi, "") +\n      "," +\n      endElementOrTextCfi.replace(rootElementCfi, "");\n  }\n  return {\n    cfi,\n    endContainerChildTextNodeIndex,\n    endContainerElementCssSelector,\n    endOffset: range.endOffset,\n    startContainerChildTextNodeIndex,\n    startContainerElementCssSelector,\n    startOffset: range.startOffset,\n  };\n}\n\nfunction convertRangeInfo(document, rangeInfo) {\n  const startElement = document.querySelector(\n    rangeInfo.startContainerElementCssSelector\n  );\n  if (!startElement) {\n    console.log("^^^ convertRangeInfo NO START ELEMENT CSS SELECTOR?!");\n    return undefined;\n  }\n  let startContainer = startElement;\n  if (rangeInfo.startContainerChildTextNodeIndex >= 0) {\n    if (\n      rangeInfo.startContainerChildTextNodeIndex >=\n      startElement.childNodes.length\n    ) {\n      console.log(\n        "^^^ convertRangeInfo rangeInfo.startContainerChildTextNodeIndex >= startElement.childNodes.length?!"\n      );\n      return undefined;\n    }\n    startContainer =\n      startElement.childNodes[rangeInfo.startContainerChildTextNodeIndex];\n    if (startContainer.nodeType !== Node.TEXT_NODE) {\n      console.log(\n        "^^^ convertRangeInfo startContainer.nodeType !== Node.TEXT_NODE?!"\n      );\n      return undefined;\n    }\n  }\n  const endElement = document.querySelector(\n    rangeInfo.endContainerElementCssSelector\n  );\n  if (!endElement) {\n    console.log("^^^ convertRangeInfo NO END ELEMENT CSS SELECTOR?!");\n    return undefined;\n  }\n  let endContainer = endElement;\n  if (rangeInfo.endContainerChildTextNodeIndex >= 0) {\n    if (\n      rangeInfo.endContainerChildTextNodeIndex >= endElement.childNodes.length\n    ) {\n      console.log(\n        "^^^ convertRangeInfo rangeInfo.endContainerChildTextNodeIndex >= endElement.childNodes.length?!"\n      );\n      return undefined;\n    }\n    endContainer =\n      endElement.childNodes[rangeInfo.endContainerChildTextNodeIndex];\n    if (endContainer.nodeType !== Node.TEXT_NODE) {\n      console.log(\n        "^^^ convertRangeInfo endContainer.nodeType !== Node.TEXT_NODE?!"\n      );\n      return undefined;\n    }\n  }\n  return createOrderedRange(\n    startContainer,\n    rangeInfo.startOffset,\n    endContainer,\n    rangeInfo.endOffset\n  );\n}\n\nfunction frameForHighlightAnnotationMarkWithID(win, id) {\n  let clientRects = frameForHighlightWithID(id);\n  if (!clientRects) return;\n\n  var topClientRect = clientRects[0];\n  var maxHeight = topClientRect.height;\n  for (const clientRect of clientRects) {\n    if (clientRect.top < topClientRect.top) topClientRect = clientRect;\n    if (clientRect.height > maxHeight) maxHeight = clientRect.height;\n  }\n\n  const document = win.document;\n\n  const scrollElement = getScrollingElement(document);\n  const paginated = isPaginated(document);\n  const bodyRect = document.body.getBoundingClientRect();\n  let yOffset;\n  if (navigator.userAgent.match(/Android/i)) {\n    yOffset = paginated ? -scrollElement.scrollTop : bodyRect.top;\n  } else if (navigator.userAgent.match(/iPhone|iPad|iPod/i)) {\n    yOffset = paginated ? 0 : bodyRect.top;\n  }\n  var newTop = topClientRect.top;\n\n  if (_highlightsContainer) {\n    do {\n      var boundingAreas = document.getElementsByClassName(\n        CLASS_ANNOTATION_BOUNDING_AREA\n      );\n      var found = false;\n      //for (let i = 0, length = boundingAreas.snapshotLength; i < length; ++i) {\n      for (\n        var i = 0, len = boundingAreas.length | 0;\n        i < len;\n        i = (i + 1) | 0\n      ) {\n        var boundingArea = boundingAreas[i];\n        if (Math.abs(boundingArea.rect.top - (newTop - yOffset)) < 3) {\n          newTop += boundingArea.rect.height;\n          found = true;\n          break;\n        }\n      }\n    } while (found);\n  }\n\n  topClientRect.top = newTop;\n  topClientRect.height = maxHeight;\n\n  return topClientRect;\n}\n\nfunction highlightWithID(id) {\n  let i = -1;\n  const highlight = _highlights.find((h, j) => {\n    i = j;\n    return h.id === id;\n  });\n  return highlight;\n}\n\nfunction frameForHighlightWithID(id) {\n  const highlight = highlightWithID(id);\n  if (!highlight) return;\n\n  const document = window.document;\n  const scrollElement = getScrollingElement(document);\n  const range = convertRangeInfo(document, highlight.rangeInfo);\n  if (!range) {\n    return undefined;\n  }\n\n  const drawUnderline = false;\n  const drawStrikeThrough = false;\n  const doNotMergeHorizontallyAlignedRects = drawUnderline || drawStrikeThrough;\n  //const clientRects = DEBUG_VISUALS ? range.getClientRects() :\n  const clientRects = highlight_getClientRectsNoOverlap(\n    range,\n    doNotMergeHorizontallyAlignedRects\n  );\n\n  return clientRects;\n}\n\nfunction rangeInfo2Location(rangeInfo) {\n  return {\n    cssSelector: rangeInfo.startContainerElementCssSelector,\n    partialCfi: rangeInfo.cfi,\n    domRange: {\n      start: {\n        cssSelector: rangeInfo.startContainerElementCssSelector,\n        textNodeIndex: rangeInfo.startContainerChildTextNodeIndex,\n        offset: rangeInfo.startOffset,\n      },\n      end: {\n        cssSelector: rangeInfo.endContainerElementCssSelector,\n        textNodeIndex: rangeInfo.endContainerChildTextNodeIndex,\n        offset: rangeInfo.endOffset,\n      },\n    },\n  };\n}\n\nfunction location2RangeInfo(location) {\n  const locations = location.locations;\n  const domRange = locations.domRange;\n  const start = domRange.start;\n  const end = domRange.end;\n\n  return {\n    cfi: location.partialCfi,\n    endContainerChildTextNodeIndex: end.textNodeIndex,\n    endContainerElementCssSelector: end.cssSelector,\n    endOffset: end.offset,\n    startContainerChildTextNodeIndex: start.textNodeIndex,\n    startContainerElementCssSelector: start.cssSelector,\n    startOffset: start.offset,\n  };\n}\n\nfunction rectangleForHighlightWithID(id) {\n  const highlight = highlightWithID(id);\n  if (!highlight) return;\n\n  const document = window.document;\n  const scrollElement = getScrollingElement(document);\n  const range = convertRangeInfo(document, highlight.rangeInfo);\n  if (!range) {\n    return undefined;\n  }\n\n  const drawUnderline = false;\n  const drawStrikeThrough = false;\n  const doNotMergeHorizontallyAlignedRects = drawUnderline || drawStrikeThrough;\n  //const clientRects = DEBUG_VISUALS ? range.getClientRects() :\n  const clientRects = highlight_getClientRectsNoOverlap(\n    range,\n    doNotMergeHorizontallyAlignedRects\n  );\n  var size = {\n    screenWidth: window.outerWidth,\n    screenHeight: window.outerHeight,\n    left: clientRects[0].left,\n    width: clientRects[0].width,\n    top: clientRects[0].top,\n    height: clientRects[0].height,\n  };\n\n  return size;\n}\n\nfunction getSelectionRect() {\n  try {\n    var sel = window.getSelection();\n    if (!sel) {\n      return;\n    }\n    var range = sel.getRangeAt(0);\n\n    const clientRect = range.getBoundingClientRect();\n\n    var handleBounds = {\n      screenWidth: window.outerWidth,\n      screenHeight: window.outerHeight,\n      left: clientRect.left,\n      width: clientRect.width,\n      top: clientRect.top,\n      height: clientRect.height,\n    };\n    return handleBounds;\n  } catch (e) {\n    return null;\n  }\n}\n\nfunction setScrollMode(flag) {\n  if (!flag) {\n    document.documentElement.classList.add(CLASS_PAGINATED);\n  } else {\n    document.documentElement.classList.remove(CLASS_PAGINATED);\n  }\n}\n\n/*\n if (document.addEventListener) { // IE >= 9; other browsers\n        document.addEventListener(\'contextmenu\', function(e) {\n            //alert("You\'ve tried to open context menu"); //here you draw your own menu\n            //e.preventDefault();\n            //let getCssSelector = fullQualifiedSelector;\n            \n\t\t\tlet str = window.getSelection();\n\t\t\tlet selectionInfo = getCurrentSelectionInfo();\n\t\t\tlet pos = createHighlight(selectionInfo,{red:10,green:50,blue:230},true);\n\t\t\tlet ret2 = createAnnotation(pos.id);\n\t\t\t\n  }, false);\n    } else { // IE < 9\n        document.attachEvent(\'oncontextmenu\', function() {\n            alert("You\'ve tried to open context menu");\n            window.event.returnValue = false;\n        });\n    }\n*/\n\n// EXTERNAL MODULE: ./node_modules/string.prototype.matchall/index.js\nvar string_prototype_matchall = __webpack_require__(4956);\nvar string_prototype_matchall_default = /*#__PURE__*/__webpack_require__.n(string_prototype_matchall);\n;// CONCATENATED MODULE: ./src/selection.js\n//\n//  Copyright 2021 Readium Foundation. All rights reserved.\n//  Use of this source code is governed by the BSD-style license\n//  available in the top-level LICENSE file of the project.\n//\n\n\n\n\n\n// Polyfill for Android API 26\n\nstring_prototype_matchall_default().shim();\n\nconst selection_debug = true;\n\nfunction getCurrentSelection() {\n  const text = getCurrentSelectionText();\n  if (!text) {\n    return null;\n  }\n  const rect = selection_getSelectionRect();\n  return { text, rect };\n}\n\nfunction selection_getSelectionRect() {\n  try {\n    let sel = window.getSelection();\n    if (!sel) {\n      return;\n    }\n    let range = sel.getRangeAt(0);\n\n    return toNativeRect(range.getBoundingClientRect());\n  } catch (e) {\n    logError(e);\n    return null;\n  }\n}\n\nfunction getCurrentSelectionText() {\n  const selection = window.getSelection();\n  if (!selection) {\n    return undefined;\n  }\n  if (selection.isCollapsed) {\n    return undefined;\n  }\n  const highlight = selection.toString();\n  const cleanHighlight = highlight\n    .trim()\n    .replace(/\\n/g, " ")\n    .replace(/\\s\\s+/g, " ");\n  if (cleanHighlight.length === 0) {\n    return undefined;\n  }\n  if (!selection.anchorNode || !selection.focusNode) {\n    return undefined;\n  }\n  const range =\n    selection.rangeCount === 1\n      ? selection.getRangeAt(0)\n      : selection_createOrderedRange(\n          selection.anchorNode,\n          selection.anchorOffset,\n          selection.focusNode,\n          selection.focusOffset\n        );\n  if (!range || range.collapsed) {\n    selection_log("$$$$$$$$$$$$$$$$$ CANNOT GET NON-COLLAPSED SELECTION RANGE?!");\n    return undefined;\n  }\n\n  const text = document.body.textContent;\n  const textRange = text_range_TextRange.fromRange(range).relativeTo(document.body);\n  const start = textRange.start.offset;\n  const end = textRange.end.offset;\n\n  const snippetLength = 200;\n\n  // Compute the text before the highlight, ignoring the first "word", which might be cut.\n  let before = text.slice(Math.max(0, start - snippetLength), start);\n  let firstWordStart = before.search(/\\P{L}\\p{L}/gu);\n  if (firstWordStart !== -1) {\n    before = before.slice(firstWordStart + 1);\n  }\n\n  // Compute the text after the highlight, ignoring the last "word", which might be cut.\n  let after = text.slice(end, Math.min(text.length, end + snippetLength));\n  let lastWordEnd = Array.from(after.matchAll(/\\p{L}\\P{L}/gu)).pop();\n  if (lastWordEnd !== undefined && lastWordEnd.index > 1) {\n    after = after.slice(0, lastWordEnd.index + 1);\n  }\n\n  return { highlight, before, after };\n}\n\nfunction selection_createOrderedRange(startNode, startOffset, endNode, endOffset) {\n  const range = new Range();\n  range.setStart(startNode, startOffset);\n  range.setEnd(endNode, endOffset);\n  if (!range.collapsed) {\n    return range;\n  }\n  selection_log(">>> createOrderedRange COLLAPSED ... RANGE REVERSE?");\n  const rangeReverse = new Range();\n  rangeReverse.setStart(endNode, endOffset);\n  rangeReverse.setEnd(startNode, startOffset);\n  if (!rangeReverse.collapsed) {\n    selection_log(">>> createOrderedRange RANGE REVERSE OK.");\n    return range;\n  }\n  selection_log(">>> createOrderedRange RANGE REVERSE ALSO COLLAPSED?!");\n  return undefined;\n}\n\nfunction selection_convertRangeInfo(document, rangeInfo) {\n  const startElement = document.querySelector(\n    rangeInfo.startContainerElementCssSelector\n  );\n  if (!startElement) {\n    selection_log("^^^ convertRangeInfo NO START ELEMENT CSS SELECTOR?!");\n    return undefined;\n  }\n  let startContainer = startElement;\n  if (rangeInfo.startContainerChildTextNodeIndex >= 0) {\n    if (\n      rangeInfo.startContainerChildTextNodeIndex >=\n      startElement.childNodes.length\n    ) {\n      selection_log(\n        "^^^ convertRangeInfo rangeInfo.startContainerChildTextNodeIndex >= startElement.childNodes.length?!"\n      );\n      return undefined;\n    }\n    startContainer =\n      startElement.childNodes[rangeInfo.startContainerChildTextNodeIndex];\n    if (startContainer.nodeType !== Node.TEXT_NODE) {\n      selection_log("^^^ convertRangeInfo startContainer.nodeType !== Node.TEXT_NODE?!");\n      return undefined;\n    }\n  }\n  const endElement = document.querySelector(\n    rangeInfo.endContainerElementCssSelector\n  );\n  if (!endElement) {\n    selection_log("^^^ convertRangeInfo NO END ELEMENT CSS SELECTOR?!");\n    return undefined;\n  }\n  let endContainer = endElement;\n  if (rangeInfo.endContainerChildTextNodeIndex >= 0) {\n    if (\n      rangeInfo.endContainerChildTextNodeIndex >= endElement.childNodes.length\n    ) {\n      selection_log(\n        "^^^ convertRangeInfo rangeInfo.endContainerChildTextNodeIndex >= endElement.childNodes.length?!"\n      );\n      return undefined;\n    }\n    endContainer =\n      endElement.childNodes[rangeInfo.endContainerChildTextNodeIndex];\n    if (endContainer.nodeType !== Node.TEXT_NODE) {\n      selection_log("^^^ convertRangeInfo endContainer.nodeType !== Node.TEXT_NODE?!");\n      return undefined;\n    }\n  }\n  return selection_createOrderedRange(\n    startContainer,\n    rangeInfo.startOffset,\n    endContainer,\n    rangeInfo.endOffset\n  );\n}\n\nfunction selection_location2RangeInfo(location) {\n  const locations = location.locations;\n  const domRange = locations.domRange;\n  const start = domRange.start;\n  const end = domRange.end;\n\n  return {\n    endContainerChildTextNodeIndex: end.textNodeIndex,\n    endContainerElementCssSelector: end.cssSelector,\n    endOffset: end.offset,\n    startContainerChildTextNodeIndex: start.textNodeIndex,\n    startContainerElementCssSelector: start.cssSelector,\n    startOffset: start.offset,\n  };\n}\n\nfunction selection_log() {\n  if (selection_debug) {\n    log.apply(null, arguments);\n  }\n}\n\n;// CONCATENATED MODULE: ./src/index.js\n//\n//  Copyright 2021 Readium Foundation. All rights reserved.\n//  Use of this source code is governed by the BSD-style license\n//  available in the top-level LICENSE file of the project.\n//\n\n// Base script used by both reflowable and fixed layout resources.\n\n\n\n\n\n\n\n// Public API used by the navigator.\nwindow.readium = {\n  // utils\n  scrollToId: scrollToId,\n  scrollToPosition: scrollToPosition,\n  scrollToText: scrollToText,\n  scrollLeft: scrollLeft,\n  scrollRight: scrollRight,\n  scrollToStart: scrollToStart,\n  scrollToEnd: scrollToEnd,\n  setProperty: setProperty,\n  removeProperty: removeProperty,\n\n  // selection\n  getCurrentSelection: getCurrentSelection,\n\n  // decoration\n  registerDecorationTemplates: registerTemplates,\n  getDecorations: getDecorations,\n};\n\n// Legacy highlights API.\nwindow.createAnnotation = createAnnotation;\nwindow.createHighlight = createHighlight;\nwindow.destroyHighlight = destroyHighlight;\nwindow.getCurrentSelectionInfo = getCurrentSelectionInfo;\nwindow.getSelectionRect = getSelectionRect;\nwindow.rectangleForHighlightWithID = rectangleForHighlightWithID;\nwindow.setScrollMode = setScrollMode;\n\n;// CONCATENATED MODULE: ./src/index-fixed.js\n//\n//  Copyright 2021 Readium Foundation. All rights reserved.\n//  Use of this source code is governed by the BSD-style license\n//  available in the top-level LICENSE file of the project.\n//\n\n// Script used for fixed layouts resources.\n\n\n//# sourceURL=[module]\n//# sourceMappingURL=data:application/json;charset=utf-8;base64,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\n//# sourceURL=webpack-internal:///7786\n')},4654:()=>{}},__webpack_module_cache__={};function __webpack_require__(n){var t=__webpack_module_cache__[n];if(void 0!==t)return t.exports;var e=__webpack_module_cache__[n]={exports:{}};return __webpack_modules__[n](e,e.exports,__webpack_require__),e.exports}__webpack_require__.n=n=>{var t=n&&n.__esModule?()=>n.default:()=>n;return __webpack_require__.d(t,{a:t}),t},__webpack_require__.d=(n,t)=>{for(var e in t)__webpack_require__.o(t,e)&&!__webpack_require__.o(n,e)&&Object.defineProperty(n,e,{enumerable:!0,get:t[e]})},__webpack_require__.o=(n,t)=>Object.prototype.hasOwnProperty.call(n,t);var __webpack_exports__=__webpack_require__(7786)})();