// MRAID-3.0 Script

var webkit = window.webkit || {};
var messageHandlers = webkit.messageHandlers || {};
var iOS = messageHandlers.mraid;

(function() {
    this.callNative = function(command) {
        if(iOS){
            iOS.postMessage(command);
	    } else if (typeof Android === 'object'){
	        Android.onMRAIDMessageReceived(JSON.stringify(command));
	    }
	}
}());

(function() {
	var oldLog = console.log;
	console.log = function(log) {
		callNative({name: "console-log", params: {message: log}});
		oldLog(log);
	}
	console.debug = console.info = console.warn = console.error = console.log;
}());

(function() {

    ////////////////// BEGIN MRAID-3 OMSDK DEPENDENCIES //////////////////
    const OMSDK_VERIFICATION_CLOSURE_BUILD_VERSION='1.2.8-dev';

    const OMSDK_EVENT_GEOMETRY_CHANGE = "geometryChange";
    const OMSDK_EVENT_AUDIO_VOLUME_CHANGE = "volumeChange";
    const OMSDK_SESSION_EVENT_START = "sessionStart";
    const OMSDK_SESSION_EVENT_ERROR = "sessionError";
    const OMSDK_SESSION_EVENT_FINISH = "sessionFinish";

    let omidVerificationClient = undefined;
    let omsdkEventTracker = undefined;
    let DEBUG = true;

    // We keep exposure change data global & cached so we can send initial state at MRAID subscribe time
    let exposureChangeOcclusionRectangles = null;
    let exposureChangeVisibleRectangle = null;
    let exposureChangePercentExposed = 0.0;

    // We want to keep audioVolumeChange data global & cached so we can send initial state at MRAID subscribe time
    let audioVolumeChangeVolumePercent = 0.0;
    
    function logAlways(message) {
        let name = "MRAID3";
        console.log(name + ": " + message);
    }

    function logD(message) {
        let name = "MRAID3";
        if(DEBUG) console.log(name + ": " + message);
    }

    /**
     * @return success/failure
     */
    function setupMraid3Dependencies() {
        let allgood = false;
        logD("setupMraid3Dependencies(): STARTING");
        if(! OmidVerificationClient) {
            logD("OMSDK VERIFICATION CLIENT NOT FOUND");
        } else {
            logD("OmidVerificationClient object is valid");
            if(! OmidVerificationClient[OMSDK_VERIFICATION_CLOSURE_BUILD_VERSION]) {
                logD("ERROR: OMSDK VERIFICATION CLIENT Version# " + OMSDK_VERIFICATION_CLOSURE_BUILD_VERSION + " NOT FOUND");
            } else {
                omidVerificationClient = new OmidVerificationClient[OMSDK_VERIFICATION_CLOSURE_BUILD_VERSION]();
                if(! omidVerificationClient) logD("ERROR: Failed to construct OmidVerificationClient[" + OMSDK_VERIFICATION_CLOSURE_BUILD_VERSION + "]");
                else if(! omidVerificationClient.isSupported()) logD("ERROR: OmidVerificationClient's isSupported() returned FALSE");
                else allgood = true;
            }
        }
        logAlways("setupMraid3Dependencies(): MRAID-3 setup " + (allgood ? "SUCCESS" : "FAILED (no MRAID-3 support)"));
        return allgood;
    }

    function setupMraid3EventTracking() {
        omsdkEventTracker = new OMSDKEventTracker(omidVerificationClient, 'pandora-mraid', omsdkGeometryChangeCallback, omsdkVolumeChangeCallback, omsdkErrorEventCallback);
        logAlways("setupMraid3EventTracking(): Done");
    }

    /**
     * @param {Object} data event data
     */
    function omsdkGeometryChangeCallback(event) {
        logD("omsdkGeometryChangeCallback(): Entered: Full event data dump: '" + JSON.stringify(event) + "'");
        logD("omsdkGeometryChangeCallback(): Timestamp [" + event.timestamp + "] / type [" + event.type + "] / adSessionId [" + event.adSessionId + "]");
        let data = event.data;

        if (!data) {
            logD("omsdkGeometryChangeCallback(): Error: No event.data value (not firing MRAID event)");
            return;
        } else {
            logD("omsdkGeometryChangeCallback(): Timestamp[" + event.timestamp + "] Got data: " + JSON.stringify(data));
            if(data.viewport) {
                logD("omsdkGeometryChangeCallback(): Timestamp[" + event.timestamp + "] Viewport [width=" + data.viewport.width + "][height=" + data.viewport.height + "]");
            }

            exposureChangePercentExposed = 0.0;
            exposureChangeVisibleRectangle = null;
            exposureChangeOcclusionRectangles = null;
            let adView = data.adView;

            if (adView) {
                exposureChangePercentExposed = adView.percentageInView;
                logD("omsdkGeometryChangeCallback(): Timestamp[" + event.timestamp + "] AdView [percentageInView=" + adView.percentageInView + "]");

                // NOTE: According to OMSDK spec, adView.geometry will contain the creative's dimensions if that exists in the web layer, and if not then this contains the native webView's dimensions
                // In case the creative's dimensions in the web layer do exist, then the native webView's dimensions will be provided in the onScreenGeometry object. Since MRAID spec refers to the ad container
                // which is always the native webView, we must try to always get the native webView's dimensions to send in the mraid event data
                let adContainerGeometry = null;
                if(adView.onScreenGeometry) {
                    adContainerGeometry = adView.onScreenGeometry;
                    if(adView.onScreenGeometry.obstructions) exposureChangeOcclusionRectangles = adView.onScreenGeometry.obstructions;
                    logD("omsdkGeometryChangeCallback(): Timestamp[" + event.timestamp + "] Dump of onScreenGeometry: |" + JSON.stringify(adView.onScreenGeometry) + "|");
                    logAlways("omsdkGeometryChangeCallback(): Timestamp[" + event.timestamp + "] Native webView's VisibleRectangle got from adView.onScreenGeometry: [width=" + adContainerGeometry.width + "][height=" + adContainerGeometry.height + "]" + " [x=" + adContainerGeometry.x + "][y=" + adContainerGeometry.y + "]");
                    if(exposureChangeOcclusionRectangles) logAlways("omsdkGeometryChangeCallback(): Got exposureChangeOcclusionRectangles...dump: " + JSON.stringify(exposureChangeOcclusionRectangles));
                } else if(adView.geometry) {
                    adContainerGeometry = adView.geometry;
                    logD("omsdkGeometryChangeCallback(): Timestamp[" + event.timestamp + "] Dump of onScreenGeometry: |" + JSON.stringify(adView.geometry) + "|");
                    logAlways("omsdkGeometryChangeCallback(): Timestamp[" + event.timestamp + "] Native webView's VisibleRectangle got from adView.geometry: [width=" + adContainerGeometry.width + "][height=" + adContainerGeometry.height + "]" + " [x=" + adContainerGeometry.x + "][y=" + adContainerGeometry.y + "]");
                }
                exposureChangeVisibleRectangle = adContainerGeometry;
            }

            // Fire the MRAID exposureChange event (if a listener was registered)
            mraid.exposureChange(exposureChangePercentExposed, exposureChangeVisibleRectangle, exposureChangeOcclusionRectangles);
        }
    }

    /**
     * @param {Object} data event data
     */
    function omsdkVolumeChangeCallback(event) {
        logD("omsdkVolumeChangeCallback(): Entered: Full event data dump: '" + JSON.stringify(event) + "'");
        logD("omsdkVolumeChangeCallback(): Timestamp [" + event.timestamp + "] / type [" + event.type + "] / adSessionId [" + event.adSessionId + "]");
        let data = event.data;

        if (!data) {
            logD("omsdkVolumeChangeCallback(): Error: No event.data value (not firing MRAID event)");
            return;
        } else {
            audioVolumeChangeVolumePercent = data.videoPlayerVolume;
            logD("omsdkVolumeChangeCallback(): Timestamp[" + event.timestamp + "] Got data: [VideoPlayerVolume=" + data.videoPlayerVolume + "]");

            // Fire the MRAID exposureChange event (if a listener was registered)
            mraid.audioVolumeChange(audioVolumeChangeVolumePercent);
        }
    }

    function omsdkErrorEventCallback(error) {
        console.log("MRAID3: omsdkErrorEventCallback(): ERROR: " + error);
    }

    class OMSDKEventTracker {
        /**
         * @param {geometryChangeEventCallback} Callback function for the geometryChange event
         * @param {volumeChangeEventCallback} Callback function for the volumeChange event
         * @param {errorEventCallback} Callback function for relaying error events
         */
        constructor(verificationClient, vendorKey, geometryChangeEventCallback, volumeChangeEventCallback, errorEventCallback) {
            this.verificationClient = verificationClient;
            this.geometryChangeEventCallback = geometryChangeEventCallback;
            this.volumeChangeEventCallback = volumeChangeEventCallback;
            this.errorEventCallback = errorEventCallback;

            logD("OMSDKEventTracker::constructor Initializing OMSDK event tracking");

            if (verificationClient.isSupported) {
                logAlways("OMSDKEventTracker::constructor OMID is supported. Registering event listeners & session observer for vendorKey: '" + vendorKey + "'.");
                this.verificationClient.registerSessionObserver((event) => this.sessionEventCallback(event), vendorKey);
                this.verificationClient.addEventListener(OMSDK_EVENT_GEOMETRY_CHANGE, (event) => geometryChangeEventCallback(event));
                this.verificationClient.addEventListener(OMSDK_EVENT_AUDIO_VOLUME_CHANGE, (event) => volumeChangeEventCallback(event));

            } else {
                logAlways("OMSDKEventTracker::constructor OMID is not supported (MRAID-3 Disabled)");
                this.errorEventCallback("MRAID3-OMSDKEventTracker::constructor OMID is not supported (MRAID-3 Disabled)");
            }
        }

        // We're currently registering session observer more for debug purposes (MRAID3 impl doesn't seem to need anything from here) to confirm that
        /**
        * @param {Object} data event data
        */
        sessionEventCallback(event) {
            logD("OMSDKEventTracker::sessionEventCallback() " + JSON.stringify(event));
            if (!event || !event.type) {
              this.errorEventCallback("sessionEventCallback(): Invalid data or data.type object received in session event handler");
              return;
            }

            if (event.type == OMSDK_SESSION_EVENT_START) {
              const verificationParameters = event.data.verificationParameters;
              logD("OMSDKEventTracker::sessionEventCallback(): Context: '" + JSON.stringify(event.data.context) + "'");
              logD("OMSDKEventTracker::sessionEventCallback(): Verification parameters '" + JSON.stringify(verificationParameters) + "'");
            }
            else if (event.type == OMSDK_SESSION_EVENT_ERROR) {
              this.errorEventCallback("sessionEventCallback(): got data.type as " + event.type);
            }
            logD("OMSDKEventTracker::sessionEventCallback(): leaving...");
        }
    }
    ////////////////// END MRAID-3 OMSDK DEPENDENCIES //////////////////


    ////////////////// MRAID 2.0 CONTINUED //////////////////
	var mraid = window.mraid = {};

	/**
	 * Constants.
	 */

	// Version of MRAID that we currently support. The Pandora app can modify this based on MRAID experiment
	// status. We are currently either supporting 2.0 or 3.0.
    var MRAID_VERSION = "3.0";

	// MRAID SDK (mraid.js) metadata.
	var SDK_VERSION = '3.0.0';
	var SDK_NAME = 'Pandora MRAID SDK';
	var IOS_APP_ID = 'com.pandora';
	var ANDROID_APP_ID = 'com.pandora.android';

	// EXPANDED, RESIZED, HIDDEN states are not yet supported.
	var STATES = mraid.STATES = {
	    "LOADING" : "loading",
	    "DEFAULT" : "default",
        "HIDDEN"  : "hidden",
        "RESIZED" : "resized",
        "EXPANDED": "expanded"
	};

	var EVENTS = mraid.EVENTS = {
	    "ERROR" : "error",
	    "READY" : "ready",
	    "SIZECHANGE" : "sizeChange",
	    "STATECHANGE" : "stateChange",
		"EXPOSURECHANGE" : "exposureChange",
		"AUDIOVOLUMECHANGE" : "audioVolumeChange",
		"VIEWABLECHANGE" : "viewableChange"
	};

	/**
	 * Variables.
	 */

	var state = STATES.LOADING;

	var isViewable = false;

	var listeners = {};

	var supportedFeatures = {};

	var maxSize = {};

	var expandProperties = {
	    //read only property for the creative
	    //we don't allow a modal ad
	    isModal : false
	};

	// The creative is responsible for setting these properties via setOrientationProperties
	var orientationProperties = {
	    "allowOrientationChange" : true,
	    "forceOrientation" : "none"
	};

	// Init orientation as the window.orientation, but the client should update this value via
	// setCurrentAppOrientation
	// This will be ignored on android as our current implementation can't handle rotations.
	// The webview is destroyed and the ad is discarded upon rotation on android.
	var currentAppOrientation = {
	    "orientation" : window.orientation,
	    "locked" : false
	};

	// width, height, offsetX, offsetY are mandatory values,
	// customClosePosition and allowOffscreen are optionals
	var resizeProperties = {
    	width: false,
    	height: false,
    	offsetX: false,
    	offsetY: false,
    	allowOffscreen: false
  	};

  	var creativeOriginCoordinates = {
  		x: false,
  		y: false,
  	};

    var creativeCurrentPosition = {
      width: false,
      height: false,
    };

    var creativeDefaultPosition = {
      x: false,
      y: false,
      width: false,
      height: false,
    };
	/**
	 * MRAID Environment.
	 */

	window.MRAID_ENV = {
		version: MRAID_VERSION,
		sdk: SDK_NAME,
		sdkVersion: SDK_VERSION,
		appId: iOS ? IOS_APP_ID : ANDROID_APP_ID,
		ifa: '',
		limitAdTracking: true,
		coppa: false,
	};

	/**
	 * Supported MRAID 2.0 methods.
	 */

	mraid.addEventListener = function(event, listener) {
    	if (!event || !listener) {
    		console.log("mraid.addEventListener error: both event and listener are required.");
    	} else if (!contains(event, EVENTS)) {
    		console.log("mraid.addEventListener error: unknown MRAID event: " + event +".");
    	} else {
    		if (!listeners[event]) {
    			listeners[event] = new EventListeners(event);
    		}
            listeners[event].add(listener);
        }
	};

	mraid.removeEventListener = function(event, listener) {
    	if (!event) {
    		console.log("mraid.removeEventListener error: event is required.");
			return;
    	}

		if (listener) {
			var success = false;
			if (listeners[event]) {
				success = listeners[event].remove(listener);
			}

			if (!success) {
				console.log("mraid.removeEventListener error: listener not currently registered for event.");
				return;
			}
		} else if (!listener && listeners[event]) {
			listeners[event].removeAll();
		}

		if (listeners[event] && listeners[event].count === 0) {
			listeners[event] = null;
			delete listeners[event];
		}
	};

  mraid.setOrientationProperties = function(properties) {

      if (properties.hasOwnProperty('allowOrientationChange')) {
        orientationProperties.allowOrientationChange = properties.allowOrientationChange;
      }

      if (properties.hasOwnProperty('forceOrientation')) {
        orientationProperties.forceOrientation = properties.forceOrientation;
      }

      callNative({name: "setOrientationProperties", params: {"allowOrientationChange": orientationProperties.allowOrientationChange, "forceOrientation": orientationProperties.forceOrientation} });
    };

    mraid.getOrientationProperties = function() {
      return {
        allowOrientationChange: orientationProperties.allowOrientationChange,
        forceOrientation: orientationProperties.forceOrientation
      };
    };


	mraid.open = function(URL) {
		callNative({name: "open", params: {url: URL}});
	};

	mraid.getVersion = function() {
	    return MRAID_VERSION;
	};

	mraid.getState = function() {
	    return state;
	};

	mraid.supports = function(feature) {
		if (!contains(feature, Object.keys(supportedFeatures))) return false;
		return supportedFeatures[feature];
	};

	mraid.setSupports = function(feature, supported) {
		supportedFeatures[feature] = supported;
	};

	mraid.isViewable = function() {
	    return isViewable;
	};

	mraid.viewableChange = function(newViewability) {
		isViewable = newViewability;
		// Fire viewability change event after changing isViewable.
		fireEvent(EVENTS.VIEWABLECHANGE, isViewable);
	};

	mraid.exposureChange = function(exposureChangePercentExposed, exposureChangeVisibleRectangle, exposureChangeOcclusionRectangles) {
		fireEvent(EVENTS.EXPOSURECHANGE, exposureChangePercentExposed, exposureChangeVisibleRectangle, exposureChangeOcclusionRectangles);
	};

	mraid.getPlacementType = function() {
		return "inline";
	};

	mraid.audioVolumeChange = function(volumePercentage) {
		fireEvent(EVENTS.AUDIOVOLUMECHANGE, volumePercentage);
	};

	/**
	 * Called from the native app when the ad container has fully loaded the creative.
	 */
	mraid.notifyReady = function() {
		mraid.stateChange(STATES.DEFAULT);
	    fireEvent(EVENTS.READY);

        setupMraid3Dependencies() && setupMraid3EventTracking();
	};

	/**
	 * Change state.
	 */
	mraid.stateChange = function(newState) {
		state = newState;
		// Fire state change event after setting state.
		fireEvent(EVENTS.STATECHANGE, state);
	};

	mraid.sizeChange = function(width, height){
	    fireEvent(EVENTS.sizeChange, width, height)
	}

	mraid.close = function() {
      callNative({name: "close", params: {} });
    };

	mraid.expand = function() {
	    mraid.stateChange(STATES.EXPANDED);
		callNative({name: "expand", params: {} });
		console.log("changed the mraid state to expanded");
	};

	mraid.getCurrentAppOrientation = function() {
        return currentAppOrientation;
	};

	mraid.setCurrentAppOrientation = function(currentOrientation) {
        currentAppOrientation.orientation = currentOrientation.orientation;
        currentAppOrientation.locked = currentOrientation.locked;
        console.log(`setCurrentAppOrientation called -> orientation = ${currentAppOrientation.orientation} locked = ${currentAppOrientation.locked}`);
	};

  mraid.setCurrentPosition = function(currentPosition) {
    console.log(`Current Position x = ${currentPosition.x}, y = ${currentPosition.y}, width = ${currentPosition.width}, height = ${currentPosition.height} `);
    creativeOriginCoordinates.x = currentPosition.x;
    creativeOriginCoordinates.y = currentPosition.y;
    creativeCurrentPosition.width = currentPosition.width;
    creativeCurrentPosition.height = currentPosition.height;
  }

	mraid.getCurrentPosition = function() {
    console.log(`x = ${creativeOriginCoordinates.x},\ny = ${creativeOriginCoordinates.y},\nwidth = ${creativeCurrentPosition.width},\nheight = ${creativeCurrentPosition.height}`);
		return {
            width: creativeCurrentPosition.width,
            height: creativeCurrentPosition.height,
            x: creativeOriginCoordinates.x,
            y: creativeOriginCoordinates.y
        };
	};

  mraid.setDefaultPosition = function(defaultPosition) {
    console.log(`Current Position = ${defaultPosition}`)
    creativeDefaultPosition.x = defaultPosition.x;
    creativeDefaultPosition.y = defaultPosition.y;
    creativeDefaultPosition.width = defaultPosition.width;
    creativeDefaultPosition.height = defaultPosition.height;
  }

	mraid.getDefaultPosition = function() {
    console.log(`x = ${creativeDefaultPosition.x},\ny = ${creativeDefaultPosition.y},\nwidth = ${creativeDefaultPosition.width},\nheight = ${creativeDefaultPosition.height}`);
    return {
      width: creativeDefaultPosition.width,
      height: creativeDefaultPosition.height,
      x: creativeDefaultPosition.x,
      y: creativeDefaultPosition.y
    };
	};

  mraid.getExpandProperties = function() {
		return expandProperties;
	};

  mraid.getMaxSize = function() {
		return maxSize;
	};

	mraid.setMaxSize = function(clientMaxSize) {
	    maxSize = {
            width: clientMaxSize.width,
            height: clientMaxSize.height
        };

        //Before setExpandProperties is called, expandProperties width and height should be set
        //to the size of the container, according to the MRAID 3.0 spec
        expandProperties.width = maxSize.width;
        expandProperties.height = maxSize.height;

        console.log(`setMaxSize called -> width = ${maxSize.width} height = ${maxSize.height}`);
	};

	mraid.setContainerCoordinates = function(containerCoordinates) {
		creativeOriginCoordinates.x = containerCoordinates.x
		creativeOriginCoordinates.y = containerCoordinates.y
	};

	mraid.getContainerCoordinates = function() {
		return creativeOriginCoordinates;
	};

	mraid.getOrientationProperties = function() {
		return orientationProperties;
	};

    mraid.getScreenSize = function() {
        return {
            width: maxSize.width,
            height: maxSize.height
        };
    };

	mraid.playVideo = function(URL) {
      if (!mraid.isViewable()) {
        fireEvent(EVENTS.ERROR, "playVideo cannot be called until the creative is viewable", "playVideo");
        return;
      }

      if (!URL) {
        fireEvent(EVENTS.ERROR, "playVideo must be called with a valid URL", "playVideo");
      } else {
        callNative({name: "playVideo", params: {url: URL}});
      }
	};

  mraid.setExpandProperties = function(properties) {
    if (validate(properties, expandPropertyValidators, 'setExpandProperties', true)) {
      if (properties.hasOwnProperty('useCustomClose')) {
        // This is deprecated in MRAID 3.0 but is implemented to support MRAID 2.0
        // we will not respect the property on the client
        expandProperties.useCustomClose = properties.useCustomClose;
      }

      expandProperties.height = properties.height
      expandProperties.width = properties.width
    }
  };

	// Won't add the customClosePosition as this property is deprecated on MRAID 3.0
	// Also on MRAID 3.0 offsetX and offsetY are not optional anymore

	mraid.resize = function() {
		if (!(this.getState() === STATES.DEFAULT || this.getState() === STATES.RESIZED)) {
	  		fireEvent(EVENTS.ERROR, 'Ad can only be resized from the default or resized state.', 'resize');
		} else if (!resizeProperties.width || !resizeProperties.height || !resizeProperties.offsetX || !resizeProperties.offsetY) {
	  		fireEvent(EVENTS.ERROR, 'Must set resize properties before calling resize()', 'resize');
		} else {
	  		var args = {"width": resizeProperties.width,
	  					"height": resizeProperties.height,
	  					"offsetX": resizeProperties.offsetX,
	  					"offsetY": resizeProperties.offsetY,
	  					"allowOffscreen": !!resizeProperties.allowOffscreen
	    				};

	  		callNative({name: "resize", params: {resizeProperties: args} });
		}
	};

	mraid.getResizeProperties = function() {
		var properties = {
			width: resizeProperties.width,
			height: resizeProperties.height,
			offsetX: resizeProperties.offsetX,
			offsetY: resizeProperties.offsetY,
			allowOffscreen: resizeProperties.allowOffscreen
		};
		orientationProperties.allowOrientationChange = properties.allowOrientationChange;
		orientationProperties.forceOrientation = properties.forceOrientation;
		return properties;
	};

	mraid.setResizeProperties = function(properties) {
		if (validate(properties, resizePropertyValidators, 'setResizeProperties', true)) {

     		var desiredProperties = ['width', 'height', 'offsetX', 'offsetY', 'allowOffscreen'];
        var length = desiredProperties.length;

    		for (var i = 0; i < length; i++) {
      		var propname = desiredProperties[i];
      		if (properties.hasOwnProperty(propname)) {
        			resizeProperties[propname] = properties[propname];
      		}
    		}
    }
  };

	/**
	 * Unsupported MRAID 2.0 methods.
	 */

	mraid.createCalendarEvent = function(URI) {
    	console.log("mraid.createCalendarEvent is not currently supported");
	};

	mraid.getLocation = function() {
		console.log("mraid.getLocation is not currently supported");
		return {};
	};

  mraid.storePicture = function(URL) {
    console.log("mraid.storePicture is not currently supported");
  };

	mraid.unload = function() {
        callNative({name: "unload", params: {} });
	};

	mraid.useCustomClose = function(boolean) {
		console.log("mraid.useCustomClose is not currently supported");
	};

	/**
	 * Helpers.
	 */

  var resizePropertyValidators = {
      width: function(v) {
        return !isNaN(v) && v > 0;
      },
      height: function(v) {
        return !isNaN(v) && v > 0;
      },
      offsetX: function(v) {
        return !isNaN(v);
      },
      offsetY: function(v) {
        return !isNaN(v);
      }

      // Deprecated on MRAID 3.0 host will always add close indicator in top right corner
      /*
      customClosePosition: function(v) {
        return (typeof v === 'string' &&
          ['top-right', 'bottom-right', 'top-left', 'bottom-left', 'center', 'top-center', 'bottom-center'].indexOf(v) > -1);
      },*/

      // This is optional for MRAID 3.0 spec, no need to validate
      // allowOffscreen: function(v) {
      //   return (typeof v === 'boolean');
      // }
    };

    var expandPropertyValidators = {
    	useCustomClose: function(v) { return (typeof v === 'boolean'); },
  	};

    var validate = function(obj, validators, action, merge) {
    	if (!merge) {
      		// Check to see if any required properties are missing.
      		if (obj === null) {
        		fireEvent(EVENTS.ERROR, 'Required object not provided.', action);
        		return false;
      		} else {
        		for (var i in validators) {
          			if (validators.hasOwnProperty(i) && obj[i] === undefined) {
            			fireEvent(EVENTS.ERROR, 'Object is missing required property: ' + i, action);
            			return false;
          			}
        		}
      		}
    	  }
        return true;
    };

	var fireEvent = function() {
	    var args = new Array(arguments.length);
	    var l = arguments.length;
	    for (var i = 0; i < l; i++) {
			     args[i] = arguments[i];
		  }
	    var event = args.shift();
	    logAlways("MRAID3: fireEvent(): Called for EVENT="+event);
	    if (listeners[event]) {
	        logAlways("MRAID3: fireEvent(): EVENT="+event + " [There's a listener registered for it...FIRING MRAID event]");
			listeners[event].broadcast(args);
		} else {
		    logD("MRAID3: fireEvent(): EVENT="+event + " [NOT firing MRAID event - no listener registered]");
		}
	};

	var contains = function(value, array) {
	    for (var i in array) {
	      	if (array[i] === value) return true;
	    }
	    return false;
	};

	var EventListeners = function(event) {
	    this.event = event;
	    this.count = 0;
	    var listeners = {};

	    this.add = function(func) {
	    	var id = String(func);
	    	if (!listeners[id]) {
	        	listeners[id] = func;
	        	this.count++;

	        	// For MRAID-3
	        	if(event === EVENTS.EXPOSURECHANGE) {
                    // Fire initial (on subscribe) MRAID exposureChange event
                    logAlways("EventListeners::add(): Firing initial exposureChange event on subscribe...data: " + JSON.stringify(exposureChangePercentExposed) + " / " + JSON.stringify(exposureChangeVisibleRectangle) + " / " + JSON.stringify(exposureChangeOcclusionRectangles));
                    mraid.exposureChange(exposureChangePercentExposed, exposureChangeVisibleRectangle, exposureChangeOcclusionRectangles);
	        	}
	        	else if(event === EVENTS.AUDIOVOLUMECHANGE) {
                    // Fire initial (on subscribe) MRAID audioVolumeChange event
                    logAlways("EventListeners::add(): Firing initial audioVolumeChange event on subscribe...data (volume%): " + JSON.stringify(audioVolumeChangeVolumePercent));
                    mraid.audioVolumeChange(audioVolumeChangeVolumePercent);
	        	}
	      	}
	    };

	    this.remove = function(func) {
	      	var id = String(func);
	      	if (listeners[id]) {
	        	listeners[id] = null;
	        	delete listeners[id];
	        	this.count--;
	        	return true;
	      	} else {
	        	return false;
	      	}
	    };

	    this.removeAll = function() {
			     for (var i = 0, keys = Object.keys(listeners); i < keys.length; i++) {
				       this.remove(keys[i]);
			     }
	    };

	    this.broadcast = function(args) {
	      	for (var id in listeners) {
	        	if (contains(id, Object.keys(listeners))) listeners[id].apply(mraid, args);
	      	}
	    };
	};

}());
