var webkit = window.webkit || {};
var messageHandlers = webkit.messageHandlers || {};
var iOS = messageHandlers.mraid;

(function() {
    this.callNative = function(command) {
        if(iOS){
            iOS.postMessage(command);
	    } else if (typeof Android === 'object'){
	        Android.onMRAIDMessageReceived(JSON.stringify(command));
	    }
	}
}());

(function() {
	var oldLog = console.log;
	console.log = function(log) {
		callNative({name: "console-log", params: {message: log}});
		oldLog(log);
	}
	console.debug = console.info = console.warn = console.error = console.log;
}());

(function() {

	var mraid = window.mraid = {};

	/**
	 * Constants.
	 */

	// Version of MRAID that we currently support. The Pandora app can modify this based on MRAID experiment
	// status. We are currently either supporting 2.0 or 3.0.
	var MRAID_VERSION = "2.0";

	// MRAID SDK (mraid.js) metadata.
	var SDK_VERSION = '2.1.1';
	var SDK_NAME = 'Pandora MRAID SDK';
	var IOS_APP_ID = 'com.pandora';
	var ANDROID_APP_ID = 'com.pandora.android';

	// EXPANDED, RESIZED, HIDDEN states are not yet supported.
	var STATES = mraid.STATES = {
	    "LOADING" : "loading",
	    "DEFAULT" : "default",
        "HIDDEN"  : "hidden",
        "RESIZED" : "resized",
        "EXPANDED": "expanded"
	};

	var EVENTS = mraid.EVENTS = {
	    "ERROR" : "error",
	    "READY" : "ready",
	    "SIZECHANGE" : "sizeChange",
	    "STATECHANGE" : "stateChange",
		"EXPOSURECHANGE" : "exposureChange",
		"AUDIOVOLUMECHANGE" : "audioVolumeChange",
		"VIEWABLECHANGE" : "viewableChange"
	};

	/**
	 * Variables.
	 */

	var state = STATES.LOADING;

	var isViewable = false;

	var listeners = {};

	var supportedFeatures = {};

	var maxSize = {};

	var expandProperties = {
	    //read only property for the creative
	    //we don't allow a modal ad
	    isModal : false
	};

	// The creative is responsible for setting these properties via setOrientationProperties
	var orientationProperties = {
	    "allowOrientationChange" : true,
	    "forceOrientation" : "none"
	};

	// Init orientation as the window.orientation, but the client should update this value via
	// setCurrentAppOrientation
	// This will be ignored on android as our current implementation can't handle rotations.
	// The webview is destroyed and the ad is discarded upon rotation on android.
	var currentAppOrientation = {
	    "orientation" : window.orientation,
	    "locked" : false
	};

	// width, height, offsetX, offsetY are mandatory values,
	// customClosePosition and allowOffscreen are optionals
	var resizeProperties = {
    	width: false,
    	height: false,
    	offsetX: false,
    	offsetY: false,
    	allowOffscreen: false
  	};

  	var creativeOriginCoordinates = {
  		x: false,
  		y: false,
  	};

    var creativeCurrentPosition = {
      width: false,
      height: false,
    };

    var creativeDefaultPosition = {
      x: false,
      y: false,
      width: false,
      height: false,
    };
	/**
	 * MRAID Environment.
	 */

	window.MRAID_ENV = {
		version: MRAID_VERSION,
		sdk: SDK_NAME,
		sdkVersion: SDK_VERSION,
		appId: iOS ? IOS_APP_ID : ANDROID_APP_ID,
		ifa: '',
		limitAdTracking: true,
		coppa: false,
	};

	/**
	 * Supported MRAID 2.0 methods.
	 */

	mraid.addEventListener = function(event, listener) {
    	if (!event || !listener) {
    		console.log("mraid.addEventListener error: both event and listener are required.");
    	} else if (!contains(event, EVENTS)) {
    		console.log("mraid.addEventListener error: unknown MRAID event: " + event +".");
    	} else {
    		if (!listeners[event]) {
    			listeners[event] = new EventListeners(event);
    		}
			     listeners[event].add(listener);
    }
	};

	mraid.removeEventListener = function(event, listener) {
    	if (!event) {
    		console.log("mraid.removeEventListener error: event is required.");
			return;
    	}

		if (listener) {
			var success = false;
			if (listeners[event]) {
				success = listeners[event].remove(listener);
			}

			if (!success) {
				console.log("mraid.removeEventListener error: listener not currently registered for event.");
				return;
			}
		} else if (!listener && listeners[event]) {
			listeners[event].removeAll();
		}

		if (listeners[event] && listeners[event].count === 0) {
			listeners[event] = null;
			delete listeners[event];
		}
	};

  mraid.setOrientationProperties = function(properties) {

      if (properties.hasOwnProperty('allowOrientationChange')) {
        orientationProperties.allowOrientationChange = properties.allowOrientationChange;
      }

      if (properties.hasOwnProperty('forceOrientation')) {
        orientationProperties.forceOrientation = properties.forceOrientation;
      }

      callNative({name: "setOrientationProperties", params: {"allowOrientationChange": orientationProperties.allowOrientationChange, "forceOrientation": orientationProperties.forceOrientation} });
    };

    mraid.getOrientationProperties = function() {
      return {
        allowOrientationChange: orientationProperties.allowOrientationChange,
        forceOrientation: orientationProperties.forceOrientation
      };
    };


	mraid.open = function(URL) {
		callNative({name: "open", params: {url: URL}});
	};

	mraid.getVersion = function() {
	    return MRAID_VERSION;
	};

	mraid.getState = function() {
	    return state;
	};

	mraid.supports = function(feature) {
		if (!contains(feature, Object.keys(supportedFeatures))) return false;
		return supportedFeatures[feature];
	};

	mraid.setSupports = function(feature, supported) {
		supportedFeatures[feature] = supported;
	};

	mraid.isViewable = function() {
	    return isViewable;
	};

	mraid.viewableChange = function(newViewability) {
		isViewable = newViewability;
		// Fire viewability change event after changing isViewable.
		fireEvent(EVENTS.VIEWABLECHANGE, isViewable);
	};

	mraid.exposureChange = function(exposedPercentage, visibleRectangle, occlusionRectangles) {
		fireEvent(EVENTS.EXPOSURECHANGE, exposedPercentage, visibleRectangle, occlusionRectangles);
	};

	mraid.getPlacementType = function() {
		return "inline";
	};

	mraid.audioVolumeChange = function(volumePercentage) {
		fireEvent(EVENTS.AUDIOVOLUMECHANGE, volumePercentage);
	};

	/**
	 * Called from the native app when the ad container has fully loaded the creative.
	 */
	mraid.notifyReady = function() {
		mraid.stateChange(STATES.DEFAULT);
	    fireEvent(EVENTS.READY);
	};

	/**
	 * Change state.
	 */
	mraid.stateChange = function(newState) {
		state = newState;
		// Fire state change event after setting state.
		fireEvent(EVENTS.STATECHANGE, state);
	};
	
	mraid.sizeChange = function(width, height){
	    fireEvent(EVENTS.sizeChange, width, height)
	}

	mraid.close = function() {
      callNative({name: "close", params: {} });
};

	mraid.expand = function() {
	    mraid.stateChange(STATES.EXPANDED);
		callNative({name: "expand", params: {} });
		console.log("changed the mraid state to expanded");
	};

	mraid.getCurrentAppOrientation = function() {
        return currentAppOrientation;
	};

	mraid.setCurrentAppOrientation = function(currentOrientation) {
        currentAppOrientation.orientation = currentOrientation.orientation;
        currentAppOrientation.locked = currentOrientation.locked;
        console.log(`setCurrentAppOrientation called -> orientation = ${currentAppOrientation.orientation} locked = ${currentAppOrientation.locked}`);
	};

  mraid.setCurrentPosition = function(currentPosition) {
    console.log(`Current Position x = ${currentPosition.x}, y = ${currentPosition.y}, width = ${currentPosition.width}, height = ${currentPosition.height} `);
    creativeOriginCoordinates.x = currentPosition.x;
    creativeOriginCoordinates.y = currentPosition.y;
    creativeCurrentPosition.width = currentPosition.width;
    creativeCurrentPosition.height = currentPosition.height;
  }

	mraid.getCurrentPosition = function() {
    console.log(`x = ${creativeOriginCoordinates.x},\ny = ${creativeOriginCoordinates.y},\nwidth = ${creativeCurrentPosition.width},\nheight = ${creativeCurrentPosition.height}`);
		return {
            width: creativeCurrentPosition.width,
            height: creativeCurrentPosition.height,
            x: creativeOriginCoordinates.x,
            y: creativeOriginCoordinates.y
        };
	};

  mraid.setDefaultPosition = function(defaultPosition) {
    console.log(`Current Position = ${defaultPosition}`)
    creativeDefaultPosition.x = defaultPosition.x;
    creativeDefaultPosition.y = defaultPosition.y;
    creativeDefaultPosition.width = defaultPosition.width;
    creativeDefaultPosition.height = defaultPosition.height;
  }

	mraid.getDefaultPosition = function() {
    console.log(`x = ${creativeDefaultPosition.x},\ny = ${creativeDefaultPosition.y},\nwidth = ${creativeDefaultPosition.width},\nheight = ${creativeDefaultPosition.height}`);
    return {
      width: creativeDefaultPosition.width,
      height: creativeDefaultPosition.height,
      x: creativeDefaultPosition.x,
      y: creativeDefaultPosition.y
    };
	};

  mraid.getExpandProperties = function() {
		return expandProperties;
	};

  mraid.getMaxSize = function() {
		return maxSize;
	};

	mraid.setMaxSize = function(clientMaxSize) {
	    maxSize = {
            width: clientMaxSize.width,
            height: clientMaxSize.height
        };

        //Before setExpandProperties is called, expandProperties width and height should be set
        //to the size of the container, according to the MRAID 3.0 spec
        expandProperties.width = maxSize.width;
        expandProperties.height = maxSize.height;

        console.log(`setMaxSize called -> width = ${maxSize.width} height = ${maxSize.height}`);
	};

	mraid.setContainerCoordinates = function(containerCoordinates) {
		creativeOriginCoordinates.x = containerCoordinates.x
		creativeOriginCoordinates.y = containerCoordinates.y
	};

	mraid.getContainerCoordinates = function() {
		return creativeOriginCoordinates;
	};

	mraid.getOrientationProperties = function() {
		return orientationProperties;
	};

    mraid.getScreenSize = function() {
        return {
            width: maxSize.width,
            height: maxSize.height
        };
    };

	mraid.playVideo = function(URL) {
      if (!mraid.isViewable()) {
        fireEvent(EVENTS.ERROR, "playVideo cannot be called until the creative is viewable", "playVideo");
        return;
      }

      if (!URL) {
        fireEvent(EVENTS.ERROR, "playVideo must be called with a valid URL", "playVideo");
      } else {
        callNative({name: "playVideo", params: {url: URL}});
      }
	};

  mraid.setExpandProperties = function(properties) {
    if (validate(properties, expandPropertyValidators, 'setExpandProperties', true)) {
      if (properties.hasOwnProperty('useCustomClose')) {
        // This is deprecated in MRAID 3.0 but is implemented to support MRAID 2.0
        // we will not respect the property on the client
        expandProperties.useCustomClose = properties.useCustomClose;
      }

      expandProperties.height = properties.height
      expandProperties.width = properties.width
    }
  };

	// Won't add the customClosePosition as this property is deprecated on MRAID 3.0
	// Also on MRAID 3.0 offsetX and offsetY are not optional anymore

	mraid.resize = function() {
		if (!(this.getState() === STATES.DEFAULT || this.getState() === STATES.RESIZED)) {
	  		fireEvent(EVENTS.ERROR, 'Ad can only be resized from the default or resized state.', 'resize');
		} else if (!resizeProperties.width || !resizeProperties.height || !resizeProperties.offsetX || !resizeProperties.offsetY) {
	  		fireEvent(EVENTS.ERROR, 'Must set resize properties before calling resize()', 'resize');
		} else {
	  		var args = {"width": resizeProperties.width,
	  					"height": resizeProperties.height,
	  					"offsetX": resizeProperties.offsetX,
	  					"offsetY": resizeProperties.offsetY,
	  					"allowOffscreen": !!resizeProperties.allowOffscreen
	    				};

	  		callNative({name: "resize", params: {resizeProperties: args} });
		}
	};

	mraid.getResizeProperties = function() {
		var properties = {
			width: resizeProperties.width,
			height: resizeProperties.height,
			offsetX: resizeProperties.offsetX,
			offsetY: resizeProperties.offsetY,
			allowOffscreen: resizeProperties.allowOffscreen
		};
		orientationProperties.allowOrientationChange = properties.allowOrientationChange;
		orientationProperties.forceOrientation = properties.forceOrientation;
		return properties;
	};

	mraid.setResizeProperties = function(properties) {
		if (validate(properties, resizePropertyValidators, 'setResizeProperties', true)) {

     		var desiredProperties = ['width', 'height', 'offsetX', 'offsetY', 'allowOffscreen'];
        var length = desiredProperties.length;

    		for (var i = 0; i < length; i++) {
      		var propname = desiredProperties[i];
      		if (properties.hasOwnProperty(propname)) {
        			resizeProperties[propname] = properties[propname];
      		}
    		}
    }
  };

	/**
	 * Unsupported MRAID 2.0 methods.
	 */

	mraid.createCalendarEvent = function(URI) {
    	console.log("mraid.createCalendarEvent is not currently supported");
	};

	mraid.getLocation = function() {
		console.log("mraid.getLocation is not currently supported");
		return {};
	};

  mraid.storePicture = function(URL) {
    console.log("mraid.storePicture is not currently supported");
  };

	mraid.unload = function() {
		console.log("mraid.unload is not currently supported");
	};

	mraid.useCustomClose = function(boolean) {
		console.log("mraid.useCustomClose is not currently supported");
	};

	/**
	 * Helpers.
	 */

  var resizePropertyValidators = {
      width: function(v) {
        return !isNaN(v) && v > 0;
      },
      height: function(v) {
        return !isNaN(v) && v > 0;
      },
      offsetX: function(v) {
        return !isNaN(v);
      },
      offsetY: function(v) {
        return !isNaN(v);
      }

      // Deprecated on MRAID 3.0 host will always add close indicator in top right corner
      /*
      customClosePosition: function(v) {
        return (typeof v === 'string' &&
          ['top-right', 'bottom-right', 'top-left', 'bottom-left', 'center', 'top-center', 'bottom-center'].indexOf(v) > -1);
      },*/

      // This is optional for MRAID 3.0 spec, no need to validate
      // allowOffscreen: function(v) {
      //   return (typeof v === 'boolean');
      // }
    };

    var expandPropertyValidators = {
    	useCustomClose: function(v) { return (typeof v === 'boolean'); },
  	};

    var validate = function(obj, validators, action, merge) {
    	if (!merge) {
      		// Check to see if any required properties are missing.
      		if (obj === null) {
        		fireEvent(EVENTS.ERROR, 'Required object not provided.', action);
        		return false;
      		} else {
        		for (var i in validators) {
          			if (validators.hasOwnProperty(i) && obj[i] === undefined) {
            			fireEvent(EVENTS.ERROR, 'Object is missing required property: ' + i, action);
            			return false;
          			}
        		}
      		}
    	  }
        return true;
    };

	var fireEvent = function() {
	    var args = new Array(arguments.length);
	    var l = arguments.length;
	    for (var i = 0; i < l; i++) {
			     args[i] = arguments[i];
		  }
	    var event = args.shift();
	    if (listeners[event]) {
			     listeners[event].broadcast(args);
		  }
	};

	var contains = function(value, array) {
	    for (var i in array) {
	      	if (array[i] === value) return true;
	    }
	    return false;
	};

	var EventListeners = function(event) {
	    this.event = event;
	    this.count = 0;
	    var listeners = {};

	    this.add = function(func) {
	    	var id = String(func);
	    	if (!listeners[id]) {
	        	listeners[id] = func;
	        	this.count++;
	      	}
	    };

	    this.remove = function(func) {
	      	var id = String(func);
	      	if (listeners[id]) {
	        	listeners[id] = null;
	        	delete listeners[id];
	        	this.count--;
	        	return true;
	      	} else {
	        	return false;
	      	}
	    };

	    this.removeAll = function() {
			     for (var i = 0, keys = Object.keys(listeners); i < keys.length; i++) {
				       this.remove(keys[i]);
			     }
	    };

	    this.broadcast = function(args) {
	      	for (var id in listeners) {
	        	if (contains(id, Object.keys(listeners))) listeners[id].apply(mraid, args);
	      	}
	    };
	};

}());
