////// Fragment Shader
precision mediump float;

#define PI 3.1415926535

const int EdgingType_Color = 0;
const int EdgingType_Blur = 1;
const int EdgingType_Pattern = 2;
const int EdgingType_Gradient = 3;
const int EdgingType_Radial_Gradient = 4;

const int maxLength = 7;
const int maxColorLength = 7 * 3;

varying highp vec2 textureCoordinate;
varying highp vec2 textureCoordinate2;

uniform sampler2D inputImageTexture;
uniform sampler2D inputImageTexture2;

uniform mat4 mEdgingMatrix;
uniform mat4 mBlurMatrix;
uniform mat4 mEdgingLimitMatrix;
uniform float gradientColor[maxColorLength];

uniform int gradientCount;
uniform float degree;
uniform int mEdgingBgType;
uniform float width;
uniform float height;
uniform int blendType;

uniform highp vec2 inputSize;
uniform float locations[maxLength];
uniform float startPoint[2];
uniform float endPoint[2];

vec4 fanGradientColor(float sx, float sy, float ex, float ey) {
    vec3 color;
    vec2 st = textureCoordinate.xy;
    st.y = 1.0 - st.y;
    sy = 1.0 - sy;
    ey = 1.0 - ey;
    st.y = st.y * inputSize.y / inputSize.x;
    sy = sy * inputSize.y / inputSize.x;
    ey = ey * inputSize.y / inputSize.x;
    vec3 v0 = vec3(ex - sx, ey - sy, 0.0);
    vec3 v1 = vec3(st.x - sx, st.y - sy, 0.0);

    //calculate cosine value: a*b/(|a|*|b|)
    float dotp = dot(v0, v1);
    float cosval = dotp / (length(v0) * length(v1));

    //calculate sine value: |axb|/(|a|*|b|)
    vec3 crossval = cross(v0, v1);
    float sinval = length(crossval) / (length(v0) * length(v1));

    //alteration if at the other side of the line
    if (((ey - sy) * (st.x - sx) - (ex - sx) * (st.y - sy)) < 0.0) {
        sinval = -sinval;
    }

    //calulate arctangent value and convert from -PI:PI to 0:2*PI
    float angle = atan(sinval, cosval);
    if (angle < 0.0) {
        angle = angle + 2.0 * PI;
    }
    float anglenorm = angle / (PI * 2.0);

    //plot figure
    for (int i = 0; i < maxLength; i++) {
        if (i >= gradientCount - 1) {
            break;
        }
        if (anglenorm >= locations[i] && anglenorm <= locations[i + 1]) {
            float mixValue = (anglenorm - locations[i]) / (locations[i + 1] - locations[i]);
            vec3 startColor = vec3(gradientColor[i * 3], gradientColor[i * 3 + 1], gradientColor[i * 3 + 2]);
            vec3 endColor = vec3(gradientColor[(i + 1) * 3], gradientColor[(i + 1) * 3 + 1], gradientColor[(i + 1) * 3 + 2]);
            color = mix(startColor, endColor, mixValue);
            break;
        }
    }
    return vec4(color, 1.0);
}

vec4 centerGradientColor(float sx, float sy, float ex, float ey) {
    vec2 st = textureCoordinate;
    vec3 color;
    float widthRatio = inputSize.x / inputSize.y;
    vec2 centerPoint = vec2(startPoint[0] * widthRatio, startPoint[1]);
    vec2 edgePoint = vec2(endPoint[0] * widthRatio, endPoint[1]);
    vec2 currentPoint = vec2(st.x * widthRatio, st.y);
    float r = distance(centerPoint, edgePoint);
    float dist = distance(centerPoint, currentPoint);
    for (int i = 0; i < maxLength; i++) {
        if (i >= gradientCount - 1) {
            if (dist / r > locations[i]) {
                color = vec3(gradientColor[i * 3], gradientColor[i * 3 + 1], gradientColor[i * 3 + 2]);
            } else {
                color = vec3(gradientColor[0], gradientColor[1], gradientColor[2]);
            }
            break;
        }
        if (dist / r >= locations[i] && dist / r <= locations[i + 1]) {
            float diff0 = dist / r - locations[i];
            float diff1 = locations[i + 1] - locations[i];
            vec3 startColor = vec3(gradientColor[i * 3], gradientColor[i * 3 + 1], gradientColor[i * 3 + 2]);
            vec3 endColor = vec3(gradientColor[(i + 1) * 3], gradientColor[(i + 1) * 3 + 1], gradientColor[(i + 1) * 3 + 2]);
            color = mix(startColor, endColor, diff0 / diff1);
            break;
        }
    }
    return vec4(color, 1.0);
}

vec4 normalBlend(vec4 s, vec4 d) {
    vec4 c;
    c.r = s.r + d.r * d.a * (1.0 - s.a);
    c.g = s.g + d.g * d.a * (1.0 - s.a);
    c.b = s.b + d.b * d.a * (1.0 - s.a);
    c.a = s.a + d.a * (1.0 - s.a);
    return c;
}

vec4 setGradientBg() {
    float degreeResult = 360.0 - degree * 3.55;
    float gradientDegree;
    if (degreeResult >= 0.0 && degreeResult < 90.0) {
        gradientDegree = degreeResult;
    } else if (degreeResult > 90.0 && degreeResult <= 180.0) {
        gradientDegree = degreeResult - 90.0;
    } else if (degreeResult > 180.0 && degreeResult <= 270.0) {
        gradientDegree = degreeResult - 180.0;
    } else {
        gradientDegree = degreeResult - 270.0;
    }

    float radian = radians(gradientDegree);
    float newWdith = width * (cos(radian)) + height * (sin(radian));
    float newHeight = width * (sin(radian)) + height * (cos(radian));

    float posx = textureCoordinate.x * width;
    float posy = textureCoordinate.y * height;

    float newY = 0.0;
    if (degreeResult >= 0.0 && degreeResult < 90.0)
    newY = posy * cos(radian) + (width - posx) * sin(radian);
    else if (degreeResult >= 90.0 && degreeResult <= 180.0) {
        float temp = newWdith;
        newWdith = newHeight;
        newHeight = temp;
        newY = newHeight - (posx * cos(radian) + posy * sin(radian));
    } else if (degreeResult > 180.0 && degreeResult <= 270.0) {
        newY = (height - posy) * cos(radian) + posx * sin(radian);
    } else {
        float temp = newWdith;
        newWdith = newHeight;
        newHeight = temp;
        newY = posx * cos(radian) + posy * sin(radian);
    }
    float percent = newY / newHeight;
    float indexF = percent * (float(gradientCount) - 1.0);
    int index = int(indexF);
    float off = indexF - float(index);
    vec3 curPix;
    curPix.r = gradientColor[0 + index * 3] + (gradientColor[3 + index * 3] - gradientColor[0 + index * 3]) * off;
    curPix.g = gradientColor[1 + index * 3] + (gradientColor[4 + index * 3] - gradientColor[1 + index * 3]) * off;
    curPix.b = gradientColor[2 + index * 3] + (gradientColor[5 + index * 3] - gradientColor[2 + index * 3]) * off;
    return vec4(curPix, 1.0);
}

vec4 setPatternBg() {
    vec4 uvPattern = vec4(textureCoordinate2, 0.0, 1.0);
    uvPattern = mBlurMatrix * uvPattern;
    if (gradientColor[0] == 0.0) {

        return texture2D(inputImageTexture2, uvPattern.xy);
    }

    float scaleCount, percentBig, percentSmall;
    if (width > height) {
        scaleCount = width / height;
        percentBig = 1.0 / 6.0;
        percentSmall = percentBig * scaleCount;
    } else {
        scaleCount = height / width;
        percentBig = 1.0 / 6.0 * gradientColor[1];
        percentSmall = percentBig * scaleCount;
    }

    int indexX, indexY;
    float resultX, resultY;
    if (width > height) {
        indexX = int(uvPattern.x / percentBig);
        indexY = int(uvPattern.y / percentSmall);
        resultX = (uvPattern.x - float(indexX) * percentBig) / percentBig;
        resultY = (uvPattern.y - float(indexY) * percentSmall) / percentSmall;
    } else {
        indexX = int(uvPattern.x / percentSmall);
        indexY = int(uvPattern.y / percentBig);
        resultX = (uvPattern.x - float(indexX) * percentSmall) / percentSmall;
        resultY = (uvPattern.y - float(indexY) * percentBig) / percentBig;
    }
    return texture2D(inputImageTexture2, vec2(resultX, resultY));
}

vec4 setBg() {
    if (mEdgingBgType == EdgingType_Color) {
        return vec4(gradientColor[0], gradientColor[1], gradientColor[2], 1.0);
    } else if (mEdgingBgType == 1) {
        vec4 coordinateBlur = vec4(textureCoordinate2, 0.0, 1.0);
        coordinateBlur = mBlurMatrix * coordinateBlur;
        return texture2D(inputImageTexture2, coordinateBlur.xy);
    } else if (mEdgingBgType == EdgingType_Pattern) {
        return setPatternBg();
    } else if (mEdgingBgType == EdgingType_Gradient) {
        return setGradientBg();
    } else if (mEdgingBgType == EdgingType_Radial_Gradient) {
        if (abs(startPoint[0] - endPoint[0]) < 0.000001 && abs(startPoint[1] - endPoint[1]) < 0.000001) {
            return vec4(0.0, 0.0, 0.0, 1.0);
        }
        return centerGradientColor(startPoint[0], startPoint[1], endPoint[0], endPoint[1]);
    }  else {
        return vec4(0.0, 0.0, 0.0, 1.0);
    }
}

vec4 multiply(vec4 s, vec4 d) {
    return s * d + s * (1.0 - d.a) + d * (1.0 - s.a);
}

vec4 blend(vec4 s, vec4 d, int blendType) {
    if (blendType == 0)  return normalBlend(s, d);
    if (blendType == 6)  return multiply(s, d);
}


void main()
{
    highp vec4 coordinate2 = vec4(textureCoordinate, 0.0, 1.0);
    coordinate2 = mEdgingMatrix * coordinate2;

    if (coordinate2.x < 0.0 || coordinate2.x > 1.0 || coordinate2.y < 0.0 || coordinate2.y > 1.0) {
        gl_FragColor = setBg();
    } else {
        vec4 background = setBg();
        vec4 result = texture2D(inputImageTexture, coordinate2.xy);

        if (result.a == 0.0) {
            gl_FragColor = background;
        } else {
            gl_FragColor = blend(result, background, blendType);
        }
    }

}