#ifdef GL_FRAGMENT_PRECISION_HIGH
precision highp float; // TODO: mediump might be enough
#else
precision mediump float;
#endif

uniform sampler2D u_tex;
uniform sampler2D u_curves;
uniform float u_curvesRes;

varying vec2 coord;

#include utils

vec3 RGBToHSL(vec3 color)
{
    float fmin = min(min(color.r, color.g), color.b);
    float fmax = max(max(color.r, color.g), color.b);
    float chroma = fmax - fmin;

    vec3 hsl = vec3(0., fmin, fmax);

    if (chroma != 0.0)
    {
        vec3 delta = - color / 6. / chroma;

        if (color.r == fmax )
            hsl.x = delta.b - delta.g;
        else if (color.g == fmax)
            hsl.x = (1.0 / 3.0) + delta.r - delta.b;
        else if (color.b == fmax)
            hsl.x = (2.0 / 3.0) + delta.g - delta.r;

        hsl.x = fract(hsl.x);
    }

    return hsl;
}

float HueToRGB(float fmin, float fmax, float hue)
{
    hue = fract(hue);

    float res;
    if ((6.0 * hue) < 1.0)
        res = fmin + (fmax - fmin) * 6.0 * hue;
    else if ((2.0 * hue) < 1.0)
        res = fmax;
    else if ((3.0 * hue) < 2.0)
        res = fmin + (fmax - fmin) * ((2.0 / 3.0) - hue) * 6.0;
    else
        res = fmin;

    return res;
}

vec3 HSLToRGB(vec3 hsl)
{
    vec3 rgb = vec3(0.);

    float fmin = hsl.y;
    float fmax = hsl.z;

    rgb.r = HueToRGB(fmin, fmax, hsl.x + (1.0/3.0));
    rgb.g = HueToRGB(fmin, fmax, hsl.x);
    rgb.b = HueToRGB(fmin, fmax, hsl.x - (1.0/3.0));

    return rgb;
}

void hslShift(inout vec3 a,in sampler2D curves)
{
	vec3 c = RGBToHSL(a);
	float hue = c.x;
    float fmin = c.y;
    float fmax = c.z;
	float g = fmax - fmin; // chroma

	vec3 h = texture2D(curves, vec2(.5 / u_curvesRes + (u_curvesRes - 1.) / u_curvesRes * hue, .125)).xyz;
	vec3 d = (1. - h * 2.); // diff
    d.yz = -d.yz;

	hue = hue - d.x * 40./360.;
	hue = fract(1. + hue);
    c.x = hue;

    // Luminosity
    g *= (2. - g);
    g *= (2. - g);
    g *= (2. - g);
    float lum = d.z * g;
    c.y *= (1. - c.y) * lum + 1.;
    c.z *= (1. - c.z) * lum + 1.;
    c.y *= (1. - c.y) * lum + 1.;
    c.z *= (1. - c.z) * lum + 1.;


    // Saturation
    if(c.z > c.y) { // True if chroma != 0
        if(d.y > 0.) {
            float dyn = (c.z - c.y) / c.z;
            float big = min(1., c.z * 16.);
            c.y = d.y * (1. - c.y) * (2. - big) * big;
            if(c.y > 1.) {
                float var = (c.y - 1.) * 2.5;
                float vint = floor(var);
                float rem = var - vint; // rem = fract(var);
                c.y = ((1. - (1. - rem) * 0.33976999) * rem + 1.) * 6. * dyn; // TODO: Simplify
                dyn = c.y / ((c.y + 1.) - dyn);
            }
            else {
                dyn /= (1. - c.y * 0.83333331 * (1. - dyn));
            }
            c.y = c.z - dyn * c.z;
        }
        else if(d.y < 0.) {
            float sat = max(-1., d.y);
            float l = (c.y + c.z) * .5;
            c.y += (c.y - l) * sat;
            c.z += (c.z - l) * sat;
        }
    }

	a = HSLToRGB(c);
}

void main()
{
    vec3 rgb = texture2D(u_tex, coord).rgb;

    rgb = colorspace(rgb, u_curves, u_curvesRes);

    hslShift(rgb, u_curves);

    rgb = colorspaceInv(rgb, u_curves, u_curvesRes);

    gl_FragColor = vec4(rgb, 1.);
}
