// Modified from https://www.shadertoy.com/view/MslGWN

precision highp float;

varying vec2 fragCoord;
uniform vec3 iResolution;
uniform sampler2D iChannel0;
uniform float iTime;
uniform float iAlpha;


////////////////////////////////////////////////////////////////////////////////
//simplicity_galaxy.fsh

//CBS
//Parallax scrolling fractal galaxy.
//Inspired by JoshP's Simplicity shader: https://www.shadertoy.com/view/lslGWr

// http://www.fractalforums.com/new-theories-and-research/very-simple-formula-for-fractal-patterns/
float field(vec3 p, float s, int iter, float time) {
	//float strength = 7. + .03 * log(1.e-6 + fract(sin(time) * 4373.11));
	float strength = 9. + .03 * log(1.e-6 + fract(sin(time) * 4373.11));
	float accum = s/4.;
	float prev = 0.;
	float tw = 0.;
	for (int i = 0; i < iter; ++i) {
		float mag = dot(p, p);
		p = abs(p) / mag + vec3(-.5, -.4, -1.5);
		float w = exp(-float(i) / 7.);
		accum += w * exp(-strength * pow(abs(mag - prev), 2.2));
		tw += w;
		prev = mag;
	}
	return max(0., 5. * accum / tw - .7);
}

// Less iterations for second layer
/*
float field2(in vec3 p, float s) {
	//float strength = 7. + .03 * log(1.e-6 + fract(sin(iTime) * 4373.11));
	float strength = 9. + .03 * log(1.e-6 + fract(sin(iTime) * 4373.11));
	float accum = s/4.;
	float prev = 0.;
	float tw = 0.;
	for (int i = 0; i < 18; ++i) {
		float mag = dot(p, p);
		p = abs(p) / mag + vec3(-.5, -.4, -1.5);
		float w = exp(-float(i) / 7.);
		accum += w * exp(-strength * pow(abs(mag - prev), 2.2));
		tw += w;
		prev = mag;
	}
	return max(0., 5. * accum / tw - .7);
}
*/

vec3 nrand3( vec2 co )
{
	vec3 a = fract( cos( co.x*8.3e-3 + co.y )*vec3(1.3e5, 4.7e5, 2.9e5) );
	vec3 b = fract( sin( co.x*0.3e-3 + co.y )*vec3(8.1e5, 1.0e5, 0.1e5) );
	vec3 c = mix(a, b, 0.5);
	return c;
}
//simplicity_galaxy.fsh
////////////////////////////////////////////////////////////////////////////////


#define BEATMOVE 1

const float FREQ_RANGE = 256.0;
const float PI = 3.1415;
const float RADIUS = 0.5;
const float BRIGHTNESS = 0.3;
const float SPEED = 0.2;

//convert HSV to RGB
vec3 hsv2rgb(vec3 c){
    vec4 K = vec4(1.0, 2.0 / 3.0, 1.0 / 3.0, 3.0);
    vec3 p = abs(fract(c.xxx + K.xyz) * 6.0 - K.www);
    return c.z * mix(K.xxx, clamp(p - K.xxx, 0.0, 1.0), c.y);
}

float luma(vec3 color) {
  return dot(color, vec3(0.299, 0.587, 0.114));
}

float getfrequency(float x) {
	return texture2D(iChannel0, vec2(floor(x * FREQ_RANGE + 1.0) / FREQ_RANGE, 0.0)).x + 0.06;
}

float getfrequency_smooth(float x) {
	float index = floor(x * FREQ_RANGE) / FREQ_RANGE;
    float next = floor(x * FREQ_RANGE + 1.0) / FREQ_RANGE;
	return mix(getfrequency(index), getfrequency(next), smoothstep(0.0, 1.0, fract(x * FREQ_RANGE)));
}

float getfrequency_blend(float x) {
    return mix(getfrequency(x), getfrequency_smooth(x), 0.5);
}

vec3 doHalo(vec2 fragment, float radius) {
	float dist = length(fragment);
	float ring = 1.0 / (abs(dist - radius) + 0.005);

	float b = dist < radius ? BRIGHTNESS * 0.6 : BRIGHTNESS;

	vec3 col = vec3(0.0);

	float angle = atan(fragment.x, fragment.y);
	col += hsv2rgb( vec3( ( angle + iTime * 0.25 ) / (PI * 2.0), 0.6, 0.5 ) ) * ring * b;

	float frequency = max(getfrequency(abs(angle / PI)) - 0.02, 0.0);
	col *= frequency * 0.5;

	// Black halo
	col *= smoothstep(radius * 0.86, radius, dist);

	return col;
}

vec3 doLine(vec2 fragment, float radius, float x) {
	vec3 col = hsv2rgb(vec3(x * 0.23 + iTime * 0.12, 1.0, 1.0));

	float freq = abs(fragment.x * 0.5);

	col *= (1.0 / abs(fragment.y)) * BRIGHTNESS * getfrequency(freq);
	col = col * smoothstep(radius, radius * 1.8, abs(fragment.x));

	return col;
}


void main() {
    vec4 outColor;

	////////////////////////////////////////////////////////////////////////////////
	//simplicity_galaxy.fsh
    {
        vec2 uv = 2. * fragCoord.xy - 1.;
        vec2 uvs = uv * iResolution.xy / max(iResolution.x, iResolution.y);
        vec3 p = vec3(uvs / 4., 0) + vec3(1., -1.3, 0.);
        float xTime = iTime * 2.0;
        p += .2 * vec3(sin(xTime / 16.), sin(xTime / 12.),  sin(xTime / 128.));

        float freqs[4];
        //Sound
        freqs[0] = texture2D( iChannel0, vec2( 0.01, 0.25 ) ).x;
        freqs[1] = texture2D( iChannel0, vec2( 0.07, 0.25 ) ).x;
        freqs[2] = texture2D( iChannel0, vec2( 0.15, 0.25 ) ).x;
        freqs[3] = texture2D( iChannel0, vec2( 0.30, 0.25 ) ).x;

        float t = field(p,freqs[2],26,xTime);
        float v = (1. - exp((abs(uv.x) - 1.) * 6.)) * (1. - exp((abs(uv.y) - 1.) * 6.));

        //Second Layer
        vec3 p2 = vec3(uvs / (4.+sin(xTime*0.11)*0.2+0.2+sin(xTime*0.15)*0.3+0.4), 1.5) + vec3(2., -1.3, -1.);
        p2 += 0.25 * vec3(sin(xTime / 16.), sin(xTime / 12.),  sin(xTime / 128.));
        float t2 = field(p2,freqs[3],18,xTime); //field2(p2,freqs[3]);
        vec4 c2 = mix(.4, 1., v) * vec4(1.3 * t2 * t2 * t2 ,1.8  * t2 * t2 , t2* freqs[0], t2);


        //Let's add some stars
        //Thanks to http://glsl.heroku.com/e#6904.0
        vec2 seed = p.xy * 2.0;
        seed = floor(seed * iResolution.x);
        vec3 rnd = nrand3( seed );
        vec4 starcolor = vec4(pow(rnd.y,40.0));

        //Second Layer
        vec2 seed2 = p2.xy * 2.0;
        seed2 = floor(seed2 * iResolution.x);
        vec3 rnd2 = nrand3( seed2 );
        starcolor += vec4(pow(rnd2.y,40.0));

        outColor = mix(freqs[3]-.3, 1., v) * vec4(1.5*freqs[2] * t * t* t , 1.2*freqs[1] * t * t, freqs[3]*t, 1.0)+c2+starcolor;
    }

    {
        vec2 fragPos = fragCoord.xy; // / iResolution.xy;
        fragPos = (fragPos - 0.5) * 2.0;
        if (iResolution.x > iResolution.y)
            fragPos.x *= iResolution.x / iResolution.y;
        else
            fragPos.y *= iResolution.y / iResolution.x;

        vec3 color = vec3(0.0); //vec3(0.0134, 0.052, 0.1);
        color += doHalo(fragPos, RADIUS);

        //float c = cos(iTime * SPEED);
        //float s = sin(iTime * SPEED);
        //vec2 rot = mat2(c,s,-s,c) * fragPos;
        //color += doLine(rot, RADIUS, rot.x);

        color += max(luma(color) - 1.0, 0.0);

        outColor += vec4(color, 1.0);
    }

	gl_FragColor = outColor;
}