// Modified from https://www.shadertoy.com/view/4sGczm

precision highp float;

varying vec2 fragCoord;
uniform vec3 iResolution;
uniform sampler2D iChannel0;
uniform float iTime;
uniform float iAlpha;

//------------------------------------------------------------------------
// HSL color space
// Mimics the implementation in THREE.js 0.90.0
//------------------------------------------------------------------------
float hue2rgb(float p, float q, float t) {
    if ( t < 0. ) t += 1.;
    if ( t > 1. ) t -= 1.;
    if ( t < 1. / 6. ) return p + ( q - p ) * 6. * t;
    if ( t < 1. / 2. ) return q;
    if ( t < 2. / 3. ) return p + ( q - p ) * 6. * ( 2. / 3. - t );
    return p;
}
vec3 hsl2rgb(vec3 hsl) {
    // h,s,l ranges are in 0.0 - 1.0
    float h = mod(hsl.x, 1.);
    float s = clamp(hsl.y, 0., 1.);
    float l = clamp(hsl.z, 0., 1.);
    if (s == 0.)
        return vec3(l);
    else {
        float p, q;
        if (l <= 0.5) {
            p = l * ( 1. + s );
        }
        else {
            p = l + s - ( l * s );
        }
        q = ( 2. * l ) - p;
        return vec3(
        	hue2rgb(q, p, h + 1. / 3.),
            hue2rgb(q, p, h),
            hue2rgb(q, p, h - 1. / 3.)
        );
    }
}


float rand2(vec2 co){
    return fract(sin(dot(co.xy ,vec2(12.9898,78.233))) * 43758.5453);
}


// +-----------------+------------------+
// | Frequency Range | Frequency Values |
// +-----------------+------------------+
// | Sub-bass        | 20 to 60 Hz      |
// | Bass            | 60 to 250 Hz     |
// | Low midrange    | 250 to 500 Hz    |
// | Midrange        | 500 Hz to 2 kHz  |
// | Upper midrange  | 2 to 4 kHz       |
// | Presence        | 4 to 6 kHz       |
// | Brilliance      | 6 to 20 kHz      |
// +-----------------+------------------+
// https://www.teachmeaudio.com/mixing/techniques/audio-spectrum
#define SA_MIN 			0.
#define SA_MAX 			22050.
#define SA_SUB_BASS 	20./(SA_MAX-SA_MIN)
#define SA_BASS 		60./(SA_MAX-SA_MIN)
#define SA_LOW_MID 		250./(SA_MAX-SA_MIN)
#define SA_MID 			500./(SA_MAX-SA_MIN)
#define SA_HI_MID 		2000./(SA_MAX-SA_MIN)
#define SA_PRESENCE 	4000./(SA_MAX-SA_MIN)
#define SA_BRILLIANCE	6000./(SA_MAX-SA_MIN)


// How accurately to sample spectrum
#define SAMPLECOUNT 3 //ORG:200 slowhand:reduced for performance

float sampleRange(float start, float end) {
    float stepsize = (end - start) / float(SAMPLECOUNT);
	float intensity = 0.;
    for(int i = 0; i < SAMPLECOUNT; i += 1) {
		intensity += texture2D(iChannel0, vec2(start + float(i)*stepsize, 0.0)).r;
	}
	intensity /= float(SAMPLECOUNT);
    if (intensity < 0.2) //to show dark lights when no sound
        intensity = 0.2;
    return intensity;
}


void main()
{
    vec2 uv = fragCoord.xy; // / iResolution.xy;

    float beat = sampleRange(SA_SUB_BASS, SA_LOW_MID);
    float low = sampleRange(0., SA_LOW_MID);
    float mid = sampleRange(SA_LOW_MID, SA_PRESENCE);
    float hi = sampleRange(SA_PRESENCE, 1.);

    float t = iTime;
    float tcolor = iTime * .05;

    vec3 col = vec3(0.,0.,0.);

    float range = 0.9;
    float saturation = 0.9;
    float lightness = 0.6;
    float max_neg_light = -.0; // if set to less than zero, shadows everything else than lights
    float attenuation = 1.6;

    float dist1 = pow(clamp(
    	1.-distance(
            vec2(0.5+0.3*sin(t),0.5+0.3*cos(t)),
            uv.xy
        )/range,
        max_neg_light,1.
    ), attenuation);
    col += hsl2rgb(vec3(sin(tcolor), saturation, mid*lightness))*dist1;

    float dist2 = pow(clamp(
    	1.-distance(
            vec2(0.5+0.2*sin(-t*0.5),0.5+0.2*cos(-t*0.5)) + vec2(0.2*sin(t),0.2*cos(t*1.5)),
            uv.xy
        )/range,
        max_neg_light,1.
    ), attenuation);
    col += hsl2rgb(vec3(0.33+sin(tcolor), saturation, low*lightness))*dist2;

    float dist3 = pow(clamp(
    	1.-distance(
            vec2(0.5+0.5*sin(t*0.25), 0.5+0.4*sin(t)),
            uv.xy
        )/range,
        max_neg_light,1.
    ), attenuation);
    col += hsl2rgb(vec3(0.66+sin(tcolor), saturation, hi*lightness))*dist3;


    float loudness = texture2D(iChannel0, vec2(uv.x, 0.0)).r;

    // add some ambient light with glow around peaks
    float a2 = abs(.5-uv.y);
    col += .15*hsl2rgb(vec3(sin(tcolor)+.15, .2, .65*beat*(.5+loudness*.5)*a2));

    // frequency visualization
    float intensity_edge = .1;
    float intensity_bar = .9;
    float height = .4;
    float top = loudness*height+.5;
    float bottom = -loudness*height+.5;
    float a = min(abs(top - uv.y), abs(bottom - uv.y))/(loudness*height); // fade out

    if (top > uv.y && bottom < uv.y) {
        vec3 barcolor = hsl2rgb(vec3(
            sin(tcolor*1.2)+0.66 - loudness*.6,
            1.,
            .3+loudness*.3 // add a little bit of punch to peaks
        ));

        // visualize as bars
        vec3 bars = barcolor * intensity_bar*a*loudness;

        // visualize as bar edge only
        float edge_alpha = smoothstep(0.0, .12, a);
        vec3 edges = barcolor * intensity_edge*(1.-edge_alpha);

        float transition = smoothstep(0.7, .9, sin(iTime*.1));
        col += mix(bars, edges, transition); // rotate between two visualization types
    }

    // Add some noise to remove color banding
    float n = rand2(uv*2.+iTime*.001);
    col += vec3((.4+n*.2)*.01);

    gl_FragColor = vec4(col, 1.0);
}