import 'dart:io';

import 'package:cronometer/dialogs/progress_dialog.dart';
import 'package:cronometer/l10n/localization.dart';
import 'package:cronometer/localizations.dart';
import 'package:cronometer/model/global_context_container.dart';
import 'package:cronometer/model/user_prefs.dart';
import 'package:flutter/material.dart';
import 'package:cronometer/l10n/generated/app_localizations.dart';

class LocaleProvider extends ChangeNotifier {
  static final LocaleProvider _instance = LocaleProvider._internal();

  LocaleProvider._internal();

  factory LocaleProvider() => _instance;

  Locale? _locale;

  Locale? get locale => _locale;

  setLocaleOnLoad() {
    _locale = _getLocale();
  }

  Locale? _getLocale() {
    String localePreference = SharedPrefs.language.getSync();
    if (localePreference == "not_set") {
      localePreference = Platform.localeName;
    }

    // Convert to crono localization
    Localization localization = Localization(locale: localePreference);
    return localization.language.locale;
  }

  String getToolTipJsonPath() {
    if (_locale == null) {
      return SupportedLanguages.en_US.tooltipPath;
    }

    // Convert to Crono localization
    Localization localization = Localization.fromLocale(_locale!);
    return localization.language.tooltipPath;
  }

  String getAssetPath({required String originalPath}) {
    if (_locale == null) {
      return originalPath;
    }

    // Convert to Crono localization
    Localization localization = Localization.fromLocale(_locale!);
    String prefix = localization.language.assetPathPrefix;

    if (prefix.isNotEmpty) {
      return originalPath.replaceFirst("/", "/$prefix");
    }

    return originalPath;
  }

  //The future delay here is used to insure there isn't a failed race case were the global context isn't updated with the new locale and
  // the app still shows the old language
  Future<void> setLocale(
      {required Locale locale, required BuildContext context}) async {
    _locale = locale;
    notifyListeners();
    await ProgressDialog.showUntil(context, toExecute: () async {
      await Future.delayed(Duration(seconds: 5));
      L10n.get = AppLocalizations.of(
          GlobalContextContainer().context!)!;
    });
    notifyListeners();
  }

  void clearLocale() {
    _locale = Locale('en', 'US');
    notifyListeners();
  }
}
